/*
 * Copyright 2013-2020 Software Radio Systems Limited
 *
 * This file is part of srsLTE.
 *
 * srsLTE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsLTE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

#include "srslte/asn1/rrc_asn1.h"
#include <sstream>

using namespace asn1;
using namespace asn1::rrc;

/*******************************************************************************
 *                              Logging Utilities
 ******************************************************************************/

void asn1::rrc::log_invalid_access_choice_id(uint32_t val, uint32_t choice_id)
{
  asn1::log_error("The access choice id is invalid (%d!=%d)\n", val, choice_id);
}

void asn1::rrc::assert_choice_type(uint32_t val, uint32_t choice_id)
{
  if (val != choice_id) {
    log_invalid_access_choice_id(val, choice_id);
  }
}

void asn1::rrc::assert_choice_type(const std::string& access_type,
                                   const std::string& current_type,
                                   const std::string& choice_type)
{
  if (access_type != current_type) {
    asn1::log_error("Invalid field access for choice type \"%s\" (\"%s\"!=\"%s\")\n",
                    choice_type.c_str(),
                    access_type.c_str(),
                    current_type.c_str());
  }
}

const char*
asn1::rrc::convert_enum_idx(const char* array[], uint32_t nof_types, uint32_t enum_val, const char* enum_type)
{
  if (enum_val >= nof_types) {
    if (enum_val == nof_types) {
      asn1::log_error("The enum of type %s was not initialized.\n", enum_type);
    } else {
      asn1::log_error("The enum value=%d of type %s is not valid.\n", enum_val, enum_type);
    }
    return "";
  }
  return array[enum_val];
}

template <class ItemType>
ItemType asn1::rrc::map_enum_number(ItemType* array, uint32_t nof_types, uint32_t enum_val, const char* enum_type)
{
  if (enum_val >= nof_types) {
    if (enum_val == nof_types) {
      asn1::log_error("The enum of type %s is not initialized.\n", enum_type);
    } else {
      asn1::log_error("The enum value=%d of type %s cannot be converted to a number.\n", enum_val, enum_type);
    }
    return 0;
  }
  return array[enum_val];
}
template const uint8_t  asn1::rrc::map_enum_number<const uint8_t>(const uint8_t* array,
                                                                 uint32_t       nof_types,
                                                                 uint32_t       enum_val,
                                                                 const char*    enum_type);
template const uint16_t asn1::rrc::map_enum_number<const uint16_t>(const uint16_t* array,
                                                                   uint32_t        nof_types,
                                                                   uint32_t        enum_val,
                                                                   const char*     enum_type);
template const uint32_t asn1::rrc::map_enum_number<const uint32_t>(const uint32_t* array,
                                                                   uint32_t        nof_types,
                                                                   uint32_t        enum_val,
                                                                   const char*     enum_type);
template const uint64_t asn1::rrc::map_enum_number<const uint64_t>(const uint64_t* array,
                                                                   uint32_t        nof_types,
                                                                   uint32_t        enum_val,
                                                                   const char*     enum_type);
template const int8_t   asn1::rrc::map_enum_number<const int8_t>(const int8_t* array,
                                                               uint32_t      nof_types,
                                                               uint32_t      enum_val,
                                                               const char*   enum_type);
template const int16_t  asn1::rrc::map_enum_number<const int16_t>(const int16_t* array,
                                                                 uint32_t       nof_types,
                                                                 uint32_t       enum_val,
                                                                 const char*    enum_type);
template const int32_t  asn1::rrc::map_enum_number<const int32_t>(const int32_t* array,
                                                                 uint32_t       nof_types,
                                                                 uint32_t       enum_val,
                                                                 const char*    enum_type);
template const int64_t  asn1::rrc::map_enum_number<const int64_t>(const int64_t* array,
                                                                 uint32_t       nof_types,
                                                                 uint32_t       enum_val,
                                                                 const char*    enum_type);
template const float    asn1::rrc::map_enum_number<const float>(const float* array,
                                                             uint32_t     nof_types,
                                                             uint32_t     enum_val,
                                                             const char*  enum_type);

void rrc_asn1_warn_assert(bool cond, const char* filename, int lineno)
{
  if (cond) {
    asn1::log_warning("Assertion in [%s][%d] failed.\n", filename, lineno);
  }
}
static void log_invalid_choice_id(uint32_t val, const char* choice_type)
{
  asn1::log_error("Invalid choice id=%d for choice type %s\n", val, choice_type);
}

/*******************************************************************************
 *                                Struct Methods
 ******************************************************************************/

// PHICH-Config ::= SEQUENCE
SRSASN_CODE phich_cfg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(phich_dur.pack(bref));
  HANDLE_CODE(phich_res.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE phich_cfg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(phich_dur.unpack(bref));
  HANDLE_CODE(phich_res.unpack(bref));

  return SRSASN_SUCCESS;
}
void phich_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("phich-Duration", phich_dur.to_string());
  j.write_str("phich-Resource", phich_res.to_string());
  j.end_obj();
}

// MasterInformationBlock ::= SEQUENCE
SRSASN_CODE mib_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(dl_bw.pack(bref));
  HANDLE_CODE(phich_cfg.pack(bref));
  HANDLE_CODE(sys_frame_num.pack(bref));
  HANDLE_CODE(pack_integer(bref, sched_info_sib1_br_r13, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(bref.pack(sys_info_unchanged_br_r15, 1));
  HANDLE_CODE(spare.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mib_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(dl_bw.unpack(bref));
  HANDLE_CODE(phich_cfg.unpack(bref));
  HANDLE_CODE(sys_frame_num.unpack(bref));
  HANDLE_CODE(unpack_integer(sched_info_sib1_br_r13, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(bref.unpack(sys_info_unchanged_br_r15, 1));
  HANDLE_CODE(spare.unpack(bref));

  return SRSASN_SUCCESS;
}
void mib_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("dl-Bandwidth", dl_bw.to_string());
  j.write_fieldname("phich-Config");
  phich_cfg.to_json(j);
  j.write_str("systemFrameNumber", sys_frame_num.to_string());
  j.write_int("schedulingInfoSIB1-BR-r13", sched_info_sib1_br_r13);
  j.write_bool("systemInfoUnchanged-BR-r15", sys_info_unchanged_br_r15);
  j.write_str("spare", spare.to_string());
  j.end_obj();
}

// BCCH-BCH-Message ::= SEQUENCE
SRSASN_CODE bcch_bch_msg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(msg.pack(bref));

  bref.align_bytes_zero();

  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_bch_msg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(msg.unpack(bref));

  bref.align_bytes();

  return SRSASN_SUCCESS;
}
void bcch_bch_msg_s::to_json(json_writer& j) const
{
  j.start_array();
  j.start_obj();
  j.start_obj("BCCH-BCH-Message");
  j.write_fieldname("message");
  msg.to_json(j);
  j.end_obj();
  j.end_obj();
  j.end_array();
}

// MasterInformationBlock-MBMS-r14 ::= SEQUENCE
SRSASN_CODE mib_mbms_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(dl_bw_mbms_r14.pack(bref));
  HANDLE_CODE(sys_frame_num_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, add_non_mbsfn_sfs_r14, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(spare.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mib_mbms_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(dl_bw_mbms_r14.unpack(bref));
  HANDLE_CODE(sys_frame_num_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(add_non_mbsfn_sfs_r14, bref, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(spare.unpack(bref));

  return SRSASN_SUCCESS;
}
void mib_mbms_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("dl-Bandwidth-MBMS-r14", dl_bw_mbms_r14.to_string());
  j.write_str("systemFrameNumber-r14", sys_frame_num_r14.to_string());
  j.write_int("additionalNonMBSFNSubframes-r14", add_non_mbsfn_sfs_r14);
  j.write_str("spare", spare.to_string());
  j.end_obj();
}

// BCCH-BCH-Message-MBMS ::= SEQUENCE
SRSASN_CODE bcch_bch_msg_mbms_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(msg.pack(bref));

  bref.align_bytes_zero();

  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_bch_msg_mbms_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(msg.unpack(bref));

  bref.align_bytes();

  return SRSASN_SUCCESS;
}
void bcch_bch_msg_mbms_s::to_json(json_writer& j) const
{
  j.start_array();
  j.start_obj();
  j.start_obj("BCCH-BCH-Message-MBMS");
  j.write_fieldname("message");
  msg.to_json(j);
  j.end_obj();
  j.end_obj();
  j.end_array();
}

// GNSS-ID-r15 ::= SEQUENCE
SRSASN_CODE gnss_id_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(gnss_id_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE gnss_id_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(gnss_id_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void gnss_id_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("gnss-id-r15", gnss_id_r15.to_string());
  j.end_obj();
}

// SBAS-ID-r15 ::= SEQUENCE
SRSASN_CODE sbas_id_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(sbas_id_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sbas_id_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(sbas_id_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void sbas_id_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sbas-id-r15", sbas_id_r15.to_string());
  j.end_obj();
}

// PLMN-Identity ::= SEQUENCE
SRSASN_CODE plmn_id_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(mcc_present, 1));

  if (mcc_present) {
    HANDLE_CODE(pack_fixed_seq_of(bref, &(mcc)[0], mcc.size(), integer_packer<uint8_t>(0, 9)));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, mnc, 2, 3, integer_packer<uint8_t>(0, 9)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_id_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(mcc_present, 1));

  if (mcc_present) {
    HANDLE_CODE(unpack_fixed_seq_of(&(mcc)[0], bref, mcc.size(), integer_packer<uint8_t>(0, 9)));
  }
  HANDLE_CODE(unpack_dyn_seq_of(mnc, bref, 2, 3, integer_packer<uint8_t>(0, 9)));

  return SRSASN_SUCCESS;
}
void plmn_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (mcc_present) {
    j.start_array("mcc");
    for (const auto& e1 : mcc) {
      j.write_int(e1);
    }
    j.end_array();
  }
  j.start_array("mnc");
  for (const auto& e1 : mnc) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// PosSIB-Type-r15 ::= SEQUENCE
SRSASN_CODE pos_sib_type_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(encrypted_r15_present, 1));
  HANDLE_CODE(bref.pack(gnss_id_r15_present, 1));
  HANDLE_CODE(bref.pack(sbas_id_r15_present, 1));

  if (gnss_id_r15_present) {
    HANDLE_CODE(gnss_id_r15.pack(bref));
  }
  if (sbas_id_r15_present) {
    HANDLE_CODE(sbas_id_r15.pack(bref));
  }
  HANDLE_CODE(pos_sib_type_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pos_sib_type_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(encrypted_r15_present, 1));
  HANDLE_CODE(bref.unpack(gnss_id_r15_present, 1));
  HANDLE_CODE(bref.unpack(sbas_id_r15_present, 1));

  if (gnss_id_r15_present) {
    HANDLE_CODE(gnss_id_r15.unpack(bref));
  }
  if (sbas_id_r15_present) {
    HANDLE_CODE(sbas_id_r15.unpack(bref));
  }
  HANDLE_CODE(pos_sib_type_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void pos_sib_type_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (encrypted_r15_present) {
    j.write_str("encrypted-r15", "true");
  }
  if (gnss_id_r15_present) {
    j.write_fieldname("gnss-id-r15");
    gnss_id_r15.to_json(j);
  }
  if (sbas_id_r15_present) {
    j.write_fieldname("sbas-id-r15");
    sbas_id_r15.to_json(j);
  }
  j.write_str("posSibType-r15", pos_sib_type_r15.to_string());
  j.end_obj();
}

// PLMN-IdentityInfo-r15 ::= SEQUENCE
SRSASN_CODE plmn_id_info_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(plmn_id_minus5_gc_r15.pack(bref));
  HANDLE_CODE(cell_reserved_for_oper_r15.pack(bref));
  HANDLE_CODE(cell_reserved_for_oper_crs_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_id_info_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(plmn_id_minus5_gc_r15.unpack(bref));
  HANDLE_CODE(cell_reserved_for_oper_r15.unpack(bref));
  HANDLE_CODE(cell_reserved_for_oper_crs_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void plmn_id_info_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("plmn-Identity-5GC-r15");
  plmn_id_minus5_gc_r15.to_json(j);
  j.write_str("cellReservedForOperatorUse-r15", cell_reserved_for_oper_r15.to_string());
  j.write_str("cellReservedForOperatorUse-CRS-r15", cell_reserved_for_oper_crs_r15.to_string());
  j.end_obj();
}

void plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::destroy_()
{
  switch (type_) {
    case types::plmn_id_r15:
      c.destroy<plmn_id_s>();
      break;
    default:
      break;
  }
}
void plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::plmn_id_r15:
      c.init<plmn_id_s>();
      break;
    case types::plmn_idx_r15:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_");
  }
}
plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::plmn_id_minus5_gc_r15_c_(
    const plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::plmn_id_r15:
      c.init(other.c.get<plmn_id_s>());
      break;
    case types::plmn_idx_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_");
  }
}
plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_& plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::
                                              operator=(const plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::plmn_id_r15:
      c.set(other.c.get<plmn_id_s>());
      break;
    case types::plmn_idx_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_");
  }

  return *this;
}
void plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::plmn_id_r15:
      j.write_fieldname("plmn-Identity-r15");
      c.get<plmn_id_s>().to_json(j);
      break;
    case types::plmn_idx_r15:
      j.write_int("plmn-Index-r15", c.get<uint8_t>());
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::plmn_id_r15:
      HANDLE_CODE(c.get<plmn_id_s>().pack(bref));
      break;
    case types::plmn_idx_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)6u));
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::plmn_id_r15:
      HANDLE_CODE(c.get<plmn_id_s>().unpack(bref));
      break;
    case types::plmn_idx_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)6u));
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info_r15_s::plmn_id_minus5_gc_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CellIdentity-5GC-r15 ::= CHOICE
void cell_id_minus5_gc_r15_c::destroy_()
{
  switch (type_) {
    case types::cell_id_r15:
      c.destroy<fixed_bitstring<28> >();
      break;
    default:
      break;
  }
}
void cell_id_minus5_gc_r15_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_r15:
      c.init<fixed_bitstring<28> >();
      break;
    case types::cell_id_idx_r15:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_minus5_gc_r15_c");
  }
}
cell_id_minus5_gc_r15_c::cell_id_minus5_gc_r15_c(const cell_id_minus5_gc_r15_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_r15:
      c.init(other.c.get<fixed_bitstring<28> >());
      break;
    case types::cell_id_idx_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_minus5_gc_r15_c");
  }
}
cell_id_minus5_gc_r15_c& cell_id_minus5_gc_r15_c::operator=(const cell_id_minus5_gc_r15_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_r15:
      c.set(other.c.get<fixed_bitstring<28> >());
      break;
    case types::cell_id_idx_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_minus5_gc_r15_c");
  }

  return *this;
}
void cell_id_minus5_gc_r15_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_r15:
      j.write_str("cellIdentity-r15", c.get<fixed_bitstring<28> >().to_string());
      break;
    case types::cell_id_idx_r15:
      j.write_int("cellId-Index-r15", c.get<uint8_t>());
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_minus5_gc_r15_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_id_minus5_gc_r15_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_r15:
      HANDLE_CODE(c.get<fixed_bitstring<28> >().pack(bref));
      break;
    case types::cell_id_idx_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)6u));
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_minus5_gc_r15_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_minus5_gc_r15_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_r15:
      HANDLE_CODE(c.get<fixed_bitstring<28> >().unpack(bref));
      break;
    case types::cell_id_idx_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)6u));
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_minus5_gc_r15_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// PLMN-IdentityInfo-v1530 ::= SEQUENCE
SRSASN_CODE plmn_id_info_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(cell_reserved_for_oper_crs_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_id_info_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(cell_reserved_for_oper_crs_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void plmn_id_info_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cellReservedForOperatorUse-CRS-r15", cell_reserved_for_oper_crs_r15.to_string());
  j.end_obj();
}

// PosSchedulingInfo-r15 ::= SEQUENCE
SRSASN_CODE pos_sched_info_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pos_si_periodicity_r15.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, pos_sib_map_info_r15, 1, 32));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pos_sched_info_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(pos_si_periodicity_r15.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(pos_sib_map_info_r15, bref, 1, 32));

  return SRSASN_SUCCESS;
}
void pos_sched_info_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("posSI-Periodicity-r15", pos_si_periodicity_r15.to_string());
  j.start_array("posSIB-MappingInfo-r15");
  for (const auto& e1 : pos_sib_map_info_r15) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// CellAccessRelatedInfo-5GC-r15 ::= SEQUENCE
SRSASN_CODE cell_access_related_info_minus5_gc_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ran_area_code_r15_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, plmn_id_list_r15, 1, 6));
  if (ran_area_code_r15_present) {
    HANDLE_CODE(pack_integer(bref, ran_area_code_r15, (uint16_t)0u, (uint16_t)255u));
  }
  HANDLE_CODE(tac_minus5_gc_r15.pack(bref));
  HANDLE_CODE(cell_id_minus5_gc_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_access_related_info_minus5_gc_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ran_area_code_r15_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(plmn_id_list_r15, bref, 1, 6));
  if (ran_area_code_r15_present) {
    HANDLE_CODE(unpack_integer(ran_area_code_r15, bref, (uint16_t)0u, (uint16_t)255u));
  }
  HANDLE_CODE(tac_minus5_gc_r15.unpack(bref));
  HANDLE_CODE(cell_id_minus5_gc_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void cell_access_related_info_minus5_gc_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("plmn-IdentityList-r15");
  for (const auto& e1 : plmn_id_list_r15) {
    e1.to_json(j);
  }
  j.end_array();
  if (ran_area_code_r15_present) {
    j.write_int("ran-AreaCode-r15", ran_area_code_r15);
  }
  j.write_str("trackingAreaCode-5GC-r15", tac_minus5_gc_r15.to_string());
  j.write_fieldname("cellIdentity-5GC-r15");
  cell_id_minus5_gc_r15.to_json(j);
  j.end_obj();
}

// CellSelectionInfoCE-v1530 ::= SEQUENCE
SRSASN_CODE cell_sel_info_ce_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pwr_class14dbm_offset_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_ce_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(pwr_class14dbm_offset_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void cell_sel_info_ce_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("powerClass14dBm-Offset-r15", pwr_class14dbm_offset_r15.to_string());
  j.end_obj();
}

// MCS-PSSCH-Range-r15 ::= SEQUENCE
SRSASN_CODE mcs_pssch_range_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, min_mcs_pssch_r15, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, max_mcs_pssch_r15, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mcs_pssch_range_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(min_mcs_pssch_r15, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(max_mcs_pssch_r15, bref, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
void mcs_pssch_range_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("minMCS-PSSCH-r15", min_mcs_pssch_r15);
  j.write_int("maxMCS-PSSCH-r15", max_mcs_pssch_r15);
  j.end_obj();
}

// PLMN-IdentityInfo ::= SEQUENCE
SRSASN_CODE plmn_id_info_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(cell_reserved_for_oper.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_id_info_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(cell_reserved_for_oper.unpack(bref));

  return SRSASN_SUCCESS;
}
void plmn_id_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("plmn-Identity");
  plmn_id.to_json(j);
  j.write_str("cellReservedForOperatorUse", cell_reserved_for_oper.to_string());
  j.end_obj();
}

// SL-TxPower-r14 ::= CHOICE
void sl_tx_pwr_r14_c::set(types::options e)
{
  type_ = e;
}
void sl_tx_pwr_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::minusinfinity_r14:
      break;
    case types::tx_pwr_r14:
      j.write_int("txPower-r14", c);
      break;
    default:
      log_invalid_choice_id(type_, "sl_tx_pwr_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE sl_tx_pwr_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::minusinfinity_r14:
      break;
    case types::tx_pwr_r14:
      HANDLE_CODE(pack_integer(bref, c, (int8_t)-41, (int8_t)31));
      break;
    default:
      log_invalid_choice_id(type_, "sl_tx_pwr_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_tx_pwr_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::minusinfinity_r14:
      break;
    case types::tx_pwr_r14:
      HANDLE_CODE(unpack_integer(c, bref, (int8_t)-41, (int8_t)31));
      break;
    default:
      log_invalid_choice_id(type_, "sl_tx_pwr_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SL-MinT2Value-r15 ::= SEQUENCE
SRSASN_CODE sl_min_t2_value_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, prio_list_r15, 1, 8, integer_packer<uint8_t>(1, 8)));
  HANDLE_CODE(pack_integer(bref, min_t2_value_r15, (uint8_t)10u, (uint8_t)20u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_min_t2_value_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(prio_list_r15, bref, 1, 8, integer_packer<uint8_t>(1, 8)));
  HANDLE_CODE(unpack_integer(min_t2_value_r15, bref, (uint8_t)10u, (uint8_t)20u));

  return SRSASN_SUCCESS;
}
void sl_min_t2_value_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("priorityList-r15");
  for (const auto& e1 : prio_list_r15) {
    j.write_int(e1);
  }
  j.end_array();
  j.write_int("minT2Value-r15", min_t2_value_r15);
  j.end_obj();
}

// SL-PPPP-TxConfigIndex-r14 ::= SEQUENCE
SRSASN_CODE sl_pppp_tx_cfg_idx_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, prio_thres_r14, (uint8_t)1u, (uint8_t)8u));
  HANDLE_CODE(pack_integer(bref, default_tx_cfg_idx_r14, (uint8_t)0u, (uint8_t)15u));
  HANDLE_CODE(pack_integer(bref, cbr_cfg_idx_r14, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(pack_dyn_seq_of(bref, tx_cfg_idx_list_r14, 1, 16, integer_packer<uint8_t>(0, 63)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pppp_tx_cfg_idx_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(prio_thres_r14, bref, (uint8_t)1u, (uint8_t)8u));
  HANDLE_CODE(unpack_integer(default_tx_cfg_idx_r14, bref, (uint8_t)0u, (uint8_t)15u));
  HANDLE_CODE(unpack_integer(cbr_cfg_idx_r14, bref, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(unpack_dyn_seq_of(tx_cfg_idx_list_r14, bref, 1, 16, integer_packer<uint8_t>(0, 63)));

  return SRSASN_SUCCESS;
}
void sl_pppp_tx_cfg_idx_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("priorityThreshold-r14", prio_thres_r14);
  j.write_int("defaultTxConfigIndex-r14", default_tx_cfg_idx_r14);
  j.write_int("cbr-ConfigIndex-r14", cbr_cfg_idx_r14);
  j.start_array("tx-ConfigIndexList-r14");
  for (const auto& e1 : tx_cfg_idx_list_r14) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// SL-PPPP-TxConfigIndex-v1530 ::= SEQUENCE
SRSASN_CODE sl_pppp_tx_cfg_idx_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(mcs_pssch_range_list_r15_present, 1));

  if (mcs_pssch_range_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mcs_pssch_range_list_r15, 1, 16));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pppp_tx_cfg_idx_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(mcs_pssch_range_list_r15_present, 1));

  if (mcs_pssch_range_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mcs_pssch_range_list_r15, bref, 1, 16));
  }

  return SRSASN_SUCCESS;
}
void sl_pppp_tx_cfg_idx_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (mcs_pssch_range_list_r15_present) {
    j.start_array("mcs-PSSCH-RangeList-r15");
    for (const auto& e1 : mcs_pssch_range_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// SL-PSSCH-TxParameters-r14 ::= SEQUENCE
SRSASN_CODE sl_pssch_tx_params_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(max_tx_pwr_r14_present, 1));

  HANDLE_CODE(pack_integer(bref, min_mcs_pssch_r14, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, max_mcs_pssch_r14, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, min_sub_ch_num_pssch_r14, (uint8_t)1u, (uint8_t)20u));
  HANDLE_CODE(pack_integer(bref, max_subch_num_pssch_r14, (uint8_t)1u, (uint8_t)20u));
  HANDLE_CODE(allowed_retx_num_pssch_r14.pack(bref));
  if (max_tx_pwr_r14_present) {
    HANDLE_CODE(max_tx_pwr_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pssch_tx_params_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(max_tx_pwr_r14_present, 1));

  HANDLE_CODE(unpack_integer(min_mcs_pssch_r14, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(max_mcs_pssch_r14, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(min_sub_ch_num_pssch_r14, bref, (uint8_t)1u, (uint8_t)20u));
  HANDLE_CODE(unpack_integer(max_subch_num_pssch_r14, bref, (uint8_t)1u, (uint8_t)20u));
  HANDLE_CODE(allowed_retx_num_pssch_r14.unpack(bref));
  if (max_tx_pwr_r14_present) {
    HANDLE_CODE(max_tx_pwr_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_pssch_tx_params_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("minMCS-PSSCH-r14", min_mcs_pssch_r14);
  j.write_int("maxMCS-PSSCH-r14", max_mcs_pssch_r14);
  j.write_int("minSubChannel-NumberPSSCH-r14", min_sub_ch_num_pssch_r14);
  j.write_int("maxSubchannel-NumberPSSCH-r14", max_subch_num_pssch_r14);
  j.write_str("allowedRetxNumberPSSCH-r14", allowed_retx_num_pssch_r14.to_string());
  if (max_tx_pwr_r14_present) {
    j.write_fieldname("maxTxPower-r14");
    max_tx_pwr_r14.to_json(j);
  }
  j.end_obj();
}

// SL-PSSCH-TxParameters-v1530 ::= SEQUENCE
SRSASN_CODE sl_pssch_tx_params_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, min_mcs_pssch_r15, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, max_mcs_pssch_r15, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pssch_tx_params_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(min_mcs_pssch_r15, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(max_mcs_pssch_r15, bref, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
void sl_pssch_tx_params_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("minMCS-PSSCH-r15", min_mcs_pssch_r15);
  j.write_int("maxMCS-PSSCH-r15", max_mcs_pssch_r15);
  j.end_obj();
}

// SystemInformationBlockType1-v1530-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1530_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(hsdn_cell_r15_present, 1));
  HANDLE_CODE(bref.pack(cell_sel_info_ce_v1530_present, 1));
  HANDLE_CODE(bref.pack(crs_intf_mitig_cfg_r15_present, 1));
  HANDLE_CODE(bref.pack(plmn_id_list_v1530_present, 1));
  HANDLE_CODE(bref.pack(pos_sched_info_list_r15_present, 1));
  HANDLE_CODE(bref.pack(cell_access_related_info_minus5_gc_r15_present, 1));
  HANDLE_CODE(bref.pack(ims_emergency_support5_gc_r15_present, 1));
  HANDLE_CODE(bref.pack(ecall_over_ims_support5_gc_r15_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (cell_sel_info_ce_v1530_present) {
    HANDLE_CODE(cell_sel_info_ce_v1530.pack(bref));
  }
  if (crs_intf_mitig_cfg_r15_present) {
    HANDLE_CODE(crs_intf_mitig_cfg_r15.pack(bref));
  }
  HANDLE_CODE(cell_barred_crs_r15.pack(bref));
  if (plmn_id_list_v1530_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, plmn_id_list_v1530, 1, 6));
  }
  if (pos_sched_info_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, pos_sched_info_list_r15, 1, 32));
  }
  if (cell_access_related_info_minus5_gc_r15_present) {
    HANDLE_CODE(cell_access_related_info_minus5_gc_r15.cell_barred_minus5_gc_r15.pack(bref));
    HANDLE_CODE(cell_access_related_info_minus5_gc_r15.cell_barred_minus5_gc_crs_r15.pack(bref));
    HANDLE_CODE(pack_dyn_seq_of(
        bref, cell_access_related_info_minus5_gc_r15.cell_access_related_info_list_minus5_gc_r15, 1, 6));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1530_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(hsdn_cell_r15_present, 1));
  HANDLE_CODE(bref.unpack(cell_sel_info_ce_v1530_present, 1));
  HANDLE_CODE(bref.unpack(crs_intf_mitig_cfg_r15_present, 1));
  HANDLE_CODE(bref.unpack(plmn_id_list_v1530_present, 1));
  HANDLE_CODE(bref.unpack(pos_sched_info_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(cell_access_related_info_minus5_gc_r15_present, 1));
  HANDLE_CODE(bref.unpack(ims_emergency_support5_gc_r15_present, 1));
  HANDLE_CODE(bref.unpack(ecall_over_ims_support5_gc_r15_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (cell_sel_info_ce_v1530_present) {
    HANDLE_CODE(cell_sel_info_ce_v1530.unpack(bref));
  }
  if (crs_intf_mitig_cfg_r15_present) {
    HANDLE_CODE(crs_intf_mitig_cfg_r15.unpack(bref));
  }
  HANDLE_CODE(cell_barred_crs_r15.unpack(bref));
  if (plmn_id_list_v1530_present) {
    HANDLE_CODE(unpack_dyn_seq_of(plmn_id_list_v1530, bref, 1, 6));
  }
  if (pos_sched_info_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(pos_sched_info_list_r15, bref, 1, 32));
  }
  if (cell_access_related_info_minus5_gc_r15_present) {
    HANDLE_CODE(cell_access_related_info_minus5_gc_r15.cell_barred_minus5_gc_r15.unpack(bref));
    HANDLE_CODE(cell_access_related_info_minus5_gc_r15.cell_barred_minus5_gc_crs_r15.unpack(bref));
    HANDLE_CODE(unpack_dyn_seq_of(
        cell_access_related_info_minus5_gc_r15.cell_access_related_info_list_minus5_gc_r15, bref, 1, 6));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1530_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (hsdn_cell_r15_present) {
    j.write_str("hsdn-Cell-r15", "true");
  }
  if (cell_sel_info_ce_v1530_present) {
    j.write_fieldname("cellSelectionInfoCE-v1530");
    cell_sel_info_ce_v1530.to_json(j);
  }
  if (crs_intf_mitig_cfg_r15_present) {
    j.write_fieldname("crs-IntfMitigConfig-r15");
    crs_intf_mitig_cfg_r15.to_json(j);
  }
  j.write_str("cellBarred-CRS-r15", cell_barred_crs_r15.to_string());
  if (plmn_id_list_v1530_present) {
    j.start_array("plmn-IdentityList-v1530");
    for (const auto& e1 : plmn_id_list_v1530) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (pos_sched_info_list_r15_present) {
    j.start_array("posSchedulingInfoList-r15");
    for (const auto& e1 : pos_sched_info_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (cell_access_related_info_minus5_gc_r15_present) {
    j.write_fieldname("cellAccessRelatedInfo-5GC-r15");
    j.start_obj();
    j.write_str("cellBarred-5GC-r15", cell_access_related_info_minus5_gc_r15.cell_barred_minus5_gc_r15.to_string());
    j.write_str("cellBarred-5GC-CRS-r15",
                cell_access_related_info_minus5_gc_r15.cell_barred_minus5_gc_crs_r15.to_string());
    j.start_array("cellAccessRelatedInfoList-5GC-r15");
    for (const auto& e1 : cell_access_related_info_minus5_gc_r15.cell_access_related_info_list_minus5_gc_r15) {
      e1.to_json(j);
    }
    j.end_array();
    j.end_obj();
  }
  if (ims_emergency_support5_gc_r15_present) {
    j.write_str("ims-EmergencySupport5GC-r15", "true");
  }
  if (ecall_over_ims_support5_gc_r15_present) {
    j.write_str("eCallOverIMS-Support5GC-r15", "true");
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

void sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_::set(types::options e)
{
  type_ = e;
}
void sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::crs_intf_mitig_enabled_minus15:
      break;
    case types::crs_intf_mitig_num_prbs_r15:
      j.write_str("crs-IntfMitigNumPRBs-r15", c.to_string());
      break;
    default:
      log_invalid_choice_id(type_, "sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::crs_intf_mitig_enabled_minus15:
      break;
    case types::crs_intf_mitig_num_prbs_r15:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::crs_intf_mitig_enabled_minus15:
      break;
    case types::crs_intf_mitig_num_prbs_r15:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type1_v1530_ies_s::crs_intf_mitig_cfg_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// TDD-Config-v1450 ::= SEQUENCE
SRSASN_CODE tdd_cfg_v1450_s::pack(bit_ref& bref) const
{
  return SRSASN_SUCCESS;
}
SRSASN_CODE tdd_cfg_v1450_s::unpack(cbit_ref& bref)
{
  return SRSASN_SUCCESS;
}
void tdd_cfg_v1450_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("specialSubframePatterns-v1450", "ssp10-CRS-LessDwPTS");
  j.end_obj();
}

// CellAccessRelatedInfo-r14 ::= SEQUENCE
SRSASN_CODE cell_access_related_info_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, plmn_id_list_r14, 1, 6));
  HANDLE_CODE(tac_r14.pack(bref));
  HANDLE_CODE(cell_id_r14.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_access_related_info_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(plmn_id_list_r14, bref, 1, 6));
  HANDLE_CODE(tac_r14.unpack(bref));
  HANDLE_CODE(cell_id_r14.unpack(bref));

  return SRSASN_SUCCESS;
}
void cell_access_related_info_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("plmn-IdentityList-r14");
  for (const auto& e1 : plmn_id_list_r14) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_str("trackingAreaCode-r14", tac_r14.to_string());
  j.write_str("cellIdentity-r14", cell_id_r14.to_string());
  j.end_obj();
}

// CellSelectionInfoCE1-v1360 ::= SEQUENCE
SRSASN_CODE cell_sel_info_ce1_v1360_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, delta_rx_lev_min_ce1_v1360, (int8_t)-8, (int8_t)-1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_ce1_v1360_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(delta_rx_lev_min_ce1_v1360, bref, (int8_t)-8, (int8_t)-1));

  return SRSASN_SUCCESS;
}
void cell_sel_info_ce1_v1360_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("delta-RxLevMinCE1-v1360", delta_rx_lev_min_ce1_v1360);
  j.end_obj();
}

// NS-PmaxValue-v10l0 ::= SEQUENCE
SRSASN_CODE ns_pmax_value_v10l0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(add_spec_emission_v10l0_present, 1));

  if (add_spec_emission_v10l0_present) {
    HANDLE_CODE(pack_integer(bref, add_spec_emission_v10l0, (uint16_t)33u, (uint16_t)288u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ns_pmax_value_v10l0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(add_spec_emission_v10l0_present, 1));

  if (add_spec_emission_v10l0_present) {
    HANDLE_CODE(unpack_integer(add_spec_emission_v10l0, bref, (uint16_t)33u, (uint16_t)288u));
  }

  return SRSASN_SUCCESS;
}
void ns_pmax_value_v10l0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (add_spec_emission_v10l0_present) {
    j.write_int("additionalSpectrumEmission-v10l0", add_spec_emission_v10l0);
  }
  j.end_obj();
}

// SL-OffsetIndicator-r12 ::= CHOICE
void sl_offset_ind_r12_c::destroy_() {}
void sl_offset_ind_r12_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
sl_offset_ind_r12_c::sl_offset_ind_r12_c(const sl_offset_ind_r12_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::small_r12:
      c.init(other.c.get<uint16_t>());
      break;
    case types::large_r12:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sl_offset_ind_r12_c");
  }
}
sl_offset_ind_r12_c& sl_offset_ind_r12_c::operator=(const sl_offset_ind_r12_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::small_r12:
      c.set(other.c.get<uint16_t>());
      break;
    case types::large_r12:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sl_offset_ind_r12_c");
  }

  return *this;
}
void sl_offset_ind_r12_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::small_r12:
      j.write_int("small-r12", c.get<uint16_t>());
      break;
    case types::large_r12:
      j.write_int("large-r12", c.get<uint16_t>());
      break;
    default:
      log_invalid_choice_id(type_, "sl_offset_ind_r12_c");
  }
  j.end_obj();
}
SRSASN_CODE sl_offset_ind_r12_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::small_r12:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)319u));
      break;
    case types::large_r12:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)10239u));
      break;
    default:
      log_invalid_choice_id(type_, "sl_offset_ind_r12_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_offset_ind_r12_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::small_r12:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)319u));
      break;
    case types::large_r12:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)10239u));
      break;
    default:
      log_invalid_choice_id(type_, "sl_offset_ind_r12_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SL-P2X-ResourceSelectionConfig-r14 ::= SEQUENCE
SRSASN_CODE sl_p2_x_res_sel_cfg_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(partial_sensing_r14_present, 1));
  HANDLE_CODE(bref.pack(random_sel_r14_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_p2_x_res_sel_cfg_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(partial_sensing_r14_present, 1));
  HANDLE_CODE(bref.unpack(random_sel_r14_present, 1));

  return SRSASN_SUCCESS;
}
void sl_p2_x_res_sel_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (partial_sensing_r14_present) {
    j.write_str("partialSensing-r14", "true");
  }
  if (random_sel_r14_present) {
    j.write_str("randomSelection-r14", "true");
  }
  j.end_obj();
}

// SL-PSSCH-TxConfig-r14 ::= SEQUENCE
SRSASN_CODE sl_pssch_tx_cfg_r14_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(type_tx_sync_r14_present, 1));

  if (type_tx_sync_r14_present) {
    HANDLE_CODE(type_tx_sync_r14.pack(bref));
  }
  HANDLE_CODE(thres_ue_speed_r14.pack(bref));
  HANDLE_CODE(params_above_thres_r14.pack(bref));
  HANDLE_CODE(params_below_thres_r14.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= params_above_thres_v1530.is_present();
    group_flags[0] |= params_below_thres_v1530.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(params_above_thres_v1530.is_present(), 1));
      HANDLE_CODE(bref.pack(params_below_thres_v1530.is_present(), 1));
      if (params_above_thres_v1530.is_present()) {
        HANDLE_CODE(params_above_thres_v1530->pack(bref));
      }
      if (params_below_thres_v1530.is_present()) {
        HANDLE_CODE(params_below_thres_v1530->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pssch_tx_cfg_r14_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(type_tx_sync_r14_present, 1));

  if (type_tx_sync_r14_present) {
    HANDLE_CODE(type_tx_sync_r14.unpack(bref));
  }
  HANDLE_CODE(thres_ue_speed_r14.unpack(bref));
  HANDLE_CODE(params_above_thres_r14.unpack(bref));
  HANDLE_CODE(params_below_thres_r14.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool params_above_thres_v1530_present;
      HANDLE_CODE(bref.unpack(params_above_thres_v1530_present, 1));
      params_above_thres_v1530.set_present(params_above_thres_v1530_present);
      bool params_below_thres_v1530_present;
      HANDLE_CODE(bref.unpack(params_below_thres_v1530_present, 1));
      params_below_thres_v1530.set_present(params_below_thres_v1530_present);
      if (params_above_thres_v1530.is_present()) {
        HANDLE_CODE(params_above_thres_v1530->unpack(bref));
      }
      if (params_below_thres_v1530.is_present()) {
        HANDLE_CODE(params_below_thres_v1530->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sl_pssch_tx_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (type_tx_sync_r14_present) {
    j.write_str("typeTxSync-r14", type_tx_sync_r14.to_string());
  }
  j.write_str("thresUE-Speed-r14", thres_ue_speed_r14.to_string());
  j.write_fieldname("parametersAboveThres-r14");
  params_above_thres_r14.to_json(j);
  j.write_fieldname("parametersBelowThres-r14");
  params_below_thres_r14.to_json(j);
  if (ext) {
    if (params_above_thres_v1530.is_present()) {
      j.write_fieldname("parametersAboveThres-v1530");
      params_above_thres_v1530->to_json(j);
    }
    if (params_below_thres_v1530.is_present()) {
      j.write_fieldname("parametersBelowThres-v1530");
      params_below_thres_v1530->to_json(j);
    }
  }
  j.end_obj();
}

// SL-SyncAllowed-r14 ::= SEQUENCE
SRSASN_CODE sl_sync_allowed_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(gnss_sync_r14_present, 1));
  HANDLE_CODE(bref.pack(enb_sync_r14_present, 1));
  HANDLE_CODE(bref.pack(ue_sync_r14_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_sync_allowed_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(gnss_sync_r14_present, 1));
  HANDLE_CODE(bref.unpack(enb_sync_r14_present, 1));
  HANDLE_CODE(bref.unpack(ue_sync_r14_present, 1));

  return SRSASN_SUCCESS;
}
void sl_sync_allowed_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (gnss_sync_r14_present) {
    j.write_str("gnss-Sync-r14", "true");
  }
  if (enb_sync_r14_present) {
    j.write_str("enb-Sync-r14", "true");
  }
  if (ue_sync_r14_present) {
    j.write_str("ue-Sync-r14", "true");
  }
  j.end_obj();
}

// SL-TxParameters-r12 ::= SEQUENCE
SRSASN_CODE sl_tx_params_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(alpha_r12.pack(bref));
  HANDLE_CODE(pack_integer(bref, p0_r12, (int8_t)-126, (int8_t)31));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_tx_params_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(alpha_r12.unpack(bref));
  HANDLE_CODE(unpack_integer(p0_r12, bref, (int8_t)-126, (int8_t)31));

  return SRSASN_SUCCESS;
}
void sl_tx_params_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("alpha-r12", alpha_r12.to_string());
  j.write_int("p0-r12", p0_r12);
  j.end_obj();
}

// SubframeBitmapSL-r12 ::= CHOICE
void sf_bitmap_sl_r12_c::destroy_()
{
  switch (type_) {
    case types::bs4_r12:
      c.destroy<fixed_bitstring<4> >();
      break;
    case types::bs8_r12:
      c.destroy<fixed_bitstring<8> >();
      break;
    case types::bs12_r12:
      c.destroy<fixed_bitstring<12> >();
      break;
    case types::bs16_r12:
      c.destroy<fixed_bitstring<16> >();
      break;
    case types::bs30_r12:
      c.destroy<fixed_bitstring<30> >();
      break;
    case types::bs40_r12:
      c.destroy<fixed_bitstring<40> >();
      break;
    case types::bs42_r12:
      c.destroy<fixed_bitstring<42> >();
      break;
    default:
      break;
  }
}
void sf_bitmap_sl_r12_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::bs4_r12:
      c.init<fixed_bitstring<4> >();
      break;
    case types::bs8_r12:
      c.init<fixed_bitstring<8> >();
      break;
    case types::bs12_r12:
      c.init<fixed_bitstring<12> >();
      break;
    case types::bs16_r12:
      c.init<fixed_bitstring<16> >();
      break;
    case types::bs30_r12:
      c.init<fixed_bitstring<30> >();
      break;
    case types::bs40_r12:
      c.init<fixed_bitstring<40> >();
      break;
    case types::bs42_r12:
      c.init<fixed_bitstring<42> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r12_c");
  }
}
sf_bitmap_sl_r12_c::sf_bitmap_sl_r12_c(const sf_bitmap_sl_r12_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::bs4_r12:
      c.init(other.c.get<fixed_bitstring<4> >());
      break;
    case types::bs8_r12:
      c.init(other.c.get<fixed_bitstring<8> >());
      break;
    case types::bs12_r12:
      c.init(other.c.get<fixed_bitstring<12> >());
      break;
    case types::bs16_r12:
      c.init(other.c.get<fixed_bitstring<16> >());
      break;
    case types::bs30_r12:
      c.init(other.c.get<fixed_bitstring<30> >());
      break;
    case types::bs40_r12:
      c.init(other.c.get<fixed_bitstring<40> >());
      break;
    case types::bs42_r12:
      c.init(other.c.get<fixed_bitstring<42> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r12_c");
  }
}
sf_bitmap_sl_r12_c& sf_bitmap_sl_r12_c::operator=(const sf_bitmap_sl_r12_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::bs4_r12:
      c.set(other.c.get<fixed_bitstring<4> >());
      break;
    case types::bs8_r12:
      c.set(other.c.get<fixed_bitstring<8> >());
      break;
    case types::bs12_r12:
      c.set(other.c.get<fixed_bitstring<12> >());
      break;
    case types::bs16_r12:
      c.set(other.c.get<fixed_bitstring<16> >());
      break;
    case types::bs30_r12:
      c.set(other.c.get<fixed_bitstring<30> >());
      break;
    case types::bs40_r12:
      c.set(other.c.get<fixed_bitstring<40> >());
      break;
    case types::bs42_r12:
      c.set(other.c.get<fixed_bitstring<42> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r12_c");
  }

  return *this;
}
void sf_bitmap_sl_r12_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::bs4_r12:
      j.write_str("bs4-r12", c.get<fixed_bitstring<4> >().to_string());
      break;
    case types::bs8_r12:
      j.write_str("bs8-r12", c.get<fixed_bitstring<8> >().to_string());
      break;
    case types::bs12_r12:
      j.write_str("bs12-r12", c.get<fixed_bitstring<12> >().to_string());
      break;
    case types::bs16_r12:
      j.write_str("bs16-r12", c.get<fixed_bitstring<16> >().to_string());
      break;
    case types::bs30_r12:
      j.write_str("bs30-r12", c.get<fixed_bitstring<30> >().to_string());
      break;
    case types::bs40_r12:
      j.write_str("bs40-r12", c.get<fixed_bitstring<40> >().to_string());
      break;
    case types::bs42_r12:
      j.write_str("bs42-r12", c.get<fixed_bitstring<42> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r12_c");
  }
  j.end_obj();
}
SRSASN_CODE sf_bitmap_sl_r12_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::bs4_r12:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().pack(bref));
      break;
    case types::bs8_r12:
      HANDLE_CODE(c.get<fixed_bitstring<8> >().pack(bref));
      break;
    case types::bs12_r12:
      HANDLE_CODE(c.get<fixed_bitstring<12> >().pack(bref));
      break;
    case types::bs16_r12:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().pack(bref));
      break;
    case types::bs30_r12:
      HANDLE_CODE(c.get<fixed_bitstring<30> >().pack(bref));
      break;
    case types::bs40_r12:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().pack(bref));
      break;
    case types::bs42_r12:
      HANDLE_CODE(c.get<fixed_bitstring<42> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r12_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sf_bitmap_sl_r12_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::bs4_r12:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().unpack(bref));
      break;
    case types::bs8_r12:
      HANDLE_CODE(c.get<fixed_bitstring<8> >().unpack(bref));
      break;
    case types::bs12_r12:
      HANDLE_CODE(c.get<fixed_bitstring<12> >().unpack(bref));
      break;
    case types::bs16_r12:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().unpack(bref));
      break;
    case types::bs30_r12:
      HANDLE_CODE(c.get<fixed_bitstring<30> >().unpack(bref));
      break;
    case types::bs40_r12:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().unpack(bref));
      break;
    case types::bs42_r12:
      HANDLE_CODE(c.get<fixed_bitstring<42> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r12_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SubframeBitmapSL-r14 ::= CHOICE
void sf_bitmap_sl_r14_c::destroy_()
{
  switch (type_) {
    case types::bs10_r14:
      c.destroy<fixed_bitstring<10> >();
      break;
    case types::bs16_r14:
      c.destroy<fixed_bitstring<16> >();
      break;
    case types::bs20_r14:
      c.destroy<fixed_bitstring<20> >();
      break;
    case types::bs30_r14:
      c.destroy<fixed_bitstring<30> >();
      break;
    case types::bs40_r14:
      c.destroy<fixed_bitstring<40> >();
      break;
    case types::bs50_r14:
      c.destroy<fixed_bitstring<50> >();
      break;
    case types::bs60_r14:
      c.destroy<fixed_bitstring<60> >();
      break;
    case types::bs100_r14:
      c.destroy<fixed_bitstring<100> >();
      break;
    default:
      break;
  }
}
void sf_bitmap_sl_r14_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::bs10_r14:
      c.init<fixed_bitstring<10> >();
      break;
    case types::bs16_r14:
      c.init<fixed_bitstring<16> >();
      break;
    case types::bs20_r14:
      c.init<fixed_bitstring<20> >();
      break;
    case types::bs30_r14:
      c.init<fixed_bitstring<30> >();
      break;
    case types::bs40_r14:
      c.init<fixed_bitstring<40> >();
      break;
    case types::bs50_r14:
      c.init<fixed_bitstring<50> >();
      break;
    case types::bs60_r14:
      c.init<fixed_bitstring<60> >();
      break;
    case types::bs100_r14:
      c.init<fixed_bitstring<100> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r14_c");
  }
}
sf_bitmap_sl_r14_c::sf_bitmap_sl_r14_c(const sf_bitmap_sl_r14_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::bs10_r14:
      c.init(other.c.get<fixed_bitstring<10> >());
      break;
    case types::bs16_r14:
      c.init(other.c.get<fixed_bitstring<16> >());
      break;
    case types::bs20_r14:
      c.init(other.c.get<fixed_bitstring<20> >());
      break;
    case types::bs30_r14:
      c.init(other.c.get<fixed_bitstring<30> >());
      break;
    case types::bs40_r14:
      c.init(other.c.get<fixed_bitstring<40> >());
      break;
    case types::bs50_r14:
      c.init(other.c.get<fixed_bitstring<50> >());
      break;
    case types::bs60_r14:
      c.init(other.c.get<fixed_bitstring<60> >());
      break;
    case types::bs100_r14:
      c.init(other.c.get<fixed_bitstring<100> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r14_c");
  }
}
sf_bitmap_sl_r14_c& sf_bitmap_sl_r14_c::operator=(const sf_bitmap_sl_r14_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::bs10_r14:
      c.set(other.c.get<fixed_bitstring<10> >());
      break;
    case types::bs16_r14:
      c.set(other.c.get<fixed_bitstring<16> >());
      break;
    case types::bs20_r14:
      c.set(other.c.get<fixed_bitstring<20> >());
      break;
    case types::bs30_r14:
      c.set(other.c.get<fixed_bitstring<30> >());
      break;
    case types::bs40_r14:
      c.set(other.c.get<fixed_bitstring<40> >());
      break;
    case types::bs50_r14:
      c.set(other.c.get<fixed_bitstring<50> >());
      break;
    case types::bs60_r14:
      c.set(other.c.get<fixed_bitstring<60> >());
      break;
    case types::bs100_r14:
      c.set(other.c.get<fixed_bitstring<100> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r14_c");
  }

  return *this;
}
void sf_bitmap_sl_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::bs10_r14:
      j.write_str("bs10-r14", c.get<fixed_bitstring<10> >().to_string());
      break;
    case types::bs16_r14:
      j.write_str("bs16-r14", c.get<fixed_bitstring<16> >().to_string());
      break;
    case types::bs20_r14:
      j.write_str("bs20-r14", c.get<fixed_bitstring<20> >().to_string());
      break;
    case types::bs30_r14:
      j.write_str("bs30-r14", c.get<fixed_bitstring<30> >().to_string());
      break;
    case types::bs40_r14:
      j.write_str("bs40-r14", c.get<fixed_bitstring<40> >().to_string());
      break;
    case types::bs50_r14:
      j.write_str("bs50-r14", c.get<fixed_bitstring<50> >().to_string());
      break;
    case types::bs60_r14:
      j.write_str("bs60-r14", c.get<fixed_bitstring<60> >().to_string());
      break;
    case types::bs100_r14:
      j.write_str("bs100-r14", c.get<fixed_bitstring<100> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE sf_bitmap_sl_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::bs10_r14:
      HANDLE_CODE(c.get<fixed_bitstring<10> >().pack(bref));
      break;
    case types::bs16_r14:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().pack(bref));
      break;
    case types::bs20_r14:
      HANDLE_CODE(c.get<fixed_bitstring<20> >().pack(bref));
      break;
    case types::bs30_r14:
      HANDLE_CODE(c.get<fixed_bitstring<30> >().pack(bref));
      break;
    case types::bs40_r14:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().pack(bref));
      break;
    case types::bs50_r14:
      HANDLE_CODE(c.get<fixed_bitstring<50> >().pack(bref));
      break;
    case types::bs60_r14:
      HANDLE_CODE(c.get<fixed_bitstring<60> >().pack(bref));
      break;
    case types::bs100_r14:
      HANDLE_CODE(c.get<fixed_bitstring<100> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sf_bitmap_sl_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::bs10_r14:
      HANDLE_CODE(c.get<fixed_bitstring<10> >().unpack(bref));
      break;
    case types::bs16_r14:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().unpack(bref));
      break;
    case types::bs20_r14:
      HANDLE_CODE(c.get<fixed_bitstring<20> >().unpack(bref));
      break;
    case types::bs30_r14:
      HANDLE_CODE(c.get<fixed_bitstring<30> >().unpack(bref));
      break;
    case types::bs40_r14:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().unpack(bref));
      break;
    case types::bs50_r14:
      HANDLE_CODE(c.get<fixed_bitstring<50> >().unpack(bref));
      break;
    case types::bs60_r14:
      HANDLE_CODE(c.get<fixed_bitstring<60> >().unpack(bref));
      break;
    case types::bs100_r14:
      HANDLE_CODE(c.get<fixed_bitstring<100> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sf_bitmap_sl_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType1-v1450-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1450_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(tdd_cfg_v1450_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (tdd_cfg_v1450_present) {
    HANDLE_CODE(tdd_cfg_v1450.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1450_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(tdd_cfg_v1450_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (tdd_cfg_v1450_present) {
    HANDLE_CODE(tdd_cfg_v1450.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1450_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (tdd_cfg_v1450_present) {
    j.write_fieldname("tdd-Config-v1450");
    tdd_cfg_v1450.to_json(j);
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// TDD-Config ::= SEQUENCE
SRSASN_CODE tdd_cfg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sf_assign.pack(bref));
  HANDLE_CODE(special_sf_patterns.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE tdd_cfg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sf_assign.unpack(bref));
  HANDLE_CODE(special_sf_patterns.unpack(bref));

  return SRSASN_SUCCESS;
}
void tdd_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("subframeAssignment", sf_assign.to_string());
  j.write_str("specialSubframePatterns", special_sf_patterns.to_string());
  j.end_obj();
}

// TDD-Config-v1430 ::= SEQUENCE
SRSASN_CODE tdd_cfg_v1430_s::pack(bit_ref& bref) const
{
  return SRSASN_SUCCESS;
}
SRSASN_CODE tdd_cfg_v1430_s::unpack(cbit_ref& bref)
{
  return SRSASN_SUCCESS;
}
void tdd_cfg_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("specialSubframePatterns-v1430", "ssp10");
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v1360 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v1360_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cell_sel_info_ce1_v1360_present, 1));

  if (cell_sel_info_ce1_v1360_present) {
    HANDLE_CODE(cell_sel_info_ce1_v1360.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v1360_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cell_sel_info_ce1_v1360_present, 1));

  if (cell_sel_info_ce1_v1360_present) {
    HANDLE_CODE(cell_sel_info_ce1_v1360.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v1360_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cell_sel_info_ce1_v1360_present) {
    j.write_fieldname("cellSelectionInfoCE1-v1360");
    cell_sel_info_ce1_v1360.to_json(j);
  }
  j.end_obj();
}

// NS-PmaxValue-r10 ::= SEQUENCE
SRSASN_CODE ns_pmax_value_r10_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(add_pmax_r10_present, 1));

  if (add_pmax_r10_present) {
    HANDLE_CODE(pack_integer(bref, add_pmax_r10, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(pack_integer(bref, add_spec_emission, (uint8_t)1u, (uint8_t)32u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ns_pmax_value_r10_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(add_pmax_r10_present, 1));

  if (add_pmax_r10_present) {
    HANDLE_CODE(unpack_integer(add_pmax_r10, bref, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(unpack_integer(add_spec_emission, bref, (uint8_t)1u, (uint8_t)32u));

  return SRSASN_SUCCESS;
}
void ns_pmax_value_r10_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (add_pmax_r10_present) {
    j.write_int("additionalPmax-r10", add_pmax_r10);
  }
  j.write_int("additionalSpectrumEmission", add_spec_emission);
  j.end_obj();
}

// SL-CommResourcePoolV2X-r14 ::= SEQUENCE
SRSASN_CODE sl_comm_res_pool_v2x_r14_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(sl_offset_ind_r14_present, 1));
  HANDLE_CODE(bref.pack(start_rb_pscch_pool_r14_present, 1));
  HANDLE_CODE(bref.pack(rx_params_ncell_r14_present, 1));
  HANDLE_CODE(bref.pack(data_tx_params_r14_present, 1));
  HANDLE_CODE(bref.pack(zone_id_r14_present, 1));
  HANDLE_CODE(bref.pack(thresh_s_rssi_cbr_r14_present, 1));
  HANDLE_CODE(bref.pack(pool_report_id_r14_present, 1));
  HANDLE_CODE(bref.pack(cbr_pssch_tx_cfg_list_r14_present, 1));
  HANDLE_CODE(bref.pack(res_sel_cfg_p2_x_r14_present, 1));
  HANDLE_CODE(bref.pack(sync_allowed_r14_present, 1));
  HANDLE_CODE(bref.pack(restrict_res_reserv_period_r14_present, 1));

  if (sl_offset_ind_r14_present) {
    HANDLE_CODE(sl_offset_ind_r14.pack(bref));
  }
  HANDLE_CODE(sl_sf_r14.pack(bref));
  HANDLE_CODE(bref.pack(adjacency_pscch_pssch_r14, 1));
  HANDLE_CODE(size_subch_r14.pack(bref));
  HANDLE_CODE(num_subch_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, start_rb_subch_r14, (uint8_t)0u, (uint8_t)99u));
  if (start_rb_pscch_pool_r14_present) {
    HANDLE_CODE(pack_integer(bref, start_rb_pscch_pool_r14, (uint8_t)0u, (uint8_t)99u));
  }
  if (rx_params_ncell_r14_present) {
    HANDLE_CODE(bref.pack(rx_params_ncell_r14.tdd_cfg_r14_present, 1));
    if (rx_params_ncell_r14.tdd_cfg_r14_present) {
      HANDLE_CODE(rx_params_ncell_r14.tdd_cfg_r14.pack(bref));
    }
    HANDLE_CODE(pack_integer(bref, rx_params_ncell_r14.sync_cfg_idx_r14, (uint8_t)0u, (uint8_t)15u));
  }
  if (data_tx_params_r14_present) {
    HANDLE_CODE(data_tx_params_r14.pack(bref));
  }
  if (zone_id_r14_present) {
    HANDLE_CODE(pack_integer(bref, zone_id_r14, (uint8_t)0u, (uint8_t)7u));
  }
  if (thresh_s_rssi_cbr_r14_present) {
    HANDLE_CODE(pack_integer(bref, thresh_s_rssi_cbr_r14, (uint8_t)0u, (uint8_t)45u));
  }
  if (pool_report_id_r14_present) {
    HANDLE_CODE(pack_integer(bref, pool_report_id_r14, (uint8_t)1u, (uint8_t)72u));
  }
  if (cbr_pssch_tx_cfg_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, cbr_pssch_tx_cfg_list_r14, 1, 8));
  }
  if (res_sel_cfg_p2_x_r14_present) {
    HANDLE_CODE(res_sel_cfg_p2_x_r14.pack(bref));
  }
  if (sync_allowed_r14_present) {
    HANDLE_CODE(sync_allowed_r14.pack(bref));
  }
  if (restrict_res_reserv_period_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, restrict_res_reserv_period_r14, 1, 16));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= sl_min_t2_value_list_r15.is_present();
    group_flags[0] |= cbr_pssch_tx_cfg_list_v1530.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sl_min_t2_value_list_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(cbr_pssch_tx_cfg_list_v1530.is_present(), 1));
      if (sl_min_t2_value_list_r15.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *sl_min_t2_value_list_r15, 1, 8));
      }
      if (cbr_pssch_tx_cfg_list_v1530.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *cbr_pssch_tx_cfg_list_v1530, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_comm_res_pool_v2x_r14_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(sl_offset_ind_r14_present, 1));
  HANDLE_CODE(bref.unpack(start_rb_pscch_pool_r14_present, 1));
  HANDLE_CODE(bref.unpack(rx_params_ncell_r14_present, 1));
  HANDLE_CODE(bref.unpack(data_tx_params_r14_present, 1));
  HANDLE_CODE(bref.unpack(zone_id_r14_present, 1));
  HANDLE_CODE(bref.unpack(thresh_s_rssi_cbr_r14_present, 1));
  HANDLE_CODE(bref.unpack(pool_report_id_r14_present, 1));
  HANDLE_CODE(bref.unpack(cbr_pssch_tx_cfg_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(res_sel_cfg_p2_x_r14_present, 1));
  HANDLE_CODE(bref.unpack(sync_allowed_r14_present, 1));
  HANDLE_CODE(bref.unpack(restrict_res_reserv_period_r14_present, 1));

  if (sl_offset_ind_r14_present) {
    HANDLE_CODE(sl_offset_ind_r14.unpack(bref));
  }
  HANDLE_CODE(sl_sf_r14.unpack(bref));
  HANDLE_CODE(bref.unpack(adjacency_pscch_pssch_r14, 1));
  HANDLE_CODE(size_subch_r14.unpack(bref));
  HANDLE_CODE(num_subch_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(start_rb_subch_r14, bref, (uint8_t)0u, (uint8_t)99u));
  if (start_rb_pscch_pool_r14_present) {
    HANDLE_CODE(unpack_integer(start_rb_pscch_pool_r14, bref, (uint8_t)0u, (uint8_t)99u));
  }
  if (rx_params_ncell_r14_present) {
    HANDLE_CODE(bref.unpack(rx_params_ncell_r14.tdd_cfg_r14_present, 1));
    if (rx_params_ncell_r14.tdd_cfg_r14_present) {
      HANDLE_CODE(rx_params_ncell_r14.tdd_cfg_r14.unpack(bref));
    }
    HANDLE_CODE(unpack_integer(rx_params_ncell_r14.sync_cfg_idx_r14, bref, (uint8_t)0u, (uint8_t)15u));
  }
  if (data_tx_params_r14_present) {
    HANDLE_CODE(data_tx_params_r14.unpack(bref));
  }
  if (zone_id_r14_present) {
    HANDLE_CODE(unpack_integer(zone_id_r14, bref, (uint8_t)0u, (uint8_t)7u));
  }
  if (thresh_s_rssi_cbr_r14_present) {
    HANDLE_CODE(unpack_integer(thresh_s_rssi_cbr_r14, bref, (uint8_t)0u, (uint8_t)45u));
  }
  if (pool_report_id_r14_present) {
    HANDLE_CODE(unpack_integer(pool_report_id_r14, bref, (uint8_t)1u, (uint8_t)72u));
  }
  if (cbr_pssch_tx_cfg_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(cbr_pssch_tx_cfg_list_r14, bref, 1, 8));
  }
  if (res_sel_cfg_p2_x_r14_present) {
    HANDLE_CODE(res_sel_cfg_p2_x_r14.unpack(bref));
  }
  if (sync_allowed_r14_present) {
    HANDLE_CODE(sync_allowed_r14.unpack(bref));
  }
  if (restrict_res_reserv_period_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(restrict_res_reserv_period_r14, bref, 1, 16));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool sl_min_t2_value_list_r15_present;
      HANDLE_CODE(bref.unpack(sl_min_t2_value_list_r15_present, 1));
      sl_min_t2_value_list_r15.set_present(sl_min_t2_value_list_r15_present);
      bool cbr_pssch_tx_cfg_list_v1530_present;
      HANDLE_CODE(bref.unpack(cbr_pssch_tx_cfg_list_v1530_present, 1));
      cbr_pssch_tx_cfg_list_v1530.set_present(cbr_pssch_tx_cfg_list_v1530_present);
      if (sl_min_t2_value_list_r15.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*sl_min_t2_value_list_r15, bref, 1, 8));
      }
      if (cbr_pssch_tx_cfg_list_v1530.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*cbr_pssch_tx_cfg_list_v1530, bref, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sl_comm_res_pool_v2x_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (sl_offset_ind_r14_present) {
    j.write_fieldname("sl-OffsetIndicator-r14");
    sl_offset_ind_r14.to_json(j);
  }
  j.write_fieldname("sl-Subframe-r14");
  sl_sf_r14.to_json(j);
  j.write_bool("adjacencyPSCCH-PSSCH-r14", adjacency_pscch_pssch_r14);
  j.write_str("sizeSubchannel-r14", size_subch_r14.to_string());
  j.write_str("numSubchannel-r14", num_subch_r14.to_string());
  j.write_int("startRB-Subchannel-r14", start_rb_subch_r14);
  if (start_rb_pscch_pool_r14_present) {
    j.write_int("startRB-PSCCH-Pool-r14", start_rb_pscch_pool_r14);
  }
  if (rx_params_ncell_r14_present) {
    j.write_fieldname("rxParametersNCell-r14");
    j.start_obj();
    if (rx_params_ncell_r14.tdd_cfg_r14_present) {
      j.write_fieldname("tdd-Config-r14");
      rx_params_ncell_r14.tdd_cfg_r14.to_json(j);
    }
    j.write_int("syncConfigIndex-r14", rx_params_ncell_r14.sync_cfg_idx_r14);
    j.end_obj();
  }
  if (data_tx_params_r14_present) {
    j.write_fieldname("dataTxParameters-r14");
    data_tx_params_r14.to_json(j);
  }
  if (zone_id_r14_present) {
    j.write_int("zoneID-r14", zone_id_r14);
  }
  if (thresh_s_rssi_cbr_r14_present) {
    j.write_int("threshS-RSSI-CBR-r14", thresh_s_rssi_cbr_r14);
  }
  if (pool_report_id_r14_present) {
    j.write_int("poolReportId-r14", pool_report_id_r14);
  }
  if (cbr_pssch_tx_cfg_list_r14_present) {
    j.start_array("cbr-pssch-TxConfigList-r14");
    for (const auto& e1 : cbr_pssch_tx_cfg_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (res_sel_cfg_p2_x_r14_present) {
    j.write_fieldname("resourceSelectionConfigP2X-r14");
    res_sel_cfg_p2_x_r14.to_json(j);
  }
  if (sync_allowed_r14_present) {
    j.write_fieldname("syncAllowed-r14");
    sync_allowed_r14.to_json(j);
  }
  if (restrict_res_reserv_period_r14_present) {
    j.start_array("restrictResourceReservationPeriod-r14");
    for (const auto& e1 : restrict_res_reserv_period_r14) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (ext) {
    if (sl_min_t2_value_list_r15.is_present()) {
      j.start_array("sl-MinT2ValueList-r15");
      for (const auto& e1 : *sl_min_t2_value_list_r15) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (cbr_pssch_tx_cfg_list_v1530.is_present()) {
      j.start_array("cbr-pssch-TxConfigList-v1530");
      for (const auto& e1 : *cbr_pssch_tx_cfg_list_v1530) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// SL-PoolSelectionConfig-r12 ::= SEQUENCE
SRSASN_CODE sl_pool_sel_cfg_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, thresh_low_r12, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, thresh_high_r12, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pool_sel_cfg_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(thresh_low_r12, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(thresh_high_r12, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void sl_pool_sel_cfg_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("threshLow-r12", thresh_low_r12);
  j.write_int("threshHigh-r12", thresh_high_r12);
  j.end_obj();
}

// SL-SyncConfigNFreq-r13 ::= SEQUENCE
SRSASN_CODE sl_sync_cfg_nfreq_r13_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(async_params_r13_present, 1));
  HANDLE_CODE(bref.pack(tx_params_r13_present, 1));
  HANDLE_CODE(bref.pack(rx_params_r13_present, 1));

  if (async_params_r13_present) {
    HANDLE_CODE(async_params_r13.sync_cp_len_r13.pack(bref));
    HANDLE_CODE(pack_integer(bref, async_params_r13.sync_offset_ind_r13, (uint8_t)0u, (uint8_t)39u));
    HANDLE_CODE(pack_integer(bref, async_params_r13.slssid_r13, (uint8_t)0u, (uint8_t)167u));
  }
  if (tx_params_r13_present) {
    HANDLE_CODE(bref.pack(tx_params_r13.sync_info_reserved_r13_present, 1));
    HANDLE_CODE(bref.pack(tx_params_r13.sync_tx_periodic_r13_present, 1));
    HANDLE_CODE(tx_params_r13.sync_tx_params_r13.pack(bref));
    HANDLE_CODE(pack_integer(bref, tx_params_r13.sync_tx_thresh_ic_r13, (uint8_t)0u, (uint8_t)13u));
    if (tx_params_r13.sync_info_reserved_r13_present) {
      HANDLE_CODE(tx_params_r13.sync_info_reserved_r13.pack(bref));
    }
  }
  if (rx_params_r13_present) {
    HANDLE_CODE(rx_params_r13.disc_sync_win_r13.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= sync_offset_ind_v1430_present;
    group_flags[0] |= gnss_sync_r14_present;
    group_flags[1] |= sync_offset_ind2_r14_present;
    group_flags[1] |= sync_offset_ind3_r14_present;
    group_flags[2] |= slss_tx_disabled_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sync_offset_ind_v1430_present, 1));
      HANDLE_CODE(bref.pack(gnss_sync_r14_present, 1));
      if (sync_offset_ind_v1430_present) {
        HANDLE_CODE(pack_integer(bref, sync_offset_ind_v1430, (uint8_t)40u, (uint8_t)159u));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sync_offset_ind2_r14_present, 1));
      HANDLE_CODE(bref.pack(sync_offset_ind3_r14_present, 1));
      if (sync_offset_ind2_r14_present) {
        HANDLE_CODE(pack_integer(bref, sync_offset_ind2_r14, (uint8_t)0u, (uint8_t)159u));
      }
      if (sync_offset_ind3_r14_present) {
        HANDLE_CODE(pack_integer(bref, sync_offset_ind3_r14, (uint8_t)0u, (uint8_t)159u));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(slss_tx_disabled_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_sync_cfg_nfreq_r13_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(async_params_r13_present, 1));
  HANDLE_CODE(bref.unpack(tx_params_r13_present, 1));
  HANDLE_CODE(bref.unpack(rx_params_r13_present, 1));

  if (async_params_r13_present) {
    HANDLE_CODE(async_params_r13.sync_cp_len_r13.unpack(bref));
    HANDLE_CODE(unpack_integer(async_params_r13.sync_offset_ind_r13, bref, (uint8_t)0u, (uint8_t)39u));
    HANDLE_CODE(unpack_integer(async_params_r13.slssid_r13, bref, (uint8_t)0u, (uint8_t)167u));
  }
  if (tx_params_r13_present) {
    HANDLE_CODE(bref.unpack(tx_params_r13.sync_info_reserved_r13_present, 1));
    HANDLE_CODE(bref.unpack(tx_params_r13.sync_tx_periodic_r13_present, 1));
    HANDLE_CODE(tx_params_r13.sync_tx_params_r13.unpack(bref));
    HANDLE_CODE(unpack_integer(tx_params_r13.sync_tx_thresh_ic_r13, bref, (uint8_t)0u, (uint8_t)13u));
    if (tx_params_r13.sync_info_reserved_r13_present) {
      HANDLE_CODE(tx_params_r13.sync_info_reserved_r13.unpack(bref));
    }
  }
  if (rx_params_r13_present) {
    HANDLE_CODE(rx_params_r13.disc_sync_win_r13.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(3);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(sync_offset_ind_v1430_present, 1));
      HANDLE_CODE(bref.unpack(gnss_sync_r14_present, 1));
      if (sync_offset_ind_v1430_present) {
        HANDLE_CODE(unpack_integer(sync_offset_ind_v1430, bref, (uint8_t)40u, (uint8_t)159u));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(sync_offset_ind2_r14_present, 1));
      HANDLE_CODE(bref.unpack(sync_offset_ind3_r14_present, 1));
      if (sync_offset_ind2_r14_present) {
        HANDLE_CODE(unpack_integer(sync_offset_ind2_r14, bref, (uint8_t)0u, (uint8_t)159u));
      }
      if (sync_offset_ind3_r14_present) {
        HANDLE_CODE(unpack_integer(sync_offset_ind3_r14, bref, (uint8_t)0u, (uint8_t)159u));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(slss_tx_disabled_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
void sl_sync_cfg_nfreq_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (async_params_r13_present) {
    j.write_fieldname("asyncParameters-r13");
    j.start_obj();
    j.write_str("syncCP-Len-r13", async_params_r13.sync_cp_len_r13.to_string());
    j.write_int("syncOffsetIndicator-r13", async_params_r13.sync_offset_ind_r13);
    j.write_int("slssid-r13", async_params_r13.slssid_r13);
    j.end_obj();
  }
  if (tx_params_r13_present) {
    j.write_fieldname("txParameters-r13");
    j.start_obj();
    j.write_fieldname("syncTxParameters-r13");
    tx_params_r13.sync_tx_params_r13.to_json(j);
    j.write_int("syncTxThreshIC-r13", tx_params_r13.sync_tx_thresh_ic_r13);
    if (tx_params_r13.sync_info_reserved_r13_present) {
      j.write_str("syncInfoReserved-r13", tx_params_r13.sync_info_reserved_r13.to_string());
    }
    if (tx_params_r13.sync_tx_periodic_r13_present) {
      j.write_str("syncTxPeriodic-r13", "true");
    }
    j.end_obj();
  }
  if (rx_params_r13_present) {
    j.write_fieldname("rxParameters-r13");
    j.start_obj();
    j.write_str("discSyncWindow-r13", rx_params_r13.disc_sync_win_r13.to_string());
    j.end_obj();
  }
  if (ext) {
    if (sync_offset_ind_v1430_present) {
      j.write_int("syncOffsetIndicator-v1430", sync_offset_ind_v1430);
    }
    if (gnss_sync_r14_present) {
      j.write_str("gnss-Sync-r14", "true");
    }
    if (sync_offset_ind2_r14_present) {
      j.write_int("syncOffsetIndicator2-r14", sync_offset_ind2_r14);
    }
    if (sync_offset_ind3_r14_present) {
      j.write_int("syncOffsetIndicator3-r14", sync_offset_ind3_r14);
    }
    if (slss_tx_disabled_r15_present) {
      j.write_str("slss-TxDisabled-r15", "true");
    }
  }
  j.end_obj();
}

// SL-TF-ResourceConfig-r12 ::= SEQUENCE
SRSASN_CODE sl_tf_res_cfg_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, prb_num_r12, (uint8_t)1u, (uint8_t)100u));
  HANDLE_CODE(pack_integer(bref, prb_start_r12, (uint8_t)0u, (uint8_t)99u));
  HANDLE_CODE(pack_integer(bref, prb_end_r12, (uint8_t)0u, (uint8_t)99u));
  HANDLE_CODE(offset_ind_r12.pack(bref));
  HANDLE_CODE(sf_bitmap_r12.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_tf_res_cfg_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(prb_num_r12, bref, (uint8_t)1u, (uint8_t)100u));
  HANDLE_CODE(unpack_integer(prb_start_r12, bref, (uint8_t)0u, (uint8_t)99u));
  HANDLE_CODE(unpack_integer(prb_end_r12, bref, (uint8_t)0u, (uint8_t)99u));
  HANDLE_CODE(offset_ind_r12.unpack(bref));
  HANDLE_CODE(sf_bitmap_r12.unpack(bref));

  return SRSASN_SUCCESS;
}
void sl_tf_res_cfg_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("prb-Num-r12", prb_num_r12);
  j.write_int("prb-Start-r12", prb_start_r12);
  j.write_int("prb-End-r12", prb_end_r12);
  j.write_fieldname("offsetIndicator-r12");
  offset_ind_r12.to_json(j);
  j.write_fieldname("subframeBitmap-r12");
  sf_bitmap_r12.to_json(j);
  j.end_obj();
}

// SystemInformationBlockType1-v1430-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1430_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ecall_over_ims_support_r14_present, 1));
  HANDLE_CODE(bref.pack(tdd_cfg_v1430_present, 1));
  HANDLE_CODE(bref.pack(cell_access_related_info_list_r14_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (tdd_cfg_v1430_present) {
    HANDLE_CODE(tdd_cfg_v1430.pack(bref));
  }
  if (cell_access_related_info_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, cell_access_related_info_list_r14, 1, 5));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1430_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ecall_over_ims_support_r14_present, 1));
  HANDLE_CODE(bref.unpack(tdd_cfg_v1430_present, 1));
  HANDLE_CODE(bref.unpack(cell_access_related_info_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (tdd_cfg_v1430_present) {
    HANDLE_CODE(tdd_cfg_v1430.unpack(bref));
  }
  if (cell_access_related_info_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(cell_access_related_info_list_r14, bref, 1, 5));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1430_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ecall_over_ims_support_r14_present) {
    j.write_str("eCallOverIMS-Support-r14", "true");
  }
  if (tdd_cfg_v1430_present) {
    j.write_fieldname("tdd-Config-v1430");
    tdd_cfg_v1430.to_json(j);
  }
  if (cell_access_related_info_list_r14_present) {
    j.start_array("cellAccessRelatedInfoList-r14");
    for (const auto& e1 : cell_access_related_info_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// TDD-Config-v1130 ::= SEQUENCE
SRSASN_CODE tdd_cfg_v1130_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(special_sf_patterns_v1130.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE tdd_cfg_v1130_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(special_sf_patterns_v1130.unpack(bref));

  return SRSASN_SUCCESS;
}
void tdd_cfg_v1130_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("specialSubframePatterns-v1130", special_sf_patterns_v1130.to_string());
  j.end_obj();
}

// BandClassInfoCDMA2000 ::= SEQUENCE
SRSASN_CODE band_class_info_cdma2000_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_resel_prio_present, 1));

  HANDLE_CODE(band_class.pack(bref));
  if (cell_resel_prio_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low, (uint8_t)0u, (uint8_t)63u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE band_class_info_cdma2000_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_resel_prio_present, 1));

  HANDLE_CODE(band_class.unpack(bref));
  if (cell_resel_prio_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(unpack_integer(thresh_x_low, bref, (uint8_t)0u, (uint8_t)63u));

  return SRSASN_SUCCESS;
}
void band_class_info_cdma2000_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("bandClass", band_class.to_string());
  if (cell_resel_prio_present) {
    j.write_int("cellReselectionPriority", cell_resel_prio);
  }
  j.write_int("threshX-High", thresh_x_high);
  j.write_int("threshX-Low", thresh_x_low);
  j.end_obj();
}

// CellSelectionInfoCE1-r13 ::= SEQUENCE
SRSASN_CODE cell_sel_info_ce1_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(q_qual_min_rsrq_ce1_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_ce1_r13, (int8_t)-70, (int8_t)-22));
  if (q_qual_min_rsrq_ce1_r13_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_rsrq_ce1_r13, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_ce1_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(q_qual_min_rsrq_ce1_r13_present, 1));

  HANDLE_CODE(unpack_integer(q_rx_lev_min_ce1_r13, bref, (int8_t)-70, (int8_t)-22));
  if (q_qual_min_rsrq_ce1_r13_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_rsrq_ce1_r13, bref, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
void cell_sel_info_ce1_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-RxLevMinCE1-r13", q_rx_lev_min_ce1_r13);
  if (q_qual_min_rsrq_ce1_r13_present) {
    j.write_int("q-QualMinRSRQ-CE1-r13", q_qual_min_rsrq_ce1_r13);
  }
  j.end_obj();
}

// NeighCellsPerBandclassCDMA2000-r11 ::= SEQUENCE
SRSASN_CODE neigh_cells_per_bandclass_cdma2000_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, arfcn, (uint16_t)0u, (uint16_t)2047u));
  HANDLE_CODE(pack_dyn_seq_of(bref, pci_list_r11, 1, 40, integer_packer<uint16_t>(0, 511)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cells_per_bandclass_cdma2000_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(arfcn, bref, (uint16_t)0u, (uint16_t)2047u));
  HANDLE_CODE(unpack_dyn_seq_of(pci_list_r11, bref, 1, 40, integer_packer<uint16_t>(0, 511)));

  return SRSASN_SUCCESS;
}
void neigh_cells_per_bandclass_cdma2000_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("arfcn", arfcn);
  j.start_array("physCellIdList-r11");
  for (const auto& e1 : pci_list_r11) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// SL-CommTxPoolSensingConfig-r14 ::= SEQUENCE
SRSASN_CODE sl_comm_tx_pool_sensing_cfg_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(restrict_res_reserv_period_r14_present, 1));
  HANDLE_CODE(bref.pack(p2x_sensing_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(sl_reselect_after_r14_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, pssch_tx_cfg_list_r14, 1, 16));
  HANDLE_CODE(pack_fixed_seq_of(
      bref, &(thres_pssch_rsrp_list_r14)[0], thres_pssch_rsrp_list_r14.size(), integer_packer<uint8_t>(0, 66)));
  if (restrict_res_reserv_period_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, restrict_res_reserv_period_r14, 1, 16));
  }
  HANDLE_CODE(prob_res_keep_r14.pack(bref));
  if (p2x_sensing_cfg_r14_present) {
    HANDLE_CODE(pack_integer(bref, p2x_sensing_cfg_r14.min_num_candidate_sf_r14, (uint8_t)1u, (uint8_t)13u));
    HANDLE_CODE(p2x_sensing_cfg_r14.gap_candidate_sensing_r14.pack(bref));
  }
  if (sl_reselect_after_r14_present) {
    HANDLE_CODE(sl_reselect_after_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_comm_tx_pool_sensing_cfg_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(restrict_res_reserv_period_r14_present, 1));
  HANDLE_CODE(bref.unpack(p2x_sensing_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(sl_reselect_after_r14_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(pssch_tx_cfg_list_r14, bref, 1, 16));
  HANDLE_CODE(unpack_fixed_seq_of(
      &(thres_pssch_rsrp_list_r14)[0], bref, thres_pssch_rsrp_list_r14.size(), integer_packer<uint8_t>(0, 66)));
  if (restrict_res_reserv_period_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(restrict_res_reserv_period_r14, bref, 1, 16));
  }
  HANDLE_CODE(prob_res_keep_r14.unpack(bref));
  if (p2x_sensing_cfg_r14_present) {
    HANDLE_CODE(unpack_integer(p2x_sensing_cfg_r14.min_num_candidate_sf_r14, bref, (uint8_t)1u, (uint8_t)13u));
    HANDLE_CODE(p2x_sensing_cfg_r14.gap_candidate_sensing_r14.unpack(bref));
  }
  if (sl_reselect_after_r14_present) {
    HANDLE_CODE(sl_reselect_after_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_comm_tx_pool_sensing_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("pssch-TxConfigList-r14");
  for (const auto& e1 : pssch_tx_cfg_list_r14) {
    e1.to_json(j);
  }
  j.end_array();
  j.start_array("thresPSSCH-RSRP-List-r14");
  for (const auto& e1 : thres_pssch_rsrp_list_r14) {
    j.write_int(e1);
  }
  j.end_array();
  if (restrict_res_reserv_period_r14_present) {
    j.start_array("restrictResourceReservationPeriod-r14");
    for (const auto& e1 : restrict_res_reserv_period_r14) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  j.write_str("probResourceKeep-r14", prob_res_keep_r14.to_string());
  if (p2x_sensing_cfg_r14_present) {
    j.write_fieldname("p2x-SensingConfig-r14");
    j.start_obj();
    j.write_int("minNumCandidateSF-r14", p2x_sensing_cfg_r14.min_num_candidate_sf_r14);
    j.write_str("gapCandidateSensing-r14", p2x_sensing_cfg_r14.gap_candidate_sensing_r14.to_string());
    j.end_obj();
  }
  if (sl_reselect_after_r14_present) {
    j.write_str("sl-ReselectAfter-r14", sl_reselect_after_r14.to_string());
  }
  j.end_obj();
}

// SL-DiscResourcePool-r12 ::= SEQUENCE
SRSASN_CODE sl_disc_res_pool_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(tx_params_r12_present, 1));
  HANDLE_CODE(bref.pack(rx_params_r12_present, 1));

  HANDLE_CODE(cp_len_r12.pack(bref));
  HANDLE_CODE(disc_period_r12.pack(bref));
  HANDLE_CODE(pack_integer(bref, num_retx_r12, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(pack_integer(bref, num_repeat_r12, (uint8_t)1u, (uint8_t)50u));
  HANDLE_CODE(tf_res_cfg_r12.pack(bref));
  if (tx_params_r12_present) {
    HANDLE_CODE(bref.pack(tx_params_r12.ue_sel_res_cfg_r12_present, 1));
    HANDLE_CODE(tx_params_r12.tx_params_general_r12.pack(bref));
    if (tx_params_r12.ue_sel_res_cfg_r12_present) {
      HANDLE_CODE(tx_params_r12.ue_sel_res_cfg_r12.pool_sel_r12.pack(bref));
      HANDLE_CODE(tx_params_r12.ue_sel_res_cfg_r12.tx_probability_r12.pack(bref));
    }
  }
  if (rx_params_r12_present) {
    HANDLE_CODE(bref.pack(rx_params_r12.tdd_cfg_r12_present, 1));
    if (rx_params_r12.tdd_cfg_r12_present) {
      HANDLE_CODE(rx_params_r12.tdd_cfg_r12.pack(bref));
    }
    HANDLE_CODE(pack_integer(bref, rx_params_r12.sync_cfg_idx_r12, (uint8_t)0u, (uint8_t)15u));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= disc_period_v1310.is_present();
    group_flags[0] |= rx_params_add_neigh_freq_r13.is_present();
    group_flags[0] |= tx_params_add_neigh_freq_r13.is_present();
    group_flags[1] |= tx_params_add_neigh_freq_v1370.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(disc_period_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(rx_params_add_neigh_freq_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(tx_params_add_neigh_freq_r13.is_present(), 1));
      if (disc_period_v1310.is_present()) {
        HANDLE_CODE(disc_period_v1310->pack(bref));
      }
      if (rx_params_add_neigh_freq_r13.is_present()) {
        HANDLE_CODE(rx_params_add_neigh_freq_r13->pack(bref));
      }
      if (tx_params_add_neigh_freq_r13.is_present()) {
        HANDLE_CODE(tx_params_add_neigh_freq_r13->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(tx_params_add_neigh_freq_v1370.is_present(), 1));
      if (tx_params_add_neigh_freq_v1370.is_present()) {
        HANDLE_CODE(tx_params_add_neigh_freq_v1370->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_res_pool_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(tx_params_r12_present, 1));
  HANDLE_CODE(bref.unpack(rx_params_r12_present, 1));

  HANDLE_CODE(cp_len_r12.unpack(bref));
  HANDLE_CODE(disc_period_r12.unpack(bref));
  HANDLE_CODE(unpack_integer(num_retx_r12, bref, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(unpack_integer(num_repeat_r12, bref, (uint8_t)1u, (uint8_t)50u));
  HANDLE_CODE(tf_res_cfg_r12.unpack(bref));
  if (tx_params_r12_present) {
    HANDLE_CODE(bref.unpack(tx_params_r12.ue_sel_res_cfg_r12_present, 1));
    HANDLE_CODE(tx_params_r12.tx_params_general_r12.unpack(bref));
    if (tx_params_r12.ue_sel_res_cfg_r12_present) {
      HANDLE_CODE(tx_params_r12.ue_sel_res_cfg_r12.pool_sel_r12.unpack(bref));
      HANDLE_CODE(tx_params_r12.ue_sel_res_cfg_r12.tx_probability_r12.unpack(bref));
    }
  }
  if (rx_params_r12_present) {
    HANDLE_CODE(bref.unpack(rx_params_r12.tdd_cfg_r12_present, 1));
    if (rx_params_r12.tdd_cfg_r12_present) {
      HANDLE_CODE(rx_params_r12.tdd_cfg_r12.unpack(bref));
    }
    HANDLE_CODE(unpack_integer(rx_params_r12.sync_cfg_idx_r12, bref, (uint8_t)0u, (uint8_t)15u));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool disc_period_v1310_present;
      HANDLE_CODE(bref.unpack(disc_period_v1310_present, 1));
      disc_period_v1310.set_present(disc_period_v1310_present);
      bool rx_params_add_neigh_freq_r13_present;
      HANDLE_CODE(bref.unpack(rx_params_add_neigh_freq_r13_present, 1));
      rx_params_add_neigh_freq_r13.set_present(rx_params_add_neigh_freq_r13_present);
      bool tx_params_add_neigh_freq_r13_present;
      HANDLE_CODE(bref.unpack(tx_params_add_neigh_freq_r13_present, 1));
      tx_params_add_neigh_freq_r13.set_present(tx_params_add_neigh_freq_r13_present);
      if (disc_period_v1310.is_present()) {
        HANDLE_CODE(disc_period_v1310->unpack(bref));
      }
      if (rx_params_add_neigh_freq_r13.is_present()) {
        HANDLE_CODE(rx_params_add_neigh_freq_r13->unpack(bref));
      }
      if (tx_params_add_neigh_freq_r13.is_present()) {
        HANDLE_CODE(tx_params_add_neigh_freq_r13->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool tx_params_add_neigh_freq_v1370_present;
      HANDLE_CODE(bref.unpack(tx_params_add_neigh_freq_v1370_present, 1));
      tx_params_add_neigh_freq_v1370.set_present(tx_params_add_neigh_freq_v1370_present);
      if (tx_params_add_neigh_freq_v1370.is_present()) {
        HANDLE_CODE(tx_params_add_neigh_freq_v1370->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sl_disc_res_pool_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cp-Len-r12", cp_len_r12.to_string());
  j.write_str("discPeriod-r12", disc_period_r12.to_string());
  j.write_int("numRetx-r12", num_retx_r12);
  j.write_int("numRepetition-r12", num_repeat_r12);
  j.write_fieldname("tf-ResourceConfig-r12");
  tf_res_cfg_r12.to_json(j);
  if (tx_params_r12_present) {
    j.write_fieldname("txParameters-r12");
    j.start_obj();
    j.write_fieldname("txParametersGeneral-r12");
    tx_params_r12.tx_params_general_r12.to_json(j);
    if (tx_params_r12.ue_sel_res_cfg_r12_present) {
      j.write_fieldname("ue-SelectedResourceConfig-r12");
      j.start_obj();
      j.write_fieldname("poolSelection-r12");
      tx_params_r12.ue_sel_res_cfg_r12.pool_sel_r12.to_json(j);
      j.write_str("txProbability-r12", tx_params_r12.ue_sel_res_cfg_r12.tx_probability_r12.to_string());
      j.end_obj();
    }
    j.end_obj();
  }
  if (rx_params_r12_present) {
    j.write_fieldname("rxParameters-r12");
    j.start_obj();
    if (rx_params_r12.tdd_cfg_r12_present) {
      j.write_fieldname("tdd-Config-r12");
      rx_params_r12.tdd_cfg_r12.to_json(j);
    }
    j.write_int("syncConfigIndex-r12", rx_params_r12.sync_cfg_idx_r12);
    j.end_obj();
  }
  if (ext) {
    if (disc_period_v1310.is_present()) {
      j.write_fieldname("discPeriod-v1310");
      disc_period_v1310->to_json(j);
    }
    if (rx_params_add_neigh_freq_r13.is_present()) {
      j.write_fieldname("rxParamsAddNeighFreq-r13");
      rx_params_add_neigh_freq_r13->to_json(j);
    }
    if (tx_params_add_neigh_freq_r13.is_present()) {
      j.write_fieldname("txParamsAddNeighFreq-r13");
      tx_params_add_neigh_freq_r13->to_json(j);
    }
    if (tx_params_add_neigh_freq_v1370.is_present()) {
      j.write_fieldname("txParamsAddNeighFreq-v1370");
      tx_params_add_neigh_freq_v1370->to_json(j);
    }
  }
  j.end_obj();
}

void sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_::set(types::options e)
{
  type_ = e;
}
void sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::rsrp_based_r12:
      j.write_fieldname("rsrpBased-r12");
      c.to_json(j);
      break;
    case types::random_r12:
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_");
  }
  j.end_obj();
}
SRSASN_CODE sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::rsrp_based_r12:
      HANDLE_CODE(c.pack(bref));
      break;
    case types::random_r12:
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::rsrp_based_r12:
      HANDLE_CODE(c.unpack(bref));
      break;
    case types::random_r12:
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_r12_s_::ue_sel_res_cfg_r12_s_::pool_sel_r12_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sl_disc_res_pool_r12_s::disc_period_v1310_c_::set(types::options e)
{
  type_ = e;
}
void sl_disc_res_pool_r12_s::disc_period_v1310_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_str("setup", c.to_string());
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::disc_period_v1310_c_");
  }
  j.end_obj();
}
SRSASN_CODE sl_disc_res_pool_r12_s::disc_period_v1310_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::disc_period_v1310_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_res_pool_r12_s::disc_period_v1310_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::disc_period_v1310_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_::set(types::options e)
{
  type_ = e;
}
void sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("physCellId-r13");
      for (const auto& e1 : c.pci_r13) {
        j.write_int(e1);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.pci_r13, 1, 16, integer_packer<uint16_t>(0, 503)));
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.pci_r13, bref, 1, 16, integer_packer<uint16_t>(0, 503)));
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::rx_params_add_neigh_freq_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_::set(types::options e)
{
  type_ = e;
}
void sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("physCellId-r13");
      for (const auto& e1 : c.pci_r13) {
        j.write_int(e1);
      }
      j.end_array();
      if (c.p_max_present) {
        j.write_int("p-Max", c.p_max);
      }
      if (c.tdd_cfg_r13_present) {
        j.write_fieldname("tdd-Config-r13");
        c.tdd_cfg_r13.to_json(j);
      }
      if (c.tdd_cfg_v1130_present) {
        j.write_fieldname("tdd-Config-v1130");
        c.tdd_cfg_v1130.to_json(j);
      }
      j.write_fieldname("freqInfo");
      j.start_obj();
      if (c.freq_info.ul_carrier_freq_present) {
        j.write_int("ul-CarrierFreq", c.freq_info.ul_carrier_freq);
      }
      if (c.freq_info.ul_bw_present) {
        j.write_str("ul-Bandwidth", c.freq_info.ul_bw.to_string());
      }
      j.write_int("additionalSpectrumEmission", c.freq_info.add_spec_emission);
      j.end_obj();
      j.write_int("referenceSignalPower", c.ref_sig_pwr);
      if (c.sync_cfg_idx_r13_present) {
        j.write_int("syncConfigIndex-r13", c.sync_cfg_idx_r13);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.p_max_present, 1));
      HANDLE_CODE(bref.pack(c.tdd_cfg_r13_present, 1));
      HANDLE_CODE(bref.pack(c.tdd_cfg_v1130_present, 1));
      HANDLE_CODE(bref.pack(c.sync_cfg_idx_r13_present, 1));
      HANDLE_CODE(pack_dyn_seq_of(bref, c.pci_r13, 1, 16, integer_packer<uint16_t>(0, 503)));
      if (c.p_max_present) {
        HANDLE_CODE(pack_integer(bref, c.p_max, (int8_t)-30, (int8_t)33));
      }
      if (c.tdd_cfg_r13_present) {
        HANDLE_CODE(c.tdd_cfg_r13.pack(bref));
      }
      if (c.tdd_cfg_v1130_present) {
        HANDLE_CODE(c.tdd_cfg_v1130.pack(bref));
      }
      HANDLE_CODE(bref.pack(c.freq_info.ul_carrier_freq_present, 1));
      HANDLE_CODE(bref.pack(c.freq_info.ul_bw_present, 1));
      if (c.freq_info.ul_carrier_freq_present) {
        HANDLE_CODE(pack_integer(bref, c.freq_info.ul_carrier_freq, (uint32_t)0u, (uint32_t)65535u));
      }
      if (c.freq_info.ul_bw_present) {
        HANDLE_CODE(c.freq_info.ul_bw.pack(bref));
      }
      HANDLE_CODE(pack_integer(bref, c.freq_info.add_spec_emission, (uint8_t)1u, (uint8_t)32u));
      HANDLE_CODE(pack_integer(bref, c.ref_sig_pwr, (int8_t)-60, (int8_t)50));
      if (c.sync_cfg_idx_r13_present) {
        HANDLE_CODE(pack_integer(bref, c.sync_cfg_idx_r13, (uint8_t)0u, (uint8_t)15u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.p_max_present, 1));
      HANDLE_CODE(bref.unpack(c.tdd_cfg_r13_present, 1));
      HANDLE_CODE(bref.unpack(c.tdd_cfg_v1130_present, 1));
      HANDLE_CODE(bref.unpack(c.sync_cfg_idx_r13_present, 1));
      HANDLE_CODE(unpack_dyn_seq_of(c.pci_r13, bref, 1, 16, integer_packer<uint16_t>(0, 503)));
      if (c.p_max_present) {
        HANDLE_CODE(unpack_integer(c.p_max, bref, (int8_t)-30, (int8_t)33));
      }
      if (c.tdd_cfg_r13_present) {
        HANDLE_CODE(c.tdd_cfg_r13.unpack(bref));
      }
      if (c.tdd_cfg_v1130_present) {
        HANDLE_CODE(c.tdd_cfg_v1130.unpack(bref));
      }
      HANDLE_CODE(bref.unpack(c.freq_info.ul_carrier_freq_present, 1));
      HANDLE_CODE(bref.unpack(c.freq_info.ul_bw_present, 1));
      if (c.freq_info.ul_carrier_freq_present) {
        HANDLE_CODE(unpack_integer(c.freq_info.ul_carrier_freq, bref, (uint32_t)0u, (uint32_t)65535u));
      }
      if (c.freq_info.ul_bw_present) {
        HANDLE_CODE(c.freq_info.ul_bw.unpack(bref));
      }
      HANDLE_CODE(unpack_integer(c.freq_info.add_spec_emission, bref, (uint8_t)1u, (uint8_t)32u));
      HANDLE_CODE(unpack_integer(c.ref_sig_pwr, bref, (int8_t)-60, (int8_t)50));
      if (c.sync_cfg_idx_r13_present) {
        HANDLE_CODE(unpack_integer(c.sync_cfg_idx_r13, bref, (uint8_t)0u, (uint8_t)15u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_::set(types::options e)
{
  type_ = e;
}
void sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_fieldname("freqInfo-v1370");
      j.start_obj();
      j.write_int("additionalSpectrumEmission-v1370", c.freq_info_v1370.add_spec_emission_v1370);
      j.end_obj();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_");
  }
  j.end_obj();
}
SRSASN_CODE sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_integer(bref, c.freq_info_v1370.add_spec_emission_v1370, (uint16_t)33u, (uint16_t)288u));
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_integer(c.freq_info_v1370.add_spec_emission_v1370, bref, (uint16_t)33u, (uint16_t)288u));
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_res_pool_r12_s::tx_params_add_neigh_freq_v1370_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SL-ZoneConfig-r14 ::= SEQUENCE
SRSASN_CODE sl_zone_cfg_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(zone_len_r14.pack(bref));
  HANDLE_CODE(zone_width_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, zone_id_longi_mod_r14, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(pack_integer(bref, zone_id_lati_mod_r14, (uint8_t)1u, (uint8_t)4u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_zone_cfg_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(zone_len_r14.unpack(bref));
  HANDLE_CODE(zone_width_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(zone_id_longi_mod_r14, bref, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(unpack_integer(zone_id_lati_mod_r14, bref, (uint8_t)1u, (uint8_t)4u));

  return SRSASN_SUCCESS;
}
void sl_zone_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("zoneLength-r14", zone_len_r14.to_string());
  j.write_str("zoneWidth-r14", zone_width_r14.to_string());
  j.write_int("zoneIdLongiMod-r14", zone_id_longi_mod_r14);
  j.write_int("zoneIdLatiMod-r14", zone_id_lati_mod_r14);
  j.end_obj();
}

// SystemInformationBlockType1-v1360-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1360_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cell_sel_info_ce1_v1360_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (cell_sel_info_ce1_v1360_present) {
    HANDLE_CODE(cell_sel_info_ce1_v1360.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1360_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cell_sel_info_ce1_v1360_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (cell_sel_info_ce1_v1360_present) {
    HANDLE_CODE(cell_sel_info_ce1_v1360.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1360_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cell_sel_info_ce1_v1360_present) {
    j.write_fieldname("cellSelectionInfoCE1-v1360");
    cell_sel_info_ce1_v1360.to_json(j);
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v10l0 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v10l0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_info_v10l0_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10l0_present, 1));

  if (freq_band_info_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, freq_band_info_v10l0, 1, 8));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10l0, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v10l0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_info_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10l0_present, 1));

  if (freq_band_info_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(freq_band_info_v10l0, bref, 1, 8));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10l0, bref, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v10l0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_info_v10l0_present) {
    j.start_array("freqBandInfo-v10l0");
    for (const auto& e1 : freq_band_info_v10l0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (multi_band_info_list_v10l0_present) {
    j.start_array("multiBandInfoList-v10l0");
    for (const auto& e1 : multi_band_info_list_v10l0) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  j.end_obj();
}

// MultiBandInfo-v9e0 ::= SEQUENCE
SRSASN_CODE multi_band_info_v9e0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_ind_v9e0_present, 1));

  if (freq_band_ind_v9e0_present) {
    HANDLE_CODE(pack_integer(bref, freq_band_ind_v9e0, (uint16_t)65u, (uint16_t)256u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE multi_band_info_v9e0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_ind_v9e0_present, 1));

  if (freq_band_ind_v9e0_present) {
    HANDLE_CODE(unpack_integer(freq_band_ind_v9e0, bref, (uint16_t)65u, (uint16_t)256u));
  }

  return SRSASN_SUCCESS;
}
void multi_band_info_v9e0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_ind_v9e0_present) {
    j.write_int("freqBandIndicator-v9e0", freq_band_ind_v9e0);
  }
  j.end_obj();
}

// NeighCellCDMA2000-r11 ::= SEQUENCE
SRSASN_CODE neigh_cell_cdma2000_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(band_class.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, neigh_freq_info_list_r11, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cell_cdma2000_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(band_class.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(neigh_freq_info_list_r11, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void neigh_cell_cdma2000_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("bandClass", band_class.to_string());
  j.start_array("neighFreqInfoList-r11");
  for (const auto& e1 : neigh_freq_info_list_r11) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// NeighCellsPerBandclassCDMA2000 ::= SEQUENCE
SRSASN_CODE neigh_cells_per_bandclass_cdma2000_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, arfcn, (uint16_t)0u, (uint16_t)2047u));
  HANDLE_CODE(pack_dyn_seq_of(bref, pci_list, 1, 16, integer_packer<uint16_t>(0, 511)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cells_per_bandclass_cdma2000_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(arfcn, bref, (uint16_t)0u, (uint16_t)2047u));
  HANDLE_CODE(unpack_dyn_seq_of(pci_list, bref, 1, 16, integer_packer<uint16_t>(0, 511)));

  return SRSASN_SUCCESS;
}
void neigh_cells_per_bandclass_cdma2000_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("arfcn", arfcn);
  j.start_array("physCellIdList");
  for (const auto& e1 : pci_list) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// NeighCellsPerBandclassCDMA2000-v920 ::= SEQUENCE
SRSASN_CODE neigh_cells_per_bandclass_cdma2000_v920_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, pci_list_v920, 0, 24, integer_packer<uint16_t>(0, 511)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cells_per_bandclass_cdma2000_v920_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(pci_list_v920, bref, 0, 24, integer_packer<uint16_t>(0, 511)));

  return SRSASN_SUCCESS;
}
void neigh_cells_per_bandclass_cdma2000_v920_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("physCellIdList-v920");
  for (const auto& e1 : pci_list_v920) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// PhysCellIdRange ::= SEQUENCE
SRSASN_CODE pci_range_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(range_present, 1));

  HANDLE_CODE(pack_integer(bref, start, (uint16_t)0u, (uint16_t)503u));
  if (range_present) {
    HANDLE_CODE(range.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pci_range_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(range_present, 1));

  HANDLE_CODE(unpack_integer(start, bref, (uint16_t)0u, (uint16_t)503u));
  if (range_present) {
    HANDLE_CODE(range.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pci_range_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("start", start);
  if (range_present) {
    j.write_str("range", range.to_string());
  }
  j.end_obj();
}

// RedistributionNeighCell-r13 ::= SEQUENCE
SRSASN_CODE redist_neigh_cell_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, pci_r13, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(pack_integer(bref, redist_factor_cell_r13, (uint8_t)1u, (uint8_t)10u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE redist_neigh_cell_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(pci_r13, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(unpack_integer(redist_factor_cell_r13, bref, (uint8_t)1u, (uint8_t)10u));

  return SRSASN_SUCCESS;
}
void redist_neigh_cell_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId-r13", pci_r13);
  j.write_int("redistributionFactorCell-r13", redist_factor_cell_r13);
  j.end_obj();
}

// SL-DiscTxPowerInfo-r12 ::= SEQUENCE
SRSASN_CODE sl_disc_tx_pwr_info_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, disc_max_tx_pwr_r12, (int8_t)-30, (int8_t)33));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_tx_pwr_info_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(disc_max_tx_pwr_r12, bref, (int8_t)-30, (int8_t)33));

  return SRSASN_SUCCESS;
}
void sl_disc_tx_pwr_info_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("discMaxTxPower-r12", disc_max_tx_pwr_r12);
  j.end_obj();
}

// SL-V2X-FreqSelectionConfig-r15 ::= SEQUENCE
SRSASN_CODE sl_v2x_freq_sel_cfg_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(thresh_cbr_freq_resel_r15_present, 1));
  HANDLE_CODE(bref.pack(thresh_cbr_freq_keeping_r15_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, prio_list_r15, 1, 8, integer_packer<uint8_t>(1, 8)));
  if (thresh_cbr_freq_resel_r15_present) {
    HANDLE_CODE(pack_integer(bref, thresh_cbr_freq_resel_r15, (uint8_t)0u, (uint8_t)100u));
  }
  if (thresh_cbr_freq_keeping_r15_present) {
    HANDLE_CODE(pack_integer(bref, thresh_cbr_freq_keeping_r15, (uint8_t)0u, (uint8_t)100u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_v2x_freq_sel_cfg_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(thresh_cbr_freq_resel_r15_present, 1));
  HANDLE_CODE(bref.unpack(thresh_cbr_freq_keeping_r15_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(prio_list_r15, bref, 1, 8, integer_packer<uint8_t>(1, 8)));
  if (thresh_cbr_freq_resel_r15_present) {
    HANDLE_CODE(unpack_integer(thresh_cbr_freq_resel_r15, bref, (uint8_t)0u, (uint8_t)100u));
  }
  if (thresh_cbr_freq_keeping_r15_present) {
    HANDLE_CODE(unpack_integer(thresh_cbr_freq_keeping_r15, bref, (uint8_t)0u, (uint8_t)100u));
  }

  return SRSASN_SUCCESS;
}
void sl_v2x_freq_sel_cfg_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("priorityList-r15");
  for (const auto& e1 : prio_list_r15) {
    j.write_int(e1);
  }
  j.end_array();
  if (thresh_cbr_freq_resel_r15_present) {
    j.write_int("threshCBR-FreqReselection-r15", thresh_cbr_freq_resel_r15);
  }
  if (thresh_cbr_freq_keeping_r15_present) {
    j.write_int("threshCBR-FreqKeeping-r15", thresh_cbr_freq_keeping_r15);
  }
  j.end_obj();
}

// SL-V2X-InterFreqUE-Config-r14 ::= SEQUENCE
SRSASN_CODE sl_v2x_inter_freq_ue_cfg_r14_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(pci_list_r14_present, 1));
  HANDLE_CODE(bref.pack(type_tx_sync_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_sync_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_comm_rx_pool_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_comm_tx_pool_normal_r14_present, 1));
  HANDLE_CODE(bref.pack(p2x_comm_tx_pool_normal_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_comm_tx_pool_exceptional_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_res_sel_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(zone_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(offset_dfn_r14_present, 1));

  if (pci_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, pci_list_r14, 1, 16, integer_packer<uint16_t>(0, 503)));
  }
  if (type_tx_sync_r14_present) {
    HANDLE_CODE(type_tx_sync_r14.pack(bref));
  }
  if (v2x_sync_cfg_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_sync_cfg_r14, 1, 16));
  }
  if (v2x_comm_rx_pool_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_comm_rx_pool_r14, 1, 16));
  }
  if (v2x_comm_tx_pool_normal_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_comm_tx_pool_normal_r14, 1, 8));
  }
  if (p2x_comm_tx_pool_normal_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, p2x_comm_tx_pool_normal_r14, 1, 8));
  }
  if (v2x_comm_tx_pool_exceptional_r14_present) {
    HANDLE_CODE(v2x_comm_tx_pool_exceptional_r14.pack(bref));
  }
  if (v2x_res_sel_cfg_r14_present) {
    HANDLE_CODE(v2x_res_sel_cfg_r14.pack(bref));
  }
  if (zone_cfg_r14_present) {
    HANDLE_CODE(zone_cfg_r14.pack(bref));
  }
  if (offset_dfn_r14_present) {
    HANDLE_CODE(pack_integer(bref, offset_dfn_r14, (uint16_t)0u, (uint16_t)1000u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_v2x_inter_freq_ue_cfg_r14_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(pci_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(type_tx_sync_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_sync_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_comm_rx_pool_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_comm_tx_pool_normal_r14_present, 1));
  HANDLE_CODE(bref.unpack(p2x_comm_tx_pool_normal_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_comm_tx_pool_exceptional_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_res_sel_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(zone_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(offset_dfn_r14_present, 1));

  if (pci_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(pci_list_r14, bref, 1, 16, integer_packer<uint16_t>(0, 503)));
  }
  if (type_tx_sync_r14_present) {
    HANDLE_CODE(type_tx_sync_r14.unpack(bref));
  }
  if (v2x_sync_cfg_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_sync_cfg_r14, bref, 1, 16));
  }
  if (v2x_comm_rx_pool_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_comm_rx_pool_r14, bref, 1, 16));
  }
  if (v2x_comm_tx_pool_normal_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_comm_tx_pool_normal_r14, bref, 1, 8));
  }
  if (p2x_comm_tx_pool_normal_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(p2x_comm_tx_pool_normal_r14, bref, 1, 8));
  }
  if (v2x_comm_tx_pool_exceptional_r14_present) {
    HANDLE_CODE(v2x_comm_tx_pool_exceptional_r14.unpack(bref));
  }
  if (v2x_res_sel_cfg_r14_present) {
    HANDLE_CODE(v2x_res_sel_cfg_r14.unpack(bref));
  }
  if (zone_cfg_r14_present) {
    HANDLE_CODE(zone_cfg_r14.unpack(bref));
  }
  if (offset_dfn_r14_present) {
    HANDLE_CODE(unpack_integer(offset_dfn_r14, bref, (uint16_t)0u, (uint16_t)1000u));
  }

  return SRSASN_SUCCESS;
}
void sl_v2x_inter_freq_ue_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (pci_list_r14_present) {
    j.start_array("physCellIdList-r14");
    for (const auto& e1 : pci_list_r14) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (type_tx_sync_r14_present) {
    j.write_str("typeTxSync-r14", type_tx_sync_r14.to_string());
  }
  if (v2x_sync_cfg_r14_present) {
    j.start_array("v2x-SyncConfig-r14");
    for (const auto& e1 : v2x_sync_cfg_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_comm_rx_pool_r14_present) {
    j.start_array("v2x-CommRxPool-r14");
    for (const auto& e1 : v2x_comm_rx_pool_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_comm_tx_pool_normal_r14_present) {
    j.start_array("v2x-CommTxPoolNormal-r14");
    for (const auto& e1 : v2x_comm_tx_pool_normal_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (p2x_comm_tx_pool_normal_r14_present) {
    j.start_array("p2x-CommTxPoolNormal-r14");
    for (const auto& e1 : p2x_comm_tx_pool_normal_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_comm_tx_pool_exceptional_r14_present) {
    j.write_fieldname("v2x-CommTxPoolExceptional-r14");
    v2x_comm_tx_pool_exceptional_r14.to_json(j);
  }
  if (v2x_res_sel_cfg_r14_present) {
    j.write_fieldname("v2x-ResourceSelectionConfig-r14");
    v2x_res_sel_cfg_r14.to_json(j);
  }
  if (zone_cfg_r14_present) {
    j.write_fieldname("zoneConfig-r14");
    zone_cfg_r14.to_json(j);
  }
  if (offset_dfn_r14_present) {
    j.write_int("offsetDFN-r14", offset_dfn_r14);
  }
  j.end_obj();
}

// SchedulingInfo-BR-r13 ::= SEQUENCE
SRSASN_CODE sched_info_br_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, si_nb_r13, (uint8_t)1u, (uint8_t)16u));
  HANDLE_CODE(si_tbs_r13.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sched_info_br_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(si_nb_r13, bref, (uint8_t)1u, (uint8_t)16u));
  HANDLE_CODE(si_tbs_r13.unpack(bref));

  return SRSASN_SUCCESS;
}
void sched_info_br_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("si-Narrowband-r13", si_nb_r13);
  j.write_str("si-TBS-r13", si_tbs_r13.to_string());
  j.end_obj();
}

// SpeedStateScaleFactors ::= SEQUENCE
SRSASN_CODE speed_state_scale_factors_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sf_medium.pack(bref));
  HANDLE_CODE(sf_high.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE speed_state_scale_factors_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sf_medium.unpack(bref));
  HANDLE_CODE(sf_high.unpack(bref));

  return SRSASN_SUCCESS;
}
void speed_state_scale_factors_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sf-Medium", sf_medium.to_string());
  j.write_str("sf-High", sf_high.to_string());
  j.end_obj();
}

// SystemInformationBlockType1-v1350-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1350_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cell_sel_info_ce1_r13_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (cell_sel_info_ce1_r13_present) {
    HANDLE_CODE(cell_sel_info_ce1_r13.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1350_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cell_sel_info_ce1_r13_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (cell_sel_info_ce1_r13_present) {
    HANDLE_CODE(cell_sel_info_ce1_r13.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1350_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cell_sel_info_ce1_r13_present) {
    j.write_fieldname("cellSelectionInfoCE1-r13");
    cell_sel_info_ce1_r13.to_json(j);
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType5-v13a0-IEs ::= SEQUENCE
SRSASN_CODE sib_type5_v13a0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v13a0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }
  if (inter_freq_carrier_freq_list_v13a0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_carrier_freq_list_v13a0, 1, 8));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type5_v13a0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v13a0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }
  if (inter_freq_carrier_freq_list_v13a0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_carrier_freq_list_v13a0, bref, 1, 8));
  }

  return SRSASN_SUCCESS;
}
void sib_type5_v13a0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (inter_freq_carrier_freq_list_v13a0_present) {
    j.start_array("interFreqCarrierFreqList-v13a0");
    for (const auto& e1 : inter_freq_carrier_freq_list_v13a0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// AC-BarringConfig1XRTT-r9 ::= SEQUENCE
SRSASN_CODE ac_barr_cfg1_xrtt_r9_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, ac_barr0to9_r9, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(pack_integer(bref, ac_barr10_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr11_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr12_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr13_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr14_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr15_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr_msg_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr_reg_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, ac_barr_emg_r9, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ac_barr_cfg1_xrtt_r9_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(ac_barr0to9_r9, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(unpack_integer(ac_barr10_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr11_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr12_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr13_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr14_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr15_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr_msg_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr_reg_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(ac_barr_emg_r9, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void ac_barr_cfg1_xrtt_r9_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("ac-Barring0to9-r9", ac_barr0to9_r9);
  j.write_int("ac-Barring10-r9", ac_barr10_r9);
  j.write_int("ac-Barring11-r9", ac_barr11_r9);
  j.write_int("ac-Barring12-r9", ac_barr12_r9);
  j.write_int("ac-Barring13-r9", ac_barr13_r9);
  j.write_int("ac-Barring14-r9", ac_barr14_r9);
  j.write_int("ac-Barring15-r9", ac_barr15_r9);
  j.write_int("ac-BarringMsg-r9", ac_barr_msg_r9);
  j.write_int("ac-BarringReg-r9", ac_barr_reg_r9);
  j.write_int("ac-BarringEmg-r9", ac_barr_emg_r9);
  j.end_obj();
}

// BarringPerACDC-Category-r13 ::= SEQUENCE
SRSASN_CODE barr_per_acdc_category_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(acdc_barr_cfg_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, acdc_category_r13, (uint8_t)1u, (uint8_t)16u));
  if (acdc_barr_cfg_r13_present) {
    HANDLE_CODE(acdc_barr_cfg_r13.ac_barr_factor_r13.pack(bref));
    HANDLE_CODE(acdc_barr_cfg_r13.ac_barr_time_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE barr_per_acdc_category_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(acdc_barr_cfg_r13_present, 1));

  HANDLE_CODE(unpack_integer(acdc_category_r13, bref, (uint8_t)1u, (uint8_t)16u));
  if (acdc_barr_cfg_r13_present) {
    HANDLE_CODE(acdc_barr_cfg_r13.ac_barr_factor_r13.unpack(bref));
    HANDLE_CODE(acdc_barr_cfg_r13.ac_barr_time_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void barr_per_acdc_category_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("acdc-Category-r13", acdc_category_r13);
  if (acdc_barr_cfg_r13_present) {
    j.write_fieldname("acdc-BarringConfig-r13");
    j.start_obj();
    j.write_str("ac-BarringFactor-r13", acdc_barr_cfg_r13.ac_barr_factor_r13.to_string());
    j.write_str("ac-BarringTime-r13", acdc_barr_cfg_r13.ac_barr_time_r13.to_string());
    j.end_obj();
  }
  j.end_obj();
}

// CSFB-RegistrationParam1XRTT ::= SEQUENCE
SRSASN_CODE csfb_regist_param1_xrtt_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sid.pack(bref));
  HANDLE_CODE(nid.pack(bref));
  HANDLE_CODE(bref.pack(multiple_sid, 1));
  HANDLE_CODE(bref.pack(multiple_nid, 1));
  HANDLE_CODE(bref.pack(home_reg, 1));
  HANDLE_CODE(bref.pack(foreign_sid_reg, 1));
  HANDLE_CODE(bref.pack(foreign_nid_reg, 1));
  HANDLE_CODE(bref.pack(param_reg, 1));
  HANDLE_CODE(bref.pack(pwr_up_reg, 1));
  HANDLE_CODE(regist_period.pack(bref));
  HANDLE_CODE(regist_zone.pack(bref));
  HANDLE_CODE(total_zone.pack(bref));
  HANDLE_CODE(zone_timer.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE csfb_regist_param1_xrtt_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sid.unpack(bref));
  HANDLE_CODE(nid.unpack(bref));
  HANDLE_CODE(bref.unpack(multiple_sid, 1));
  HANDLE_CODE(bref.unpack(multiple_nid, 1));
  HANDLE_CODE(bref.unpack(home_reg, 1));
  HANDLE_CODE(bref.unpack(foreign_sid_reg, 1));
  HANDLE_CODE(bref.unpack(foreign_nid_reg, 1));
  HANDLE_CODE(bref.unpack(param_reg, 1));
  HANDLE_CODE(bref.unpack(pwr_up_reg, 1));
  HANDLE_CODE(regist_period.unpack(bref));
  HANDLE_CODE(regist_zone.unpack(bref));
  HANDLE_CODE(total_zone.unpack(bref));
  HANDLE_CODE(zone_timer.unpack(bref));

  return SRSASN_SUCCESS;
}
void csfb_regist_param1_xrtt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sid", sid.to_string());
  j.write_str("nid", nid.to_string());
  j.write_bool("multipleSID", multiple_sid);
  j.write_bool("multipleNID", multiple_nid);
  j.write_bool("homeReg", home_reg);
  j.write_bool("foreignSIDReg", foreign_sid_reg);
  j.write_bool("foreignNIDReg", foreign_nid_reg);
  j.write_bool("parameterReg", param_reg);
  j.write_bool("powerUpReg", pwr_up_reg);
  j.write_str("registrationPeriod", regist_period.to_string());
  j.write_str("registrationZone", regist_zone.to_string());
  j.write_str("totalZone", total_zone.to_string());
  j.write_str("zoneTimer", zone_timer.to_string());
  j.end_obj();
}

// CSFB-RegistrationParam1XRTT-v920 ::= SEQUENCE
SRSASN_CODE csfb_regist_param1_xrtt_v920_s::pack(bit_ref& bref) const
{
  return SRSASN_SUCCESS;
}
SRSASN_CODE csfb_regist_param1_xrtt_v920_s::unpack(cbit_ref& bref)
{
  return SRSASN_SUCCESS;
}
void csfb_regist_param1_xrtt_v920_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("powerDownReg-r9", "true");
  j.end_obj();
}

// CellReselectionParametersCDMA2000-r11 ::= SEQUENCE
SRSASN_CODE cell_resel_params_cdma2000_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(t_resel_cdma2000_sf_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, band_class_list, 1, 32));
  HANDLE_CODE(pack_dyn_seq_of(bref, neigh_cell_list_r11, 1, 16));
  HANDLE_CODE(pack_integer(bref, t_resel_cdma2000, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_cdma2000_sf_present) {
    HANDLE_CODE(t_resel_cdma2000_sf.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_resel_params_cdma2000_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(t_resel_cdma2000_sf_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(band_class_list, bref, 1, 32));
  HANDLE_CODE(unpack_dyn_seq_of(neigh_cell_list_r11, bref, 1, 16));
  HANDLE_CODE(unpack_integer(t_resel_cdma2000, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_cdma2000_sf_present) {
    HANDLE_CODE(t_resel_cdma2000_sf.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_resel_params_cdma2000_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("bandClassList");
  for (const auto& e1 : band_class_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.start_array("neighCellList-r11");
  for (const auto& e1 : neigh_cell_list_r11) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_int("t-ReselectionCDMA2000", t_resel_cdma2000);
  if (t_resel_cdma2000_sf_present) {
    j.write_fieldname("t-ReselectionCDMA2000-SF");
    t_resel_cdma2000_sf.to_json(j);
  }
  j.end_obj();
}

// CellSelectionInfoCE-r13 ::= SEQUENCE
SRSASN_CODE cell_sel_info_ce_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(q_qual_min_rsrq_ce_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_ce_r13, (int8_t)-70, (int8_t)-22));
  if (q_qual_min_rsrq_ce_r13_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_rsrq_ce_r13, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_ce_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(q_qual_min_rsrq_ce_r13_present, 1));

  HANDLE_CODE(unpack_integer(q_rx_lev_min_ce_r13, bref, (int8_t)-70, (int8_t)-22));
  if (q_qual_min_rsrq_ce_r13_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_rsrq_ce_r13, bref, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
void cell_sel_info_ce_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-RxLevMinCE-r13", q_rx_lev_min_ce_r13);
  if (q_qual_min_rsrq_ce_r13_present) {
    j.write_int("q-QualMinRSRQ-CE-r13", q_qual_min_rsrq_ce_r13);
  }
  j.end_obj();
}

// CellSelectionInfoNFreq-r13 ::= SEQUENCE
SRSASN_CODE cell_sel_info_nfreq_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(q_rx_lev_min_offset_present, 1));

  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_r13, (int8_t)-70, (int8_t)-22));
  if (q_rx_lev_min_offset_present) {
    HANDLE_CODE(pack_integer(bref, q_rx_lev_min_offset, (uint8_t)1u, (uint8_t)8u));
  }
  HANDLE_CODE(q_hyst_r13.pack(bref));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_resel_r13, (int8_t)-70, (int8_t)-22));
  HANDLE_CODE(pack_integer(bref, t_resel_eutra_r13, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_nfreq_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(q_rx_lev_min_offset_present, 1));

  HANDLE_CODE(unpack_integer(q_rx_lev_min_r13, bref, (int8_t)-70, (int8_t)-22));
  if (q_rx_lev_min_offset_present) {
    HANDLE_CODE(unpack_integer(q_rx_lev_min_offset, bref, (uint8_t)1u, (uint8_t)8u));
  }
  HANDLE_CODE(q_hyst_r13.unpack(bref));
  HANDLE_CODE(unpack_integer(q_rx_lev_min_resel_r13, bref, (int8_t)-70, (int8_t)-22));
  HANDLE_CODE(unpack_integer(t_resel_eutra_r13, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void cell_sel_info_nfreq_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-RxLevMin-r13", q_rx_lev_min_r13);
  if (q_rx_lev_min_offset_present) {
    j.write_int("q-RxLevMinOffset", q_rx_lev_min_offset);
  }
  j.write_str("q-Hyst-r13", q_hyst_r13.to_string());
  j.write_int("q-RxLevMinReselection-r13", q_rx_lev_min_resel_r13);
  j.write_int("t-ReselectionEUTRA-r13", t_resel_eutra_r13);
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v10j0 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v10j0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_info_r10_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10j0_present, 1));

  if (freq_band_info_r10_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, freq_band_info_r10, 1, 8));
  }
  if (multi_band_info_list_v10j0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10j0, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v10j0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_info_r10_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10j0_present, 1));

  if (freq_band_info_r10_present) {
    HANDLE_CODE(unpack_dyn_seq_of(freq_band_info_r10, bref, 1, 8));
  }
  if (multi_band_info_list_v10j0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10j0, bref, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v10j0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_info_r10_present) {
    j.start_array("freqBandInfo-r10");
    for (const auto& e1 : freq_band_info_r10) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (multi_band_info_list_v10j0_present) {
    j.start_array("multiBandInfoList-v10j0");
    for (const auto& e1 : multi_band_info_list_v10j0) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  j.end_obj();
}

// InterFreqNeighCellInfo ::= SEQUENCE
SRSASN_CODE inter_freq_neigh_cell_info_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, pci, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(q_offset_cell.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_neigh_cell_info_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(pci, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(q_offset_cell.unpack(bref));

  return SRSASN_SUCCESS;
}
void inter_freq_neigh_cell_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId", pci);
  j.write_str("q-OffsetCell", q_offset_cell.to_string());
  j.end_obj();
}

// NS-PmaxValueNR-r15 ::= SEQUENCE
SRSASN_CODE ns_pmax_value_nr_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(add_pmax_nr_r15_present, 1));

  if (add_pmax_nr_r15_present) {
    HANDLE_CODE(pack_integer(bref, add_pmax_nr_r15, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(pack_integer(bref, add_spec_emission_nr_r15, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ns_pmax_value_nr_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(add_pmax_nr_r15_present, 1));

  if (add_pmax_nr_r15_present) {
    HANDLE_CODE(unpack_integer(add_pmax_nr_r15, bref, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(unpack_integer(add_spec_emission_nr_r15, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void ns_pmax_value_nr_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (add_pmax_nr_r15_present) {
    j.write_int("additionalPmaxNR-r15", add_pmax_nr_r15);
  }
  j.write_int("additionalSpectrumEmissionNR-r15", add_spec_emission_nr_r15);
  j.end_obj();
}

// PLMN-IdentityInfo2-r12 ::= CHOICE
void plmn_id_info2_r12_c::destroy_()
{
  switch (type_) {
    case types::plmn_id_r12:
      c.destroy<plmn_id_s>();
      break;
    default:
      break;
  }
}
void plmn_id_info2_r12_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::plmn_idx_r12:
      break;
    case types::plmn_id_r12:
      c.init<plmn_id_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info2_r12_c");
  }
}
plmn_id_info2_r12_c::plmn_id_info2_r12_c(const plmn_id_info2_r12_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::plmn_idx_r12:
      c.init(other.c.get<uint8_t>());
      break;
    case types::plmn_id_r12:
      c.init(other.c.get<plmn_id_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info2_r12_c");
  }
}
plmn_id_info2_r12_c& plmn_id_info2_r12_c::operator=(const plmn_id_info2_r12_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::plmn_idx_r12:
      c.set(other.c.get<uint8_t>());
      break;
    case types::plmn_id_r12:
      c.set(other.c.get<plmn_id_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info2_r12_c");
  }

  return *this;
}
void plmn_id_info2_r12_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::plmn_idx_r12:
      j.write_int("plmn-Index-r12", c.get<uint8_t>());
      break;
    case types::plmn_id_r12:
      j.write_fieldname("plmnIdentity-r12");
      c.get<plmn_id_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info2_r12_c");
  }
  j.end_obj();
}
SRSASN_CODE plmn_id_info2_r12_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::plmn_idx_r12:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)6u));
      break;
    case types::plmn_id_r12:
      HANDLE_CODE(c.get<plmn_id_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info2_r12_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_id_info2_r12_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::plmn_idx_r12:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)6u));
      break;
    case types::plmn_id_r12:
      HANDLE_CODE(c.get<plmn_id_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "plmn_id_info2_r12_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// PRACH-ParametersCE-r13 ::= SEQUENCE
SRSASN_CODE prach_params_ce_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(prach_start_sf_r13_present, 1));
  HANDLE_CODE(bref.pack(max_num_preamb_attempt_ce_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, prach_cfg_idx_r13, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(pack_integer(bref, prach_freq_offset_r13, (uint8_t)0u, (uint8_t)94u));
  if (prach_start_sf_r13_present) {
    HANDLE_CODE(prach_start_sf_r13.pack(bref));
  }
  if (max_num_preamb_attempt_ce_r13_present) {
    HANDLE_CODE(max_num_preamb_attempt_ce_r13.pack(bref));
  }
  HANDLE_CODE(num_repeat_per_preamb_attempt_r13.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, mpdcch_nbs_to_monitor_r13, 1, 2, integer_packer<uint8_t>(1, 16)));
  HANDLE_CODE(mpdcch_num_repeat_ra_r13.pack(bref));
  HANDLE_CODE(prach_hop_cfg_r13.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_params_ce_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(prach_start_sf_r13_present, 1));
  HANDLE_CODE(bref.unpack(max_num_preamb_attempt_ce_r13_present, 1));

  HANDLE_CODE(unpack_integer(prach_cfg_idx_r13, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(unpack_integer(prach_freq_offset_r13, bref, (uint8_t)0u, (uint8_t)94u));
  if (prach_start_sf_r13_present) {
    HANDLE_CODE(prach_start_sf_r13.unpack(bref));
  }
  if (max_num_preamb_attempt_ce_r13_present) {
    HANDLE_CODE(max_num_preamb_attempt_ce_r13.unpack(bref));
  }
  HANDLE_CODE(num_repeat_per_preamb_attempt_r13.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(mpdcch_nbs_to_monitor_r13, bref, 1, 2, integer_packer<uint8_t>(1, 16)));
  HANDLE_CODE(mpdcch_num_repeat_ra_r13.unpack(bref));
  HANDLE_CODE(prach_hop_cfg_r13.unpack(bref));

  return SRSASN_SUCCESS;
}
void prach_params_ce_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("prach-ConfigIndex-r13", prach_cfg_idx_r13);
  j.write_int("prach-FreqOffset-r13", prach_freq_offset_r13);
  if (prach_start_sf_r13_present) {
    j.write_str("prach-StartingSubframe-r13", prach_start_sf_r13.to_string());
  }
  if (max_num_preamb_attempt_ce_r13_present) {
    j.write_str("maxNumPreambleAttemptCE-r13", max_num_preamb_attempt_ce_r13.to_string());
  }
  j.write_str("numRepetitionPerPreambleAttempt-r13", num_repeat_per_preamb_attempt_r13.to_string());
  j.start_array("mpdcch-NarrowbandsToMonitor-r13");
  for (const auto& e1 : mpdcch_nbs_to_monitor_r13) {
    j.write_int(e1);
  }
  j.end_array();
  j.write_str("mpdcch-NumRepetition-RA-r13", mpdcch_num_repeat_ra_r13.to_string());
  j.write_str("prach-HoppingConfig-r13", prach_hop_cfg_r13.to_string());
  j.end_obj();
}

// PreRegistrationInfoHRPD ::= SEQUENCE
SRSASN_CODE pre_regist_info_hrpd_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(pre_regist_zone_id_present, 1));
  HANDLE_CODE(bref.pack(secondary_pre_regist_zone_id_list_present, 1));

  HANDLE_CODE(bref.pack(pre_regist_allowed, 1));
  if (pre_regist_zone_id_present) {
    HANDLE_CODE(pack_integer(bref, pre_regist_zone_id, (uint16_t)0u, (uint16_t)255u));
  }
  if (secondary_pre_regist_zone_id_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, secondary_pre_regist_zone_id_list, 1, 2, integer_packer<uint16_t>(0, 255)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pre_regist_info_hrpd_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(pre_regist_zone_id_present, 1));
  HANDLE_CODE(bref.unpack(secondary_pre_regist_zone_id_list_present, 1));

  HANDLE_CODE(bref.unpack(pre_regist_allowed, 1));
  if (pre_regist_zone_id_present) {
    HANDLE_CODE(unpack_integer(pre_regist_zone_id, bref, (uint16_t)0u, (uint16_t)255u));
  }
  if (secondary_pre_regist_zone_id_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(secondary_pre_regist_zone_id_list, bref, 1, 2, integer_packer<uint16_t>(0, 255)));
  }

  return SRSASN_SUCCESS;
}
void pre_regist_info_hrpd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_bool("preRegistrationAllowed", pre_regist_allowed);
  if (pre_regist_zone_id_present) {
    j.write_int("preRegistrationZoneId", pre_regist_zone_id);
  }
  if (secondary_pre_regist_zone_id_list_present) {
    j.start_array("secondaryPreRegistrationZoneIdList");
    for (const auto& e1 : secondary_pre_regist_zone_id_list) {
      j.write_int(e1);
    }
    j.end_array();
  }
  j.end_obj();
}

// RACH-CE-LevelInfo-r13 ::= SEQUENCE
SRSASN_CODE rach_ce_level_info_r13_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, preamb_map_info_r13.first_preamb_r13, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(pack_integer(bref, preamb_map_info_r13.last_preamb_r13, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(ra_resp_win_size_r13.pack(bref));
  HANDLE_CODE(mac_contention_resolution_timer_r13.pack(bref));
  HANDLE_CODE(rar_hop_cfg_r13.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= edt_params_r15.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(edt_params_r15.is_present(), 1));
      if (edt_params_r15.is_present()) {
        HANDLE_CODE(bref.pack(edt_params_r15->mac_contention_resolution_timer_r15_present, 1));
        HANDLE_CODE(pack_integer(bref, edt_params_r15->edt_last_preamb_r15, (uint8_t)0u, (uint8_t)63u));
        HANDLE_CODE(bref.pack(edt_params_r15->edt_small_tbs_enabled_r15, 1));
        HANDLE_CODE(edt_params_r15->edt_tbs_r15.pack(bref));
        if (edt_params_r15->mac_contention_resolution_timer_r15_present) {
          HANDLE_CODE(edt_params_r15->mac_contention_resolution_timer_r15.pack(bref));
        }
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rach_ce_level_info_r13_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(preamb_map_info_r13.first_preamb_r13, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(unpack_integer(preamb_map_info_r13.last_preamb_r13, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(ra_resp_win_size_r13.unpack(bref));
  HANDLE_CODE(mac_contention_resolution_timer_r13.unpack(bref));
  HANDLE_CODE(rar_hop_cfg_r13.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool edt_params_r15_present;
      HANDLE_CODE(bref.unpack(edt_params_r15_present, 1));
      edt_params_r15.set_present(edt_params_r15_present);
      if (edt_params_r15.is_present()) {
        HANDLE_CODE(bref.unpack(edt_params_r15->mac_contention_resolution_timer_r15_present, 1));
        HANDLE_CODE(unpack_integer(edt_params_r15->edt_last_preamb_r15, bref, (uint8_t)0u, (uint8_t)63u));
        HANDLE_CODE(bref.unpack(edt_params_r15->edt_small_tbs_enabled_r15, 1));
        HANDLE_CODE(edt_params_r15->edt_tbs_r15.unpack(bref));
        if (edt_params_r15->mac_contention_resolution_timer_r15_present) {
          HANDLE_CODE(edt_params_r15->mac_contention_resolution_timer_r15.unpack(bref));
        }
      }
    }
  }
  return SRSASN_SUCCESS;
}
void rach_ce_level_info_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("preambleMappingInfo-r13");
  j.start_obj();
  j.write_int("firstPreamble-r13", preamb_map_info_r13.first_preamb_r13);
  j.write_int("lastPreamble-r13", preamb_map_info_r13.last_preamb_r13);
  j.end_obj();
  j.write_str("ra-ResponseWindowSize-r13", ra_resp_win_size_r13.to_string());
  j.write_str("mac-ContentionResolutionTimer-r13", mac_contention_resolution_timer_r13.to_string());
  j.write_str("rar-HoppingConfig-r13", rar_hop_cfg_r13.to_string());
  if (ext) {
    if (edt_params_r15.is_present()) {
      j.write_fieldname("edt-Parameters-r15");
      j.start_obj();
      j.write_int("edt-LastPreamble-r15", edt_params_r15->edt_last_preamb_r15);
      j.write_bool("edt-SmallTBS-Enabled-r15", edt_params_r15->edt_small_tbs_enabled_r15);
      j.write_str("edt-TBS-r15", edt_params_r15->edt_tbs_r15.to_string());
      if (edt_params_r15->mac_contention_resolution_timer_r15_present) {
        j.write_str("mac-ContentionResolutionTimer-r15",
                    edt_params_r15->mac_contention_resolution_timer_r15.to_string());
      }
      j.end_obj();
    }
  }
  j.end_obj();
}

// SL-AllowedCarrierFreqList-r15 ::= SEQUENCE
SRSASN_CODE sl_allowed_carrier_freq_list_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, allowed_carrier_freq_set1, 1, 8, integer_packer<uint32_t>(0, 262143)));
  HANDLE_CODE(pack_dyn_seq_of(bref, allowed_carrier_freq_set2, 1, 8, integer_packer<uint32_t>(0, 262143)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_allowed_carrier_freq_list_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(allowed_carrier_freq_set1, bref, 1, 8, integer_packer<uint32_t>(0, 262143)));
  HANDLE_CODE(unpack_dyn_seq_of(allowed_carrier_freq_set2, bref, 1, 8, integer_packer<uint32_t>(0, 262143)));

  return SRSASN_SUCCESS;
}
void sl_allowed_carrier_freq_list_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("allowedCarrierFreqSet1");
  for (const auto& e1 : allowed_carrier_freq_set1) {
    j.write_int(e1);
  }
  j.end_array();
  j.start_array("allowedCarrierFreqSet2");
  for (const auto& e1 : allowed_carrier_freq_set2) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// SL-DiscTxResourcesInterFreq-r13 ::= CHOICE
void sl_disc_tx_res_inter_freq_r13_c::set(types::options e)
{
  type_ = e;
}
void sl_disc_tx_res_inter_freq_r13_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::acquire_si_from_carrier_r13:
      break;
    case types::disc_tx_pool_common_r13:
      j.start_array("discTxPoolCommon-r13");
      for (const auto& e1 : c) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::request_ded_r13:
      break;
    case types::no_tx_on_carrier_r13:
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_tx_res_inter_freq_r13_c");
  }
  j.end_obj();
}
SRSASN_CODE sl_disc_tx_res_inter_freq_r13_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::acquire_si_from_carrier_r13:
      break;
    case types::disc_tx_pool_common_r13:
      HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 4));
      break;
    case types::request_ded_r13:
      break;
    case types::no_tx_on_carrier_r13:
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_tx_res_inter_freq_r13_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_tx_res_inter_freq_r13_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::acquire_si_from_carrier_r13:
      break;
    case types::disc_tx_pool_common_r13:
      HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 4));
      break;
    case types::request_ded_r13:
      break;
    case types::no_tx_on_carrier_r13:
      break;
    default:
      log_invalid_choice_id(type_, "sl_disc_tx_res_inter_freq_r13_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType1-v1320-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1320_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_hop_params_dl_r13_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_hop_params_dl_r13_present) {
    HANDLE_CODE(bref.pack(freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13_present, 1));
    HANDLE_CODE(bref.pack(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13_present, 1));
    HANDLE_CODE(bref.pack(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13_present, 1));
    HANDLE_CODE(bref.pack(freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13_present, 1));
    if (freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13_present) {
      HANDLE_CODE(freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13.pack(bref));
    }
    if (freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13_present) {
      HANDLE_CODE(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13.pack(bref));
    }
    if (freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13_present) {
      HANDLE_CODE(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13.pack(bref));
    }
    if (freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13_present) {
      HANDLE_CODE(pack_integer(bref, freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13, (uint8_t)1u, (uint8_t)16u));
    }
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1320_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_hop_params_dl_r13_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_hop_params_dl_r13_present) {
    HANDLE_CODE(bref.unpack(freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13_present, 1));
    HANDLE_CODE(bref.unpack(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13_present, 1));
    HANDLE_CODE(bref.unpack(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13_present, 1));
    HANDLE_CODE(bref.unpack(freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13_present, 1));
    if (freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13_present) {
      HANDLE_CODE(freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13.unpack(bref));
    }
    if (freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13_present) {
      HANDLE_CODE(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13.unpack(bref));
    }
    if (freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13_present) {
      HANDLE_CODE(freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13.unpack(bref));
    }
    if (freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13_present) {
      HANDLE_CODE(unpack_integer(freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13, bref, (uint8_t)1u, (uint8_t)16u));
    }
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1320_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_hop_params_dl_r13_present) {
    j.write_fieldname("freqHoppingParametersDL-r13");
    j.start_obj();
    if (freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13_present) {
      j.write_str("mpdcch-pdsch-HoppingNB-r13", freq_hop_params_dl_r13.mpdcch_pdsch_hop_nb_r13.to_string());
    }
    if (freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13_present) {
      j.write_fieldname("interval-DLHoppingConfigCommonModeA-r13");
      freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_a_r13.to_json(j);
    }
    if (freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13_present) {
      j.write_fieldname("interval-DLHoppingConfigCommonModeB-r13");
      freq_hop_params_dl_r13.interv_dl_hop_cfg_common_mode_b_r13.to_json(j);
    }
    if (freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13_present) {
      j.write_int("mpdcch-pdsch-HoppingOffset-r13", freq_hop_params_dl_r13.mpdcch_pdsch_hop_offset_r13);
    }
    j.end_obj();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

void sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::destroy_() {}
void sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::
    interv_dl_hop_cfg_common_mode_a_r13_c_(
        const sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::interv_fdd_r13:
      c.init(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.init(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_");
  }
}
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_&
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::
operator=(const sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::interv_fdd_r13:
      c.set(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.set(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_");
  }

  return *this;
}
void sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::to_json(
    json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::interv_fdd_r13:
      j.write_str("interval-FDD-r13", c.get<interv_fdd_r13_e_>().to_string());
      break;
    case types::interv_tdd_r13:
      j.write_str("interval-TDD-r13", c.get<interv_tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().pack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().unpack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_a_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::destroy_() {}
void sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::
    interv_dl_hop_cfg_common_mode_b_r13_c_(
        const sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::interv_fdd_r13:
      c.init(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.init(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_");
  }
}
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_&
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::
operator=(const sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::interv_fdd_r13:
      c.set(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.set(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_");
  }

  return *this;
}
void sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::to_json(
    json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::interv_fdd_r13:
      j.write_str("interval-FDD-r13", c.get<interv_fdd_r13_e_>().to_string());
      break;
    case types::interv_tdd_r13:
      j.write_str("interval-TDD-r13", c.get<interv_tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().pack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE
sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().unpack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_,
                            "sib_type1_v1320_ies_s::freq_hop_params_dl_r13_s_::interv_dl_hop_cfg_common_mode_b_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType2-v10m0-IEs ::= SEQUENCE
SRSASN_CODE sib_type2_v10m0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_info_v10l0_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10l0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_info_v10l0_present) {
    HANDLE_CODE(pack_integer(bref, freq_info_v10l0.add_spec_emission_v10l0, (uint16_t)33u, (uint16_t)288u));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10l0, 1, 8, integer_packer<uint16_t>(33, 288)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type2_v10m0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_info_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_info_v10l0_present) {
    HANDLE_CODE(unpack_integer(freq_info_v10l0.add_spec_emission_v10l0, bref, (uint16_t)33u, (uint16_t)288u));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10l0, bref, 1, 8, integer_packer<uint16_t>(33, 288)));
  }

  return SRSASN_SUCCESS;
}
void sib_type2_v10m0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_info_v10l0_present) {
    j.write_fieldname("freqInfo-v10l0");
    j.start_obj();
    j.write_int("additionalSpectrumEmission-v10l0", freq_info_v10l0.add_spec_emission_v10l0);
    j.end_obj();
  }
  if (multi_band_info_list_v10l0_present) {
    j.start_array("multiBandInfoList-v10l0");
    for (const auto& e1 : multi_band_info_list_v10l0) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// SystemInformationBlockType5-v10l0-IEs ::= SEQUENCE
SRSASN_CODE sib_type5_v10l0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v10l0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_carrier_freq_list_v10l0, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type5_v10l0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_carrier_freq_list_v10l0, bref, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type5_v10l0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (inter_freq_carrier_freq_list_v10l0_present) {
    j.start_array("interFreqCarrierFreqList-v10l0");
    for (const auto& e1 : inter_freq_carrier_freq_list_v10l0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemTimeInfoCDMA2000 ::= SEQUENCE
SRSASN_CODE sys_time_info_cdma2000_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cdma_eutra_synchronisation, 1));
  HANDLE_CODE(cdma_sys_time.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_time_info_cdma2000_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cdma_eutra_synchronisation, 1));
  HANDLE_CODE(cdma_sys_time.unpack(bref));

  return SRSASN_SUCCESS;
}
void sys_time_info_cdma2000_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_bool("cdma-EUTRA-Synchronisation", cdma_eutra_synchronisation);
  j.write_fieldname("cdma-SystemTime");
  cdma_sys_time.to_json(j);
  j.end_obj();
}

void sys_time_info_cdma2000_s::cdma_sys_time_c_::destroy_()
{
  switch (type_) {
    case types::sync_sys_time:
      c.destroy<fixed_bitstring<39> >();
      break;
    case types::async_sys_time:
      c.destroy<fixed_bitstring<49> >();
      break;
    default:
      break;
  }
}
void sys_time_info_cdma2000_s::cdma_sys_time_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sync_sys_time:
      c.init<fixed_bitstring<39> >();
      break;
    case types::async_sys_time:
      c.init<fixed_bitstring<49> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_time_info_cdma2000_s::cdma_sys_time_c_");
  }
}
sys_time_info_cdma2000_s::cdma_sys_time_c_::cdma_sys_time_c_(const sys_time_info_cdma2000_s::cdma_sys_time_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sync_sys_time:
      c.init(other.c.get<fixed_bitstring<39> >());
      break;
    case types::async_sys_time:
      c.init(other.c.get<fixed_bitstring<49> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_time_info_cdma2000_s::cdma_sys_time_c_");
  }
}
sys_time_info_cdma2000_s::cdma_sys_time_c_& sys_time_info_cdma2000_s::cdma_sys_time_c_::
                                            operator=(const sys_time_info_cdma2000_s::cdma_sys_time_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sync_sys_time:
      c.set(other.c.get<fixed_bitstring<39> >());
      break;
    case types::async_sys_time:
      c.set(other.c.get<fixed_bitstring<49> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_time_info_cdma2000_s::cdma_sys_time_c_");
  }

  return *this;
}
void sys_time_info_cdma2000_s::cdma_sys_time_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sync_sys_time:
      j.write_str("synchronousSystemTime", c.get<fixed_bitstring<39> >().to_string());
      break;
    case types::async_sys_time:
      j.write_str("asynchronousSystemTime", c.get<fixed_bitstring<49> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "sys_time_info_cdma2000_s::cdma_sys_time_c_");
  }
  j.end_obj();
}
SRSASN_CODE sys_time_info_cdma2000_s::cdma_sys_time_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sync_sys_time:
      HANDLE_CODE(c.get<fixed_bitstring<39> >().pack(bref));
      break;
    case types::async_sys_time:
      HANDLE_CODE(c.get<fixed_bitstring<49> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sys_time_info_cdma2000_s::cdma_sys_time_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_time_info_cdma2000_s::cdma_sys_time_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sync_sys_time:
      HANDLE_CODE(c.get<fixed_bitstring<39> >().unpack(bref));
      break;
    case types::async_sys_time:
      HANDLE_CODE(c.get<fixed_bitstring<49> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sys_time_info_cdma2000_s::cdma_sys_time_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// UAC-BarringPerCat-r15 ::= SEQUENCE
SRSASN_CODE uac_barr_per_cat_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, access_category_r15, (uint8_t)1u, (uint8_t)63u));
  HANDLE_CODE(pack_integer(bref, uac_barr_info_set_idx_r15, (uint8_t)1u, (uint8_t)8u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE uac_barr_per_cat_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(access_category_r15, bref, (uint8_t)1u, (uint8_t)63u));
  HANDLE_CODE(unpack_integer(uac_barr_info_set_idx_r15, bref, (uint8_t)1u, (uint8_t)8u));

  return SRSASN_SUCCESS;
}
void uac_barr_per_cat_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("accessCategory-r15", access_category_r15);
  j.write_int("uac-barringInfoSetIndex-r15", uac_barr_info_set_idx_r15);
  j.end_obj();
}

// AC-BarringConfig ::= SEQUENCE
SRSASN_CODE ac_barr_cfg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(ac_barr_factor.pack(bref));
  HANDLE_CODE(ac_barr_time.pack(bref));
  HANDLE_CODE(ac_barr_for_special_ac.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ac_barr_cfg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(ac_barr_factor.unpack(bref));
  HANDLE_CODE(ac_barr_time.unpack(bref));
  HANDLE_CODE(ac_barr_for_special_ac.unpack(bref));

  return SRSASN_SUCCESS;
}
void ac_barr_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("ac-BarringFactor", ac_barr_factor.to_string());
  j.write_str("ac-BarringTime", ac_barr_time.to_string());
  j.write_str("ac-BarringForSpecialAC", ac_barr_for_special_ac.to_string());
  j.end_obj();
}

// CarrierFreqsGERAN ::= SEQUENCE
SRSASN_CODE carrier_freqs_geran_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, start_arfcn, (uint16_t)0u, (uint16_t)1023u));
  HANDLE_CODE(band_ind.pack(bref));
  HANDLE_CODE(following_arfcns.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freqs_geran_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(start_arfcn, bref, (uint16_t)0u, (uint16_t)1023u));
  HANDLE_CODE(band_ind.unpack(bref));
  HANDLE_CODE(following_arfcns.unpack(bref));

  return SRSASN_SUCCESS;
}
void carrier_freqs_geran_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("startingARFCN", start_arfcn);
  j.write_str("bandIndicator", band_ind.to_string());
  j.write_fieldname("followingARFCNs");
  following_arfcns.to_json(j);
  j.end_obj();
}

void carrier_freqs_geran_s::following_arfcns_c_::destroy_()
{
  switch (type_) {
    case types::explicit_list_of_arfcns:
      c.destroy<explicit_list_of_arfcns_l>();
      break;
    case types::equally_spaced_arfcns:
      c.destroy<equally_spaced_arfcns_s_>();
      break;
    case types::variable_bit_map_of_arfcns:
      c.destroy<dyn_octstring>();
      break;
    default:
      break;
  }
}
void carrier_freqs_geran_s::following_arfcns_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::explicit_list_of_arfcns:
      c.init<explicit_list_of_arfcns_l>();
      break;
    case types::equally_spaced_arfcns:
      c.init<equally_spaced_arfcns_s_>();
      break;
    case types::variable_bit_map_of_arfcns:
      c.init<dyn_octstring>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "carrier_freqs_geran_s::following_arfcns_c_");
  }
}
carrier_freqs_geran_s::following_arfcns_c_::following_arfcns_c_(const carrier_freqs_geran_s::following_arfcns_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::explicit_list_of_arfcns:
      c.init(other.c.get<explicit_list_of_arfcns_l>());
      break;
    case types::equally_spaced_arfcns:
      c.init(other.c.get<equally_spaced_arfcns_s_>());
      break;
    case types::variable_bit_map_of_arfcns:
      c.init(other.c.get<dyn_octstring>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "carrier_freqs_geran_s::following_arfcns_c_");
  }
}
carrier_freqs_geran_s::following_arfcns_c_& carrier_freqs_geran_s::following_arfcns_c_::
                                            operator=(const carrier_freqs_geran_s::following_arfcns_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::explicit_list_of_arfcns:
      c.set(other.c.get<explicit_list_of_arfcns_l>());
      break;
    case types::equally_spaced_arfcns:
      c.set(other.c.get<equally_spaced_arfcns_s_>());
      break;
    case types::variable_bit_map_of_arfcns:
      c.set(other.c.get<dyn_octstring>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "carrier_freqs_geran_s::following_arfcns_c_");
  }

  return *this;
}
void carrier_freqs_geran_s::following_arfcns_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::explicit_list_of_arfcns:
      j.start_array("explicitListOfARFCNs");
      for (const auto& e1 : c.get<explicit_list_of_arfcns_l>()) {
        j.write_int(e1);
      }
      j.end_array();
      break;
    case types::equally_spaced_arfcns:
      j.write_fieldname("equallySpacedARFCNs");
      j.start_obj();
      j.write_int("arfcn-Spacing", c.get<equally_spaced_arfcns_s_>().arfcn_spacing);
      j.write_int("numberOfFollowingARFCNs", c.get<equally_spaced_arfcns_s_>().nof_following_arfcns);
      j.end_obj();
      break;
    case types::variable_bit_map_of_arfcns:
      j.write_str("variableBitMapOfARFCNs", c.get<dyn_octstring>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "carrier_freqs_geran_s::following_arfcns_c_");
  }
  j.end_obj();
}
SRSASN_CODE carrier_freqs_geran_s::following_arfcns_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::explicit_list_of_arfcns:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<explicit_list_of_arfcns_l>(), 0, 31, integer_packer<uint16_t>(0, 1023)));
      break;
    case types::equally_spaced_arfcns:
      HANDLE_CODE(pack_integer(bref, c.get<equally_spaced_arfcns_s_>().arfcn_spacing, (uint8_t)1u, (uint8_t)8u));
      HANDLE_CODE(
          pack_integer(bref, c.get<equally_spaced_arfcns_s_>().nof_following_arfcns, (uint8_t)0u, (uint8_t)31u));
      break;
    case types::variable_bit_map_of_arfcns:
      HANDLE_CODE(c.get<dyn_octstring>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "carrier_freqs_geran_s::following_arfcns_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freqs_geran_s::following_arfcns_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::explicit_list_of_arfcns:
      HANDLE_CODE(
          unpack_dyn_seq_of(c.get<explicit_list_of_arfcns_l>(), bref, 0, 31, integer_packer<uint16_t>(0, 1023)));
      break;
    case types::equally_spaced_arfcns:
      HANDLE_CODE(unpack_integer(c.get<equally_spaced_arfcns_s_>().arfcn_spacing, bref, (uint8_t)1u, (uint8_t)8u));
      HANDLE_CODE(
          unpack_integer(c.get<equally_spaced_arfcns_s_>().nof_following_arfcns, bref, (uint8_t)0u, (uint8_t)31u));
      break;
    case types::variable_bit_map_of_arfcns:
      HANDLE_CODE(c.get<dyn_octstring>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "carrier_freqs_geran_s::following_arfcns_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CellSelectionInfo-v1250 ::= SEQUENCE
SRSASN_CODE cell_sel_info_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, q_qual_min_rsrq_on_all_symbols_r12, (int8_t)-34, (int8_t)-3));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(q_qual_min_rsrq_on_all_symbols_r12, bref, (int8_t)-34, (int8_t)-3));

  return SRSASN_SUCCESS;
}
void cell_sel_info_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-QualMinRSRQ-OnAllSymbols-r12", q_qual_min_rsrq_on_all_symbols_r12);
  j.end_obj();
}

// DeltaFList-PUCCH ::= SEQUENCE
SRSASN_CODE delta_flist_pucch_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(delta_f_pucch_format1.pack(bref));
  HANDLE_CODE(delta_f_pucch_format1b.pack(bref));
  HANDLE_CODE(delta_f_pucch_format2.pack(bref));
  HANDLE_CODE(delta_f_pucch_format2a.pack(bref));
  HANDLE_CODE(delta_f_pucch_format2b.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE delta_flist_pucch_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(delta_f_pucch_format1.unpack(bref));
  HANDLE_CODE(delta_f_pucch_format1b.unpack(bref));
  HANDLE_CODE(delta_f_pucch_format2.unpack(bref));
  HANDLE_CODE(delta_f_pucch_format2a.unpack(bref));
  HANDLE_CODE(delta_f_pucch_format2b.unpack(bref));

  return SRSASN_SUCCESS;
}
void delta_flist_pucch_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("deltaF-PUCCH-Format1", delta_f_pucch_format1.to_string());
  j.write_str("deltaF-PUCCH-Format1b", delta_f_pucch_format1b.to_string());
  j.write_str("deltaF-PUCCH-Format2", delta_f_pucch_format2.to_string());
  j.write_str("deltaF-PUCCH-Format2a", delta_f_pucch_format2a.to_string());
  j.write_str("deltaF-PUCCH-Format2b", delta_f_pucch_format2b.to_string());
  j.end_obj();
}

// EDT-PRACH-ParametersCE-r15 ::= SEQUENCE
SRSASN_CODE edt_prach_params_ce_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(edt_prach_params_ce_r15_present, 1));

  if (edt_prach_params_ce_r15_present) {
    HANDLE_CODE(bref.pack(edt_prach_params_ce_r15.prach_start_sf_r15_present, 1));
    HANDLE_CODE(pack_integer(bref, edt_prach_params_ce_r15.prach_cfg_idx_r15, (uint8_t)0u, (uint8_t)63u));
    HANDLE_CODE(pack_integer(bref, edt_prach_params_ce_r15.prach_freq_offset_r15, (uint8_t)0u, (uint8_t)94u));
    if (edt_prach_params_ce_r15.prach_start_sf_r15_present) {
      HANDLE_CODE(edt_prach_params_ce_r15.prach_start_sf_r15.pack(bref));
    }
    HANDLE_CODE(
        pack_dyn_seq_of(bref, edt_prach_params_ce_r15.mpdcch_nbs_to_monitor_r15, 1, 2, integer_packer<uint8_t>(1, 16)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE edt_prach_params_ce_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(edt_prach_params_ce_r15_present, 1));

  if (edt_prach_params_ce_r15_present) {
    HANDLE_CODE(bref.unpack(edt_prach_params_ce_r15.prach_start_sf_r15_present, 1));
    HANDLE_CODE(unpack_integer(edt_prach_params_ce_r15.prach_cfg_idx_r15, bref, (uint8_t)0u, (uint8_t)63u));
    HANDLE_CODE(unpack_integer(edt_prach_params_ce_r15.prach_freq_offset_r15, bref, (uint8_t)0u, (uint8_t)94u));
    if (edt_prach_params_ce_r15.prach_start_sf_r15_present) {
      HANDLE_CODE(edt_prach_params_ce_r15.prach_start_sf_r15.unpack(bref));
    }
    HANDLE_CODE(unpack_dyn_seq_of(
        edt_prach_params_ce_r15.mpdcch_nbs_to_monitor_r15, bref, 1, 2, integer_packer<uint8_t>(1, 16)));
  }

  return SRSASN_SUCCESS;
}
void edt_prach_params_ce_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (edt_prach_params_ce_r15_present) {
    j.write_fieldname("edt-PRACH-ParametersCE-r15");
    j.start_obj();
    j.write_int("prach-ConfigIndex-r15", edt_prach_params_ce_r15.prach_cfg_idx_r15);
    j.write_int("prach-FreqOffset-r15", edt_prach_params_ce_r15.prach_freq_offset_r15);
    if (edt_prach_params_ce_r15.prach_start_sf_r15_present) {
      j.write_str("prach-StartingSubframe-r15", edt_prach_params_ce_r15.prach_start_sf_r15.to_string());
    }
    j.start_array("mpdcch-NarrowbandsToMonitor-r15");
    for (const auto& e1 : edt_prach_params_ce_r15.mpdcch_nbs_to_monitor_r15) {
      j.write_int(e1);
    }
    j.end_array();
    j.end_obj();
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v9e0 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v9e0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(dl_carrier_freq_v9e0_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v9e0_present, 1));

  if (dl_carrier_freq_v9e0_present) {
    HANDLE_CODE(pack_integer(bref, dl_carrier_freq_v9e0, (uint32_t)65536u, (uint32_t)262143u));
  }
  if (multi_band_info_list_v9e0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v9e0, 1, 8));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v9e0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(dl_carrier_freq_v9e0_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v9e0_present, 1));

  if (dl_carrier_freq_v9e0_present) {
    HANDLE_CODE(unpack_integer(dl_carrier_freq_v9e0, bref, (uint32_t)65536u, (uint32_t)262143u));
  }
  if (multi_band_info_list_v9e0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v9e0, bref, 1, 8));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v9e0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dl_carrier_freq_v9e0_present) {
    j.write_int("dl-CarrierFreq-v9e0", dl_carrier_freq_v9e0);
  }
  if (multi_band_info_list_v9e0_present) {
    j.start_array("multiBandInfoList-v9e0");
    for (const auto& e1 : multi_band_info_list_v9e0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// MTC-SSB-NR-r15 ::= SEQUENCE
SRSASN_CODE mtc_ssb_nr_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(periodicity_and_offset_r15.pack(bref));
  HANDLE_CODE(ssb_dur_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mtc_ssb_nr_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(periodicity_and_offset_r15.unpack(bref));
  HANDLE_CODE(ssb_dur_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void mtc_ssb_nr_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("periodicityAndOffset-r15");
  periodicity_and_offset_r15.to_json(j);
  j.write_str("ssb-Duration-r15", ssb_dur_r15.to_string());
  j.end_obj();
}

void mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::destroy_() {}
void mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::periodicity_and_offset_r15_c_(
    const mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sf5_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf10_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf20_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf40_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf80_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf160_r15:
      c.init(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_");
  }
}
mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_& mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::
                                                 operator=(const mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sf5_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf10_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf20_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf40_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf80_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf160_r15:
      c.set(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_");
  }

  return *this;
}
void mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sf5_r15:
      j.write_int("sf5-r15", c.get<uint8_t>());
      break;
    case types::sf10_r15:
      j.write_int("sf10-r15", c.get<uint8_t>());
      break;
    case types::sf20_r15:
      j.write_int("sf20-r15", c.get<uint8_t>());
      break;
    case types::sf40_r15:
      j.write_int("sf40-r15", c.get<uint8_t>());
      break;
    case types::sf80_r15:
      j.write_int("sf80-r15", c.get<uint8_t>());
      break;
    case types::sf160_r15:
      j.write_int("sf160-r15", c.get<uint8_t>());
      break;
    default:
      log_invalid_choice_id(type_, "mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sf5_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)4u));
      break;
    case types::sf10_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)9u));
      break;
    case types::sf20_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)19u));
      break;
    case types::sf40_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)39u));
      break;
    case types::sf80_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)79u));
      break;
    case types::sf160_r15:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)159u));
      break;
    default:
      log_invalid_choice_id(type_, "mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sf5_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)4u));
      break;
    case types::sf10_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)9u));
      break;
    case types::sf20_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)19u));
      break;
    case types::sf40_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)39u));
      break;
    case types::sf80_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)79u));
      break;
    case types::sf160_r15:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)159u));
      break;
    default:
      log_invalid_choice_id(type_, "mtc_ssb_nr_r15_s::periodicity_and_offset_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// MeasIdleCarrierEUTRA-r15 ::= SEQUENCE
SRSASN_CODE meas_idle_carrier_eutra_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(validity_area_r15_present, 1));
  HANDLE_CODE(bref.pack(meas_cell_list_r15_present, 1));
  HANDLE_CODE(bref.pack(quality_thres_r15_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq_r15, (uint32_t)0u, (uint32_t)262143u));
  HANDLE_CODE(allowed_meas_bw_r15.pack(bref));
  if (validity_area_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, validity_area_r15, 1, 8));
  }
  if (meas_cell_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, meas_cell_list_r15, 1, 8));
  }
  HANDLE_CODE(report_quantities.pack(bref));
  if (quality_thres_r15_present) {
    HANDLE_CODE(bref.pack(quality_thres_r15.idle_rsrp_thres_r15_present, 1));
    HANDLE_CODE(bref.pack(quality_thres_r15.idle_rsrq_thres_r15_present, 1));
    if (quality_thres_r15.idle_rsrp_thres_r15_present) {
      HANDLE_CODE(pack_integer(bref, quality_thres_r15.idle_rsrp_thres_r15, (uint8_t)0u, (uint8_t)97u));
    }
    if (quality_thres_r15.idle_rsrq_thres_r15_present) {
      HANDLE_CODE(pack_integer(bref, quality_thres_r15.idle_rsrq_thres_r15, (int8_t)-30, (int8_t)46));
    }
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE meas_idle_carrier_eutra_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(validity_area_r15_present, 1));
  HANDLE_CODE(bref.unpack(meas_cell_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(quality_thres_r15_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq_r15, bref, (uint32_t)0u, (uint32_t)262143u));
  HANDLE_CODE(allowed_meas_bw_r15.unpack(bref));
  if (validity_area_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(validity_area_r15, bref, 1, 8));
  }
  if (meas_cell_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(meas_cell_list_r15, bref, 1, 8));
  }
  HANDLE_CODE(report_quantities.unpack(bref));
  if (quality_thres_r15_present) {
    HANDLE_CODE(bref.unpack(quality_thres_r15.idle_rsrp_thres_r15_present, 1));
    HANDLE_CODE(bref.unpack(quality_thres_r15.idle_rsrq_thres_r15_present, 1));
    if (quality_thres_r15.idle_rsrp_thres_r15_present) {
      HANDLE_CODE(unpack_integer(quality_thres_r15.idle_rsrp_thres_r15, bref, (uint8_t)0u, (uint8_t)97u));
    }
    if (quality_thres_r15.idle_rsrq_thres_r15_present) {
      HANDLE_CODE(unpack_integer(quality_thres_r15.idle_rsrq_thres_r15, bref, (int8_t)-30, (int8_t)46));
    }
  }

  return SRSASN_SUCCESS;
}
void meas_idle_carrier_eutra_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq-r15", carrier_freq_r15);
  j.write_str("allowedMeasBandwidth-r15", allowed_meas_bw_r15.to_string());
  if (validity_area_r15_present) {
    j.start_array("validityArea-r15");
    for (const auto& e1 : validity_area_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (meas_cell_list_r15_present) {
    j.start_array("measCellList-r15");
    for (const auto& e1 : meas_cell_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.write_str("reportQuantities", report_quantities.to_string());
  if (quality_thres_r15_present) {
    j.write_fieldname("qualityThreshold-r15");
    j.start_obj();
    if (quality_thres_r15.idle_rsrp_thres_r15_present) {
      j.write_int("idleRSRP-Threshold-r15", quality_thres_r15.idle_rsrp_thres_r15);
    }
    if (quality_thres_r15.idle_rsrq_thres_r15_present) {
      j.write_int("idleRSRQ-Threshold-r15", quality_thres_r15.idle_rsrq_thres_r15);
    }
    j.end_obj();
  }
  j.end_obj();
}

// NeighCellCDMA2000 ::= SEQUENCE
SRSASN_CODE neigh_cell_cdma2000_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(band_class.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, neigh_cells_per_freq_list, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cell_cdma2000_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(band_class.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(neigh_cells_per_freq_list, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void neigh_cell_cdma2000_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("bandClass", band_class.to_string());
  j.start_array("neighCellsPerFreqList");
  for (const auto& e1 : neigh_cells_per_freq_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// NeighCellCDMA2000-v920 ::= SEQUENCE
SRSASN_CODE neigh_cell_cdma2000_v920_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, neigh_cells_per_freq_list_v920, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cell_cdma2000_v920_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(neigh_cells_per_freq_list_v920, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void neigh_cell_cdma2000_v920_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("neighCellsPerFreqList-v920");
  for (const auto& e1 : neigh_cells_per_freq_list_v920) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// PRACH-ConfigInfo ::= SEQUENCE
SRSASN_CODE prach_cfg_info_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, prach_cfg_idx, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(bref.pack(high_speed_flag, 1));
  HANDLE_CODE(pack_integer(bref, zero_correlation_zone_cfg, (uint8_t)0u, (uint8_t)15u));
  HANDLE_CODE(pack_integer(bref, prach_freq_offset, (uint8_t)0u, (uint8_t)94u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_cfg_info_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(prach_cfg_idx, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(bref.unpack(high_speed_flag, 1));
  HANDLE_CODE(unpack_integer(zero_correlation_zone_cfg, bref, (uint8_t)0u, (uint8_t)15u));
  HANDLE_CODE(unpack_integer(prach_freq_offset, bref, (uint8_t)0u, (uint8_t)94u));

  return SRSASN_SUCCESS;
}
void prach_cfg_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("prach-ConfigIndex", prach_cfg_idx);
  j.write_bool("highSpeedFlag", high_speed_flag);
  j.write_int("zeroCorrelationZoneConfig", zero_correlation_zone_cfg);
  j.write_int("prach-FreqOffset", prach_freq_offset);
  j.end_obj();
}

// ParametersCDMA2000-r11 ::= SEQUENCE
SRSASN_CODE params_cdma2000_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(sys_time_info_r11_present, 1));
  HANDLE_CODE(bref.pack(params_hrpd_r11_present, 1));
  HANDLE_CODE(bref.pack(params1_xrtt_r11_present, 1));

  if (sys_time_info_r11_present) {
    HANDLE_CODE(sys_time_info_r11.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, search_win_size_r11, (uint8_t)0u, (uint8_t)15u));
  if (params_hrpd_r11_present) {
    HANDLE_CODE(bref.pack(params_hrpd_r11.cell_resel_params_hrpd_r11_present, 1));
    HANDLE_CODE(params_hrpd_r11.pre_regist_info_hrpd_r11.pack(bref));
    if (params_hrpd_r11.cell_resel_params_hrpd_r11_present) {
      HANDLE_CODE(params_hrpd_r11.cell_resel_params_hrpd_r11.pack(bref));
    }
  }
  if (params1_xrtt_r11_present) {
    HANDLE_CODE(bref.pack(params1_xrtt_r11.csfb_regist_param1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt_r11.long_code_state1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt_r11.cell_resel_params1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt_r11.ac_barr_cfg1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt_r11.csfb_dual_rx_tx_support_r11_present, 1));
    if (params1_xrtt_r11.csfb_regist_param1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.csfb_regist_param1_xrtt_r11.pack(bref));
    }
    if (params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11.pack(bref));
    }
    if (params1_xrtt_r11.long_code_state1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.long_code_state1_xrtt_r11.pack(bref));
    }
    if (params1_xrtt_r11.cell_resel_params1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.cell_resel_params1_xrtt_r11.pack(bref));
    }
    if (params1_xrtt_r11.ac_barr_cfg1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.ac_barr_cfg1_xrtt_r11.pack(bref));
    }
    if (params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11_present) {
      HANDLE_CODE(bref.pack(params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11, 1));
    }
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE params_cdma2000_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(sys_time_info_r11_present, 1));
  HANDLE_CODE(bref.unpack(params_hrpd_r11_present, 1));
  HANDLE_CODE(bref.unpack(params1_xrtt_r11_present, 1));

  if (sys_time_info_r11_present) {
    HANDLE_CODE(sys_time_info_r11.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(search_win_size_r11, bref, (uint8_t)0u, (uint8_t)15u));
  if (params_hrpd_r11_present) {
    HANDLE_CODE(bref.unpack(params_hrpd_r11.cell_resel_params_hrpd_r11_present, 1));
    HANDLE_CODE(params_hrpd_r11.pre_regist_info_hrpd_r11.unpack(bref));
    if (params_hrpd_r11.cell_resel_params_hrpd_r11_present) {
      HANDLE_CODE(params_hrpd_r11.cell_resel_params_hrpd_r11.unpack(bref));
    }
  }
  if (params1_xrtt_r11_present) {
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.csfb_regist_param1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.long_code_state1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.cell_resel_params1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.ac_barr_cfg1_xrtt_r11_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt_r11.csfb_dual_rx_tx_support_r11_present, 1));
    if (params1_xrtt_r11.csfb_regist_param1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.csfb_regist_param1_xrtt_r11.unpack(bref));
    }
    if (params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11.unpack(bref));
    }
    if (params1_xrtt_r11.long_code_state1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.long_code_state1_xrtt_r11.unpack(bref));
    }
    if (params1_xrtt_r11.cell_resel_params1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.cell_resel_params1_xrtt_r11.unpack(bref));
    }
    if (params1_xrtt_r11.ac_barr_cfg1_xrtt_r11_present) {
      HANDLE_CODE(params1_xrtt_r11.ac_barr_cfg1_xrtt_r11.unpack(bref));
    }
    if (params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11_present) {
      HANDLE_CODE(bref.unpack(params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11, 1));
    }
  }

  return SRSASN_SUCCESS;
}
void params_cdma2000_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (sys_time_info_r11_present) {
    j.write_fieldname("systemTimeInfo-r11");
    sys_time_info_r11.to_json(j);
  }
  j.write_int("searchWindowSize-r11", search_win_size_r11);
  if (params_hrpd_r11_present) {
    j.write_fieldname("parametersHRPD-r11");
    j.start_obj();
    j.write_fieldname("preRegistrationInfoHRPD-r11");
    params_hrpd_r11.pre_regist_info_hrpd_r11.to_json(j);
    if (params_hrpd_r11.cell_resel_params_hrpd_r11_present) {
      j.write_fieldname("cellReselectionParametersHRPD-r11");
      params_hrpd_r11.cell_resel_params_hrpd_r11.to_json(j);
    }
    j.end_obj();
  }
  if (params1_xrtt_r11_present) {
    j.write_fieldname("parameters1XRTT-r11");
    j.start_obj();
    if (params1_xrtt_r11.csfb_regist_param1_xrtt_r11_present) {
      j.write_fieldname("csfb-RegistrationParam1XRTT-r11");
      params1_xrtt_r11.csfb_regist_param1_xrtt_r11.to_json(j);
    }
    if (params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11_present) {
      j.write_fieldname("csfb-RegistrationParam1XRTT-Ext-r11");
      params1_xrtt_r11.csfb_regist_param1_xrtt_ext_r11.to_json(j);
    }
    if (params1_xrtt_r11.long_code_state1_xrtt_r11_present) {
      j.write_str("longCodeState1XRTT-r11", params1_xrtt_r11.long_code_state1_xrtt_r11.to_string());
    }
    if (params1_xrtt_r11.cell_resel_params1_xrtt_r11_present) {
      j.write_fieldname("cellReselectionParameters1XRTT-r11");
      params1_xrtt_r11.cell_resel_params1_xrtt_r11.to_json(j);
    }
    if (params1_xrtt_r11.ac_barr_cfg1_xrtt_r11_present) {
      j.write_fieldname("ac-BarringConfig1XRTT-r11");
      params1_xrtt_r11.ac_barr_cfg1_xrtt_r11.to_json(j);
    }
    if (params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11_present) {
      j.write_bool("csfb-SupportForDualRxUEs-r11", params1_xrtt_r11.csfb_support_for_dual_rx_ues_r11);
    }
    if (params1_xrtt_r11.csfb_dual_rx_tx_support_r11_present) {
      j.write_str("csfb-DualRxTxSupport-r11", "true");
    }
    j.end_obj();
  }
  j.end_obj();
}

void params_cdma2000_r11_s::sys_time_info_r11_c_::set(types::options e)
{
  type_ = e;
}
void params_cdma2000_r11_s::sys_time_info_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::explicit_value:
      j.write_fieldname("explicitValue");
      c.to_json(j);
      break;
    case types::default_value:
      break;
    default:
      log_invalid_choice_id(type_, "params_cdma2000_r11_s::sys_time_info_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE params_cdma2000_r11_s::sys_time_info_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::explicit_value:
      HANDLE_CODE(c.pack(bref));
      break;
    case types::default_value:
      break;
    default:
      log_invalid_choice_id(type_, "params_cdma2000_r11_s::sys_time_info_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE params_cdma2000_r11_s::sys_time_info_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::explicit_value:
      HANDLE_CODE(c.unpack(bref));
      break;
    case types::default_value:
      break;
    default:
      log_invalid_choice_id(type_, "params_cdma2000_r11_s::sys_time_info_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// PowerRampingParameters ::= SEQUENCE
SRSASN_CODE pwr_ramp_params_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pwr_ramp_step.pack(bref));
  HANDLE_CODE(preamb_init_rx_target_pwr.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pwr_ramp_params_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(pwr_ramp_step.unpack(bref));
  HANDLE_CODE(preamb_init_rx_target_pwr.unpack(bref));

  return SRSASN_SUCCESS;
}
void pwr_ramp_params_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("powerRampingStep", pwr_ramp_step.to_string());
  j.write_str("preambleInitialReceivedTargetPower", preamb_init_rx_target_pwr.to_string());
  j.end_obj();
}

// RedistributionInterFreqInfo-r13 ::= SEQUENCE
SRSASN_CODE redist_inter_freq_info_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(redist_factor_freq_r13_present, 1));
  HANDLE_CODE(bref.pack(redist_neigh_cell_list_r13_present, 1));

  if (redist_factor_freq_r13_present) {
    HANDLE_CODE(pack_integer(bref, redist_factor_freq_r13, (uint8_t)1u, (uint8_t)10u));
  }
  if (redist_neigh_cell_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, redist_neigh_cell_list_r13, 1, 16));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE redist_inter_freq_info_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(redist_factor_freq_r13_present, 1));
  HANDLE_CODE(bref.unpack(redist_neigh_cell_list_r13_present, 1));

  if (redist_factor_freq_r13_present) {
    HANDLE_CODE(unpack_integer(redist_factor_freq_r13, bref, (uint8_t)1u, (uint8_t)10u));
  }
  if (redist_neigh_cell_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(redist_neigh_cell_list_r13, bref, 1, 16));
  }

  return SRSASN_SUCCESS;
}
void redist_inter_freq_info_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (redist_factor_freq_r13_present) {
    j.write_int("redistributionFactorFreq-r13", redist_factor_freq_r13);
  }
  if (redist_neigh_cell_list_r13_present) {
    j.start_array("redistributionNeighCellList-r13");
    for (const auto& e1 : redist_neigh_cell_list_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// SL-CBR-PSSCH-TxConfig-r14 ::= SEQUENCE
SRSASN_CODE sl_cbr_pssch_tx_cfg_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, cr_limit_r14, (uint16_t)0u, (uint16_t)10000u));
  HANDLE_CODE(tx_params_r14.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_cbr_pssch_tx_cfg_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(cr_limit_r14, bref, (uint16_t)0u, (uint16_t)10000u));
  HANDLE_CODE(tx_params_r14.unpack(bref));

  return SRSASN_SUCCESS;
}
void sl_cbr_pssch_tx_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("cr-Limit-r14", cr_limit_r14);
  j.write_fieldname("tx-Parameters-r14");
  tx_params_r14.to_json(j);
  j.end_obj();
}

// SL-DiscConfigOtherInterFreq-r13 ::= SEQUENCE
SRSASN_CODE sl_disc_cfg_other_inter_freq_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(tx_pwr_info_r13_present, 1));
  HANDLE_CODE(bref.pack(ref_carrier_common_r13_present, 1));
  HANDLE_CODE(bref.pack(disc_sync_cfg_r13_present, 1));
  HANDLE_CODE(bref.pack(disc_cell_sel_info_r13_present, 1));

  if (tx_pwr_info_r13_present) {
    HANDLE_CODE(pack_fixed_seq_of(bref, &(tx_pwr_info_r13)[0], tx_pwr_info_r13.size()));
  }
  if (disc_sync_cfg_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, disc_sync_cfg_r13, 1, 16));
  }
  if (disc_cell_sel_info_r13_present) {
    HANDLE_CODE(disc_cell_sel_info_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_cfg_other_inter_freq_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(tx_pwr_info_r13_present, 1));
  HANDLE_CODE(bref.unpack(ref_carrier_common_r13_present, 1));
  HANDLE_CODE(bref.unpack(disc_sync_cfg_r13_present, 1));
  HANDLE_CODE(bref.unpack(disc_cell_sel_info_r13_present, 1));

  if (tx_pwr_info_r13_present) {
    HANDLE_CODE(unpack_fixed_seq_of(&(tx_pwr_info_r13)[0], bref, tx_pwr_info_r13.size()));
  }
  if (disc_sync_cfg_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(disc_sync_cfg_r13, bref, 1, 16));
  }
  if (disc_cell_sel_info_r13_present) {
    HANDLE_CODE(disc_cell_sel_info_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_disc_cfg_other_inter_freq_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (tx_pwr_info_r13_present) {
    j.start_array("txPowerInfo-r13");
    for (const auto& e1 : tx_pwr_info_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ref_carrier_common_r13_present) {
    j.write_str("refCarrierCommon-r13", "pCell");
  }
  if (disc_sync_cfg_r13_present) {
    j.start_array("discSyncConfig-r13");
    for (const auto& e1 : disc_sync_cfg_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (disc_cell_sel_info_r13_present) {
    j.write_fieldname("discCellSelectionInfo-r13");
    disc_cell_sel_info_r13.to_json(j);
  }
  j.end_obj();
}

// SL-HoppingConfigComm-r12 ::= SEQUENCE
SRSASN_CODE sl_hop_cfg_comm_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, hop_param_r12, (uint16_t)0u, (uint16_t)504u));
  HANDLE_CODE(num_subbands_r12.pack(bref));
  HANDLE_CODE(pack_integer(bref, rb_offset_r12, (uint8_t)0u, (uint8_t)110u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_hop_cfg_comm_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(hop_param_r12, bref, (uint16_t)0u, (uint16_t)504u));
  HANDLE_CODE(num_subbands_r12.unpack(bref));
  HANDLE_CODE(unpack_integer(rb_offset_r12, bref, (uint8_t)0u, (uint8_t)110u));

  return SRSASN_SUCCESS;
}
void sl_hop_cfg_comm_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("hoppingParameter-r12", hop_param_r12);
  j.write_str("numSubbands-r12", num_subbands_r12.to_string());
  j.write_int("rb-Offset-r12", rb_offset_r12);
  j.end_obj();
}

// SL-InterFreqInfoV2X-r14 ::= SEQUENCE
SRSASN_CODE sl_inter_freq_info_v2x_r14_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(plmn_id_list_r14_present, 1));
  HANDLE_CODE(bref.pack(sl_max_tx_pwr_r14_present, 1));
  HANDLE_CODE(bref.pack(sl_bw_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_sched_pool_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_ue_cfg_list_r14_present, 1));

  if (plmn_id_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, plmn_id_list_r14, 1, 6));
  }
  HANDLE_CODE(pack_integer(bref, v2x_comm_carrier_freq_r14, (uint32_t)0u, (uint32_t)262143u));
  if (sl_max_tx_pwr_r14_present) {
    HANDLE_CODE(pack_integer(bref, sl_max_tx_pwr_r14, (int8_t)-30, (int8_t)33));
  }
  if (sl_bw_r14_present) {
    HANDLE_CODE(sl_bw_r14.pack(bref));
  }
  if (v2x_sched_pool_r14_present) {
    HANDLE_CODE(v2x_sched_pool_r14.pack(bref));
  }
  if (v2x_ue_cfg_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_ue_cfg_list_r14, 1, 16));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= add_spec_emission_v2x_r14.is_present();
    group_flags[1] |= v2x_freq_sel_cfg_list_r15.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(add_spec_emission_v2x_r14.is_present(), 1));
      if (add_spec_emission_v2x_r14.is_present()) {
        HANDLE_CODE(add_spec_emission_v2x_r14->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(v2x_freq_sel_cfg_list_r15.is_present(), 1));
      if (v2x_freq_sel_cfg_list_r15.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *v2x_freq_sel_cfg_list_r15, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_inter_freq_info_v2x_r14_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(plmn_id_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(sl_max_tx_pwr_r14_present, 1));
  HANDLE_CODE(bref.unpack(sl_bw_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_sched_pool_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_ue_cfg_list_r14_present, 1));

  if (plmn_id_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(plmn_id_list_r14, bref, 1, 6));
  }
  HANDLE_CODE(unpack_integer(v2x_comm_carrier_freq_r14, bref, (uint32_t)0u, (uint32_t)262143u));
  if (sl_max_tx_pwr_r14_present) {
    HANDLE_CODE(unpack_integer(sl_max_tx_pwr_r14, bref, (int8_t)-30, (int8_t)33));
  }
  if (sl_bw_r14_present) {
    HANDLE_CODE(sl_bw_r14.unpack(bref));
  }
  if (v2x_sched_pool_r14_present) {
    HANDLE_CODE(v2x_sched_pool_r14.unpack(bref));
  }
  if (v2x_ue_cfg_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_ue_cfg_list_r14, bref, 1, 16));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool add_spec_emission_v2x_r14_present;
      HANDLE_CODE(bref.unpack(add_spec_emission_v2x_r14_present, 1));
      add_spec_emission_v2x_r14.set_present(add_spec_emission_v2x_r14_present);
      if (add_spec_emission_v2x_r14.is_present()) {
        HANDLE_CODE(add_spec_emission_v2x_r14->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool v2x_freq_sel_cfg_list_r15_present;
      HANDLE_CODE(bref.unpack(v2x_freq_sel_cfg_list_r15_present, 1));
      v2x_freq_sel_cfg_list_r15.set_present(v2x_freq_sel_cfg_list_r15_present);
      if (v2x_freq_sel_cfg_list_r15.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*v2x_freq_sel_cfg_list_r15, bref, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sl_inter_freq_info_v2x_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (plmn_id_list_r14_present) {
    j.start_array("plmn-IdentityList-r14");
    for (const auto& e1 : plmn_id_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.write_int("v2x-CommCarrierFreq-r14", v2x_comm_carrier_freq_r14);
  if (sl_max_tx_pwr_r14_present) {
    j.write_int("sl-MaxTxPower-r14", sl_max_tx_pwr_r14);
  }
  if (sl_bw_r14_present) {
    j.write_str("sl-Bandwidth-r14", sl_bw_r14.to_string());
  }
  if (v2x_sched_pool_r14_present) {
    j.write_fieldname("v2x-SchedulingPool-r14");
    v2x_sched_pool_r14.to_json(j);
  }
  if (v2x_ue_cfg_list_r14_present) {
    j.start_array("v2x-UE-ConfigList-r14");
    for (const auto& e1 : v2x_ue_cfg_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ext) {
    if (add_spec_emission_v2x_r14.is_present()) {
      j.write_fieldname("additionalSpectrumEmissionV2X-r14");
      add_spec_emission_v2x_r14->to_json(j);
    }
    if (v2x_freq_sel_cfg_list_r15.is_present()) {
      j.start_array("v2x-FreqSelectionConfigList-r15");
      for (const auto& e1 : *v2x_freq_sel_cfg_list_r15) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

void sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::destroy_() {}
void sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::add_spec_emission_v2x_r14_c_(
    const sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::add_spec_emission_r14:
      c.init(other.c.get<uint8_t>());
      break;
    case types::add_spec_emission_v1440:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_");
  }
}
sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_& sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::
                                                            operator=(const sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::add_spec_emission_r14:
      c.set(other.c.get<uint8_t>());
      break;
    case types::add_spec_emission_v1440:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_");
  }

  return *this;
}
void sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::add_spec_emission_r14:
      j.write_int("additionalSpectrumEmission-r14", c.get<uint8_t>());
      break;
    case types::add_spec_emission_v1440:
      j.write_int("additionalSpectrumEmission-v1440", c.get<uint16_t>());
      break;
    default:
      log_invalid_choice_id(type_, "sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_");
  }
  j.end_obj();
}
SRSASN_CODE sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::add_spec_emission_r14:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)32u));
      break;
    case types::add_spec_emission_v1440:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)33u, (uint16_t)288u));
      break;
    default:
      log_invalid_choice_id(type_, "sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::add_spec_emission_r14:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)32u));
      break;
    case types::add_spec_emission_v1440:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)33u, (uint16_t)288u));
      break;
    default:
      log_invalid_choice_id(type_, "sl_inter_freq_info_v2x_r14_s::add_spec_emission_v2x_r14_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SL-PPPR-Dest-CarrierFreq ::= SEQUENCE
SRSASN_CODE sl_pppr_dest_carrier_freq_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(dest_info_list_r15_present, 1));
  HANDLE_CODE(bref.pack(allowed_carrier_freq_list_r15_present, 1));

  if (dest_info_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, dest_info_list_r15, 1, 16));
  }
  if (allowed_carrier_freq_list_r15_present) {
    HANDLE_CODE(allowed_carrier_freq_list_r15.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pppr_dest_carrier_freq_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(dest_info_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(allowed_carrier_freq_list_r15_present, 1));

  if (dest_info_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(dest_info_list_r15, bref, 1, 16));
  }
  if (allowed_carrier_freq_list_r15_present) {
    HANDLE_CODE(allowed_carrier_freq_list_r15.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_pppr_dest_carrier_freq_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dest_info_list_r15_present) {
    j.start_array("destinationInfoList-r15");
    for (const auto& e1 : dest_info_list_r15) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (allowed_carrier_freq_list_r15_present) {
    j.write_fieldname("allowedCarrierFreqList-r15");
    allowed_carrier_freq_list_r15.to_json(j);
  }
  j.end_obj();
}

// SL-ResourcesInterFreq-r13 ::= SEQUENCE
SRSASN_CODE sl_res_inter_freq_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(disc_rx_res_inter_freq_r13_present, 1));
  HANDLE_CODE(bref.pack(disc_tx_res_inter_freq_r13_present, 1));

  if (disc_rx_res_inter_freq_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, disc_rx_res_inter_freq_r13, 1, 16));
  }
  if (disc_tx_res_inter_freq_r13_present) {
    HANDLE_CODE(disc_tx_res_inter_freq_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_res_inter_freq_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(disc_rx_res_inter_freq_r13_present, 1));
  HANDLE_CODE(bref.unpack(disc_tx_res_inter_freq_r13_present, 1));

  if (disc_rx_res_inter_freq_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(disc_rx_res_inter_freq_r13, bref, 1, 16));
  }
  if (disc_tx_res_inter_freq_r13_present) {
    HANDLE_CODE(disc_tx_res_inter_freq_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_res_inter_freq_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (disc_rx_res_inter_freq_r13_present) {
    j.start_array("discRxResourcesInterFreq-r13");
    for (const auto& e1 : disc_rx_res_inter_freq_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (disc_tx_res_inter_freq_r13_present) {
    j.write_fieldname("discTxResourcesInterFreq-r13");
    disc_tx_res_inter_freq_r13.to_json(j);
  }
  j.end_obj();
}

// SL-SyncConfig-r12 ::= SEQUENCE
SRSASN_CODE sl_sync_cfg_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(tx_params_r12_present, 1));
  HANDLE_CODE(bref.pack(rx_params_ncell_r12_present, 1));

  HANDLE_CODE(sync_cp_len_r12.pack(bref));
  HANDLE_CODE(pack_integer(bref, sync_offset_ind_r12, (uint8_t)0u, (uint8_t)39u));
  HANDLE_CODE(pack_integer(bref, slssid_r12, (uint8_t)0u, (uint8_t)167u));
  if (tx_params_r12_present) {
    HANDLE_CODE(bref.pack(tx_params_r12.sync_info_reserved_r12_present, 1));
    HANDLE_CODE(tx_params_r12.sync_tx_params_r12.pack(bref));
    HANDLE_CODE(pack_integer(bref, tx_params_r12.sync_tx_thresh_ic_r12, (uint8_t)0u, (uint8_t)13u));
    if (tx_params_r12.sync_info_reserved_r12_present) {
      HANDLE_CODE(tx_params_r12.sync_info_reserved_r12.pack(bref));
    }
  }
  if (rx_params_ncell_r12_present) {
    HANDLE_CODE(pack_integer(bref, rx_params_ncell_r12.pci_r12, (uint16_t)0u, (uint16_t)503u));
    HANDLE_CODE(rx_params_ncell_r12.disc_sync_win_r12.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= sync_tx_periodic_r13_present;
    group_flags[1] |= sync_offset_ind_v1430_present;
    group_flags[1] |= gnss_sync_r14_present;
    group_flags[2] |= sync_offset_ind2_r14_present;
    group_flags[2] |= sync_offset_ind3_r14_present;
    group_flags[3] |= slss_tx_disabled_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sync_tx_periodic_r13_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sync_offset_ind_v1430_present, 1));
      HANDLE_CODE(bref.pack(gnss_sync_r14_present, 1));
      if (sync_offset_ind_v1430_present) {
        HANDLE_CODE(pack_integer(bref, sync_offset_ind_v1430, (uint8_t)40u, (uint8_t)159u));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sync_offset_ind2_r14_present, 1));
      HANDLE_CODE(bref.pack(sync_offset_ind3_r14_present, 1));
      if (sync_offset_ind2_r14_present) {
        HANDLE_CODE(pack_integer(bref, sync_offset_ind2_r14, (uint8_t)0u, (uint8_t)159u));
      }
      if (sync_offset_ind3_r14_present) {
        HANDLE_CODE(pack_integer(bref, sync_offset_ind3_r14, (uint8_t)0u, (uint8_t)159u));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(slss_tx_disabled_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_sync_cfg_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(tx_params_r12_present, 1));
  HANDLE_CODE(bref.unpack(rx_params_ncell_r12_present, 1));

  HANDLE_CODE(sync_cp_len_r12.unpack(bref));
  HANDLE_CODE(unpack_integer(sync_offset_ind_r12, bref, (uint8_t)0u, (uint8_t)39u));
  HANDLE_CODE(unpack_integer(slssid_r12, bref, (uint8_t)0u, (uint8_t)167u));
  if (tx_params_r12_present) {
    HANDLE_CODE(bref.unpack(tx_params_r12.sync_info_reserved_r12_present, 1));
    HANDLE_CODE(tx_params_r12.sync_tx_params_r12.unpack(bref));
    HANDLE_CODE(unpack_integer(tx_params_r12.sync_tx_thresh_ic_r12, bref, (uint8_t)0u, (uint8_t)13u));
    if (tx_params_r12.sync_info_reserved_r12_present) {
      HANDLE_CODE(tx_params_r12.sync_info_reserved_r12.unpack(bref));
    }
  }
  if (rx_params_ncell_r12_present) {
    HANDLE_CODE(unpack_integer(rx_params_ncell_r12.pci_r12, bref, (uint16_t)0u, (uint16_t)503u));
    HANDLE_CODE(rx_params_ncell_r12.disc_sync_win_r12.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(4);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(sync_tx_periodic_r13_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(sync_offset_ind_v1430_present, 1));
      HANDLE_CODE(bref.unpack(gnss_sync_r14_present, 1));
      if (sync_offset_ind_v1430_present) {
        HANDLE_CODE(unpack_integer(sync_offset_ind_v1430, bref, (uint8_t)40u, (uint8_t)159u));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(sync_offset_ind2_r14_present, 1));
      HANDLE_CODE(bref.unpack(sync_offset_ind3_r14_present, 1));
      if (sync_offset_ind2_r14_present) {
        HANDLE_CODE(unpack_integer(sync_offset_ind2_r14, bref, (uint8_t)0u, (uint8_t)159u));
      }
      if (sync_offset_ind3_r14_present) {
        HANDLE_CODE(unpack_integer(sync_offset_ind3_r14, bref, (uint8_t)0u, (uint8_t)159u));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(slss_tx_disabled_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
void sl_sync_cfg_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("syncCP-Len-r12", sync_cp_len_r12.to_string());
  j.write_int("syncOffsetIndicator-r12", sync_offset_ind_r12);
  j.write_int("slssid-r12", slssid_r12);
  if (tx_params_r12_present) {
    j.write_fieldname("txParameters-r12");
    j.start_obj();
    j.write_fieldname("syncTxParameters-r12");
    tx_params_r12.sync_tx_params_r12.to_json(j);
    j.write_int("syncTxThreshIC-r12", tx_params_r12.sync_tx_thresh_ic_r12);
    if (tx_params_r12.sync_info_reserved_r12_present) {
      j.write_str("syncInfoReserved-r12", tx_params_r12.sync_info_reserved_r12.to_string());
    }
    j.end_obj();
  }
  if (rx_params_ncell_r12_present) {
    j.write_fieldname("rxParamsNCell-r12");
    j.start_obj();
    j.write_int("physCellId-r12", rx_params_ncell_r12.pci_r12);
    j.write_str("discSyncWindow-r12", rx_params_ncell_r12.disc_sync_win_r12.to_string());
    j.end_obj();
  }
  if (ext) {
    if (sync_tx_periodic_r13_present) {
      j.write_str("syncTxPeriodic-r13", "true");
    }
    if (sync_offset_ind_v1430_present) {
      j.write_int("syncOffsetIndicator-v1430", sync_offset_ind_v1430);
    }
    if (gnss_sync_r14_present) {
      j.write_str("gnss-Sync-r14", "true");
    }
    if (sync_offset_ind2_r14_present) {
      j.write_int("syncOffsetIndicator2-r14", sync_offset_ind2_r14);
    }
    if (sync_offset_ind3_r14_present) {
      j.write_int("syncOffsetIndicator3-r14", sync_offset_ind3_r14);
    }
    if (slss_tx_disabled_r15_present) {
      j.write_str("slss-TxDisabled-r15", "true");
    }
  }
  j.end_obj();
}

// SS-RSSI-Measurement-r15 ::= SEQUENCE
SRSASN_CODE ss_rssi_meas_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(meas_slots_r15.pack(bref));
  HANDLE_CODE(pack_integer(bref, end_symbol_r15, (uint8_t)0u, (uint8_t)3u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ss_rssi_meas_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(meas_slots_r15.unpack(bref));
  HANDLE_CODE(unpack_integer(end_symbol_r15, bref, (uint8_t)0u, (uint8_t)3u));

  return SRSASN_SUCCESS;
}
void ss_rssi_meas_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("measurementSlots-r15", meas_slots_r15.to_string());
  j.write_int("endSymbol-r15", end_symbol_r15);
  j.end_obj();
}

// SystemInformationBlockType1-v10l0-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v10l0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_info_v10l0_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10l0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_band_info_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, freq_band_info_v10l0, 1, 8));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10l0, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v10l0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_info_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_band_info_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(freq_band_info_v10l0, bref, 1, 8));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10l0, bref, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v10l0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_info_v10l0_present) {
    j.start_array("freqBandInfo-v10l0");
    for (const auto& e1 : freq_band_info_v10l0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (multi_band_info_list_v10l0_present) {
    j.start_array("multiBandInfoList-v10l0");
    for (const auto& e1 : multi_band_info_list_v10l0) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// SystemInformationBlockType1-v1310-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1310_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(hyper_sfn_r13_present, 1));
  HANDLE_CODE(bref.pack(edrx_allowed_r13_present, 1));
  HANDLE_CODE(bref.pack(cell_sel_info_ce_r13_present, 1));
  HANDLE_CODE(bref.pack(bw_reduced_access_related_info_r13_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (hyper_sfn_r13_present) {
    HANDLE_CODE(hyper_sfn_r13.pack(bref));
  }
  if (cell_sel_info_ce_r13_present) {
    HANDLE_CODE(cell_sel_info_ce_r13.pack(bref));
  }
  if (bw_reduced_access_related_info_r13_present) {
    HANDLE_CODE(bref.pack(bw_reduced_access_related_info_r13.sched_info_list_br_r13_present, 1));
    HANDLE_CODE(bref.pack(bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13_present, 1));
    HANDLE_CODE(bref.pack(bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13_present, 1));
    HANDLE_CODE(bref.pack(bw_reduced_access_related_info_r13.si_validity_time_r13_present, 1));
    HANDLE_CODE(bref.pack(bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13_present, 1));
    HANDLE_CODE(bw_reduced_access_related_info_r13.si_win_len_br_r13.pack(bref));
    HANDLE_CODE(bw_reduced_access_related_info_r13.si_repeat_pattern_r13.pack(bref));
    if (bw_reduced_access_related_info_r13.sched_info_list_br_r13_present) {
      HANDLE_CODE(pack_dyn_seq_of(bref, bw_reduced_access_related_info_r13.sched_info_list_br_r13, 1, 32));
    }
    if (bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13_present) {
      HANDLE_CODE(bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13.pack(bref));
    }
    if (bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13_present) {
      HANDLE_CODE(bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13.pack(bref));
    }
    HANDLE_CODE(pack_integer(bref, bw_reduced_access_related_info_r13.start_symbol_br_r13, (uint8_t)1u, (uint8_t)4u));
    HANDLE_CODE(bw_reduced_access_related_info_r13.si_hop_cfg_common_r13.pack(bref));
    if (bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13_present) {
      HANDLE_CODE(pack_dyn_seq_of(
          bref, bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13, 1, 32, integer_packer<uint8_t>(0, 3)));
    }
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1310_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(hyper_sfn_r13_present, 1));
  HANDLE_CODE(bref.unpack(edrx_allowed_r13_present, 1));
  HANDLE_CODE(bref.unpack(cell_sel_info_ce_r13_present, 1));
  HANDLE_CODE(bref.unpack(bw_reduced_access_related_info_r13_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (hyper_sfn_r13_present) {
    HANDLE_CODE(hyper_sfn_r13.unpack(bref));
  }
  if (cell_sel_info_ce_r13_present) {
    HANDLE_CODE(cell_sel_info_ce_r13.unpack(bref));
  }
  if (bw_reduced_access_related_info_r13_present) {
    HANDLE_CODE(bref.unpack(bw_reduced_access_related_info_r13.sched_info_list_br_r13_present, 1));
    HANDLE_CODE(bref.unpack(bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13_present, 1));
    HANDLE_CODE(bref.unpack(bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13_present, 1));
    HANDLE_CODE(bref.unpack(bw_reduced_access_related_info_r13.si_validity_time_r13_present, 1));
    HANDLE_CODE(bref.unpack(bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13_present, 1));
    HANDLE_CODE(bw_reduced_access_related_info_r13.si_win_len_br_r13.unpack(bref));
    HANDLE_CODE(bw_reduced_access_related_info_r13.si_repeat_pattern_r13.unpack(bref));
    if (bw_reduced_access_related_info_r13.sched_info_list_br_r13_present) {
      HANDLE_CODE(unpack_dyn_seq_of(bw_reduced_access_related_info_r13.sched_info_list_br_r13, bref, 1, 32));
    }
    if (bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13_present) {
      HANDLE_CODE(bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13.unpack(bref));
    }
    if (bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13_present) {
      HANDLE_CODE(bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13.unpack(bref));
    }
    HANDLE_CODE(unpack_integer(bw_reduced_access_related_info_r13.start_symbol_br_r13, bref, (uint8_t)1u, (uint8_t)4u));
    HANDLE_CODE(bw_reduced_access_related_info_r13.si_hop_cfg_common_r13.unpack(bref));
    if (bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13_present) {
      HANDLE_CODE(unpack_dyn_seq_of(
          bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13, bref, 1, 32, integer_packer<uint8_t>(0, 3)));
    }
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1310_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (hyper_sfn_r13_present) {
    j.write_str("hyperSFN-r13", hyper_sfn_r13.to_string());
  }
  if (edrx_allowed_r13_present) {
    j.write_str("eDRX-Allowed-r13", "true");
  }
  if (cell_sel_info_ce_r13_present) {
    j.write_fieldname("cellSelectionInfoCE-r13");
    cell_sel_info_ce_r13.to_json(j);
  }
  if (bw_reduced_access_related_info_r13_present) {
    j.write_fieldname("bandwidthReducedAccessRelatedInfo-r13");
    j.start_obj();
    j.write_str("si-WindowLength-BR-r13", bw_reduced_access_related_info_r13.si_win_len_br_r13.to_string());
    j.write_str("si-RepetitionPattern-r13", bw_reduced_access_related_info_r13.si_repeat_pattern_r13.to_string());
    if (bw_reduced_access_related_info_r13.sched_info_list_br_r13_present) {
      j.start_array("schedulingInfoList-BR-r13");
      for (const auto& e1 : bw_reduced_access_related_info_r13.sched_info_list_br_r13) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13_present) {
      j.write_fieldname("fdd-DownlinkOrTddSubframeBitmapBR-r13");
      bw_reduced_access_related_info_r13.fdd_dl_or_tdd_sf_bitmap_br_r13.to_json(j);
    }
    if (bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13_present) {
      j.write_str("fdd-UplinkSubframeBitmapBR-r13",
                  bw_reduced_access_related_info_r13.fdd_ul_sf_bitmap_br_r13.to_string());
    }
    j.write_int("startSymbolBR-r13", bw_reduced_access_related_info_r13.start_symbol_br_r13);
    j.write_str("si-HoppingConfigCommon-r13", bw_reduced_access_related_info_r13.si_hop_cfg_common_r13.to_string());
    if (bw_reduced_access_related_info_r13.si_validity_time_r13_present) {
      j.write_str("si-ValidityTime-r13", "true");
    }
    if (bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13_present) {
      j.start_array("systemInfoValueTagList-r13");
      for (const auto& e1 : bw_reduced_access_related_info_r13.sys_info_value_tag_list_r13) {
        j.write_int(e1);
      }
      j.end_array();
    }
    j.end_obj();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

void sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::destroy_()
{
  switch (type_) {
    case types::sf_pattern10_r13:
      c.destroy<fixed_bitstring<10> >();
      break;
    case types::sf_pattern40_r13:
      c.destroy<fixed_bitstring<40> >();
      break;
    default:
      break;
  }
}
void sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::set(
    types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sf_pattern10_r13:
      c.init<fixed_bitstring<10> >();
      break;
    case types::sf_pattern40_r13:
      c.init<fixed_bitstring<40> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(
          type_, "sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_");
  }
}
sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::
    fdd_dl_or_tdd_sf_bitmap_br_r13_c_(
        const sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sf_pattern10_r13:
      c.init(other.c.get<fixed_bitstring<10> >());
      break;
    case types::sf_pattern40_r13:
      c.init(other.c.get<fixed_bitstring<40> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(
          type_, "sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_");
  }
}
sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_&
sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::
operator=(const sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sf_pattern10_r13:
      c.set(other.c.get<fixed_bitstring<10> >());
      break;
    case types::sf_pattern40_r13:
      c.set(other.c.get<fixed_bitstring<40> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(
          type_, "sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_");
  }

  return *this;
}
void sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::to_json(
    json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sf_pattern10_r13:
      j.write_str("subframePattern10-r13", c.get<fixed_bitstring<10> >().to_string());
      break;
    case types::sf_pattern40_r13:
      j.write_str("subframePattern40-r13", c.get<fixed_bitstring<40> >().to_string());
      break;
    default:
      log_invalid_choice_id(
          type_, "sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::pack(
    bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sf_pattern10_r13:
      HANDLE_CODE(c.get<fixed_bitstring<10> >().pack(bref));
      break;
    case types::sf_pattern40_r13:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(
          type_, "sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE
sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sf_pattern10_r13:
      HANDLE_CODE(c.get<fixed_bitstring<10> >().unpack(bref));
      break;
    case types::sf_pattern40_r13:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(
          type_, "sib_type1_v1310_ies_s::bw_reduced_access_related_info_r13_s_::fdd_dl_or_tdd_sf_bitmap_br_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType2-v9i0-IEs ::= SEQUENCE
SRSASN_CODE sib_type2_v9i0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));
  HANDLE_CODE(bref.pack(dummy_present, 1));

  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type2_v9i0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));
  HANDLE_CODE(bref.unpack(dummy_present, 1));

  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type2_v9i0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (non_crit_ext_present) {
    j.write_str("nonCriticalExtension", non_crit_ext.to_string());
  }
  if (dummy_present) {
    j.write_fieldname("dummy");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// SystemInformationBlockType5-v10j0-IEs ::= SEQUENCE
SRSASN_CODE sib_type5_v10j0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v10j0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v10j0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_carrier_freq_list_v10j0, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type5_v10j0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v10j0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v10j0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_carrier_freq_list_v10j0, bref, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type5_v10j0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (inter_freq_carrier_freq_list_v10j0_present) {
    j.start_array("interFreqCarrierFreqList-v10j0");
    for (const auto& e1 : inter_freq_carrier_freq_list_v10j0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// ThresholdListNR-r15 ::= SEQUENCE
SRSASN_CODE thres_list_nr_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(nr_rsrp_r15_present, 1));
  HANDLE_CODE(bref.pack(nr_rsrq_r15_present, 1));
  HANDLE_CODE(bref.pack(nr_sinr_r15_present, 1));

  if (nr_rsrp_r15_present) {
    HANDLE_CODE(pack_integer(bref, nr_rsrp_r15, (uint8_t)0u, (uint8_t)127u));
  }
  if (nr_rsrq_r15_present) {
    HANDLE_CODE(pack_integer(bref, nr_rsrq_r15, (uint8_t)0u, (uint8_t)127u));
  }
  if (nr_sinr_r15_present) {
    HANDLE_CODE(pack_integer(bref, nr_sinr_r15, (uint8_t)0u, (uint8_t)127u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE thres_list_nr_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(nr_rsrp_r15_present, 1));
  HANDLE_CODE(bref.unpack(nr_rsrq_r15_present, 1));
  HANDLE_CODE(bref.unpack(nr_sinr_r15_present, 1));

  if (nr_rsrp_r15_present) {
    HANDLE_CODE(unpack_integer(nr_rsrp_r15, bref, (uint8_t)0u, (uint8_t)127u));
  }
  if (nr_rsrq_r15_present) {
    HANDLE_CODE(unpack_integer(nr_rsrq_r15, bref, (uint8_t)0u, (uint8_t)127u));
  }
  if (nr_sinr_r15_present) {
    HANDLE_CODE(unpack_integer(nr_sinr_r15, bref, (uint8_t)0u, (uint8_t)127u));
  }

  return SRSASN_SUCCESS;
}
void thres_list_nr_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (nr_rsrp_r15_present) {
    j.write_int("nr-RSRP-r15", nr_rsrp_r15);
  }
  if (nr_rsrq_r15_present) {
    j.write_int("nr-RSRQ-r15", nr_rsrq_r15);
  }
  if (nr_sinr_r15_present) {
    j.write_int("nr-SINR-r15", nr_sinr_r15);
  }
  j.end_obj();
}

// UDT-Restricting-r13 ::= SEQUENCE
SRSASN_CODE udt_restricting_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(udt_restricting_r13_present, 1));
  HANDLE_CODE(bref.pack(udt_restricting_time_r13_present, 1));

  if (udt_restricting_time_r13_present) {
    HANDLE_CODE(udt_restricting_time_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE udt_restricting_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(udt_restricting_r13_present, 1));
  HANDLE_CODE(bref.unpack(udt_restricting_time_r13_present, 1));

  if (udt_restricting_time_r13_present) {
    HANDLE_CODE(udt_restricting_time_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void udt_restricting_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (udt_restricting_r13_present) {
    j.write_str("udt-Restricting-r13", "true");
  }
  if (udt_restricting_time_r13_present) {
    j.write_str("udt-RestrictingTime-r13", udt_restricting_time_r13.to_string());
  }
  j.end_obj();
}

// UL-ReferenceSignalsPUSCH ::= SEQUENCE
SRSASN_CODE ul_ref_sigs_pusch_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(group_hop_enabled, 1));
  HANDLE_CODE(pack_integer(bref, group_assign_pusch, (uint8_t)0u, (uint8_t)29u));
  HANDLE_CODE(bref.pack(seq_hop_enabled, 1));
  HANDLE_CODE(pack_integer(bref, cyclic_shift, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_ref_sigs_pusch_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(group_hop_enabled, 1));
  HANDLE_CODE(unpack_integer(group_assign_pusch, bref, (uint8_t)0u, (uint8_t)29u));
  HANDLE_CODE(bref.unpack(seq_hop_enabled, 1));
  HANDLE_CODE(unpack_integer(cyclic_shift, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void ul_ref_sigs_pusch_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_bool("groupHoppingEnabled", group_hop_enabled);
  j.write_int("groupAssignmentPUSCH", group_assign_pusch);
  j.write_bool("sequenceHoppingEnabled", seq_hop_enabled);
  j.write_int("cyclicShift", cyclic_shift);
  j.end_obj();
}

// WLAN-Identifiers-r12 ::= SEQUENCE
SRSASN_CODE wlan_ids_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ssid_r12_present, 1));
  HANDLE_CODE(bref.pack(bssid_r12_present, 1));
  HANDLE_CODE(bref.pack(hessid_r12_present, 1));

  if (ssid_r12_present) {
    HANDLE_CODE(ssid_r12.pack(bref));
  }
  if (bssid_r12_present) {
    HANDLE_CODE(bssid_r12.pack(bref));
  }
  if (hessid_r12_present) {
    HANDLE_CODE(hessid_r12.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE wlan_ids_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ssid_r12_present, 1));
  HANDLE_CODE(bref.unpack(bssid_r12_present, 1));
  HANDLE_CODE(bref.unpack(hessid_r12_present, 1));

  if (ssid_r12_present) {
    HANDLE_CODE(ssid_r12.unpack(bref));
  }
  if (bssid_r12_present) {
    HANDLE_CODE(bssid_r12.unpack(bref));
  }
  if (hessid_r12_present) {
    HANDLE_CODE(hessid_r12.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void wlan_ids_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ssid_r12_present) {
    j.write_str("ssid-r12", ssid_r12.to_string());
  }
  if (bssid_r12_present) {
    j.write_str("bssid-r12", bssid_r12.to_string());
  }
  if (hessid_r12_present) {
    j.write_str("hessid-r12", hessid_r12.to_string());
  }
  j.end_obj();
}

// AC-BarringPerPLMN-r12 ::= SEQUENCE
SRSASN_CODE ac_barr_per_plmn_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ac_barr_info_r12_present, 1));
  HANDLE_CODE(bref.pack(ac_barr_skip_for_mmtel_voice_r12_present, 1));
  HANDLE_CODE(bref.pack(ac_barr_skip_for_mmtel_video_r12_present, 1));
  HANDLE_CODE(bref.pack(ac_barr_skip_for_sms_r12_present, 1));
  HANDLE_CODE(bref.pack(ac_barr_for_csfb_r12_present, 1));
  HANDLE_CODE(bref.pack(ssac_barr_for_mmtel_voice_r12_present, 1));
  HANDLE_CODE(bref.pack(ssac_barr_for_mmtel_video_r12_present, 1));

  HANDLE_CODE(pack_integer(bref, plmn_id_idx_r12, (uint8_t)1u, (uint8_t)6u));
  if (ac_barr_info_r12_present) {
    HANDLE_CODE(bref.pack(ac_barr_info_r12.ac_barr_for_mo_sig_r12_present, 1));
    HANDLE_CODE(bref.pack(ac_barr_info_r12.ac_barr_for_mo_data_r12_present, 1));
    HANDLE_CODE(bref.pack(ac_barr_info_r12.ac_barr_for_emergency_r12, 1));
    if (ac_barr_info_r12.ac_barr_for_mo_sig_r12_present) {
      HANDLE_CODE(ac_barr_info_r12.ac_barr_for_mo_sig_r12.pack(bref));
    }
    if (ac_barr_info_r12.ac_barr_for_mo_data_r12_present) {
      HANDLE_CODE(ac_barr_info_r12.ac_barr_for_mo_data_r12.pack(bref));
    }
  }
  if (ac_barr_for_csfb_r12_present) {
    HANDLE_CODE(ac_barr_for_csfb_r12.pack(bref));
  }
  if (ssac_barr_for_mmtel_voice_r12_present) {
    HANDLE_CODE(ssac_barr_for_mmtel_voice_r12.pack(bref));
  }
  if (ssac_barr_for_mmtel_video_r12_present) {
    HANDLE_CODE(ssac_barr_for_mmtel_video_r12.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ac_barr_per_plmn_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ac_barr_info_r12_present, 1));
  HANDLE_CODE(bref.unpack(ac_barr_skip_for_mmtel_voice_r12_present, 1));
  HANDLE_CODE(bref.unpack(ac_barr_skip_for_mmtel_video_r12_present, 1));
  HANDLE_CODE(bref.unpack(ac_barr_skip_for_sms_r12_present, 1));
  HANDLE_CODE(bref.unpack(ac_barr_for_csfb_r12_present, 1));
  HANDLE_CODE(bref.unpack(ssac_barr_for_mmtel_voice_r12_present, 1));
  HANDLE_CODE(bref.unpack(ssac_barr_for_mmtel_video_r12_present, 1));

  HANDLE_CODE(unpack_integer(plmn_id_idx_r12, bref, (uint8_t)1u, (uint8_t)6u));
  if (ac_barr_info_r12_present) {
    HANDLE_CODE(bref.unpack(ac_barr_info_r12.ac_barr_for_mo_sig_r12_present, 1));
    HANDLE_CODE(bref.unpack(ac_barr_info_r12.ac_barr_for_mo_data_r12_present, 1));
    HANDLE_CODE(bref.unpack(ac_barr_info_r12.ac_barr_for_emergency_r12, 1));
    if (ac_barr_info_r12.ac_barr_for_mo_sig_r12_present) {
      HANDLE_CODE(ac_barr_info_r12.ac_barr_for_mo_sig_r12.unpack(bref));
    }
    if (ac_barr_info_r12.ac_barr_for_mo_data_r12_present) {
      HANDLE_CODE(ac_barr_info_r12.ac_barr_for_mo_data_r12.unpack(bref));
    }
  }
  if (ac_barr_for_csfb_r12_present) {
    HANDLE_CODE(ac_barr_for_csfb_r12.unpack(bref));
  }
  if (ssac_barr_for_mmtel_voice_r12_present) {
    HANDLE_CODE(ssac_barr_for_mmtel_voice_r12.unpack(bref));
  }
  if (ssac_barr_for_mmtel_video_r12_present) {
    HANDLE_CODE(ssac_barr_for_mmtel_video_r12.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ac_barr_per_plmn_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("plmn-IdentityIndex-r12", plmn_id_idx_r12);
  if (ac_barr_info_r12_present) {
    j.write_fieldname("ac-BarringInfo-r12");
    j.start_obj();
    j.write_bool("ac-BarringForEmergency-r12", ac_barr_info_r12.ac_barr_for_emergency_r12);
    if (ac_barr_info_r12.ac_barr_for_mo_sig_r12_present) {
      j.write_fieldname("ac-BarringForMO-Signalling-r12");
      ac_barr_info_r12.ac_barr_for_mo_sig_r12.to_json(j);
    }
    if (ac_barr_info_r12.ac_barr_for_mo_data_r12_present) {
      j.write_fieldname("ac-BarringForMO-Data-r12");
      ac_barr_info_r12.ac_barr_for_mo_data_r12.to_json(j);
    }
    j.end_obj();
  }
  if (ac_barr_skip_for_mmtel_voice_r12_present) {
    j.write_str("ac-BarringSkipForMMTELVoice-r12", "true");
  }
  if (ac_barr_skip_for_mmtel_video_r12_present) {
    j.write_str("ac-BarringSkipForMMTELVideo-r12", "true");
  }
  if (ac_barr_skip_for_sms_r12_present) {
    j.write_str("ac-BarringSkipForSMS-r12", "true");
  }
  if (ac_barr_for_csfb_r12_present) {
    j.write_fieldname("ac-BarringForCSFB-r12");
    ac_barr_for_csfb_r12.to_json(j);
  }
  if (ssac_barr_for_mmtel_voice_r12_present) {
    j.write_fieldname("ssac-BarringForMMTEL-Voice-r12");
    ssac_barr_for_mmtel_voice_r12.to_json(j);
  }
  if (ssac_barr_for_mmtel_video_r12_present) {
    j.write_fieldname("ssac-BarringForMMTEL-Video-r12");
    ssac_barr_for_mmtel_video_r12.to_json(j);
  }
  j.end_obj();
}

// ACDC-BarringPerPLMN-r13 ::= SEQUENCE
SRSASN_CODE acdc_barr_per_plmn_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, plmn_id_idx_r13, (uint8_t)1u, (uint8_t)6u));
  HANDLE_CODE(bref.pack(acdc_only_for_hplmn_r13, 1));
  HANDLE_CODE(pack_dyn_seq_of(bref, barr_per_acdc_category_list_r13, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE acdc_barr_per_plmn_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(plmn_id_idx_r13, bref, (uint8_t)1u, (uint8_t)6u));
  HANDLE_CODE(bref.unpack(acdc_only_for_hplmn_r13, 1));
  HANDLE_CODE(unpack_dyn_seq_of(barr_per_acdc_category_list_r13, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void acdc_barr_per_plmn_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("plmn-IdentityIndex-r13", plmn_id_idx_r13);
  j.write_bool("acdc-OnlyForHPLMN-r13", acdc_only_for_hplmn_r13);
  j.start_array("barringPerACDC-CategoryList-r13");
  for (const auto& e1 : barr_per_acdc_category_list_r13) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// BCCH-Config ::= SEQUENCE
SRSASN_CODE bcch_cfg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(mod_period_coeff.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_cfg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(mod_period_coeff.unpack(bref));

  return SRSASN_SUCCESS;
}
void bcch_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("modificationPeriodCoeff", mod_period_coeff.to_string());
  j.end_obj();
}

// BCCH-Config-v1310 ::= SEQUENCE
SRSASN_CODE bcch_cfg_v1310_s::pack(bit_ref& bref) const
{
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_cfg_v1310_s::unpack(cbit_ref& bref)
{
  return SRSASN_SUCCESS;
}
void bcch_cfg_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("modificationPeriodCoeff-v1310", "n64");
  j.end_obj();
}

// CIOT-OptimisationPLMN-r13 ::= SEQUENCE
SRSASN_CODE ciot_optim_plmn_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(up_cio_t_eps_optim_r13_present, 1));
  HANDLE_CODE(bref.pack(cp_cio_t_eps_optim_r13_present, 1));
  HANDLE_CODE(bref.pack(attach_without_pdn_connect_r13_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ciot_optim_plmn_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(up_cio_t_eps_optim_r13_present, 1));
  HANDLE_CODE(bref.unpack(cp_cio_t_eps_optim_r13_present, 1));
  HANDLE_CODE(bref.unpack(attach_without_pdn_connect_r13_present, 1));

  return SRSASN_SUCCESS;
}
void ciot_optim_plmn_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (up_cio_t_eps_optim_r13_present) {
    j.write_str("up-CIoT-EPS-Optimisation-r13", "true");
  }
  if (cp_cio_t_eps_optim_r13_present) {
    j.write_str("cp-CIoT-EPS-Optimisation-r13", "true");
  }
  if (attach_without_pdn_connect_r13_present) {
    j.write_str("attachWithoutPDN-Connectivity-r13", "true");
  }
  j.end_obj();
}

// CarrierFreqInfoUTRA-FDD-v8h0 ::= SEQUENCE
SRSASN_CODE carrier_freq_info_utra_fdd_v8h0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(multi_band_info_list_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list, 1, 8, integer_packer<uint8_t>(1, 86)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_info_utra_fdd_v8h0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(multi_band_info_list_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list, bref, 1, 8, integer_packer<uint8_t>(1, 86)));
  }

  return SRSASN_SUCCESS;
}
void carrier_freq_info_utra_fdd_v8h0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (multi_band_info_list_present) {
    j.start_array("multiBandInfoList");
    for (const auto& e1 : multi_band_info_list) {
      j.write_int(e1);
    }
    j.end_array();
  }
  j.end_obj();
}

// CarrierFreqNR-r15 ::= SEQUENCE
SRSASN_CODE carrier_freq_nr_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(multi_band_info_list_r15_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_sul_r15_present, 1));
  HANDLE_CODE(bref.pack(meas_timing_cfg_r15_present, 1));
  HANDLE_CODE(bref.pack(ss_rssi_meas_r15_present, 1));
  HANDLE_CODE(bref.pack(cell_resel_prio_r15_present, 1));
  HANDLE_CODE(bref.pack(cell_resel_sub_prio_r15_present, 1));
  HANDLE_CODE(bref.pack(thresh_x_q_r15_present, 1));
  HANDLE_CODE(bref.pack(q_rx_lev_min_sul_r15_present, 1));
  HANDLE_CODE(bref.pack(ns_pmax_list_nr_r15_present, 1));
  HANDLE_CODE(bref.pack(q_qual_min_r15_present, 1));
  HANDLE_CODE(bref.pack(max_rs_idx_cell_qual_r15_present, 1));
  HANDLE_CODE(bref.pack(thresh_rs_idx_r15_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq_r15, (uint32_t)0u, (uint32_t)3279165u));
  if (multi_band_info_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_r15, 1, 32, integer_packer<uint16_t>(1, 1024)));
  }
  if (multi_band_info_list_sul_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_sul_r15, 1, 32, integer_packer<uint16_t>(1, 1024)));
  }
  if (meas_timing_cfg_r15_present) {
    HANDLE_CODE(meas_timing_cfg_r15.pack(bref));
  }
  HANDLE_CODE(subcarrier_spacing_ssb_r15.pack(bref));
  if (ss_rssi_meas_r15_present) {
    HANDLE_CODE(ss_rssi_meas_r15.pack(bref));
  }
  if (cell_resel_prio_r15_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio_r15, (uint8_t)0u, (uint8_t)7u));
  }
  if (cell_resel_sub_prio_r15_present) {
    HANDLE_CODE(cell_resel_sub_prio_r15.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high_r15, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low_r15, (uint8_t)0u, (uint8_t)31u));
  if (thresh_x_q_r15_present) {
    HANDLE_CODE(pack_integer(bref, thresh_x_q_r15.thresh_x_high_q_r15, (uint8_t)0u, (uint8_t)31u));
    HANDLE_CODE(pack_integer(bref, thresh_x_q_r15.thresh_x_low_q_r15, (uint8_t)0u, (uint8_t)31u));
  }
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_r15, (int8_t)-70, (int8_t)-22));
  if (q_rx_lev_min_sul_r15_present) {
    HANDLE_CODE(pack_integer(bref, q_rx_lev_min_sul_r15, (int8_t)-70, (int8_t)-22));
  }
  HANDLE_CODE(pack_integer(bref, p_max_nr_r15, (int8_t)-30, (int8_t)33));
  if (ns_pmax_list_nr_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ns_pmax_list_nr_r15, 1, 8));
  }
  if (q_qual_min_r15_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_r15, (int8_t)-34, (int8_t)-3));
  }
  HANDLE_CODE(bref.pack(derive_ssb_idx_from_cell_r15, 1));
  if (max_rs_idx_cell_qual_r15_present) {
    HANDLE_CODE(pack_integer(bref, max_rs_idx_cell_qual_r15, (uint8_t)1u, (uint8_t)16u));
  }
  if (thresh_rs_idx_r15_present) {
    HANDLE_CODE(thresh_rs_idx_r15.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_nr_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(multi_band_info_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_sul_r15_present, 1));
  HANDLE_CODE(bref.unpack(meas_timing_cfg_r15_present, 1));
  HANDLE_CODE(bref.unpack(ss_rssi_meas_r15_present, 1));
  HANDLE_CODE(bref.unpack(cell_resel_prio_r15_present, 1));
  HANDLE_CODE(bref.unpack(cell_resel_sub_prio_r15_present, 1));
  HANDLE_CODE(bref.unpack(thresh_x_q_r15_present, 1));
  HANDLE_CODE(bref.unpack(q_rx_lev_min_sul_r15_present, 1));
  HANDLE_CODE(bref.unpack(ns_pmax_list_nr_r15_present, 1));
  HANDLE_CODE(bref.unpack(q_qual_min_r15_present, 1));
  HANDLE_CODE(bref.unpack(max_rs_idx_cell_qual_r15_present, 1));
  HANDLE_CODE(bref.unpack(thresh_rs_idx_r15_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq_r15, bref, (uint32_t)0u, (uint32_t)3279165u));
  if (multi_band_info_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_r15, bref, 1, 32, integer_packer<uint16_t>(1, 1024)));
  }
  if (multi_band_info_list_sul_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_sul_r15, bref, 1, 32, integer_packer<uint16_t>(1, 1024)));
  }
  if (meas_timing_cfg_r15_present) {
    HANDLE_CODE(meas_timing_cfg_r15.unpack(bref));
  }
  HANDLE_CODE(subcarrier_spacing_ssb_r15.unpack(bref));
  if (ss_rssi_meas_r15_present) {
    HANDLE_CODE(ss_rssi_meas_r15.unpack(bref));
  }
  if (cell_resel_prio_r15_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio_r15, bref, (uint8_t)0u, (uint8_t)7u));
  }
  if (cell_resel_sub_prio_r15_present) {
    HANDLE_CODE(cell_resel_sub_prio_r15.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high_r15, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low_r15, bref, (uint8_t)0u, (uint8_t)31u));
  if (thresh_x_q_r15_present) {
    HANDLE_CODE(unpack_integer(thresh_x_q_r15.thresh_x_high_q_r15, bref, (uint8_t)0u, (uint8_t)31u));
    HANDLE_CODE(unpack_integer(thresh_x_q_r15.thresh_x_low_q_r15, bref, (uint8_t)0u, (uint8_t)31u));
  }
  HANDLE_CODE(unpack_integer(q_rx_lev_min_r15, bref, (int8_t)-70, (int8_t)-22));
  if (q_rx_lev_min_sul_r15_present) {
    HANDLE_CODE(unpack_integer(q_rx_lev_min_sul_r15, bref, (int8_t)-70, (int8_t)-22));
  }
  HANDLE_CODE(unpack_integer(p_max_nr_r15, bref, (int8_t)-30, (int8_t)33));
  if (ns_pmax_list_nr_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ns_pmax_list_nr_r15, bref, 1, 8));
  }
  if (q_qual_min_r15_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_r15, bref, (int8_t)-34, (int8_t)-3));
  }
  HANDLE_CODE(bref.unpack(derive_ssb_idx_from_cell_r15, 1));
  if (max_rs_idx_cell_qual_r15_present) {
    HANDLE_CODE(unpack_integer(max_rs_idx_cell_qual_r15, bref, (uint8_t)1u, (uint8_t)16u));
  }
  if (thresh_rs_idx_r15_present) {
    HANDLE_CODE(thresh_rs_idx_r15.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void carrier_freq_nr_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq-r15", carrier_freq_r15);
  if (multi_band_info_list_r15_present) {
    j.start_array("multiBandInfoList-r15");
    for (const auto& e1 : multi_band_info_list_r15) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (multi_band_info_list_sul_r15_present) {
    j.start_array("multiBandInfoListSUL-r15");
    for (const auto& e1 : multi_band_info_list_sul_r15) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (meas_timing_cfg_r15_present) {
    j.write_fieldname("measTimingConfig-r15");
    meas_timing_cfg_r15.to_json(j);
  }
  j.write_str("subcarrierSpacingSSB-r15", subcarrier_spacing_ssb_r15.to_string());
  if (ss_rssi_meas_r15_present) {
    j.write_fieldname("ss-RSSI-Measurement-r15");
    ss_rssi_meas_r15.to_json(j);
  }
  if (cell_resel_prio_r15_present) {
    j.write_int("cellReselectionPriority-r15", cell_resel_prio_r15);
  }
  if (cell_resel_sub_prio_r15_present) {
    j.write_str("cellReselectionSubPriority-r15", cell_resel_sub_prio_r15.to_string());
  }
  j.write_int("threshX-High-r15", thresh_x_high_r15);
  j.write_int("threshX-Low-r15", thresh_x_low_r15);
  if (thresh_x_q_r15_present) {
    j.write_fieldname("threshX-Q-r15");
    j.start_obj();
    j.write_int("threshX-HighQ-r15", thresh_x_q_r15.thresh_x_high_q_r15);
    j.write_int("threshX-LowQ-r15", thresh_x_q_r15.thresh_x_low_q_r15);
    j.end_obj();
  }
  j.write_int("q-RxLevMin-r15", q_rx_lev_min_r15);
  if (q_rx_lev_min_sul_r15_present) {
    j.write_int("q-RxLevMinSUL-r15", q_rx_lev_min_sul_r15);
  }
  j.write_int("p-MaxNR-r15", p_max_nr_r15);
  if (ns_pmax_list_nr_r15_present) {
    j.start_array("ns-PmaxListNR-r15");
    for (const auto& e1 : ns_pmax_list_nr_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (q_qual_min_r15_present) {
    j.write_int("q-QualMin-r15", q_qual_min_r15);
  }
  j.write_bool("deriveSSB-IndexFromCell-r15", derive_ssb_idx_from_cell_r15);
  if (max_rs_idx_cell_qual_r15_present) {
    j.write_int("maxRS-IndexCellQual-r15", max_rs_idx_cell_qual_r15);
  }
  if (thresh_rs_idx_r15_present) {
    j.write_fieldname("threshRS-Index-r15");
    thresh_rs_idx_r15.to_json(j);
  }
  j.end_obj();
}

// CarrierFreqUTRA-FDD ::= SEQUENCE
SRSASN_CODE carrier_freq_utra_fdd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_resel_prio_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(pack_integer(bref, p_max_utra, (int8_t)-50, (int8_t)33));
  HANDLE_CODE(pack_integer(bref, q_qual_min, (int8_t)-24, (int8_t)0));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= thresh_x_q_r9.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(thresh_x_q_r9.is_present(), 1));
      if (thresh_x_q_r9.is_present()) {
        HANDLE_CODE(pack_integer(bref, thresh_x_q_r9->thresh_x_high_q_r9, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(pack_integer(bref, thresh_x_q_r9->thresh_x_low_q_r9, (uint8_t)0u, (uint8_t)31u));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_utra_fdd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_resel_prio_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq, bref, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(q_rx_lev_min, bref, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(unpack_integer(p_max_utra, bref, (int8_t)-50, (int8_t)33));
  HANDLE_CODE(unpack_integer(q_qual_min, bref, (int8_t)-24, (int8_t)0));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool thresh_x_q_r9_present;
      HANDLE_CODE(bref.unpack(thresh_x_q_r9_present, 1));
      thresh_x_q_r9.set_present(thresh_x_q_r9_present);
      if (thresh_x_q_r9.is_present()) {
        HANDLE_CODE(unpack_integer(thresh_x_q_r9->thresh_x_high_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(unpack_integer(thresh_x_q_r9->thresh_x_low_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void carrier_freq_utra_fdd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq", carrier_freq);
  if (cell_resel_prio_present) {
    j.write_int("cellReselectionPriority", cell_resel_prio);
  }
  j.write_int("threshX-High", thresh_x_high);
  j.write_int("threshX-Low", thresh_x_low);
  j.write_int("q-RxLevMin", q_rx_lev_min);
  j.write_int("p-MaxUTRA", p_max_utra);
  j.write_int("q-QualMin", q_qual_min);
  if (ext) {
    if (thresh_x_q_r9.is_present()) {
      j.write_fieldname("threshX-Q-r9");
      j.start_obj();
      j.write_int("threshX-HighQ-r9", thresh_x_q_r9->thresh_x_high_q_r9);
      j.write_int("threshX-LowQ-r9", thresh_x_q_r9->thresh_x_low_q_r9);
      j.end_obj();
    }
  }
  j.end_obj();
}

// CarrierFreqUTRA-FDD-Ext-r12 ::= SEQUENCE
SRSASN_CODE carrier_freq_utra_fdd_ext_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_resel_prio_r12_present, 1));
  HANDLE_CODE(bref.pack(thresh_x_q_r12_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_r12_present, 1));
  HANDLE_CODE(bref.pack(reduced_meas_performance_r12_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq_r12, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_r12_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio_r12, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_r12, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(pack_integer(bref, p_max_utra_r12, (int8_t)-50, (int8_t)33));
  HANDLE_CODE(pack_integer(bref, q_qual_min_r12, (int8_t)-24, (int8_t)0));
  if (thresh_x_q_r12_present) {
    HANDLE_CODE(pack_integer(bref, thresh_x_q_r12.thresh_x_high_q_r12, (uint8_t)0u, (uint8_t)31u));
    HANDLE_CODE(pack_integer(bref, thresh_x_q_r12.thresh_x_low_q_r12, (uint8_t)0u, (uint8_t)31u));
  }
  if (multi_band_info_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_r12, 1, 8, integer_packer<uint8_t>(1, 86)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_utra_fdd_ext_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_resel_prio_r12_present, 1));
  HANDLE_CODE(bref.unpack(thresh_x_q_r12_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_r12_present, 1));
  HANDLE_CODE(bref.unpack(reduced_meas_performance_r12_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq_r12, bref, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_r12_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio_r12, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(q_rx_lev_min_r12, bref, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(unpack_integer(p_max_utra_r12, bref, (int8_t)-50, (int8_t)33));
  HANDLE_CODE(unpack_integer(q_qual_min_r12, bref, (int8_t)-24, (int8_t)0));
  if (thresh_x_q_r12_present) {
    HANDLE_CODE(unpack_integer(thresh_x_q_r12.thresh_x_high_q_r12, bref, (uint8_t)0u, (uint8_t)31u));
    HANDLE_CODE(unpack_integer(thresh_x_q_r12.thresh_x_low_q_r12, bref, (uint8_t)0u, (uint8_t)31u));
  }
  if (multi_band_info_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_r12, bref, 1, 8, integer_packer<uint8_t>(1, 86)));
  }

  return SRSASN_SUCCESS;
}
void carrier_freq_utra_fdd_ext_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq-r12", carrier_freq_r12);
  if (cell_resel_prio_r12_present) {
    j.write_int("cellReselectionPriority-r12", cell_resel_prio_r12);
  }
  j.write_int("threshX-High-r12", thresh_x_high_r12);
  j.write_int("threshX-Low-r12", thresh_x_low_r12);
  j.write_int("q-RxLevMin-r12", q_rx_lev_min_r12);
  j.write_int("p-MaxUTRA-r12", p_max_utra_r12);
  j.write_int("q-QualMin-r12", q_qual_min_r12);
  if (thresh_x_q_r12_present) {
    j.write_fieldname("threshX-Q-r12");
    j.start_obj();
    j.write_int("threshX-HighQ-r12", thresh_x_q_r12.thresh_x_high_q_r12);
    j.write_int("threshX-LowQ-r12", thresh_x_q_r12.thresh_x_low_q_r12);
    j.end_obj();
  }
  if (multi_band_info_list_r12_present) {
    j.start_array("multiBandInfoList-r12");
    for (const auto& e1 : multi_band_info_list_r12) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (reduced_meas_performance_r12_present) {
    j.write_str("reducedMeasPerformance-r12", "true");
  }
  j.end_obj();
}

// CarrierFreqUTRA-TDD ::= SEQUENCE
SRSASN_CODE carrier_freq_utra_tdd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_resel_prio_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(pack_integer(bref, p_max_utra, (int8_t)-50, (int8_t)33));

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_utra_tdd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_resel_prio_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq, bref, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(q_rx_lev_min, bref, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(unpack_integer(p_max_utra, bref, (int8_t)-50, (int8_t)33));

  return SRSASN_SUCCESS;
}
void carrier_freq_utra_tdd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq", carrier_freq);
  if (cell_resel_prio_present) {
    j.write_int("cellReselectionPriority", cell_resel_prio);
  }
  j.write_int("threshX-High", thresh_x_high);
  j.write_int("threshX-Low", thresh_x_low);
  j.write_int("q-RxLevMin", q_rx_lev_min);
  j.write_int("p-MaxUTRA", p_max_utra);
  j.end_obj();
}

// CarrierFreqUTRA-TDD-r12 ::= SEQUENCE
SRSASN_CODE carrier_freq_utra_tdd_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_resel_prio_r12_present, 1));
  HANDLE_CODE(bref.pack(reduced_meas_performance_r12_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq_r12, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_r12_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio_r12, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_r12, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(pack_integer(bref, p_max_utra_r12, (int8_t)-50, (int8_t)33));

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_utra_tdd_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_resel_prio_r12_present, 1));
  HANDLE_CODE(bref.unpack(reduced_meas_performance_r12_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq_r12, bref, (uint16_t)0u, (uint16_t)16383u));
  if (cell_resel_prio_r12_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio_r12, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(q_rx_lev_min_r12, bref, (int8_t)-60, (int8_t)-13));
  HANDLE_CODE(unpack_integer(p_max_utra_r12, bref, (int8_t)-50, (int8_t)33));

  return SRSASN_SUCCESS;
}
void carrier_freq_utra_tdd_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq-r12", carrier_freq_r12);
  if (cell_resel_prio_r12_present) {
    j.write_int("cellReselectionPriority-r12", cell_resel_prio_r12);
  }
  j.write_int("threshX-High-r12", thresh_x_high_r12);
  j.write_int("threshX-Low-r12", thresh_x_low_r12);
  j.write_int("q-RxLevMin-r12", q_rx_lev_min_r12);
  j.write_int("p-MaxUTRA-r12", p_max_utra_r12);
  if (reduced_meas_performance_r12_present) {
    j.write_str("reducedMeasPerformance-r12", "true");
  }
  j.end_obj();
}

// CarrierFreqsInfoGERAN ::= SEQUENCE
SRSASN_CODE carrier_freqs_info_geran_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(carrier_freqs.pack(bref));
  HANDLE_CODE(bref.pack(common_info.cell_resel_prio_present, 1));
  HANDLE_CODE(bref.pack(common_info.p_max_geran_present, 1));
  if (common_info.cell_resel_prio_present) {
    HANDLE_CODE(pack_integer(bref, common_info.cell_resel_prio, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(common_info.ncc_permitted.pack(bref));
  HANDLE_CODE(pack_integer(bref, common_info.q_rx_lev_min, (uint8_t)0u, (uint8_t)45u));
  if (common_info.p_max_geran_present) {
    HANDLE_CODE(pack_integer(bref, common_info.p_max_geran, (uint8_t)0u, (uint8_t)39u));
  }
  HANDLE_CODE(pack_integer(bref, common_info.thresh_x_high, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, common_info.thresh_x_low, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freqs_info_geran_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(carrier_freqs.unpack(bref));
  HANDLE_CODE(bref.unpack(common_info.cell_resel_prio_present, 1));
  HANDLE_CODE(bref.unpack(common_info.p_max_geran_present, 1));
  if (common_info.cell_resel_prio_present) {
    HANDLE_CODE(unpack_integer(common_info.cell_resel_prio, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(common_info.ncc_permitted.unpack(bref));
  HANDLE_CODE(unpack_integer(common_info.q_rx_lev_min, bref, (uint8_t)0u, (uint8_t)45u));
  if (common_info.p_max_geran_present) {
    HANDLE_CODE(unpack_integer(common_info.p_max_geran, bref, (uint8_t)0u, (uint8_t)39u));
  }
  HANDLE_CODE(unpack_integer(common_info.thresh_x_high, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(common_info.thresh_x_low, bref, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
void carrier_freqs_info_geran_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("carrierFreqs");
  carrier_freqs.to_json(j);
  j.write_fieldname("commonInfo");
  j.start_obj();
  if (common_info.cell_resel_prio_present) {
    j.write_int("cellReselectionPriority", common_info.cell_resel_prio);
  }
  j.write_str("ncc-Permitted", common_info.ncc_permitted.to_string());
  j.write_int("q-RxLevMin", common_info.q_rx_lev_min);
  if (common_info.p_max_geran_present) {
    j.write_int("p-MaxGERAN", common_info.p_max_geran);
  }
  j.write_int("threshX-High", common_info.thresh_x_high);
  j.write_int("threshX-Low", common_info.thresh_x_low);
  j.end_obj();
  j.end_obj();
}

// CellSelectionInfo-v1130 ::= SEQUENCE
SRSASN_CODE cell_sel_info_v1130_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, q_qual_min_wb_r11, (int8_t)-34, (int8_t)-3));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_v1130_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(q_qual_min_wb_r11, bref, (int8_t)-34, (int8_t)-3));

  return SRSASN_SUCCESS;
}
void cell_sel_info_v1130_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-QualMinWB-r11", q_qual_min_wb_r11);
  j.end_obj();
}

// EAB-Config-r11 ::= SEQUENCE
SRSASN_CODE eab_cfg_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(eab_category_r11.pack(bref));
  HANDLE_CODE(eab_barr_bitmap_r11.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE eab_cfg_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(eab_category_r11.unpack(bref));
  HANDLE_CODE(eab_barr_bitmap_r11.unpack(bref));

  return SRSASN_SUCCESS;
}
void eab_cfg_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("eab-Category-r11", eab_category_r11.to_string());
  j.write_str("eab-BarringBitmap-r11", eab_barr_bitmap_r11.to_string());
  j.end_obj();
}

// FreqHoppingParameters-r13 ::= SEQUENCE
SRSASN_CODE freq_hop_params_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(dummy_present, 1));
  HANDLE_CODE(bref.pack(dummy2_present, 1));
  HANDLE_CODE(bref.pack(dummy3_present, 1));
  HANDLE_CODE(bref.pack(interv_ul_hop_cfg_common_mode_a_r13_present, 1));
  HANDLE_CODE(bref.pack(interv_ul_hop_cfg_common_mode_b_r13_present, 1));
  HANDLE_CODE(bref.pack(dummy4_present, 1));

  if (dummy_present) {
    HANDLE_CODE(dummy.pack(bref));
  }
  if (dummy2_present) {
    HANDLE_CODE(dummy2.pack(bref));
  }
  if (dummy3_present) {
    HANDLE_CODE(dummy3.pack(bref));
  }
  if (interv_ul_hop_cfg_common_mode_a_r13_present) {
    HANDLE_CODE(interv_ul_hop_cfg_common_mode_a_r13.pack(bref));
  }
  if (interv_ul_hop_cfg_common_mode_b_r13_present) {
    HANDLE_CODE(interv_ul_hop_cfg_common_mode_b_r13.pack(bref));
  }
  if (dummy4_present) {
    HANDLE_CODE(pack_integer(bref, dummy4, (uint8_t)1u, (uint8_t)16u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE freq_hop_params_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(dummy_present, 1));
  HANDLE_CODE(bref.unpack(dummy2_present, 1));
  HANDLE_CODE(bref.unpack(dummy3_present, 1));
  HANDLE_CODE(bref.unpack(interv_ul_hop_cfg_common_mode_a_r13_present, 1));
  HANDLE_CODE(bref.unpack(interv_ul_hop_cfg_common_mode_b_r13_present, 1));
  HANDLE_CODE(bref.unpack(dummy4_present, 1));

  if (dummy_present) {
    HANDLE_CODE(dummy.unpack(bref));
  }
  if (dummy2_present) {
    HANDLE_CODE(dummy2.unpack(bref));
  }
  if (dummy3_present) {
    HANDLE_CODE(dummy3.unpack(bref));
  }
  if (interv_ul_hop_cfg_common_mode_a_r13_present) {
    HANDLE_CODE(interv_ul_hop_cfg_common_mode_a_r13.unpack(bref));
  }
  if (interv_ul_hop_cfg_common_mode_b_r13_present) {
    HANDLE_CODE(interv_ul_hop_cfg_common_mode_b_r13.unpack(bref));
  }
  if (dummy4_present) {
    HANDLE_CODE(unpack_integer(dummy4, bref, (uint8_t)1u, (uint8_t)16u));
  }

  return SRSASN_SUCCESS;
}
void freq_hop_params_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dummy_present) {
    j.write_str("dummy", dummy.to_string());
  }
  if (dummy2_present) {
    j.write_fieldname("dummy2");
    dummy2.to_json(j);
  }
  if (dummy3_present) {
    j.write_fieldname("dummy3");
    dummy3.to_json(j);
  }
  if (interv_ul_hop_cfg_common_mode_a_r13_present) {
    j.write_fieldname("interval-ULHoppingConfigCommonModeA-r13");
    interv_ul_hop_cfg_common_mode_a_r13.to_json(j);
  }
  if (interv_ul_hop_cfg_common_mode_b_r13_present) {
    j.write_fieldname("interval-ULHoppingConfigCommonModeB-r13");
    interv_ul_hop_cfg_common_mode_b_r13.to_json(j);
  }
  if (dummy4_present) {
    j.write_int("dummy4", dummy4);
  }
  j.end_obj();
}

void freq_hop_params_r13_s::dummy2_c_::destroy_() {}
void freq_hop_params_r13_s::dummy2_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
freq_hop_params_r13_s::dummy2_c_::dummy2_c_(const freq_hop_params_r13_s::dummy2_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::interv_fdd_r13:
      c.init(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.init(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy2_c_");
  }
}
freq_hop_params_r13_s::dummy2_c_& freq_hop_params_r13_s::dummy2_c_::
                                  operator=(const freq_hop_params_r13_s::dummy2_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::interv_fdd_r13:
      c.set(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.set(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy2_c_");
  }

  return *this;
}
void freq_hop_params_r13_s::dummy2_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::interv_fdd_r13:
      j.write_str("interval-FDD-r13", c.get<interv_fdd_r13_e_>().to_string());
      break;
    case types::interv_tdd_r13:
      j.write_str("interval-TDD-r13", c.get<interv_tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy2_c_");
  }
  j.end_obj();
}
SRSASN_CODE freq_hop_params_r13_s::dummy2_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().pack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy2_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE freq_hop_params_r13_s::dummy2_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().unpack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy2_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void freq_hop_params_r13_s::dummy3_c_::destroy_() {}
void freq_hop_params_r13_s::dummy3_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
freq_hop_params_r13_s::dummy3_c_::dummy3_c_(const freq_hop_params_r13_s::dummy3_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::interv_fdd_r13:
      c.init(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.init(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy3_c_");
  }
}
freq_hop_params_r13_s::dummy3_c_& freq_hop_params_r13_s::dummy3_c_::
                                  operator=(const freq_hop_params_r13_s::dummy3_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::interv_fdd_r13:
      c.set(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.set(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy3_c_");
  }

  return *this;
}
void freq_hop_params_r13_s::dummy3_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::interv_fdd_r13:
      j.write_str("interval-FDD-r13", c.get<interv_fdd_r13_e_>().to_string());
      break;
    case types::interv_tdd_r13:
      j.write_str("interval-TDD-r13", c.get<interv_tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy3_c_");
  }
  j.end_obj();
}
SRSASN_CODE freq_hop_params_r13_s::dummy3_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().pack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy3_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE freq_hop_params_r13_s::dummy3_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().unpack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::dummy3_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::destroy_() {}
void freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::interv_ul_hop_cfg_common_mode_a_r13_c_(
    const freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::interv_fdd_r13:
      c.init(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.init(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_");
  }
}
freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_&
freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::
operator=(const freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::interv_fdd_r13:
      c.set(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.set(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_");
  }

  return *this;
}
void freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::interv_fdd_r13:
      j.write_str("interval-FDD-r13", c.get<interv_fdd_r13_e_>().to_string());
      break;
    case types::interv_tdd_r13:
      j.write_str("interval-TDD-r13", c.get<interv_tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().pack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().unpack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_a_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::destroy_() {}
void freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::interv_ul_hop_cfg_common_mode_b_r13_c_(
    const freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::interv_fdd_r13:
      c.init(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.init(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_");
  }
}
freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_&
freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::
operator=(const freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::interv_fdd_r13:
      c.set(other.c.get<interv_fdd_r13_e_>());
      break;
    case types::interv_tdd_r13:
      c.set(other.c.get<interv_tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_");
  }

  return *this;
}
void freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::interv_fdd_r13:
      j.write_str("interval-FDD-r13", c.get<interv_fdd_r13_e_>().to_string());
      break;
    case types::interv_tdd_r13:
      j.write_str("interval-TDD-r13", c.get<interv_tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().pack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::interv_fdd_r13:
      HANDLE_CODE(c.get<interv_fdd_r13_e_>().unpack(bref));
      break;
    case types::interv_tdd_r13:
      HANDLE_CODE(c.get<interv_tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "freq_hop_params_r13_s::interv_ul_hop_cfg_common_mode_b_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// HighSpeedConfig-r14 ::= SEQUENCE
SRSASN_CODE high_speed_cfg_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(high_speed_enhanced_meas_flag_r14_present, 1));
  HANDLE_CODE(bref.pack(high_speed_enhanced_demod_flag_r14_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE high_speed_cfg_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(high_speed_enhanced_meas_flag_r14_present, 1));
  HANDLE_CODE(bref.unpack(high_speed_enhanced_demod_flag_r14_present, 1));

  return SRSASN_SUCCESS;
}
void high_speed_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (high_speed_enhanced_meas_flag_r14_present) {
    j.write_str("highSpeedEnhancedMeasFlag-r14", "true");
  }
  if (high_speed_enhanced_demod_flag_r14_present) {
    j.write_str("highSpeedEnhancedDemodulationFlag-r14", "true");
  }
  j.end_obj();
}

// HighSpeedConfig-v1530 ::= SEQUENCE
SRSASN_CODE high_speed_cfg_v1530_s::pack(bit_ref& bref) const
{
  return SRSASN_SUCCESS;
}
SRSASN_CODE high_speed_cfg_v1530_s::unpack(cbit_ref& bref)
{
  return SRSASN_SUCCESS;
}
void high_speed_cfg_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("highSpeedMeasGapCE-ModeA-r15", "true");
  j.end_obj();
}

// InterFreqCarrierFreqInfo ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(p_max_present, 1));
  HANDLE_CODE(bref.pack(t_resel_eutra_sf_present, 1));
  HANDLE_CODE(bref.pack(cell_resel_prio_present, 1));
  HANDLE_CODE(bref.pack(q_offset_freq_present, 1));
  HANDLE_CODE(bref.pack(inter_freq_neigh_cell_list_present, 1));
  HANDLE_CODE(bref.pack(inter_freq_black_cell_list_present, 1));

  HANDLE_CODE(pack_integer(bref, dl_carrier_freq, (uint32_t)0u, (uint32_t)65535u));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min, (int8_t)-70, (int8_t)-22));
  if (p_max_present) {
    HANDLE_CODE(pack_integer(bref, p_max, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(pack_integer(bref, t_resel_eutra, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_eutra_sf_present) {
    HANDLE_CODE(t_resel_eutra_sf.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(allowed_meas_bw.pack(bref));
  HANDLE_CODE(bref.pack(presence_ant_port1, 1));
  if (cell_resel_prio_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(neigh_cell_cfg.pack(bref));
  if (q_offset_freq_present) {
    HANDLE_CODE(q_offset_freq.pack(bref));
  }
  if (inter_freq_neigh_cell_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_neigh_cell_list, 1, 16));
  }
  if (inter_freq_black_cell_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_black_cell_list, 1, 16));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= q_qual_min_r9_present;
    group_flags[0] |= thresh_x_q_r9.is_present();
    group_flags[1] |= q_qual_min_wb_r11_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(q_qual_min_r9_present, 1));
      HANDLE_CODE(bref.pack(thresh_x_q_r9.is_present(), 1));
      if (q_qual_min_r9_present) {
        HANDLE_CODE(pack_integer(bref, q_qual_min_r9, (int8_t)-34, (int8_t)-3));
      }
      if (thresh_x_q_r9.is_present()) {
        HANDLE_CODE(pack_integer(bref, thresh_x_q_r9->thresh_x_high_q_r9, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(pack_integer(bref, thresh_x_q_r9->thresh_x_low_q_r9, (uint8_t)0u, (uint8_t)31u));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(q_qual_min_wb_r11_present, 1));
      if (q_qual_min_wb_r11_present) {
        HANDLE_CODE(pack_integer(bref, q_qual_min_wb_r11, (int8_t)-34, (int8_t)-3));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(p_max_present, 1));
  HANDLE_CODE(bref.unpack(t_resel_eutra_sf_present, 1));
  HANDLE_CODE(bref.unpack(cell_resel_prio_present, 1));
  HANDLE_CODE(bref.unpack(q_offset_freq_present, 1));
  HANDLE_CODE(bref.unpack(inter_freq_neigh_cell_list_present, 1));
  HANDLE_CODE(bref.unpack(inter_freq_black_cell_list_present, 1));

  HANDLE_CODE(unpack_integer(dl_carrier_freq, bref, (uint32_t)0u, (uint32_t)65535u));
  HANDLE_CODE(unpack_integer(q_rx_lev_min, bref, (int8_t)-70, (int8_t)-22));
  if (p_max_present) {
    HANDLE_CODE(unpack_integer(p_max, bref, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(unpack_integer(t_resel_eutra, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_eutra_sf_present) {
    HANDLE_CODE(t_resel_eutra_sf.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(allowed_meas_bw.unpack(bref));
  HANDLE_CODE(bref.unpack(presence_ant_port1, 1));
  if (cell_resel_prio_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(neigh_cell_cfg.unpack(bref));
  if (q_offset_freq_present) {
    HANDLE_CODE(q_offset_freq.unpack(bref));
  }
  if (inter_freq_neigh_cell_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_neigh_cell_list, bref, 1, 16));
  }
  if (inter_freq_black_cell_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_black_cell_list, bref, 1, 16));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(q_qual_min_r9_present, 1));
      bool thresh_x_q_r9_present;
      HANDLE_CODE(bref.unpack(thresh_x_q_r9_present, 1));
      thresh_x_q_r9.set_present(thresh_x_q_r9_present);
      if (q_qual_min_r9_present) {
        HANDLE_CODE(unpack_integer(q_qual_min_r9, bref, (int8_t)-34, (int8_t)-3));
      }
      if (thresh_x_q_r9.is_present()) {
        HANDLE_CODE(unpack_integer(thresh_x_q_r9->thresh_x_high_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(unpack_integer(thresh_x_q_r9->thresh_x_low_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(q_qual_min_wb_r11_present, 1));
      if (q_qual_min_wb_r11_present) {
        HANDLE_CODE(unpack_integer(q_qual_min_wb_r11, bref, (int8_t)-34, (int8_t)-3));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dl-CarrierFreq", dl_carrier_freq);
  j.write_int("q-RxLevMin", q_rx_lev_min);
  if (p_max_present) {
    j.write_int("p-Max", p_max);
  }
  j.write_int("t-ReselectionEUTRA", t_resel_eutra);
  if (t_resel_eutra_sf_present) {
    j.write_fieldname("t-ReselectionEUTRA-SF");
    t_resel_eutra_sf.to_json(j);
  }
  j.write_int("threshX-High", thresh_x_high);
  j.write_int("threshX-Low", thresh_x_low);
  j.write_str("allowedMeasBandwidth", allowed_meas_bw.to_string());
  j.write_bool("presenceAntennaPort1", presence_ant_port1);
  if (cell_resel_prio_present) {
    j.write_int("cellReselectionPriority", cell_resel_prio);
  }
  j.write_str("neighCellConfig", neigh_cell_cfg.to_string());
  if (q_offset_freq_present) {
    j.write_str("q-OffsetFreq", q_offset_freq.to_string());
  }
  if (inter_freq_neigh_cell_list_present) {
    j.start_array("interFreqNeighCellList");
    for (const auto& e1 : inter_freq_neigh_cell_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (inter_freq_black_cell_list_present) {
    j.start_array("interFreqBlackCellList");
    for (const auto& e1 : inter_freq_black_cell_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ext) {
    if (q_qual_min_r9_present) {
      j.write_int("q-QualMin-r9", q_qual_min_r9);
    }
    if (thresh_x_q_r9.is_present()) {
      j.write_fieldname("threshX-Q-r9");
      j.start_obj();
      j.write_int("threshX-HighQ-r9", thresh_x_q_r9->thresh_x_high_q_r9);
      j.write_int("threshX-LowQ-r9", thresh_x_q_r9->thresh_x_low_q_r9);
      j.end_obj();
    }
    if (q_qual_min_wb_r11_present) {
      j.write_int("q-QualMinWB-r11", q_qual_min_wb_r11);
    }
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-r12 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(p_max_r12_present, 1));
  HANDLE_CODE(bref.pack(t_resel_eutra_sf_r12_present, 1));
  HANDLE_CODE(bref.pack(cell_resel_prio_r12_present, 1));
  HANDLE_CODE(bref.pack(q_offset_freq_r12_present, 1));
  HANDLE_CODE(bref.pack(inter_freq_neigh_cell_list_r12_present, 1));
  HANDLE_CODE(bref.pack(inter_freq_black_cell_list_r12_present, 1));
  HANDLE_CODE(bref.pack(q_qual_min_r12_present, 1));
  HANDLE_CODE(bref.pack(thresh_x_q_r12_present, 1));
  HANDLE_CODE(bref.pack(q_qual_min_wb_r12_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_r12_present, 1));
  HANDLE_CODE(bref.pack(reduced_meas_performance_r12_present, 1));
  HANDLE_CODE(bref.pack(q_qual_min_rsrq_on_all_symbols_r12_present, 1));

  HANDLE_CODE(pack_integer(bref, dl_carrier_freq_r12, (uint32_t)0u, (uint32_t)262143u));
  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_r12, (int8_t)-70, (int8_t)-22));
  if (p_max_r12_present) {
    HANDLE_CODE(pack_integer(bref, p_max_r12, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(pack_integer(bref, t_resel_eutra_r12, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_eutra_sf_r12_present) {
    HANDLE_CODE(t_resel_eutra_sf_r12.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, thresh_x_high_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, thresh_x_low_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(allowed_meas_bw_r12.pack(bref));
  HANDLE_CODE(bref.pack(presence_ant_port1_r12, 1));
  if (cell_resel_prio_r12_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_prio_r12, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(neigh_cell_cfg_r12.pack(bref));
  if (q_offset_freq_r12_present) {
    HANDLE_CODE(q_offset_freq_r12.pack(bref));
  }
  if (inter_freq_neigh_cell_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_neigh_cell_list_r12, 1, 16));
  }
  if (inter_freq_black_cell_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_black_cell_list_r12, 1, 16));
  }
  if (q_qual_min_r12_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_r12, (int8_t)-34, (int8_t)-3));
  }
  if (thresh_x_q_r12_present) {
    HANDLE_CODE(pack_integer(bref, thresh_x_q_r12.thresh_x_high_q_r12, (uint8_t)0u, (uint8_t)31u));
    HANDLE_CODE(pack_integer(bref, thresh_x_q_r12.thresh_x_low_q_r12, (uint8_t)0u, (uint8_t)31u));
  }
  if (q_qual_min_wb_r12_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_wb_r12, (int8_t)-34, (int8_t)-3));
  }
  if (multi_band_info_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_r12, 1, 8, integer_packer<uint16_t>(1, 256)));
  }
  if (q_qual_min_rsrq_on_all_symbols_r12_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_rsrq_on_all_symbols_r12, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(p_max_r12_present, 1));
  HANDLE_CODE(bref.unpack(t_resel_eutra_sf_r12_present, 1));
  HANDLE_CODE(bref.unpack(cell_resel_prio_r12_present, 1));
  HANDLE_CODE(bref.unpack(q_offset_freq_r12_present, 1));
  HANDLE_CODE(bref.unpack(inter_freq_neigh_cell_list_r12_present, 1));
  HANDLE_CODE(bref.unpack(inter_freq_black_cell_list_r12_present, 1));
  HANDLE_CODE(bref.unpack(q_qual_min_r12_present, 1));
  HANDLE_CODE(bref.unpack(thresh_x_q_r12_present, 1));
  HANDLE_CODE(bref.unpack(q_qual_min_wb_r12_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_r12_present, 1));
  HANDLE_CODE(bref.unpack(reduced_meas_performance_r12_present, 1));
  HANDLE_CODE(bref.unpack(q_qual_min_rsrq_on_all_symbols_r12_present, 1));

  HANDLE_CODE(unpack_integer(dl_carrier_freq_r12, bref, (uint32_t)0u, (uint32_t)262143u));
  HANDLE_CODE(unpack_integer(q_rx_lev_min_r12, bref, (int8_t)-70, (int8_t)-22));
  if (p_max_r12_present) {
    HANDLE_CODE(unpack_integer(p_max_r12, bref, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(unpack_integer(t_resel_eutra_r12, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_eutra_sf_r12_present) {
    HANDLE_CODE(t_resel_eutra_sf_r12.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(thresh_x_high_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(thresh_x_low_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(allowed_meas_bw_r12.unpack(bref));
  HANDLE_CODE(bref.unpack(presence_ant_port1_r12, 1));
  if (cell_resel_prio_r12_present) {
    HANDLE_CODE(unpack_integer(cell_resel_prio_r12, bref, (uint8_t)0u, (uint8_t)7u));
  }
  HANDLE_CODE(neigh_cell_cfg_r12.unpack(bref));
  if (q_offset_freq_r12_present) {
    HANDLE_CODE(q_offset_freq_r12.unpack(bref));
  }
  if (inter_freq_neigh_cell_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_neigh_cell_list_r12, bref, 1, 16));
  }
  if (inter_freq_black_cell_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_black_cell_list_r12, bref, 1, 16));
  }
  if (q_qual_min_r12_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_r12, bref, (int8_t)-34, (int8_t)-3));
  }
  if (thresh_x_q_r12_present) {
    HANDLE_CODE(unpack_integer(thresh_x_q_r12.thresh_x_high_q_r12, bref, (uint8_t)0u, (uint8_t)31u));
    HANDLE_CODE(unpack_integer(thresh_x_q_r12.thresh_x_low_q_r12, bref, (uint8_t)0u, (uint8_t)31u));
  }
  if (q_qual_min_wb_r12_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_wb_r12, bref, (int8_t)-34, (int8_t)-3));
  }
  if (multi_band_info_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_r12, bref, 1, 8, integer_packer<uint16_t>(1, 256)));
  }
  if (q_qual_min_rsrq_on_all_symbols_r12_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_rsrq_on_all_symbols_r12, bref, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dl-CarrierFreq-r12", dl_carrier_freq_r12);
  j.write_int("q-RxLevMin-r12", q_rx_lev_min_r12);
  if (p_max_r12_present) {
    j.write_int("p-Max-r12", p_max_r12);
  }
  j.write_int("t-ReselectionEUTRA-r12", t_resel_eutra_r12);
  if (t_resel_eutra_sf_r12_present) {
    j.write_fieldname("t-ReselectionEUTRA-SF-r12");
    t_resel_eutra_sf_r12.to_json(j);
  }
  j.write_int("threshX-High-r12", thresh_x_high_r12);
  j.write_int("threshX-Low-r12", thresh_x_low_r12);
  j.write_str("allowedMeasBandwidth-r12", allowed_meas_bw_r12.to_string());
  j.write_bool("presenceAntennaPort1-r12", presence_ant_port1_r12);
  if (cell_resel_prio_r12_present) {
    j.write_int("cellReselectionPriority-r12", cell_resel_prio_r12);
  }
  j.write_str("neighCellConfig-r12", neigh_cell_cfg_r12.to_string());
  if (q_offset_freq_r12_present) {
    j.write_str("q-OffsetFreq-r12", q_offset_freq_r12.to_string());
  }
  if (inter_freq_neigh_cell_list_r12_present) {
    j.start_array("interFreqNeighCellList-r12");
    for (const auto& e1 : inter_freq_neigh_cell_list_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (inter_freq_black_cell_list_r12_present) {
    j.start_array("interFreqBlackCellList-r12");
    for (const auto& e1 : inter_freq_black_cell_list_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (q_qual_min_r12_present) {
    j.write_int("q-QualMin-r12", q_qual_min_r12);
  }
  if (thresh_x_q_r12_present) {
    j.write_fieldname("threshX-Q-r12");
    j.start_obj();
    j.write_int("threshX-HighQ-r12", thresh_x_q_r12.thresh_x_high_q_r12);
    j.write_int("threshX-LowQ-r12", thresh_x_q_r12.thresh_x_low_q_r12);
    j.end_obj();
  }
  if (q_qual_min_wb_r12_present) {
    j.write_int("q-QualMinWB-r12", q_qual_min_wb_r12);
  }
  if (multi_band_info_list_r12_present) {
    j.start_array("multiBandInfoList-r12");
    for (const auto& e1 : multi_band_info_list_r12) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (reduced_meas_performance_r12_present) {
    j.write_str("reducedMeasPerformance-r12", "true");
  }
  if (q_qual_min_rsrq_on_all_symbols_r12_present) {
    j.write_int("q-QualMinRSRQ-OnAllSymbols-r12", q_qual_min_rsrq_on_all_symbols_r12);
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v1250 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(reduced_meas_performance_r12_present, 1));
  HANDLE_CODE(bref.pack(q_qual_min_rsrq_on_all_symbols_r12_present, 1));

  if (q_qual_min_rsrq_on_all_symbols_r12_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_rsrq_on_all_symbols_r12, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(reduced_meas_performance_r12_present, 1));
  HANDLE_CODE(bref.unpack(q_qual_min_rsrq_on_all_symbols_r12_present, 1));

  if (q_qual_min_rsrq_on_all_symbols_r12_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_rsrq_on_all_symbols_r12, bref, (int8_t)-34, (int8_t)-3));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (reduced_meas_performance_r12_present) {
    j.write_str("reducedMeasPerformance-r12", "true");
  }
  if (q_qual_min_rsrq_on_all_symbols_r12_present) {
    j.write_int("q-QualMinRSRQ-OnAllSymbols-r12", q_qual_min_rsrq_on_all_symbols_r12);
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v1310 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cell_resel_sub_prio_r13_present, 1));
  HANDLE_CODE(bref.pack(redist_inter_freq_info_r13_present, 1));
  HANDLE_CODE(bref.pack(cell_sel_info_ce_r13_present, 1));
  HANDLE_CODE(bref.pack(t_resel_eutra_ce_r13_present, 1));

  if (cell_resel_sub_prio_r13_present) {
    HANDLE_CODE(cell_resel_sub_prio_r13.pack(bref));
  }
  if (redist_inter_freq_info_r13_present) {
    HANDLE_CODE(redist_inter_freq_info_r13.pack(bref));
  }
  if (cell_sel_info_ce_r13_present) {
    HANDLE_CODE(cell_sel_info_ce_r13.pack(bref));
  }
  if (t_resel_eutra_ce_r13_present) {
    HANDLE_CODE(pack_integer(bref, t_resel_eutra_ce_r13, (uint8_t)0u, (uint8_t)15u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cell_resel_sub_prio_r13_present, 1));
  HANDLE_CODE(bref.unpack(redist_inter_freq_info_r13_present, 1));
  HANDLE_CODE(bref.unpack(cell_sel_info_ce_r13_present, 1));
  HANDLE_CODE(bref.unpack(t_resel_eutra_ce_r13_present, 1));

  if (cell_resel_sub_prio_r13_present) {
    HANDLE_CODE(cell_resel_sub_prio_r13.unpack(bref));
  }
  if (redist_inter_freq_info_r13_present) {
    HANDLE_CODE(redist_inter_freq_info_r13.unpack(bref));
  }
  if (cell_sel_info_ce_r13_present) {
    HANDLE_CODE(cell_sel_info_ce_r13.unpack(bref));
  }
  if (t_resel_eutra_ce_r13_present) {
    HANDLE_CODE(unpack_integer(t_resel_eutra_ce_r13, bref, (uint8_t)0u, (uint8_t)15u));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cell_resel_sub_prio_r13_present) {
    j.write_str("cellReselectionSubPriority-r13", cell_resel_sub_prio_r13.to_string());
  }
  if (redist_inter_freq_info_r13_present) {
    j.write_fieldname("redistributionInterFreqInfo-r13");
    redist_inter_freq_info_r13.to_json(j);
  }
  if (cell_sel_info_ce_r13_present) {
    j.write_fieldname("cellSelectionInfoCE-r13");
    cell_sel_info_ce_r13.to_json(j);
  }
  if (t_resel_eutra_ce_r13_present) {
    j.write_int("t-ReselectionEUTRA-CE-r13", t_resel_eutra_ce_r13);
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v1350 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v1350_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cell_sel_info_ce1_r13_present, 1));

  if (cell_sel_info_ce1_r13_present) {
    HANDLE_CODE(cell_sel_info_ce1_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v1350_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cell_sel_info_ce1_r13_present, 1));

  if (cell_sel_info_ce1_r13_present) {
    HANDLE_CODE(cell_sel_info_ce1_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v1350_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cell_sel_info_ce1_r13_present) {
    j.write_fieldname("cellSelectionInfoCE1-r13");
    cell_sel_info_ce1_r13.to_json(j);
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v1530 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(inter_freq_neigh_hsdn_cell_list_r15_present, 1));
  HANDLE_CODE(bref.pack(cell_sel_info_ce_v1530_present, 1));

  HANDLE_CODE(bref.pack(hsdn_ind_r15, 1));
  if (inter_freq_neigh_hsdn_cell_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_neigh_hsdn_cell_list_r15, 1, 16));
  }
  if (cell_sel_info_ce_v1530_present) {
    HANDLE_CODE(cell_sel_info_ce_v1530.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(inter_freq_neigh_hsdn_cell_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(cell_sel_info_ce_v1530_present, 1));

  HANDLE_CODE(bref.unpack(hsdn_ind_r15, 1));
  if (inter_freq_neigh_hsdn_cell_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_neigh_hsdn_cell_list_r15, bref, 1, 16));
  }
  if (cell_sel_info_ce_v1530_present) {
    HANDLE_CODE(cell_sel_info_ce_v1530.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_bool("hsdn-Indication-r15", hsdn_ind_r15);
  if (inter_freq_neigh_hsdn_cell_list_r15_present) {
    j.start_array("interFreqNeighHSDN-CellList-r15");
    for (const auto& e1 : inter_freq_neigh_hsdn_cell_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (cell_sel_info_ce_v1530_present) {
    j.write_fieldname("cellSelectionInfoCE-v1530");
    cell_sel_info_ce_v1530.to_json(j);
  }
  j.end_obj();
}

// InterFreqCarrierFreqInfo-v8h0 ::= SEQUENCE
SRSASN_CODE inter_freq_carrier_freq_info_v8h0_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(multi_band_info_list_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list, 1, 8, integer_packer<uint8_t>(1, 64)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_freq_carrier_freq_info_v8h0_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(multi_band_info_list_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list, bref, 1, 8, integer_packer<uint8_t>(1, 64)));
  }

  return SRSASN_SUCCESS;
}
void inter_freq_carrier_freq_info_v8h0_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (multi_band_info_list_present) {
    j.start_array("multiBandInfoList");
    for (const auto& e1 : multi_band_info_list) {
      j.write_int(e1);
    }
    j.end_array();
  }
  j.end_obj();
}

// IntraFreqNeighCellInfo ::= SEQUENCE
SRSASN_CODE intra_freq_neigh_cell_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, pci, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(q_offset_cell.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE intra_freq_neigh_cell_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(pci, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(q_offset_cell.unpack(bref));

  return SRSASN_SUCCESS;
}
void intra_freq_neigh_cell_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId", pci);
  j.write_str("q-OffsetCell", q_offset_cell.to_string());
  j.end_obj();
}

// MBMS-CarrierType-r14 ::= SEQUENCE
SRSASN_CODE mbms_carrier_type_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(frame_offset_r14_present, 1));

  HANDLE_CODE(carrier_type_r14.pack(bref));
  if (frame_offset_r14_present) {
    HANDLE_CODE(pack_integer(bref, frame_offset_r14, (uint8_t)0u, (uint8_t)3u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbms_carrier_type_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(frame_offset_r14_present, 1));

  HANDLE_CODE(carrier_type_r14.unpack(bref));
  if (frame_offset_r14_present) {
    HANDLE_CODE(unpack_integer(frame_offset_r14, bref, (uint8_t)0u, (uint8_t)3u));
  }

  return SRSASN_SUCCESS;
}
void mbms_carrier_type_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("carrierType-r14", carrier_type_r14.to_string());
  if (frame_offset_r14_present) {
    j.write_int("frameOffset-r14", frame_offset_r14);
  }
  j.end_obj();
}

// MBMS-SAI-InterFreq-r11 ::= SEQUENCE
SRSASN_CODE mbms_sai_inter_freq_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, dl_carrier_freq_r11, (uint32_t)0u, (uint32_t)262143u));
  HANDLE_CODE(pack_dyn_seq_of(bref, mbms_sai_list_r11, 1, 64, integer_packer<uint32_t>(0, 65535)));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbms_sai_inter_freq_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(dl_carrier_freq_r11, bref, (uint32_t)0u, (uint32_t)262143u));
  HANDLE_CODE(unpack_dyn_seq_of(mbms_sai_list_r11, bref, 1, 64, integer_packer<uint32_t>(0, 65535)));

  return SRSASN_SUCCESS;
}
void mbms_sai_inter_freq_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dl-CarrierFreq-r11", dl_carrier_freq_r11);
  j.start_array("mbms-SAI-List-r11");
  for (const auto& e1 : mbms_sai_list_r11) {
    j.write_int(e1);
  }
  j.end_array();
  j.end_obj();
}

// MBMS-SAI-InterFreq-v1140 ::= SEQUENCE
SRSASN_CODE mbms_sai_inter_freq_v1140_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(multi_band_info_list_r11_present, 1));

  if (multi_band_info_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_r11, 1, 8, integer_packer<uint16_t>(1, 256)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbms_sai_inter_freq_v1140_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(multi_band_info_list_r11_present, 1));

  if (multi_band_info_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_r11, bref, 1, 8, integer_packer<uint16_t>(1, 256)));
  }

  return SRSASN_SUCCESS;
}
void mbms_sai_inter_freq_v1140_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (multi_band_info_list_r11_present) {
    j.start_array("multiBandInfoList-r11");
    for (const auto& e1 : multi_band_info_list_r11) {
      j.write_int(e1);
    }
    j.end_array();
  }
  j.end_obj();
}

// MBSFN-AreaInfo-r9 ::= SEQUENCE
SRSASN_CODE mbsfn_area_info_r9_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, mbsfn_area_id_r9, (uint16_t)0u, (uint16_t)255u));
  HANDLE_CODE(non_mbsfn_region_len.pack(bref));
  HANDLE_CODE(pack_integer(bref, notif_ind_r9, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(mcch_cfg_r9.mcch_repeat_period_r9.pack(bref));
  HANDLE_CODE(pack_integer(bref, mcch_cfg_r9.mcch_offset_r9, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(mcch_cfg_r9.mcch_mod_period_r9.pack(bref));
  HANDLE_CODE(mcch_cfg_r9.sf_alloc_info_r9.pack(bref));
  HANDLE_CODE(mcch_cfg_r9.sig_mcs_r9.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= mcch_cfg_r14.is_present();
    group_flags[0] |= subcarrier_spacing_mbms_r14_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mcch_cfg_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(subcarrier_spacing_mbms_r14_present, 1));
      if (mcch_cfg_r14.is_present()) {
        HANDLE_CODE(bref.pack(mcch_cfg_r14->mcch_repeat_period_v1430_present, 1));
        HANDLE_CODE(bref.pack(mcch_cfg_r14->mcch_mod_period_v1430_present, 1));
        if (mcch_cfg_r14->mcch_repeat_period_v1430_present) {
          HANDLE_CODE(mcch_cfg_r14->mcch_repeat_period_v1430.pack(bref));
        }
        if (mcch_cfg_r14->mcch_mod_period_v1430_present) {
          HANDLE_CODE(mcch_cfg_r14->mcch_mod_period_v1430.pack(bref));
        }
      }
      if (subcarrier_spacing_mbms_r14_present) {
        HANDLE_CODE(subcarrier_spacing_mbms_r14.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_area_info_r9_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(mbsfn_area_id_r9, bref, (uint16_t)0u, (uint16_t)255u));
  HANDLE_CODE(non_mbsfn_region_len.unpack(bref));
  HANDLE_CODE(unpack_integer(notif_ind_r9, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(mcch_cfg_r9.mcch_repeat_period_r9.unpack(bref));
  HANDLE_CODE(unpack_integer(mcch_cfg_r9.mcch_offset_r9, bref, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(mcch_cfg_r9.mcch_mod_period_r9.unpack(bref));
  HANDLE_CODE(mcch_cfg_r9.sf_alloc_info_r9.unpack(bref));
  HANDLE_CODE(mcch_cfg_r9.sig_mcs_r9.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mcch_cfg_r14_present;
      HANDLE_CODE(bref.unpack(mcch_cfg_r14_present, 1));
      mcch_cfg_r14.set_present(mcch_cfg_r14_present);
      HANDLE_CODE(bref.unpack(subcarrier_spacing_mbms_r14_present, 1));
      if (mcch_cfg_r14.is_present()) {
        HANDLE_CODE(bref.unpack(mcch_cfg_r14->mcch_repeat_period_v1430_present, 1));
        HANDLE_CODE(bref.unpack(mcch_cfg_r14->mcch_mod_period_v1430_present, 1));
        if (mcch_cfg_r14->mcch_repeat_period_v1430_present) {
          HANDLE_CODE(mcch_cfg_r14->mcch_repeat_period_v1430.unpack(bref));
        }
        if (mcch_cfg_r14->mcch_mod_period_v1430_present) {
          HANDLE_CODE(mcch_cfg_r14->mcch_mod_period_v1430.unpack(bref));
        }
      }
      if (subcarrier_spacing_mbms_r14_present) {
        HANDLE_CODE(subcarrier_spacing_mbms_r14.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void mbsfn_area_info_r9_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("mbsfn-AreaId-r9", mbsfn_area_id_r9);
  j.write_str("non-MBSFNregionLength", non_mbsfn_region_len.to_string());
  j.write_int("notificationIndicator-r9", notif_ind_r9);
  j.write_fieldname("mcch-Config-r9");
  j.start_obj();
  j.write_str("mcch-RepetitionPeriod-r9", mcch_cfg_r9.mcch_repeat_period_r9.to_string());
  j.write_int("mcch-Offset-r9", mcch_cfg_r9.mcch_offset_r9);
  j.write_str("mcch-ModificationPeriod-r9", mcch_cfg_r9.mcch_mod_period_r9.to_string());
  j.write_str("sf-AllocInfo-r9", mcch_cfg_r9.sf_alloc_info_r9.to_string());
  j.write_str("signallingMCS-r9", mcch_cfg_r9.sig_mcs_r9.to_string());
  j.end_obj();
  if (ext) {
    if (mcch_cfg_r14.is_present()) {
      j.write_fieldname("mcch-Config-r14");
      j.start_obj();
      if (mcch_cfg_r14->mcch_repeat_period_v1430_present) {
        j.write_str("mcch-RepetitionPeriod-v1430", mcch_cfg_r14->mcch_repeat_period_v1430.to_string());
      }
      if (mcch_cfg_r14->mcch_mod_period_v1430_present) {
        j.write_str("mcch-ModificationPeriod-v1430", mcch_cfg_r14->mcch_mod_period_v1430.to_string());
      }
      j.end_obj();
    }
    if (subcarrier_spacing_mbms_r14_present) {
      j.write_str("subcarrierSpacingMBMS-r14", subcarrier_spacing_mbms_r14.to_string());
    }
  }
  j.end_obj();
}

// MBSFN-SubframeConfig ::= SEQUENCE
SRSASN_CODE mbsfn_sf_cfg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(radioframe_alloc_period.pack(bref));
  HANDLE_CODE(pack_integer(bref, radioframe_alloc_offset, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(sf_alloc.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_sf_cfg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(radioframe_alloc_period.unpack(bref));
  HANDLE_CODE(unpack_integer(radioframe_alloc_offset, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(sf_alloc.unpack(bref));

  return SRSASN_SUCCESS;
}
void mbsfn_sf_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("radioframeAllocationPeriod", radioframe_alloc_period.to_string());
  j.write_int("radioframeAllocationOffset", radioframe_alloc_offset);
  j.write_fieldname("subframeAllocation");
  sf_alloc.to_json(j);
  j.end_obj();
}

void mbsfn_sf_cfg_s::sf_alloc_c_::destroy_()
{
  switch (type_) {
    case types::one_frame:
      c.destroy<fixed_bitstring<6> >();
      break;
    case types::four_frames:
      c.destroy<fixed_bitstring<24> >();
      break;
    default:
      break;
  }
}
void mbsfn_sf_cfg_s::sf_alloc_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::one_frame:
      c.init<fixed_bitstring<6> >();
      break;
    case types::four_frames:
      c.init<fixed_bitstring<24> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_s::sf_alloc_c_");
  }
}
mbsfn_sf_cfg_s::sf_alloc_c_::sf_alloc_c_(const mbsfn_sf_cfg_s::sf_alloc_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::one_frame:
      c.init(other.c.get<fixed_bitstring<6> >());
      break;
    case types::four_frames:
      c.init(other.c.get<fixed_bitstring<24> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_s::sf_alloc_c_");
  }
}
mbsfn_sf_cfg_s::sf_alloc_c_& mbsfn_sf_cfg_s::sf_alloc_c_::operator=(const mbsfn_sf_cfg_s::sf_alloc_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::one_frame:
      c.set(other.c.get<fixed_bitstring<6> >());
      break;
    case types::four_frames:
      c.set(other.c.get<fixed_bitstring<24> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_s::sf_alloc_c_");
  }

  return *this;
}
void mbsfn_sf_cfg_s::sf_alloc_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::one_frame:
      j.write_str("oneFrame", c.get<fixed_bitstring<6> >().to_string());
      break;
    case types::four_frames:
      j.write_str("fourFrames", c.get<fixed_bitstring<24> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_s::sf_alloc_c_");
  }
  j.end_obj();
}
SRSASN_CODE mbsfn_sf_cfg_s::sf_alloc_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::one_frame:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().pack(bref));
      break;
    case types::four_frames:
      HANDLE_CODE(c.get<fixed_bitstring<24> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_s::sf_alloc_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_sf_cfg_s::sf_alloc_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::one_frame:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().unpack(bref));
      break;
    case types::four_frames:
      HANDLE_CODE(c.get<fixed_bitstring<24> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_s::sf_alloc_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// MBSFN-SubframeConfig-v1430 ::= SEQUENCE
SRSASN_CODE mbsfn_sf_cfg_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sf_alloc_v1430.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_sf_cfg_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sf_alloc_v1430.unpack(bref));

  return SRSASN_SUCCESS;
}
void mbsfn_sf_cfg_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("subframeAllocation-v1430");
  sf_alloc_v1430.to_json(j);
  j.end_obj();
}

void mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::destroy_()
{
  switch (type_) {
    case types::one_frame_v1430:
      c.destroy<fixed_bitstring<2> >();
      break;
    case types::four_frames_v1430:
      c.destroy<fixed_bitstring<8> >();
      break;
    default:
      break;
  }
}
void mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::one_frame_v1430:
      c.init<fixed_bitstring<2> >();
      break;
    case types::four_frames_v1430:
      c.init<fixed_bitstring<8> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_");
  }
}
mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::sf_alloc_v1430_c_(const mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::one_frame_v1430:
      c.init(other.c.get<fixed_bitstring<2> >());
      break;
    case types::four_frames_v1430:
      c.init(other.c.get<fixed_bitstring<8> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_");
  }
}
mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_& mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::
                                         operator=(const mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::one_frame_v1430:
      c.set(other.c.get<fixed_bitstring<2> >());
      break;
    case types::four_frames_v1430:
      c.set(other.c.get<fixed_bitstring<8> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_");
  }

  return *this;
}
void mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::one_frame_v1430:
      j.write_str("oneFrame-v1430", c.get<fixed_bitstring<2> >().to_string());
      break;
    case types::four_frames_v1430:
      j.write_str("fourFrames-v1430", c.get<fixed_bitstring<8> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_");
  }
  j.end_obj();
}
SRSASN_CODE mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::one_frame_v1430:
      HANDLE_CODE(c.get<fixed_bitstring<2> >().pack(bref));
      break;
    case types::four_frames_v1430:
      HANDLE_CODE(c.get<fixed_bitstring<8> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::one_frame_v1430:
      HANDLE_CODE(c.get<fixed_bitstring<2> >().unpack(bref));
      break;
    case types::four_frames_v1430:
      HANDLE_CODE(c.get<fixed_bitstring<8> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "mbsfn_sf_cfg_v1430_s::sf_alloc_v1430_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// PCCH-Config ::= SEQUENCE
SRSASN_CODE pcch_cfg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(default_paging_cycle.pack(bref));
  HANDLE_CODE(nb.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pcch_cfg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(default_paging_cycle.unpack(bref));
  HANDLE_CODE(nb.unpack(bref));

  return SRSASN_SUCCESS;
}
void pcch_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("defaultPagingCycle", default_paging_cycle.to_string());
  j.write_str("nB", nb.to_string());
  j.end_obj();
}

// PCCH-Config-v1310 ::= SEQUENCE
SRSASN_CODE pcch_cfg_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(nb_v1310_present, 1));

  HANDLE_CODE(pack_integer(bref, paging_narrow_bands_r13, (uint8_t)1u, (uint8_t)16u));
  HANDLE_CODE(mpdcch_num_repeat_paging_r13.pack(bref));
  if (nb_v1310_present) {
    HANDLE_CODE(nb_v1310.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pcch_cfg_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(nb_v1310_present, 1));

  HANDLE_CODE(unpack_integer(paging_narrow_bands_r13, bref, (uint8_t)1u, (uint8_t)16u));
  HANDLE_CODE(mpdcch_num_repeat_paging_r13.unpack(bref));
  if (nb_v1310_present) {
    HANDLE_CODE(nb_v1310.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pcch_cfg_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("paging-narrowBands-r13", paging_narrow_bands_r13);
  j.write_str("mpdcch-NumRepetition-Paging-r13", mpdcch_num_repeat_paging_r13.to_string());
  if (nb_v1310_present) {
    j.write_str("nB-v1310", nb_v1310.to_string());
  }
  j.end_obj();
}

// PDSCH-ConfigCommon ::= SEQUENCE
SRSASN_CODE pdsch_cfg_common_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, ref_sig_pwr, (int8_t)-60, (int8_t)50));
  HANDLE_CODE(pack_integer(bref, p_b, (uint8_t)0u, (uint8_t)3u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdsch_cfg_common_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(ref_sig_pwr, bref, (int8_t)-60, (int8_t)50));
  HANDLE_CODE(unpack_integer(p_b, bref, (uint8_t)0u, (uint8_t)3u));

  return SRSASN_SUCCESS;
}
void pdsch_cfg_common_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("referenceSignalPower", ref_sig_pwr);
  j.write_int("p-b", p_b);
  j.end_obj();
}

// PDSCH-ConfigCommon-v1310 ::= SEQUENCE
SRSASN_CODE pdsch_cfg_common_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(pdsch_max_num_repeat_cemode_a_r13_present, 1));
  HANDLE_CODE(bref.pack(pdsch_max_num_repeat_cemode_b_r13_present, 1));

  if (pdsch_max_num_repeat_cemode_a_r13_present) {
    HANDLE_CODE(pdsch_max_num_repeat_cemode_a_r13.pack(bref));
  }
  if (pdsch_max_num_repeat_cemode_b_r13_present) {
    HANDLE_CODE(pdsch_max_num_repeat_cemode_b_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdsch_cfg_common_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(pdsch_max_num_repeat_cemode_a_r13_present, 1));
  HANDLE_CODE(bref.unpack(pdsch_max_num_repeat_cemode_b_r13_present, 1));

  if (pdsch_max_num_repeat_cemode_a_r13_present) {
    HANDLE_CODE(pdsch_max_num_repeat_cemode_a_r13.unpack(bref));
  }
  if (pdsch_max_num_repeat_cemode_b_r13_present) {
    HANDLE_CODE(pdsch_max_num_repeat_cemode_b_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdsch_cfg_common_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (pdsch_max_num_repeat_cemode_a_r13_present) {
    j.write_str("pdsch-maxNumRepetitionCEmodeA-r13", pdsch_max_num_repeat_cemode_a_r13.to_string());
  }
  if (pdsch_max_num_repeat_cemode_b_r13_present) {
    j.write_str("pdsch-maxNumRepetitionCEmodeB-r13", pdsch_max_num_repeat_cemode_b_r13.to_string());
  }
  j.end_obj();
}

// PLMN-Info-r15 ::= SEQUENCE
SRSASN_CODE plmn_info_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(upper_layer_ind_r15_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_info_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(upper_layer_ind_r15_present, 1));

  return SRSASN_SUCCESS;
}
void plmn_info_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (upper_layer_ind_r15_present) {
    j.write_str("upperLayerIndication-r15", "true");
  }
  j.end_obj();
}

// PRACH-Config-v1430 ::= SEQUENCE
SRSASN_CODE prach_cfg_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, root_seq_idx_high_speed_r14, (uint16_t)0u, (uint16_t)837u));
  HANDLE_CODE(pack_integer(bref, zero_correlation_zone_cfg_high_speed_r14, (uint8_t)0u, (uint8_t)12u));
  HANDLE_CODE(pack_integer(bref, prach_cfg_idx_high_speed_r14, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(pack_integer(bref, prach_freq_offset_high_speed_r14, (uint8_t)0u, (uint8_t)94u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_cfg_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(root_seq_idx_high_speed_r14, bref, (uint16_t)0u, (uint16_t)837u));
  HANDLE_CODE(unpack_integer(zero_correlation_zone_cfg_high_speed_r14, bref, (uint8_t)0u, (uint8_t)12u));
  HANDLE_CODE(unpack_integer(prach_cfg_idx_high_speed_r14, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(unpack_integer(prach_freq_offset_high_speed_r14, bref, (uint8_t)0u, (uint8_t)94u));

  return SRSASN_SUCCESS;
}
void prach_cfg_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("rootSequenceIndexHighSpeed-r14", root_seq_idx_high_speed_r14);
  j.write_int("zeroCorrelationZoneConfigHighSpeed-r14", zero_correlation_zone_cfg_high_speed_r14);
  j.write_int("prach-ConfigIndexHighSpeed-r14", prach_cfg_idx_high_speed_r14);
  j.write_int("prach-FreqOffsetHighSpeed-r14", prach_freq_offset_high_speed_r14);
  j.end_obj();
}

// PRACH-ConfigSIB ::= SEQUENCE
SRSASN_CODE prach_cfg_sib_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, root_seq_idx, (uint16_t)0u, (uint16_t)837u));
  HANDLE_CODE(prach_cfg_info.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_cfg_sib_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(root_seq_idx, bref, (uint16_t)0u, (uint16_t)837u));
  HANDLE_CODE(prach_cfg_info.unpack(bref));

  return SRSASN_SUCCESS;
}
void prach_cfg_sib_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("rootSequenceIndex", root_seq_idx);
  j.write_fieldname("prach-ConfigInfo");
  prach_cfg_info.to_json(j);
  j.end_obj();
}

// PRACH-ConfigSIB-v1310 ::= SEQUENCE
SRSASN_CODE prach_cfg_sib_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(mpdcch_start_sf_css_ra_r13_present, 1));
  HANDLE_CODE(bref.pack(prach_hop_offset_r13_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, rsrp_thress_prach_info_list_r13, 1, 3, integer_packer<uint8_t>(0, 97)));
  if (mpdcch_start_sf_css_ra_r13_present) {
    HANDLE_CODE(mpdcch_start_sf_css_ra_r13.pack(bref));
  }
  if (prach_hop_offset_r13_present) {
    HANDLE_CODE(pack_integer(bref, prach_hop_offset_r13, (uint8_t)0u, (uint8_t)94u));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, prach_params_list_ce_r13, 1, 4));

  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_cfg_sib_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(mpdcch_start_sf_css_ra_r13_present, 1));
  HANDLE_CODE(bref.unpack(prach_hop_offset_r13_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(rsrp_thress_prach_info_list_r13, bref, 1, 3, integer_packer<uint8_t>(0, 97)));
  if (mpdcch_start_sf_css_ra_r13_present) {
    HANDLE_CODE(mpdcch_start_sf_css_ra_r13.unpack(bref));
  }
  if (prach_hop_offset_r13_present) {
    HANDLE_CODE(unpack_integer(prach_hop_offset_r13, bref, (uint8_t)0u, (uint8_t)94u));
  }
  HANDLE_CODE(unpack_dyn_seq_of(prach_params_list_ce_r13, bref, 1, 4));

  return SRSASN_SUCCESS;
}
void prach_cfg_sib_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("rsrp-ThresholdsPrachInfoList-r13");
  for (const auto& e1 : rsrp_thress_prach_info_list_r13) {
    j.write_int(e1);
  }
  j.end_array();
  if (mpdcch_start_sf_css_ra_r13_present) {
    j.write_fieldname("mpdcch-startSF-CSS-RA-r13");
    mpdcch_start_sf_css_ra_r13.to_json(j);
  }
  if (prach_hop_offset_r13_present) {
    j.write_int("prach-HoppingOffset-r13", prach_hop_offset_r13);
  }
  j.start_array("prach-ParametersListCE-r13");
  for (const auto& e1 : prach_params_list_ce_r13) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

void prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::destroy_() {}
void prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::mpdcch_start_sf_css_ra_r13_c_(
    const prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::fdd_r13:
      c.init(other.c.get<fdd_r13_e_>());
      break;
    case types::tdd_r13:
      c.init(other.c.get<tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_");
  }
}
prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_& prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::
                                                      operator=(const prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::fdd_r13:
      c.set(other.c.get<fdd_r13_e_>());
      break;
    case types::tdd_r13:
      c.set(other.c.get<tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_");
  }

  return *this;
}
void prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::fdd_r13:
      j.write_str("fdd-r13", c.get<fdd_r13_e_>().to_string());
      break;
    case types::tdd_r13:
      j.write_str("tdd-r13", c.get<tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::fdd_r13:
      HANDLE_CODE(c.get<fdd_r13_e_>().pack(bref));
      break;
    case types::tdd_r13:
      HANDLE_CODE(c.get<tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::fdd_r13:
      HANDLE_CODE(c.get<fdd_r13_e_>().unpack(bref));
      break;
    case types::tdd_r13:
      HANDLE_CODE(c.get<tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "prach_cfg_sib_v1310_s::mpdcch_start_sf_css_ra_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// PRACH-ConfigSIB-v1530 ::= SEQUENCE
SRSASN_CODE prach_cfg_sib_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, edt_prach_params_list_ce_r15, 1, 4));

  return SRSASN_SUCCESS;
}
SRSASN_CODE prach_cfg_sib_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(edt_prach_params_list_ce_r15, bref, 1, 4));

  return SRSASN_SUCCESS;
}
void prach_cfg_sib_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("edt-PRACH-ParametersListCE-r15");
  for (const auto& e1 : edt_prach_params_list_ce_r15) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// PUCCH-ConfigCommon ::= SEQUENCE
SRSASN_CODE pucch_cfg_common_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(delta_pucch_shift.pack(bref));
  HANDLE_CODE(pack_integer(bref, nrb_cqi, (uint8_t)0u, (uint8_t)98u));
  HANDLE_CODE(pack_integer(bref, ncs_an, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(pack_integer(bref, n1_pucch_an, (uint16_t)0u, (uint16_t)2047u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pucch_cfg_common_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(delta_pucch_shift.unpack(bref));
  HANDLE_CODE(unpack_integer(nrb_cqi, bref, (uint8_t)0u, (uint8_t)98u));
  HANDLE_CODE(unpack_integer(ncs_an, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(unpack_integer(n1_pucch_an, bref, (uint16_t)0u, (uint16_t)2047u));

  return SRSASN_SUCCESS;
}
void pucch_cfg_common_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("deltaPUCCH-Shift", delta_pucch_shift.to_string());
  j.write_int("nRB-CQI", nrb_cqi);
  j.write_int("nCS-AN", ncs_an);
  j.write_int("n1PUCCH-AN", n1_pucch_an);
  j.end_obj();
}

// PUCCH-ConfigCommon-v1310 ::= SEQUENCE
SRSASN_CODE pucch_cfg_common_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(n1_pucch_an_info_list_r13_present, 1));
  HANDLE_CODE(bref.pack(pucch_num_repeat_ce_msg4_level0_r13_present, 1));
  HANDLE_CODE(bref.pack(pucch_num_repeat_ce_msg4_level1_r13_present, 1));
  HANDLE_CODE(bref.pack(pucch_num_repeat_ce_msg4_level2_r13_present, 1));
  HANDLE_CODE(bref.pack(pucch_num_repeat_ce_msg4_level3_r13_present, 1));

  if (n1_pucch_an_info_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, n1_pucch_an_info_list_r13, 1, 4, integer_packer<uint16_t>(0, 2047)));
  }
  if (pucch_num_repeat_ce_msg4_level0_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level0_r13.pack(bref));
  }
  if (pucch_num_repeat_ce_msg4_level1_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level1_r13.pack(bref));
  }
  if (pucch_num_repeat_ce_msg4_level2_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level2_r13.pack(bref));
  }
  if (pucch_num_repeat_ce_msg4_level3_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level3_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pucch_cfg_common_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(n1_pucch_an_info_list_r13_present, 1));
  HANDLE_CODE(bref.unpack(pucch_num_repeat_ce_msg4_level0_r13_present, 1));
  HANDLE_CODE(bref.unpack(pucch_num_repeat_ce_msg4_level1_r13_present, 1));
  HANDLE_CODE(bref.unpack(pucch_num_repeat_ce_msg4_level2_r13_present, 1));
  HANDLE_CODE(bref.unpack(pucch_num_repeat_ce_msg4_level3_r13_present, 1));

  if (n1_pucch_an_info_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(n1_pucch_an_info_list_r13, bref, 1, 4, integer_packer<uint16_t>(0, 2047)));
  }
  if (pucch_num_repeat_ce_msg4_level0_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level0_r13.unpack(bref));
  }
  if (pucch_num_repeat_ce_msg4_level1_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level1_r13.unpack(bref));
  }
  if (pucch_num_repeat_ce_msg4_level2_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level2_r13.unpack(bref));
  }
  if (pucch_num_repeat_ce_msg4_level3_r13_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level3_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pucch_cfg_common_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (n1_pucch_an_info_list_r13_present) {
    j.start_array("n1PUCCH-AN-InfoList-r13");
    for (const auto& e1 : n1_pucch_an_info_list_r13) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (pucch_num_repeat_ce_msg4_level0_r13_present) {
    j.write_str("pucch-NumRepetitionCE-Msg4-Level0-r13", pucch_num_repeat_ce_msg4_level0_r13.to_string());
  }
  if (pucch_num_repeat_ce_msg4_level1_r13_present) {
    j.write_str("pucch-NumRepetitionCE-Msg4-Level1-r13", pucch_num_repeat_ce_msg4_level1_r13.to_string());
  }
  if (pucch_num_repeat_ce_msg4_level2_r13_present) {
    j.write_str("pucch-NumRepetitionCE-Msg4-Level2-r13", pucch_num_repeat_ce_msg4_level2_r13.to_string());
  }
  if (pucch_num_repeat_ce_msg4_level3_r13_present) {
    j.write_str("pucch-NumRepetitionCE-Msg4-Level3-r13", pucch_num_repeat_ce_msg4_level3_r13.to_string());
  }
  j.end_obj();
}

// PUCCH-ConfigCommon-v1430 ::= SEQUENCE
SRSASN_CODE pucch_cfg_common_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(pucch_num_repeat_ce_msg4_level3_r14_present, 1));

  if (pucch_num_repeat_ce_msg4_level3_r14_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level3_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pucch_cfg_common_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(pucch_num_repeat_ce_msg4_level3_r14_present, 1));

  if (pucch_num_repeat_ce_msg4_level3_r14_present) {
    HANDLE_CODE(pucch_num_repeat_ce_msg4_level3_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pucch_cfg_common_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (pucch_num_repeat_ce_msg4_level3_r14_present) {
    j.write_str("pucch-NumRepetitionCE-Msg4-Level3-r14", pucch_num_repeat_ce_msg4_level3_r14.to_string());
  }
  j.end_obj();
}

// PUSCH-ConfigCommon ::= SEQUENCE
SRSASN_CODE pusch_cfg_common_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, pusch_cfg_basic.n_sb, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(pusch_cfg_basic.hop_mode.pack(bref));
  HANDLE_CODE(pack_integer(bref, pusch_cfg_basic.pusch_hop_offset, (uint8_t)0u, (uint8_t)98u));
  HANDLE_CODE(bref.pack(pusch_cfg_basic.enable64_qam, 1));
  HANDLE_CODE(ul_ref_sigs_pusch.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE pusch_cfg_common_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(pusch_cfg_basic.n_sb, bref, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(pusch_cfg_basic.hop_mode.unpack(bref));
  HANDLE_CODE(unpack_integer(pusch_cfg_basic.pusch_hop_offset, bref, (uint8_t)0u, (uint8_t)98u));
  HANDLE_CODE(bref.unpack(pusch_cfg_basic.enable64_qam, 1));
  HANDLE_CODE(ul_ref_sigs_pusch.unpack(bref));

  return SRSASN_SUCCESS;
}
void pusch_cfg_common_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("pusch-ConfigBasic");
  j.start_obj();
  j.write_int("n-SB", pusch_cfg_basic.n_sb);
  j.write_str("hoppingMode", pusch_cfg_basic.hop_mode.to_string());
  j.write_int("pusch-HoppingOffset", pusch_cfg_basic.pusch_hop_offset);
  j.write_bool("enable64QAM", pusch_cfg_basic.enable64_qam);
  j.end_obj();
  j.write_fieldname("ul-ReferenceSignalsPUSCH");
  ul_ref_sigs_pusch.to_json(j);
  j.end_obj();
}

// PUSCH-ConfigCommon-v1270 ::= SEQUENCE
SRSASN_CODE pusch_cfg_common_v1270_s::pack(bit_ref& bref) const
{
  return SRSASN_SUCCESS;
}
SRSASN_CODE pusch_cfg_common_v1270_s::unpack(cbit_ref& bref)
{
  return SRSASN_SUCCESS;
}
void pusch_cfg_common_v1270_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("enable64QAM-v1270", "true");
  j.end_obj();
}

// PUSCH-ConfigCommon-v1310 ::= SEQUENCE
SRSASN_CODE pusch_cfg_common_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(pusch_max_num_repeat_cemode_a_r13_present, 1));
  HANDLE_CODE(bref.pack(pusch_max_num_repeat_cemode_b_r13_present, 1));
  HANDLE_CODE(bref.pack(pusch_hop_offset_v1310_present, 1));

  if (pusch_max_num_repeat_cemode_a_r13_present) {
    HANDLE_CODE(pusch_max_num_repeat_cemode_a_r13.pack(bref));
  }
  if (pusch_max_num_repeat_cemode_b_r13_present) {
    HANDLE_CODE(pusch_max_num_repeat_cemode_b_r13.pack(bref));
  }
  if (pusch_hop_offset_v1310_present) {
    HANDLE_CODE(pack_integer(bref, pusch_hop_offset_v1310, (uint8_t)1u, (uint8_t)16u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pusch_cfg_common_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(pusch_max_num_repeat_cemode_a_r13_present, 1));
  HANDLE_CODE(bref.unpack(pusch_max_num_repeat_cemode_b_r13_present, 1));
  HANDLE_CODE(bref.unpack(pusch_hop_offset_v1310_present, 1));

  if (pusch_max_num_repeat_cemode_a_r13_present) {
    HANDLE_CODE(pusch_max_num_repeat_cemode_a_r13.unpack(bref));
  }
  if (pusch_max_num_repeat_cemode_b_r13_present) {
    HANDLE_CODE(pusch_max_num_repeat_cemode_b_r13.unpack(bref));
  }
  if (pusch_hop_offset_v1310_present) {
    HANDLE_CODE(unpack_integer(pusch_hop_offset_v1310, bref, (uint8_t)1u, (uint8_t)16u));
  }

  return SRSASN_SUCCESS;
}
void pusch_cfg_common_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (pusch_max_num_repeat_cemode_a_r13_present) {
    j.write_str("pusch-maxNumRepetitionCEmodeA-r13", pusch_max_num_repeat_cemode_a_r13.to_string());
  }
  if (pusch_max_num_repeat_cemode_b_r13_present) {
    j.write_str("pusch-maxNumRepetitionCEmodeB-r13", pusch_max_num_repeat_cemode_b_r13.to_string());
  }
  if (pusch_hop_offset_v1310_present) {
    j.write_int("pusch-HoppingOffset-v1310", pusch_hop_offset_v1310);
  }
  j.end_obj();
}

// RACH-ConfigCommon ::= SEQUENCE
SRSASN_CODE rach_cfg_common_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(preamb_info.preambs_group_a_cfg_present, 1));
  HANDLE_CODE(preamb_info.nof_ra_preambs.pack(bref));
  if (preamb_info.preambs_group_a_cfg_present) {
    bref.pack(preamb_info.preambs_group_a_cfg.ext, 1);
    HANDLE_CODE(preamb_info.preambs_group_a_cfg.size_of_ra_preambs_group_a.pack(bref));
    HANDLE_CODE(preamb_info.preambs_group_a_cfg.msg_size_group_a.pack(bref));
    HANDLE_CODE(preamb_info.preambs_group_a_cfg.msg_pwr_offset_group_b.pack(bref));
  }
  HANDLE_CODE(pwr_ramp_params.pack(bref));
  HANDLE_CODE(ra_supervision_info.preamb_trans_max.pack(bref));
  HANDLE_CODE(ra_supervision_info.ra_resp_win_size.pack(bref));
  HANDLE_CODE(ra_supervision_info.mac_contention_resolution_timer.pack(bref));
  HANDLE_CODE(pack_integer(bref, max_harq_msg3_tx, (uint8_t)1u, (uint8_t)8u));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= preamb_trans_max_ce_r13_present;
    group_flags[0] |= rach_ce_level_info_list_r13.is_present();
    group_flags[1] |= edt_small_tbs_subset_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(preamb_trans_max_ce_r13_present, 1));
      HANDLE_CODE(bref.pack(rach_ce_level_info_list_r13.is_present(), 1));
      if (preamb_trans_max_ce_r13_present) {
        HANDLE_CODE(preamb_trans_max_ce_r13.pack(bref));
      }
      if (rach_ce_level_info_list_r13.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *rach_ce_level_info_list_r13, 1, 4));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(edt_small_tbs_subset_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rach_cfg_common_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(preamb_info.preambs_group_a_cfg_present, 1));
  HANDLE_CODE(preamb_info.nof_ra_preambs.unpack(bref));
  if (preamb_info.preambs_group_a_cfg_present) {
    bref.unpack(preamb_info.preambs_group_a_cfg.ext, 1);
    HANDLE_CODE(preamb_info.preambs_group_a_cfg.size_of_ra_preambs_group_a.unpack(bref));
    HANDLE_CODE(preamb_info.preambs_group_a_cfg.msg_size_group_a.unpack(bref));
    HANDLE_CODE(preamb_info.preambs_group_a_cfg.msg_pwr_offset_group_b.unpack(bref));
  }
  HANDLE_CODE(pwr_ramp_params.unpack(bref));
  HANDLE_CODE(ra_supervision_info.preamb_trans_max.unpack(bref));
  HANDLE_CODE(ra_supervision_info.ra_resp_win_size.unpack(bref));
  HANDLE_CODE(ra_supervision_info.mac_contention_resolution_timer.unpack(bref));
  HANDLE_CODE(unpack_integer(max_harq_msg3_tx, bref, (uint8_t)1u, (uint8_t)8u));

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(preamb_trans_max_ce_r13_present, 1));
      bool rach_ce_level_info_list_r13_present;
      HANDLE_CODE(bref.unpack(rach_ce_level_info_list_r13_present, 1));
      rach_ce_level_info_list_r13.set_present(rach_ce_level_info_list_r13_present);
      if (preamb_trans_max_ce_r13_present) {
        HANDLE_CODE(preamb_trans_max_ce_r13.unpack(bref));
      }
      if (rach_ce_level_info_list_r13.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*rach_ce_level_info_list_r13, bref, 1, 4));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(edt_small_tbs_subset_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
void rach_cfg_common_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("preambleInfo");
  j.start_obj();
  j.write_str("numberOfRA-Preambles", preamb_info.nof_ra_preambs.to_string());
  if (preamb_info.preambs_group_a_cfg_present) {
    j.write_fieldname("preamblesGroupAConfig");
    j.start_obj();
    j.write_str("sizeOfRA-PreamblesGroupA", preamb_info.preambs_group_a_cfg.size_of_ra_preambs_group_a.to_string());
    j.write_str("messageSizeGroupA", preamb_info.preambs_group_a_cfg.msg_size_group_a.to_string());
    j.write_str("messagePowerOffsetGroupB", preamb_info.preambs_group_a_cfg.msg_pwr_offset_group_b.to_string());
    j.end_obj();
  }
  j.end_obj();
  j.write_fieldname("powerRampingParameters");
  pwr_ramp_params.to_json(j);
  j.write_fieldname("ra-SupervisionInfo");
  j.start_obj();
  j.write_str("preambleTransMax", ra_supervision_info.preamb_trans_max.to_string());
  j.write_str("ra-ResponseWindowSize", ra_supervision_info.ra_resp_win_size.to_string());
  j.write_str("mac-ContentionResolutionTimer", ra_supervision_info.mac_contention_resolution_timer.to_string());
  j.end_obj();
  j.write_int("maxHARQ-Msg3Tx", max_harq_msg3_tx);
  if (ext) {
    if (preamb_trans_max_ce_r13_present) {
      j.write_str("preambleTransMax-CE-r13", preamb_trans_max_ce_r13.to_string());
    }
    if (rach_ce_level_info_list_r13.is_present()) {
      j.start_array("rach-CE-LevelInfoList-r13");
      for (const auto& e1 : *rach_ce_level_info_list_r13) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (edt_small_tbs_subset_r15_present) {
      j.write_str("edt-SmallTBS-Subset-r15", "true");
    }
  }
  j.end_obj();
}

// RACH-ConfigCommon-v1250 ::= SEQUENCE
SRSASN_CODE rach_cfg_common_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(tx_fail_params_r12.conn_est_fail_offset_r12_present, 1));
  HANDLE_CODE(tx_fail_params_r12.conn_est_fail_count_r12.pack(bref));
  HANDLE_CODE(tx_fail_params_r12.conn_est_fail_offset_validity_r12.pack(bref));
  if (tx_fail_params_r12.conn_est_fail_offset_r12_present) {
    HANDLE_CODE(pack_integer(bref, tx_fail_params_r12.conn_est_fail_offset_r12, (uint8_t)0u, (uint8_t)15u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE rach_cfg_common_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(tx_fail_params_r12.conn_est_fail_offset_r12_present, 1));
  HANDLE_CODE(tx_fail_params_r12.conn_est_fail_count_r12.unpack(bref));
  HANDLE_CODE(tx_fail_params_r12.conn_est_fail_offset_validity_r12.unpack(bref));
  if (tx_fail_params_r12.conn_est_fail_offset_r12_present) {
    HANDLE_CODE(unpack_integer(tx_fail_params_r12.conn_est_fail_offset_r12, bref, (uint8_t)0u, (uint8_t)15u));
  }

  return SRSASN_SUCCESS;
}
void rach_cfg_common_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("txFailParams-r12");
  j.start_obj();
  j.write_str("connEstFailCount-r12", tx_fail_params_r12.conn_est_fail_count_r12.to_string());
  j.write_str("connEstFailOffsetValidity-r12", tx_fail_params_r12.conn_est_fail_offset_validity_r12.to_string());
  if (tx_fail_params_r12.conn_est_fail_offset_r12_present) {
    j.write_int("connEstFailOffset-r12", tx_fail_params_r12.conn_est_fail_offset_r12);
  }
  j.end_obj();
  j.end_obj();
}

// RSS-Config-r15 ::= SEQUENCE
SRSASN_CODE rss_cfg_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(dur_r15.pack(bref));
  HANDLE_CODE(pack_integer(bref, freq_location_r15, (uint8_t)0u, (uint8_t)98u));
  HANDLE_CODE(periodicity_r15.pack(bref));
  HANDLE_CODE(pwr_boost_r15.pack(bref));
  HANDLE_CODE(pack_integer(bref, time_offset_r15, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE rss_cfg_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(dur_r15.unpack(bref));
  HANDLE_CODE(unpack_integer(freq_location_r15, bref, (uint8_t)0u, (uint8_t)98u));
  HANDLE_CODE(periodicity_r15.unpack(bref));
  HANDLE_CODE(pwr_boost_r15.unpack(bref));
  HANDLE_CODE(unpack_integer(time_offset_r15, bref, (uint8_t)0u, (uint8_t)31u));

  return SRSASN_SUCCESS;
}
void rss_cfg_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("duration-r15", dur_r15.to_string());
  j.write_int("freqLocation-r15", freq_location_r15);
  j.write_str("periodicity-r15", periodicity_r15.to_string());
  j.write_str("powerBoost-r15", pwr_boost_r15.to_string());
  j.write_int("timeOffset-r15", time_offset_r15);
  j.end_obj();
}

// ReferenceTime-r15 ::= SEQUENCE
SRSASN_CODE ref_time_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, ref_days_r15, (uint32_t)0u, (uint32_t)72999u));
  HANDLE_CODE(pack_integer(bref, ref_seconds_r15, (uint32_t)0u, (uint32_t)86399u));
  HANDLE_CODE(pack_integer(bref, ref_milli_seconds_r15, (uint16_t)0u, (uint16_t)999u));
  HANDLE_CODE(pack_integer(bref, ref_quarter_micro_seconds_r15, (uint16_t)0u, (uint16_t)3999u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ref_time_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(ref_days_r15, bref, (uint32_t)0u, (uint32_t)72999u));
  HANDLE_CODE(unpack_integer(ref_seconds_r15, bref, (uint32_t)0u, (uint32_t)86399u));
  HANDLE_CODE(unpack_integer(ref_milli_seconds_r15, bref, (uint16_t)0u, (uint16_t)999u));
  HANDLE_CODE(unpack_integer(ref_quarter_micro_seconds_r15, bref, (uint16_t)0u, (uint16_t)3999u));

  return SRSASN_SUCCESS;
}
void ref_time_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("refDays-r15", ref_days_r15);
  j.write_int("refSeconds-r15", ref_seconds_r15);
  j.write_int("refMilliSeconds-r15", ref_milli_seconds_r15);
  j.write_int("refQuarterMicroSeconds-r15", ref_quarter_micro_seconds_r15);
  j.end_obj();
}

// ReselectionInfoRelay-r13 ::= SEQUENCE
SRSASN_CODE resel_info_relay_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(min_hyst_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, q_rx_lev_min_r13, (int8_t)-70, (int8_t)-22));
  HANDLE_CODE(filt_coef_r13.pack(bref));
  if (min_hyst_r13_present) {
    HANDLE_CODE(min_hyst_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE resel_info_relay_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(min_hyst_r13_present, 1));

  HANDLE_CODE(unpack_integer(q_rx_lev_min_r13, bref, (int8_t)-70, (int8_t)-22));
  HANDLE_CODE(filt_coef_r13.unpack(bref));
  if (min_hyst_r13_present) {
    HANDLE_CODE(min_hyst_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void resel_info_relay_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-RxLevMin-r13", q_rx_lev_min_r13);
  j.write_str("filterCoefficient-r13", filt_coef_r13.to_string());
  if (min_hyst_r13_present) {
    j.write_str("minHyst-r13", min_hyst_r13.to_string());
  }
  j.end_obj();
}

// SIB8-PerPLMN-r11 ::= SEQUENCE
SRSASN_CODE sib8_per_plmn_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, plmn_id_r11, (uint8_t)1u, (uint8_t)6u));
  HANDLE_CODE(params_cdma2000_r11.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib8_per_plmn_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(plmn_id_r11, bref, (uint8_t)1u, (uint8_t)6u));
  HANDLE_CODE(params_cdma2000_r11.unpack(bref));

  return SRSASN_SUCCESS;
}
void sib8_per_plmn_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("plmn-Identity-r11", plmn_id_r11);
  j.write_fieldname("parametersCDMA2000-r11");
  params_cdma2000_r11.to_json(j);
  j.end_obj();
}

void sib8_per_plmn_r11_s::params_cdma2000_r11_c_::set(types::options e)
{
  type_ = e;
}
void sib8_per_plmn_r11_s::params_cdma2000_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::explicit_value:
      j.write_fieldname("explicitValue");
      c.to_json(j);
      break;
    case types::default_value:
      break;
    default:
      log_invalid_choice_id(type_, "sib8_per_plmn_r11_s::params_cdma2000_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE sib8_per_plmn_r11_s::params_cdma2000_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::explicit_value:
      HANDLE_CODE(c.pack(bref));
      break;
    case types::default_value:
      break;
    default:
      log_invalid_choice_id(type_, "sib8_per_plmn_r11_s::params_cdma2000_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib8_per_plmn_r11_s::params_cdma2000_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::explicit_value:
      HANDLE_CODE(c.unpack(bref));
      break;
    case types::default_value:
      break;
    default:
      log_invalid_choice_id(type_, "sib8_per_plmn_r11_s::params_cdma2000_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SL-CBR-CommonTxConfigList-r14 ::= SEQUENCE
SRSASN_CODE sl_cbr_common_tx_cfg_list_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref,
                              cbr_range_common_cfg_list_r14,
                              1,
                              4,
                              SeqOfPacker<integer_packer<uint8_t> >(1, 16, integer_packer<uint8_t>(0, 100))));
  HANDLE_CODE(pack_dyn_seq_of(bref, sl_cbr_pssch_tx_cfg_list_r14, 1, 64));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_cbr_common_tx_cfg_list_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(cbr_range_common_cfg_list_r14,
                                bref,
                                1,
                                4,
                                SeqOfPacker<integer_packer<uint8_t> >(1, 16, integer_packer<uint8_t>(0, 100))));
  HANDLE_CODE(unpack_dyn_seq_of(sl_cbr_pssch_tx_cfg_list_r14, bref, 1, 64));

  return SRSASN_SUCCESS;
}
void sl_cbr_common_tx_cfg_list_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("cbr-RangeCommonConfigList-r14");
  for (const auto& e1 : cbr_range_common_cfg_list_r14) {
    j.start_array();
    for (const auto& e2 : e1) {
      j.write_int(e2);
    }
    j.end_array();
  }
  j.end_array();
  j.start_array("sl-CBR-PSSCH-TxConfigList-r14");
  for (const auto& e1 : sl_cbr_pssch_tx_cfg_list_r14) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// SL-CarrierFreqInfo-r12 ::= SEQUENCE
SRSASN_CODE sl_carrier_freq_info_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(plmn_id_list_r12_present, 1));

  HANDLE_CODE(pack_integer(bref, carrier_freq_r12, (uint32_t)0u, (uint32_t)262143u));
  if (plmn_id_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, plmn_id_list_r12, 1, 6));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_carrier_freq_info_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(plmn_id_list_r12_present, 1));

  HANDLE_CODE(unpack_integer(carrier_freq_r12, bref, (uint32_t)0u, (uint32_t)262143u));
  if (plmn_id_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(plmn_id_list_r12, bref, 1, 6));
  }

  return SRSASN_SUCCESS;
}
void sl_carrier_freq_info_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("carrierFreq-r12", carrier_freq_r12);
  if (plmn_id_list_r12_present) {
    j.start_array("plmn-IdentityList-r12");
    for (const auto& e1 : plmn_id_list_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// SL-CarrierFreqInfo-v1310 ::= SEQUENCE
SRSASN_CODE sl_carrier_freq_info_v1310_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(disc_res_non_ps_r13_present, 1));
  HANDLE_CODE(bref.pack(disc_res_ps_r13_present, 1));
  HANDLE_CODE(bref.pack(disc_cfg_other_r13_present, 1));

  if (disc_res_non_ps_r13_present) {
    HANDLE_CODE(disc_res_non_ps_r13.pack(bref));
  }
  if (disc_res_ps_r13_present) {
    HANDLE_CODE(disc_res_ps_r13.pack(bref));
  }
  if (disc_cfg_other_r13_present) {
    HANDLE_CODE(disc_cfg_other_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_carrier_freq_info_v1310_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(disc_res_non_ps_r13_present, 1));
  HANDLE_CODE(bref.unpack(disc_res_ps_r13_present, 1));
  HANDLE_CODE(bref.unpack(disc_cfg_other_r13_present, 1));

  if (disc_res_non_ps_r13_present) {
    HANDLE_CODE(disc_res_non_ps_r13.unpack(bref));
  }
  if (disc_res_ps_r13_present) {
    HANDLE_CODE(disc_res_ps_r13.unpack(bref));
  }
  if (disc_cfg_other_r13_present) {
    HANDLE_CODE(disc_cfg_other_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_carrier_freq_info_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (disc_res_non_ps_r13_present) {
    j.write_fieldname("discResourcesNonPS-r13");
    disc_res_non_ps_r13.to_json(j);
  }
  if (disc_res_ps_r13_present) {
    j.write_fieldname("discResourcesPS-r13");
    disc_res_ps_r13.to_json(j);
  }
  if (disc_cfg_other_r13_present) {
    j.write_fieldname("discConfigOther-r13");
    disc_cfg_other_r13.to_json(j);
  }
  j.end_obj();
}

// SL-CommResourcePool-r12 ::= SEQUENCE
SRSASN_CODE sl_comm_res_pool_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ue_sel_res_cfg_r12_present, 1));
  HANDLE_CODE(bref.pack(rx_params_ncell_r12_present, 1));
  HANDLE_CODE(bref.pack(tx_params_r12_present, 1));

  HANDLE_CODE(sc_cp_len_r12.pack(bref));
  HANDLE_CODE(sc_period_r12.pack(bref));
  HANDLE_CODE(sc_tf_res_cfg_r12.pack(bref));
  HANDLE_CODE(data_cp_len_r12.pack(bref));
  HANDLE_CODE(data_hop_cfg_r12.pack(bref));
  if (ue_sel_res_cfg_r12_present) {
    HANDLE_CODE(bref.pack(ue_sel_res_cfg_r12.trpt_subset_r12_present, 1));
    HANDLE_CODE(ue_sel_res_cfg_r12.data_tf_res_cfg_r12.pack(bref));
    if (ue_sel_res_cfg_r12.trpt_subset_r12_present) {
      HANDLE_CODE(ue_sel_res_cfg_r12.trpt_subset_r12.pack(bref));
    }
  }
  if (rx_params_ncell_r12_present) {
    HANDLE_CODE(bref.pack(rx_params_ncell_r12.tdd_cfg_r12_present, 1));
    if (rx_params_ncell_r12.tdd_cfg_r12_present) {
      HANDLE_CODE(rx_params_ncell_r12.tdd_cfg_r12.pack(bref));
    }
    HANDLE_CODE(pack_integer(bref, rx_params_ncell_r12.sync_cfg_idx_r12, (uint8_t)0u, (uint8_t)15u));
  }
  if (tx_params_r12_present) {
    HANDLE_CODE(tx_params_r12.sc_tx_params_r12.pack(bref));
    HANDLE_CODE(tx_params_r12.data_tx_params_r12.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= prio_list_r13.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(prio_list_r13.is_present(), 1));
      if (prio_list_r13.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *prio_list_r13, 1, 8, integer_packer<uint8_t>(1, 8)));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_comm_res_pool_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ue_sel_res_cfg_r12_present, 1));
  HANDLE_CODE(bref.unpack(rx_params_ncell_r12_present, 1));
  HANDLE_CODE(bref.unpack(tx_params_r12_present, 1));

  HANDLE_CODE(sc_cp_len_r12.unpack(bref));
  HANDLE_CODE(sc_period_r12.unpack(bref));
  HANDLE_CODE(sc_tf_res_cfg_r12.unpack(bref));
  HANDLE_CODE(data_cp_len_r12.unpack(bref));
  HANDLE_CODE(data_hop_cfg_r12.unpack(bref));
  if (ue_sel_res_cfg_r12_present) {
    HANDLE_CODE(bref.unpack(ue_sel_res_cfg_r12.trpt_subset_r12_present, 1));
    HANDLE_CODE(ue_sel_res_cfg_r12.data_tf_res_cfg_r12.unpack(bref));
    if (ue_sel_res_cfg_r12.trpt_subset_r12_present) {
      HANDLE_CODE(ue_sel_res_cfg_r12.trpt_subset_r12.unpack(bref));
    }
  }
  if (rx_params_ncell_r12_present) {
    HANDLE_CODE(bref.unpack(rx_params_ncell_r12.tdd_cfg_r12_present, 1));
    if (rx_params_ncell_r12.tdd_cfg_r12_present) {
      HANDLE_CODE(rx_params_ncell_r12.tdd_cfg_r12.unpack(bref));
    }
    HANDLE_CODE(unpack_integer(rx_params_ncell_r12.sync_cfg_idx_r12, bref, (uint8_t)0u, (uint8_t)15u));
  }
  if (tx_params_r12_present) {
    HANDLE_CODE(tx_params_r12.sc_tx_params_r12.unpack(bref));
    HANDLE_CODE(tx_params_r12.data_tx_params_r12.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool prio_list_r13_present;
      HANDLE_CODE(bref.unpack(prio_list_r13_present, 1));
      prio_list_r13.set_present(prio_list_r13_present);
      if (prio_list_r13.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*prio_list_r13, bref, 1, 8, integer_packer<uint8_t>(1, 8)));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sl_comm_res_pool_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sc-CP-Len-r12", sc_cp_len_r12.to_string());
  j.write_str("sc-Period-r12", sc_period_r12.to_string());
  j.write_fieldname("sc-TF-ResourceConfig-r12");
  sc_tf_res_cfg_r12.to_json(j);
  j.write_str("data-CP-Len-r12", data_cp_len_r12.to_string());
  j.write_fieldname("dataHoppingConfig-r12");
  data_hop_cfg_r12.to_json(j);
  if (ue_sel_res_cfg_r12_present) {
    j.write_fieldname("ue-SelectedResourceConfig-r12");
    j.start_obj();
    j.write_fieldname("data-TF-ResourceConfig-r12");
    ue_sel_res_cfg_r12.data_tf_res_cfg_r12.to_json(j);
    if (ue_sel_res_cfg_r12.trpt_subset_r12_present) {
      j.write_str("trpt-Subset-r12", ue_sel_res_cfg_r12.trpt_subset_r12.to_string());
    }
    j.end_obj();
  }
  if (rx_params_ncell_r12_present) {
    j.write_fieldname("rxParametersNCell-r12");
    j.start_obj();
    if (rx_params_ncell_r12.tdd_cfg_r12_present) {
      j.write_fieldname("tdd-Config-r12");
      rx_params_ncell_r12.tdd_cfg_r12.to_json(j);
    }
    j.write_int("syncConfigIndex-r12", rx_params_ncell_r12.sync_cfg_idx_r12);
    j.end_obj();
  }
  if (tx_params_r12_present) {
    j.write_fieldname("txParameters-r12");
    j.start_obj();
    j.write_fieldname("sc-TxParameters-r12");
    tx_params_r12.sc_tx_params_r12.to_json(j);
    j.write_fieldname("dataTxParameters-r12");
    tx_params_r12.data_tx_params_r12.to_json(j);
    j.end_obj();
  }
  if (ext) {
    if (prio_list_r13.is_present()) {
      j.start_array("priorityList-r13");
      for (const auto& e1 : *prio_list_r13) {
        j.write_int(e1);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// SL-PPPP-TxConfigIndex-r15 ::= SEQUENCE
SRSASN_CODE sl_pppp_tx_cfg_idx_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, prio_thres_r15, (uint8_t)1u, (uint8_t)8u));
  HANDLE_CODE(pack_integer(bref, default_tx_cfg_idx_r15, (uint8_t)0u, (uint8_t)15u));
  HANDLE_CODE(pack_integer(bref, cbr_cfg_idx_r15, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(pack_dyn_seq_of(bref, tx_cfg_idx_list_r15, 1, 16, integer_packer<uint8_t>(0, 63)));
  HANDLE_CODE(pack_dyn_seq_of(bref, mcs_pssch_range_list_r15, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_pppp_tx_cfg_idx_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(prio_thres_r15, bref, (uint8_t)1u, (uint8_t)8u));
  HANDLE_CODE(unpack_integer(default_tx_cfg_idx_r15, bref, (uint8_t)0u, (uint8_t)15u));
  HANDLE_CODE(unpack_integer(cbr_cfg_idx_r15, bref, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(unpack_dyn_seq_of(tx_cfg_idx_list_r15, bref, 1, 16, integer_packer<uint8_t>(0, 63)));
  HANDLE_CODE(unpack_dyn_seq_of(mcs_pssch_range_list_r15, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void sl_pppp_tx_cfg_idx_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("priorityThreshold-r15", prio_thres_r15);
  j.write_int("defaultTxConfigIndex-r15", default_tx_cfg_idx_r15);
  j.write_int("cbr-ConfigIndex-r15", cbr_cfg_idx_r15);
  j.start_array("tx-ConfigIndexList-r15");
  for (const auto& e1 : tx_cfg_idx_list_r15) {
    j.write_int(e1);
  }
  j.end_array();
  j.start_array("mcs-PSSCH-RangeList-r15");
  for (const auto& e1 : mcs_pssch_range_list_r15) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// SoundingRS-UL-ConfigCommon ::= CHOICE
void srs_ul_cfg_common_c::set(types::options e)
{
  type_ = e;
}
void srs_ul_cfg_common_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("srs-BandwidthConfig", c.srs_bw_cfg.to_string());
      j.write_str("srs-SubframeConfig", c.srs_sf_cfg.to_string());
      j.write_bool("ackNackSRS-SimultaneousTransmission", c.ack_nack_srs_simul_tx);
      if (c.srs_max_up_pts_present) {
        j.write_str("srs-MaxUpPts", "true");
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "srs_ul_cfg_common_c");
  }
  j.end_obj();
}
SRSASN_CODE srs_ul_cfg_common_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.srs_max_up_pts_present, 1));
      HANDLE_CODE(c.srs_bw_cfg.pack(bref));
      HANDLE_CODE(c.srs_sf_cfg.pack(bref));
      HANDLE_CODE(bref.pack(c.ack_nack_srs_simul_tx, 1));
      break;
    default:
      log_invalid_choice_id(type_, "srs_ul_cfg_common_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE srs_ul_cfg_common_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.srs_max_up_pts_present, 1));
      HANDLE_CODE(c.srs_bw_cfg.unpack(bref));
      HANDLE_CODE(c.srs_sf_cfg.unpack(bref));
      HANDLE_CODE(bref.unpack(c.ack_nack_srs_simul_tx, 1));
      break;
    default:
      log_invalid_choice_id(type_, "srs_ul_cfg_common_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType1-v10j0-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v10j0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_info_r10_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10j0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_band_info_r10_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, freq_band_info_r10, 1, 8));
  }
  if (multi_band_info_list_v10j0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10j0, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v10j0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_info_r10_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10j0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_band_info_r10_present) {
    HANDLE_CODE(unpack_dyn_seq_of(freq_band_info_r10, bref, 1, 8));
  }
  if (multi_band_info_list_v10j0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10j0, bref, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v10j0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_info_r10_present) {
    j.start_array("freqBandInfo-r10");
    for (const auto& e1 : freq_band_info_r10) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (multi_band_info_list_v10j0_present) {
    j.start_array("multiBandInfoList-v10j0");
    for (const auto& e1 : multi_band_info_list_v10j0) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType1-v1250-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1250_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cell_sel_info_v1250_present, 1));
  HANDLE_CODE(bref.pack(freq_band_ind_prio_r12_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  HANDLE_CODE(bref.pack(cell_access_related_info_v1250.category0_allowed_r12_present, 1));
  if (cell_sel_info_v1250_present) {
    HANDLE_CODE(cell_sel_info_v1250.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1250_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cell_sel_info_v1250_present, 1));
  HANDLE_CODE(bref.unpack(freq_band_ind_prio_r12_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  HANDLE_CODE(bref.unpack(cell_access_related_info_v1250.category0_allowed_r12_present, 1));
  if (cell_sel_info_v1250_present) {
    HANDLE_CODE(cell_sel_info_v1250.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1250_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellAccessRelatedInfo-v1250");
  j.start_obj();
  if (cell_access_related_info_v1250.category0_allowed_r12_present) {
    j.write_str("category0Allowed-r12", "true");
  }
  j.end_obj();
  if (cell_sel_info_v1250_present) {
    j.write_fieldname("cellSelectionInfo-v1250");
    cell_sel_info_v1250.to_json(j);
  }
  if (freq_band_ind_prio_r12_present) {
    j.write_str("freqBandIndicatorPriority-r12", "true");
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType2-v9e0-IEs ::= SEQUENCE
SRSASN_CODE sib_type2_v9e0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ul_carrier_freq_v9e0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (ul_carrier_freq_v9e0_present) {
    HANDLE_CODE(pack_integer(bref, ul_carrier_freq_v9e0, (uint32_t)65536u, (uint32_t)262143u));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type2_v9e0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ul_carrier_freq_v9e0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (ul_carrier_freq_v9e0_present) {
    HANDLE_CODE(unpack_integer(ul_carrier_freq_v9e0, bref, (uint32_t)65536u, (uint32_t)262143u));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type2_v9e0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ul_carrier_freq_v9e0_present) {
    j.write_int("ul-CarrierFreq-v9e0", ul_carrier_freq_v9e0);
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType3-v10l0-IEs ::= SEQUENCE
SRSASN_CODE sib_type3_v10l0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_info_v10l0_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10l0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_band_info_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, freq_band_info_v10l0, 1, 8));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10l0, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type3_v10l0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_info_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10l0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_band_info_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(freq_band_info_v10l0, bref, 1, 8));
  }
  if (multi_band_info_list_v10l0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10l0, bref, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }

  return SRSASN_SUCCESS;
}
void sib_type3_v10l0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_info_v10l0_present) {
    j.start_array("freqBandInfo-v10l0");
    for (const auto& e1 : freq_band_info_v10l0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (multi_band_info_list_v10l0_present) {
    j.start_array("multiBandInfoList-v10l0");
    for (const auto& e1 : multi_band_info_list_v10l0) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// SystemInformationBlockType5-v9e0-IEs ::= SEQUENCE
SRSASN_CODE sib_type5_v9e0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v9e0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v9e0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_carrier_freq_list_v9e0, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type5_v9e0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v9e0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v9e0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_carrier_freq_list_v9e0, bref, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type5_v9e0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (inter_freq_carrier_freq_list_v9e0_present) {
    j.start_array("interFreqCarrierFreqList-v9e0");
    for (const auto& e1 : inter_freq_carrier_freq_list_v9e0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// UAC-BarringInfoSet-r15 ::= SEQUENCE
SRSASN_CODE uac_barr_info_set_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(uac_barr_factor_r15.pack(bref));
  HANDLE_CODE(uac_barr_time_r15.pack(bref));
  HANDLE_CODE(uac_barr_for_access_id_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE uac_barr_info_set_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(uac_barr_factor_r15.unpack(bref));
  HANDLE_CODE(uac_barr_time_r15.unpack(bref));
  HANDLE_CODE(uac_barr_for_access_id_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void uac_barr_info_set_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("uac-BarringFactor-r15", uac_barr_factor_r15.to_string());
  j.write_str("uac-BarringTime-r15", uac_barr_time_r15.to_string());
  j.write_str("uac-BarringForAccessIdentity-r15", uac_barr_for_access_id_r15.to_string());
  j.end_obj();
}

// UAC-BarringPerPLMN-r15 ::= SEQUENCE
SRSASN_CODE uac_barr_per_plmn_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(uac_ac_barr_list_type_r15_present, 1));

  HANDLE_CODE(pack_integer(bref, plmn_id_idx_r15, (uint8_t)1u, (uint8_t)6u));
  if (uac_ac_barr_list_type_r15_present) {
    HANDLE_CODE(uac_ac_barr_list_type_r15.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE uac_barr_per_plmn_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(uac_ac_barr_list_type_r15_present, 1));

  HANDLE_CODE(unpack_integer(plmn_id_idx_r15, bref, (uint8_t)1u, (uint8_t)6u));
  if (uac_ac_barr_list_type_r15_present) {
    HANDLE_CODE(uac_ac_barr_list_type_r15.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void uac_barr_per_plmn_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("plmn-IdentityIndex-r15", plmn_id_idx_r15);
  if (uac_ac_barr_list_type_r15_present) {
    j.write_fieldname("uac-AC-BarringListType-r15");
    uac_ac_barr_list_type_r15.to_json(j);
  }
  j.end_obj();
}

void uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::destroy_()
{
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      c.destroy<uac_implicit_ac_barr_list_r15_l_>();
      break;
    case types::uac_explicit_ac_barr_list_r15:
      c.destroy<uac_barr_per_cat_list_r15_l>();
      break;
    default:
      break;
  }
}
void uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      c.init<uac_implicit_ac_barr_list_r15_l_>();
      break;
    case types::uac_explicit_ac_barr_list_r15:
      c.init<uac_barr_per_cat_list_r15_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_");
  }
}
uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::uac_ac_barr_list_type_r15_c_(
    const uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      c.init(other.c.get<uac_implicit_ac_barr_list_r15_l_>());
      break;
    case types::uac_explicit_ac_barr_list_r15:
      c.init(other.c.get<uac_barr_per_cat_list_r15_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_");
  }
}
uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_& uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::
                                                       operator=(const uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      c.set(other.c.get<uac_implicit_ac_barr_list_r15_l_>());
      break;
    case types::uac_explicit_ac_barr_list_r15:
      c.set(other.c.get<uac_barr_per_cat_list_r15_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_");
  }

  return *this;
}
void uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      j.start_array("uac-ImplicitAC-BarringList-r15");
      for (const auto& e1 : c.get<uac_implicit_ac_barr_list_r15_l_>()) {
        j.write_int(e1);
      }
      j.end_array();
      break;
    case types::uac_explicit_ac_barr_list_r15:
      j.start_array("uac-ExplicitAC-BarringList-r15");
      for (const auto& e1 : c.get<uac_barr_per_cat_list_r15_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      HANDLE_CODE(pack_fixed_seq_of(bref,
                                    &(c.get<uac_implicit_ac_barr_list_r15_l_>())[0],
                                    c.get<uac_implicit_ac_barr_list_r15_l_>().size(),
                                    integer_packer<uint8_t>(1, 8)));
      break;
    case types::uac_explicit_ac_barr_list_r15:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<uac_barr_per_cat_list_r15_l>(), 1, 63));
      break;
    default:
      log_invalid_choice_id(type_, "uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::uac_implicit_ac_barr_list_r15:
      HANDLE_CODE(unpack_fixed_seq_of(&(c.get<uac_implicit_ac_barr_list_r15_l_>())[0],
                                      bref,
                                      c.get<uac_implicit_ac_barr_list_r15_l_>().size(),
                                      integer_packer<uint8_t>(1, 8)));
      break;
    case types::uac_explicit_ac_barr_list_r15:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<uac_barr_per_cat_list_r15_l>(), bref, 1, 63));
      break;
    default:
      log_invalid_choice_id(type_, "uac_barr_per_plmn_r15_s::uac_ac_barr_list_type_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// UDT-RestrictingPerPLMN-r13 ::= SEQUENCE
SRSASN_CODE udt_restricting_per_plmn_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(udt_restricting_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, plmn_id_idx_r13, (uint8_t)1u, (uint8_t)6u));
  if (udt_restricting_r13_present) {
    HANDLE_CODE(udt_restricting_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE udt_restricting_per_plmn_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(udt_restricting_r13_present, 1));

  HANDLE_CODE(unpack_integer(plmn_id_idx_r13, bref, (uint8_t)1u, (uint8_t)6u));
  if (udt_restricting_r13_present) {
    HANDLE_CODE(udt_restricting_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void udt_restricting_per_plmn_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("plmn-IdentityIndex-r13", plmn_id_idx_r13);
  if (udt_restricting_r13_present) {
    j.write_fieldname("udt-Restricting-r13");
    udt_restricting_r13.to_json(j);
  }
  j.end_obj();
}

// UplinkPowerControlCommon ::= SEQUENCE
SRSASN_CODE ul_pwr_ctrl_common_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, p0_nominal_pusch, (int8_t)-126, (int8_t)24));
  HANDLE_CODE(alpha.pack(bref));
  HANDLE_CODE(pack_integer(bref, p0_nominal_pucch, (int8_t)-127, (int8_t)-96));
  HANDLE_CODE(delta_flist_pucch.pack(bref));
  HANDLE_CODE(pack_integer(bref, delta_preamb_msg3, (int8_t)-1, (int8_t)6));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_pwr_ctrl_common_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(p0_nominal_pusch, bref, (int8_t)-126, (int8_t)24));
  HANDLE_CODE(alpha.unpack(bref));
  HANDLE_CODE(unpack_integer(p0_nominal_pucch, bref, (int8_t)-127, (int8_t)-96));
  HANDLE_CODE(delta_flist_pucch.unpack(bref));
  HANDLE_CODE(unpack_integer(delta_preamb_msg3, bref, (int8_t)-1, (int8_t)6));

  return SRSASN_SUCCESS;
}
void ul_pwr_ctrl_common_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("p0-NominalPUSCH", p0_nominal_pusch);
  j.write_str("alpha", alpha.to_string());
  j.write_int("p0-NominalPUCCH", p0_nominal_pucch);
  j.write_fieldname("deltaFList-PUCCH");
  delta_flist_pucch.to_json(j);
  j.write_int("deltaPreambleMsg3", delta_preamb_msg3);
  j.end_obj();
}

// UplinkPowerControlCommon-v1020 ::= SEQUENCE
SRSASN_CODE ul_pwr_ctrl_common_v1020_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(delta_f_pucch_format3_r10.pack(bref));
  HANDLE_CODE(delta_f_pucch_format1b_cs_r10.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_pwr_ctrl_common_v1020_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(delta_f_pucch_format3_r10.unpack(bref));
  HANDLE_CODE(delta_f_pucch_format1b_cs_r10.unpack(bref));

  return SRSASN_SUCCESS;
}
void ul_pwr_ctrl_common_v1020_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("deltaF-PUCCH-Format3-r10", delta_f_pucch_format3_r10.to_string());
  j.write_str("deltaF-PUCCH-Format1bCS-r10", delta_f_pucch_format1b_cs_r10.to_string());
  j.end_obj();
}

// WLAN-OffloadConfig-r12 ::= SEQUENCE
SRSASN_CODE wlan_offload_cfg_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(thres_rsrp_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_rsrq_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_rsrq_on_all_symbols_with_wb_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_rsrq_on_all_symbols_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_rsrq_wb_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_ch_utilization_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_backhaul_bw_r12_present, 1));
  HANDLE_CODE(bref.pack(thres_wlan_rssi_r12_present, 1));
  HANDLE_CODE(bref.pack(offload_pref_ind_r12_present, 1));
  HANDLE_CODE(bref.pack(t_steering_wlan_r12_present, 1));

  if (thres_rsrp_r12_present) {
    HANDLE_CODE(pack_integer(bref, thres_rsrp_r12.thres_rsrp_low_r12, (uint8_t)0u, (uint8_t)97u));
    HANDLE_CODE(pack_integer(bref, thres_rsrp_r12.thres_rsrp_high_r12, (uint8_t)0u, (uint8_t)97u));
  }
  if (thres_rsrq_r12_present) {
    HANDLE_CODE(pack_integer(bref, thres_rsrq_r12.thres_rsrq_low_r12, (uint8_t)0u, (uint8_t)34u));
    HANDLE_CODE(pack_integer(bref, thres_rsrq_r12.thres_rsrq_high_r12, (uint8_t)0u, (uint8_t)34u));
  }
  if (thres_rsrq_on_all_symbols_with_wb_r12_present) {
    HANDLE_CODE(pack_integer(bref,
                             thres_rsrq_on_all_symbols_with_wb_r12.thres_rsrq_on_all_symbols_with_wb_low_r12,
                             (uint8_t)0u,
                             (uint8_t)34u));
    HANDLE_CODE(pack_integer(bref,
                             thres_rsrq_on_all_symbols_with_wb_r12.thres_rsrq_on_all_symbols_with_wb_high_r12,
                             (uint8_t)0u,
                             (uint8_t)34u));
  }
  if (thres_rsrq_on_all_symbols_r12_present) {
    HANDLE_CODE(
        pack_integer(bref, thres_rsrq_on_all_symbols_r12.thres_rsrq_on_all_symbols_low_r12, (uint8_t)0u, (uint8_t)34u));
    HANDLE_CODE(pack_integer(
        bref, thres_rsrq_on_all_symbols_r12.thres_rsrq_on_all_symbols_high_r12, (uint8_t)0u, (uint8_t)34u));
  }
  if (thres_rsrq_wb_r12_present) {
    HANDLE_CODE(pack_integer(bref, thres_rsrq_wb_r12.thres_rsrq_wb_low_r12, (uint8_t)0u, (uint8_t)34u));
    HANDLE_CODE(pack_integer(bref, thres_rsrq_wb_r12.thres_rsrq_wb_high_r12, (uint8_t)0u, (uint8_t)34u));
  }
  if (thres_ch_utilization_r12_present) {
    HANDLE_CODE(
        pack_integer(bref, thres_ch_utilization_r12.thres_ch_utilization_low_r12, (uint16_t)0u, (uint16_t)255u));
    HANDLE_CODE(
        pack_integer(bref, thres_ch_utilization_r12.thres_ch_utilization_high_r12, (uint16_t)0u, (uint16_t)255u));
  }
  if (thres_backhaul_bw_r12_present) {
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_dl_bw_low_r12.pack(bref));
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_dl_bw_high_r12.pack(bref));
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_ul_bw_low_r12.pack(bref));
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_ul_bw_high_r12.pack(bref));
  }
  if (thres_wlan_rssi_r12_present) {
    HANDLE_CODE(pack_integer(bref, thres_wlan_rssi_r12.thres_wlan_rssi_low_r12, (uint16_t)0u, (uint16_t)255u));
    HANDLE_CODE(pack_integer(bref, thres_wlan_rssi_r12.thres_wlan_rssi_high_r12, (uint16_t)0u, (uint16_t)255u));
  }
  if (offload_pref_ind_r12_present) {
    HANDLE_CODE(offload_pref_ind_r12.pack(bref));
  }
  if (t_steering_wlan_r12_present) {
    HANDLE_CODE(pack_integer(bref, t_steering_wlan_r12, (uint8_t)0u, (uint8_t)7u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE wlan_offload_cfg_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(thres_rsrp_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_rsrq_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_rsrq_on_all_symbols_with_wb_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_rsrq_on_all_symbols_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_rsrq_wb_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_ch_utilization_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_backhaul_bw_r12_present, 1));
  HANDLE_CODE(bref.unpack(thres_wlan_rssi_r12_present, 1));
  HANDLE_CODE(bref.unpack(offload_pref_ind_r12_present, 1));
  HANDLE_CODE(bref.unpack(t_steering_wlan_r12_present, 1));

  if (thres_rsrp_r12_present) {
    HANDLE_CODE(unpack_integer(thres_rsrp_r12.thres_rsrp_low_r12, bref, (uint8_t)0u, (uint8_t)97u));
    HANDLE_CODE(unpack_integer(thres_rsrp_r12.thres_rsrp_high_r12, bref, (uint8_t)0u, (uint8_t)97u));
  }
  if (thres_rsrq_r12_present) {
    HANDLE_CODE(unpack_integer(thres_rsrq_r12.thres_rsrq_low_r12, bref, (uint8_t)0u, (uint8_t)34u));
    HANDLE_CODE(unpack_integer(thres_rsrq_r12.thres_rsrq_high_r12, bref, (uint8_t)0u, (uint8_t)34u));
  }
  if (thres_rsrq_on_all_symbols_with_wb_r12_present) {
    HANDLE_CODE(unpack_integer(thres_rsrq_on_all_symbols_with_wb_r12.thres_rsrq_on_all_symbols_with_wb_low_r12,
                               bref,
                               (uint8_t)0u,
                               (uint8_t)34u));
    HANDLE_CODE(unpack_integer(thres_rsrq_on_all_symbols_with_wb_r12.thres_rsrq_on_all_symbols_with_wb_high_r12,
                               bref,
                               (uint8_t)0u,
                               (uint8_t)34u));
  }
  if (thres_rsrq_on_all_symbols_r12_present) {
    HANDLE_CODE(unpack_integer(
        thres_rsrq_on_all_symbols_r12.thres_rsrq_on_all_symbols_low_r12, bref, (uint8_t)0u, (uint8_t)34u));
    HANDLE_CODE(unpack_integer(
        thres_rsrq_on_all_symbols_r12.thres_rsrq_on_all_symbols_high_r12, bref, (uint8_t)0u, (uint8_t)34u));
  }
  if (thres_rsrq_wb_r12_present) {
    HANDLE_CODE(unpack_integer(thres_rsrq_wb_r12.thres_rsrq_wb_low_r12, bref, (uint8_t)0u, (uint8_t)34u));
    HANDLE_CODE(unpack_integer(thres_rsrq_wb_r12.thres_rsrq_wb_high_r12, bref, (uint8_t)0u, (uint8_t)34u));
  }
  if (thres_ch_utilization_r12_present) {
    HANDLE_CODE(
        unpack_integer(thres_ch_utilization_r12.thres_ch_utilization_low_r12, bref, (uint16_t)0u, (uint16_t)255u));
    HANDLE_CODE(
        unpack_integer(thres_ch_utilization_r12.thres_ch_utilization_high_r12, bref, (uint16_t)0u, (uint16_t)255u));
  }
  if (thres_backhaul_bw_r12_present) {
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_dl_bw_low_r12.unpack(bref));
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_dl_bw_high_r12.unpack(bref));
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_ul_bw_low_r12.unpack(bref));
    HANDLE_CODE(thres_backhaul_bw_r12.thres_backhaul_ul_bw_high_r12.unpack(bref));
  }
  if (thres_wlan_rssi_r12_present) {
    HANDLE_CODE(unpack_integer(thres_wlan_rssi_r12.thres_wlan_rssi_low_r12, bref, (uint16_t)0u, (uint16_t)255u));
    HANDLE_CODE(unpack_integer(thres_wlan_rssi_r12.thres_wlan_rssi_high_r12, bref, (uint16_t)0u, (uint16_t)255u));
  }
  if (offload_pref_ind_r12_present) {
    HANDLE_CODE(offload_pref_ind_r12.unpack(bref));
  }
  if (t_steering_wlan_r12_present) {
    HANDLE_CODE(unpack_integer(t_steering_wlan_r12, bref, (uint8_t)0u, (uint8_t)7u));
  }

  return SRSASN_SUCCESS;
}
void wlan_offload_cfg_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (thres_rsrp_r12_present) {
    j.write_fieldname("thresholdRSRP-r12");
    j.start_obj();
    j.write_int("thresholdRSRP-Low-r12", thres_rsrp_r12.thres_rsrp_low_r12);
    j.write_int("thresholdRSRP-High-r12", thres_rsrp_r12.thres_rsrp_high_r12);
    j.end_obj();
  }
  if (thres_rsrq_r12_present) {
    j.write_fieldname("thresholdRSRQ-r12");
    j.start_obj();
    j.write_int("thresholdRSRQ-Low-r12", thres_rsrq_r12.thres_rsrq_low_r12);
    j.write_int("thresholdRSRQ-High-r12", thres_rsrq_r12.thres_rsrq_high_r12);
    j.end_obj();
  }
  if (thres_rsrq_on_all_symbols_with_wb_r12_present) {
    j.write_fieldname("thresholdRSRQ-OnAllSymbolsWithWB-r12");
    j.start_obj();
    j.write_int("thresholdRSRQ-OnAllSymbolsWithWB-Low-r12",
                thres_rsrq_on_all_symbols_with_wb_r12.thres_rsrq_on_all_symbols_with_wb_low_r12);
    j.write_int("thresholdRSRQ-OnAllSymbolsWithWB-High-r12",
                thres_rsrq_on_all_symbols_with_wb_r12.thres_rsrq_on_all_symbols_with_wb_high_r12);
    j.end_obj();
  }
  if (thres_rsrq_on_all_symbols_r12_present) {
    j.write_fieldname("thresholdRSRQ-OnAllSymbols-r12");
    j.start_obj();
    j.write_int("thresholdRSRQ-OnAllSymbolsLow-r12", thres_rsrq_on_all_symbols_r12.thres_rsrq_on_all_symbols_low_r12);
    j.write_int("thresholdRSRQ-OnAllSymbolsHigh-r12", thres_rsrq_on_all_symbols_r12.thres_rsrq_on_all_symbols_high_r12);
    j.end_obj();
  }
  if (thres_rsrq_wb_r12_present) {
    j.write_fieldname("thresholdRSRQ-WB-r12");
    j.start_obj();
    j.write_int("thresholdRSRQ-WB-Low-r12", thres_rsrq_wb_r12.thres_rsrq_wb_low_r12);
    j.write_int("thresholdRSRQ-WB-High-r12", thres_rsrq_wb_r12.thres_rsrq_wb_high_r12);
    j.end_obj();
  }
  if (thres_ch_utilization_r12_present) {
    j.write_fieldname("thresholdChannelUtilization-r12");
    j.start_obj();
    j.write_int("thresholdChannelUtilizationLow-r12", thres_ch_utilization_r12.thres_ch_utilization_low_r12);
    j.write_int("thresholdChannelUtilizationHigh-r12", thres_ch_utilization_r12.thres_ch_utilization_high_r12);
    j.end_obj();
  }
  if (thres_backhaul_bw_r12_present) {
    j.write_fieldname("thresholdBackhaul-Bandwidth-r12");
    j.start_obj();
    j.write_str("thresholdBackhaulDL-BandwidthLow-r12", thres_backhaul_bw_r12.thres_backhaul_dl_bw_low_r12.to_string());
    j.write_str("thresholdBackhaulDL-BandwidthHigh-r12",
                thres_backhaul_bw_r12.thres_backhaul_dl_bw_high_r12.to_string());
    j.write_str("thresholdBackhaulUL-BandwidthLow-r12", thres_backhaul_bw_r12.thres_backhaul_ul_bw_low_r12.to_string());
    j.write_str("thresholdBackhaulUL-BandwidthHigh-r12",
                thres_backhaul_bw_r12.thres_backhaul_ul_bw_high_r12.to_string());
    j.end_obj();
  }
  if (thres_wlan_rssi_r12_present) {
    j.write_fieldname("thresholdWLAN-RSSI-r12");
    j.start_obj();
    j.write_int("thresholdWLAN-RSSI-Low-r12", thres_wlan_rssi_r12.thres_wlan_rssi_low_r12);
    j.write_int("thresholdWLAN-RSSI-High-r12", thres_wlan_rssi_r12.thres_wlan_rssi_high_r12);
    j.end_obj();
  }
  if (offload_pref_ind_r12_present) {
    j.write_str("offloadPreferenceIndicator-r12", offload_pref_ind_r12.to_string());
  }
  if (t_steering_wlan_r12_present) {
    j.write_int("t-SteeringWLAN-r12", t_steering_wlan_r12);
  }
  j.end_obj();
}

// WUS-Config-r15 ::= SEQUENCE
SRSASN_CODE wus_cfg_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(num_pos_r15_present, 1));
  HANDLE_CODE(bref.pack(time_offset_e_drx_long_r15_present, 1));

  HANDLE_CODE(max_dur_factor_r15.pack(bref));
  if (num_pos_r15_present) {
    HANDLE_CODE(num_pos_r15.pack(bref));
  }
  HANDLE_CODE(freq_location_r15.pack(bref));
  HANDLE_CODE(time_offset_drx_r15.pack(bref));
  HANDLE_CODE(time_offset_e_drx_short_r15.pack(bref));
  if (time_offset_e_drx_long_r15_present) {
    HANDLE_CODE(time_offset_e_drx_long_r15.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE wus_cfg_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(num_pos_r15_present, 1));
  HANDLE_CODE(bref.unpack(time_offset_e_drx_long_r15_present, 1));

  HANDLE_CODE(max_dur_factor_r15.unpack(bref));
  if (num_pos_r15_present) {
    HANDLE_CODE(num_pos_r15.unpack(bref));
  }
  HANDLE_CODE(freq_location_r15.unpack(bref));
  HANDLE_CODE(time_offset_drx_r15.unpack(bref));
  HANDLE_CODE(time_offset_e_drx_short_r15.unpack(bref));
  if (time_offset_e_drx_long_r15_present) {
    HANDLE_CODE(time_offset_e_drx_long_r15.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void wus_cfg_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("maxDurationFactor-r15", max_dur_factor_r15.to_string());
  if (num_pos_r15_present) {
    j.write_str("numPOs-r15", num_pos_r15.to_string());
  }
  j.write_str("freqLocation-r15", freq_location_r15.to_string());
  j.write_str("timeOffsetDRX-r15", time_offset_drx_r15.to_string());
  j.write_str("timeOffset-eDRX-Short-r15", time_offset_e_drx_short_r15.to_string());
  if (time_offset_e_drx_long_r15_present) {
    j.write_str("timeOffset-eDRX-Long-r15", time_offset_e_drx_long_r15.to_string());
  }
  j.end_obj();
}

// ACDC-BarringForCommon-r13 ::= SEQUENCE
SRSASN_CODE acdc_barr_for_common_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(acdc_hplm_nonly_r13, 1));
  HANDLE_CODE(pack_dyn_seq_of(bref, barr_per_acdc_category_list_r13, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE acdc_barr_for_common_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(acdc_hplm_nonly_r13, 1));
  HANDLE_CODE(unpack_dyn_seq_of(barr_per_acdc_category_list_r13, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void acdc_barr_for_common_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_bool("acdc-HPLMNonly-r13", acdc_hplm_nonly_r13);
  j.start_array("barringPerACDC-CategoryList-r13");
  for (const auto& e1 : barr_per_acdc_category_list_r13) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// CarrierFreqInfoUTRA-v1250 ::= SEQUENCE
SRSASN_CODE carrier_freq_info_utra_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(reduced_meas_performance_r12_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE carrier_freq_info_utra_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(reduced_meas_performance_r12_present, 1));

  return SRSASN_SUCCESS;
}
void carrier_freq_info_utra_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (reduced_meas_performance_r12_present) {
    j.write_str("reducedMeasPerformance-r12", "true");
  }
  j.end_obj();
}

// CellReselectionInfoCommon-v1460 ::= SEQUENCE
SRSASN_CODE cell_resel_info_common_v1460_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(s_search_delta_p_r14.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_resel_info_common_v1460_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(s_search_delta_p_r14.unpack(bref));

  return SRSASN_SUCCESS;
}
void cell_resel_info_common_v1460_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("s-SearchDeltaP-r14", s_search_delta_p_r14.to_string());
  j.end_obj();
}

// CellReselectionInfoHSDN-r15 ::= SEQUENCE
SRSASN_CODE cell_resel_info_hsdn_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, cell_equivalent_size_r15, (uint8_t)2u, (uint8_t)16u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_resel_info_hsdn_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(cell_equivalent_size_r15, bref, (uint8_t)2u, (uint8_t)16u));

  return SRSASN_SUCCESS;
}
void cell_resel_info_hsdn_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("cellEquivalentSize-r15", cell_equivalent_size_r15);
  j.end_obj();
}

// CellReselectionParametersCDMA2000 ::= SEQUENCE
SRSASN_CODE cell_resel_params_cdma2000_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(t_resel_cdma2000_sf_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, band_class_list, 1, 32));
  HANDLE_CODE(pack_dyn_seq_of(bref, neigh_cell_list, 1, 16));
  HANDLE_CODE(pack_integer(bref, t_resel_cdma2000, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_cdma2000_sf_present) {
    HANDLE_CODE(t_resel_cdma2000_sf.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_resel_params_cdma2000_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(t_resel_cdma2000_sf_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(band_class_list, bref, 1, 32));
  HANDLE_CODE(unpack_dyn_seq_of(neigh_cell_list, bref, 1, 16));
  HANDLE_CODE(unpack_integer(t_resel_cdma2000, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_cdma2000_sf_present) {
    HANDLE_CODE(t_resel_cdma2000_sf.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_resel_params_cdma2000_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("bandClassList");
  for (const auto& e1 : band_class_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.start_array("neighCellList");
  for (const auto& e1 : neigh_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_int("t-ReselectionCDMA2000", t_resel_cdma2000);
  if (t_resel_cdma2000_sf_present) {
    j.write_fieldname("t-ReselectionCDMA2000-SF");
    t_resel_cdma2000_sf.to_json(j);
  }
  j.end_obj();
}

// CellReselectionParametersCDMA2000-v920 ::= SEQUENCE
SRSASN_CODE cell_resel_params_cdma2000_v920_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_dyn_seq_of(bref, neigh_cell_list_v920, 1, 16));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_resel_params_cdma2000_v920_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_dyn_seq_of(neigh_cell_list_v920, bref, 1, 16));

  return SRSASN_SUCCESS;
}
void cell_resel_params_cdma2000_v920_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("neighCellList-v920");
  for (const auto& e1 : neigh_cell_list_v920) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// CellReselectionServingFreqInfo-v1310 ::= SEQUENCE
SRSASN_CODE cell_resel_serving_freq_info_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(cell_resel_sub_prio_r13.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_resel_serving_freq_info_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(cell_resel_sub_prio_r13.unpack(bref));

  return SRSASN_SUCCESS;
}
void cell_resel_serving_freq_info_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cellReselectionSubPriority-r13", cell_resel_sub_prio_r13.to_string());
  j.end_obj();
}

// CellSelectionInfo-v920 ::= SEQUENCE
SRSASN_CODE cell_sel_info_v920_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(q_qual_min_offset_r9_present, 1));

  HANDLE_CODE(pack_integer(bref, q_qual_min_r9, (int8_t)-34, (int8_t)-3));
  if (q_qual_min_offset_r9_present) {
    HANDLE_CODE(pack_integer(bref, q_qual_min_offset_r9, (uint8_t)1u, (uint8_t)8u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_sel_info_v920_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(q_qual_min_offset_r9_present, 1));

  HANDLE_CODE(unpack_integer(q_qual_min_r9, bref, (int8_t)-34, (int8_t)-3));
  if (q_qual_min_offset_r9_present) {
    HANDLE_CODE(unpack_integer(q_qual_min_offset_r9, bref, (uint8_t)1u, (uint8_t)8u));
  }

  return SRSASN_SUCCESS;
}
void cell_sel_info_v920_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("q-QualMin-r9", q_qual_min_r9);
  if (q_qual_min_offset_r9_present) {
    j.write_int("q-QualMinOffset-r9", q_qual_min_offset_r9);
  }
  j.end_obj();
}

// EAB-ConfigPLMN-r11 ::= SEQUENCE
SRSASN_CODE eab_cfg_plmn_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(eab_cfg_r11_present, 1));

  if (eab_cfg_r11_present) {
    HANDLE_CODE(eab_cfg_r11.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE eab_cfg_plmn_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(eab_cfg_r11_present, 1));

  if (eab_cfg_r11_present) {
    HANDLE_CODE(eab_cfg_r11.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void eab_cfg_plmn_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (eab_cfg_r11_present) {
    j.write_fieldname("eab-Config-r11");
    eab_cfg_r11.to_json(j);
  }
  j.end_obj();
}

// MBMS-NotificationConfig-r9 ::= SEQUENCE
SRSASN_CODE mbms_notif_cfg_r9_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(notif_repeat_coeff_r9.pack(bref));
  HANDLE_CODE(pack_integer(bref, notif_offset_r9, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(pack_integer(bref, notif_sf_idx_r9, (uint8_t)1u, (uint8_t)6u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbms_notif_cfg_r9_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(notif_repeat_coeff_r9.unpack(bref));
  HANDLE_CODE(unpack_integer(notif_offset_r9, bref, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(unpack_integer(notif_sf_idx_r9, bref, (uint8_t)1u, (uint8_t)6u));

  return SRSASN_SUCCESS;
}
void mbms_notif_cfg_r9_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("notificationRepetitionCoeff-r9", notif_repeat_coeff_r9.to_string());
  j.write_int("notificationOffset-r9", notif_offset_r9);
  j.write_int("notificationSF-Index-r9", notif_sf_idx_r9);
  j.end_obj();
}

// MBMS-NotificationConfig-v1430 ::= SEQUENCE
SRSASN_CODE mbms_notif_cfg_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, notif_sf_idx_v1430, (uint8_t)7u, (uint8_t)10u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbms_notif_cfg_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(notif_sf_idx_v1430, bref, (uint8_t)7u, (uint8_t)10u));

  return SRSASN_SUCCESS;
}
void mbms_notif_cfg_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("notificationSF-Index-v1430", notif_sf_idx_v1430);
  j.end_obj();
}

// MeasIdleConfigSIB-r15 ::= SEQUENCE
SRSASN_CODE meas_idle_cfg_sib_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, meas_idle_carrier_list_eutra_r15, 1, 8));

  return SRSASN_SUCCESS;
}
SRSASN_CODE meas_idle_cfg_sib_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(meas_idle_carrier_list_eutra_r15, bref, 1, 8));

  return SRSASN_SUCCESS;
}
void meas_idle_cfg_sib_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("measIdleCarrierListEUTRA-r15");
  for (const auto& e1 : meas_idle_carrier_list_eutra_r15) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// MobilityStateParameters ::= SEQUENCE
SRSASN_CODE mob_state_params_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(t_eval.pack(bref));
  HANDLE_CODE(t_hyst_normal.pack(bref));
  HANDLE_CODE(pack_integer(bref, n_cell_change_medium, (uint8_t)1u, (uint8_t)16u));
  HANDLE_CODE(pack_integer(bref, n_cell_change_high, (uint8_t)1u, (uint8_t)16u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE mob_state_params_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(t_eval.unpack(bref));
  HANDLE_CODE(t_hyst_normal.unpack(bref));
  HANDLE_CODE(unpack_integer(n_cell_change_medium, bref, (uint8_t)1u, (uint8_t)16u));
  HANDLE_CODE(unpack_integer(n_cell_change_high, bref, (uint8_t)1u, (uint8_t)16u));

  return SRSASN_SUCCESS;
}
void mob_state_params_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("t-Evaluation", t_eval.to_string());
  j.write_str("t-HystNormal", t_hyst_normal.to_string());
  j.write_int("n-CellChangeMedium", n_cell_change_medium);
  j.write_int("n-CellChangeHigh", n_cell_change_high);
  j.end_obj();
}

// RadioResourceConfigCommonSIB ::= SEQUENCE
SRSASN_CODE rr_cfg_common_sib_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(rach_cfg_common.pack(bref));
  HANDLE_CODE(bcch_cfg.pack(bref));
  HANDLE_CODE(pcch_cfg.pack(bref));
  HANDLE_CODE(prach_cfg.pack(bref));
  HANDLE_CODE(pdsch_cfg_common.pack(bref));
  HANDLE_CODE(pusch_cfg_common.pack(bref));
  HANDLE_CODE(pucch_cfg_common.pack(bref));
  HANDLE_CODE(srs_ul_cfg_common.pack(bref));
  HANDLE_CODE(ul_pwr_ctrl_common.pack(bref));
  HANDLE_CODE(ul_cp_len.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= ul_pwr_ctrl_common_v1020.is_present();
    group_flags[1] |= rach_cfg_common_v1250.is_present();
    group_flags[2] |= pusch_cfg_common_v1270.is_present();
    group_flags[3] |= bcch_cfg_v1310.is_present();
    group_flags[3] |= pcch_cfg_v1310.is_present();
    group_flags[3] |= freq_hop_params_r13.is_present();
    group_flags[3] |= pdsch_cfg_common_v1310.is_present();
    group_flags[3] |= pusch_cfg_common_v1310.is_present();
    group_flags[3] |= prach_cfg_common_v1310.is_present();
    group_flags[3] |= pucch_cfg_common_v1310.is_present();
    group_flags[4] |= high_speed_cfg_r14.is_present();
    group_flags[4] |= prach_cfg_v1430.is_present();
    group_flags[4] |= pucch_cfg_common_v1430.is_present();
    group_flags[5] |= prach_cfg_v1530.is_present();
    group_flags[5] |= ce_rss_cfg_r15.is_present();
    group_flags[5] |= wus_cfg_r15.is_present();
    group_flags[5] |= high_speed_cfg_v1530.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ul_pwr_ctrl_common_v1020.is_present(), 1));
      if (ul_pwr_ctrl_common_v1020.is_present()) {
        HANDLE_CODE(ul_pwr_ctrl_common_v1020->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(rach_cfg_common_v1250.is_present(), 1));
      if (rach_cfg_common_v1250.is_present()) {
        HANDLE_CODE(rach_cfg_common_v1250->pack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(pusch_cfg_common_v1270.is_present(), 1));
      if (pusch_cfg_common_v1270.is_present()) {
        HANDLE_CODE(pusch_cfg_common_v1270->pack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(bcch_cfg_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(pcch_cfg_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(freq_hop_params_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(pdsch_cfg_common_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(pusch_cfg_common_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(prach_cfg_common_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(pucch_cfg_common_v1310.is_present(), 1));
      if (bcch_cfg_v1310.is_present()) {
        HANDLE_CODE(bcch_cfg_v1310->pack(bref));
      }
      if (pcch_cfg_v1310.is_present()) {
        HANDLE_CODE(pcch_cfg_v1310->pack(bref));
      }
      if (freq_hop_params_r13.is_present()) {
        HANDLE_CODE(freq_hop_params_r13->pack(bref));
      }
      if (pdsch_cfg_common_v1310.is_present()) {
        HANDLE_CODE(pdsch_cfg_common_v1310->pack(bref));
      }
      if (pusch_cfg_common_v1310.is_present()) {
        HANDLE_CODE(pusch_cfg_common_v1310->pack(bref));
      }
      if (prach_cfg_common_v1310.is_present()) {
        HANDLE_CODE(prach_cfg_common_v1310->pack(bref));
      }
      if (pucch_cfg_common_v1310.is_present()) {
        HANDLE_CODE(pucch_cfg_common_v1310->pack(bref));
      }
    }
    if (group_flags[4]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(high_speed_cfg_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(prach_cfg_v1430.is_present(), 1));
      HANDLE_CODE(bref.pack(pucch_cfg_common_v1430.is_present(), 1));
      if (high_speed_cfg_r14.is_present()) {
        HANDLE_CODE(high_speed_cfg_r14->pack(bref));
      }
      if (prach_cfg_v1430.is_present()) {
        HANDLE_CODE(prach_cfg_v1430->pack(bref));
      }
      if (pucch_cfg_common_v1430.is_present()) {
        HANDLE_CODE(pucch_cfg_common_v1430->pack(bref));
      }
    }
    if (group_flags[5]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(prach_cfg_v1530.is_present(), 1));
      HANDLE_CODE(bref.pack(ce_rss_cfg_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(wus_cfg_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(high_speed_cfg_v1530.is_present(), 1));
      if (prach_cfg_v1530.is_present()) {
        HANDLE_CODE(prach_cfg_v1530->pack(bref));
      }
      if (ce_rss_cfg_r15.is_present()) {
        HANDLE_CODE(ce_rss_cfg_r15->pack(bref));
      }
      if (wus_cfg_r15.is_present()) {
        HANDLE_CODE(wus_cfg_r15->pack(bref));
      }
      if (high_speed_cfg_v1530.is_present()) {
        HANDLE_CODE(high_speed_cfg_v1530->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rr_cfg_common_sib_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(rach_cfg_common.unpack(bref));
  HANDLE_CODE(bcch_cfg.unpack(bref));
  HANDLE_CODE(pcch_cfg.unpack(bref));
  HANDLE_CODE(prach_cfg.unpack(bref));
  HANDLE_CODE(pdsch_cfg_common.unpack(bref));
  HANDLE_CODE(pusch_cfg_common.unpack(bref));
  HANDLE_CODE(pucch_cfg_common.unpack(bref));
  HANDLE_CODE(srs_ul_cfg_common.unpack(bref));
  HANDLE_CODE(ul_pwr_ctrl_common.unpack(bref));
  HANDLE_CODE(ul_cp_len.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(6);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool ul_pwr_ctrl_common_v1020_present;
      HANDLE_CODE(bref.unpack(ul_pwr_ctrl_common_v1020_present, 1));
      ul_pwr_ctrl_common_v1020.set_present(ul_pwr_ctrl_common_v1020_present);
      if (ul_pwr_ctrl_common_v1020.is_present()) {
        HANDLE_CODE(ul_pwr_ctrl_common_v1020->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool rach_cfg_common_v1250_present;
      HANDLE_CODE(bref.unpack(rach_cfg_common_v1250_present, 1));
      rach_cfg_common_v1250.set_present(rach_cfg_common_v1250_present);
      if (rach_cfg_common_v1250.is_present()) {
        HANDLE_CODE(rach_cfg_common_v1250->unpack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool pusch_cfg_common_v1270_present;
      HANDLE_CODE(bref.unpack(pusch_cfg_common_v1270_present, 1));
      pusch_cfg_common_v1270.set_present(pusch_cfg_common_v1270_present);
      if (pusch_cfg_common_v1270.is_present()) {
        HANDLE_CODE(pusch_cfg_common_v1270->unpack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool bcch_cfg_v1310_present;
      HANDLE_CODE(bref.unpack(bcch_cfg_v1310_present, 1));
      bcch_cfg_v1310.set_present(bcch_cfg_v1310_present);
      bool pcch_cfg_v1310_present;
      HANDLE_CODE(bref.unpack(pcch_cfg_v1310_present, 1));
      pcch_cfg_v1310.set_present(pcch_cfg_v1310_present);
      bool freq_hop_params_r13_present;
      HANDLE_CODE(bref.unpack(freq_hop_params_r13_present, 1));
      freq_hop_params_r13.set_present(freq_hop_params_r13_present);
      bool pdsch_cfg_common_v1310_present;
      HANDLE_CODE(bref.unpack(pdsch_cfg_common_v1310_present, 1));
      pdsch_cfg_common_v1310.set_present(pdsch_cfg_common_v1310_present);
      bool pusch_cfg_common_v1310_present;
      HANDLE_CODE(bref.unpack(pusch_cfg_common_v1310_present, 1));
      pusch_cfg_common_v1310.set_present(pusch_cfg_common_v1310_present);
      bool prach_cfg_common_v1310_present;
      HANDLE_CODE(bref.unpack(prach_cfg_common_v1310_present, 1));
      prach_cfg_common_v1310.set_present(prach_cfg_common_v1310_present);
      bool pucch_cfg_common_v1310_present;
      HANDLE_CODE(bref.unpack(pucch_cfg_common_v1310_present, 1));
      pucch_cfg_common_v1310.set_present(pucch_cfg_common_v1310_present);
      if (bcch_cfg_v1310.is_present()) {
        HANDLE_CODE(bcch_cfg_v1310->unpack(bref));
      }
      if (pcch_cfg_v1310.is_present()) {
        HANDLE_CODE(pcch_cfg_v1310->unpack(bref));
      }
      if (freq_hop_params_r13.is_present()) {
        HANDLE_CODE(freq_hop_params_r13->unpack(bref));
      }
      if (pdsch_cfg_common_v1310.is_present()) {
        HANDLE_CODE(pdsch_cfg_common_v1310->unpack(bref));
      }
      if (pusch_cfg_common_v1310.is_present()) {
        HANDLE_CODE(pusch_cfg_common_v1310->unpack(bref));
      }
      if (prach_cfg_common_v1310.is_present()) {
        HANDLE_CODE(prach_cfg_common_v1310->unpack(bref));
      }
      if (pucch_cfg_common_v1310.is_present()) {
        HANDLE_CODE(pucch_cfg_common_v1310->unpack(bref));
      }
    }
    if (group_flags[4]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool high_speed_cfg_r14_present;
      HANDLE_CODE(bref.unpack(high_speed_cfg_r14_present, 1));
      high_speed_cfg_r14.set_present(high_speed_cfg_r14_present);
      bool prach_cfg_v1430_present;
      HANDLE_CODE(bref.unpack(prach_cfg_v1430_present, 1));
      prach_cfg_v1430.set_present(prach_cfg_v1430_present);
      bool pucch_cfg_common_v1430_present;
      HANDLE_CODE(bref.unpack(pucch_cfg_common_v1430_present, 1));
      pucch_cfg_common_v1430.set_present(pucch_cfg_common_v1430_present);
      if (high_speed_cfg_r14.is_present()) {
        HANDLE_CODE(high_speed_cfg_r14->unpack(bref));
      }
      if (prach_cfg_v1430.is_present()) {
        HANDLE_CODE(prach_cfg_v1430->unpack(bref));
      }
      if (pucch_cfg_common_v1430.is_present()) {
        HANDLE_CODE(pucch_cfg_common_v1430->unpack(bref));
      }
    }
    if (group_flags[5]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool prach_cfg_v1530_present;
      HANDLE_CODE(bref.unpack(prach_cfg_v1530_present, 1));
      prach_cfg_v1530.set_present(prach_cfg_v1530_present);
      bool ce_rss_cfg_r15_present;
      HANDLE_CODE(bref.unpack(ce_rss_cfg_r15_present, 1));
      ce_rss_cfg_r15.set_present(ce_rss_cfg_r15_present);
      bool wus_cfg_r15_present;
      HANDLE_CODE(bref.unpack(wus_cfg_r15_present, 1));
      wus_cfg_r15.set_present(wus_cfg_r15_present);
      bool high_speed_cfg_v1530_present;
      HANDLE_CODE(bref.unpack(high_speed_cfg_v1530_present, 1));
      high_speed_cfg_v1530.set_present(high_speed_cfg_v1530_present);
      if (prach_cfg_v1530.is_present()) {
        HANDLE_CODE(prach_cfg_v1530->unpack(bref));
      }
      if (ce_rss_cfg_r15.is_present()) {
        HANDLE_CODE(ce_rss_cfg_r15->unpack(bref));
      }
      if (wus_cfg_r15.is_present()) {
        HANDLE_CODE(wus_cfg_r15->unpack(bref));
      }
      if (high_speed_cfg_v1530.is_present()) {
        HANDLE_CODE(high_speed_cfg_v1530->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void rr_cfg_common_sib_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("rach-ConfigCommon");
  rach_cfg_common.to_json(j);
  j.write_fieldname("bcch-Config");
  bcch_cfg.to_json(j);
  j.write_fieldname("pcch-Config");
  pcch_cfg.to_json(j);
  j.write_fieldname("prach-Config");
  prach_cfg.to_json(j);
  j.write_fieldname("pdsch-ConfigCommon");
  pdsch_cfg_common.to_json(j);
  j.write_fieldname("pusch-ConfigCommon");
  pusch_cfg_common.to_json(j);
  j.write_fieldname("pucch-ConfigCommon");
  pucch_cfg_common.to_json(j);
  j.write_fieldname("soundingRS-UL-ConfigCommon");
  srs_ul_cfg_common.to_json(j);
  j.write_fieldname("uplinkPowerControlCommon");
  ul_pwr_ctrl_common.to_json(j);
  j.write_str("ul-CyclicPrefixLength", ul_cp_len.to_string());
  if (ext) {
    if (ul_pwr_ctrl_common_v1020.is_present()) {
      j.write_fieldname("uplinkPowerControlCommon-v1020");
      ul_pwr_ctrl_common_v1020->to_json(j);
    }
    if (rach_cfg_common_v1250.is_present()) {
      j.write_fieldname("rach-ConfigCommon-v1250");
      rach_cfg_common_v1250->to_json(j);
    }
    if (pusch_cfg_common_v1270.is_present()) {
      j.write_fieldname("pusch-ConfigCommon-v1270");
      pusch_cfg_common_v1270->to_json(j);
    }
    if (bcch_cfg_v1310.is_present()) {
      j.write_fieldname("bcch-Config-v1310");
      bcch_cfg_v1310->to_json(j);
    }
    if (pcch_cfg_v1310.is_present()) {
      j.write_fieldname("pcch-Config-v1310");
      pcch_cfg_v1310->to_json(j);
    }
    if (freq_hop_params_r13.is_present()) {
      j.write_fieldname("freqHoppingParameters-r13");
      freq_hop_params_r13->to_json(j);
    }
    if (pdsch_cfg_common_v1310.is_present()) {
      j.write_fieldname("pdsch-ConfigCommon-v1310");
      pdsch_cfg_common_v1310->to_json(j);
    }
    if (pusch_cfg_common_v1310.is_present()) {
      j.write_fieldname("pusch-ConfigCommon-v1310");
      pusch_cfg_common_v1310->to_json(j);
    }
    if (prach_cfg_common_v1310.is_present()) {
      j.write_fieldname("prach-ConfigCommon-v1310");
      prach_cfg_common_v1310->to_json(j);
    }
    if (pucch_cfg_common_v1310.is_present()) {
      j.write_fieldname("pucch-ConfigCommon-v1310");
      pucch_cfg_common_v1310->to_json(j);
    }
    if (high_speed_cfg_r14.is_present()) {
      j.write_fieldname("highSpeedConfig-r14");
      high_speed_cfg_r14->to_json(j);
    }
    if (prach_cfg_v1430.is_present()) {
      j.write_fieldname("prach-Config-v1430");
      prach_cfg_v1430->to_json(j);
    }
    if (pucch_cfg_common_v1430.is_present()) {
      j.write_fieldname("pucch-ConfigCommon-v1430");
      pucch_cfg_common_v1430->to_json(j);
    }
    if (prach_cfg_v1530.is_present()) {
      j.write_fieldname("prach-Config-v1530");
      prach_cfg_v1530->to_json(j);
    }
    if (ce_rss_cfg_r15.is_present()) {
      j.write_fieldname("ce-RSS-Config-r15");
      ce_rss_cfg_r15->to_json(j);
    }
    if (wus_cfg_r15.is_present()) {
      j.write_fieldname("wus-Config-r15");
      wus_cfg_r15->to_json(j);
    }
    if (high_speed_cfg_v1530.is_present()) {
      j.write_fieldname("highSpeedConfig-v1530");
      high_speed_cfg_v1530->to_json(j);
    }
  }
  j.end_obj();
}

// RedistributionServingInfo-r13 ::= SEQUENCE
SRSASN_CODE redist_serving_info_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(redist_factor_cell_r13_present, 1));
  HANDLE_CODE(bref.pack(redistr_on_paging_only_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, redist_factor_serving_r13, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(t360_r13.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE redist_serving_info_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(redist_factor_cell_r13_present, 1));
  HANDLE_CODE(bref.unpack(redistr_on_paging_only_r13_present, 1));

  HANDLE_CODE(unpack_integer(redist_factor_serving_r13, bref, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(t360_r13.unpack(bref));

  return SRSASN_SUCCESS;
}
void redist_serving_info_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("redistributionFactorServing-r13", redist_factor_serving_r13);
  if (redist_factor_cell_r13_present) {
    j.write_str("redistributionFactorCell-r13", "true");
  }
  j.write_str("t360-r13", t360_r13.to_string());
  if (redistr_on_paging_only_r13_present) {
    j.write_str("redistrOnPagingOnly-r13", "true");
  }
  j.end_obj();
}

// SC-MCCH-SchedulingInfo-r14 ::= SEQUENCE
SRSASN_CODE sc_mcch_sched_info_r14_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(on_dur_timer_scptm_r14.pack(bref));
  HANDLE_CODE(drx_inactivity_timer_scptm_r14.pack(bref));
  HANDLE_CODE(sched_period_start_offset_scptm_r14.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sc_mcch_sched_info_r14_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(on_dur_timer_scptm_r14.unpack(bref));
  HANDLE_CODE(drx_inactivity_timer_scptm_r14.unpack(bref));
  HANDLE_CODE(sched_period_start_offset_scptm_r14.unpack(bref));

  return SRSASN_SUCCESS;
}
void sc_mcch_sched_info_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("onDurationTimerSCPTM-r14", on_dur_timer_scptm_r14.to_string());
  j.write_str("drx-InactivityTimerSCPTM-r14", drx_inactivity_timer_scptm_r14.to_string());
  j.write_fieldname("schedulingPeriodStartOffsetSCPTM-r14");
  sched_period_start_offset_scptm_r14.to_json(j);
  j.end_obj();
}

void sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::destroy_() {}
void sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::sched_period_start_offset_scptm_r14_c_(
    const sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sf10:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf20:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf32:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf40:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf64:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf80:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf128:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf160:
      c.init(other.c.get<uint8_t>());
      break;
    case types::sf256:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf320:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf512:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf640:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf1024:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf2048:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf4096:
      c.init(other.c.get<uint16_t>());
      break;
    case types::sf8192:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_");
  }
}
sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_&
sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::
operator=(const sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sf10:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf20:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf32:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf40:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf64:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf80:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf128:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf160:
      c.set(other.c.get<uint8_t>());
      break;
    case types::sf256:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf320:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf512:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf640:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf1024:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf2048:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf4096:
      c.set(other.c.get<uint16_t>());
      break;
    case types::sf8192:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_");
  }

  return *this;
}
void sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sf10:
      j.write_int("sf10", c.get<uint8_t>());
      break;
    case types::sf20:
      j.write_int("sf20", c.get<uint8_t>());
      break;
    case types::sf32:
      j.write_int("sf32", c.get<uint8_t>());
      break;
    case types::sf40:
      j.write_int("sf40", c.get<uint8_t>());
      break;
    case types::sf64:
      j.write_int("sf64", c.get<uint8_t>());
      break;
    case types::sf80:
      j.write_int("sf80", c.get<uint8_t>());
      break;
    case types::sf128:
      j.write_int("sf128", c.get<uint8_t>());
      break;
    case types::sf160:
      j.write_int("sf160", c.get<uint8_t>());
      break;
    case types::sf256:
      j.write_int("sf256", c.get<uint16_t>());
      break;
    case types::sf320:
      j.write_int("sf320", c.get<uint16_t>());
      break;
    case types::sf512:
      j.write_int("sf512", c.get<uint16_t>());
      break;
    case types::sf640:
      j.write_int("sf640", c.get<uint16_t>());
      break;
    case types::sf1024:
      j.write_int("sf1024", c.get<uint16_t>());
      break;
    case types::sf2048:
      j.write_int("sf2048", c.get<uint16_t>());
      break;
    case types::sf4096:
      j.write_int("sf4096", c.get<uint16_t>());
      break;
    case types::sf8192:
      j.write_int("sf8192", c.get<uint16_t>());
      break;
    default:
      log_invalid_choice_id(type_, "sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_");
  }
  j.end_obj();
}
SRSASN_CODE sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sf10:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)9u));
      break;
    case types::sf20:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)19u));
      break;
    case types::sf32:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)31u));
      break;
    case types::sf40:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)39u));
      break;
    case types::sf64:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)63u));
      break;
    case types::sf80:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)79u));
      break;
    case types::sf128:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)127u));
      break;
    case types::sf160:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)159u));
      break;
    case types::sf256:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u));
      break;
    case types::sf320:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)319u));
      break;
    case types::sf512:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)511u));
      break;
    case types::sf640:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)639u));
      break;
    case types::sf1024:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)1023u));
      break;
    case types::sf2048:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)2047u));
      break;
    case types::sf4096:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)4095u));
      break;
    case types::sf8192:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)8191u));
      break;
    default:
      log_invalid_choice_id(type_, "sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sf10:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)9u));
      break;
    case types::sf20:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)19u));
      break;
    case types::sf32:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)31u));
      break;
    case types::sf40:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)39u));
      break;
    case types::sf64:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)63u));
      break;
    case types::sf80:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)79u));
      break;
    case types::sf128:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)127u));
      break;
    case types::sf160:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)159u));
      break;
    case types::sf256:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u));
      break;
    case types::sf320:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)319u));
      break;
    case types::sf512:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)511u));
      break;
    case types::sf640:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)639u));
      break;
    case types::sf1024:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)1023u));
      break;
    case types::sf2048:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)2047u));
      break;
    case types::sf4096:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)4095u));
      break;
    case types::sf8192:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)8191u));
      break;
    default:
      log_invalid_choice_id(type_, "sc_mcch_sched_info_r14_s::sched_period_start_offset_scptm_r14_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SL-DiscConfigRelayUE-r13 ::= SEQUENCE
SRSASN_CODE sl_disc_cfg_relay_ue_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(thresh_high_r13_present, 1));
  HANDLE_CODE(bref.pack(thresh_low_r13_present, 1));
  HANDLE_CODE(bref.pack(hyst_max_r13_present, 1));
  HANDLE_CODE(bref.pack(hyst_min_r13_present, 1));

  if (thresh_high_r13_present) {
    HANDLE_CODE(pack_integer(bref, thresh_high_r13, (uint8_t)0u, (uint8_t)49u));
  }
  if (thresh_low_r13_present) {
    HANDLE_CODE(pack_integer(bref, thresh_low_r13, (uint8_t)0u, (uint8_t)49u));
  }
  if (hyst_max_r13_present) {
    HANDLE_CODE(hyst_max_r13.pack(bref));
  }
  if (hyst_min_r13_present) {
    HANDLE_CODE(hyst_min_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_cfg_relay_ue_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(thresh_high_r13_present, 1));
  HANDLE_CODE(bref.unpack(thresh_low_r13_present, 1));
  HANDLE_CODE(bref.unpack(hyst_max_r13_present, 1));
  HANDLE_CODE(bref.unpack(hyst_min_r13_present, 1));

  if (thresh_high_r13_present) {
    HANDLE_CODE(unpack_integer(thresh_high_r13, bref, (uint8_t)0u, (uint8_t)49u));
  }
  if (thresh_low_r13_present) {
    HANDLE_CODE(unpack_integer(thresh_low_r13, bref, (uint8_t)0u, (uint8_t)49u));
  }
  if (hyst_max_r13_present) {
    HANDLE_CODE(hyst_max_r13.unpack(bref));
  }
  if (hyst_min_r13_present) {
    HANDLE_CODE(hyst_min_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_disc_cfg_relay_ue_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (thresh_high_r13_present) {
    j.write_int("threshHigh-r13", thresh_high_r13);
  }
  if (thresh_low_r13_present) {
    j.write_int("threshLow-r13", thresh_low_r13);
  }
  if (hyst_max_r13_present) {
    j.write_str("hystMax-r13", hyst_max_r13.to_string());
  }
  if (hyst_min_r13_present) {
    j.write_str("hystMin-r13", hyst_min_r13.to_string());
  }
  j.end_obj();
}

// SL-DiscConfigRemoteUE-r13 ::= SEQUENCE
SRSASN_CODE sl_disc_cfg_remote_ue_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(thresh_high_r13_present, 1));
  HANDLE_CODE(bref.pack(hyst_max_r13_present, 1));

  if (thresh_high_r13_present) {
    HANDLE_CODE(pack_integer(bref, thresh_high_r13, (uint8_t)0u, (uint8_t)49u));
  }
  if (hyst_max_r13_present) {
    HANDLE_CODE(hyst_max_r13.pack(bref));
  }
  HANDLE_CODE(resel_info_ic_r13.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_disc_cfg_remote_ue_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(thresh_high_r13_present, 1));
  HANDLE_CODE(bref.unpack(hyst_max_r13_present, 1));

  if (thresh_high_r13_present) {
    HANDLE_CODE(unpack_integer(thresh_high_r13, bref, (uint8_t)0u, (uint8_t)49u));
  }
  if (hyst_max_r13_present) {
    HANDLE_CODE(hyst_max_r13.unpack(bref));
  }
  HANDLE_CODE(resel_info_ic_r13.unpack(bref));

  return SRSASN_SUCCESS;
}
void sl_disc_cfg_remote_ue_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (thresh_high_r13_present) {
    j.write_int("threshHigh-r13", thresh_high_r13);
  }
  if (hyst_max_r13_present) {
    j.write_str("hystMax-r13", hyst_max_r13.to_string());
  }
  j.write_fieldname("reselectionInfoIC-r13");
  resel_info_ic_r13.to_json(j);
  j.end_obj();
}

// SL-V2X-ConfigCommon-r14 ::= SEQUENCE
SRSASN_CODE sl_v2x_cfg_common_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(v2x_comm_rx_pool_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_comm_tx_pool_normal_common_r14_present, 1));
  HANDLE_CODE(bref.pack(p2x_comm_tx_pool_normal_common_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_comm_tx_pool_exceptional_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_sync_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_inter_freq_info_list_r14_present, 1));
  HANDLE_CODE(bref.pack(v2x_res_sel_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(zone_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(type_tx_sync_r14_present, 1));
  HANDLE_CODE(bref.pack(thres_sl_tx_prioritization_r14_present, 1));
  HANDLE_CODE(bref.pack(anchor_carrier_freq_list_r14_present, 1));
  HANDLE_CODE(bref.pack(offset_dfn_r14_present, 1));
  HANDLE_CODE(bref.pack(cbr_common_tx_cfg_list_r14_present, 1));

  if (v2x_comm_rx_pool_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_comm_rx_pool_r14, 1, 16));
  }
  if (v2x_comm_tx_pool_normal_common_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_comm_tx_pool_normal_common_r14, 1, 8));
  }
  if (p2x_comm_tx_pool_normal_common_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, p2x_comm_tx_pool_normal_common_r14, 1, 8));
  }
  if (v2x_comm_tx_pool_exceptional_r14_present) {
    HANDLE_CODE(v2x_comm_tx_pool_exceptional_r14.pack(bref));
  }
  if (v2x_sync_cfg_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_sync_cfg_r14, 1, 16));
  }
  if (v2x_inter_freq_info_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_inter_freq_info_list_r14, 0, 7));
  }
  if (v2x_res_sel_cfg_r14_present) {
    HANDLE_CODE(v2x_res_sel_cfg_r14.pack(bref));
  }
  if (zone_cfg_r14_present) {
    HANDLE_CODE(zone_cfg_r14.pack(bref));
  }
  if (type_tx_sync_r14_present) {
    HANDLE_CODE(type_tx_sync_r14.pack(bref));
  }
  if (thres_sl_tx_prioritization_r14_present) {
    HANDLE_CODE(pack_integer(bref, thres_sl_tx_prioritization_r14, (uint8_t)1u, (uint8_t)8u));
  }
  if (anchor_carrier_freq_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, anchor_carrier_freq_list_r14, 1, 8, integer_packer<uint32_t>(0, 262143)));
  }
  if (offset_dfn_r14_present) {
    HANDLE_CODE(pack_integer(bref, offset_dfn_r14, (uint16_t)0u, (uint16_t)1000u));
  }
  if (cbr_common_tx_cfg_list_r14_present) {
    HANDLE_CODE(cbr_common_tx_cfg_list_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_v2x_cfg_common_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(v2x_comm_rx_pool_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_comm_tx_pool_normal_common_r14_present, 1));
  HANDLE_CODE(bref.unpack(p2x_comm_tx_pool_normal_common_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_comm_tx_pool_exceptional_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_sync_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_inter_freq_info_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(v2x_res_sel_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(zone_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(type_tx_sync_r14_present, 1));
  HANDLE_CODE(bref.unpack(thres_sl_tx_prioritization_r14_present, 1));
  HANDLE_CODE(bref.unpack(anchor_carrier_freq_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(offset_dfn_r14_present, 1));
  HANDLE_CODE(bref.unpack(cbr_common_tx_cfg_list_r14_present, 1));

  if (v2x_comm_rx_pool_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_comm_rx_pool_r14, bref, 1, 16));
  }
  if (v2x_comm_tx_pool_normal_common_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_comm_tx_pool_normal_common_r14, bref, 1, 8));
  }
  if (p2x_comm_tx_pool_normal_common_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(p2x_comm_tx_pool_normal_common_r14, bref, 1, 8));
  }
  if (v2x_comm_tx_pool_exceptional_r14_present) {
    HANDLE_CODE(v2x_comm_tx_pool_exceptional_r14.unpack(bref));
  }
  if (v2x_sync_cfg_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_sync_cfg_r14, bref, 1, 16));
  }
  if (v2x_inter_freq_info_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_inter_freq_info_list_r14, bref, 0, 7));
  }
  if (v2x_res_sel_cfg_r14_present) {
    HANDLE_CODE(v2x_res_sel_cfg_r14.unpack(bref));
  }
  if (zone_cfg_r14_present) {
    HANDLE_CODE(zone_cfg_r14.unpack(bref));
  }
  if (type_tx_sync_r14_present) {
    HANDLE_CODE(type_tx_sync_r14.unpack(bref));
  }
  if (thres_sl_tx_prioritization_r14_present) {
    HANDLE_CODE(unpack_integer(thres_sl_tx_prioritization_r14, bref, (uint8_t)1u, (uint8_t)8u));
  }
  if (anchor_carrier_freq_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(anchor_carrier_freq_list_r14, bref, 1, 8, integer_packer<uint32_t>(0, 262143)));
  }
  if (offset_dfn_r14_present) {
    HANDLE_CODE(unpack_integer(offset_dfn_r14, bref, (uint16_t)0u, (uint16_t)1000u));
  }
  if (cbr_common_tx_cfg_list_r14_present) {
    HANDLE_CODE(cbr_common_tx_cfg_list_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sl_v2x_cfg_common_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (v2x_comm_rx_pool_r14_present) {
    j.start_array("v2x-CommRxPool-r14");
    for (const auto& e1 : v2x_comm_rx_pool_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_comm_tx_pool_normal_common_r14_present) {
    j.start_array("v2x-CommTxPoolNormalCommon-r14");
    for (const auto& e1 : v2x_comm_tx_pool_normal_common_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (p2x_comm_tx_pool_normal_common_r14_present) {
    j.start_array("p2x-CommTxPoolNormalCommon-r14");
    for (const auto& e1 : p2x_comm_tx_pool_normal_common_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_comm_tx_pool_exceptional_r14_present) {
    j.write_fieldname("v2x-CommTxPoolExceptional-r14");
    v2x_comm_tx_pool_exceptional_r14.to_json(j);
  }
  if (v2x_sync_cfg_r14_present) {
    j.start_array("v2x-SyncConfig-r14");
    for (const auto& e1 : v2x_sync_cfg_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_inter_freq_info_list_r14_present) {
    j.start_array("v2x-InterFreqInfoList-r14");
    for (const auto& e1 : v2x_inter_freq_info_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_res_sel_cfg_r14_present) {
    j.write_fieldname("v2x-ResourceSelectionConfig-r14");
    v2x_res_sel_cfg_r14.to_json(j);
  }
  if (zone_cfg_r14_present) {
    j.write_fieldname("zoneConfig-r14");
    zone_cfg_r14.to_json(j);
  }
  if (type_tx_sync_r14_present) {
    j.write_str("typeTxSync-r14", type_tx_sync_r14.to_string());
  }
  if (thres_sl_tx_prioritization_r14_present) {
    j.write_int("thresSL-TxPrioritization-r14", thres_sl_tx_prioritization_r14);
  }
  if (anchor_carrier_freq_list_r14_present) {
    j.start_array("anchorCarrierFreqList-r14");
    for (const auto& e1 : anchor_carrier_freq_list_r14) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (offset_dfn_r14_present) {
    j.write_int("offsetDFN-r14", offset_dfn_r14);
  }
  if (cbr_common_tx_cfg_list_r14_present) {
    j.write_fieldname("cbr-CommonTxConfigList-r14");
    cbr_common_tx_cfg_list_r14.to_json(j);
  }
  j.end_obj();
}

// SL-V2X-PacketDuplicationConfig-r15 ::= SEQUENCE
SRSASN_CODE sl_v2x_packet_dupl_cfg_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(allowed_carrier_freq_cfg_r15_present, 1));

  HANDLE_CODE(pack_integer(bref, thresh_sl_reliability_r15, (uint8_t)1u, (uint8_t)8u));
  if (allowed_carrier_freq_cfg_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, allowed_carrier_freq_cfg_r15, 1, 16));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sl_v2x_packet_dupl_cfg_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(allowed_carrier_freq_cfg_r15_present, 1));

  HANDLE_CODE(unpack_integer(thresh_sl_reliability_r15, bref, (uint8_t)1u, (uint8_t)8u));
  if (allowed_carrier_freq_cfg_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(allowed_carrier_freq_cfg_r15, bref, 1, 16));
  }

  return SRSASN_SUCCESS;
}
void sl_v2x_packet_dupl_cfg_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("threshSL-Reliability-r15", thresh_sl_reliability_r15);
  if (allowed_carrier_freq_cfg_r15_present) {
    j.start_array("allowedCarrierFreqConfig-r15");
    for (const auto& e1 : allowed_carrier_freq_cfg_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// SystemInformationBlockType1-v1130-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v1130_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(tdd_cfg_v1130_present, 1));
  HANDLE_CODE(bref.pack(cell_sel_info_v1130_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (tdd_cfg_v1130_present) {
    HANDLE_CODE(tdd_cfg_v1130.pack(bref));
  }
  if (cell_sel_info_v1130_present) {
    HANDLE_CODE(cell_sel_info_v1130.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v1130_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(tdd_cfg_v1130_present, 1));
  HANDLE_CODE(bref.unpack(cell_sel_info_v1130_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (tdd_cfg_v1130_present) {
    HANDLE_CODE(tdd_cfg_v1130.unpack(bref));
  }
  if (cell_sel_info_v1130_present) {
    HANDLE_CODE(cell_sel_info_v1130.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v1130_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (tdd_cfg_v1130_present) {
    j.write_fieldname("tdd-Config-v1130");
    tdd_cfg_v1130.to_json(j);
  }
  if (cell_sel_info_v1130_present) {
    j.write_fieldname("cellSelectionInfo-v1130");
    cell_sel_info_v1130.to_json(j);
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType1-v9e0-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v9e0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_ind_v9e0_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v9e0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_band_ind_v9e0_present) {
    HANDLE_CODE(pack_integer(bref, freq_band_ind_v9e0, (uint16_t)65u, (uint16_t)256u));
  }
  if (multi_band_info_list_v9e0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v9e0, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v9e0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_ind_v9e0_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v9e0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_band_ind_v9e0_present) {
    HANDLE_CODE(unpack_integer(freq_band_ind_v9e0, bref, (uint16_t)65u, (uint16_t)256u));
  }
  if (multi_band_info_list_v9e0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v9e0, bref, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v9e0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_ind_v9e0_present) {
    j.write_int("freqBandIndicator-v9e0", freq_band_ind_v9e0);
  }
  if (multi_band_info_list_v9e0_present) {
    j.start_array("multiBandInfoList-v9e0");
    for (const auto& e1 : multi_band_info_list_v9e0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType2-v8h0-IEs ::= SEQUENCE
SRSASN_CODE sib_type2_v8h0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(multi_band_info_list_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list, 1, 8, integer_packer<uint8_t>(1, 32)));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type2_v8h0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(multi_band_info_list_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list, bref, 1, 8, integer_packer<uint8_t>(1, 32)));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type2_v8h0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (multi_band_info_list_present) {
    j.start_array("multiBandInfoList");
    for (const auto& e1 : multi_band_info_list) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType3-v10j0-IEs ::= SEQUENCE
SRSASN_CODE sib_type3_v10j0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(freq_band_info_r10_present, 1));
  HANDLE_CODE(bref.pack(multi_band_info_list_v10j0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (freq_band_info_r10_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, freq_band_info_r10, 1, 8));
  }
  if (multi_band_info_list_v10j0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_v10j0, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type3_v10j0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(freq_band_info_r10_present, 1));
  HANDLE_CODE(bref.unpack(multi_band_info_list_v10j0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (freq_band_info_r10_present) {
    HANDLE_CODE(unpack_dyn_seq_of(freq_band_info_r10, bref, 1, 8));
  }
  if (multi_band_info_list_v10j0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_v10j0, bref, 1, 8, SeqOfPacker<Packer>(1, 8, Packer())));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type3_v10j0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (freq_band_info_r10_present) {
    j.start_array("freqBandInfo-r10");
    for (const auto& e1 : freq_band_info_r10) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (multi_band_info_list_v10j0_present) {
    j.start_array("multiBandInfoList-v10j0");
    for (const auto& e1 : multi_band_info_list_v10j0) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType5-v8h0-IEs ::= SEQUENCE
SRSASN_CODE sib_type5_v8h0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v8h0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v8h0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_carrier_freq_list_v8h0, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type5_v8h0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v8h0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (inter_freq_carrier_freq_list_v8h0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(inter_freq_carrier_freq_list_v8h0, bref, 1, 8));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type5_v8h0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (inter_freq_carrier_freq_list_v8h0_present) {
    j.start_array("interFreqCarrierFreqList-v8h0");
    for (const auto& e1 : inter_freq_carrier_freq_list_v8h0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType6-v8h0-IEs ::= SEQUENCE
SRSASN_CODE sib_type6_v8h0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(carrier_freq_list_utra_fdd_v8h0_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (carrier_freq_list_utra_fdd_v8h0_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, carrier_freq_list_utra_fdd_v8h0, 1, 16));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type6_v8h0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(carrier_freq_list_utra_fdd_v8h0_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (carrier_freq_list_utra_fdd_v8h0_present) {
    HANDLE_CODE(unpack_dyn_seq_of(carrier_freq_list_utra_fdd_v8h0, bref, 1, 16));
  }

  return SRSASN_SUCCESS;
}
void sib_type6_v8h0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (carrier_freq_list_utra_fdd_v8h0_present) {
    j.start_array("carrierFreqListUTRA-FDD-v8h0");
    for (const auto& e1 : carrier_freq_list_utra_fdd_v8h0) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// TimeReferenceInfo-r15 ::= SEQUENCE
SRSASN_CODE time_ref_info_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(uncertainty_r15_present, 1));
  HANDLE_CODE(bref.pack(time_info_type_r15_present, 1));
  HANDLE_CODE(bref.pack(ref_sfn_r15_present, 1));

  HANDLE_CODE(time_r15.pack(bref));
  if (uncertainty_r15_present) {
    HANDLE_CODE(pack_integer(bref, uncertainty_r15, (uint8_t)0u, (uint8_t)12u));
  }
  if (ref_sfn_r15_present) {
    HANDLE_CODE(pack_integer(bref, ref_sfn_r15, (uint16_t)0u, (uint16_t)1023u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE time_ref_info_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(uncertainty_r15_present, 1));
  HANDLE_CODE(bref.unpack(time_info_type_r15_present, 1));
  HANDLE_CODE(bref.unpack(ref_sfn_r15_present, 1));

  HANDLE_CODE(time_r15.unpack(bref));
  if (uncertainty_r15_present) {
    HANDLE_CODE(unpack_integer(uncertainty_r15, bref, (uint8_t)0u, (uint8_t)12u));
  }
  if (ref_sfn_r15_present) {
    HANDLE_CODE(unpack_integer(ref_sfn_r15, bref, (uint16_t)0u, (uint16_t)1023u));
  }

  return SRSASN_SUCCESS;
}
void time_ref_info_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("time-r15");
  time_r15.to_json(j);
  if (uncertainty_r15_present) {
    j.write_int("uncertainty-r15", uncertainty_r15);
  }
  if (time_info_type_r15_present) {
    j.write_str("timeInfoType-r15", "localClock");
  }
  if (ref_sfn_r15_present) {
    j.write_int("referenceSFN-r15", ref_sfn_r15);
  }
  j.end_obj();
}

// UE-TimersAndConstants ::= SEQUENCE
SRSASN_CODE ue_timers_and_consts_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(t300.pack(bref));
  HANDLE_CODE(t301.pack(bref));
  HANDLE_CODE(t310.pack(bref));
  HANDLE_CODE(n310.pack(bref));
  HANDLE_CODE(t311.pack(bref));
  HANDLE_CODE(n311.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= t300_v1310_present;
    group_flags[0] |= t301_v1310_present;
    group_flags[1] |= t310_v1330_present;
    group_flags[2] |= t300_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(t300_v1310_present, 1));
      HANDLE_CODE(bref.pack(t301_v1310_present, 1));
      if (t300_v1310_present) {
        HANDLE_CODE(t300_v1310.pack(bref));
      }
      if (t301_v1310_present) {
        HANDLE_CODE(t301_v1310.pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(t310_v1330_present, 1));
      if (t310_v1330_present) {
        HANDLE_CODE(t310_v1330.pack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(t300_r15_present, 1));
      if (t300_r15_present) {
        HANDLE_CODE(t300_r15.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_timers_and_consts_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(t300.unpack(bref));
  HANDLE_CODE(t301.unpack(bref));
  HANDLE_CODE(t310.unpack(bref));
  HANDLE_CODE(n310.unpack(bref));
  HANDLE_CODE(t311.unpack(bref));
  HANDLE_CODE(n311.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(3);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(t300_v1310_present, 1));
      HANDLE_CODE(bref.unpack(t301_v1310_present, 1));
      if (t300_v1310_present) {
        HANDLE_CODE(t300_v1310.unpack(bref));
      }
      if (t301_v1310_present) {
        HANDLE_CODE(t301_v1310.unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(t310_v1330_present, 1));
      if (t310_v1330_present) {
        HANDLE_CODE(t310_v1330.unpack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(t300_r15_present, 1));
      if (t300_r15_present) {
        HANDLE_CODE(t300_r15.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void ue_timers_and_consts_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("t300", t300.to_string());
  j.write_str("t301", t301.to_string());
  j.write_str("t310", t310.to_string());
  j.write_str("n310", n310.to_string());
  j.write_str("t311", t311.to_string());
  j.write_str("n311", n311.to_string());
  if (ext) {
    if (t300_v1310_present) {
      j.write_str("t300-v1310", t300_v1310.to_string());
    }
    if (t301_v1310_present) {
      j.write_str("t301-v1310", t301_v1310.to_string());
    }
    if (t310_v1330_present) {
      j.write_str("t310-v1330", t310_v1330.to_string());
    }
    if (t300_r15_present) {
      j.write_str("t300-r15", t300_r15.to_string());
    }
  }
  j.end_obj();
}

// WLAN-OffloadInfoPerPLMN-r12 ::= SEQUENCE
SRSASN_CODE wlan_offload_info_per_plmn_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(wlan_offload_cfg_common_r12_present, 1));
  HANDLE_CODE(bref.pack(wlan_id_list_r12_present, 1));

  if (wlan_offload_cfg_common_r12_present) {
    HANDLE_CODE(wlan_offload_cfg_common_r12.pack(bref));
  }
  if (wlan_id_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, wlan_id_list_r12, 1, 16));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE wlan_offload_info_per_plmn_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(wlan_offload_cfg_common_r12_present, 1));
  HANDLE_CODE(bref.unpack(wlan_id_list_r12_present, 1));

  if (wlan_offload_cfg_common_r12_present) {
    HANDLE_CODE(wlan_offload_cfg_common_r12.unpack(bref));
  }
  if (wlan_id_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(wlan_id_list_r12, bref, 1, 16));
  }

  return SRSASN_SUCCESS;
}
void wlan_offload_info_per_plmn_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (wlan_offload_cfg_common_r12_present) {
    j.write_fieldname("wlan-OffloadConfigCommon-r12");
    wlan_offload_cfg_common_r12.to_json(j);
  }
  if (wlan_id_list_r12_present) {
    j.start_array("wlan-Id-List-r12");
    for (const auto& e1 : wlan_id_list_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// SchedulingInfo ::= SEQUENCE
SRSASN_CODE sched_info_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(si_periodicity.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, sib_map_info, 0, 31));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sched_info_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(si_periodicity.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(sib_map_info, bref, 0, 31));

  return SRSASN_SUCCESS;
}
void sched_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("si-Periodicity", si_periodicity.to_string());
  j.start_array("sib-MappingInfo");
  for (const auto& e1 : sib_map_info) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.end_obj();
}

// SystemInformation-v8a0-IEs ::= SEQUENCE
SRSASN_CODE sys_info_v8a0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_info_v8a0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sys_info_v8a0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// SystemInformationBlockPos-r15 ::= SEQUENCE
SRSASN_CODE sib_pos_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  HANDLE_CODE(assist_data_sib_elem_r15.pack(bref));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_pos_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  HANDLE_CODE(assist_data_sib_elem_r15.unpack(bref));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_pos_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("assistanceDataSIB-Element-r15", assist_data_sib_elem_r15.to_string());
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  j.end_obj();
}

// SystemInformationBlockType1-v8h0-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v8h0_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(multi_band_info_list_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list, 1, 8, integer_packer<uint8_t>(1, 64)));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v8h0_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(multi_band_info_list_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (multi_band_info_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list, bref, 1, 8, integer_packer<uint8_t>(1, 64)));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v8h0_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (multi_band_info_list_present) {
    j.start_array("multiBandInfoList");
    for (const auto& e1 : multi_band_info_list) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType1-v920-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v920_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ims_emergency_support_r9_present, 1));
  HANDLE_CODE(bref.pack(cell_sel_info_v920_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (cell_sel_info_v920_present) {
    HANDLE_CODE(cell_sel_info_v920.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v920_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ims_emergency_support_r9_present, 1));
  HANDLE_CODE(bref.unpack(cell_sel_info_v920_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (cell_sel_info_v920_present) {
    HANDLE_CODE(cell_sel_info_v920.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v920_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ims_emergency_support_r9_present) {
    j.write_str("ims-EmergencySupport-r9", "true");
  }
  if (cell_sel_info_v920_present) {
    j.write_fieldname("cellSelectionInfo-v920");
    cell_sel_info_v920.to_json(j);
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType10 ::= SEQUENCE
SRSASN_CODE sib_type10_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dummy_present, 1));

  HANDLE_CODE(msg_id.pack(bref));
  HANDLE_CODE(serial_num.pack(bref));
  HANDLE_CODE(warning_type.pack(bref));
  if (dummy_present) {
    HANDLE_CODE(dummy.pack(bref));
  }

  if (ext) {
    HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.pack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type10_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dummy_present, 1));

  HANDLE_CODE(msg_id.unpack(bref));
  HANDLE_CODE(serial_num.unpack(bref));
  HANDLE_CODE(warning_type.unpack(bref));
  if (dummy_present) {
    HANDLE_CODE(dummy.unpack(bref));
  }

  if (ext) {
    HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.unpack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type10_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("messageIdentifier", msg_id.to_string());
  j.write_str("serialNumber", serial_num.to_string());
  j.write_str("warningType", warning_type.to_string());
  if (dummy_present) {
    j.write_str("dummy", dummy.to_string());
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
  }
  j.end_obj();
}

// SystemInformationBlockType11 ::= SEQUENCE
SRSASN_CODE sib_type11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(data_coding_scheme_present, 1));

  HANDLE_CODE(msg_id.pack(bref));
  HANDLE_CODE(serial_num.pack(bref));
  HANDLE_CODE(warning_msg_segment_type.pack(bref));
  HANDLE_CODE(pack_integer(bref, warning_msg_segment_num, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(warning_msg_segment.pack(bref));
  if (data_coding_scheme_present) {
    HANDLE_CODE(data_coding_scheme.pack(bref));
  }

  if (ext) {
    HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.pack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(data_coding_scheme_present, 1));

  HANDLE_CODE(msg_id.unpack(bref));
  HANDLE_CODE(serial_num.unpack(bref));
  HANDLE_CODE(warning_msg_segment_type.unpack(bref));
  HANDLE_CODE(unpack_integer(warning_msg_segment_num, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(warning_msg_segment.unpack(bref));
  if (data_coding_scheme_present) {
    HANDLE_CODE(data_coding_scheme.unpack(bref));
  }

  if (ext) {
    HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.unpack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("messageIdentifier", msg_id.to_string());
  j.write_str("serialNumber", serial_num.to_string());
  j.write_str("warningMessageSegmentType", warning_msg_segment_type.to_string());
  j.write_int("warningMessageSegmentNumber", warning_msg_segment_num);
  j.write_str("warningMessageSegment", warning_msg_segment.to_string());
  if (data_coding_scheme_present) {
    j.write_str("dataCodingScheme", data_coding_scheme.to_string());
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
  }
  j.end_obj();
}

// SystemInformationBlockType12-r9 ::= SEQUENCE
SRSASN_CODE sib_type12_r9_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(data_coding_scheme_r9_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  HANDLE_CODE(msg_id_r9.pack(bref));
  HANDLE_CODE(serial_num_r9.pack(bref));
  HANDLE_CODE(warning_msg_segment_type_r9.pack(bref));
  HANDLE_CODE(pack_integer(bref, warning_msg_segment_num_r9, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(warning_msg_segment_r9.pack(bref));
  if (data_coding_scheme_r9_present) {
    HANDLE_CODE(data_coding_scheme_r9.pack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= warning_area_coordinates_segment_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(warning_area_coordinates_segment_r15_present, 1));
      if (warning_area_coordinates_segment_r15_present) {
        HANDLE_CODE(warning_area_coordinates_segment_r15.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type12_r9_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(data_coding_scheme_r9_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  HANDLE_CODE(msg_id_r9.unpack(bref));
  HANDLE_CODE(serial_num_r9.unpack(bref));
  HANDLE_CODE(warning_msg_segment_type_r9.unpack(bref));
  HANDLE_CODE(unpack_integer(warning_msg_segment_num_r9, bref, (uint8_t)0u, (uint8_t)63u));
  HANDLE_CODE(warning_msg_segment_r9.unpack(bref));
  if (data_coding_scheme_r9_present) {
    HANDLE_CODE(data_coding_scheme_r9.unpack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(warning_area_coordinates_segment_r15_present, 1));
      if (warning_area_coordinates_segment_r15_present) {
        HANDLE_CODE(warning_area_coordinates_segment_r15.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type12_r9_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("messageIdentifier-r9", msg_id_r9.to_string());
  j.write_str("serialNumber-r9", serial_num_r9.to_string());
  j.write_str("warningMessageSegmentType-r9", warning_msg_segment_type_r9.to_string());
  j.write_int("warningMessageSegmentNumber-r9", warning_msg_segment_num_r9);
  j.write_str("warningMessageSegment-r9", warning_msg_segment_r9.to_string());
  if (data_coding_scheme_r9_present) {
    j.write_str("dataCodingScheme-r9", data_coding_scheme_r9.to_string());
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (warning_area_coordinates_segment_r15_present) {
      j.write_str("warningAreaCoordinatesSegment-r15", warning_area_coordinates_segment_r15.to_string());
    }
  }
  j.end_obj();
}

// SystemInformationBlockType13-r9 ::= SEQUENCE
SRSASN_CODE sib_type13_r9_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_area_info_list_r9, 1, 8));
  HANDLE_CODE(notif_cfg_r9.pack(bref));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= notif_cfg_v1430.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(notif_cfg_v1430.is_present(), 1));
      if (notif_cfg_v1430.is_present()) {
        HANDLE_CODE(notif_cfg_v1430->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type13_r9_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(mbsfn_area_info_list_r9, bref, 1, 8));
  HANDLE_CODE(notif_cfg_r9.unpack(bref));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool notif_cfg_v1430_present;
      HANDLE_CODE(bref.unpack(notif_cfg_v1430_present, 1));
      notif_cfg_v1430.set_present(notif_cfg_v1430_present);
      if (notif_cfg_v1430.is_present()) {
        HANDLE_CODE(notif_cfg_v1430->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type13_r9_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("mbsfn-AreaInfoList-r9");
  for (const auto& e1 : mbsfn_area_info_list_r9) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_fieldname("notificationConfig-r9");
  notif_cfg_r9.to_json(j);
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (notif_cfg_v1430.is_present()) {
      j.write_fieldname("notificationConfig-v1430");
      notif_cfg_v1430->to_json(j);
    }
  }
  j.end_obj();
}

// SystemInformationBlockType14-r11 ::= SEQUENCE
SRSASN_CODE sib_type14_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(eab_param_r11_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (eab_param_r11_present) {
    HANDLE_CODE(eab_param_r11.pack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= eab_per_rsrp_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(eab_per_rsrp_r15_present, 1));
      if (eab_per_rsrp_r15_present) {
        HANDLE_CODE(eab_per_rsrp_r15.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type14_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(eab_param_r11_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (eab_param_r11_present) {
    HANDLE_CODE(eab_param_r11.unpack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(eab_per_rsrp_r15_present, 1));
      if (eab_per_rsrp_r15_present) {
        HANDLE_CODE(eab_per_rsrp_r15.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type14_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (eab_param_r11_present) {
    j.write_fieldname("eab-Param-r11");
    eab_param_r11.to_json(j);
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (eab_per_rsrp_r15_present) {
      j.write_str("eab-PerRSRP-r15", eab_per_rsrp_r15.to_string());
    }
  }
  j.end_obj();
}

void sib_type14_r11_s::eab_param_r11_c_::destroy_()
{
  switch (type_) {
    case types::eab_common_r11:
      c.destroy<eab_cfg_r11_s>();
      break;
    case types::eab_per_plmn_list_r11:
      c.destroy<eab_per_plmn_list_r11_l_>();
      break;
    default:
      break;
  }
}
void sib_type14_r11_s::eab_param_r11_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eab_common_r11:
      c.init<eab_cfg_r11_s>();
      break;
    case types::eab_per_plmn_list_r11:
      c.init<eab_per_plmn_list_r11_l_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type14_r11_s::eab_param_r11_c_");
  }
}
sib_type14_r11_s::eab_param_r11_c_::eab_param_r11_c_(const sib_type14_r11_s::eab_param_r11_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eab_common_r11:
      c.init(other.c.get<eab_cfg_r11_s>());
      break;
    case types::eab_per_plmn_list_r11:
      c.init(other.c.get<eab_per_plmn_list_r11_l_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type14_r11_s::eab_param_r11_c_");
  }
}
sib_type14_r11_s::eab_param_r11_c_& sib_type14_r11_s::eab_param_r11_c_::
                                    operator=(const sib_type14_r11_s::eab_param_r11_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eab_common_r11:
      c.set(other.c.get<eab_cfg_r11_s>());
      break;
    case types::eab_per_plmn_list_r11:
      c.set(other.c.get<eab_per_plmn_list_r11_l_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type14_r11_s::eab_param_r11_c_");
  }

  return *this;
}
void sib_type14_r11_s::eab_param_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eab_common_r11:
      j.write_fieldname("eab-Common-r11");
      c.get<eab_cfg_r11_s>().to_json(j);
      break;
    case types::eab_per_plmn_list_r11:
      j.start_array("eab-PerPLMN-List-r11");
      for (const auto& e1 : c.get<eab_per_plmn_list_r11_l_>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "sib_type14_r11_s::eab_param_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE sib_type14_r11_s::eab_param_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eab_common_r11:
      HANDLE_CODE(c.get<eab_cfg_r11_s>().pack(bref));
      break;
    case types::eab_per_plmn_list_r11:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<eab_per_plmn_list_r11_l_>(), 1, 6));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type14_r11_s::eab_param_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type14_r11_s::eab_param_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eab_common_r11:
      HANDLE_CODE(c.get<eab_cfg_r11_s>().unpack(bref));
      break;
    case types::eab_per_plmn_list_r11:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<eab_per_plmn_list_r11_l_>(), bref, 1, 6));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type14_r11_s::eab_param_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType15-r11 ::= SEQUENCE
SRSASN_CODE sib_type15_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbms_sai_intra_freq_r11_present, 1));
  HANDLE_CODE(bref.pack(mbms_sai_inter_freq_list_r11_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (mbms_sai_intra_freq_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbms_sai_intra_freq_r11, 1, 64, integer_packer<uint32_t>(0, 65535)));
  }
  if (mbms_sai_inter_freq_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbms_sai_inter_freq_list_r11, 1, 8));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= mbms_sai_inter_freq_list_v1140.is_present();
    group_flags[1] |= mbms_intra_freq_carrier_type_r14.is_present();
    group_flags[1] |= mbms_inter_freq_carrier_type_list_r14.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbms_sai_inter_freq_list_v1140.is_present(), 1));
      if (mbms_sai_inter_freq_list_v1140.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *mbms_sai_inter_freq_list_v1140, 1, 8));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbms_intra_freq_carrier_type_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(mbms_inter_freq_carrier_type_list_r14.is_present(), 1));
      if (mbms_intra_freq_carrier_type_r14.is_present()) {
        HANDLE_CODE(mbms_intra_freq_carrier_type_r14->pack(bref));
      }
      if (mbms_inter_freq_carrier_type_list_r14.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *mbms_inter_freq_carrier_type_list_r14, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type15_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbms_sai_intra_freq_r11_present, 1));
  HANDLE_CODE(bref.unpack(mbms_sai_inter_freq_list_r11_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (mbms_sai_intra_freq_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbms_sai_intra_freq_r11, bref, 1, 64, integer_packer<uint32_t>(0, 65535)));
  }
  if (mbms_sai_inter_freq_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbms_sai_inter_freq_list_r11, bref, 1, 8));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbms_sai_inter_freq_list_v1140_present;
      HANDLE_CODE(bref.unpack(mbms_sai_inter_freq_list_v1140_present, 1));
      mbms_sai_inter_freq_list_v1140.set_present(mbms_sai_inter_freq_list_v1140_present);
      if (mbms_sai_inter_freq_list_v1140.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*mbms_sai_inter_freq_list_v1140, bref, 1, 8));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbms_intra_freq_carrier_type_r14_present;
      HANDLE_CODE(bref.unpack(mbms_intra_freq_carrier_type_r14_present, 1));
      mbms_intra_freq_carrier_type_r14.set_present(mbms_intra_freq_carrier_type_r14_present);
      bool mbms_inter_freq_carrier_type_list_r14_present;
      HANDLE_CODE(bref.unpack(mbms_inter_freq_carrier_type_list_r14_present, 1));
      mbms_inter_freq_carrier_type_list_r14.set_present(mbms_inter_freq_carrier_type_list_r14_present);
      if (mbms_intra_freq_carrier_type_r14.is_present()) {
        HANDLE_CODE(mbms_intra_freq_carrier_type_r14->unpack(bref));
      }
      if (mbms_inter_freq_carrier_type_list_r14.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*mbms_inter_freq_carrier_type_list_r14, bref, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type15_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (mbms_sai_intra_freq_r11_present) {
    j.start_array("mbms-SAI-IntraFreq-r11");
    for (const auto& e1 : mbms_sai_intra_freq_r11) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (mbms_sai_inter_freq_list_r11_present) {
    j.start_array("mbms-SAI-InterFreqList-r11");
    for (const auto& e1 : mbms_sai_inter_freq_list_r11) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (mbms_sai_inter_freq_list_v1140.is_present()) {
      j.start_array("mbms-SAI-InterFreqList-v1140");
      for (const auto& e1 : *mbms_sai_inter_freq_list_v1140) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (mbms_intra_freq_carrier_type_r14.is_present()) {
      j.write_fieldname("mbms-IntraFreqCarrierType-r14");
      mbms_intra_freq_carrier_type_r14->to_json(j);
    }
    if (mbms_inter_freq_carrier_type_list_r14.is_present()) {
      j.start_array("mbms-InterFreqCarrierTypeList-r14");
      for (const auto& e1 : *mbms_inter_freq_carrier_type_list_r14) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// SystemInformationBlockType16-r11 ::= SEQUENCE
SRSASN_CODE sib_type16_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_info_r11_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (time_info_r11_present) {
    HANDLE_CODE(bref.pack(time_info_r11.day_light_saving_time_r11_present, 1));
    HANDLE_CODE(bref.pack(time_info_r11.leap_seconds_r11_present, 1));
    HANDLE_CODE(bref.pack(time_info_r11.local_time_offset_r11_present, 1));
    HANDLE_CODE(pack_integer(bref, time_info_r11.time_info_utc_r11, (uint64_t)0u, (uint64_t)549755813887u));
    if (time_info_r11.day_light_saving_time_r11_present) {
      HANDLE_CODE(time_info_r11.day_light_saving_time_r11.pack(bref));
    }
    if (time_info_r11.leap_seconds_r11_present) {
      HANDLE_CODE(pack_integer(bref, time_info_r11.leap_seconds_r11, (int16_t)-127, (int16_t)128));
    }
    if (time_info_r11.local_time_offset_r11_present) {
      HANDLE_CODE(pack_integer(bref, time_info_r11.local_time_offset_r11, (int8_t)-63, (int8_t)64));
    }
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= time_ref_info_r15.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(time_ref_info_r15.is_present(), 1));
      if (time_ref_info_r15.is_present()) {
        HANDLE_CODE(time_ref_info_r15->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type16_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_info_r11_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (time_info_r11_present) {
    HANDLE_CODE(bref.unpack(time_info_r11.day_light_saving_time_r11_present, 1));
    HANDLE_CODE(bref.unpack(time_info_r11.leap_seconds_r11_present, 1));
    HANDLE_CODE(bref.unpack(time_info_r11.local_time_offset_r11_present, 1));
    HANDLE_CODE(unpack_integer(time_info_r11.time_info_utc_r11, bref, (uint64_t)0u, (uint64_t)549755813887u));
    if (time_info_r11.day_light_saving_time_r11_present) {
      HANDLE_CODE(time_info_r11.day_light_saving_time_r11.unpack(bref));
    }
    if (time_info_r11.leap_seconds_r11_present) {
      HANDLE_CODE(unpack_integer(time_info_r11.leap_seconds_r11, bref, (int16_t)-127, (int16_t)128));
    }
    if (time_info_r11.local_time_offset_r11_present) {
      HANDLE_CODE(unpack_integer(time_info_r11.local_time_offset_r11, bref, (int8_t)-63, (int8_t)64));
    }
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool time_ref_info_r15_present;
      HANDLE_CODE(bref.unpack(time_ref_info_r15_present, 1));
      time_ref_info_r15.set_present(time_ref_info_r15_present);
      if (time_ref_info_r15.is_present()) {
        HANDLE_CODE(time_ref_info_r15->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type16_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (time_info_r11_present) {
    j.write_fieldname("timeInfo-r11");
    j.start_obj();
    j.write_int("timeInfoUTC-r11", time_info_r11.time_info_utc_r11);
    if (time_info_r11.day_light_saving_time_r11_present) {
      j.write_str("dayLightSavingTime-r11", time_info_r11.day_light_saving_time_r11.to_string());
    }
    if (time_info_r11.leap_seconds_r11_present) {
      j.write_int("leapSeconds-r11", time_info_r11.leap_seconds_r11);
    }
    if (time_info_r11.local_time_offset_r11_present) {
      j.write_int("localTimeOffset-r11", time_info_r11.local_time_offset_r11);
    }
    j.end_obj();
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (time_ref_info_r15.is_present()) {
      j.write_fieldname("timeReferenceInfo-r15");
      time_ref_info_r15->to_json(j);
    }
  }
  j.end_obj();
}

// SystemInformationBlockType17-r12 ::= SEQUENCE
SRSASN_CODE sib_type17_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(wlan_offload_info_per_plmn_list_r12_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (wlan_offload_info_per_plmn_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, wlan_offload_info_per_plmn_list_r12, 1, 6));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type17_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(wlan_offload_info_per_plmn_list_r12_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (wlan_offload_info_per_plmn_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(wlan_offload_info_per_plmn_list_r12, bref, 1, 6));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type17_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (wlan_offload_info_per_plmn_list_r12_present) {
    j.start_array("wlan-OffloadInfoPerPLMN-List-r12");
    for (const auto& e1 : wlan_offload_info_per_plmn_list_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  j.end_obj();
}

// SystemInformationBlockType18-r12 ::= SEQUENCE
SRSASN_CODE sib_type18_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(comm_cfg_r12_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (comm_cfg_r12_present) {
    HANDLE_CODE(bref.pack(comm_cfg_r12.comm_tx_pool_normal_common_r12_present, 1));
    HANDLE_CODE(bref.pack(comm_cfg_r12.comm_tx_pool_exceptional_r12_present, 1));
    HANDLE_CODE(bref.pack(comm_cfg_r12.comm_sync_cfg_r12_present, 1));
    HANDLE_CODE(pack_dyn_seq_of(bref, comm_cfg_r12.comm_rx_pool_r12, 1, 16));
    if (comm_cfg_r12.comm_tx_pool_normal_common_r12_present) {
      HANDLE_CODE(pack_dyn_seq_of(bref, comm_cfg_r12.comm_tx_pool_normal_common_r12, 1, 4));
    }
    if (comm_cfg_r12.comm_tx_pool_exceptional_r12_present) {
      HANDLE_CODE(pack_dyn_seq_of(bref, comm_cfg_r12.comm_tx_pool_exceptional_r12, 1, 4));
    }
    if (comm_cfg_r12.comm_sync_cfg_r12_present) {
      HANDLE_CODE(pack_dyn_seq_of(bref, comm_cfg_r12.comm_sync_cfg_r12, 1, 16));
    }
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= comm_tx_pool_normal_common_ext_r13.is_present();
    group_flags[0] |= comm_tx_res_uc_req_allowed_r13_present;
    group_flags[0] |= comm_tx_allow_relay_common_r13_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(comm_tx_pool_normal_common_ext_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(comm_tx_res_uc_req_allowed_r13_present, 1));
      HANDLE_CODE(bref.pack(comm_tx_allow_relay_common_r13_present, 1));
      if (comm_tx_pool_normal_common_ext_r13.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *comm_tx_pool_normal_common_ext_r13, 1, 4));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type18_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(comm_cfg_r12_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (comm_cfg_r12_present) {
    HANDLE_CODE(bref.unpack(comm_cfg_r12.comm_tx_pool_normal_common_r12_present, 1));
    HANDLE_CODE(bref.unpack(comm_cfg_r12.comm_tx_pool_exceptional_r12_present, 1));
    HANDLE_CODE(bref.unpack(comm_cfg_r12.comm_sync_cfg_r12_present, 1));
    HANDLE_CODE(unpack_dyn_seq_of(comm_cfg_r12.comm_rx_pool_r12, bref, 1, 16));
    if (comm_cfg_r12.comm_tx_pool_normal_common_r12_present) {
      HANDLE_CODE(unpack_dyn_seq_of(comm_cfg_r12.comm_tx_pool_normal_common_r12, bref, 1, 4));
    }
    if (comm_cfg_r12.comm_tx_pool_exceptional_r12_present) {
      HANDLE_CODE(unpack_dyn_seq_of(comm_cfg_r12.comm_tx_pool_exceptional_r12, bref, 1, 4));
    }
    if (comm_cfg_r12.comm_sync_cfg_r12_present) {
      HANDLE_CODE(unpack_dyn_seq_of(comm_cfg_r12.comm_sync_cfg_r12, bref, 1, 16));
    }
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool comm_tx_pool_normal_common_ext_r13_present;
      HANDLE_CODE(bref.unpack(comm_tx_pool_normal_common_ext_r13_present, 1));
      comm_tx_pool_normal_common_ext_r13.set_present(comm_tx_pool_normal_common_ext_r13_present);
      HANDLE_CODE(bref.unpack(comm_tx_res_uc_req_allowed_r13_present, 1));
      HANDLE_CODE(bref.unpack(comm_tx_allow_relay_common_r13_present, 1));
      if (comm_tx_pool_normal_common_ext_r13.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*comm_tx_pool_normal_common_ext_r13, bref, 1, 4));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type18_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (comm_cfg_r12_present) {
    j.write_fieldname("commConfig-r12");
    j.start_obj();
    j.start_array("commRxPool-r12");
    for (const auto& e1 : comm_cfg_r12.comm_rx_pool_r12) {
      e1.to_json(j);
    }
    j.end_array();
    if (comm_cfg_r12.comm_tx_pool_normal_common_r12_present) {
      j.start_array("commTxPoolNormalCommon-r12");
      for (const auto& e1 : comm_cfg_r12.comm_tx_pool_normal_common_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (comm_cfg_r12.comm_tx_pool_exceptional_r12_present) {
      j.start_array("commTxPoolExceptional-r12");
      for (const auto& e1 : comm_cfg_r12.comm_tx_pool_exceptional_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (comm_cfg_r12.comm_sync_cfg_r12_present) {
      j.start_array("commSyncConfig-r12");
      for (const auto& e1 : comm_cfg_r12.comm_sync_cfg_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    j.end_obj();
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (comm_tx_pool_normal_common_ext_r13.is_present()) {
      j.start_array("commTxPoolNormalCommonExt-r13");
      for (const auto& e1 : *comm_tx_pool_normal_common_ext_r13) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (comm_tx_res_uc_req_allowed_r13_present) {
      j.write_str("commTxResourceUC-ReqAllowed-r13", "true");
    }
    if (comm_tx_allow_relay_common_r13_present) {
      j.write_str("commTxAllowRelayCommon-r13", "true");
    }
  }
  j.end_obj();
}

// SystemInformationBlockType19-r12 ::= SEQUENCE
SRSASN_CODE sib_type19_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(disc_cfg_r12_present, 1));
  HANDLE_CODE(bref.pack(disc_inter_freq_list_r12_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (disc_cfg_r12_present) {
    HANDLE_CODE(bref.pack(disc_cfg_r12.disc_tx_pool_common_r12_present, 1));
    HANDLE_CODE(bref.pack(disc_cfg_r12.disc_tx_pwr_info_r12_present, 1));
    HANDLE_CODE(bref.pack(disc_cfg_r12.disc_sync_cfg_r12_present, 1));
    HANDLE_CODE(pack_dyn_seq_of(bref, disc_cfg_r12.disc_rx_pool_r12, 1, 16));
    if (disc_cfg_r12.disc_tx_pool_common_r12_present) {
      HANDLE_CODE(pack_dyn_seq_of(bref, disc_cfg_r12.disc_tx_pool_common_r12, 1, 4));
    }
    if (disc_cfg_r12.disc_tx_pwr_info_r12_present) {
      HANDLE_CODE(
          pack_fixed_seq_of(bref, &(disc_cfg_r12.disc_tx_pwr_info_r12)[0], disc_cfg_r12.disc_tx_pwr_info_r12.size()));
    }
    if (disc_cfg_r12.disc_sync_cfg_r12_present) {
      HANDLE_CODE(pack_dyn_seq_of(bref, disc_cfg_r12.disc_sync_cfg_r12, 1, 16));
    }
  }
  if (disc_inter_freq_list_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, disc_inter_freq_list_r12, 1, 8));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= disc_cfg_v1310.is_present();
    group_flags[0] |= disc_cfg_relay_r13.is_present();
    group_flags[0] |= disc_cfg_ps_minus13.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(disc_cfg_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(disc_cfg_relay_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(disc_cfg_ps_minus13.is_present(), 1));
      if (disc_cfg_v1310.is_present()) {
        HANDLE_CODE(bref.pack(disc_cfg_v1310->disc_inter_freq_list_v1310_present, 1));
        HANDLE_CODE(bref.pack(disc_cfg_v1310->gap_requests_allowed_common_present, 1));
        if (disc_cfg_v1310->disc_inter_freq_list_v1310_present) {
          HANDLE_CODE(pack_dyn_seq_of(bref, disc_cfg_v1310->disc_inter_freq_list_v1310, 1, 8));
        }
      }
      if (disc_cfg_relay_r13.is_present()) {
        HANDLE_CODE(disc_cfg_relay_r13->relay_ue_cfg_r13.pack(bref));
        HANDLE_CODE(disc_cfg_relay_r13->remote_ue_cfg_r13.pack(bref));
      }
      if (disc_cfg_ps_minus13.is_present()) {
        HANDLE_CODE(bref.pack(disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13_present, 1));
        HANDLE_CODE(pack_dyn_seq_of(bref, disc_cfg_ps_minus13->disc_rx_pool_ps_r13, 1, 16));
        if (disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13_present) {
          HANDLE_CODE(pack_dyn_seq_of(bref, disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13, 1, 4));
        }
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type19_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(disc_cfg_r12_present, 1));
  HANDLE_CODE(bref.unpack(disc_inter_freq_list_r12_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (disc_cfg_r12_present) {
    HANDLE_CODE(bref.unpack(disc_cfg_r12.disc_tx_pool_common_r12_present, 1));
    HANDLE_CODE(bref.unpack(disc_cfg_r12.disc_tx_pwr_info_r12_present, 1));
    HANDLE_CODE(bref.unpack(disc_cfg_r12.disc_sync_cfg_r12_present, 1));
    HANDLE_CODE(unpack_dyn_seq_of(disc_cfg_r12.disc_rx_pool_r12, bref, 1, 16));
    if (disc_cfg_r12.disc_tx_pool_common_r12_present) {
      HANDLE_CODE(unpack_dyn_seq_of(disc_cfg_r12.disc_tx_pool_common_r12, bref, 1, 4));
    }
    if (disc_cfg_r12.disc_tx_pwr_info_r12_present) {
      HANDLE_CODE(
          unpack_fixed_seq_of(&(disc_cfg_r12.disc_tx_pwr_info_r12)[0], bref, disc_cfg_r12.disc_tx_pwr_info_r12.size()));
    }
    if (disc_cfg_r12.disc_sync_cfg_r12_present) {
      HANDLE_CODE(unpack_dyn_seq_of(disc_cfg_r12.disc_sync_cfg_r12, bref, 1, 16));
    }
  }
  if (disc_inter_freq_list_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(disc_inter_freq_list_r12, bref, 1, 8));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool disc_cfg_v1310_present;
      HANDLE_CODE(bref.unpack(disc_cfg_v1310_present, 1));
      disc_cfg_v1310.set_present(disc_cfg_v1310_present);
      bool disc_cfg_relay_r13_present;
      HANDLE_CODE(bref.unpack(disc_cfg_relay_r13_present, 1));
      disc_cfg_relay_r13.set_present(disc_cfg_relay_r13_present);
      bool disc_cfg_ps_minus13_present;
      HANDLE_CODE(bref.unpack(disc_cfg_ps_minus13_present, 1));
      disc_cfg_ps_minus13.set_present(disc_cfg_ps_minus13_present);
      if (disc_cfg_v1310.is_present()) {
        HANDLE_CODE(bref.unpack(disc_cfg_v1310->disc_inter_freq_list_v1310_present, 1));
        HANDLE_CODE(bref.unpack(disc_cfg_v1310->gap_requests_allowed_common_present, 1));
        if (disc_cfg_v1310->disc_inter_freq_list_v1310_present) {
          HANDLE_CODE(unpack_dyn_seq_of(disc_cfg_v1310->disc_inter_freq_list_v1310, bref, 1, 8));
        }
      }
      if (disc_cfg_relay_r13.is_present()) {
        HANDLE_CODE(disc_cfg_relay_r13->relay_ue_cfg_r13.unpack(bref));
        HANDLE_CODE(disc_cfg_relay_r13->remote_ue_cfg_r13.unpack(bref));
      }
      if (disc_cfg_ps_minus13.is_present()) {
        HANDLE_CODE(bref.unpack(disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13_present, 1));
        HANDLE_CODE(unpack_dyn_seq_of(disc_cfg_ps_minus13->disc_rx_pool_ps_r13, bref, 1, 16));
        if (disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13_present) {
          HANDLE_CODE(unpack_dyn_seq_of(disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13, bref, 1, 4));
        }
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type19_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (disc_cfg_r12_present) {
    j.write_fieldname("discConfig-r12");
    j.start_obj();
    j.start_array("discRxPool-r12");
    for (const auto& e1 : disc_cfg_r12.disc_rx_pool_r12) {
      e1.to_json(j);
    }
    j.end_array();
    if (disc_cfg_r12.disc_tx_pool_common_r12_present) {
      j.start_array("discTxPoolCommon-r12");
      for (const auto& e1 : disc_cfg_r12.disc_tx_pool_common_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (disc_cfg_r12.disc_tx_pwr_info_r12_present) {
      j.start_array("discTxPowerInfo-r12");
      for (const auto& e1 : disc_cfg_r12.disc_tx_pwr_info_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (disc_cfg_r12.disc_sync_cfg_r12_present) {
      j.start_array("discSyncConfig-r12");
      for (const auto& e1 : disc_cfg_r12.disc_sync_cfg_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    j.end_obj();
  }
  if (disc_inter_freq_list_r12_present) {
    j.start_array("discInterFreqList-r12");
    for (const auto& e1 : disc_inter_freq_list_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (disc_cfg_v1310.is_present()) {
      j.write_fieldname("discConfig-v1310");
      j.start_obj();
      if (disc_cfg_v1310->disc_inter_freq_list_v1310_present) {
        j.start_array("discInterFreqList-v1310");
        for (const auto& e1 : disc_cfg_v1310->disc_inter_freq_list_v1310) {
          e1.to_json(j);
        }
        j.end_array();
      }
      if (disc_cfg_v1310->gap_requests_allowed_common_present) {
        j.write_str("gapRequestsAllowedCommon", "true");
      }
      j.end_obj();
    }
    if (disc_cfg_relay_r13.is_present()) {
      j.write_fieldname("discConfigRelay-r13");
      j.start_obj();
      j.write_fieldname("relayUE-Config-r13");
      disc_cfg_relay_r13->relay_ue_cfg_r13.to_json(j);
      j.write_fieldname("remoteUE-Config-r13");
      disc_cfg_relay_r13->remote_ue_cfg_r13.to_json(j);
      j.end_obj();
    }
    if (disc_cfg_ps_minus13.is_present()) {
      j.write_fieldname("discConfigPS-13");
      j.start_obj();
      j.start_array("discRxPoolPS-r13");
      for (const auto& e1 : disc_cfg_ps_minus13->disc_rx_pool_ps_r13) {
        e1.to_json(j);
      }
      j.end_array();
      if (disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13_present) {
        j.start_array("discTxPoolPS-Common-r13");
        for (const auto& e1 : disc_cfg_ps_minus13->disc_tx_pool_ps_common_r13) {
          e1.to_json(j);
        }
        j.end_array();
      }
      j.end_obj();
    }
  }
  j.end_obj();
}

// SystemInformationBlockType2 ::= SEQUENCE
SRSASN_CODE sib_type2_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ac_barr_info_present, 1));
  HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_present, 1));

  if (ac_barr_info_present) {
    HANDLE_CODE(bref.pack(ac_barr_info.ac_barr_for_mo_sig_present, 1));
    HANDLE_CODE(bref.pack(ac_barr_info.ac_barr_for_mo_data_present, 1));
    HANDLE_CODE(bref.pack(ac_barr_info.ac_barr_for_emergency, 1));
    if (ac_barr_info.ac_barr_for_mo_sig_present) {
      HANDLE_CODE(ac_barr_info.ac_barr_for_mo_sig.pack(bref));
    }
    if (ac_barr_info.ac_barr_for_mo_data_present) {
      HANDLE_CODE(ac_barr_info.ac_barr_for_mo_data.pack(bref));
    }
  }
  HANDLE_CODE(rr_cfg_common.pack(bref));
  HANDLE_CODE(ue_timers_and_consts.pack(bref));
  HANDLE_CODE(bref.pack(freq_info.ul_carrier_freq_present, 1));
  HANDLE_CODE(bref.pack(freq_info.ul_bw_present, 1));
  if (freq_info.ul_carrier_freq_present) {
    HANDLE_CODE(pack_integer(bref, freq_info.ul_carrier_freq, (uint32_t)0u, (uint32_t)65535u));
  }
  if (freq_info.ul_bw_present) {
    HANDLE_CODE(freq_info.ul_bw.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, freq_info.add_spec_emission, (uint8_t)1u, (uint8_t)32u));
  if (mbsfn_sf_cfg_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_sf_cfg_list, 1, 8));
  }
  HANDLE_CODE(time_align_timer_common.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= late_non_crit_ext_present;
    group_flags[1] |= ssac_barr_for_mmtel_voice_r9.is_present();
    group_flags[1] |= ssac_barr_for_mmtel_video_r9.is_present();
    group_flags[2] |= ac_barr_for_csfb_r10.is_present();
    group_flags[3] |= ac_barr_skip_for_mmtel_voice_r12_present;
    group_flags[3] |= ac_barr_skip_for_mmtel_video_r12_present;
    group_flags[3] |= ac_barr_skip_for_sms_r12_present;
    group_flags[3] |= ac_barr_per_plmn_list_r12.is_present();
    group_flags[4] |= voice_service_cause_ind_r12_present;
    group_flags[5] |= acdc_barr_for_common_r13.is_present();
    group_flags[5] |= acdc_barr_per_plmn_list_r13.is_present();
    group_flags[6] |= udt_restricting_for_common_r13.is_present();
    group_flags[6] |= udt_restricting_per_plmn_list_r13.is_present();
    group_flags[6] |= cio_t_eps_optim_info_r13.is_present();
    group_flags[6] |= use_full_resume_id_r13_present;
    group_flags[7] |= unicast_freq_hop_ind_r13_present;
    group_flags[8] |= mbsfn_sf_cfg_list_v1430.is_present();
    group_flags[8] |= video_service_cause_ind_r14_present;
    group_flags[9] |= plmn_info_list_r15.is_present();
    group_flags[10] |= cp_edt_r15_present;
    group_flags[10] |= up_edt_r15_present;
    group_flags[10] |= idle_mode_meass_r15_present;
    group_flags[10] |= reduced_cp_latency_enabled_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ssac_barr_for_mmtel_voice_r9.is_present(), 1));
      HANDLE_CODE(bref.pack(ssac_barr_for_mmtel_video_r9.is_present(), 1));
      if (ssac_barr_for_mmtel_voice_r9.is_present()) {
        HANDLE_CODE(ssac_barr_for_mmtel_voice_r9->pack(bref));
      }
      if (ssac_barr_for_mmtel_video_r9.is_present()) {
        HANDLE_CODE(ssac_barr_for_mmtel_video_r9->pack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ac_barr_for_csfb_r10.is_present(), 1));
      if (ac_barr_for_csfb_r10.is_present()) {
        HANDLE_CODE(ac_barr_for_csfb_r10->pack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ac_barr_skip_for_mmtel_voice_r12_present, 1));
      HANDLE_CODE(bref.pack(ac_barr_skip_for_mmtel_video_r12_present, 1));
      HANDLE_CODE(bref.pack(ac_barr_skip_for_sms_r12_present, 1));
      HANDLE_CODE(bref.pack(ac_barr_per_plmn_list_r12.is_present(), 1));
      if (ac_barr_per_plmn_list_r12.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *ac_barr_per_plmn_list_r12, 1, 6));
      }
    }
    if (group_flags[4]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(voice_service_cause_ind_r12_present, 1));
    }
    if (group_flags[5]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(acdc_barr_for_common_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(acdc_barr_per_plmn_list_r13.is_present(), 1));
      if (acdc_barr_for_common_r13.is_present()) {
        HANDLE_CODE(acdc_barr_for_common_r13->pack(bref));
      }
      if (acdc_barr_per_plmn_list_r13.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *acdc_barr_per_plmn_list_r13, 1, 6));
      }
    }
    if (group_flags[6]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(udt_restricting_for_common_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(udt_restricting_per_plmn_list_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(cio_t_eps_optim_info_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(use_full_resume_id_r13_present, 1));
      if (udt_restricting_for_common_r13.is_present()) {
        HANDLE_CODE(udt_restricting_for_common_r13->pack(bref));
      }
      if (udt_restricting_per_plmn_list_r13.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *udt_restricting_per_plmn_list_r13, 1, 6));
      }
      if (cio_t_eps_optim_info_r13.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *cio_t_eps_optim_info_r13, 1, 6));
      }
    }
    if (group_flags[7]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(unicast_freq_hop_ind_r13_present, 1));
    }
    if (group_flags[8]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_v1430.is_present(), 1));
      HANDLE_CODE(bref.pack(video_service_cause_ind_r14_present, 1));
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *mbsfn_sf_cfg_list_v1430, 1, 8));
      }
    }
    if (group_flags[9]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(plmn_info_list_r15.is_present(), 1));
      if (plmn_info_list_r15.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *plmn_info_list_r15, 1, 6));
      }
    }
    if (group_flags[10]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cp_edt_r15_present, 1));
      HANDLE_CODE(bref.pack(up_edt_r15_present, 1));
      HANDLE_CODE(bref.pack(idle_mode_meass_r15_present, 1));
      HANDLE_CODE(bref.pack(reduced_cp_latency_enabled_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type2_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ac_barr_info_present, 1));
  HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_present, 1));

  if (ac_barr_info_present) {
    HANDLE_CODE(bref.unpack(ac_barr_info.ac_barr_for_mo_sig_present, 1));
    HANDLE_CODE(bref.unpack(ac_barr_info.ac_barr_for_mo_data_present, 1));
    HANDLE_CODE(bref.unpack(ac_barr_info.ac_barr_for_emergency, 1));
    if (ac_barr_info.ac_barr_for_mo_sig_present) {
      HANDLE_CODE(ac_barr_info.ac_barr_for_mo_sig.unpack(bref));
    }
    if (ac_barr_info.ac_barr_for_mo_data_present) {
      HANDLE_CODE(ac_barr_info.ac_barr_for_mo_data.unpack(bref));
    }
  }
  HANDLE_CODE(rr_cfg_common.unpack(bref));
  HANDLE_CODE(ue_timers_and_consts.unpack(bref));
  HANDLE_CODE(bref.unpack(freq_info.ul_carrier_freq_present, 1));
  HANDLE_CODE(bref.unpack(freq_info.ul_bw_present, 1));
  if (freq_info.ul_carrier_freq_present) {
    HANDLE_CODE(unpack_integer(freq_info.ul_carrier_freq, bref, (uint32_t)0u, (uint32_t)65535u));
  }
  if (freq_info.ul_bw_present) {
    HANDLE_CODE(freq_info.ul_bw.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(freq_info.add_spec_emission, bref, (uint8_t)1u, (uint8_t)32u));
  if (mbsfn_sf_cfg_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbsfn_sf_cfg_list, bref, 1, 8));
  }
  HANDLE_CODE(time_align_timer_common.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(11);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool ssac_barr_for_mmtel_voice_r9_present;
      HANDLE_CODE(bref.unpack(ssac_barr_for_mmtel_voice_r9_present, 1));
      ssac_barr_for_mmtel_voice_r9.set_present(ssac_barr_for_mmtel_voice_r9_present);
      bool ssac_barr_for_mmtel_video_r9_present;
      HANDLE_CODE(bref.unpack(ssac_barr_for_mmtel_video_r9_present, 1));
      ssac_barr_for_mmtel_video_r9.set_present(ssac_barr_for_mmtel_video_r9_present);
      if (ssac_barr_for_mmtel_voice_r9.is_present()) {
        HANDLE_CODE(ssac_barr_for_mmtel_voice_r9->unpack(bref));
      }
      if (ssac_barr_for_mmtel_video_r9.is_present()) {
        HANDLE_CODE(ssac_barr_for_mmtel_video_r9->unpack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool ac_barr_for_csfb_r10_present;
      HANDLE_CODE(bref.unpack(ac_barr_for_csfb_r10_present, 1));
      ac_barr_for_csfb_r10.set_present(ac_barr_for_csfb_r10_present);
      if (ac_barr_for_csfb_r10.is_present()) {
        HANDLE_CODE(ac_barr_for_csfb_r10->unpack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(ac_barr_skip_for_mmtel_voice_r12_present, 1));
      HANDLE_CODE(bref.unpack(ac_barr_skip_for_mmtel_video_r12_present, 1));
      HANDLE_CODE(bref.unpack(ac_barr_skip_for_sms_r12_present, 1));
      bool ac_barr_per_plmn_list_r12_present;
      HANDLE_CODE(bref.unpack(ac_barr_per_plmn_list_r12_present, 1));
      ac_barr_per_plmn_list_r12.set_present(ac_barr_per_plmn_list_r12_present);
      if (ac_barr_per_plmn_list_r12.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*ac_barr_per_plmn_list_r12, bref, 1, 6));
      }
    }
    if (group_flags[4]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(voice_service_cause_ind_r12_present, 1));
    }
    if (group_flags[5]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool acdc_barr_for_common_r13_present;
      HANDLE_CODE(bref.unpack(acdc_barr_for_common_r13_present, 1));
      acdc_barr_for_common_r13.set_present(acdc_barr_for_common_r13_present);
      bool acdc_barr_per_plmn_list_r13_present;
      HANDLE_CODE(bref.unpack(acdc_barr_per_plmn_list_r13_present, 1));
      acdc_barr_per_plmn_list_r13.set_present(acdc_barr_per_plmn_list_r13_present);
      if (acdc_barr_for_common_r13.is_present()) {
        HANDLE_CODE(acdc_barr_for_common_r13->unpack(bref));
      }
      if (acdc_barr_per_plmn_list_r13.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*acdc_barr_per_plmn_list_r13, bref, 1, 6));
      }
    }
    if (group_flags[6]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool udt_restricting_for_common_r13_present;
      HANDLE_CODE(bref.unpack(udt_restricting_for_common_r13_present, 1));
      udt_restricting_for_common_r13.set_present(udt_restricting_for_common_r13_present);
      bool udt_restricting_per_plmn_list_r13_present;
      HANDLE_CODE(bref.unpack(udt_restricting_per_plmn_list_r13_present, 1));
      udt_restricting_per_plmn_list_r13.set_present(udt_restricting_per_plmn_list_r13_present);
      bool cio_t_eps_optim_info_r13_present;
      HANDLE_CODE(bref.unpack(cio_t_eps_optim_info_r13_present, 1));
      cio_t_eps_optim_info_r13.set_present(cio_t_eps_optim_info_r13_present);
      HANDLE_CODE(bref.unpack(use_full_resume_id_r13_present, 1));
      if (udt_restricting_for_common_r13.is_present()) {
        HANDLE_CODE(udt_restricting_for_common_r13->unpack(bref));
      }
      if (udt_restricting_per_plmn_list_r13.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*udt_restricting_per_plmn_list_r13, bref, 1, 6));
      }
      if (cio_t_eps_optim_info_r13.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*cio_t_eps_optim_info_r13, bref, 1, 6));
      }
    }
    if (group_flags[7]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(unicast_freq_hop_ind_r13_present, 1));
    }
    if (group_flags[8]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbsfn_sf_cfg_list_v1430_present;
      HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_v1430_present, 1));
      mbsfn_sf_cfg_list_v1430.set_present(mbsfn_sf_cfg_list_v1430_present);
      HANDLE_CODE(bref.unpack(video_service_cause_ind_r14_present, 1));
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*mbsfn_sf_cfg_list_v1430, bref, 1, 8));
      }
    }
    if (group_flags[9]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool plmn_info_list_r15_present;
      HANDLE_CODE(bref.unpack(plmn_info_list_r15_present, 1));
      plmn_info_list_r15.set_present(plmn_info_list_r15_present);
      if (plmn_info_list_r15.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*plmn_info_list_r15, bref, 1, 6));
      }
    }
    if (group_flags[10]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(cp_edt_r15_present, 1));
      HANDLE_CODE(bref.unpack(up_edt_r15_present, 1));
      HANDLE_CODE(bref.unpack(idle_mode_meass_r15_present, 1));
      HANDLE_CODE(bref.unpack(reduced_cp_latency_enabled_r15_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type2_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ac_barr_info_present) {
    j.write_fieldname("ac-BarringInfo");
    j.start_obj();
    j.write_bool("ac-BarringForEmergency", ac_barr_info.ac_barr_for_emergency);
    if (ac_barr_info.ac_barr_for_mo_sig_present) {
      j.write_fieldname("ac-BarringForMO-Signalling");
      ac_barr_info.ac_barr_for_mo_sig.to_json(j);
    }
    if (ac_barr_info.ac_barr_for_mo_data_present) {
      j.write_fieldname("ac-BarringForMO-Data");
      ac_barr_info.ac_barr_for_mo_data.to_json(j);
    }
    j.end_obj();
  }
  j.write_fieldname("radioResourceConfigCommon");
  rr_cfg_common.to_json(j);
  j.write_fieldname("ue-TimersAndConstants");
  ue_timers_and_consts.to_json(j);
  j.write_fieldname("freqInfo");
  j.start_obj();
  if (freq_info.ul_carrier_freq_present) {
    j.write_int("ul-CarrierFreq", freq_info.ul_carrier_freq);
  }
  if (freq_info.ul_bw_present) {
    j.write_str("ul-Bandwidth", freq_info.ul_bw.to_string());
  }
  j.write_int("additionalSpectrumEmission", freq_info.add_spec_emission);
  j.end_obj();
  if (mbsfn_sf_cfg_list_present) {
    j.start_array("mbsfn-SubframeConfigList");
    for (const auto& e1 : mbsfn_sf_cfg_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.write_str("timeAlignmentTimerCommon", time_align_timer_common.to_string());
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
    if (ssac_barr_for_mmtel_voice_r9.is_present()) {
      j.write_fieldname("ssac-BarringForMMTEL-Voice-r9");
      ssac_barr_for_mmtel_voice_r9->to_json(j);
    }
    if (ssac_barr_for_mmtel_video_r9.is_present()) {
      j.write_fieldname("ssac-BarringForMMTEL-Video-r9");
      ssac_barr_for_mmtel_video_r9->to_json(j);
    }
    if (ac_barr_for_csfb_r10.is_present()) {
      j.write_fieldname("ac-BarringForCSFB-r10");
      ac_barr_for_csfb_r10->to_json(j);
    }
    if (ac_barr_skip_for_mmtel_voice_r12_present) {
      j.write_str("ac-BarringSkipForMMTELVoice-r12", "true");
    }
    if (ac_barr_skip_for_mmtel_video_r12_present) {
      j.write_str("ac-BarringSkipForMMTELVideo-r12", "true");
    }
    if (ac_barr_skip_for_sms_r12_present) {
      j.write_str("ac-BarringSkipForSMS-r12", "true");
    }
    if (ac_barr_per_plmn_list_r12.is_present()) {
      j.start_array("ac-BarringPerPLMN-List-r12");
      for (const auto& e1 : *ac_barr_per_plmn_list_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (voice_service_cause_ind_r12_present) {
      j.write_str("voiceServiceCauseIndication-r12", "true");
    }
    if (acdc_barr_for_common_r13.is_present()) {
      j.write_fieldname("acdc-BarringForCommon-r13");
      acdc_barr_for_common_r13->to_json(j);
    }
    if (acdc_barr_per_plmn_list_r13.is_present()) {
      j.start_array("acdc-BarringPerPLMN-List-r13");
      for (const auto& e1 : *acdc_barr_per_plmn_list_r13) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (udt_restricting_for_common_r13.is_present()) {
      j.write_fieldname("udt-RestrictingForCommon-r13");
      udt_restricting_for_common_r13->to_json(j);
    }
    if (udt_restricting_per_plmn_list_r13.is_present()) {
      j.start_array("udt-RestrictingPerPLMN-List-r13");
      for (const auto& e1 : *udt_restricting_per_plmn_list_r13) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (cio_t_eps_optim_info_r13.is_present()) {
      j.start_array("cIoT-EPS-OptimisationInfo-r13");
      for (const auto& e1 : *cio_t_eps_optim_info_r13) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (use_full_resume_id_r13_present) {
      j.write_str("useFullResumeID-r13", "true");
    }
    if (unicast_freq_hop_ind_r13_present) {
      j.write_str("unicastFreqHoppingInd-r13", "true");
    }
    if (mbsfn_sf_cfg_list_v1430.is_present()) {
      j.start_array("mbsfn-SubframeConfigList-v1430");
      for (const auto& e1 : *mbsfn_sf_cfg_list_v1430) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (video_service_cause_ind_r14_present) {
      j.write_str("videoServiceCauseIndication-r14", "true");
    }
    if (plmn_info_list_r15.is_present()) {
      j.start_array("plmn-InfoList-r15");
      for (const auto& e1 : *plmn_info_list_r15) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (cp_edt_r15_present) {
      j.write_str("cp-EDT-r15", "true");
    }
    if (up_edt_r15_present) {
      j.write_str("up-EDT-r15", "true");
    }
    if (idle_mode_meass_r15_present) {
      j.write_str("idleModeMeasurements-r15", "true");
    }
    if (reduced_cp_latency_enabled_r15_present) {
      j.write_str("reducedCP-LatencyEnabled-r15", "true");
    }
  }
  j.end_obj();
}

// SystemInformationBlockType20-r13 ::= SEQUENCE
SRSASN_CODE sib_type20_r13_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(sc_mcch_dur_r13_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  HANDLE_CODE(sc_mcch_repeat_period_r13.pack(bref));
  HANDLE_CODE(pack_integer(bref, sc_mcch_offset_r13, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(pack_integer(bref, sc_mcch_first_sf_r13, (uint8_t)0u, (uint8_t)9u));
  if (sc_mcch_dur_r13_present) {
    HANDLE_CODE(pack_integer(bref, sc_mcch_dur_r13, (uint8_t)2u, (uint8_t)9u));
  }
  HANDLE_CODE(sc_mcch_mod_period_r13.pack(bref));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= br_bcch_cfg_r14.is_present();
    group_flags[0] |= sc_mcch_sched_info_r14.is_present();
    group_flags[0] |= pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present;
    group_flags[0] |= pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present;
    group_flags[1] |= sc_mcch_repeat_period_v1470_present;
    group_flags[1] |= sc_mcch_mod_period_v1470_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(br_bcch_cfg_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(sc_mcch_sched_info_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present, 1));
      HANDLE_CODE(bref.pack(pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present, 1));
      if (br_bcch_cfg_r14.is_present()) {
        HANDLE_CODE(pack_integer(bref, br_bcch_cfg_r14->mpdcch_nb_sc_mcch_r14, (uint8_t)1u, (uint8_t)16u));
        HANDLE_CODE(br_bcch_cfg_r14->mpdcch_num_repeat_sc_mcch_r14.pack(bref));
        HANDLE_CODE(br_bcch_cfg_r14->mpdcch_start_sf_sc_mcch_r14.pack(bref));
        HANDLE_CODE(br_bcch_cfg_r14->mpdcch_pdsch_hop_cfg_sc_mcch_r14.pack(bref));
        HANDLE_CODE(pack_integer(bref, br_bcch_cfg_r14->sc_mcch_carrier_freq_r14, (uint32_t)0u, (uint32_t)262143u));
        HANDLE_CODE(pack_integer(bref, br_bcch_cfg_r14->sc_mcch_offset_br_r14, (uint8_t)0u, (uint8_t)10u));
        HANDLE_CODE(br_bcch_cfg_r14->sc_mcch_repeat_period_br_r14.pack(bref));
        HANDLE_CODE(br_bcch_cfg_r14->sc_mcch_mod_period_br_r14.pack(bref));
      }
      if (sc_mcch_sched_info_r14.is_present()) {
        HANDLE_CODE(sc_mcch_sched_info_r14->pack(bref));
      }
      if (pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present) {
        HANDLE_CODE(pdsch_max_num_repeat_cemode_a_sc_mtch_r14.pack(bref));
      }
      if (pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present) {
        HANDLE_CODE(pdsch_max_num_repeat_cemode_b_sc_mtch_r14.pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sc_mcch_repeat_period_v1470_present, 1));
      HANDLE_CODE(bref.pack(sc_mcch_mod_period_v1470_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type20_r13_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(sc_mcch_dur_r13_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  HANDLE_CODE(sc_mcch_repeat_period_r13.unpack(bref));
  HANDLE_CODE(unpack_integer(sc_mcch_offset_r13, bref, (uint8_t)0u, (uint8_t)10u));
  HANDLE_CODE(unpack_integer(sc_mcch_first_sf_r13, bref, (uint8_t)0u, (uint8_t)9u));
  if (sc_mcch_dur_r13_present) {
    HANDLE_CODE(unpack_integer(sc_mcch_dur_r13, bref, (uint8_t)2u, (uint8_t)9u));
  }
  HANDLE_CODE(sc_mcch_mod_period_r13.unpack(bref));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool br_bcch_cfg_r14_present;
      HANDLE_CODE(bref.unpack(br_bcch_cfg_r14_present, 1));
      br_bcch_cfg_r14.set_present(br_bcch_cfg_r14_present);
      bool sc_mcch_sched_info_r14_present;
      HANDLE_CODE(bref.unpack(sc_mcch_sched_info_r14_present, 1));
      sc_mcch_sched_info_r14.set_present(sc_mcch_sched_info_r14_present);
      HANDLE_CODE(bref.unpack(pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present, 1));
      HANDLE_CODE(bref.unpack(pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present, 1));
      if (br_bcch_cfg_r14.is_present()) {
        HANDLE_CODE(unpack_integer(br_bcch_cfg_r14->mpdcch_nb_sc_mcch_r14, bref, (uint8_t)1u, (uint8_t)16u));
        HANDLE_CODE(br_bcch_cfg_r14->mpdcch_num_repeat_sc_mcch_r14.unpack(bref));
        HANDLE_CODE(br_bcch_cfg_r14->mpdcch_start_sf_sc_mcch_r14.unpack(bref));
        HANDLE_CODE(br_bcch_cfg_r14->mpdcch_pdsch_hop_cfg_sc_mcch_r14.unpack(bref));
        HANDLE_CODE(unpack_integer(br_bcch_cfg_r14->sc_mcch_carrier_freq_r14, bref, (uint32_t)0u, (uint32_t)262143u));
        HANDLE_CODE(unpack_integer(br_bcch_cfg_r14->sc_mcch_offset_br_r14, bref, (uint8_t)0u, (uint8_t)10u));
        HANDLE_CODE(br_bcch_cfg_r14->sc_mcch_repeat_period_br_r14.unpack(bref));
        HANDLE_CODE(br_bcch_cfg_r14->sc_mcch_mod_period_br_r14.unpack(bref));
      }
      if (sc_mcch_sched_info_r14.is_present()) {
        HANDLE_CODE(sc_mcch_sched_info_r14->unpack(bref));
      }
      if (pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present) {
        HANDLE_CODE(pdsch_max_num_repeat_cemode_a_sc_mtch_r14.unpack(bref));
      }
      if (pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present) {
        HANDLE_CODE(pdsch_max_num_repeat_cemode_b_sc_mtch_r14.unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(sc_mcch_repeat_period_v1470_present, 1));
      HANDLE_CODE(bref.unpack(sc_mcch_mod_period_v1470_present, 1));
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type20_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sc-mcch-RepetitionPeriod-r13", sc_mcch_repeat_period_r13.to_string());
  j.write_int("sc-mcch-Offset-r13", sc_mcch_offset_r13);
  j.write_int("sc-mcch-FirstSubframe-r13", sc_mcch_first_sf_r13);
  if (sc_mcch_dur_r13_present) {
    j.write_int("sc-mcch-duration-r13", sc_mcch_dur_r13);
  }
  j.write_str("sc-mcch-ModificationPeriod-r13", sc_mcch_mod_period_r13.to_string());
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (ext) {
    if (br_bcch_cfg_r14.is_present()) {
      j.write_fieldname("br-BCCH-Config-r14");
      j.start_obj();
      j.write_str("dummy", "rf1");
      j.write_str("dummy2", "rf1");
      j.write_int("mpdcch-Narrowband-SC-MCCH-r14", br_bcch_cfg_r14->mpdcch_nb_sc_mcch_r14);
      j.write_str("mpdcch-NumRepetition-SC-MCCH-r14", br_bcch_cfg_r14->mpdcch_num_repeat_sc_mcch_r14.to_string());
      j.write_fieldname("mpdcch-StartSF-SC-MCCH-r14");
      br_bcch_cfg_r14->mpdcch_start_sf_sc_mcch_r14.to_json(j);
      j.write_str("mpdcch-PDSCH-HoppingConfig-SC-MCCH-r14",
                  br_bcch_cfg_r14->mpdcch_pdsch_hop_cfg_sc_mcch_r14.to_string());
      j.write_int("sc-mcch-CarrierFreq-r14", br_bcch_cfg_r14->sc_mcch_carrier_freq_r14);
      j.write_int("sc-mcch-Offset-BR-r14", br_bcch_cfg_r14->sc_mcch_offset_br_r14);
      j.write_str("sc-mcch-RepetitionPeriod-BR-r14", br_bcch_cfg_r14->sc_mcch_repeat_period_br_r14.to_string());
      j.write_str("sc-mcch-ModificationPeriod-BR-r14", br_bcch_cfg_r14->sc_mcch_mod_period_br_r14.to_string());
      j.end_obj();
    }
    if (sc_mcch_sched_info_r14.is_present()) {
      j.write_fieldname("sc-mcch-SchedulingInfo-r14");
      sc_mcch_sched_info_r14->to_json(j);
    }
    if (pdsch_max_num_repeat_cemode_a_sc_mtch_r14_present) {
      j.write_str("pdsch-maxNumRepetitionCEmodeA-SC-MTCH-r14", pdsch_max_num_repeat_cemode_a_sc_mtch_r14.to_string());
    }
    if (pdsch_max_num_repeat_cemode_b_sc_mtch_r14_present) {
      j.write_str("pdsch-maxNumRepetitionCEmodeB-SC-MTCH-r14", pdsch_max_num_repeat_cemode_b_sc_mtch_r14.to_string());
    }
    if (sc_mcch_repeat_period_v1470_present) {
      j.write_str("sc-mcch-RepetitionPeriod-v1470", "rf1");
    }
    if (sc_mcch_mod_period_v1470_present) {
      j.write_str("sc-mcch-ModificationPeriod-v1470", "rf1");
    }
  }
  j.end_obj();
}

void sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::destroy_() {}
void sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::mpdcch_start_sf_sc_mcch_r14_c_(
    const sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::fdd_r14:
      c.init(other.c.get<fdd_r14_e_>());
      break;
    case types::tdd_r14:
      c.init(other.c.get<tdd_r14_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_");
  }
}
sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_&
sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::
operator=(const sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::fdd_r14:
      c.set(other.c.get<fdd_r14_e_>());
      break;
    case types::tdd_r14:
      c.set(other.c.get<tdd_r14_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_");
  }

  return *this;
}
void sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::fdd_r14:
      j.write_str("fdd-r14", c.get<fdd_r14_e_>().to_string());
      break;
    case types::tdd_r14:
      j.write_str("tdd-r14", c.get<tdd_r14_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_");
  }
  j.end_obj();
}
SRSASN_CODE sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::fdd_r14:
      HANDLE_CODE(c.get<fdd_r14_e_>().pack(bref));
      break;
    case types::tdd_r14:
      HANDLE_CODE(c.get<tdd_r14_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::fdd_r14:
      HANDLE_CODE(c.get<fdd_r14_e_>().unpack(bref));
      break;
    case types::tdd_r14:
      HANDLE_CODE(c.get<tdd_r14_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type20_r13_s::br_bcch_cfg_r14_s_::mpdcch_start_sf_sc_mcch_r14_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType21-r14 ::= SEQUENCE
SRSASN_CODE sib_type21_r14_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(sl_v2x_cfg_common_r14_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (sl_v2x_cfg_common_r14_present) {
    HANDLE_CODE(sl_v2x_cfg_common_r14.pack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type21_r14_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(sl_v2x_cfg_common_r14_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (sl_v2x_cfg_common_r14_present) {
    HANDLE_CODE(sl_v2x_cfg_common_r14.unpack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type21_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (sl_v2x_cfg_common_r14_present) {
    j.write_fieldname("sl-V2X-ConfigCommon-r14");
    sl_v2x_cfg_common_r14.to_json(j);
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  j.end_obj();
}

// SystemInformationBlockType24-r15 ::= SEQUENCE
SRSASN_CODE sib_type24_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(carrier_freq_list_nr_r15_present, 1));
  HANDLE_CODE(bref.pack(t_resel_nr_sf_r15_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (carrier_freq_list_nr_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, carrier_freq_list_nr_r15, 1, 8));
  }
  HANDLE_CODE(pack_integer(bref, t_resel_nr_r15, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_nr_sf_r15_present) {
    HANDLE_CODE(t_resel_nr_sf_r15.pack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type24_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(carrier_freq_list_nr_r15_present, 1));
  HANDLE_CODE(bref.unpack(t_resel_nr_sf_r15_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (carrier_freq_list_nr_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(carrier_freq_list_nr_r15, bref, 1, 8));
  }
  HANDLE_CODE(unpack_integer(t_resel_nr_r15, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_nr_sf_r15_present) {
    HANDLE_CODE(t_resel_nr_sf_r15.unpack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type24_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (carrier_freq_list_nr_r15_present) {
    j.start_array("carrierFreqListNR-r15");
    for (const auto& e1 : carrier_freq_list_nr_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.write_int("t-ReselectionNR-r15", t_resel_nr_r15);
  if (t_resel_nr_sf_r15_present) {
    j.write_fieldname("t-ReselectionNR-SF-r15");
    t_resel_nr_sf_r15.to_json(j);
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  j.end_obj();
}

// SystemInformationBlockType25-r15 ::= SEQUENCE
SRSASN_CODE sib_type25_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(uac_barr_for_common_r15_present, 1));
  HANDLE_CODE(bref.pack(uac_barr_per_plmn_list_r15_present, 1));
  HANDLE_CODE(bref.pack(uac_ac1_select_assist_info_r15_present, 1));
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

  if (uac_barr_for_common_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, uac_barr_for_common_r15, 1, 63));
  }
  if (uac_barr_per_plmn_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, uac_barr_per_plmn_list_r15, 1, 6));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, uac_barr_info_set_list_r15, 1, 8));
  if (uac_ac1_select_assist_info_r15_present) {
    HANDLE_CODE(uac_ac1_select_assist_info_r15.pack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type25_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(uac_barr_for_common_r15_present, 1));
  HANDLE_CODE(bref.unpack(uac_barr_per_plmn_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(uac_ac1_select_assist_info_r15_present, 1));
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

  if (uac_barr_for_common_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(uac_barr_for_common_r15, bref, 1, 63));
  }
  if (uac_barr_per_plmn_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(uac_barr_per_plmn_list_r15, bref, 1, 6));
  }
  HANDLE_CODE(unpack_dyn_seq_of(uac_barr_info_set_list_r15, bref, 1, 8));
  if (uac_ac1_select_assist_info_r15_present) {
    HANDLE_CODE(uac_ac1_select_assist_info_r15.unpack(bref));
  }
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type25_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (uac_barr_for_common_r15_present) {
    j.start_array("uac-BarringForCommon-r15");
    for (const auto& e1 : uac_barr_for_common_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (uac_barr_per_plmn_list_r15_present) {
    j.start_array("uac-BarringPerPLMN-List-r15");
    for (const auto& e1 : uac_barr_per_plmn_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.start_array("uac-BarringInfoSetList-r15");
  for (const auto& e1 : uac_barr_info_set_list_r15) {
    e1.to_json(j);
  }
  j.end_array();
  if (uac_ac1_select_assist_info_r15_present) {
    j.write_fieldname("uac-AC1-SelectAssistInfo-r15");
    uac_ac1_select_assist_info_r15.to_json(j);
  }
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  j.end_obj();
}

void sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::destroy_()
{
  switch (type_) {
    case types::individual_plmn_list_r15:
      c.destroy<individual_plmn_list_r15_l_>();
      break;
    default:
      break;
  }
}
void sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::plmn_common_r15:
      break;
    case types::individual_plmn_list_r15:
      c.init<individual_plmn_list_r15_l_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_");
  }
}
sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::uac_ac1_select_assist_info_r15_c_(
    const sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::plmn_common_r15:
      c.init(other.c.get<uac_ac1_select_assist_info_r15_e>());
      break;
    case types::individual_plmn_list_r15:
      c.init(other.c.get<individual_plmn_list_r15_l_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_");
  }
}
sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_& sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::
                                                     operator=(const sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::plmn_common_r15:
      c.set(other.c.get<uac_ac1_select_assist_info_r15_e>());
      break;
    case types::individual_plmn_list_r15:
      c.set(other.c.get<individual_plmn_list_r15_l_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_");
  }

  return *this;
}
void sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::plmn_common_r15:
      j.write_str("plmnCommon-r15", c.get<uac_ac1_select_assist_info_r15_e>().to_string());
      break;
    case types::individual_plmn_list_r15:
      j.start_array("individualPLMNList-r15");
      for (const auto& e1 : c.get<individual_plmn_list_r15_l_>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::plmn_common_r15:
      HANDLE_CODE(c.get<uac_ac1_select_assist_info_r15_e>().pack(bref));
      break;
    case types::individual_plmn_list_r15:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<individual_plmn_list_r15_l_>(), 2, 6));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::plmn_common_r15:
      HANDLE_CODE(c.get<uac_ac1_select_assist_info_r15_e>().unpack(bref));
      break;
    case types::individual_plmn_list_r15:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<individual_plmn_list_r15_l_>(), bref, 2, 6));
      break;
    default:
      log_invalid_choice_id(type_, "sib_type25_r15_s::uac_ac1_select_assist_info_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType26-r15 ::= SEQUENCE
SRSASN_CODE sib_type26_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(v2x_inter_freq_info_list_r15_present, 1));
  HANDLE_CODE(bref.pack(cbr_pssch_tx_cfg_list_r15_present, 1));
  HANDLE_CODE(bref.pack(v2x_packet_dupl_cfg_r15_present, 1));
  HANDLE_CODE(bref.pack(sync_freq_list_r15_present, 1));
  HANDLE_CODE(bref.pack(slss_tx_multi_freq_r15_present, 1));
  HANDLE_CODE(bref.pack(v2x_freq_sel_cfg_list_r15_present, 1));
  HANDLE_CODE(bref.pack(thresh_s_rssi_cbr_r14_present, 1));

  if (v2x_inter_freq_info_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_inter_freq_info_list_r15, 0, 7));
  }
  if (cbr_pssch_tx_cfg_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, cbr_pssch_tx_cfg_list_r15, 1, 8));
  }
  if (v2x_packet_dupl_cfg_r15_present) {
    HANDLE_CODE(v2x_packet_dupl_cfg_r15.pack(bref));
  }
  if (sync_freq_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, sync_freq_list_r15, 1, 8, integer_packer<uint32_t>(0, 262143)));
  }
  if (v2x_freq_sel_cfg_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, v2x_freq_sel_cfg_list_r15, 1, 8));
  }
  if (thresh_s_rssi_cbr_r14_present) {
    HANDLE_CODE(pack_integer(bref, thresh_s_rssi_cbr_r14, (uint8_t)0u, (uint8_t)45u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type26_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(v2x_inter_freq_info_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(cbr_pssch_tx_cfg_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(v2x_packet_dupl_cfg_r15_present, 1));
  HANDLE_CODE(bref.unpack(sync_freq_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(slss_tx_multi_freq_r15_present, 1));
  HANDLE_CODE(bref.unpack(v2x_freq_sel_cfg_list_r15_present, 1));
  HANDLE_CODE(bref.unpack(thresh_s_rssi_cbr_r14_present, 1));

  if (v2x_inter_freq_info_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_inter_freq_info_list_r15, bref, 0, 7));
  }
  if (cbr_pssch_tx_cfg_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(cbr_pssch_tx_cfg_list_r15, bref, 1, 8));
  }
  if (v2x_packet_dupl_cfg_r15_present) {
    HANDLE_CODE(v2x_packet_dupl_cfg_r15.unpack(bref));
  }
  if (sync_freq_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(sync_freq_list_r15, bref, 1, 8, integer_packer<uint32_t>(0, 262143)));
  }
  if (v2x_freq_sel_cfg_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(v2x_freq_sel_cfg_list_r15, bref, 1, 8));
  }
  if (thresh_s_rssi_cbr_r14_present) {
    HANDLE_CODE(unpack_integer(thresh_s_rssi_cbr_r14, bref, (uint8_t)0u, (uint8_t)45u));
  }

  return SRSASN_SUCCESS;
}
void sib_type26_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (v2x_inter_freq_info_list_r15_present) {
    j.start_array("v2x-InterFreqInfoList-r15");
    for (const auto& e1 : v2x_inter_freq_info_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (cbr_pssch_tx_cfg_list_r15_present) {
    j.start_array("cbr-pssch-TxConfigList-r15");
    for (const auto& e1 : cbr_pssch_tx_cfg_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (v2x_packet_dupl_cfg_r15_present) {
    j.write_fieldname("v2x-PacketDuplicationConfig-r15");
    v2x_packet_dupl_cfg_r15.to_json(j);
  }
  if (sync_freq_list_r15_present) {
    j.start_array("syncFreqList-r15");
    for (const auto& e1 : sync_freq_list_r15) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (slss_tx_multi_freq_r15_present) {
    j.write_str("slss-TxMultiFreq-r15", "true");
  }
  if (v2x_freq_sel_cfg_list_r15_present) {
    j.start_array("v2x-FreqSelectionConfigList-r15");
    for (const auto& e1 : v2x_freq_sel_cfg_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (thresh_s_rssi_cbr_r14_present) {
    j.write_int("threshS-RSSI-CBR-r14", thresh_s_rssi_cbr_r14);
  }
  j.end_obj();
}

// SystemInformationBlockType3 ::= SEQUENCE
SRSASN_CODE sib_type3_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_resel_info_common.speed_state_resel_pars_present, 1));
  HANDLE_CODE(cell_resel_info_common.q_hyst.pack(bref));
  if (cell_resel_info_common.speed_state_resel_pars_present) {
    HANDLE_CODE(cell_resel_info_common.speed_state_resel_pars.mob_state_params.pack(bref));
    HANDLE_CODE(cell_resel_info_common.speed_state_resel_pars.q_hyst_sf.sf_medium.pack(bref));
    HANDLE_CODE(cell_resel_info_common.speed_state_resel_pars.q_hyst_sf.sf_high.pack(bref));
  }
  HANDLE_CODE(bref.pack(cell_resel_serving_freq_info.s_non_intra_search_present, 1));
  if (cell_resel_serving_freq_info.s_non_intra_search_present) {
    HANDLE_CODE(pack_integer(bref, cell_resel_serving_freq_info.s_non_intra_search, (uint8_t)0u, (uint8_t)31u));
  }
  HANDLE_CODE(pack_integer(bref, cell_resel_serving_freq_info.thresh_serving_low, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, cell_resel_serving_freq_info.cell_resel_prio, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(bref.pack(intra_freq_cell_resel_info.p_max_present, 1));
  HANDLE_CODE(bref.pack(intra_freq_cell_resel_info.s_intra_search_present, 1));
  HANDLE_CODE(bref.pack(intra_freq_cell_resel_info.allowed_meas_bw_present, 1));
  HANDLE_CODE(bref.pack(intra_freq_cell_resel_info.t_resel_eutra_sf_present, 1));
  HANDLE_CODE(pack_integer(bref, intra_freq_cell_resel_info.q_rx_lev_min, (int8_t)-70, (int8_t)-22));
  if (intra_freq_cell_resel_info.p_max_present) {
    HANDLE_CODE(pack_integer(bref, intra_freq_cell_resel_info.p_max, (int8_t)-30, (int8_t)33));
  }
  if (intra_freq_cell_resel_info.s_intra_search_present) {
    HANDLE_CODE(pack_integer(bref, intra_freq_cell_resel_info.s_intra_search, (uint8_t)0u, (uint8_t)31u));
  }
  if (intra_freq_cell_resel_info.allowed_meas_bw_present) {
    HANDLE_CODE(intra_freq_cell_resel_info.allowed_meas_bw.pack(bref));
  }
  HANDLE_CODE(bref.pack(intra_freq_cell_resel_info.presence_ant_port1, 1));
  HANDLE_CODE(intra_freq_cell_resel_info.neigh_cell_cfg.pack(bref));
  HANDLE_CODE(pack_integer(bref, intra_freq_cell_resel_info.t_resel_eutra, (uint8_t)0u, (uint8_t)7u));
  if (intra_freq_cell_resel_info.t_resel_eutra_sf_present) {
    HANDLE_CODE(intra_freq_cell_resel_info.t_resel_eutra_sf.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= late_non_crit_ext_present;
    group_flags[1] |= s_intra_search_v920.is_present();
    group_flags[1] |= s_non_intra_search_v920.is_present();
    group_flags[1] |= q_qual_min_r9_present;
    group_flags[1] |= thresh_serving_low_q_r9_present;
    group_flags[2] |= q_qual_min_wb_r11_present;
    group_flags[3] |= q_qual_min_rsrq_on_all_symbols_r12_present;
    group_flags[4] |= cell_resel_serving_freq_info_v1310.is_present();
    group_flags[4] |= redist_serving_info_r13.is_present();
    group_flags[4] |= cell_sel_info_ce_r13.is_present();
    group_flags[4] |= t_resel_eutra_ce_r13_present;
    group_flags[5] |= cell_sel_info_ce1_r13.is_present();
    group_flags[6] |= cell_sel_info_ce1_v1360.is_present();
    group_flags[7] |= cell_resel_info_common_v1460.is_present();
    group_flags[8] |= cell_resel_info_hsdn_r15.is_present();
    group_flags[8] |= cell_sel_info_ce_v1530.is_present();
    group_flags[8] |= crs_intf_mitig_neigh_cells_ce_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(s_intra_search_v920.is_present(), 1));
      HANDLE_CODE(bref.pack(s_non_intra_search_v920.is_present(), 1));
      HANDLE_CODE(bref.pack(q_qual_min_r9_present, 1));
      HANDLE_CODE(bref.pack(thresh_serving_low_q_r9_present, 1));
      if (s_intra_search_v920.is_present()) {
        HANDLE_CODE(pack_integer(bref, s_intra_search_v920->s_intra_search_p_r9, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(pack_integer(bref, s_intra_search_v920->s_intra_search_q_r9, (uint8_t)0u, (uint8_t)31u));
      }
      if (s_non_intra_search_v920.is_present()) {
        HANDLE_CODE(pack_integer(bref, s_non_intra_search_v920->s_non_intra_search_p_r9, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(pack_integer(bref, s_non_intra_search_v920->s_non_intra_search_q_r9, (uint8_t)0u, (uint8_t)31u));
      }
      if (q_qual_min_r9_present) {
        HANDLE_CODE(pack_integer(bref, q_qual_min_r9, (int8_t)-34, (int8_t)-3));
      }
      if (thresh_serving_low_q_r9_present) {
        HANDLE_CODE(pack_integer(bref, thresh_serving_low_q_r9, (uint8_t)0u, (uint8_t)31u));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(q_qual_min_wb_r11_present, 1));
      if (q_qual_min_wb_r11_present) {
        HANDLE_CODE(pack_integer(bref, q_qual_min_wb_r11, (int8_t)-34, (int8_t)-3));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(q_qual_min_rsrq_on_all_symbols_r12_present, 1));
      if (q_qual_min_rsrq_on_all_symbols_r12_present) {
        HANDLE_CODE(pack_integer(bref, q_qual_min_rsrq_on_all_symbols_r12, (int8_t)-34, (int8_t)-3));
      }
    }
    if (group_flags[4]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cell_resel_serving_freq_info_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(redist_serving_info_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(cell_sel_info_ce_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(t_resel_eutra_ce_r13_present, 1));
      if (cell_resel_serving_freq_info_v1310.is_present()) {
        HANDLE_CODE(cell_resel_serving_freq_info_v1310->pack(bref));
      }
      if (redist_serving_info_r13.is_present()) {
        HANDLE_CODE(redist_serving_info_r13->pack(bref));
      }
      if (cell_sel_info_ce_r13.is_present()) {
        HANDLE_CODE(cell_sel_info_ce_r13->pack(bref));
      }
      if (t_resel_eutra_ce_r13_present) {
        HANDLE_CODE(pack_integer(bref, t_resel_eutra_ce_r13, (uint8_t)0u, (uint8_t)15u));
      }
    }
    if (group_flags[5]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cell_sel_info_ce1_r13.is_present(), 1));
      if (cell_sel_info_ce1_r13.is_present()) {
        HANDLE_CODE(cell_sel_info_ce1_r13->pack(bref));
      }
    }
    if (group_flags[6]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cell_sel_info_ce1_v1360.is_present(), 1));
      if (cell_sel_info_ce1_v1360.is_present()) {
        HANDLE_CODE(cell_sel_info_ce1_v1360->pack(bref));
      }
    }
    if (group_flags[7]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cell_resel_info_common_v1460.is_present(), 1));
      if (cell_resel_info_common_v1460.is_present()) {
        HANDLE_CODE(cell_resel_info_common_v1460->pack(bref));
      }
    }
    if (group_flags[8]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cell_resel_info_hsdn_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(cell_sel_info_ce_v1530.is_present(), 1));
      HANDLE_CODE(bref.pack(crs_intf_mitig_neigh_cells_ce_r15_present, 1));
      if (cell_resel_info_hsdn_r15.is_present()) {
        HANDLE_CODE(cell_resel_info_hsdn_r15->pack(bref));
      }
      if (cell_sel_info_ce_v1530.is_present()) {
        HANDLE_CODE(cell_sel_info_ce_v1530->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type3_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_resel_info_common.speed_state_resel_pars_present, 1));
  HANDLE_CODE(cell_resel_info_common.q_hyst.unpack(bref));
  if (cell_resel_info_common.speed_state_resel_pars_present) {
    HANDLE_CODE(cell_resel_info_common.speed_state_resel_pars.mob_state_params.unpack(bref));
    HANDLE_CODE(cell_resel_info_common.speed_state_resel_pars.q_hyst_sf.sf_medium.unpack(bref));
    HANDLE_CODE(cell_resel_info_common.speed_state_resel_pars.q_hyst_sf.sf_high.unpack(bref));
  }
  HANDLE_CODE(bref.unpack(cell_resel_serving_freq_info.s_non_intra_search_present, 1));
  if (cell_resel_serving_freq_info.s_non_intra_search_present) {
    HANDLE_CODE(unpack_integer(cell_resel_serving_freq_info.s_non_intra_search, bref, (uint8_t)0u, (uint8_t)31u));
  }
  HANDLE_CODE(unpack_integer(cell_resel_serving_freq_info.thresh_serving_low, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(cell_resel_serving_freq_info.cell_resel_prio, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(bref.unpack(intra_freq_cell_resel_info.p_max_present, 1));
  HANDLE_CODE(bref.unpack(intra_freq_cell_resel_info.s_intra_search_present, 1));
  HANDLE_CODE(bref.unpack(intra_freq_cell_resel_info.allowed_meas_bw_present, 1));
  HANDLE_CODE(bref.unpack(intra_freq_cell_resel_info.t_resel_eutra_sf_present, 1));
  HANDLE_CODE(unpack_integer(intra_freq_cell_resel_info.q_rx_lev_min, bref, (int8_t)-70, (int8_t)-22));
  if (intra_freq_cell_resel_info.p_max_present) {
    HANDLE_CODE(unpack_integer(intra_freq_cell_resel_info.p_max, bref, (int8_t)-30, (int8_t)33));
  }
  if (intra_freq_cell_resel_info.s_intra_search_present) {
    HANDLE_CODE(unpack_integer(intra_freq_cell_resel_info.s_intra_search, bref, (uint8_t)0u, (uint8_t)31u));
  }
  if (intra_freq_cell_resel_info.allowed_meas_bw_present) {
    HANDLE_CODE(intra_freq_cell_resel_info.allowed_meas_bw.unpack(bref));
  }
  HANDLE_CODE(bref.unpack(intra_freq_cell_resel_info.presence_ant_port1, 1));
  HANDLE_CODE(intra_freq_cell_resel_info.neigh_cell_cfg.unpack(bref));
  HANDLE_CODE(unpack_integer(intra_freq_cell_resel_info.t_resel_eutra, bref, (uint8_t)0u, (uint8_t)7u));
  if (intra_freq_cell_resel_info.t_resel_eutra_sf_present) {
    HANDLE_CODE(intra_freq_cell_resel_info.t_resel_eutra_sf.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(9);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool s_intra_search_v920_present;
      HANDLE_CODE(bref.unpack(s_intra_search_v920_present, 1));
      s_intra_search_v920.set_present(s_intra_search_v920_present);
      bool s_non_intra_search_v920_present;
      HANDLE_CODE(bref.unpack(s_non_intra_search_v920_present, 1));
      s_non_intra_search_v920.set_present(s_non_intra_search_v920_present);
      HANDLE_CODE(bref.unpack(q_qual_min_r9_present, 1));
      HANDLE_CODE(bref.unpack(thresh_serving_low_q_r9_present, 1));
      if (s_intra_search_v920.is_present()) {
        HANDLE_CODE(unpack_integer(s_intra_search_v920->s_intra_search_p_r9, bref, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(unpack_integer(s_intra_search_v920->s_intra_search_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
      }
      if (s_non_intra_search_v920.is_present()) {
        HANDLE_CODE(unpack_integer(s_non_intra_search_v920->s_non_intra_search_p_r9, bref, (uint8_t)0u, (uint8_t)31u));
        HANDLE_CODE(unpack_integer(s_non_intra_search_v920->s_non_intra_search_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
      }
      if (q_qual_min_r9_present) {
        HANDLE_CODE(unpack_integer(q_qual_min_r9, bref, (int8_t)-34, (int8_t)-3));
      }
      if (thresh_serving_low_q_r9_present) {
        HANDLE_CODE(unpack_integer(thresh_serving_low_q_r9, bref, (uint8_t)0u, (uint8_t)31u));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(q_qual_min_wb_r11_present, 1));
      if (q_qual_min_wb_r11_present) {
        HANDLE_CODE(unpack_integer(q_qual_min_wb_r11, bref, (int8_t)-34, (int8_t)-3));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(q_qual_min_rsrq_on_all_symbols_r12_present, 1));
      if (q_qual_min_rsrq_on_all_symbols_r12_present) {
        HANDLE_CODE(unpack_integer(q_qual_min_rsrq_on_all_symbols_r12, bref, (int8_t)-34, (int8_t)-3));
      }
    }
    if (group_flags[4]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cell_resel_serving_freq_info_v1310_present;
      HANDLE_CODE(bref.unpack(cell_resel_serving_freq_info_v1310_present, 1));
      cell_resel_serving_freq_info_v1310.set_present(cell_resel_serving_freq_info_v1310_present);
      bool redist_serving_info_r13_present;
      HANDLE_CODE(bref.unpack(redist_serving_info_r13_present, 1));
      redist_serving_info_r13.set_present(redist_serving_info_r13_present);
      bool cell_sel_info_ce_r13_present;
      HANDLE_CODE(bref.unpack(cell_sel_info_ce_r13_present, 1));
      cell_sel_info_ce_r13.set_present(cell_sel_info_ce_r13_present);
      HANDLE_CODE(bref.unpack(t_resel_eutra_ce_r13_present, 1));
      if (cell_resel_serving_freq_info_v1310.is_present()) {
        HANDLE_CODE(cell_resel_serving_freq_info_v1310->unpack(bref));
      }
      if (redist_serving_info_r13.is_present()) {
        HANDLE_CODE(redist_serving_info_r13->unpack(bref));
      }
      if (cell_sel_info_ce_r13.is_present()) {
        HANDLE_CODE(cell_sel_info_ce_r13->unpack(bref));
      }
      if (t_resel_eutra_ce_r13_present) {
        HANDLE_CODE(unpack_integer(t_resel_eutra_ce_r13, bref, (uint8_t)0u, (uint8_t)15u));
      }
    }
    if (group_flags[5]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cell_sel_info_ce1_r13_present;
      HANDLE_CODE(bref.unpack(cell_sel_info_ce1_r13_present, 1));
      cell_sel_info_ce1_r13.set_present(cell_sel_info_ce1_r13_present);
      if (cell_sel_info_ce1_r13.is_present()) {
        HANDLE_CODE(cell_sel_info_ce1_r13->unpack(bref));
      }
    }
    if (group_flags[6]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cell_sel_info_ce1_v1360_present;
      HANDLE_CODE(bref.unpack(cell_sel_info_ce1_v1360_present, 1));
      cell_sel_info_ce1_v1360.set_present(cell_sel_info_ce1_v1360_present);
      if (cell_sel_info_ce1_v1360.is_present()) {
        HANDLE_CODE(cell_sel_info_ce1_v1360->unpack(bref));
      }
    }
    if (group_flags[7]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cell_resel_info_common_v1460_present;
      HANDLE_CODE(bref.unpack(cell_resel_info_common_v1460_present, 1));
      cell_resel_info_common_v1460.set_present(cell_resel_info_common_v1460_present);
      if (cell_resel_info_common_v1460.is_present()) {
        HANDLE_CODE(cell_resel_info_common_v1460->unpack(bref));
      }
    }
    if (group_flags[8]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cell_resel_info_hsdn_r15_present;
      HANDLE_CODE(bref.unpack(cell_resel_info_hsdn_r15_present, 1));
      cell_resel_info_hsdn_r15.set_present(cell_resel_info_hsdn_r15_present);
      bool cell_sel_info_ce_v1530_present;
      HANDLE_CODE(bref.unpack(cell_sel_info_ce_v1530_present, 1));
      cell_sel_info_ce_v1530.set_present(cell_sel_info_ce_v1530_present);
      HANDLE_CODE(bref.unpack(crs_intf_mitig_neigh_cells_ce_r15_present, 1));
      if (cell_resel_info_hsdn_r15.is_present()) {
        HANDLE_CODE(cell_resel_info_hsdn_r15->unpack(bref));
      }
      if (cell_sel_info_ce_v1530.is_present()) {
        HANDLE_CODE(cell_sel_info_ce_v1530->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type3_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellReselectionInfoCommon");
  j.start_obj();
  j.write_str("q-Hyst", cell_resel_info_common.q_hyst.to_string());
  if (cell_resel_info_common.speed_state_resel_pars_present) {
    j.write_fieldname("speedStateReselectionPars");
    j.start_obj();
    j.write_fieldname("mobilityStateParameters");
    cell_resel_info_common.speed_state_resel_pars.mob_state_params.to_json(j);
    j.write_fieldname("q-HystSF");
    j.start_obj();
    j.write_str("sf-Medium", cell_resel_info_common.speed_state_resel_pars.q_hyst_sf.sf_medium.to_string());
    j.write_str("sf-High", cell_resel_info_common.speed_state_resel_pars.q_hyst_sf.sf_high.to_string());
    j.end_obj();
    j.end_obj();
  }
  j.end_obj();
  j.write_fieldname("cellReselectionServingFreqInfo");
  j.start_obj();
  if (cell_resel_serving_freq_info.s_non_intra_search_present) {
    j.write_int("s-NonIntraSearch", cell_resel_serving_freq_info.s_non_intra_search);
  }
  j.write_int("threshServingLow", cell_resel_serving_freq_info.thresh_serving_low);
  j.write_int("cellReselectionPriority", cell_resel_serving_freq_info.cell_resel_prio);
  j.end_obj();
  j.write_fieldname("intraFreqCellReselectionInfo");
  j.start_obj();
  j.write_int("q-RxLevMin", intra_freq_cell_resel_info.q_rx_lev_min);
  if (intra_freq_cell_resel_info.p_max_present) {
    j.write_int("p-Max", intra_freq_cell_resel_info.p_max);
  }
  if (intra_freq_cell_resel_info.s_intra_search_present) {
    j.write_int("s-IntraSearch", intra_freq_cell_resel_info.s_intra_search);
  }
  if (intra_freq_cell_resel_info.allowed_meas_bw_present) {
    j.write_str("allowedMeasBandwidth", intra_freq_cell_resel_info.allowed_meas_bw.to_string());
  }
  j.write_bool("presenceAntennaPort1", intra_freq_cell_resel_info.presence_ant_port1);
  j.write_str("neighCellConfig", intra_freq_cell_resel_info.neigh_cell_cfg.to_string());
  j.write_int("t-ReselectionEUTRA", intra_freq_cell_resel_info.t_resel_eutra);
  if (intra_freq_cell_resel_info.t_resel_eutra_sf_present) {
    j.write_fieldname("t-ReselectionEUTRA-SF");
    intra_freq_cell_resel_info.t_resel_eutra_sf.to_json(j);
  }
  j.end_obj();
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
    if (s_intra_search_v920.is_present()) {
      j.write_fieldname("s-IntraSearch-v920");
      j.start_obj();
      j.write_int("s-IntraSearchP-r9", s_intra_search_v920->s_intra_search_p_r9);
      j.write_int("s-IntraSearchQ-r9", s_intra_search_v920->s_intra_search_q_r9);
      j.end_obj();
    }
    if (s_non_intra_search_v920.is_present()) {
      j.write_fieldname("s-NonIntraSearch-v920");
      j.start_obj();
      j.write_int("s-NonIntraSearchP-r9", s_non_intra_search_v920->s_non_intra_search_p_r9);
      j.write_int("s-NonIntraSearchQ-r9", s_non_intra_search_v920->s_non_intra_search_q_r9);
      j.end_obj();
    }
    if (q_qual_min_r9_present) {
      j.write_int("q-QualMin-r9", q_qual_min_r9);
    }
    if (thresh_serving_low_q_r9_present) {
      j.write_int("threshServingLowQ-r9", thresh_serving_low_q_r9);
    }
    if (q_qual_min_wb_r11_present) {
      j.write_int("q-QualMinWB-r11", q_qual_min_wb_r11);
    }
    if (q_qual_min_rsrq_on_all_symbols_r12_present) {
      j.write_int("q-QualMinRSRQ-OnAllSymbols-r12", q_qual_min_rsrq_on_all_symbols_r12);
    }
    if (cell_resel_serving_freq_info_v1310.is_present()) {
      j.write_fieldname("cellReselectionServingFreqInfo-v1310");
      cell_resel_serving_freq_info_v1310->to_json(j);
    }
    if (redist_serving_info_r13.is_present()) {
      j.write_fieldname("redistributionServingInfo-r13");
      redist_serving_info_r13->to_json(j);
    }
    if (cell_sel_info_ce_r13.is_present()) {
      j.write_fieldname("cellSelectionInfoCE-r13");
      cell_sel_info_ce_r13->to_json(j);
    }
    if (t_resel_eutra_ce_r13_present) {
      j.write_int("t-ReselectionEUTRA-CE-r13", t_resel_eutra_ce_r13);
    }
    if (cell_sel_info_ce1_r13.is_present()) {
      j.write_fieldname("cellSelectionInfoCE1-r13");
      cell_sel_info_ce1_r13->to_json(j);
    }
    if (cell_sel_info_ce1_v1360.is_present()) {
      j.write_fieldname("cellSelectionInfoCE1-v1360");
      cell_sel_info_ce1_v1360->to_json(j);
    }
    if (cell_resel_info_common_v1460.is_present()) {
      j.write_fieldname("cellReselectionInfoCommon-v1460");
      cell_resel_info_common_v1460->to_json(j);
    }
    if (cell_resel_info_hsdn_r15.is_present()) {
      j.write_fieldname("cellReselectionInfoHSDN-r15");
      cell_resel_info_hsdn_r15->to_json(j);
    }
    if (cell_sel_info_ce_v1530.is_present()) {
      j.write_fieldname("cellSelectionInfoCE-v1530");
      cell_sel_info_ce_v1530->to_json(j);
    }
    if (crs_intf_mitig_neigh_cells_ce_r15_present) {
      j.write_str("crs-IntfMitigNeighCellsCE-r15", "enabled");
    }
  }
  j.end_obj();
}

// SystemInformationBlockType4 ::= SEQUENCE
SRSASN_CODE sib_type4_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(intra_freq_neigh_cell_list_present, 1));
  HANDLE_CODE(bref.pack(intra_freq_black_cell_list_present, 1));
  HANDLE_CODE(bref.pack(csg_pci_range_present, 1));

  if (intra_freq_neigh_cell_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, intra_freq_neigh_cell_list, 1, 16));
  }
  if (intra_freq_black_cell_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, intra_freq_black_cell_list, 1, 16));
  }
  if (csg_pci_range_present) {
    HANDLE_CODE(csg_pci_range.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= late_non_crit_ext_present;
    group_flags[1] |= intra_freq_neigh_hsdn_cell_list_r15.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(intra_freq_neigh_hsdn_cell_list_r15.is_present(), 1));
      if (intra_freq_neigh_hsdn_cell_list_r15.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *intra_freq_neigh_hsdn_cell_list_r15, 1, 16));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type4_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(intra_freq_neigh_cell_list_present, 1));
  HANDLE_CODE(bref.unpack(intra_freq_black_cell_list_present, 1));
  HANDLE_CODE(bref.unpack(csg_pci_range_present, 1));

  if (intra_freq_neigh_cell_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(intra_freq_neigh_cell_list, bref, 1, 16));
  }
  if (intra_freq_black_cell_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(intra_freq_black_cell_list, bref, 1, 16));
  }
  if (csg_pci_range_present) {
    HANDLE_CODE(csg_pci_range.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool intra_freq_neigh_hsdn_cell_list_r15_present;
      HANDLE_CODE(bref.unpack(intra_freq_neigh_hsdn_cell_list_r15_present, 1));
      intra_freq_neigh_hsdn_cell_list_r15.set_present(intra_freq_neigh_hsdn_cell_list_r15_present);
      if (intra_freq_neigh_hsdn_cell_list_r15.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*intra_freq_neigh_hsdn_cell_list_r15, bref, 1, 16));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type4_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (intra_freq_neigh_cell_list_present) {
    j.start_array("intraFreqNeighCellList");
    for (const auto& e1 : intra_freq_neigh_cell_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (intra_freq_black_cell_list_present) {
    j.start_array("intraFreqBlackCellList");
    for (const auto& e1 : intra_freq_black_cell_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (csg_pci_range_present) {
    j.write_fieldname("csg-PhysCellIdRange");
    csg_pci_range.to_json(j);
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
    if (intra_freq_neigh_hsdn_cell_list_r15.is_present()) {
      j.start_array("intraFreqNeighHSDN-CellList-r15");
      for (const auto& e1 : *intra_freq_neigh_hsdn_cell_list_r15) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// SystemInformationBlockType5 ::= SEQUENCE
SRSASN_CODE sib_type5_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, inter_freq_carrier_freq_list, 1, 8));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= late_non_crit_ext_present;
    group_flags[1] |= inter_freq_carrier_freq_list_v1250.is_present();
    group_flags[1] |= inter_freq_carrier_freq_list_ext_r12.is_present();
    group_flags[2] |= inter_freq_carrier_freq_list_ext_v1280.is_present();
    group_flags[3] |= inter_freq_carrier_freq_list_v1310.is_present();
    group_flags[3] |= inter_freq_carrier_freq_list_ext_v1310.is_present();
    group_flags[4] |= inter_freq_carrier_freq_list_v1350.is_present();
    group_flags[4] |= inter_freq_carrier_freq_list_ext_v1350.is_present();
    group_flags[5] |= inter_freq_carrier_freq_list_ext_v1360.is_present();
    group_flags[6] |= scptm_freq_offset_r14_present;
    group_flags[7] |= inter_freq_carrier_freq_list_v1530.is_present();
    group_flags[7] |= inter_freq_carrier_freq_list_ext_v1530.is_present();
    group_flags[7] |= meas_idle_cfg_sib_r15.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v1250.is_present(), 1));
      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_ext_r12.is_present(), 1));
      if (inter_freq_carrier_freq_list_v1250.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_v1250, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_r12.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_ext_r12, 1, 8));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_ext_v1280.is_present(), 1));
      if (inter_freq_carrier_freq_list_ext_v1280.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_ext_v1280, 1, 8));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_ext_v1310.is_present(), 1));
      if (inter_freq_carrier_freq_list_v1310.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_v1310, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_v1310.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_ext_v1310, 1, 8));
      }
    }
    if (group_flags[4]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v1350.is_present(), 1));
      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_ext_v1350.is_present(), 1));
      if (inter_freq_carrier_freq_list_v1350.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_v1350, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_v1350.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_ext_v1350, 1, 8));
      }
    }
    if (group_flags[5]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_ext_v1360.is_present(), 1));
      if (inter_freq_carrier_freq_list_ext_v1360.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_ext_v1360, 1, 8));
      }
    }
    if (group_flags[6]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(scptm_freq_offset_r14_present, 1));
      if (scptm_freq_offset_r14_present) {
        HANDLE_CODE(pack_integer(bref, scptm_freq_offset_r14, (uint8_t)1u, (uint8_t)8u));
      }
    }
    if (group_flags[7]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_v1530.is_present(), 1));
      HANDLE_CODE(bref.pack(inter_freq_carrier_freq_list_ext_v1530.is_present(), 1));
      HANDLE_CODE(bref.pack(meas_idle_cfg_sib_r15.is_present(), 1));
      if (inter_freq_carrier_freq_list_v1530.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_v1530, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_v1530.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *inter_freq_carrier_freq_list_ext_v1530, 1, 8));
      }
      if (meas_idle_cfg_sib_r15.is_present()) {
        HANDLE_CODE(meas_idle_cfg_sib_r15->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type5_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(inter_freq_carrier_freq_list, bref, 1, 8));

  if (ext) {
    ext_groups_unpacker_guard group_flags(8);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool inter_freq_carrier_freq_list_v1250_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v1250_present, 1));
      inter_freq_carrier_freq_list_v1250.set_present(inter_freq_carrier_freq_list_v1250_present);
      bool inter_freq_carrier_freq_list_ext_r12_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_ext_r12_present, 1));
      inter_freq_carrier_freq_list_ext_r12.set_present(inter_freq_carrier_freq_list_ext_r12_present);
      if (inter_freq_carrier_freq_list_v1250.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_v1250, bref, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_r12.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_ext_r12, bref, 1, 8));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool inter_freq_carrier_freq_list_ext_v1280_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_ext_v1280_present, 1));
      inter_freq_carrier_freq_list_ext_v1280.set_present(inter_freq_carrier_freq_list_ext_v1280_present);
      if (inter_freq_carrier_freq_list_ext_v1280.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_ext_v1280, bref, 1, 8));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool inter_freq_carrier_freq_list_v1310_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v1310_present, 1));
      inter_freq_carrier_freq_list_v1310.set_present(inter_freq_carrier_freq_list_v1310_present);
      bool inter_freq_carrier_freq_list_ext_v1310_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_ext_v1310_present, 1));
      inter_freq_carrier_freq_list_ext_v1310.set_present(inter_freq_carrier_freq_list_ext_v1310_present);
      if (inter_freq_carrier_freq_list_v1310.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_v1310, bref, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_v1310.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_ext_v1310, bref, 1, 8));
      }
    }
    if (group_flags[4]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool inter_freq_carrier_freq_list_v1350_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v1350_present, 1));
      inter_freq_carrier_freq_list_v1350.set_present(inter_freq_carrier_freq_list_v1350_present);
      bool inter_freq_carrier_freq_list_ext_v1350_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_ext_v1350_present, 1));
      inter_freq_carrier_freq_list_ext_v1350.set_present(inter_freq_carrier_freq_list_ext_v1350_present);
      if (inter_freq_carrier_freq_list_v1350.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_v1350, bref, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_v1350.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_ext_v1350, bref, 1, 8));
      }
    }
    if (group_flags[5]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool inter_freq_carrier_freq_list_ext_v1360_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_ext_v1360_present, 1));
      inter_freq_carrier_freq_list_ext_v1360.set_present(inter_freq_carrier_freq_list_ext_v1360_present);
      if (inter_freq_carrier_freq_list_ext_v1360.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_ext_v1360, bref, 1, 8));
      }
    }
    if (group_flags[6]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(scptm_freq_offset_r14_present, 1));
      if (scptm_freq_offset_r14_present) {
        HANDLE_CODE(unpack_integer(scptm_freq_offset_r14, bref, (uint8_t)1u, (uint8_t)8u));
      }
    }
    if (group_flags[7]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool inter_freq_carrier_freq_list_v1530_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_v1530_present, 1));
      inter_freq_carrier_freq_list_v1530.set_present(inter_freq_carrier_freq_list_v1530_present);
      bool inter_freq_carrier_freq_list_ext_v1530_present;
      HANDLE_CODE(bref.unpack(inter_freq_carrier_freq_list_ext_v1530_present, 1));
      inter_freq_carrier_freq_list_ext_v1530.set_present(inter_freq_carrier_freq_list_ext_v1530_present);
      bool meas_idle_cfg_sib_r15_present;
      HANDLE_CODE(bref.unpack(meas_idle_cfg_sib_r15_present, 1));
      meas_idle_cfg_sib_r15.set_present(meas_idle_cfg_sib_r15_present);
      if (inter_freq_carrier_freq_list_v1530.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_v1530, bref, 1, 8));
      }
      if (inter_freq_carrier_freq_list_ext_v1530.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*inter_freq_carrier_freq_list_ext_v1530, bref, 1, 8));
      }
      if (meas_idle_cfg_sib_r15.is_present()) {
        HANDLE_CODE(meas_idle_cfg_sib_r15->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type5_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("interFreqCarrierFreqList");
  for (const auto& e1 : inter_freq_carrier_freq_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
    if (inter_freq_carrier_freq_list_v1250.is_present()) {
      j.start_array("interFreqCarrierFreqList-v1250");
      for (const auto& e1 : *inter_freq_carrier_freq_list_v1250) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_ext_r12.is_present()) {
      j.start_array("interFreqCarrierFreqListExt-r12");
      for (const auto& e1 : *inter_freq_carrier_freq_list_ext_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_ext_v1280.is_present()) {
      j.start_array("interFreqCarrierFreqListExt-v1280");
      for (const auto& e1 : *inter_freq_carrier_freq_list_ext_v1280) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_v1310.is_present()) {
      j.start_array("interFreqCarrierFreqList-v1310");
      for (const auto& e1 : *inter_freq_carrier_freq_list_v1310) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_ext_v1310.is_present()) {
      j.start_array("interFreqCarrierFreqListExt-v1310");
      for (const auto& e1 : *inter_freq_carrier_freq_list_ext_v1310) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_v1350.is_present()) {
      j.start_array("interFreqCarrierFreqList-v1350");
      for (const auto& e1 : *inter_freq_carrier_freq_list_v1350) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_ext_v1350.is_present()) {
      j.start_array("interFreqCarrierFreqListExt-v1350");
      for (const auto& e1 : *inter_freq_carrier_freq_list_ext_v1350) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_ext_v1360.is_present()) {
      j.start_array("interFreqCarrierFreqListExt-v1360");
      for (const auto& e1 : *inter_freq_carrier_freq_list_ext_v1360) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (scptm_freq_offset_r14_present) {
      j.write_int("scptm-FreqOffset-r14", scptm_freq_offset_r14);
    }
    if (inter_freq_carrier_freq_list_v1530.is_present()) {
      j.start_array("interFreqCarrierFreqList-v1530");
      for (const auto& e1 : *inter_freq_carrier_freq_list_v1530) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (inter_freq_carrier_freq_list_ext_v1530.is_present()) {
      j.start_array("interFreqCarrierFreqListExt-v1530");
      for (const auto& e1 : *inter_freq_carrier_freq_list_ext_v1530) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (meas_idle_cfg_sib_r15.is_present()) {
      j.write_fieldname("measIdleConfigSIB-r15");
      meas_idle_cfg_sib_r15->to_json(j);
    }
  }
  j.end_obj();
}

// SystemInformationBlockType6 ::= SEQUENCE
SRSASN_CODE sib_type6_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(carrier_freq_list_utra_fdd_present, 1));
  HANDLE_CODE(bref.pack(carrier_freq_list_utra_tdd_present, 1));
  HANDLE_CODE(bref.pack(t_resel_utra_sf_present, 1));

  if (carrier_freq_list_utra_fdd_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, carrier_freq_list_utra_fdd, 1, 16));
  }
  if (carrier_freq_list_utra_tdd_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, carrier_freq_list_utra_tdd, 1, 16));
  }
  HANDLE_CODE(pack_integer(bref, t_resel_utra, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_utra_sf_present) {
    HANDLE_CODE(t_resel_utra_sf.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= late_non_crit_ext_present;
    group_flags[1] |= carrier_freq_list_utra_fdd_v1250.is_present();
    group_flags[1] |= carrier_freq_list_utra_tdd_v1250.is_present();
    group_flags[1] |= carrier_freq_list_utra_fdd_ext_r12.is_present();
    group_flags[1] |= carrier_freq_list_utra_tdd_ext_r12.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(carrier_freq_list_utra_fdd_v1250.is_present(), 1));
      HANDLE_CODE(bref.pack(carrier_freq_list_utra_tdd_v1250.is_present(), 1));
      HANDLE_CODE(bref.pack(carrier_freq_list_utra_fdd_ext_r12.is_present(), 1));
      HANDLE_CODE(bref.pack(carrier_freq_list_utra_tdd_ext_r12.is_present(), 1));
      if (carrier_freq_list_utra_fdd_v1250.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *carrier_freq_list_utra_fdd_v1250, 1, 16));
      }
      if (carrier_freq_list_utra_tdd_v1250.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *carrier_freq_list_utra_tdd_v1250, 1, 16));
      }
      if (carrier_freq_list_utra_fdd_ext_r12.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *carrier_freq_list_utra_fdd_ext_r12, 1, 16));
      }
      if (carrier_freq_list_utra_tdd_ext_r12.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *carrier_freq_list_utra_tdd_ext_r12, 1, 16));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type6_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(carrier_freq_list_utra_fdd_present, 1));
  HANDLE_CODE(bref.unpack(carrier_freq_list_utra_tdd_present, 1));
  HANDLE_CODE(bref.unpack(t_resel_utra_sf_present, 1));

  if (carrier_freq_list_utra_fdd_present) {
    HANDLE_CODE(unpack_dyn_seq_of(carrier_freq_list_utra_fdd, bref, 1, 16));
  }
  if (carrier_freq_list_utra_tdd_present) {
    HANDLE_CODE(unpack_dyn_seq_of(carrier_freq_list_utra_tdd, bref, 1, 16));
  }
  HANDLE_CODE(unpack_integer(t_resel_utra, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_utra_sf_present) {
    HANDLE_CODE(t_resel_utra_sf.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool carrier_freq_list_utra_fdd_v1250_present;
      HANDLE_CODE(bref.unpack(carrier_freq_list_utra_fdd_v1250_present, 1));
      carrier_freq_list_utra_fdd_v1250.set_present(carrier_freq_list_utra_fdd_v1250_present);
      bool carrier_freq_list_utra_tdd_v1250_present;
      HANDLE_CODE(bref.unpack(carrier_freq_list_utra_tdd_v1250_present, 1));
      carrier_freq_list_utra_tdd_v1250.set_present(carrier_freq_list_utra_tdd_v1250_present);
      bool carrier_freq_list_utra_fdd_ext_r12_present;
      HANDLE_CODE(bref.unpack(carrier_freq_list_utra_fdd_ext_r12_present, 1));
      carrier_freq_list_utra_fdd_ext_r12.set_present(carrier_freq_list_utra_fdd_ext_r12_present);
      bool carrier_freq_list_utra_tdd_ext_r12_present;
      HANDLE_CODE(bref.unpack(carrier_freq_list_utra_tdd_ext_r12_present, 1));
      carrier_freq_list_utra_tdd_ext_r12.set_present(carrier_freq_list_utra_tdd_ext_r12_present);
      if (carrier_freq_list_utra_fdd_v1250.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*carrier_freq_list_utra_fdd_v1250, bref, 1, 16));
      }
      if (carrier_freq_list_utra_tdd_v1250.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*carrier_freq_list_utra_tdd_v1250, bref, 1, 16));
      }
      if (carrier_freq_list_utra_fdd_ext_r12.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*carrier_freq_list_utra_fdd_ext_r12, bref, 1, 16));
      }
      if (carrier_freq_list_utra_tdd_ext_r12.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*carrier_freq_list_utra_tdd_ext_r12, bref, 1, 16));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type6_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (carrier_freq_list_utra_fdd_present) {
    j.start_array("carrierFreqListUTRA-FDD");
    for (const auto& e1 : carrier_freq_list_utra_fdd) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (carrier_freq_list_utra_tdd_present) {
    j.start_array("carrierFreqListUTRA-TDD");
    for (const auto& e1 : carrier_freq_list_utra_tdd) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.write_int("t-ReselectionUTRA", t_resel_utra);
  if (t_resel_utra_sf_present) {
    j.write_fieldname("t-ReselectionUTRA-SF");
    t_resel_utra_sf.to_json(j);
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
    if (carrier_freq_list_utra_fdd_v1250.is_present()) {
      j.start_array("carrierFreqListUTRA-FDD-v1250");
      for (const auto& e1 : *carrier_freq_list_utra_fdd_v1250) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (carrier_freq_list_utra_tdd_v1250.is_present()) {
      j.start_array("carrierFreqListUTRA-TDD-v1250");
      for (const auto& e1 : *carrier_freq_list_utra_tdd_v1250) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (carrier_freq_list_utra_fdd_ext_r12.is_present()) {
      j.start_array("carrierFreqListUTRA-FDD-Ext-r12");
      for (const auto& e1 : *carrier_freq_list_utra_fdd_ext_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
    if (carrier_freq_list_utra_tdd_ext_r12.is_present()) {
      j.start_array("carrierFreqListUTRA-TDD-Ext-r12");
      for (const auto& e1 : *carrier_freq_list_utra_tdd_ext_r12) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// SystemInformationBlockType7 ::= SEQUENCE
SRSASN_CODE sib_type7_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(t_resel_geran_sf_present, 1));
  HANDLE_CODE(bref.pack(carrier_freqs_info_list_present, 1));

  HANDLE_CODE(pack_integer(bref, t_resel_geran, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_geran_sf_present) {
    HANDLE_CODE(t_resel_geran_sf.pack(bref));
  }
  if (carrier_freqs_info_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, carrier_freqs_info_list, 1, 16));
  }

  if (ext) {
    HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.pack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type7_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(t_resel_geran_sf_present, 1));
  HANDLE_CODE(bref.unpack(carrier_freqs_info_list_present, 1));

  HANDLE_CODE(unpack_integer(t_resel_geran, bref, (uint8_t)0u, (uint8_t)7u));
  if (t_resel_geran_sf_present) {
    HANDLE_CODE(t_resel_geran_sf.unpack(bref));
  }
  if (carrier_freqs_info_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(carrier_freqs_info_list, bref, 1, 16));
  }

  if (ext) {
    HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.unpack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type7_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("t-ReselectionGERAN", t_resel_geran);
  if (t_resel_geran_sf_present) {
    j.write_fieldname("t-ReselectionGERAN-SF");
    t_resel_geran_sf.to_json(j);
  }
  if (carrier_freqs_info_list_present) {
    j.start_array("carrierFreqsInfoList");
    for (const auto& e1 : carrier_freqs_info_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
  }
  j.end_obj();
}

// SystemInformationBlockType8 ::= SEQUENCE
SRSASN_CODE sib_type8_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(sys_time_info_present, 1));
  HANDLE_CODE(bref.pack(search_win_size_present, 1));
  HANDLE_CODE(bref.pack(params_hrpd_present, 1));
  HANDLE_CODE(bref.pack(params1_xrtt_present, 1));

  if (sys_time_info_present) {
    HANDLE_CODE(sys_time_info.pack(bref));
  }
  if (search_win_size_present) {
    HANDLE_CODE(pack_integer(bref, search_win_size, (uint8_t)0u, (uint8_t)15u));
  }
  if (params_hrpd_present) {
    HANDLE_CODE(bref.pack(params_hrpd.cell_resel_params_hrpd_present, 1));
    HANDLE_CODE(params_hrpd.pre_regist_info_hrpd.pack(bref));
    if (params_hrpd.cell_resel_params_hrpd_present) {
      HANDLE_CODE(params_hrpd.cell_resel_params_hrpd.pack(bref));
    }
  }
  if (params1_xrtt_present) {
    HANDLE_CODE(bref.pack(params1_xrtt.csfb_regist_param1_xrtt_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt.long_code_state1_xrtt_present, 1));
    HANDLE_CODE(bref.pack(params1_xrtt.cell_resel_params1_xrtt_present, 1));
    if (params1_xrtt.csfb_regist_param1_xrtt_present) {
      HANDLE_CODE(params1_xrtt.csfb_regist_param1_xrtt.pack(bref));
    }
    if (params1_xrtt.long_code_state1_xrtt_present) {
      HANDLE_CODE(params1_xrtt.long_code_state1_xrtt.pack(bref));
    }
    if (params1_xrtt.cell_resel_params1_xrtt_present) {
      HANDLE_CODE(params1_xrtt.cell_resel_params1_xrtt.pack(bref));
    }
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= late_non_crit_ext_present;
    group_flags[1] |= csfb_support_for_dual_rx_ues_r9_present;
    group_flags[1] |= cell_resel_params_hrpd_v920.is_present();
    group_flags[1] |= cell_resel_params1_xrtt_v920.is_present();
    group_flags[1] |= csfb_regist_param1_xrtt_v920.is_present();
    group_flags[1] |= ac_barr_cfg1_xrtt_r9.is_present();
    group_flags[2] |= csfb_dual_rx_tx_support_r10_present;
    group_flags[3] |= sib8_per_plmn_list_r11.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(csfb_support_for_dual_rx_ues_r9_present, 1));
      HANDLE_CODE(bref.pack(cell_resel_params_hrpd_v920.is_present(), 1));
      HANDLE_CODE(bref.pack(cell_resel_params1_xrtt_v920.is_present(), 1));
      HANDLE_CODE(bref.pack(csfb_regist_param1_xrtt_v920.is_present(), 1));
      HANDLE_CODE(bref.pack(ac_barr_cfg1_xrtt_r9.is_present(), 1));
      if (csfb_support_for_dual_rx_ues_r9_present) {
        HANDLE_CODE(bref.pack(csfb_support_for_dual_rx_ues_r9, 1));
      }
      if (cell_resel_params_hrpd_v920.is_present()) {
        HANDLE_CODE(cell_resel_params_hrpd_v920->pack(bref));
      }
      if (cell_resel_params1_xrtt_v920.is_present()) {
        HANDLE_CODE(cell_resel_params1_xrtt_v920->pack(bref));
      }
      if (csfb_regist_param1_xrtt_v920.is_present()) {
        HANDLE_CODE(csfb_regist_param1_xrtt_v920->pack(bref));
      }
      if (ac_barr_cfg1_xrtt_r9.is_present()) {
        HANDLE_CODE(ac_barr_cfg1_xrtt_r9->pack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(csfb_dual_rx_tx_support_r10_present, 1));
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(sib8_per_plmn_list_r11.is_present(), 1));
      if (sib8_per_plmn_list_r11.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *sib8_per_plmn_list_r11, 1, 6));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type8_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(sys_time_info_present, 1));
  HANDLE_CODE(bref.unpack(search_win_size_present, 1));
  HANDLE_CODE(bref.unpack(params_hrpd_present, 1));
  HANDLE_CODE(bref.unpack(params1_xrtt_present, 1));

  if (sys_time_info_present) {
    HANDLE_CODE(sys_time_info.unpack(bref));
  }
  if (search_win_size_present) {
    HANDLE_CODE(unpack_integer(search_win_size, bref, (uint8_t)0u, (uint8_t)15u));
  }
  if (params_hrpd_present) {
    HANDLE_CODE(bref.unpack(params_hrpd.cell_resel_params_hrpd_present, 1));
    HANDLE_CODE(params_hrpd.pre_regist_info_hrpd.unpack(bref));
    if (params_hrpd.cell_resel_params_hrpd_present) {
      HANDLE_CODE(params_hrpd.cell_resel_params_hrpd.unpack(bref));
    }
  }
  if (params1_xrtt_present) {
    HANDLE_CODE(bref.unpack(params1_xrtt.csfb_regist_param1_xrtt_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt.long_code_state1_xrtt_present, 1));
    HANDLE_CODE(bref.unpack(params1_xrtt.cell_resel_params1_xrtt_present, 1));
    if (params1_xrtt.csfb_regist_param1_xrtt_present) {
      HANDLE_CODE(params1_xrtt.csfb_regist_param1_xrtt.unpack(bref));
    }
    if (params1_xrtt.long_code_state1_xrtt_present) {
      HANDLE_CODE(params1_xrtt.long_code_state1_xrtt.unpack(bref));
    }
    if (params1_xrtt.cell_resel_params1_xrtt_present) {
      HANDLE_CODE(params1_xrtt.cell_resel_params1_xrtt.unpack(bref));
    }
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(4);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(csfb_support_for_dual_rx_ues_r9_present, 1));
      bool cell_resel_params_hrpd_v920_present;
      HANDLE_CODE(bref.unpack(cell_resel_params_hrpd_v920_present, 1));
      cell_resel_params_hrpd_v920.set_present(cell_resel_params_hrpd_v920_present);
      bool cell_resel_params1_xrtt_v920_present;
      HANDLE_CODE(bref.unpack(cell_resel_params1_xrtt_v920_present, 1));
      cell_resel_params1_xrtt_v920.set_present(cell_resel_params1_xrtt_v920_present);
      bool csfb_regist_param1_xrtt_v920_present;
      HANDLE_CODE(bref.unpack(csfb_regist_param1_xrtt_v920_present, 1));
      csfb_regist_param1_xrtt_v920.set_present(csfb_regist_param1_xrtt_v920_present);
      bool ac_barr_cfg1_xrtt_r9_present;
      HANDLE_CODE(bref.unpack(ac_barr_cfg1_xrtt_r9_present, 1));
      ac_barr_cfg1_xrtt_r9.set_present(ac_barr_cfg1_xrtt_r9_present);
      if (csfb_support_for_dual_rx_ues_r9_present) {
        HANDLE_CODE(bref.unpack(csfb_support_for_dual_rx_ues_r9, 1));
      }
      if (cell_resel_params_hrpd_v920.is_present()) {
        HANDLE_CODE(cell_resel_params_hrpd_v920->unpack(bref));
      }
      if (cell_resel_params1_xrtt_v920.is_present()) {
        HANDLE_CODE(cell_resel_params1_xrtt_v920->unpack(bref));
      }
      if (csfb_regist_param1_xrtt_v920.is_present()) {
        HANDLE_CODE(csfb_regist_param1_xrtt_v920->unpack(bref));
      }
      if (ac_barr_cfg1_xrtt_r9.is_present()) {
        HANDLE_CODE(ac_barr_cfg1_xrtt_r9->unpack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(csfb_dual_rx_tx_support_r10_present, 1));
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool sib8_per_plmn_list_r11_present;
      HANDLE_CODE(bref.unpack(sib8_per_plmn_list_r11_present, 1));
      sib8_per_plmn_list_r11.set_present(sib8_per_plmn_list_r11_present);
      if (sib8_per_plmn_list_r11.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*sib8_per_plmn_list_r11, bref, 1, 6));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type8_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (sys_time_info_present) {
    j.write_fieldname("systemTimeInfo");
    sys_time_info.to_json(j);
  }
  if (search_win_size_present) {
    j.write_int("searchWindowSize", search_win_size);
  }
  if (params_hrpd_present) {
    j.write_fieldname("parametersHRPD");
    j.start_obj();
    j.write_fieldname("preRegistrationInfoHRPD");
    params_hrpd.pre_regist_info_hrpd.to_json(j);
    if (params_hrpd.cell_resel_params_hrpd_present) {
      j.write_fieldname("cellReselectionParametersHRPD");
      params_hrpd.cell_resel_params_hrpd.to_json(j);
    }
    j.end_obj();
  }
  if (params1_xrtt_present) {
    j.write_fieldname("parameters1XRTT");
    j.start_obj();
    if (params1_xrtt.csfb_regist_param1_xrtt_present) {
      j.write_fieldname("csfb-RegistrationParam1XRTT");
      params1_xrtt.csfb_regist_param1_xrtt.to_json(j);
    }
    if (params1_xrtt.long_code_state1_xrtt_present) {
      j.write_str("longCodeState1XRTT", params1_xrtt.long_code_state1_xrtt.to_string());
    }
    if (params1_xrtt.cell_resel_params1_xrtt_present) {
      j.write_fieldname("cellReselectionParameters1XRTT");
      params1_xrtt.cell_resel_params1_xrtt.to_json(j);
    }
    j.end_obj();
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
    if (csfb_support_for_dual_rx_ues_r9_present) {
      j.write_bool("csfb-SupportForDualRxUEs-r9", csfb_support_for_dual_rx_ues_r9);
    }
    if (cell_resel_params_hrpd_v920.is_present()) {
      j.write_fieldname("cellReselectionParametersHRPD-v920");
      cell_resel_params_hrpd_v920->to_json(j);
    }
    if (cell_resel_params1_xrtt_v920.is_present()) {
      j.write_fieldname("cellReselectionParameters1XRTT-v920");
      cell_resel_params1_xrtt_v920->to_json(j);
    }
    if (csfb_regist_param1_xrtt_v920.is_present()) {
      j.write_fieldname("csfb-RegistrationParam1XRTT-v920");
      csfb_regist_param1_xrtt_v920->to_json(j);
    }
    if (ac_barr_cfg1_xrtt_r9.is_present()) {
      j.write_fieldname("ac-BarringConfig1XRTT-r9");
      ac_barr_cfg1_xrtt_r9->to_json(j);
    }
    if (csfb_dual_rx_tx_support_r10_present) {
      j.write_str("csfb-DualRxTxSupport-r10", "true");
    }
    if (sib8_per_plmn_list_r11.is_present()) {
      j.start_array("sib8-PerPLMN-List-r11");
      for (const auto& e1 : *sib8_per_plmn_list_r11) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// SystemInformationBlockType9 ::= SEQUENCE
SRSASN_CODE sib_type9_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(hnb_name_present, 1));

  if (hnb_name_present) {
    HANDLE_CODE(hnb_name.pack(bref));
  }

  if (ext) {
    HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.pack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type9_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(hnb_name_present, 1));

  if (hnb_name_present) {
    HANDLE_CODE(hnb_name.unpack(bref));
  }

  if (ext) {
    HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));

    if (late_non_crit_ext_present) {
      HANDLE_CODE(late_non_crit_ext.unpack(bref));
    }
  }
  return SRSASN_SUCCESS;
}
void sib_type9_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (hnb_name_present) {
    j.write_str("hnb-Name", hnb_name.to_string());
  }
  if (ext) {
    if (late_non_crit_ext_present) {
      j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
    }
  }
  j.end_obj();
}

// PosSystemInformation-r15-IEs ::= SEQUENCE
SRSASN_CODE pos_sys_info_r15_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, pos_sib_type_and_info_r15, 1, 32));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pos_sys_info_r15_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(pos_sib_type_and_info_r15, bref, 1, 32));
  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pos_sys_info_r15_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("posSIB-TypeAndInfo-r15");
  for (const auto& e1 : pos_sib_type_and_info_r15) {
    e1.to_json(j);
  }
  j.end_array();
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

void pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::destroy_()
{
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus2_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus3_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus4_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus5_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus6_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus7_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus1_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus2_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus3_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus4_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus5_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus6_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus7_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus8_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus9_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus10_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus11_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus12_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus13_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus14_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus15_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus16_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus17_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus18_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus19_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    case types::pos_sib3_minus1_r15:
      c.destroy<sib_pos_r15_s>();
      break;
    default:
      break;
  }
}
void pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus2_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus3_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus4_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus5_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus6_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib1_minus7_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus1_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus2_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus3_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus4_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus5_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus6_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus7_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus8_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus9_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus10_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus11_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus12_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus13_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus14_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus15_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus16_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus17_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus18_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib2_minus19_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::pos_sib3_minus1_r15:
      c.init<sib_pos_r15_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_");
  }
}
pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::pos_sib_type_and_info_r15_item_c_(
    const pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus2_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus3_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus4_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus5_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus6_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus7_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus1_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus2_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus3_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus4_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus5_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus6_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus7_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus8_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus9_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus10_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus11_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus12_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus13_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus14_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus15_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus16_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus17_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus18_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus19_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib3_minus1_r15:
      c.init(other.c.get<sib_pos_r15_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_");
  }
}
pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_& pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::
                                                           operator=(const pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus2_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus3_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus4_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus5_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus6_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib1_minus7_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus1_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus2_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus3_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus4_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus5_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus6_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus7_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus8_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus9_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus10_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus11_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus12_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus13_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus14_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus15_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus16_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus17_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus18_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib2_minus19_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::pos_sib3_minus1_r15:
      c.set(other.c.get<sib_pos_r15_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_");
  }

  return *this;
}
void pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      j.write_fieldname("posSib1-1-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib1_minus2_r15:
      j.write_fieldname("posSib1-2-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib1_minus3_r15:
      j.write_fieldname("posSib1-3-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib1_minus4_r15:
      j.write_fieldname("posSib1-4-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib1_minus5_r15:
      j.write_fieldname("posSib1-5-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib1_minus6_r15:
      j.write_fieldname("posSib1-6-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib1_minus7_r15:
      j.write_fieldname("posSib1-7-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus1_r15:
      j.write_fieldname("posSib2-1-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus2_r15:
      j.write_fieldname("posSib2-2-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus3_r15:
      j.write_fieldname("posSib2-3-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus4_r15:
      j.write_fieldname("posSib2-4-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus5_r15:
      j.write_fieldname("posSib2-5-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus6_r15:
      j.write_fieldname("posSib2-6-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus7_r15:
      j.write_fieldname("posSib2-7-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus8_r15:
      j.write_fieldname("posSib2-8-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus9_r15:
      j.write_fieldname("posSib2-9-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus10_r15:
      j.write_fieldname("posSib2-10-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus11_r15:
      j.write_fieldname("posSib2-11-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus12_r15:
      j.write_fieldname("posSib2-12-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus13_r15:
      j.write_fieldname("posSib2-13-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus14_r15:
      j.write_fieldname("posSib2-14-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus15_r15:
      j.write_fieldname("posSib2-15-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus16_r15:
      j.write_fieldname("posSib2-16-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus17_r15:
      j.write_fieldname("posSib2-17-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus18_r15:
      j.write_fieldname("posSib2-18-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib2_minus19_r15:
      j.write_fieldname("posSib2-19-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    case types::pos_sib3_minus1_r15:
      j.write_fieldname("posSib3-1-r15");
      c.get<sib_pos_r15_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_");
  }
  j.end_obj();
}
SRSASN_CODE pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib1_minus2_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib1_minus3_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib1_minus4_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib1_minus5_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib1_minus6_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib1_minus7_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus1_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus2_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus3_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus4_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus5_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus6_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus7_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus8_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus9_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus10_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus11_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus12_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus13_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus14_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus15_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus16_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus17_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus18_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib2_minus19_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    case types::pos_sib3_minus1_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::pos_sib1_minus1_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib1_minus2_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib1_minus3_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib1_minus4_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib1_minus5_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib1_minus6_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib1_minus7_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus1_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus2_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus3_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus4_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus5_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus6_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus7_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus8_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus9_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus10_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus11_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus12_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus13_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus14_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus15_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus16_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus17_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus18_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib2_minus19_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    case types::pos_sib3_minus1_r15:
      HANDLE_CODE(c.get<sib_pos_r15_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pos_sys_info_r15_ies_s::pos_sib_type_and_info_r15_item_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sib_info_item_c::destroy_()
{
  switch (type_) {
    case types::sib2:
      c.destroy<sib_type2_s>();
      break;
    case types::sib3:
      c.destroy<sib_type3_s>();
      break;
    case types::sib4:
      c.destroy<sib_type4_s>();
      break;
    case types::sib5:
      c.destroy<sib_type5_s>();
      break;
    case types::sib6:
      c.destroy<sib_type6_s>();
      break;
    case types::sib7:
      c.destroy<sib_type7_s>();
      break;
    case types::sib8:
      c.destroy<sib_type8_s>();
      break;
    case types::sib9:
      c.destroy<sib_type9_s>();
      break;
    case types::sib10:
      c.destroy<sib_type10_s>();
      break;
    case types::sib11:
      c.destroy<sib_type11_s>();
      break;
    case types::sib12_v920:
      c.destroy<sib_type12_r9_s>();
      break;
    case types::sib13_v920:
      c.destroy<sib_type13_r9_s>();
      break;
    case types::sib14_v1130:
      c.destroy<sib_type14_r11_s>();
      break;
    case types::sib15_v1130:
      c.destroy<sib_type15_r11_s>();
      break;
    case types::sib16_v1130:
      c.destroy<sib_type16_r11_s>();
      break;
    case types::sib17_v1250:
      c.destroy<sib_type17_r12_s>();
      break;
    case types::sib18_v1250:
      c.destroy<sib_type18_r12_s>();
      break;
    case types::sib19_v1250:
      c.destroy<sib_type19_r12_s>();
      break;
    case types::sib20_v1310:
      c.destroy<sib_type20_r13_s>();
      break;
    case types::sib21_v1430:
      c.destroy<sib_type21_r14_s>();
      break;
    case types::sib24_v1530:
      c.destroy<sib_type24_r15_s>();
      break;
    case types::sib25_v1530:
      c.destroy<sib_type25_r15_s>();
      break;
    case types::sib26_v1530:
      c.destroy<sib_type26_r15_s>();
      break;
    default:
      break;
  }
}
void sib_info_item_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sib2:
      c.init<sib_type2_s>();
      break;
    case types::sib3:
      c.init<sib_type3_s>();
      break;
    case types::sib4:
      c.init<sib_type4_s>();
      break;
    case types::sib5:
      c.init<sib_type5_s>();
      break;
    case types::sib6:
      c.init<sib_type6_s>();
      break;
    case types::sib7:
      c.init<sib_type7_s>();
      break;
    case types::sib8:
      c.init<sib_type8_s>();
      break;
    case types::sib9:
      c.init<sib_type9_s>();
      break;
    case types::sib10:
      c.init<sib_type10_s>();
      break;
    case types::sib11:
      c.init<sib_type11_s>();
      break;
    case types::sib12_v920:
      c.init<sib_type12_r9_s>();
      break;
    case types::sib13_v920:
      c.init<sib_type13_r9_s>();
      break;
    case types::sib14_v1130:
      c.init<sib_type14_r11_s>();
      break;
    case types::sib15_v1130:
      c.init<sib_type15_r11_s>();
      break;
    case types::sib16_v1130:
      c.init<sib_type16_r11_s>();
      break;
    case types::sib17_v1250:
      c.init<sib_type17_r12_s>();
      break;
    case types::sib18_v1250:
      c.init<sib_type18_r12_s>();
      break;
    case types::sib19_v1250:
      c.init<sib_type19_r12_s>();
      break;
    case types::sib20_v1310:
      c.init<sib_type20_r13_s>();
      break;
    case types::sib21_v1430:
      c.init<sib_type21_r14_s>();
      break;
    case types::sib24_v1530:
      c.init<sib_type24_r15_s>();
      break;
    case types::sib25_v1530:
      c.init<sib_type25_r15_s>();
      break;
    case types::sib26_v1530:
      c.init<sib_type26_r15_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_info_item_c");
  }
}
sib_info_item_c::sib_info_item_c(const sib_info_item_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sib2:
      c.init(other.c.get<sib_type2_s>());
      break;
    case types::sib3:
      c.init(other.c.get<sib_type3_s>());
      break;
    case types::sib4:
      c.init(other.c.get<sib_type4_s>());
      break;
    case types::sib5:
      c.init(other.c.get<sib_type5_s>());
      break;
    case types::sib6:
      c.init(other.c.get<sib_type6_s>());
      break;
    case types::sib7:
      c.init(other.c.get<sib_type7_s>());
      break;
    case types::sib8:
      c.init(other.c.get<sib_type8_s>());
      break;
    case types::sib9:
      c.init(other.c.get<sib_type9_s>());
      break;
    case types::sib10:
      c.init(other.c.get<sib_type10_s>());
      break;
    case types::sib11:
      c.init(other.c.get<sib_type11_s>());
      break;
    case types::sib12_v920:
      c.init(other.c.get<sib_type12_r9_s>());
      break;
    case types::sib13_v920:
      c.init(other.c.get<sib_type13_r9_s>());
      break;
    case types::sib14_v1130:
      c.init(other.c.get<sib_type14_r11_s>());
      break;
    case types::sib15_v1130:
      c.init(other.c.get<sib_type15_r11_s>());
      break;
    case types::sib16_v1130:
      c.init(other.c.get<sib_type16_r11_s>());
      break;
    case types::sib17_v1250:
      c.init(other.c.get<sib_type17_r12_s>());
      break;
    case types::sib18_v1250:
      c.init(other.c.get<sib_type18_r12_s>());
      break;
    case types::sib19_v1250:
      c.init(other.c.get<sib_type19_r12_s>());
      break;
    case types::sib20_v1310:
      c.init(other.c.get<sib_type20_r13_s>());
      break;
    case types::sib21_v1430:
      c.init(other.c.get<sib_type21_r14_s>());
      break;
    case types::sib24_v1530:
      c.init(other.c.get<sib_type24_r15_s>());
      break;
    case types::sib25_v1530:
      c.init(other.c.get<sib_type25_r15_s>());
      break;
    case types::sib26_v1530:
      c.init(other.c.get<sib_type26_r15_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_info_item_c");
  }
}
sib_info_item_c& sib_info_item_c::operator=(const sib_info_item_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sib2:
      c.set(other.c.get<sib_type2_s>());
      break;
    case types::sib3:
      c.set(other.c.get<sib_type3_s>());
      break;
    case types::sib4:
      c.set(other.c.get<sib_type4_s>());
      break;
    case types::sib5:
      c.set(other.c.get<sib_type5_s>());
      break;
    case types::sib6:
      c.set(other.c.get<sib_type6_s>());
      break;
    case types::sib7:
      c.set(other.c.get<sib_type7_s>());
      break;
    case types::sib8:
      c.set(other.c.get<sib_type8_s>());
      break;
    case types::sib9:
      c.set(other.c.get<sib_type9_s>());
      break;
    case types::sib10:
      c.set(other.c.get<sib_type10_s>());
      break;
    case types::sib11:
      c.set(other.c.get<sib_type11_s>());
      break;
    case types::sib12_v920:
      c.set(other.c.get<sib_type12_r9_s>());
      break;
    case types::sib13_v920:
      c.set(other.c.get<sib_type13_r9_s>());
      break;
    case types::sib14_v1130:
      c.set(other.c.get<sib_type14_r11_s>());
      break;
    case types::sib15_v1130:
      c.set(other.c.get<sib_type15_r11_s>());
      break;
    case types::sib16_v1130:
      c.set(other.c.get<sib_type16_r11_s>());
      break;
    case types::sib17_v1250:
      c.set(other.c.get<sib_type17_r12_s>());
      break;
    case types::sib18_v1250:
      c.set(other.c.get<sib_type18_r12_s>());
      break;
    case types::sib19_v1250:
      c.set(other.c.get<sib_type19_r12_s>());
      break;
    case types::sib20_v1310:
      c.set(other.c.get<sib_type20_r13_s>());
      break;
    case types::sib21_v1430:
      c.set(other.c.get<sib_type21_r14_s>());
      break;
    case types::sib24_v1530:
      c.set(other.c.get<sib_type24_r15_s>());
      break;
    case types::sib25_v1530:
      c.set(other.c.get<sib_type25_r15_s>());
      break;
    case types::sib26_v1530:
      c.set(other.c.get<sib_type26_r15_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sib_info_item_c");
  }

  return *this;
}
void sib_info_item_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sib2:
      j.write_fieldname("sib2");
      c.get<sib_type2_s>().to_json(j);
      break;
    case types::sib3:
      j.write_fieldname("sib3");
      c.get<sib_type3_s>().to_json(j);
      break;
    case types::sib4:
      j.write_fieldname("sib4");
      c.get<sib_type4_s>().to_json(j);
      break;
    case types::sib5:
      j.write_fieldname("sib5");
      c.get<sib_type5_s>().to_json(j);
      break;
    case types::sib6:
      j.write_fieldname("sib6");
      c.get<sib_type6_s>().to_json(j);
      break;
    case types::sib7:
      j.write_fieldname("sib7");
      c.get<sib_type7_s>().to_json(j);
      break;
    case types::sib8:
      j.write_fieldname("sib8");
      c.get<sib_type8_s>().to_json(j);
      break;
    case types::sib9:
      j.write_fieldname("sib9");
      c.get<sib_type9_s>().to_json(j);
      break;
    case types::sib10:
      j.write_fieldname("sib10");
      c.get<sib_type10_s>().to_json(j);
      break;
    case types::sib11:
      j.write_fieldname("sib11");
      c.get<sib_type11_s>().to_json(j);
      break;
    case types::sib12_v920:
      j.write_fieldname("sib12-v920");
      c.get<sib_type12_r9_s>().to_json(j);
      break;
    case types::sib13_v920:
      j.write_fieldname("sib13-v920");
      c.get<sib_type13_r9_s>().to_json(j);
      break;
    case types::sib14_v1130:
      j.write_fieldname("sib14-v1130");
      c.get<sib_type14_r11_s>().to_json(j);
      break;
    case types::sib15_v1130:
      j.write_fieldname("sib15-v1130");
      c.get<sib_type15_r11_s>().to_json(j);
      break;
    case types::sib16_v1130:
      j.write_fieldname("sib16-v1130");
      c.get<sib_type16_r11_s>().to_json(j);
      break;
    case types::sib17_v1250:
      j.write_fieldname("sib17-v1250");
      c.get<sib_type17_r12_s>().to_json(j);
      break;
    case types::sib18_v1250:
      j.write_fieldname("sib18-v1250");
      c.get<sib_type18_r12_s>().to_json(j);
      break;
    case types::sib19_v1250:
      j.write_fieldname("sib19-v1250");
      c.get<sib_type19_r12_s>().to_json(j);
      break;
    case types::sib20_v1310:
      j.write_fieldname("sib20-v1310");
      c.get<sib_type20_r13_s>().to_json(j);
      break;
    case types::sib21_v1430:
      j.write_fieldname("sib21-v1430");
      c.get<sib_type21_r14_s>().to_json(j);
      break;
    case types::sib24_v1530:
      j.write_fieldname("sib24-v1530");
      c.get<sib_type24_r15_s>().to_json(j);
      break;
    case types::sib25_v1530:
      j.write_fieldname("sib25-v1530");
      c.get<sib_type25_r15_s>().to_json(j);
      break;
    case types::sib26_v1530:
      j.write_fieldname("sib26-v1530");
      c.get<sib_type26_r15_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "sib_info_item_c");
  }
  j.end_obj();
}
SRSASN_CODE sib_info_item_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sib2:
      HANDLE_CODE(c.get<sib_type2_s>().pack(bref));
      break;
    case types::sib3:
      HANDLE_CODE(c.get<sib_type3_s>().pack(bref));
      break;
    case types::sib4:
      HANDLE_CODE(c.get<sib_type4_s>().pack(bref));
      break;
    case types::sib5:
      HANDLE_CODE(c.get<sib_type5_s>().pack(bref));
      break;
    case types::sib6:
      HANDLE_CODE(c.get<sib_type6_s>().pack(bref));
      break;
    case types::sib7:
      HANDLE_CODE(c.get<sib_type7_s>().pack(bref));
      break;
    case types::sib8:
      HANDLE_CODE(c.get<sib_type8_s>().pack(bref));
      break;
    case types::sib9:
      HANDLE_CODE(c.get<sib_type9_s>().pack(bref));
      break;
    case types::sib10:
      HANDLE_CODE(c.get<sib_type10_s>().pack(bref));
      break;
    case types::sib11:
      HANDLE_CODE(c.get<sib_type11_s>().pack(bref));
      break;
    case types::sib12_v920: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type12_r9_s>().pack(bref));
    } break;
    case types::sib13_v920: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type13_r9_s>().pack(bref));
    } break;
    case types::sib14_v1130: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type14_r11_s>().pack(bref));
    } break;
    case types::sib15_v1130: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type15_r11_s>().pack(bref));
    } break;
    case types::sib16_v1130: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type16_r11_s>().pack(bref));
    } break;
    case types::sib17_v1250: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type17_r12_s>().pack(bref));
    } break;
    case types::sib18_v1250: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type18_r12_s>().pack(bref));
    } break;
    case types::sib19_v1250: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type19_r12_s>().pack(bref));
    } break;
    case types::sib20_v1310: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type20_r13_s>().pack(bref));
    } break;
    case types::sib21_v1430: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type21_r14_s>().pack(bref));
    } break;
    case types::sib24_v1530: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type24_r15_s>().pack(bref));
    } break;
    case types::sib25_v1530: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type25_r15_s>().pack(bref));
    } break;
    case types::sib26_v1530: {
      varlength_field_pack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type26_r15_s>().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "sib_info_item_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_info_item_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sib2:
      HANDLE_CODE(c.get<sib_type2_s>().unpack(bref));
      break;
    case types::sib3:
      HANDLE_CODE(c.get<sib_type3_s>().unpack(bref));
      break;
    case types::sib4:
      HANDLE_CODE(c.get<sib_type4_s>().unpack(bref));
      break;
    case types::sib5:
      HANDLE_CODE(c.get<sib_type5_s>().unpack(bref));
      break;
    case types::sib6:
      HANDLE_CODE(c.get<sib_type6_s>().unpack(bref));
      break;
    case types::sib7:
      HANDLE_CODE(c.get<sib_type7_s>().unpack(bref));
      break;
    case types::sib8:
      HANDLE_CODE(c.get<sib_type8_s>().unpack(bref));
      break;
    case types::sib9:
      HANDLE_CODE(c.get<sib_type9_s>().unpack(bref));
      break;
    case types::sib10:
      HANDLE_CODE(c.get<sib_type10_s>().unpack(bref));
      break;
    case types::sib11:
      HANDLE_CODE(c.get<sib_type11_s>().unpack(bref));
      break;
    case types::sib12_v920: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type12_r9_s>().unpack(bref));
    } break;
    case types::sib13_v920: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type13_r9_s>().unpack(bref));
    } break;
    case types::sib14_v1130: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type14_r11_s>().unpack(bref));
    } break;
    case types::sib15_v1130: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type15_r11_s>().unpack(bref));
    } break;
    case types::sib16_v1130: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type16_r11_s>().unpack(bref));
    } break;
    case types::sib17_v1250: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type17_r12_s>().unpack(bref));
    } break;
    case types::sib18_v1250: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type18_r12_s>().unpack(bref));
    } break;
    case types::sib19_v1250: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type19_r12_s>().unpack(bref));
    } break;
    case types::sib20_v1310: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type20_r13_s>().unpack(bref));
    } break;
    case types::sib21_v1430: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type21_r14_s>().unpack(bref));
    } break;
    case types::sib24_v1530: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type24_r15_s>().unpack(bref));
    } break;
    case types::sib25_v1530: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type25_r15_s>().unpack(bref));
    } break;
    case types::sib26_v1530: {
      varlength_field_unpack_guard varlen_scope(bref, false);
      HANDLE_CODE(c.get<sib_type26_r15_s>().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "sib_info_item_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformation-r8-IEs ::= SEQUENCE
SRSASN_CODE sys_info_r8_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, sib_type_and_info, 1, 32));
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_info_r8_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(sib_type_and_info, bref, 1, 32));
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sys_info_r8_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("sib-TypeAndInfo");
  for (const auto& e1 : sib_type_and_info) {
    e1.to_json(j);
  }
  j.end_array();
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformationBlockType1-v890-IEs ::= SEQUENCE
SRSASN_CODE sib_type1_v890_ies_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.pack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_v890_ies_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(late_non_crit_ext_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  if (late_non_crit_ext_present) {
    HANDLE_CODE(late_non_crit_ext.unpack(bref));
  }
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_v890_ies_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (late_non_crit_ext_present) {
    j.write_str("lateNonCriticalExtension", late_non_crit_ext.to_string());
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// SystemInformation ::= SEQUENCE
SRSASN_CODE sys_info_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(crit_exts.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_info_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(crit_exts.unpack(bref));

  return SRSASN_SUCCESS;
}
void sys_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("criticalExtensions");
  crit_exts.to_json(j);
  j.end_obj();
}

void sys_info_s::crit_exts_c_::destroy_()
{
  switch (type_) {
    case types::sys_info_r8:
      c.destroy<sys_info_r8_ies_s>();
      break;
    case types::crit_exts_future_r15:
      c.destroy<crit_exts_future_r15_c_>();
      break;
    default:
      break;
  }
}
void sys_info_s::crit_exts_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sys_info_r8:
      c.init<sys_info_r8_ies_s>();
      break;
    case types::crit_exts_future_r15:
      c.init<crit_exts_future_r15_c_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_");
  }
}
sys_info_s::crit_exts_c_::crit_exts_c_(const sys_info_s::crit_exts_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sys_info_r8:
      c.init(other.c.get<sys_info_r8_ies_s>());
      break;
    case types::crit_exts_future_r15:
      c.init(other.c.get<crit_exts_future_r15_c_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_");
  }
}
sys_info_s::crit_exts_c_& sys_info_s::crit_exts_c_::operator=(const sys_info_s::crit_exts_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sys_info_r8:
      c.set(other.c.get<sys_info_r8_ies_s>());
      break;
    case types::crit_exts_future_r15:
      c.set(other.c.get<crit_exts_future_r15_c_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_");
  }

  return *this;
}
void sys_info_s::crit_exts_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sys_info_r8:
      j.write_fieldname("systemInformation-r8");
      c.get<sys_info_r8_ies_s>().to_json(j);
      break;
    case types::crit_exts_future_r15:
      j.write_fieldname("criticalExtensionsFuture-r15");
      c.get<crit_exts_future_r15_c_>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_");
  }
  j.end_obj();
}
SRSASN_CODE sys_info_s::crit_exts_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sys_info_r8:
      HANDLE_CODE(c.get<sys_info_r8_ies_s>().pack(bref));
      break;
    case types::crit_exts_future_r15:
      HANDLE_CODE(c.get<crit_exts_future_r15_c_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_info_s::crit_exts_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sys_info_r8:
      HANDLE_CODE(c.get<sys_info_r8_ies_s>().unpack(bref));
      break;
    case types::crit_exts_future_r15:
      HANDLE_CODE(c.get<crit_exts_future_r15_c_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::destroy_()
{
  switch (type_) {
    case types::pos_sys_info_r15:
      c.destroy<pos_sys_info_r15_ies_s>();
      break;
    default:
      break;
  }
}
void sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::pos_sys_info_r15:
      c.init<pos_sys_info_r15_ies_s>();
      break;
    case types::crit_exts_future:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_::crit_exts_future_r15_c_");
  }
}
sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::crit_exts_future_r15_c_(
    const sys_info_s::crit_exts_c_::crit_exts_future_r15_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::pos_sys_info_r15:
      c.init(other.c.get<pos_sys_info_r15_ies_s>());
      break;
    case types::crit_exts_future:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_::crit_exts_future_r15_c_");
  }
}
sys_info_s::crit_exts_c_::crit_exts_future_r15_c_& sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::
                                                   operator=(const sys_info_s::crit_exts_c_::crit_exts_future_r15_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::pos_sys_info_r15:
      c.set(other.c.get<pos_sys_info_r15_ies_s>());
      break;
    case types::crit_exts_future:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_::crit_exts_future_r15_c_");
  }

  return *this;
}
void sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::pos_sys_info_r15:
      j.write_fieldname("posSystemInformation-r15");
      c.get<pos_sys_info_r15_ies_s>().to_json(j);
      break;
    case types::crit_exts_future:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_::crit_exts_future_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::pos_sys_info_r15:
      HANDLE_CODE(c.get<pos_sys_info_r15_ies_s>().pack(bref));
      break;
    case types::crit_exts_future:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_::crit_exts_future_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE sys_info_s::crit_exts_c_::crit_exts_future_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::pos_sys_info_r15:
      HANDLE_CODE(c.get<pos_sys_info_r15_ies_s>().unpack(bref));
      break;
    case types::crit_exts_future:
      break;
    default:
      log_invalid_choice_id(type_, "sys_info_s::crit_exts_c_::crit_exts_future_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SystemInformationBlockType1 ::= SEQUENCE
SRSASN_CODE sib_type1_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(p_max_present, 1));
  HANDLE_CODE(bref.pack(tdd_cfg_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  HANDLE_CODE(bref.pack(cell_access_related_info.csg_id_present, 1));
  HANDLE_CODE(pack_dyn_seq_of(bref, cell_access_related_info.plmn_id_list, 1, 6));
  HANDLE_CODE(cell_access_related_info.tac.pack(bref));
  HANDLE_CODE(cell_access_related_info.cell_id.pack(bref));
  HANDLE_CODE(cell_access_related_info.cell_barred.pack(bref));
  HANDLE_CODE(cell_access_related_info.intra_freq_resel.pack(bref));
  HANDLE_CODE(bref.pack(cell_access_related_info.csg_ind, 1));
  if (cell_access_related_info.csg_id_present) {
    HANDLE_CODE(cell_access_related_info.csg_id.pack(bref));
  }
  HANDLE_CODE(bref.pack(cell_sel_info.q_rx_lev_min_offset_present, 1));
  HANDLE_CODE(pack_integer(bref, cell_sel_info.q_rx_lev_min, (int8_t)-70, (int8_t)-22));
  if (cell_sel_info.q_rx_lev_min_offset_present) {
    HANDLE_CODE(pack_integer(bref, cell_sel_info.q_rx_lev_min_offset, (uint8_t)1u, (uint8_t)8u));
  }
  if (p_max_present) {
    HANDLE_CODE(pack_integer(bref, p_max, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(pack_integer(bref, freq_band_ind, (uint8_t)1u, (uint8_t)64u));
  HANDLE_CODE(pack_dyn_seq_of(bref, sched_info_list, 1, 32));
  if (tdd_cfg_present) {
    HANDLE_CODE(tdd_cfg.pack(bref));
  }
  HANDLE_CODE(si_win_len.pack(bref));
  HANDLE_CODE(pack_integer(bref, sys_info_value_tag, (uint8_t)0u, (uint8_t)31u));
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(p_max_present, 1));
  HANDLE_CODE(bref.unpack(tdd_cfg_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  HANDLE_CODE(bref.unpack(cell_access_related_info.csg_id_present, 1));
  HANDLE_CODE(unpack_dyn_seq_of(cell_access_related_info.plmn_id_list, bref, 1, 6));
  HANDLE_CODE(cell_access_related_info.tac.unpack(bref));
  HANDLE_CODE(cell_access_related_info.cell_id.unpack(bref));
  HANDLE_CODE(cell_access_related_info.cell_barred.unpack(bref));
  HANDLE_CODE(cell_access_related_info.intra_freq_resel.unpack(bref));
  HANDLE_CODE(bref.unpack(cell_access_related_info.csg_ind, 1));
  if (cell_access_related_info.csg_id_present) {
    HANDLE_CODE(cell_access_related_info.csg_id.unpack(bref));
  }
  HANDLE_CODE(bref.unpack(cell_sel_info.q_rx_lev_min_offset_present, 1));
  HANDLE_CODE(unpack_integer(cell_sel_info.q_rx_lev_min, bref, (int8_t)-70, (int8_t)-22));
  if (cell_sel_info.q_rx_lev_min_offset_present) {
    HANDLE_CODE(unpack_integer(cell_sel_info.q_rx_lev_min_offset, bref, (uint8_t)1u, (uint8_t)8u));
  }
  if (p_max_present) {
    HANDLE_CODE(unpack_integer(p_max, bref, (int8_t)-30, (int8_t)33));
  }
  HANDLE_CODE(unpack_integer(freq_band_ind, bref, (uint8_t)1u, (uint8_t)64u));
  HANDLE_CODE(unpack_dyn_seq_of(sched_info_list, bref, 1, 32));
  if (tdd_cfg_present) {
    HANDLE_CODE(tdd_cfg.unpack(bref));
  }
  HANDLE_CODE(si_win_len.unpack(bref));
  HANDLE_CODE(unpack_integer(sys_info_value_tag, bref, (uint8_t)0u, (uint8_t)31u));
  if (non_crit_ext_present) {
    HANDLE_CODE(non_crit_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellAccessRelatedInfo");
  j.start_obj();
  j.start_array("plmn-IdentityList");
  for (const auto& e1 : cell_access_related_info.plmn_id_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_str("trackingAreaCode", cell_access_related_info.tac.to_string());
  j.write_str("cellIdentity", cell_access_related_info.cell_id.to_string());
  j.write_str("cellBarred", cell_access_related_info.cell_barred.to_string());
  j.write_str("intraFreqReselection", cell_access_related_info.intra_freq_resel.to_string());
  j.write_bool("csg-Indication", cell_access_related_info.csg_ind);
  if (cell_access_related_info.csg_id_present) {
    j.write_str("csg-Identity", cell_access_related_info.csg_id.to_string());
  }
  j.end_obj();
  j.write_fieldname("cellSelectionInfo");
  j.start_obj();
  j.write_int("q-RxLevMin", cell_sel_info.q_rx_lev_min);
  if (cell_sel_info.q_rx_lev_min_offset_present) {
    j.write_int("q-RxLevMinOffset", cell_sel_info.q_rx_lev_min_offset);
  }
  j.end_obj();
  if (p_max_present) {
    j.write_int("p-Max", p_max);
  }
  j.write_int("freqBandIndicator", freq_band_ind);
  j.start_array("schedulingInfoList");
  for (const auto& e1 : sched_info_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (tdd_cfg_present) {
    j.write_fieldname("tdd-Config");
    tdd_cfg.to_json(j);
  }
  j.write_str("si-WindowLength", si_win_len.to_string());
  j.write_int("systemInfoValueTag", sys_info_value_tag);
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    non_crit_ext.to_json(j);
  }
  j.end_obj();
}

// BCCH-DL-SCH-MessageType ::= CHOICE
void bcch_dl_sch_msg_type_c::destroy_()
{
  switch (type_) {
    case types::c1:
      c.destroy<c1_c_>();
      break;
    default:
      break;
  }
}
void bcch_dl_sch_msg_type_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::c1:
      c.init<c1_c_>();
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c");
  }
}
bcch_dl_sch_msg_type_c::bcch_dl_sch_msg_type_c(const bcch_dl_sch_msg_type_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::c1:
      c.init(other.c.get<c1_c_>());
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c");
  }
}
bcch_dl_sch_msg_type_c& bcch_dl_sch_msg_type_c::operator=(const bcch_dl_sch_msg_type_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::c1:
      c.set(other.c.get<c1_c_>());
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c");
  }

  return *this;
}
void bcch_dl_sch_msg_type_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::c1:
      j.write_fieldname("c1");
      c.get<c1_c_>().to_json(j);
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c");
  }
  j.end_obj();
}
SRSASN_CODE bcch_dl_sch_msg_type_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::c1:
      HANDLE_CODE(c.get<c1_c_>().pack(bref));
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_type_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::c1:
      HANDLE_CODE(c.get<c1_c_>().unpack(bref));
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void bcch_dl_sch_msg_type_c::c1_c_::destroy_()
{
  switch (type_) {
    case types::sys_info:
      c.destroy<sys_info_s>();
      break;
    case types::sib_type1:
      c.destroy<sib_type1_s>();
      break;
    default:
      break;
  }
}
void bcch_dl_sch_msg_type_c::c1_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sys_info:
      c.init<sys_info_s>();
      break;
    case types::sib_type1:
      c.init<sib_type1_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c::c1_c_");
  }
}
bcch_dl_sch_msg_type_c::c1_c_::c1_c_(const bcch_dl_sch_msg_type_c::c1_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sys_info:
      c.init(other.c.get<sys_info_s>());
      break;
    case types::sib_type1:
      c.init(other.c.get<sib_type1_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c::c1_c_");
  }
}
bcch_dl_sch_msg_type_c::c1_c_& bcch_dl_sch_msg_type_c::c1_c_::operator=(const bcch_dl_sch_msg_type_c::c1_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sys_info:
      c.set(other.c.get<sys_info_s>());
      break;
    case types::sib_type1:
      c.set(other.c.get<sib_type1_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c::c1_c_");
  }

  return *this;
}
void bcch_dl_sch_msg_type_c::c1_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sys_info:
      j.write_fieldname("systemInformation");
      c.get<sys_info_s>().to_json(j);
      break;
    case types::sib_type1:
      j.write_fieldname("systemInformationBlockType1");
      c.get<sib_type1_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c::c1_c_");
  }
  j.end_obj();
}
SRSASN_CODE bcch_dl_sch_msg_type_c::c1_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sys_info:
      HANDLE_CODE(c.get<sys_info_s>().pack(bref));
      break;
    case types::sib_type1:
      HANDLE_CODE(c.get<sib_type1_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c::c1_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_type_c::c1_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sys_info:
      HANDLE_CODE(c.get<sys_info_s>().unpack(bref));
      break;
    case types::sib_type1:
      HANDLE_CODE(c.get<sib_type1_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_c::c1_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// BCCH-DL-SCH-Message ::= SEQUENCE
SRSASN_CODE bcch_dl_sch_msg_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(msg.pack(bref));

  bref.align_bytes_zero();

  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(msg.unpack(bref));

  bref.align_bytes();

  return SRSASN_SUCCESS;
}
void bcch_dl_sch_msg_s::to_json(json_writer& j) const
{
  j.start_array();
  j.start_obj();
  j.start_obj("BCCH-DL-SCH-Message");
  j.write_fieldname("message");
  msg.to_json(j);
  j.end_obj();
  j.end_obj();
  j.end_array();
}

// BCCH-DL-SCH-MessageType-BR-r13 ::= CHOICE
void bcch_dl_sch_msg_type_br_r13_c::destroy_()
{
  switch (type_) {
    case types::c1:
      c.destroy<c1_c_>();
      break;
    default:
      break;
  }
}
void bcch_dl_sch_msg_type_br_r13_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::c1:
      c.init<c1_c_>();
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c");
  }
}
bcch_dl_sch_msg_type_br_r13_c::bcch_dl_sch_msg_type_br_r13_c(const bcch_dl_sch_msg_type_br_r13_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::c1:
      c.init(other.c.get<c1_c_>());
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c");
  }
}
bcch_dl_sch_msg_type_br_r13_c& bcch_dl_sch_msg_type_br_r13_c::operator=(const bcch_dl_sch_msg_type_br_r13_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::c1:
      c.set(other.c.get<c1_c_>());
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c");
  }

  return *this;
}
void bcch_dl_sch_msg_type_br_r13_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::c1:
      j.write_fieldname("c1");
      c.get<c1_c_>().to_json(j);
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c");
  }
  j.end_obj();
}
SRSASN_CODE bcch_dl_sch_msg_type_br_r13_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::c1:
      HANDLE_CODE(c.get<c1_c_>().pack(bref));
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_type_br_r13_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::c1:
      HANDLE_CODE(c.get<c1_c_>().unpack(bref));
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void bcch_dl_sch_msg_type_br_r13_c::c1_c_::destroy_()
{
  switch (type_) {
    case types::sys_info_br_r13:
      c.destroy<sys_info_br_r13_s>();
      break;
    case types::sib_type1_br_r13:
      c.destroy<sib_type1_br_r13_s>();
      break;
    default:
      break;
  }
}
void bcch_dl_sch_msg_type_br_r13_c::c1_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sys_info_br_r13:
      c.init<sys_info_br_r13_s>();
      break;
    case types::sib_type1_br_r13:
      c.init<sib_type1_br_r13_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c::c1_c_");
  }
}
bcch_dl_sch_msg_type_br_r13_c::c1_c_::c1_c_(const bcch_dl_sch_msg_type_br_r13_c::c1_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sys_info_br_r13:
      c.init(other.c.get<sys_info_br_r13_s>());
      break;
    case types::sib_type1_br_r13:
      c.init(other.c.get<sib_type1_br_r13_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c::c1_c_");
  }
}
bcch_dl_sch_msg_type_br_r13_c::c1_c_& bcch_dl_sch_msg_type_br_r13_c::c1_c_::
                                      operator=(const bcch_dl_sch_msg_type_br_r13_c::c1_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sys_info_br_r13:
      c.set(other.c.get<sys_info_br_r13_s>());
      break;
    case types::sib_type1_br_r13:
      c.set(other.c.get<sib_type1_br_r13_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c::c1_c_");
  }

  return *this;
}
void bcch_dl_sch_msg_type_br_r13_c::c1_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sys_info_br_r13:
      j.write_fieldname("systemInformation-BR-r13");
      c.get<sys_info_br_r13_s>().to_json(j);
      break;
    case types::sib_type1_br_r13:
      j.write_fieldname("systemInformationBlockType1-BR-r13");
      c.get<sib_type1_br_r13_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c::c1_c_");
  }
  j.end_obj();
}
SRSASN_CODE bcch_dl_sch_msg_type_br_r13_c::c1_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sys_info_br_r13:
      HANDLE_CODE(c.get<sys_info_br_r13_s>().pack(bref));
      break;
    case types::sib_type1_br_r13:
      HANDLE_CODE(c.get<sib_type1_br_r13_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c::c1_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_type_br_r13_c::c1_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sys_info_br_r13:
      HANDLE_CODE(c.get<sys_info_br_r13_s>().unpack(bref));
      break;
    case types::sib_type1_br_r13:
      HANDLE_CODE(c.get<sib_type1_br_r13_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_br_r13_c::c1_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// BCCH-DL-SCH-Message-BR ::= SEQUENCE
SRSASN_CODE bcch_dl_sch_msg_br_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(msg.pack(bref));

  bref.align_bytes_zero();

  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_br_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(msg.unpack(bref));

  bref.align_bytes();

  return SRSASN_SUCCESS;
}
void bcch_dl_sch_msg_br_s::to_json(json_writer& j) const
{
  j.start_array();
  j.start_obj();
  j.start_obj("BCCH-DL-SCH-Message-BR");
  j.write_fieldname("message");
  msg.to_json(j);
  j.end_obj();
  j.end_obj();
  j.end_array();
}

// SchedulingInfo-MBMS-r14 ::= SEQUENCE
SRSASN_CODE sched_info_mbms_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(si_periodicity_r14.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, sib_map_info_r14, 0, 31));

  return SRSASN_SUCCESS;
}
SRSASN_CODE sched_info_mbms_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(si_periodicity_r14.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(sib_map_info_r14, bref, 0, 31));

  return SRSASN_SUCCESS;
}
void sched_info_mbms_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("si-Periodicity-r14", si_periodicity_r14.to_string());
  j.start_array("sib-MappingInfo-r14");
  for (const auto& e1 : sib_map_info_r14) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.end_obj();
}

// NonMBSFN-SubframeConfig-r14 ::= SEQUENCE
SRSASN_CODE non_mbsfn_sf_cfg_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(radio_frame_alloc_period_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, radio_frame_alloc_offset_r14, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(sf_alloc_r14.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE non_mbsfn_sf_cfg_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(radio_frame_alloc_period_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(radio_frame_alloc_offset_r14, bref, (uint8_t)0u, (uint8_t)7u));
  HANDLE_CODE(sf_alloc_r14.unpack(bref));

  return SRSASN_SUCCESS;
}
void non_mbsfn_sf_cfg_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("radioFrameAllocationPeriod-r14", radio_frame_alloc_period_r14.to_string());
  j.write_int("radioFrameAllocationOffset-r14", radio_frame_alloc_offset_r14);
  j.write_str("subframeAllocation-r14", sf_alloc_r14.to_string());
  j.end_obj();
}

// SystemInformationBlockType1-MBMS-r14 ::= SEQUENCE
SRSASN_CODE sib_type1_mbms_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(multi_band_info_list_r14_present, 1));
  HANDLE_CODE(bref.pack(non_mbsfn_sf_cfg_r14_present, 1));
  HANDLE_CODE(bref.pack(sib_type13_r14_present, 1));
  HANDLE_CODE(bref.pack(cell_access_related_info_list_r14_present, 1));
  HANDLE_CODE(bref.pack(non_crit_ext_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, cell_access_related_info_r14.plmn_id_list_r14, 1, 6));
  HANDLE_CODE(cell_access_related_info_r14.tac_r14.pack(bref));
  HANDLE_CODE(cell_access_related_info_r14.cell_id_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, freq_band_ind_r14, (uint16_t)1u, (uint16_t)256u));
  if (multi_band_info_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, multi_band_info_list_r14, 1, 8, integer_packer<uint16_t>(1, 256)));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, sched_info_list_mbms_r14, 1, 32));
  HANDLE_CODE(si_win_len_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, sys_info_value_tag_r14, (uint8_t)0u, (uint8_t)31u));
  if (non_mbsfn_sf_cfg_r14_present) {
    HANDLE_CODE(non_mbsfn_sf_cfg_r14.pack(bref));
  }
  HANDLE_CODE(pdsch_cfg_common_r14.pack(bref));
  if (sib_type13_r14_present) {
    HANDLE_CODE(sib_type13_r14.pack(bref));
  }
  if (cell_access_related_info_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, cell_access_related_info_list_r14, 1, 5));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sib_type1_mbms_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(multi_band_info_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(non_mbsfn_sf_cfg_r14_present, 1));
  HANDLE_CODE(bref.unpack(sib_type13_r14_present, 1));
  HANDLE_CODE(bref.unpack(cell_access_related_info_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(non_crit_ext_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(cell_access_related_info_r14.plmn_id_list_r14, bref, 1, 6));
  HANDLE_CODE(cell_access_related_info_r14.tac_r14.unpack(bref));
  HANDLE_CODE(cell_access_related_info_r14.cell_id_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(freq_band_ind_r14, bref, (uint16_t)1u, (uint16_t)256u));
  if (multi_band_info_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(multi_band_info_list_r14, bref, 1, 8, integer_packer<uint16_t>(1, 256)));
  }
  HANDLE_CODE(unpack_dyn_seq_of(sched_info_list_mbms_r14, bref, 1, 32));
  HANDLE_CODE(si_win_len_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(sys_info_value_tag_r14, bref, (uint8_t)0u, (uint8_t)31u));
  if (non_mbsfn_sf_cfg_r14_present) {
    HANDLE_CODE(non_mbsfn_sf_cfg_r14.unpack(bref));
  }
  HANDLE_CODE(pdsch_cfg_common_r14.unpack(bref));
  if (sib_type13_r14_present) {
    HANDLE_CODE(sib_type13_r14.unpack(bref));
  }
  if (cell_access_related_info_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(cell_access_related_info_list_r14, bref, 1, 5));
  }

  return SRSASN_SUCCESS;
}
void sib_type1_mbms_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellAccessRelatedInfo-r14");
  j.start_obj();
  j.start_array("plmn-IdentityList-r14");
  for (const auto& e1 : cell_access_related_info_r14.plmn_id_list_r14) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_str("trackingAreaCode-r14", cell_access_related_info_r14.tac_r14.to_string());
  j.write_str("cellIdentity-r14", cell_access_related_info_r14.cell_id_r14.to_string());
  j.end_obj();
  j.write_int("freqBandIndicator-r14", freq_band_ind_r14);
  if (multi_band_info_list_r14_present) {
    j.start_array("multiBandInfoList-r14");
    for (const auto& e1 : multi_band_info_list_r14) {
      j.write_int(e1);
    }
    j.end_array();
  }
  j.start_array("schedulingInfoList-MBMS-r14");
  for (const auto& e1 : sched_info_list_mbms_r14) {
    e1.to_json(j);
  }
  j.end_array();
  j.write_str("si-WindowLength-r14", si_win_len_r14.to_string());
  j.write_int("systemInfoValueTag-r14", sys_info_value_tag_r14);
  if (non_mbsfn_sf_cfg_r14_present) {
    j.write_fieldname("nonMBSFN-SubframeConfig-r14");
    non_mbsfn_sf_cfg_r14.to_json(j);
  }
  j.write_fieldname("pdsch-ConfigCommon-r14");
  pdsch_cfg_common_r14.to_json(j);
  if (sib_type13_r14_present) {
    j.write_fieldname("systemInformationBlockType13-r14");
    sib_type13_r14.to_json(j);
  }
  if (cell_access_related_info_list_r14_present) {
    j.start_array("cellAccessRelatedInfoList-r14");
    for (const auto& e1 : cell_access_related_info_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (non_crit_ext_present) {
    j.write_fieldname("nonCriticalExtension");
    j.start_obj();
    j.end_obj();
  }
  j.end_obj();
}

// BCCH-DL-SCH-MessageType-MBMS-r14 ::= CHOICE
void bcch_dl_sch_msg_type_mbms_r14_c::destroy_()
{
  switch (type_) {
    case types::c1:
      c.destroy<c1_c_>();
      break;
    default:
      break;
  }
}
void bcch_dl_sch_msg_type_mbms_r14_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::c1:
      c.init<c1_c_>();
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c");
  }
}
bcch_dl_sch_msg_type_mbms_r14_c::bcch_dl_sch_msg_type_mbms_r14_c(const bcch_dl_sch_msg_type_mbms_r14_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::c1:
      c.init(other.c.get<c1_c_>());
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c");
  }
}
bcch_dl_sch_msg_type_mbms_r14_c& bcch_dl_sch_msg_type_mbms_r14_c::
                                 operator=(const bcch_dl_sch_msg_type_mbms_r14_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::c1:
      c.set(other.c.get<c1_c_>());
      break;
    case types::msg_class_ext:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c");
  }

  return *this;
}
void bcch_dl_sch_msg_type_mbms_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::c1:
      j.write_fieldname("c1");
      c.get<c1_c_>().to_json(j);
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE bcch_dl_sch_msg_type_mbms_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::c1:
      HANDLE_CODE(c.get<c1_c_>().pack(bref));
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_type_mbms_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::c1:
      HANDLE_CODE(c.get<c1_c_>().unpack(bref));
      break;
    case types::msg_class_ext:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::destroy_()
{
  switch (type_) {
    case types::sys_info_mbms_r14:
      c.destroy<sys_info_mbms_r14_s>();
      break;
    case types::sib_type1_mbms_r14:
      c.destroy<sib_type1_mbms_r14_s>();
      break;
    default:
      break;
  }
}
void bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sys_info_mbms_r14:
      c.init<sys_info_mbms_r14_s>();
      break;
    case types::sib_type1_mbms_r14:
      c.init<sib_type1_mbms_r14_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c::c1_c_");
  }
}
bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::c1_c_(const bcch_dl_sch_msg_type_mbms_r14_c::c1_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sys_info_mbms_r14:
      c.init(other.c.get<sys_info_mbms_r14_s>());
      break;
    case types::sib_type1_mbms_r14:
      c.init(other.c.get<sib_type1_mbms_r14_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c::c1_c_");
  }
}
bcch_dl_sch_msg_type_mbms_r14_c::c1_c_& bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::
                                        operator=(const bcch_dl_sch_msg_type_mbms_r14_c::c1_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sys_info_mbms_r14:
      c.set(other.c.get<sys_info_mbms_r14_s>());
      break;
    case types::sib_type1_mbms_r14:
      c.set(other.c.get<sib_type1_mbms_r14_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c::c1_c_");
  }

  return *this;
}
void bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sys_info_mbms_r14:
      j.write_fieldname("systemInformation-MBMS-r14");
      c.get<sys_info_mbms_r14_s>().to_json(j);
      break;
    case types::sib_type1_mbms_r14:
      j.write_fieldname("systemInformationBlockType1-MBMS-r14");
      c.get<sib_type1_mbms_r14_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c::c1_c_");
  }
  j.end_obj();
}
SRSASN_CODE bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sys_info_mbms_r14:
      HANDLE_CODE(c.get<sys_info_mbms_r14_s>().pack(bref));
      break;
    case types::sib_type1_mbms_r14:
      HANDLE_CODE(c.get<sib_type1_mbms_r14_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c::c1_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_type_mbms_r14_c::c1_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sys_info_mbms_r14:
      HANDLE_CODE(c.get<sys_info_mbms_r14_s>().unpack(bref));
      break;
    case types::sib_type1_mbms_r14:
      HANDLE_CODE(c.get<sib_type1_mbms_r14_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "bcch_dl_sch_msg_type_mbms_r14_c::c1_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// BCCH-DL-SCH-Message-MBMS ::= SEQUENCE
SRSASN_CODE bcch_dl_sch_msg_mbms_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(msg.pack(bref));

  bref.align_bytes_zero();

  return SRSASN_SUCCESS;
}
SRSASN_CODE bcch_dl_sch_msg_mbms_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(msg.unpack(bref));

  bref.align_bytes();

  return SRSASN_SUCCESS;
}
void bcch_dl_sch_msg_mbms_s::to_json(json_writer& j) const
{
  j.start_array();
  j.start_obj();
  j.start_obj("BCCH-DL-SCH-Message-MBMS");
  j.write_fieldname("message");
  msg.to_json(j);
  j.end_obj();
  j.end_obj();
  j.end_array();
}

// P-C-AndCBSR-r13 ::= SEQUENCE
SRSASN_CODE p_c_and_cbsr_r13_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, p_c_r13, (int8_t)-8, (int8_t)15));
  HANDLE_CODE(cbsr_sel_r13.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE p_c_and_cbsr_r13_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(p_c_r13, bref, (int8_t)-8, (int8_t)15));
  HANDLE_CODE(cbsr_sel_r13.unpack(bref));

  return SRSASN_SUCCESS;
}
void p_c_and_cbsr_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("p-C-r13", p_c_r13);
  j.write_fieldname("cbsr-Selection-r13");
  cbsr_sel_r13.to_json(j);
  j.end_obj();
}

void p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::destroy_()
{
  switch (type_) {
    case types::non_precoded_r13:
      c.destroy<non_precoded_r13_s_>();
      break;
    case types::bf_k1a_r13:
      c.destroy<bf_k1a_r13_s_>();
      break;
    case types::bf_kn_r13:
      c.destroy<bf_kn_r13_s_>();
      break;
    default:
      break;
  }
}
void p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::non_precoded_r13:
      c.init<non_precoded_r13_s_>();
      break;
    case types::bf_k1a_r13:
      c.init<bf_k1a_r13_s_>();
      break;
    case types::bf_kn_r13:
      c.init<bf_kn_r13_s_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "p_c_and_cbsr_r13_s::cbsr_sel_r13_c_");
  }
}
p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::cbsr_sel_r13_c_(const p_c_and_cbsr_r13_s::cbsr_sel_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::non_precoded_r13:
      c.init(other.c.get<non_precoded_r13_s_>());
      break;
    case types::bf_k1a_r13:
      c.init(other.c.get<bf_k1a_r13_s_>());
      break;
    case types::bf_kn_r13:
      c.init(other.c.get<bf_kn_r13_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "p_c_and_cbsr_r13_s::cbsr_sel_r13_c_");
  }
}
p_c_and_cbsr_r13_s::cbsr_sel_r13_c_& p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::
                                     operator=(const p_c_and_cbsr_r13_s::cbsr_sel_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::non_precoded_r13:
      c.set(other.c.get<non_precoded_r13_s_>());
      break;
    case types::bf_k1a_r13:
      c.set(other.c.get<bf_k1a_r13_s_>());
      break;
    case types::bf_kn_r13:
      c.set(other.c.get<bf_kn_r13_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "p_c_and_cbsr_r13_s::cbsr_sel_r13_c_");
  }

  return *this;
}
void p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::non_precoded_r13:
      j.write_fieldname("nonPrecoded-r13");
      j.start_obj();
      j.write_str("codebookSubsetRestriction1-r13",
                  c.get<non_precoded_r13_s_>().codebook_subset_restrict1_r13.to_string());
      j.write_str("codebookSubsetRestriction2-r13",
                  c.get<non_precoded_r13_s_>().codebook_subset_restrict2_r13.to_string());
      j.end_obj();
      break;
    case types::bf_k1a_r13:
      j.write_fieldname("beamformedK1a-r13");
      j.start_obj();
      j.write_str("codebookSubsetRestriction3-r13", c.get<bf_k1a_r13_s_>().codebook_subset_restrict3_r13.to_string());
      j.end_obj();
      break;
    case types::bf_kn_r13:
      j.write_fieldname("beamformedKN-r13");
      j.start_obj();
      j.write_str("codebookSubsetRestriction-r13", c.get<bf_kn_r13_s_>().codebook_subset_restrict_r13.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "p_c_and_cbsr_r13_s::cbsr_sel_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::non_precoded_r13:
      HANDLE_CODE(c.get<non_precoded_r13_s_>().codebook_subset_restrict1_r13.pack(bref));
      HANDLE_CODE(c.get<non_precoded_r13_s_>().codebook_subset_restrict2_r13.pack(bref));
      break;
    case types::bf_k1a_r13:
      HANDLE_CODE(c.get<bf_k1a_r13_s_>().codebook_subset_restrict3_r13.pack(bref));
      break;
    case types::bf_kn_r13:
      HANDLE_CODE(c.get<bf_kn_r13_s_>().codebook_subset_restrict_r13.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "p_c_and_cbsr_r13_s::cbsr_sel_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE p_c_and_cbsr_r13_s::cbsr_sel_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::non_precoded_r13:
      HANDLE_CODE(c.get<non_precoded_r13_s_>().codebook_subset_restrict1_r13.unpack(bref));
      HANDLE_CODE(c.get<non_precoded_r13_s_>().codebook_subset_restrict2_r13.unpack(bref));
      break;
    case types::bf_k1a_r13:
      HANDLE_CODE(c.get<bf_k1a_r13_s_>().codebook_subset_restrict3_r13.unpack(bref));
      break;
    case types::bf_kn_r13:
      HANDLE_CODE(c.get<bf_kn_r13_s_>().codebook_subset_restrict_r13.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "p_c_and_cbsr_r13_s::cbsr_sel_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-Config-NZP-v1430 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_nzp_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(tx_comb_r14_present, 1));
  HANDLE_CODE(bref.pack(freq_density_r14_present, 1));

  if (tx_comb_r14_present) {
    HANDLE_CODE(pack_integer(bref, tx_comb_r14, (uint8_t)0u, (uint8_t)2u));
  }
  if (freq_density_r14_present) {
    HANDLE_CODE(freq_density_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(tx_comb_r14_present, 1));
  HANDLE_CODE(bref.unpack(freq_density_r14_present, 1));

  if (tx_comb_r14_present) {
    HANDLE_CODE(unpack_integer(tx_comb_r14, bref, (uint8_t)0u, (uint8_t)2u));
  }
  if (freq_density_r14_present) {
    HANDLE_CODE(freq_density_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_nzp_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (tx_comb_r14_present) {
    j.write_int("transmissionComb-r14", tx_comb_r14);
  }
  if (freq_density_r14_present) {
    j.write_str("frequencyDensity-r14", freq_density_r14.to_string());
  }
  j.end_obj();
}

// CSI-RS-ConfigNZP-Activation-r14 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_nzp_activation_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(csi_rs_nzp_mode_r14.pack(bref));
  HANDLE_CODE(pack_integer(bref, activ_res_r14, (uint8_t)0u, (uint8_t)4u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_activation_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(csi_rs_nzp_mode_r14.unpack(bref));
  HANDLE_CODE(unpack_integer(activ_res_r14, bref, (uint8_t)0u, (uint8_t)4u));

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_nzp_activation_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("csi-RS-NZP-mode-r14", csi_rs_nzp_mode_r14.to_string());
  j.write_int("activatedResources-r14", activ_res_r14);
  j.end_obj();
}

// CSI-RS-ConfigNZP-r11 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_nzp_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(qcl_crs_info_r11_present, 1));

  HANDLE_CODE(pack_integer(bref, csi_rs_cfg_nzp_id_r11, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(ant_ports_count_r11.pack(bref));
  HANDLE_CODE(pack_integer(bref, res_cfg_r11, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, sf_cfg_r11, (uint8_t)0u, (uint8_t)154u));
  HANDLE_CODE(pack_integer(bref, scrambling_id_r11, (uint16_t)0u, (uint16_t)503u));
  if (qcl_crs_info_r11_present) {
    HANDLE_CODE(bref.pack(qcl_crs_info_r11.mbsfn_sf_cfg_list_r11_present, 1));
    HANDLE_CODE(pack_integer(bref, qcl_crs_info_r11.qcl_scrambling_id_r11, (uint16_t)0u, (uint16_t)503u));
    HANDLE_CODE(qcl_crs_info_r11.crs_ports_count_r11.pack(bref));
    if (qcl_crs_info_r11.mbsfn_sf_cfg_list_r11_present) {
      HANDLE_CODE(qcl_crs_info_r11.mbsfn_sf_cfg_list_r11.pack(bref));
    }
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= csi_rs_cfg_nzp_id_v1310_present;
    group_flags[1] |= tx_comb_r14_present;
    group_flags[1] |= freq_density_r14_present;
    group_flags[2] |= mbsfn_sf_cfg_list_v1430.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_id_v1310_present, 1));
      if (csi_rs_cfg_nzp_id_v1310_present) {
        HANDLE_CODE(pack_integer(bref, csi_rs_cfg_nzp_id_v1310, (uint8_t)4u, (uint8_t)24u));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(tx_comb_r14_present, 1));
      HANDLE_CODE(bref.pack(freq_density_r14_present, 1));
      if (tx_comb_r14_present) {
        HANDLE_CODE(pack_integer(bref, tx_comb_r14, (uint8_t)0u, (uint8_t)2u));
      }
      if (freq_density_r14_present) {
        HANDLE_CODE(freq_density_r14.pack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_v1430.is_present(), 1));
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(mbsfn_sf_cfg_list_v1430->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(qcl_crs_info_r11_present, 1));

  HANDLE_CODE(unpack_integer(csi_rs_cfg_nzp_id_r11, bref, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(ant_ports_count_r11.unpack(bref));
  HANDLE_CODE(unpack_integer(res_cfg_r11, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(sf_cfg_r11, bref, (uint8_t)0u, (uint8_t)154u));
  HANDLE_CODE(unpack_integer(scrambling_id_r11, bref, (uint16_t)0u, (uint16_t)503u));
  if (qcl_crs_info_r11_present) {
    HANDLE_CODE(bref.unpack(qcl_crs_info_r11.mbsfn_sf_cfg_list_r11_present, 1));
    HANDLE_CODE(unpack_integer(qcl_crs_info_r11.qcl_scrambling_id_r11, bref, (uint16_t)0u, (uint16_t)503u));
    HANDLE_CODE(qcl_crs_info_r11.crs_ports_count_r11.unpack(bref));
    if (qcl_crs_info_r11.mbsfn_sf_cfg_list_r11_present) {
      HANDLE_CODE(qcl_crs_info_r11.mbsfn_sf_cfg_list_r11.unpack(bref));
    }
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(3);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_id_v1310_present, 1));
      if (csi_rs_cfg_nzp_id_v1310_present) {
        HANDLE_CODE(unpack_integer(csi_rs_cfg_nzp_id_v1310, bref, (uint8_t)4u, (uint8_t)24u));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(tx_comb_r14_present, 1));
      HANDLE_CODE(bref.unpack(freq_density_r14_present, 1));
      if (tx_comb_r14_present) {
        HANDLE_CODE(unpack_integer(tx_comb_r14, bref, (uint8_t)0u, (uint8_t)2u));
      }
      if (freq_density_r14_present) {
        HANDLE_CODE(freq_density_r14.unpack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbsfn_sf_cfg_list_v1430_present;
      HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_v1430_present, 1));
      mbsfn_sf_cfg_list_v1430.set_present(mbsfn_sf_cfg_list_v1430_present);
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(mbsfn_sf_cfg_list_v1430->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void csi_rs_cfg_nzp_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("csi-RS-ConfigNZPId-r11", csi_rs_cfg_nzp_id_r11);
  j.write_str("antennaPortsCount-r11", ant_ports_count_r11.to_string());
  j.write_int("resourceConfig-r11", res_cfg_r11);
  j.write_int("subframeConfig-r11", sf_cfg_r11);
  j.write_int("scramblingIdentity-r11", scrambling_id_r11);
  if (qcl_crs_info_r11_present) {
    j.write_fieldname("qcl-CRS-Info-r11");
    j.start_obj();
    j.write_int("qcl-ScramblingIdentity-r11", qcl_crs_info_r11.qcl_scrambling_id_r11);
    j.write_str("crs-PortsCount-r11", qcl_crs_info_r11.crs_ports_count_r11.to_string());
    if (qcl_crs_info_r11.mbsfn_sf_cfg_list_r11_present) {
      j.write_fieldname("mbsfn-SubframeConfigList-r11");
      qcl_crs_info_r11.mbsfn_sf_cfg_list_r11.to_json(j);
    }
    j.end_obj();
  }
  if (ext) {
    if (csi_rs_cfg_nzp_id_v1310_present) {
      j.write_int("csi-RS-ConfigNZPId-v1310", csi_rs_cfg_nzp_id_v1310);
    }
    if (tx_comb_r14_present) {
      j.write_int("transmissionComb-r14", tx_comb_r14);
    }
    if (freq_density_r14_present) {
      j.write_str("frequencyDensity-r14", freq_density_r14.to_string());
    }
    if (mbsfn_sf_cfg_list_v1430.is_present()) {
      j.write_fieldname("mbsfn-SubframeConfigList-v1430");
      mbsfn_sf_cfg_list_v1430->to_json(j);
    }
  }
  j.end_obj();
}

void csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("subframeConfigList");
      for (const auto& e1 : c.sf_cfg_list) {
        e1.to_json(j);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.sf_cfg_list, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.sf_cfg_list, bref, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_r11_s::qcl_crs_info_r11_s_::mbsfn_sf_cfg_list_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("subframeConfigList-v1430");
      for (const auto& e1 : c.sf_cfg_list_v1430) {
        e1.to_json(j);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.sf_cfg_list_v1430, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.sf_cfg_list_v1430, bref, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_r11_s::mbsfn_sf_cfg_list_v1430_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// NZP-ResourceConfig-r13 ::= SEQUENCE
SRSASN_CODE nzp_res_cfg_r13_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, res_cfg_r13, (uint8_t)0u, (uint8_t)31u));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= tx_comb_r14_present;
    group_flags[0] |= freq_density_r14_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(tx_comb_r14_present, 1));
      HANDLE_CODE(bref.pack(freq_density_r14_present, 1));
      if (tx_comb_r14_present) {
        HANDLE_CODE(pack_integer(bref, tx_comb_r14, (uint8_t)0u, (uint8_t)2u));
      }
      if (freq_density_r14_present) {
        HANDLE_CODE(freq_density_r14.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE nzp_res_cfg_r13_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(res_cfg_r13, bref, (uint8_t)0u, (uint8_t)31u));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(tx_comb_r14_present, 1));
      HANDLE_CODE(bref.unpack(freq_density_r14_present, 1));
      if (tx_comb_r14_present) {
        HANDLE_CODE(unpack_integer(tx_comb_r14, bref, (uint8_t)0u, (uint8_t)2u));
      }
      if (freq_density_r14_present) {
        HANDLE_CODE(freq_density_r14.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void nzp_res_cfg_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("resourceConfig-r13", res_cfg_r13);
  if (ext) {
    if (tx_comb_r14_present) {
      j.write_int("transmissionComb-r14", tx_comb_r14);
    }
    if (freq_density_r14_present) {
      j.write_str("frequencyDensity-r14", freq_density_r14.to_string());
    }
  }
  j.end_obj();
}

// P-C-AndCBSR-r15 ::= SEQUENCE
SRSASN_CODE p_c_and_cbsr_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, p_c_r15, (int8_t)-8, (int8_t)15));
  HANDLE_CODE(codebook_subset_restrict4_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE p_c_and_cbsr_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(p_c_r15, bref, (int8_t)-8, (int8_t)15));
  HANDLE_CODE(codebook_subset_restrict4_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void p_c_and_cbsr_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("p-C-r15", p_c_r15);
  j.write_str("codebookSubsetRestriction4-r15", codebook_subset_restrict4_r15.to_string());
  j.end_obj();
}

// CSI-RS-ConfigBeamformed-r14 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_bf_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_id_list_ext_r14_present, 1));
  HANDLE_CODE(bref.pack(csi_im_cfg_id_list_r14_present, 1));
  HANDLE_CODE(bref.pack(p_c_and_cbsr_per_res_cfg_list_r14_present, 1));
  HANDLE_CODE(bref.pack(ace_for4_tx_per_res_cfg_list_r14_present, 1));
  HANDLE_CODE(bref.pack(alt_codebook_enabled_bf_r14_present, 1));
  HANDLE_CODE(bref.pack(ch_meas_restrict_r14_present, 1));
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_ap_list_r14_present, 1));
  HANDLE_CODE(bref.pack(nzp_res_cfg_original_v1430_present, 1));
  HANDLE_CODE(bref.pack(csi_rs_nzp_activation_r14_present, 1));

  if (csi_rs_cfg_nzp_id_list_ext_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_rs_cfg_nzp_id_list_ext_r14, 1, 7, integer_packer<uint8_t>(1, 24)));
  }
  if (csi_im_cfg_id_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_id_list_r14, 1, 8, integer_packer<uint8_t>(1, 24)));
  }
  if (p_c_and_cbsr_per_res_cfg_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, p_c_and_cbsr_per_res_cfg_list_r14, 1, 8, SeqOfPacker<Packer>(1, 2, Packer())));
  }
  if (ace_for4_tx_per_res_cfg_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ace_for4_tx_per_res_cfg_list_r14, 1, 7, BitPacker(1)));
  }
  if (csi_rs_cfg_nzp_ap_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_rs_cfg_nzp_ap_list_r14, 1, 8));
  }
  if (nzp_res_cfg_original_v1430_present) {
    HANDLE_CODE(nzp_res_cfg_original_v1430.pack(bref));
  }
  if (csi_rs_nzp_activation_r14_present) {
    HANDLE_CODE(csi_rs_nzp_activation_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_bf_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_id_list_ext_r14_present, 1));
  HANDLE_CODE(bref.unpack(csi_im_cfg_id_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(p_c_and_cbsr_per_res_cfg_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(ace_for4_tx_per_res_cfg_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(alt_codebook_enabled_bf_r14_present, 1));
  HANDLE_CODE(bref.unpack(ch_meas_restrict_r14_present, 1));
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_ap_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(nzp_res_cfg_original_v1430_present, 1));
  HANDLE_CODE(bref.unpack(csi_rs_nzp_activation_r14_present, 1));

  if (csi_rs_cfg_nzp_id_list_ext_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_rs_cfg_nzp_id_list_ext_r14, bref, 1, 7, integer_packer<uint8_t>(1, 24)));
  }
  if (csi_im_cfg_id_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_id_list_r14, bref, 1, 8, integer_packer<uint8_t>(1, 24)));
  }
  if (p_c_and_cbsr_per_res_cfg_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(p_c_and_cbsr_per_res_cfg_list_r14, bref, 1, 8, SeqOfPacker<Packer>(1, 2, Packer())));
  }
  if (ace_for4_tx_per_res_cfg_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ace_for4_tx_per_res_cfg_list_r14, bref, 1, 7, BitPacker(1)));
  }
  if (csi_rs_cfg_nzp_ap_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_rs_cfg_nzp_ap_list_r14, bref, 1, 8));
  }
  if (nzp_res_cfg_original_v1430_present) {
    HANDLE_CODE(nzp_res_cfg_original_v1430.unpack(bref));
  }
  if (csi_rs_nzp_activation_r14_present) {
    HANDLE_CODE(csi_rs_nzp_activation_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_bf_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_rs_cfg_nzp_id_list_ext_r14_present) {
    j.start_array("csi-RS-ConfigNZPIdListExt-r14");
    for (const auto& e1 : csi_rs_cfg_nzp_id_list_ext_r14) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (csi_im_cfg_id_list_r14_present) {
    j.start_array("csi-IM-ConfigIdList-r14");
    for (const auto& e1 : csi_im_cfg_id_list_r14) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (p_c_and_cbsr_per_res_cfg_list_r14_present) {
    j.start_array("p-C-AndCBSR-PerResourceConfigList-r14");
    for (const auto& e1 : p_c_and_cbsr_per_res_cfg_list_r14) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  if (ace_for4_tx_per_res_cfg_list_r14_present) {
    j.start_array("ace-For4Tx-PerResourceConfigList-r14");
    for (const auto& e1 : ace_for4_tx_per_res_cfg_list_r14) {
      j.write_bool(e1);
    }
    j.end_array();
  }
  if (alt_codebook_enabled_bf_r14_present) {
    j.write_str("alternativeCodebookEnabledBeamformed-r14", "true");
  }
  if (ch_meas_restrict_r14_present) {
    j.write_str("channelMeasRestriction-r14", "on");
  }
  if (csi_rs_cfg_nzp_ap_list_r14_present) {
    j.start_array("csi-RS-ConfigNZP-ApList-r14");
    for (const auto& e1 : csi_rs_cfg_nzp_ap_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (nzp_res_cfg_original_v1430_present) {
    j.write_fieldname("nzp-ResourceConfigOriginal-v1430");
    nzp_res_cfg_original_v1430.to_json(j);
  }
  if (csi_rs_nzp_activation_r14_present) {
    j.write_fieldname("csi-RS-NZP-Activation-r14");
    csi_rs_nzp_activation_r14.to_json(j);
  }
  j.end_obj();
}

// CSI-RS-ConfigNZP-EMIMO-r13 ::= CHOICE
void csi_rs_cfg_nzp_emimo_r13_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_nzp_emimo_r13_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("nzp-resourceConfigList-r13");
      for (const auto& e1 : c.nzp_res_cfg_list_r13) {
        e1.to_json(j);
      }
      j.end_array();
      if (c.cdm_type_r13_present) {
        j.write_str("cdmType-r13", c.cdm_type_r13.to_string());
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_emimo_r13_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_nzp_emimo_r13_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.cdm_type_r13_present, 1));
      HANDLE_CODE(pack_dyn_seq_of(bref, c.nzp_res_cfg_list_r13, 1, 2));
      if (c.cdm_type_r13_present) {
        HANDLE_CODE(c.cdm_type_r13.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_emimo_r13_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_emimo_r13_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.cdm_type_r13_present, 1));
      HANDLE_CODE(unpack_dyn_seq_of(c.nzp_res_cfg_list_r13, bref, 1, 2));
      if (c.cdm_type_r13_present) {
        HANDLE_CODE(c.cdm_type_r13.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_nzp_emimo_r13_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigNZP-EMIMO-v1430 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_nzp_emimo_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cdm_type_v1430_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, nzp_res_cfg_list_ext_r14, 0, 4));

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_nzp_emimo_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cdm_type_v1430_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(nzp_res_cfg_list_ext_r14, bref, 0, 4));

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_nzp_emimo_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("nzp-resourceConfigListExt-r14");
  for (const auto& e1 : nzp_res_cfg_list_ext_r14) {
    e1.to_json(j);
  }
  j.end_array();
  if (cdm_type_v1430_present) {
    j.write_str("cdmType-v1430", "cdm8");
  }
  j.end_obj();
}

// CSI-RS-ConfigBeamformed-r13 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_bf_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_id_list_ext_r13_present, 1));
  HANDLE_CODE(bref.pack(csi_im_cfg_id_list_r13_present, 1));
  HANDLE_CODE(bref.pack(p_c_and_cbsr_per_res_cfg_list_r13_present, 1));
  HANDLE_CODE(bref.pack(ace_for4_tx_per_res_cfg_list_r13_present, 1));
  HANDLE_CODE(bref.pack(alt_codebook_enabled_bf_r13_present, 1));
  HANDLE_CODE(bref.pack(ch_meas_restrict_r13_present, 1));

  if (csi_rs_cfg_nzp_id_list_ext_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_rs_cfg_nzp_id_list_ext_r13, 1, 7, integer_packer<uint8_t>(1, 24)));
  }
  if (csi_im_cfg_id_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_id_list_r13, 1, 8, integer_packer<uint8_t>(1, 24)));
  }
  if (p_c_and_cbsr_per_res_cfg_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, p_c_and_cbsr_per_res_cfg_list_r13, 1, 8, SeqOfPacker<Packer>(1, 2, Packer())));
  }
  if (ace_for4_tx_per_res_cfg_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ace_for4_tx_per_res_cfg_list_r13, 1, 7, BitPacker(1)));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_bf_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_id_list_ext_r13_present, 1));
  HANDLE_CODE(bref.unpack(csi_im_cfg_id_list_r13_present, 1));
  HANDLE_CODE(bref.unpack(p_c_and_cbsr_per_res_cfg_list_r13_present, 1));
  HANDLE_CODE(bref.unpack(ace_for4_tx_per_res_cfg_list_r13_present, 1));
  HANDLE_CODE(bref.unpack(alt_codebook_enabled_bf_r13_present, 1));
  HANDLE_CODE(bref.unpack(ch_meas_restrict_r13_present, 1));

  if (csi_rs_cfg_nzp_id_list_ext_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_rs_cfg_nzp_id_list_ext_r13, bref, 1, 7, integer_packer<uint8_t>(1, 24)));
  }
  if (csi_im_cfg_id_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_id_list_r13, bref, 1, 8, integer_packer<uint8_t>(1, 24)));
  }
  if (p_c_and_cbsr_per_res_cfg_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(p_c_and_cbsr_per_res_cfg_list_r13, bref, 1, 8, SeqOfPacker<Packer>(1, 2, Packer())));
  }
  if (ace_for4_tx_per_res_cfg_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ace_for4_tx_per_res_cfg_list_r13, bref, 1, 7, BitPacker(1)));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_bf_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_rs_cfg_nzp_id_list_ext_r13_present) {
    j.start_array("csi-RS-ConfigNZPIdListExt-r13");
    for (const auto& e1 : csi_rs_cfg_nzp_id_list_ext_r13) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (csi_im_cfg_id_list_r13_present) {
    j.start_array("csi-IM-ConfigIdList-r13");
    for (const auto& e1 : csi_im_cfg_id_list_r13) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (p_c_and_cbsr_per_res_cfg_list_r13_present) {
    j.start_array("p-C-AndCBSR-PerResourceConfigList-r13");
    for (const auto& e1 : p_c_and_cbsr_per_res_cfg_list_r13) {
      j.start_array();
      for (const auto& e2 : e1) {
        e2.to_json(j);
      }
      j.end_array();
    }
    j.end_array();
  }
  if (ace_for4_tx_per_res_cfg_list_r13_present) {
    j.start_array("ace-For4Tx-PerResourceConfigList-r13");
    for (const auto& e1 : ace_for4_tx_per_res_cfg_list_r13) {
      j.write_bool(e1);
    }
    j.end_array();
  }
  if (alt_codebook_enabled_bf_r13_present) {
    j.write_str("alternativeCodebookEnabledBeamformed-r13", "true");
  }
  if (ch_meas_restrict_r13_present) {
    j.write_str("channelMeasRestriction-r13", "on");
  }
  j.end_obj();
}

// CSI-RS-ConfigBeamformed-v1430 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_bf_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_ap_list_r14_present, 1));
  HANDLE_CODE(bref.pack(nzp_res_cfg_original_v1430_present, 1));
  HANDLE_CODE(bref.pack(csi_rs_nzp_activation_r14_present, 1));

  if (csi_rs_cfg_nzp_ap_list_r14_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_rs_cfg_nzp_ap_list_r14, 1, 8));
  }
  if (nzp_res_cfg_original_v1430_present) {
    HANDLE_CODE(nzp_res_cfg_original_v1430.pack(bref));
  }
  if (csi_rs_nzp_activation_r14_present) {
    HANDLE_CODE(csi_rs_nzp_activation_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_bf_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_ap_list_r14_present, 1));
  HANDLE_CODE(bref.unpack(nzp_res_cfg_original_v1430_present, 1));
  HANDLE_CODE(bref.unpack(csi_rs_nzp_activation_r14_present, 1));

  if (csi_rs_cfg_nzp_ap_list_r14_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_rs_cfg_nzp_ap_list_r14, bref, 1, 8));
  }
  if (nzp_res_cfg_original_v1430_present) {
    HANDLE_CODE(nzp_res_cfg_original_v1430.unpack(bref));
  }
  if (csi_rs_nzp_activation_r14_present) {
    HANDLE_CODE(csi_rs_nzp_activation_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_bf_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_rs_cfg_nzp_ap_list_r14_present) {
    j.start_array("csi-RS-ConfigNZP-ApList-r14");
    for (const auto& e1 : csi_rs_cfg_nzp_ap_list_r14) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (nzp_res_cfg_original_v1430_present) {
    j.write_fieldname("nzp-ResourceConfigOriginal-v1430");
    nzp_res_cfg_original_v1430.to_json(j);
  }
  if (csi_rs_nzp_activation_r14_present) {
    j.write_fieldname("csi-RS-NZP-Activation-r14");
    csi_rs_nzp_activation_r14.to_json(j);
  }
  j.end_obj();
}

// CSI-RS-ConfigEMIMO2-r14 ::= CHOICE
void csi_rs_cfg_emimo2_r14_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_emimo2_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo2_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo2_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo2_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo2_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo2_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigNonPrecoded-r13 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_non_precoded_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(p_c_and_cbsr_list_r13_present, 1));
  HANDLE_CODE(bref.pack(codebook_over_sampling_rate_cfg_o1_r13_present, 1));
  HANDLE_CODE(bref.pack(codebook_over_sampling_rate_cfg_o2_r13_present, 1));
  HANDLE_CODE(bref.pack(csi_im_cfg_id_list_r13_present, 1));
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_emimo_r13_present, 1));

  if (p_c_and_cbsr_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, p_c_and_cbsr_list_r13, 1, 2));
  }
  HANDLE_CODE(codebook_cfg_n1_r13.pack(bref));
  HANDLE_CODE(codebook_cfg_n2_r13.pack(bref));
  if (codebook_over_sampling_rate_cfg_o1_r13_present) {
    HANDLE_CODE(codebook_over_sampling_rate_cfg_o1_r13.pack(bref));
  }
  if (codebook_over_sampling_rate_cfg_o2_r13_present) {
    HANDLE_CODE(codebook_over_sampling_rate_cfg_o2_r13.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, codebook_cfg_r13, (uint8_t)1u, (uint8_t)4u));
  if (csi_im_cfg_id_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_id_list_r13, 1, 2, integer_packer<uint8_t>(1, 24)));
  }
  if (csi_rs_cfg_nzp_emimo_r13_present) {
    HANDLE_CODE(csi_rs_cfg_nzp_emimo_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_non_precoded_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(p_c_and_cbsr_list_r13_present, 1));
  HANDLE_CODE(bref.unpack(codebook_over_sampling_rate_cfg_o1_r13_present, 1));
  HANDLE_CODE(bref.unpack(codebook_over_sampling_rate_cfg_o2_r13_present, 1));
  HANDLE_CODE(bref.unpack(csi_im_cfg_id_list_r13_present, 1));
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_emimo_r13_present, 1));

  if (p_c_and_cbsr_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(p_c_and_cbsr_list_r13, bref, 1, 2));
  }
  HANDLE_CODE(codebook_cfg_n1_r13.unpack(bref));
  HANDLE_CODE(codebook_cfg_n2_r13.unpack(bref));
  if (codebook_over_sampling_rate_cfg_o1_r13_present) {
    HANDLE_CODE(codebook_over_sampling_rate_cfg_o1_r13.unpack(bref));
  }
  if (codebook_over_sampling_rate_cfg_o2_r13_present) {
    HANDLE_CODE(codebook_over_sampling_rate_cfg_o2_r13.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(codebook_cfg_r13, bref, (uint8_t)1u, (uint8_t)4u));
  if (csi_im_cfg_id_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_id_list_r13, bref, 1, 2, integer_packer<uint8_t>(1, 24)));
  }
  if (csi_rs_cfg_nzp_emimo_r13_present) {
    HANDLE_CODE(csi_rs_cfg_nzp_emimo_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_non_precoded_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (p_c_and_cbsr_list_r13_present) {
    j.start_array("p-C-AndCBSRList-r13");
    for (const auto& e1 : p_c_and_cbsr_list_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.write_str("codebookConfigN1-r13", codebook_cfg_n1_r13.to_string());
  j.write_str("codebookConfigN2-r13", codebook_cfg_n2_r13.to_string());
  if (codebook_over_sampling_rate_cfg_o1_r13_present) {
    j.write_str("codebookOverSamplingRateConfig-O1-r13", codebook_over_sampling_rate_cfg_o1_r13.to_string());
  }
  if (codebook_over_sampling_rate_cfg_o2_r13_present) {
    j.write_str("codebookOverSamplingRateConfig-O2-r13", codebook_over_sampling_rate_cfg_o2_r13.to_string());
  }
  j.write_int("codebookConfig-r13", codebook_cfg_r13);
  if (csi_im_cfg_id_list_r13_present) {
    j.start_array("csi-IM-ConfigIdList-r13");
    for (const auto& e1 : csi_im_cfg_id_list_r13) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (csi_rs_cfg_nzp_emimo_r13_present) {
    j.write_fieldname("csi-RS-ConfigNZP-EMIMO-r13");
    csi_rs_cfg_nzp_emimo_r13.to_json(j);
  }
  j.end_obj();
}

// CSI-RS-ConfigNonPrecoded-v1430 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_non_precoded_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_emimo_v1430_present, 1));

  if (csi_rs_cfg_nzp_emimo_v1430_present) {
    HANDLE_CODE(csi_rs_cfg_nzp_emimo_v1430.pack(bref));
  }
  HANDLE_CODE(codebook_cfg_n1_v1430.pack(bref));
  HANDLE_CODE(codebook_cfg_n2_v1430.pack(bref));
  HANDLE_CODE(nzp_res_cfg_tm9_original_v1430.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_non_precoded_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_emimo_v1430_present, 1));

  if (csi_rs_cfg_nzp_emimo_v1430_present) {
    HANDLE_CODE(csi_rs_cfg_nzp_emimo_v1430.unpack(bref));
  }
  HANDLE_CODE(codebook_cfg_n1_v1430.unpack(bref));
  HANDLE_CODE(codebook_cfg_n2_v1430.unpack(bref));
  HANDLE_CODE(nzp_res_cfg_tm9_original_v1430.unpack(bref));

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_non_precoded_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_rs_cfg_nzp_emimo_v1430_present) {
    j.write_fieldname("csi-RS-ConfigNZP-EMIMO-v1430");
    csi_rs_cfg_nzp_emimo_v1430.to_json(j);
  }
  j.write_str("codebookConfigN1-v1430", codebook_cfg_n1_v1430.to_string());
  j.write_str("codebookConfigN2-v1430", codebook_cfg_n2_v1430.to_string());
  j.write_fieldname("nzp-ResourceConfigTM9-Original-v1430");
  nzp_res_cfg_tm9_original_v1430.to_json(j);
  j.end_obj();
}

// CSI-RS-ConfigNonPrecoded-v1480 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_non_precoded_v1480_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_rs_cfg_nzp_emimo_v1480_present, 1));
  HANDLE_CODE(bref.pack(codebook_cfg_n1_v1480_present, 1));
  HANDLE_CODE(bref.pack(codebook_cfg_n2_r1480_present, 1));

  if (csi_rs_cfg_nzp_emimo_v1480_present) {
    HANDLE_CODE(csi_rs_cfg_nzp_emimo_v1480.pack(bref));
  }
  if (codebook_cfg_n1_v1480_present) {
    HANDLE_CODE(codebook_cfg_n1_v1480.pack(bref));
  }
  if (codebook_cfg_n2_r1480_present) {
    HANDLE_CODE(codebook_cfg_n2_r1480.pack(bref));
  }
  HANDLE_CODE(nzp_res_cfg_tm9_original_v1480.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_non_precoded_v1480_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_rs_cfg_nzp_emimo_v1480_present, 1));
  HANDLE_CODE(bref.unpack(codebook_cfg_n1_v1480_present, 1));
  HANDLE_CODE(bref.unpack(codebook_cfg_n2_r1480_present, 1));

  if (csi_rs_cfg_nzp_emimo_v1480_present) {
    HANDLE_CODE(csi_rs_cfg_nzp_emimo_v1480.unpack(bref));
  }
  if (codebook_cfg_n1_v1480_present) {
    HANDLE_CODE(codebook_cfg_n1_v1480.unpack(bref));
  }
  if (codebook_cfg_n2_r1480_present) {
    HANDLE_CODE(codebook_cfg_n2_r1480.unpack(bref));
  }
  HANDLE_CODE(nzp_res_cfg_tm9_original_v1480.unpack(bref));

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_non_precoded_v1480_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_rs_cfg_nzp_emimo_v1480_present) {
    j.write_fieldname("csi-RS-ConfigNZP-EMIMO-v1480");
    csi_rs_cfg_nzp_emimo_v1480.to_json(j);
  }
  if (codebook_cfg_n1_v1480_present) {
    j.write_str("codebookConfigN1-v1480", codebook_cfg_n1_v1480.to_string());
  }
  if (codebook_cfg_n2_r1480_present) {
    j.write_str("codebookConfigN2-r1480", codebook_cfg_n2_r1480.to_string());
  }
  j.write_fieldname("nzp-ResourceConfigTM9-Original-v1480");
  nzp_res_cfg_tm9_original_v1480.to_json(j);
  j.end_obj();
}

// CSI-RS-ConfigNonPrecoded-v1530 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_non_precoded_v1530_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(p_c_and_cbsr_list_r15_present, 1));

  if (p_c_and_cbsr_list_r15_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, p_c_and_cbsr_list_r15, 1, 2));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_non_precoded_v1530_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(p_c_and_cbsr_list_r15_present, 1));

  if (p_c_and_cbsr_list_r15_present) {
    HANDLE_CODE(unpack_dyn_seq_of(p_c_and_cbsr_list_r15, bref, 1, 2));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_non_precoded_v1530_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (p_c_and_cbsr_list_r15_present) {
    j.start_array("p-C-AndCBSRList-r15");
    for (const auto& e1 : p_c_and_cbsr_list_r15) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// P-C-AndCBSR-r11 ::= SEQUENCE
SRSASN_CODE p_c_and_cbsr_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, p_c_r11, (int8_t)-8, (int8_t)15));
  HANDLE_CODE(codebook_subset_restrict_r11.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE p_c_and_cbsr_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(p_c_r11, bref, (int8_t)-8, (int8_t)15));
  HANDLE_CODE(codebook_subset_restrict_r11.unpack(bref));

  return SRSASN_SUCCESS;
}
void p_c_and_cbsr_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("p-C-r11", p_c_r11);
  j.write_str("codebookSubsetRestriction-r11", codebook_subset_restrict_r11.to_string());
  j.end_obj();
}

// CQI-ReportAperiodicProc-r11 ::= SEQUENCE
SRSASN_CODE cqi_report_aperiodic_proc_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(cqi_report_mode_aperiodic_r11.pack(bref));
  HANDLE_CODE(bref.pack(trigger01_r11, 1));
  HANDLE_CODE(bref.pack(trigger10_r11, 1));
  HANDLE_CODE(bref.pack(trigger11_r11, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_proc_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(cqi_report_mode_aperiodic_r11.unpack(bref));
  HANDLE_CODE(bref.unpack(trigger01_r11, 1));
  HANDLE_CODE(bref.unpack(trigger10_r11, 1));
  HANDLE_CODE(bref.unpack(trigger11_r11, 1));

  return SRSASN_SUCCESS;
}
void cqi_report_aperiodic_proc_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cqi-ReportModeAperiodic-r11", cqi_report_mode_aperiodic_r11.to_string());
  j.write_bool("trigger01-r11", trigger01_r11);
  j.write_bool("trigger10-r11", trigger10_r11);
  j.write_bool("trigger11-r11", trigger11_r11);
  j.end_obj();
}

// CQI-ReportAperiodicProc-v1310 ::= SEQUENCE
SRSASN_CODE cqi_report_aperiodic_proc_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(trigger001_r13, 1));
  HANDLE_CODE(bref.pack(trigger010_r13, 1));
  HANDLE_CODE(bref.pack(trigger011_r13, 1));
  HANDLE_CODE(bref.pack(trigger100_r13, 1));
  HANDLE_CODE(bref.pack(trigger101_r13, 1));
  HANDLE_CODE(bref.pack(trigger110_r13, 1));
  HANDLE_CODE(bref.pack(trigger111_r13, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_proc_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(trigger001_r13, 1));
  HANDLE_CODE(bref.unpack(trigger010_r13, 1));
  HANDLE_CODE(bref.unpack(trigger011_r13, 1));
  HANDLE_CODE(bref.unpack(trigger100_r13, 1));
  HANDLE_CODE(bref.unpack(trigger101_r13, 1));
  HANDLE_CODE(bref.unpack(trigger110_r13, 1));
  HANDLE_CODE(bref.unpack(trigger111_r13, 1));

  return SRSASN_SUCCESS;
}
void cqi_report_aperiodic_proc_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_bool("trigger001-r13", trigger001_r13);
  j.write_bool("trigger010-r13", trigger010_r13);
  j.write_bool("trigger011-r13", trigger011_r13);
  j.write_bool("trigger100-r13", trigger100_r13);
  j.write_bool("trigger101-r13", trigger101_r13);
  j.write_bool("trigger110-r13", trigger110_r13);
  j.write_bool("trigger111-r13", trigger111_r13);
  j.end_obj();
}

// CQI-ReportBothProc-r11 ::= SEQUENCE
SRSASN_CODE cqi_report_both_proc_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ri_ref_csi_process_id_r11_present, 1));
  HANDLE_CODE(bref.pack(pmi_ri_report_r11_present, 1));

  if (ri_ref_csi_process_id_r11_present) {
    HANDLE_CODE(pack_integer(bref, ri_ref_csi_process_id_r11, (uint8_t)1u, (uint8_t)4u));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_both_proc_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ri_ref_csi_process_id_r11_present, 1));
  HANDLE_CODE(bref.unpack(pmi_ri_report_r11_present, 1));

  if (ri_ref_csi_process_id_r11_present) {
    HANDLE_CODE(unpack_integer(ri_ref_csi_process_id_r11, bref, (uint8_t)1u, (uint8_t)4u));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_both_proc_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ri_ref_csi_process_id_r11_present) {
    j.write_int("ri-Ref-CSI-ProcessId-r11", ri_ref_csi_process_id_r11);
  }
  if (pmi_ri_report_r11_present) {
    j.write_str("pmi-RI-Report-r11", "setup");
  }
  j.end_obj();
}

// CRI-ReportConfig-r13 ::= CHOICE
void cri_report_cfg_r13_c::set(types::options e)
{
  type_ = e;
}
void cri_report_cfg_r13_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("cri-ConfigIndex-r13", c.cri_cfg_idx_r13);
      if (c.cri_cfg_idx2_r13_present) {
        j.write_int("cri-ConfigIndex2-r13", c.cri_cfg_idx2_r13);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cri_report_cfg_r13_c");
  }
  j.end_obj();
}
SRSASN_CODE cri_report_cfg_r13_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.cri_cfg_idx2_r13_present, 1));
      HANDLE_CODE(pack_integer(bref, c.cri_cfg_idx_r13, (uint16_t)0u, (uint16_t)1023u));
      if (c.cri_cfg_idx2_r13_present) {
        HANDLE_CODE(pack_integer(bref, c.cri_cfg_idx2_r13, (uint16_t)0u, (uint16_t)1023u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cri_report_cfg_r13_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cri_report_cfg_r13_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.cri_cfg_idx2_r13_present, 1));
      HANDLE_CODE(unpack_integer(c.cri_cfg_idx_r13, bref, (uint16_t)0u, (uint16_t)1023u));
      if (c.cri_cfg_idx2_r13_present) {
        HANDLE_CODE(unpack_integer(c.cri_cfg_idx2_r13, bref, (uint16_t)0u, (uint16_t)1023u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cri_report_cfg_r13_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigEMIMO-Hybrid-r14 ::= CHOICE
void csi_rs_cfg_emimo_hybrid_r14_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_emimo_hybrid_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.periodicity_offset_idx_r14_present) {
        j.write_int("periodicityOffsetIndex-r14", c.periodicity_offset_idx_r14);
      }
      if (c.emimo_type2_r14_present) {
        j.write_fieldname("eMIMO-Type2-r14");
        c.emimo_type2_r14.to_json(j);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_hybrid_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_hybrid_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.periodicity_offset_idx_r14_present, 1));
      HANDLE_CODE(bref.pack(c.emimo_type2_r14_present, 1));
      if (c.periodicity_offset_idx_r14_present) {
        HANDLE_CODE(pack_integer(bref, c.periodicity_offset_idx_r14, (uint16_t)0u, (uint16_t)1023u));
      }
      if (c.emimo_type2_r14_present) {
        HANDLE_CODE(c.emimo_type2_r14.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_hybrid_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_hybrid_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.periodicity_offset_idx_r14_present, 1));
      HANDLE_CODE(bref.unpack(c.emimo_type2_r14_present, 1));
      if (c.periodicity_offset_idx_r14_present) {
        HANDLE_CODE(unpack_integer(c.periodicity_offset_idx_r14, bref, (uint16_t)0u, (uint16_t)1023u));
      }
      if (c.emimo_type2_r14_present) {
        HANDLE_CODE(c.emimo_type2_r14.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_hybrid_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigEMIMO-r13 ::= CHOICE
void csi_rs_cfg_emimo_r13_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_emimo_r13_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_r13_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_r13_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_rs_cfg_emimo_r13_c::setup_c_::destroy_()
{
  switch (type_) {
    case types::non_precoded_r13:
      c.destroy<csi_rs_cfg_non_precoded_r13_s>();
      break;
    case types::bf_r13:
      c.destroy<csi_rs_cfg_bf_r13_s>();
      break;
    default:
      break;
  }
}
void csi_rs_cfg_emimo_r13_c::setup_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::non_precoded_r13:
      c.init<csi_rs_cfg_non_precoded_r13_s>();
      break;
    case types::bf_r13:
      c.init<csi_rs_cfg_bf_r13_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c::setup_c_");
  }
}
csi_rs_cfg_emimo_r13_c::setup_c_::setup_c_(const csi_rs_cfg_emimo_r13_c::setup_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::non_precoded_r13:
      c.init(other.c.get<csi_rs_cfg_non_precoded_r13_s>());
      break;
    case types::bf_r13:
      c.init(other.c.get<csi_rs_cfg_bf_r13_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c::setup_c_");
  }
}
csi_rs_cfg_emimo_r13_c::setup_c_& csi_rs_cfg_emimo_r13_c::setup_c_::
                                  operator=(const csi_rs_cfg_emimo_r13_c::setup_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::non_precoded_r13:
      c.set(other.c.get<csi_rs_cfg_non_precoded_r13_s>());
      break;
    case types::bf_r13:
      c.set(other.c.get<csi_rs_cfg_bf_r13_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c::setup_c_");
  }

  return *this;
}
void csi_rs_cfg_emimo_r13_c::setup_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::non_precoded_r13:
      j.write_fieldname("nonPrecoded-r13");
      c.get<csi_rs_cfg_non_precoded_r13_s>().to_json(j);
      break;
    case types::bf_r13:
      j.write_fieldname("beamformed-r13");
      c.get<csi_rs_cfg_bf_r13_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c::setup_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_r13_c::setup_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::non_precoded_r13:
      HANDLE_CODE(c.get<csi_rs_cfg_non_precoded_r13_s>().pack(bref));
      break;
    case types::bf_r13:
      HANDLE_CODE(c.get<csi_rs_cfg_bf_r13_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c::setup_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_r13_c::setup_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::non_precoded_r13:
      HANDLE_CODE(c.get<csi_rs_cfg_non_precoded_r13_s>().unpack(bref));
      break;
    case types::bf_r13:
      HANDLE_CODE(c.get<csi_rs_cfg_bf_r13_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_r13_c::setup_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigEMIMO-v1430 ::= CHOICE
void csi_rs_cfg_emimo_v1430_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_emimo_v1430_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_v1430_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_v1430_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_rs_cfg_emimo_v1430_c::setup_c_::destroy_()
{
  switch (type_) {
    case types::non_precoded_v1430:
      c.destroy<csi_rs_cfg_non_precoded_v1430_s>();
      break;
    case types::bf_v1430:
      c.destroy<csi_rs_cfg_bf_v1430_s>();
      break;
    default:
      break;
  }
}
void csi_rs_cfg_emimo_v1430_c::setup_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::non_precoded_v1430:
      c.init<csi_rs_cfg_non_precoded_v1430_s>();
      break;
    case types::bf_v1430:
      c.init<csi_rs_cfg_bf_v1430_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c::setup_c_");
  }
}
csi_rs_cfg_emimo_v1430_c::setup_c_::setup_c_(const csi_rs_cfg_emimo_v1430_c::setup_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::non_precoded_v1430:
      c.init(other.c.get<csi_rs_cfg_non_precoded_v1430_s>());
      break;
    case types::bf_v1430:
      c.init(other.c.get<csi_rs_cfg_bf_v1430_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c::setup_c_");
  }
}
csi_rs_cfg_emimo_v1430_c::setup_c_& csi_rs_cfg_emimo_v1430_c::setup_c_::
                                    operator=(const csi_rs_cfg_emimo_v1430_c::setup_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::non_precoded_v1430:
      c.set(other.c.get<csi_rs_cfg_non_precoded_v1430_s>());
      break;
    case types::bf_v1430:
      c.set(other.c.get<csi_rs_cfg_bf_v1430_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c::setup_c_");
  }

  return *this;
}
void csi_rs_cfg_emimo_v1430_c::setup_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::non_precoded_v1430:
      j.write_fieldname("nonPrecoded-v1430");
      c.get<csi_rs_cfg_non_precoded_v1430_s>().to_json(j);
      break;
    case types::bf_v1430:
      j.write_fieldname("beamformed-v1430");
      c.get<csi_rs_cfg_bf_v1430_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c::setup_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_v1430_c::setup_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::non_precoded_v1430:
      HANDLE_CODE(c.get<csi_rs_cfg_non_precoded_v1430_s>().pack(bref));
      break;
    case types::bf_v1430:
      HANDLE_CODE(c.get<csi_rs_cfg_bf_v1430_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c::setup_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_v1430_c::setup_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::non_precoded_v1430:
      HANDLE_CODE(c.get<csi_rs_cfg_non_precoded_v1430_s>().unpack(bref));
      break;
    case types::bf_v1430:
      HANDLE_CODE(c.get<csi_rs_cfg_bf_v1430_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1430_c::setup_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigEMIMO-v1480 ::= CHOICE
void csi_rs_cfg_emimo_v1480_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_emimo_v1480_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_v1480_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_v1480_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_rs_cfg_emimo_v1480_c::setup_c_::destroy_()
{
  switch (type_) {
    case types::non_precoded_v1480:
      c.destroy<csi_rs_cfg_non_precoded_v1480_s>();
      break;
    case types::bf_v1480:
      c.destroy<csi_rs_cfg_bf_v1430_s>();
      break;
    default:
      break;
  }
}
void csi_rs_cfg_emimo_v1480_c::setup_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::non_precoded_v1480:
      c.init<csi_rs_cfg_non_precoded_v1480_s>();
      break;
    case types::bf_v1480:
      c.init<csi_rs_cfg_bf_v1430_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c::setup_c_");
  }
}
csi_rs_cfg_emimo_v1480_c::setup_c_::setup_c_(const csi_rs_cfg_emimo_v1480_c::setup_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::non_precoded_v1480:
      c.init(other.c.get<csi_rs_cfg_non_precoded_v1480_s>());
      break;
    case types::bf_v1480:
      c.init(other.c.get<csi_rs_cfg_bf_v1430_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c::setup_c_");
  }
}
csi_rs_cfg_emimo_v1480_c::setup_c_& csi_rs_cfg_emimo_v1480_c::setup_c_::
                                    operator=(const csi_rs_cfg_emimo_v1480_c::setup_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::non_precoded_v1480:
      c.set(other.c.get<csi_rs_cfg_non_precoded_v1480_s>());
      break;
    case types::bf_v1480:
      c.set(other.c.get<csi_rs_cfg_bf_v1430_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c::setup_c_");
  }

  return *this;
}
void csi_rs_cfg_emimo_v1480_c::setup_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::non_precoded_v1480:
      j.write_fieldname("nonPrecoded-v1480");
      c.get<csi_rs_cfg_non_precoded_v1480_s>().to_json(j);
      break;
    case types::bf_v1480:
      j.write_fieldname("beamformed-v1480");
      c.get<csi_rs_cfg_bf_v1430_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c::setup_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_v1480_c::setup_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::non_precoded_v1480:
      HANDLE_CODE(c.get<csi_rs_cfg_non_precoded_v1480_s>().pack(bref));
      break;
    case types::bf_v1480:
      HANDLE_CODE(c.get<csi_rs_cfg_bf_v1430_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c::setup_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_v1480_c::setup_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::non_precoded_v1480:
      HANDLE_CODE(c.get<csi_rs_cfg_non_precoded_v1480_s>().unpack(bref));
      break;
    case types::bf_v1480:
      HANDLE_CODE(c.get<csi_rs_cfg_bf_v1430_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1480_c::setup_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigEMIMO-v1530 ::= CHOICE
void csi_rs_cfg_emimo_v1530_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_emimo_v1530_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1530_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_v1530_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1530_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_v1530_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_emimo_v1530_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_rs_cfg_emimo_v1530_c::setup_c_::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nonPrecoded-v1530");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_emimo_v1530_c::setup_c_::pack(bit_ref& bref) const
{
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_emimo_v1530_c::setup_c_::unpack(cbit_ref& bref)
{
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

// CQI-ReportPeriodicProcExt-r11 ::= SEQUENCE
SRSASN_CODE cqi_report_periodic_proc_ext_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ri_cfg_idx_r11_present, 1));
  HANDLE_CODE(bref.pack(csi_cfg_idx_r11_present, 1));

  HANDLE_CODE(pack_integer(bref, cqi_report_periodic_proc_ext_id_r11, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(pack_integer(bref, cqi_pmi_cfg_idx_r11, (uint16_t)0u, (uint16_t)1023u));
  HANDLE_CODE(cqi_format_ind_periodic_r11.pack(bref));
  if (ri_cfg_idx_r11_present) {
    HANDLE_CODE(pack_integer(bref, ri_cfg_idx_r11, (uint16_t)0u, (uint16_t)1023u));
  }
  if (csi_cfg_idx_r11_present) {
    HANDLE_CODE(csi_cfg_idx_r11.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= cri_report_cfg_r13.is_present();
    group_flags[1] |= periodicity_factor_wb_r13_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cri_report_cfg_r13.is_present(), 1));
      if (cri_report_cfg_r13.is_present()) {
        HANDLE_CODE(cri_report_cfg_r13->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(periodicity_factor_wb_r13_present, 1));
      if (periodicity_factor_wb_r13_present) {
        HANDLE_CODE(periodicity_factor_wb_r13.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_proc_ext_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ri_cfg_idx_r11_present, 1));
  HANDLE_CODE(bref.unpack(csi_cfg_idx_r11_present, 1));

  HANDLE_CODE(unpack_integer(cqi_report_periodic_proc_ext_id_r11, bref, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(unpack_integer(cqi_pmi_cfg_idx_r11, bref, (uint16_t)0u, (uint16_t)1023u));
  HANDLE_CODE(cqi_format_ind_periodic_r11.unpack(bref));
  if (ri_cfg_idx_r11_present) {
    HANDLE_CODE(unpack_integer(ri_cfg_idx_r11, bref, (uint16_t)0u, (uint16_t)1023u));
  }
  if (csi_cfg_idx_r11_present) {
    HANDLE_CODE(csi_cfg_idx_r11.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cri_report_cfg_r13_present;
      HANDLE_CODE(bref.unpack(cri_report_cfg_r13_present, 1));
      cri_report_cfg_r13.set_present(cri_report_cfg_r13_present);
      if (cri_report_cfg_r13.is_present()) {
        HANDLE_CODE(cri_report_cfg_r13->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(periodicity_factor_wb_r13_present, 1));
      if (periodicity_factor_wb_r13_present) {
        HANDLE_CODE(periodicity_factor_wb_r13.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void cqi_report_periodic_proc_ext_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("cqi-ReportPeriodicProcExtId-r11", cqi_report_periodic_proc_ext_id_r11);
  j.write_int("cqi-pmi-ConfigIndex-r11", cqi_pmi_cfg_idx_r11);
  j.write_fieldname("cqi-FormatIndicatorPeriodic-r11");
  cqi_format_ind_periodic_r11.to_json(j);
  if (ri_cfg_idx_r11_present) {
    j.write_int("ri-ConfigIndex-r11", ri_cfg_idx_r11);
  }
  if (csi_cfg_idx_r11_present) {
    j.write_fieldname("csi-ConfigIndex-r11");
    csi_cfg_idx_r11.to_json(j);
  }
  if (ext) {
    if (cri_report_cfg_r13.is_present()) {
      j.write_fieldname("cri-ReportConfig-r13");
      cri_report_cfg_r13->to_json(j);
    }
    if (periodicity_factor_wb_r13_present) {
      j.write_str("periodicityFactorWB-r13", periodicity_factor_wb_r13.to_string());
    }
  }
  j.end_obj();
}

void cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::destroy_()
{
  switch (type_) {
    case types::wideband_cqi_r11:
      c.destroy<wideband_cqi_r11_s_>();
      break;
    case types::subband_cqi_r11:
      c.destroy<subband_cqi_r11_s_>();
      break;
    default:
      break;
  }
}
void cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::wideband_cqi_r11:
      c.init<wideband_cqi_r11_s_>();
      break;
    case types::subband_cqi_r11:
      c.init<subband_cqi_r11_s_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_");
  }
}
cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::cqi_format_ind_periodic_r11_c_(
    const cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::wideband_cqi_r11:
      c.init(other.c.get<wideband_cqi_r11_s_>());
      break;
    case types::subband_cqi_r11:
      c.init(other.c.get<subband_cqi_r11_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_");
  }
}
cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_&
cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::
operator=(const cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::wideband_cqi_r11:
      c.set(other.c.get<wideband_cqi_r11_s_>());
      break;
    case types::subband_cqi_r11:
      c.set(other.c.get<subband_cqi_r11_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_");
  }

  return *this;
}
void cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::wideband_cqi_r11:
      j.write_fieldname("widebandCQI-r11");
      j.start_obj();
      if (c.get<wideband_cqi_r11_s_>().csi_report_mode_r11_present) {
        j.write_str("csi-ReportMode-r11", c.get<wideband_cqi_r11_s_>().csi_report_mode_r11.to_string());
      }
      j.end_obj();
      break;
    case types::subband_cqi_r11:
      j.write_fieldname("subbandCQI-r11");
      j.start_obj();
      j.write_int("k", c.get<subband_cqi_r11_s_>().k);
      j.write_str("periodicityFactor-r11", c.get<subband_cqi_r11_s_>().periodicity_factor_r11.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::wideband_cqi_r11:
      HANDLE_CODE(bref.pack(c.get<wideband_cqi_r11_s_>().csi_report_mode_r11_present, 1));
      if (c.get<wideband_cqi_r11_s_>().csi_report_mode_r11_present) {
        HANDLE_CODE(c.get<wideband_cqi_r11_s_>().csi_report_mode_r11.pack(bref));
      }
      break;
    case types::subband_cqi_r11:
      HANDLE_CODE(pack_integer(bref, c.get<subband_cqi_r11_s_>().k, (uint8_t)1u, (uint8_t)4u));
      HANDLE_CODE(c.get<subband_cqi_r11_s_>().periodicity_factor_r11.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::wideband_cqi_r11:
      HANDLE_CODE(bref.unpack(c.get<wideband_cqi_r11_s_>().csi_report_mode_r11_present, 1));
      if (c.get<wideband_cqi_r11_s_>().csi_report_mode_r11_present) {
        HANDLE_CODE(c.get<wideband_cqi_r11_s_>().csi_report_mode_r11.unpack(bref));
      }
      break;
    case types::subband_cqi_r11:
      HANDLE_CODE(unpack_integer(c.get<subband_cqi_r11_s_>().k, bref, (uint8_t)1u, (uint8_t)4u));
      HANDLE_CODE(c.get<subband_cqi_r11_s_>().periodicity_factor_r11.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::cqi_format_ind_periodic_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_::set(types::options e)
{
  type_ = e;
}
void cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("cqi-pmi-ConfigIndex2-r11", c.cqi_pmi_cfg_idx2_r11);
      if (c.ri_cfg_idx2_r11_present) {
        j.write_int("ri-ConfigIndex2-r11", c.ri_cfg_idx2_r11);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.ri_cfg_idx2_r11_present, 1));
      HANDLE_CODE(pack_integer(bref, c.cqi_pmi_cfg_idx2_r11, (uint16_t)0u, (uint16_t)1023u));
      if (c.ri_cfg_idx2_r11_present) {
        HANDLE_CODE(pack_integer(bref, c.ri_cfg_idx2_r11, (uint16_t)0u, (uint16_t)1023u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.ri_cfg_idx2_r11_present, 1));
      HANDLE_CODE(unpack_integer(c.cqi_pmi_cfg_idx2_r11, bref, (uint16_t)0u, (uint16_t)1023u));
      if (c.ri_cfg_idx2_r11_present) {
        HANDLE_CODE(unpack_integer(c.ri_cfg_idx2_r11, bref, (uint16_t)0u, (uint16_t)1023u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_proc_ext_r11_s::csi_cfg_idx_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-IM-Config-r11 ::= SEQUENCE
SRSASN_CODE csi_im_cfg_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, csi_im_cfg_id_r11, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(pack_integer(bref, res_cfg_r11, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, sf_cfg_r11, (uint8_t)0u, (uint8_t)154u));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= interference_meas_restrict_r13_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(interference_meas_restrict_r13_present, 1));
      if (interference_meas_restrict_r13_present) {
        HANDLE_CODE(bref.pack(interference_meas_restrict_r13, 1));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_im_cfg_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(csi_im_cfg_id_r11, bref, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(unpack_integer(res_cfg_r11, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(sf_cfg_r11, bref, (uint8_t)0u, (uint8_t)154u));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(interference_meas_restrict_r13_present, 1));
      if (interference_meas_restrict_r13_present) {
        HANDLE_CODE(bref.unpack(interference_meas_restrict_r13, 1));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void csi_im_cfg_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("csi-IM-ConfigId-r11", csi_im_cfg_id_r11);
  j.write_int("resourceConfig-r11", res_cfg_r11);
  j.write_int("subframeConfig-r11", sf_cfg_r11);
  if (ext) {
    if (interference_meas_restrict_r13_present) {
      j.write_bool("interferenceMeasRestriction-r13", interference_meas_restrict_r13);
    }
  }
  j.end_obj();
}

// CSI-IM-ConfigExt-r12 ::= SEQUENCE
SRSASN_CODE csi_im_cfg_ext_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, csi_im_cfg_id_v1250, (uint8_t)4u, (uint8_t)4u));
  HANDLE_CODE(pack_integer(bref, res_cfg_r12, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(pack_integer(bref, sf_cfg_r12, (uint8_t)0u, (uint8_t)154u));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= interference_meas_restrict_r13_present;
    group_flags[0] |= csi_im_cfg_id_v1310_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(interference_meas_restrict_r13_present, 1));
      HANDLE_CODE(bref.pack(csi_im_cfg_id_v1310_present, 1));
      if (interference_meas_restrict_r13_present) {
        HANDLE_CODE(bref.pack(interference_meas_restrict_r13, 1));
      }
      if (csi_im_cfg_id_v1310_present) {
        HANDLE_CODE(pack_integer(bref, csi_im_cfg_id_v1310, (uint8_t)5u, (uint8_t)24u));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_im_cfg_ext_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(csi_im_cfg_id_v1250, bref, (uint8_t)4u, (uint8_t)4u));
  HANDLE_CODE(unpack_integer(res_cfg_r12, bref, (uint8_t)0u, (uint8_t)31u));
  HANDLE_CODE(unpack_integer(sf_cfg_r12, bref, (uint8_t)0u, (uint8_t)154u));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(interference_meas_restrict_r13_present, 1));
      HANDLE_CODE(bref.unpack(csi_im_cfg_id_v1310_present, 1));
      if (interference_meas_restrict_r13_present) {
        HANDLE_CODE(bref.unpack(interference_meas_restrict_r13, 1));
      }
      if (csi_im_cfg_id_v1310_present) {
        HANDLE_CODE(unpack_integer(csi_im_cfg_id_v1310, bref, (uint8_t)5u, (uint8_t)24u));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void csi_im_cfg_ext_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("csi-IM-ConfigId-v1250", csi_im_cfg_id_v1250);
  j.write_int("resourceConfig-r12", res_cfg_r12);
  j.write_int("subframeConfig-r12", sf_cfg_r12);
  if (ext) {
    if (interference_meas_restrict_r13_present) {
      j.write_bool("interferenceMeasRestriction-r13", interference_meas_restrict_r13);
    }
    if (csi_im_cfg_id_v1310_present) {
      j.write_int("csi-IM-ConfigId-v1310", csi_im_cfg_id_v1310);
    }
  }
  j.end_obj();
}

// CSI-Process-r11 ::= SEQUENCE
SRSASN_CODE csi_process_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cqi_report_both_proc_r11_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_periodic_proc_id_r11_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_aperiodic_proc_r11_present, 1));

  HANDLE_CODE(pack_integer(bref, csi_process_id_r11, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(pack_integer(bref, csi_rs_cfg_nzp_id_r11, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(pack_integer(bref, csi_im_cfg_id_r11, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(pack_dyn_seq_of(bref, p_c_and_cbsr_list_r11, 1, 2));
  if (cqi_report_both_proc_r11_present) {
    HANDLE_CODE(cqi_report_both_proc_r11.pack(bref));
  }
  if (cqi_report_periodic_proc_id_r11_present) {
    HANDLE_CODE(pack_integer(bref, cqi_report_periodic_proc_id_r11, (uint8_t)0u, (uint8_t)3u));
  }
  if (cqi_report_aperiodic_proc_r11_present) {
    HANDLE_CODE(cqi_report_aperiodic_proc_r11.pack(bref));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= alt_codebook_enabled_for4_tx_proc_r12_present;
    group_flags[0] |= csi_im_cfg_id_list_r12.is_present();
    group_flags[0] |= cqi_report_aperiodic_proc2_r12.is_present();
    group_flags[1] |= cqi_report_aperiodic_proc_v1310.is_present();
    group_flags[1] |= cqi_report_aperiodic_proc2_v1310.is_present();
    group_flags[1] |= emimo_type_r13.is_present();
    group_flags[2] |= dummy.is_present();
    group_flags[2] |= emimo_hybrid_r14.is_present();
    group_flags[2] |= advanced_codebook_enabled_r14_present;
    group_flags[3] |= emimo_type_v1480.is_present();
    group_flags[4] |= fe_comp_csi_enabled_v1530_present;
    group_flags[4] |= emimo_type_v1530.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(alt_codebook_enabled_for4_tx_proc_r12_present, 1));
      HANDLE_CODE(bref.pack(csi_im_cfg_id_list_r12.is_present(), 1));
      HANDLE_CODE(bref.pack(cqi_report_aperiodic_proc2_r12.is_present(), 1));
      if (csi_im_cfg_id_list_r12.is_present()) {
        HANDLE_CODE(csi_im_cfg_id_list_r12->pack(bref));
      }
      if (cqi_report_aperiodic_proc2_r12.is_present()) {
        HANDLE_CODE(cqi_report_aperiodic_proc2_r12->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(cqi_report_aperiodic_proc_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(cqi_report_aperiodic_proc2_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(emimo_type_r13.is_present(), 1));
      if (cqi_report_aperiodic_proc_v1310.is_present()) {
        HANDLE_CODE(cqi_report_aperiodic_proc_v1310->pack(bref));
      }
      if (cqi_report_aperiodic_proc2_v1310.is_present()) {
        HANDLE_CODE(cqi_report_aperiodic_proc2_v1310->pack(bref));
      }
      if (emimo_type_r13.is_present()) {
        HANDLE_CODE(emimo_type_r13->pack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(dummy.is_present(), 1));
      HANDLE_CODE(bref.pack(emimo_hybrid_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(advanced_codebook_enabled_r14_present, 1));
      if (dummy.is_present()) {
        HANDLE_CODE(dummy->pack(bref));
      }
      if (emimo_hybrid_r14.is_present()) {
        HANDLE_CODE(emimo_hybrid_r14->pack(bref));
      }
      if (advanced_codebook_enabled_r14_present) {
        HANDLE_CODE(bref.pack(advanced_codebook_enabled_r14, 1));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(emimo_type_v1480.is_present(), 1));
      if (emimo_type_v1480.is_present()) {
        HANDLE_CODE(emimo_type_v1480->pack(bref));
      }
    }
    if (group_flags[4]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(fe_comp_csi_enabled_v1530_present, 1));
      HANDLE_CODE(bref.pack(emimo_type_v1530.is_present(), 1));
      if (fe_comp_csi_enabled_v1530_present) {
        HANDLE_CODE(bref.pack(fe_comp_csi_enabled_v1530, 1));
      }
      if (emimo_type_v1530.is_present()) {
        HANDLE_CODE(emimo_type_v1530->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_process_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cqi_report_both_proc_r11_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_periodic_proc_id_r11_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_aperiodic_proc_r11_present, 1));

  HANDLE_CODE(unpack_integer(csi_process_id_r11, bref, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(unpack_integer(csi_rs_cfg_nzp_id_r11, bref, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(unpack_integer(csi_im_cfg_id_r11, bref, (uint8_t)1u, (uint8_t)3u));
  HANDLE_CODE(unpack_dyn_seq_of(p_c_and_cbsr_list_r11, bref, 1, 2));
  if (cqi_report_both_proc_r11_present) {
    HANDLE_CODE(cqi_report_both_proc_r11.unpack(bref));
  }
  if (cqi_report_periodic_proc_id_r11_present) {
    HANDLE_CODE(unpack_integer(cqi_report_periodic_proc_id_r11, bref, (uint8_t)0u, (uint8_t)3u));
  }
  if (cqi_report_aperiodic_proc_r11_present) {
    HANDLE_CODE(cqi_report_aperiodic_proc_r11.unpack(bref));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(5);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(alt_codebook_enabled_for4_tx_proc_r12_present, 1));
      bool csi_im_cfg_id_list_r12_present;
      HANDLE_CODE(bref.unpack(csi_im_cfg_id_list_r12_present, 1));
      csi_im_cfg_id_list_r12.set_present(csi_im_cfg_id_list_r12_present);
      bool cqi_report_aperiodic_proc2_r12_present;
      HANDLE_CODE(bref.unpack(cqi_report_aperiodic_proc2_r12_present, 1));
      cqi_report_aperiodic_proc2_r12.set_present(cqi_report_aperiodic_proc2_r12_present);
      if (csi_im_cfg_id_list_r12.is_present()) {
        HANDLE_CODE(csi_im_cfg_id_list_r12->unpack(bref));
      }
      if (cqi_report_aperiodic_proc2_r12.is_present()) {
        HANDLE_CODE(cqi_report_aperiodic_proc2_r12->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool cqi_report_aperiodic_proc_v1310_present;
      HANDLE_CODE(bref.unpack(cqi_report_aperiodic_proc_v1310_present, 1));
      cqi_report_aperiodic_proc_v1310.set_present(cqi_report_aperiodic_proc_v1310_present);
      bool cqi_report_aperiodic_proc2_v1310_present;
      HANDLE_CODE(bref.unpack(cqi_report_aperiodic_proc2_v1310_present, 1));
      cqi_report_aperiodic_proc2_v1310.set_present(cqi_report_aperiodic_proc2_v1310_present);
      bool emimo_type_r13_present;
      HANDLE_CODE(bref.unpack(emimo_type_r13_present, 1));
      emimo_type_r13.set_present(emimo_type_r13_present);
      if (cqi_report_aperiodic_proc_v1310.is_present()) {
        HANDLE_CODE(cqi_report_aperiodic_proc_v1310->unpack(bref));
      }
      if (cqi_report_aperiodic_proc2_v1310.is_present()) {
        HANDLE_CODE(cqi_report_aperiodic_proc2_v1310->unpack(bref));
      }
      if (emimo_type_r13.is_present()) {
        HANDLE_CODE(emimo_type_r13->unpack(bref));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool dummy_present;
      HANDLE_CODE(bref.unpack(dummy_present, 1));
      dummy.set_present(dummy_present);
      bool emimo_hybrid_r14_present;
      HANDLE_CODE(bref.unpack(emimo_hybrid_r14_present, 1));
      emimo_hybrid_r14.set_present(emimo_hybrid_r14_present);
      HANDLE_CODE(bref.unpack(advanced_codebook_enabled_r14_present, 1));
      if (dummy.is_present()) {
        HANDLE_CODE(dummy->unpack(bref));
      }
      if (emimo_hybrid_r14.is_present()) {
        HANDLE_CODE(emimo_hybrid_r14->unpack(bref));
      }
      if (advanced_codebook_enabled_r14_present) {
        HANDLE_CODE(bref.unpack(advanced_codebook_enabled_r14, 1));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool emimo_type_v1480_present;
      HANDLE_CODE(bref.unpack(emimo_type_v1480_present, 1));
      emimo_type_v1480.set_present(emimo_type_v1480_present);
      if (emimo_type_v1480.is_present()) {
        HANDLE_CODE(emimo_type_v1480->unpack(bref));
      }
    }
    if (group_flags[4]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(fe_comp_csi_enabled_v1530_present, 1));
      bool emimo_type_v1530_present;
      HANDLE_CODE(bref.unpack(emimo_type_v1530_present, 1));
      emimo_type_v1530.set_present(emimo_type_v1530_present);
      if (fe_comp_csi_enabled_v1530_present) {
        HANDLE_CODE(bref.unpack(fe_comp_csi_enabled_v1530, 1));
      }
      if (emimo_type_v1530.is_present()) {
        HANDLE_CODE(emimo_type_v1530->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void csi_process_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("csi-ProcessId-r11", csi_process_id_r11);
  j.write_int("csi-RS-ConfigNZPId-r11", csi_rs_cfg_nzp_id_r11);
  j.write_int("csi-IM-ConfigId-r11", csi_im_cfg_id_r11);
  j.start_array("p-C-AndCBSRList-r11");
  for (const auto& e1 : p_c_and_cbsr_list_r11) {
    e1.to_json(j);
  }
  j.end_array();
  if (cqi_report_both_proc_r11_present) {
    j.write_fieldname("cqi-ReportBothProc-r11");
    cqi_report_both_proc_r11.to_json(j);
  }
  if (cqi_report_periodic_proc_id_r11_present) {
    j.write_int("cqi-ReportPeriodicProcId-r11", cqi_report_periodic_proc_id_r11);
  }
  if (cqi_report_aperiodic_proc_r11_present) {
    j.write_fieldname("cqi-ReportAperiodicProc-r11");
    cqi_report_aperiodic_proc_r11.to_json(j);
  }
  if (ext) {
    if (alt_codebook_enabled_for4_tx_proc_r12_present) {
      j.write_str("alternativeCodebookEnabledFor4TXProc-r12", "true");
    }
    if (csi_im_cfg_id_list_r12.is_present()) {
      j.write_fieldname("csi-IM-ConfigIdList-r12");
      csi_im_cfg_id_list_r12->to_json(j);
    }
    if (cqi_report_aperiodic_proc2_r12.is_present()) {
      j.write_fieldname("cqi-ReportAperiodicProc2-r12");
      cqi_report_aperiodic_proc2_r12->to_json(j);
    }
    if (cqi_report_aperiodic_proc_v1310.is_present()) {
      j.write_fieldname("cqi-ReportAperiodicProc-v1310");
      cqi_report_aperiodic_proc_v1310->to_json(j);
    }
    if (cqi_report_aperiodic_proc2_v1310.is_present()) {
      j.write_fieldname("cqi-ReportAperiodicProc2-v1310");
      cqi_report_aperiodic_proc2_v1310->to_json(j);
    }
    if (emimo_type_r13.is_present()) {
      j.write_fieldname("eMIMO-Type-r13");
      emimo_type_r13->to_json(j);
    }
    if (dummy.is_present()) {
      j.write_fieldname("dummy");
      dummy->to_json(j);
    }
    if (emimo_hybrid_r14.is_present()) {
      j.write_fieldname("eMIMO-Hybrid-r14");
      emimo_hybrid_r14->to_json(j);
    }
    if (advanced_codebook_enabled_r14_present) {
      j.write_bool("advancedCodebookEnabled-r14", advanced_codebook_enabled_r14);
    }
    if (emimo_type_v1480.is_present()) {
      j.write_fieldname("eMIMO-Type-v1480");
      emimo_type_v1480->to_json(j);
    }
    if (fe_comp_csi_enabled_v1530_present) {
      j.write_bool("feCOMP-CSI-Enabled-v1530", fe_comp_csi_enabled_v1530);
    }
    if (emimo_type_v1530.is_present()) {
      j.write_fieldname("eMIMO-Type-v1530");
      emimo_type_v1530->to_json(j);
    }
  }
  j.end_obj();
}

void csi_process_r11_s::csi_im_cfg_id_list_r12_c_::set(types::options e)
{
  type_ = e;
}
void csi_process_r11_s::csi_im_cfg_id_list_r12_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.start_array("setup");
      for (const auto& e1 : c) {
        j.write_int(e1);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::csi_im_cfg_id_list_r12_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_process_r11_s::csi_im_cfg_id_list_r12_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 2, integer_packer<uint8_t>(1, 4)));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::csi_im_cfg_id_list_r12_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_process_r11_s::csi_im_cfg_id_list_r12_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 2, integer_packer<uint8_t>(1, 4)));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::csi_im_cfg_id_list_r12_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_::set(types::options e)
{
  type_ = e;
}
void csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc2_r12_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_::set(types::options e)
{
  type_ = e;
}
void csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc_v1310_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_::set(types::options e)
{
  type_ = e;
}
void csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "csi_process_r11_s::cqi_report_aperiodic_proc2_v1310_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// N4SPUCCH-Resource-r15 ::= SEQUENCE
SRSASN_CODE n4_spucch_res_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, n4start_prb_r15, (uint8_t)0u, (uint8_t)109u));
  HANDLE_CODE(pack_integer(bref, n4nof_prb_r15, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE n4_spucch_res_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(n4start_prb_r15, bref, (uint8_t)0u, (uint8_t)109u));
  HANDLE_CODE(unpack_integer(n4nof_prb_r15, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void n4_spucch_res_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("n4startingPRB-r15", n4start_prb_r15);
  j.write_int("n4numberOfPRB-r15", n4nof_prb_r15);
  j.end_obj();
}

// ZeroTxPowerCSI-RS-r12 ::= SEQUENCE
SRSASN_CODE zero_tx_pwr_csi_rs_r12_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(zero_tx_pwr_res_cfg_list_r12.pack(bref));
  HANDLE_CODE(pack_integer(bref, zero_tx_pwr_sf_cfg_r12, (uint8_t)0u, (uint8_t)154u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE zero_tx_pwr_csi_rs_r12_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(zero_tx_pwr_res_cfg_list_r12.unpack(bref));
  HANDLE_CODE(unpack_integer(zero_tx_pwr_sf_cfg_r12, bref, (uint8_t)0u, (uint8_t)154u));

  return SRSASN_SUCCESS;
}
void zero_tx_pwr_csi_rs_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("zeroTxPowerResourceConfigList-r12", zero_tx_pwr_res_cfg_list_r12.to_string());
  j.write_int("zeroTxPowerSubframeConfig-r12", zero_tx_pwr_sf_cfg_r12);
  j.end_obj();
}

// CQI-ReportAperiodic-r10 ::= CHOICE
void cqi_report_aperiodic_r10_c::set(types::options e)
{
  type_ = e;
}
void cqi_report_aperiodic_r10_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("cqi-ReportModeAperiodic-r10", c.cqi_report_mode_aperiodic_r10.to_string());
      if (c.aperiodic_csi_trigger_r10_present) {
        j.write_fieldname("aperiodicCSI-Trigger-r10");
        j.start_obj();
        j.write_str("trigger1-r10", c.aperiodic_csi_trigger_r10.trigger1_r10.to_string());
        j.write_str("trigger2-r10", c.aperiodic_csi_trigger_r10.trigger2_r10.to_string());
        j.end_obj();
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_r10_c");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_aperiodic_r10_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.aperiodic_csi_trigger_r10_present, 1));
      HANDLE_CODE(c.cqi_report_mode_aperiodic_r10.pack(bref));
      if (c.aperiodic_csi_trigger_r10_present) {
        HANDLE_CODE(c.aperiodic_csi_trigger_r10.trigger1_r10.pack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_r10.trigger2_r10.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_r10_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_r10_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.aperiodic_csi_trigger_r10_present, 1));
      HANDLE_CODE(c.cqi_report_mode_aperiodic_r10.unpack(bref));
      if (c.aperiodic_csi_trigger_r10_present) {
        HANDLE_CODE(c.aperiodic_csi_trigger_r10.trigger1_r10.unpack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_r10.trigger2_r10.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_r10_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportAperiodic-v1250 ::= CHOICE
void cqi_report_aperiodic_v1250_c::set(types::options e)
{
  type_ = e;
}
void cqi_report_aperiodic_v1250_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_fieldname("aperiodicCSI-Trigger-v1250");
      j.start_obj();
      j.write_str("trigger-SubframeSetIndicator-r12", c.aperiodic_csi_trigger_v1250.trigger_sf_set_ind_r12.to_string());
      j.write_str("trigger1-SubframeSetIndicator-r12",
                  c.aperiodic_csi_trigger_v1250.trigger1_sf_set_ind_r12.to_string());
      j.write_str("trigger2-SubframeSetIndicator-r12",
                  c.aperiodic_csi_trigger_v1250.trigger2_sf_set_ind_r12.to_string());
      j.end_obj();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1250_c");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_aperiodic_v1250_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.aperiodic_csi_trigger_v1250.trigger_sf_set_ind_r12.pack(bref));
      HANDLE_CODE(c.aperiodic_csi_trigger_v1250.trigger1_sf_set_ind_r12.pack(bref));
      HANDLE_CODE(c.aperiodic_csi_trigger_v1250.trigger2_sf_set_ind_r12.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1250_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_v1250_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.aperiodic_csi_trigger_v1250.trigger_sf_set_ind_r12.unpack(bref));
      HANDLE_CODE(c.aperiodic_csi_trigger_v1250.trigger1_sf_set_ind_r12.unpack(bref));
      HANDLE_CODE(c.aperiodic_csi_trigger_v1250.trigger2_sf_set_ind_r12.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1250_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportAperiodic-v1310 ::= CHOICE
void cqi_report_aperiodic_v1310_c::set(types::options e)
{
  type_ = e;
}
void cqi_report_aperiodic_v1310_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.aperiodic_csi_trigger_v1310_present) {
        j.write_fieldname("aperiodicCSI-Trigger-v1310");
        j.start_obj();
        j.write_str("trigger1-r13", c.aperiodic_csi_trigger_v1310.trigger1_r13.to_string());
        j.write_str("trigger2-r13", c.aperiodic_csi_trigger_v1310.trigger2_r13.to_string());
        j.write_str("trigger3-r13", c.aperiodic_csi_trigger_v1310.trigger3_r13.to_string());
        j.write_str("trigger4-r13", c.aperiodic_csi_trigger_v1310.trigger4_r13.to_string());
        j.write_str("trigger5-r13", c.aperiodic_csi_trigger_v1310.trigger5_r13.to_string());
        j.write_str("trigger6-r13", c.aperiodic_csi_trigger_v1310.trigger6_r13.to_string());
        j.end_obj();
      }
      if (c.aperiodic_csi_trigger2_r13_present) {
        j.write_fieldname("aperiodicCSI-Trigger2-r13");
        c.aperiodic_csi_trigger2_r13.to_json(j);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1310_c");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_aperiodic_v1310_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.aperiodic_csi_trigger_v1310_present, 1));
      HANDLE_CODE(bref.pack(c.aperiodic_csi_trigger2_r13_present, 1));
      if (c.aperiodic_csi_trigger_v1310_present) {
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger1_r13.pack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger2_r13.pack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger3_r13.pack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger4_r13.pack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger5_r13.pack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger6_r13.pack(bref));
      }
      if (c.aperiodic_csi_trigger2_r13_present) {
        HANDLE_CODE(c.aperiodic_csi_trigger2_r13.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1310_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_v1310_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.aperiodic_csi_trigger_v1310_present, 1));
      HANDLE_CODE(bref.unpack(c.aperiodic_csi_trigger2_r13_present, 1));
      if (c.aperiodic_csi_trigger_v1310_present) {
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger1_r13.unpack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger2_r13.unpack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger3_r13.unpack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger4_r13.unpack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger5_r13.unpack(bref));
        HANDLE_CODE(c.aperiodic_csi_trigger_v1310.trigger6_r13.unpack(bref));
      }
      if (c.aperiodic_csi_trigger2_r13_present) {
        HANDLE_CODE(c.aperiodic_csi_trigger2_r13.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1310_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_::set(types::options e)
{
  type_ = e;
}
void cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("trigger1-SubframeSetIndicator-r13", c.trigger1_sf_set_ind_r13.to_string());
      j.write_str("trigger2-SubframeSetIndicator-r13", c.trigger2_sf_set_ind_r13.to_string());
      j.write_str("trigger3-SubframeSetIndicator-r13", c.trigger3_sf_set_ind_r13.to_string());
      j.write_str("trigger4-SubframeSetIndicator-r13", c.trigger4_sf_set_ind_r13.to_string());
      j.write_str("trigger5-SubframeSetIndicator-r13", c.trigger5_sf_set_ind_r13.to_string());
      j.write_str("trigger6-SubframeSetIndicator-r13", c.trigger6_sf_set_ind_r13.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.trigger1_sf_set_ind_r13.pack(bref));
      HANDLE_CODE(c.trigger2_sf_set_ind_r13.pack(bref));
      HANDLE_CODE(c.trigger3_sf_set_ind_r13.pack(bref));
      HANDLE_CODE(c.trigger4_sf_set_ind_r13.pack(bref));
      HANDLE_CODE(c.trigger5_sf_set_ind_r13.pack(bref));
      HANDLE_CODE(c.trigger6_sf_set_ind_r13.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.trigger1_sf_set_ind_r13.unpack(bref));
      HANDLE_CODE(c.trigger2_sf_set_ind_r13.unpack(bref));
      HANDLE_CODE(c.trigger3_sf_set_ind_r13.unpack(bref));
      HANDLE_CODE(c.trigger4_sf_set_ind_r13.unpack(bref));
      HANDLE_CODE(c.trigger5_sf_set_ind_r13.unpack(bref));
      HANDLE_CODE(c.trigger6_sf_set_ind_r13.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_v1310_c::setup_s_::aperiodic_csi_trigger2_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportAperiodicHybrid-r14 ::= SEQUENCE
SRSASN_CODE cqi_report_aperiodic_hybrid_r14_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(triggers_r14_present, 1));

  if (triggers_r14_present) {
    HANDLE_CODE(triggers_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_hybrid_r14_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(triggers_r14_present, 1));

  if (triggers_r14_present) {
    HANDLE_CODE(triggers_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_aperiodic_hybrid_r14_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (triggers_r14_present) {
    j.write_fieldname("triggers-r14");
    triggers_r14.to_json(j);
  }
  j.end_obj();
}

void cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::destroy_()
{
  switch (type_) {
    case types::one_bit_r14:
      c.destroy<one_bit_r14_s_>();
      break;
    case types::two_bit_r14:
      c.destroy<two_bit_r14_s_>();
      break;
    case types::three_bit_r14:
      c.destroy<three_bit_r14_s_>();
      break;
    default:
      break;
  }
}
void cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::one_bit_r14:
      c.init<one_bit_r14_s_>();
      break;
    case types::two_bit_r14:
      c.init<two_bit_r14_s_>();
      break;
    case types::three_bit_r14:
      c.init<three_bit_r14_s_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_");
  }
}
cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::triggers_r14_c_(
    const cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::one_bit_r14:
      c.init(other.c.get<one_bit_r14_s_>());
      break;
    case types::two_bit_r14:
      c.init(other.c.get<two_bit_r14_s_>());
      break;
    case types::three_bit_r14:
      c.init(other.c.get<three_bit_r14_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_");
  }
}
cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_& cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::
                                                    operator=(const cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::one_bit_r14:
      c.set(other.c.get<one_bit_r14_s_>());
      break;
    case types::two_bit_r14:
      c.set(other.c.get<two_bit_r14_s_>());
      break;
    case types::three_bit_r14:
      c.set(other.c.get<three_bit_r14_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_");
  }

  return *this;
}
void cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::one_bit_r14:
      j.write_fieldname("oneBit-r14");
      j.start_obj();
      j.write_str("trigger1-Indicator-r14", c.get<one_bit_r14_s_>().trigger1_ind_r14.to_string());
      j.end_obj();
      break;
    case types::two_bit_r14:
      j.write_fieldname("twoBit-r14");
      j.start_obj();
      j.write_str("trigger01-Indicator-r14", c.get<two_bit_r14_s_>().trigger01_ind_r14.to_string());
      j.write_str("trigger10-Indicator-r14", c.get<two_bit_r14_s_>().trigger10_ind_r14.to_string());
      j.write_str("trigger11-Indicator-r14", c.get<two_bit_r14_s_>().trigger11_ind_r14.to_string());
      j.end_obj();
      break;
    case types::three_bit_r14:
      j.write_fieldname("threeBit-r14");
      j.start_obj();
      j.write_str("trigger001-Indicator-r14", c.get<three_bit_r14_s_>().trigger001_ind_r14.to_string());
      j.write_str("trigger010-Indicator-r14", c.get<three_bit_r14_s_>().trigger010_ind_r14.to_string());
      j.write_str("trigger011-Indicator-r14", c.get<three_bit_r14_s_>().trigger011_ind_r14.to_string());
      j.write_str("trigger100-Indicator-r14", c.get<three_bit_r14_s_>().trigger100_ind_r14.to_string());
      j.write_str("trigger101-Indicator-r14", c.get<three_bit_r14_s_>().trigger101_ind_r14.to_string());
      j.write_str("trigger110-Indicator-r14", c.get<three_bit_r14_s_>().trigger110_ind_r14.to_string());
      j.write_str("trigger111-Indicator-r14", c.get<three_bit_r14_s_>().trigger111_ind_r14.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::one_bit_r14:
      HANDLE_CODE(c.get<one_bit_r14_s_>().trigger1_ind_r14.pack(bref));
      break;
    case types::two_bit_r14:
      HANDLE_CODE(c.get<two_bit_r14_s_>().trigger01_ind_r14.pack(bref));
      HANDLE_CODE(c.get<two_bit_r14_s_>().trigger10_ind_r14.pack(bref));
      HANDLE_CODE(c.get<two_bit_r14_s_>().trigger11_ind_r14.pack(bref));
      break;
    case types::three_bit_r14:
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger001_ind_r14.pack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger010_ind_r14.pack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger011_ind_r14.pack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger100_ind_r14.pack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger101_ind_r14.pack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger110_ind_r14.pack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger111_ind_r14.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::one_bit_r14:
      HANDLE_CODE(c.get<one_bit_r14_s_>().trigger1_ind_r14.unpack(bref));
      break;
    case types::two_bit_r14:
      HANDLE_CODE(c.get<two_bit_r14_s_>().trigger01_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<two_bit_r14_s_>().trigger10_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<two_bit_r14_s_>().trigger11_ind_r14.unpack(bref));
      break;
    case types::three_bit_r14:
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger001_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger010_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger011_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger100_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger101_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger110_ind_r14.unpack(bref));
      HANDLE_CODE(c.get<three_bit_r14_s_>().trigger111_ind_r14.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_aperiodic_hybrid_r14_s::triggers_r14_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportBoth-r11 ::= SEQUENCE
SRSASN_CODE cqi_report_both_r11_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_im_cfg_to_release_list_r11_present, 1));
  HANDLE_CODE(bref.pack(csi_im_cfg_to_add_mod_list_r11_present, 1));
  HANDLE_CODE(bref.pack(csi_process_to_release_list_r11_present, 1));
  HANDLE_CODE(bref.pack(csi_process_to_add_mod_list_r11_present, 1));

  if (csi_im_cfg_to_release_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_to_release_list_r11, 1, 3, integer_packer<uint8_t>(1, 3)));
  }
  if (csi_im_cfg_to_add_mod_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_to_add_mod_list_r11, 1, 3));
  }
  if (csi_process_to_release_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_process_to_release_list_r11, 1, 4, integer_packer<uint8_t>(1, 4)));
  }
  if (csi_process_to_add_mod_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_process_to_add_mod_list_r11, 1, 4));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_both_r11_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_im_cfg_to_release_list_r11_present, 1));
  HANDLE_CODE(bref.unpack(csi_im_cfg_to_add_mod_list_r11_present, 1));
  HANDLE_CODE(bref.unpack(csi_process_to_release_list_r11_present, 1));
  HANDLE_CODE(bref.unpack(csi_process_to_add_mod_list_r11_present, 1));

  if (csi_im_cfg_to_release_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_to_release_list_r11, bref, 1, 3, integer_packer<uint8_t>(1, 3)));
  }
  if (csi_im_cfg_to_add_mod_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_to_add_mod_list_r11, bref, 1, 3));
  }
  if (csi_process_to_release_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_process_to_release_list_r11, bref, 1, 4, integer_packer<uint8_t>(1, 4)));
  }
  if (csi_process_to_add_mod_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_process_to_add_mod_list_r11, bref, 1, 4));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_both_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_im_cfg_to_release_list_r11_present) {
    j.start_array("csi-IM-ConfigToReleaseList-r11");
    for (const auto& e1 : csi_im_cfg_to_release_list_r11) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (csi_im_cfg_to_add_mod_list_r11_present) {
    j.start_array("csi-IM-ConfigToAddModList-r11");
    for (const auto& e1 : csi_im_cfg_to_add_mod_list_r11) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (csi_process_to_release_list_r11_present) {
    j.start_array("csi-ProcessToReleaseList-r11");
    for (const auto& e1 : csi_process_to_release_list_r11) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (csi_process_to_add_mod_list_r11_present) {
    j.start_array("csi-ProcessToAddModList-r11");
    for (const auto& e1 : csi_process_to_add_mod_list_r11) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// CQI-ReportBoth-v1250 ::= SEQUENCE
SRSASN_CODE cqi_report_both_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_im_cfg_to_release_list_ext_r12_present, 1));
  HANDLE_CODE(bref.pack(csi_im_cfg_to_add_mod_list_ext_r12_present, 1));

  if (csi_im_cfg_to_release_list_ext_r12_present) {
    HANDLE_CODE(pack_integer(bref, csi_im_cfg_to_release_list_ext_r12, (uint8_t)4u, (uint8_t)4u));
  }
  if (csi_im_cfg_to_add_mod_list_ext_r12_present) {
    HANDLE_CODE(csi_im_cfg_to_add_mod_list_ext_r12.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_both_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_im_cfg_to_release_list_ext_r12_present, 1));
  HANDLE_CODE(bref.unpack(csi_im_cfg_to_add_mod_list_ext_r12_present, 1));

  if (csi_im_cfg_to_release_list_ext_r12_present) {
    HANDLE_CODE(unpack_integer(csi_im_cfg_to_release_list_ext_r12, bref, (uint8_t)4u, (uint8_t)4u));
  }
  if (csi_im_cfg_to_add_mod_list_ext_r12_present) {
    HANDLE_CODE(csi_im_cfg_to_add_mod_list_ext_r12.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_both_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_im_cfg_to_release_list_ext_r12_present) {
    j.write_int("csi-IM-ConfigToReleaseListExt-r12", csi_im_cfg_to_release_list_ext_r12);
  }
  if (csi_im_cfg_to_add_mod_list_ext_r12_present) {
    j.write_fieldname("csi-IM-ConfigToAddModListExt-r12");
    csi_im_cfg_to_add_mod_list_ext_r12.to_json(j);
  }
  j.end_obj();
}

// CQI-ReportBoth-v1310 ::= SEQUENCE
SRSASN_CODE cqi_report_both_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_im_cfg_to_release_list_ext_r13_present, 1));
  HANDLE_CODE(bref.pack(csi_im_cfg_to_add_mod_list_ext_r13_present, 1));

  if (csi_im_cfg_to_release_list_ext_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_to_release_list_ext_r13, 1, 20, integer_packer<uint8_t>(5, 24)));
  }
  if (csi_im_cfg_to_add_mod_list_ext_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, csi_im_cfg_to_add_mod_list_ext_r13, 1, 20));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_both_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_im_cfg_to_release_list_ext_r13_present, 1));
  HANDLE_CODE(bref.unpack(csi_im_cfg_to_add_mod_list_ext_r13_present, 1));

  if (csi_im_cfg_to_release_list_ext_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_to_release_list_ext_r13, bref, 1, 20, integer_packer<uint8_t>(5, 24)));
  }
  if (csi_im_cfg_to_add_mod_list_ext_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(csi_im_cfg_to_add_mod_list_ext_r13, bref, 1, 20));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_both_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_im_cfg_to_release_list_ext_r13_present) {
    j.start_array("csi-IM-ConfigToReleaseListExt-r13");
    for (const auto& e1 : csi_im_cfg_to_release_list_ext_r13) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (csi_im_cfg_to_add_mod_list_ext_r13_present) {
    j.start_array("csi-IM-ConfigToAddModListExt-r13");
    for (const auto& e1 : csi_im_cfg_to_add_mod_list_ext_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// CQI-ReportPeriodic-r10 ::= CHOICE
void cqi_report_periodic_r10_c::set(types::options e)
{
  type_ = e;
}
void cqi_report_periodic_r10_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("cqi-PUCCH-ResourceIndex-r10", c.cqi_pucch_res_idx_r10);
      if (c.cqi_pucch_res_idx_p1_r10_present) {
        j.write_int("cqi-PUCCH-ResourceIndexP1-r10", c.cqi_pucch_res_idx_p1_r10);
      }
      j.write_int("cqi-pmi-ConfigIndex", c.cqi_pmi_cfg_idx);
      j.write_fieldname("cqi-FormatIndicatorPeriodic-r10");
      c.cqi_format_ind_periodic_r10.to_json(j);
      if (c.ri_cfg_idx_present) {
        j.write_int("ri-ConfigIndex", c.ri_cfg_idx);
      }
      j.write_bool("simultaneousAckNackAndCQI", c.simul_ack_nack_and_cqi);
      if (c.cqi_mask_r9_present) {
        j.write_str("cqi-Mask-r9", "setup");
      }
      if (c.csi_cfg_idx_r10_present) {
        j.write_fieldname("csi-ConfigIndex-r10");
        c.csi_cfg_idx_r10.to_json(j);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_r10_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.cqi_pucch_res_idx_p1_r10_present, 1));
      HANDLE_CODE(bref.pack(c.ri_cfg_idx_present, 1));
      HANDLE_CODE(bref.pack(c.cqi_mask_r9_present, 1));
      HANDLE_CODE(bref.pack(c.csi_cfg_idx_r10_present, 1));
      HANDLE_CODE(pack_integer(bref, c.cqi_pucch_res_idx_r10, (uint16_t)0u, (uint16_t)1184u));
      if (c.cqi_pucch_res_idx_p1_r10_present) {
        HANDLE_CODE(pack_integer(bref, c.cqi_pucch_res_idx_p1_r10, (uint16_t)0u, (uint16_t)1184u));
      }
      HANDLE_CODE(pack_integer(bref, c.cqi_pmi_cfg_idx, (uint16_t)0u, (uint16_t)1023u));
      HANDLE_CODE(c.cqi_format_ind_periodic_r10.pack(bref));
      if (c.ri_cfg_idx_present) {
        HANDLE_CODE(pack_integer(bref, c.ri_cfg_idx, (uint16_t)0u, (uint16_t)1023u));
      }
      HANDLE_CODE(bref.pack(c.simul_ack_nack_and_cqi, 1));
      if (c.csi_cfg_idx_r10_present) {
        HANDLE_CODE(c.csi_cfg_idx_r10.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_r10_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.cqi_pucch_res_idx_p1_r10_present, 1));
      HANDLE_CODE(bref.unpack(c.ri_cfg_idx_present, 1));
      HANDLE_CODE(bref.unpack(c.cqi_mask_r9_present, 1));
      HANDLE_CODE(bref.unpack(c.csi_cfg_idx_r10_present, 1));
      HANDLE_CODE(unpack_integer(c.cqi_pucch_res_idx_r10, bref, (uint16_t)0u, (uint16_t)1184u));
      if (c.cqi_pucch_res_idx_p1_r10_present) {
        HANDLE_CODE(unpack_integer(c.cqi_pucch_res_idx_p1_r10, bref, (uint16_t)0u, (uint16_t)1184u));
      }
      HANDLE_CODE(unpack_integer(c.cqi_pmi_cfg_idx, bref, (uint16_t)0u, (uint16_t)1023u));
      HANDLE_CODE(c.cqi_format_ind_periodic_r10.unpack(bref));
      if (c.ri_cfg_idx_present) {
        HANDLE_CODE(unpack_integer(c.ri_cfg_idx, bref, (uint16_t)0u, (uint16_t)1023u));
      }
      HANDLE_CODE(bref.unpack(c.simul_ack_nack_and_cqi, 1));
      if (c.csi_cfg_idx_r10_present) {
        HANDLE_CODE(c.csi_cfg_idx_r10.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::destroy_()
{
  switch (type_) {
    case types::wideband_cqi_r10:
      c.destroy<wideband_cqi_r10_s_>();
      break;
    case types::subband_cqi_r10:
      c.destroy<subband_cqi_r10_s_>();
      break;
    default:
      break;
  }
}
void cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::wideband_cqi_r10:
      c.init<wideband_cqi_r10_s_>();
      break;
    case types::subband_cqi_r10:
      c.init<subband_cqi_r10_s_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_");
  }
}
cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::cqi_format_ind_periodic_r10_c_(
    const cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::wideband_cqi_r10:
      c.init(other.c.get<wideband_cqi_r10_s_>());
      break;
    case types::subband_cqi_r10:
      c.init(other.c.get<subband_cqi_r10_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_");
  }
}
cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_&
cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::
operator=(const cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::wideband_cqi_r10:
      c.set(other.c.get<wideband_cqi_r10_s_>());
      break;
    case types::subband_cqi_r10:
      c.set(other.c.get<subband_cqi_r10_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_");
  }

  return *this;
}
void cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::wideband_cqi_r10:
      j.write_fieldname("widebandCQI-r10");
      j.start_obj();
      if (c.get<wideband_cqi_r10_s_>().csi_report_mode_r10_present) {
        j.write_str("csi-ReportMode-r10", c.get<wideband_cqi_r10_s_>().csi_report_mode_r10.to_string());
      }
      j.end_obj();
      break;
    case types::subband_cqi_r10:
      j.write_fieldname("subbandCQI-r10");
      j.start_obj();
      j.write_int("k", c.get<subband_cqi_r10_s_>().k);
      j.write_str("periodicityFactor-r10", c.get<subband_cqi_r10_s_>().periodicity_factor_r10.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::wideband_cqi_r10:
      HANDLE_CODE(bref.pack(c.get<wideband_cqi_r10_s_>().csi_report_mode_r10_present, 1));
      if (c.get<wideband_cqi_r10_s_>().csi_report_mode_r10_present) {
        HANDLE_CODE(c.get<wideband_cqi_r10_s_>().csi_report_mode_r10.pack(bref));
      }
      break;
    case types::subband_cqi_r10:
      HANDLE_CODE(pack_integer(bref, c.get<subband_cqi_r10_s_>().k, (uint8_t)1u, (uint8_t)4u));
      HANDLE_CODE(c.get<subband_cqi_r10_s_>().periodicity_factor_r10.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::wideband_cqi_r10:
      HANDLE_CODE(bref.unpack(c.get<wideband_cqi_r10_s_>().csi_report_mode_r10_present, 1));
      if (c.get<wideband_cqi_r10_s_>().csi_report_mode_r10_present) {
        HANDLE_CODE(c.get<wideband_cqi_r10_s_>().csi_report_mode_r10.unpack(bref));
      }
      break;
    case types::subband_cqi_r10:
      HANDLE_CODE(unpack_integer(c.get<subband_cqi_r10_s_>().k, bref, (uint8_t)1u, (uint8_t)4u));
      HANDLE_CODE(c.get<subband_cqi_r10_s_>().periodicity_factor_r10.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::cqi_format_ind_periodic_r10_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_::set(types::options e)
{
  type_ = e;
}
void cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("cqi-pmi-ConfigIndex2-r10", c.cqi_pmi_cfg_idx2_r10);
      if (c.ri_cfg_idx2_r10_present) {
        j.write_int("ri-ConfigIndex2-r10", c.ri_cfg_idx2_r10);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.ri_cfg_idx2_r10_present, 1));
      HANDLE_CODE(pack_integer(bref, c.cqi_pmi_cfg_idx2_r10, (uint16_t)0u, (uint16_t)1023u));
      if (c.ri_cfg_idx2_r10_present) {
        HANDLE_CODE(pack_integer(bref, c.ri_cfg_idx2_r10, (uint16_t)0u, (uint16_t)1023u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.ri_cfg_idx2_r10_present, 1));
      HANDLE_CODE(unpack_integer(c.cqi_pmi_cfg_idx2_r10, bref, (uint16_t)0u, (uint16_t)1023u));
      if (c.ri_cfg_idx2_r10_present) {
        HANDLE_CODE(unpack_integer(c.ri_cfg_idx2_r10, bref, (uint16_t)0u, (uint16_t)1023u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_r10_c::setup_s_::csi_cfg_idx_r10_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportPeriodic-v1130 ::= SEQUENCE
SRSASN_CODE cqi_report_periodic_v1130_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(simul_ack_nack_and_cqi_format3_r11_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_periodic_proc_ext_to_release_list_r11_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_periodic_proc_ext_to_add_mod_list_r11_present, 1));

  if (cqi_report_periodic_proc_ext_to_release_list_r11_present) {
    HANDLE_CODE(
        pack_dyn_seq_of(bref, cqi_report_periodic_proc_ext_to_release_list_r11, 1, 3, integer_packer<uint8_t>(1, 3)));
  }
  if (cqi_report_periodic_proc_ext_to_add_mod_list_r11_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, cqi_report_periodic_proc_ext_to_add_mod_list_r11, 1, 3));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_v1130_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(simul_ack_nack_and_cqi_format3_r11_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_periodic_proc_ext_to_release_list_r11_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_periodic_proc_ext_to_add_mod_list_r11_present, 1));

  if (cqi_report_periodic_proc_ext_to_release_list_r11_present) {
    HANDLE_CODE(
        unpack_dyn_seq_of(cqi_report_periodic_proc_ext_to_release_list_r11, bref, 1, 3, integer_packer<uint8_t>(1, 3)));
  }
  if (cqi_report_periodic_proc_ext_to_add_mod_list_r11_present) {
    HANDLE_CODE(unpack_dyn_seq_of(cqi_report_periodic_proc_ext_to_add_mod_list_r11, bref, 1, 3));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_periodic_v1130_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (simul_ack_nack_and_cqi_format3_r11_present) {
    j.write_str("simultaneousAckNackAndCQI-Format3-r11", "setup");
  }
  if (cqi_report_periodic_proc_ext_to_release_list_r11_present) {
    j.start_array("cqi-ReportPeriodicProcExtToReleaseList-r11");
    for (const auto& e1 : cqi_report_periodic_proc_ext_to_release_list_r11) {
      j.write_int(e1);
    }
    j.end_array();
  }
  if (cqi_report_periodic_proc_ext_to_add_mod_list_r11_present) {
    j.start_array("cqi-ReportPeriodicProcExtToAddModList-r11");
    for (const auto& e1 : cqi_report_periodic_proc_ext_to_add_mod_list_r11) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.end_obj();
}

// CQI-ReportPeriodic-v1310 ::= SEQUENCE
SRSASN_CODE cqi_report_periodic_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cri_report_cfg_r13_present, 1));
  HANDLE_CODE(bref.pack(simul_ack_nack_and_cqi_format4_format5_r13_present, 1));

  if (cri_report_cfg_r13_present) {
    HANDLE_CODE(cri_report_cfg_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cri_report_cfg_r13_present, 1));
  HANDLE_CODE(bref.unpack(simul_ack_nack_and_cqi_format4_format5_r13_present, 1));

  if (cri_report_cfg_r13_present) {
    HANDLE_CODE(cri_report_cfg_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_periodic_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cri_report_cfg_r13_present) {
    j.write_fieldname("cri-ReportConfig-r13");
    cri_report_cfg_r13.to_json(j);
  }
  if (simul_ack_nack_and_cqi_format4_format5_r13_present) {
    j.write_str("simultaneousAckNackAndCQI-Format4-Format5-r13", "setup");
  }
  j.end_obj();
}

// CQI-ReportPeriodic-v1320 ::= SEQUENCE
SRSASN_CODE cqi_report_periodic_v1320_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(periodicity_factor_wb_r13_present, 1));

  if (periodicity_factor_wb_r13_present) {
    HANDLE_CODE(periodicity_factor_wb_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_v1320_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(periodicity_factor_wb_r13_present, 1));

  if (periodicity_factor_wb_r13_present) {
    HANDLE_CODE(periodicity_factor_wb_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_periodic_v1320_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (periodicity_factor_wb_r13_present) {
    j.write_str("periodicityFactorWB-r13", periodicity_factor_wb_r13.to_string());
  }
  j.end_obj();
}

// DL-AM-RLC-r15 ::= SEQUENCE
SRSASN_CODE dl_am_rlc_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(t_reordering_r15.pack(bref));
  HANDLE_CODE(t_status_prohibit_r15.pack(bref));
  HANDLE_CODE(bref.pack(extended_rlc_li_field_r15, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_am_rlc_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(t_reordering_r15.unpack(bref));
  HANDLE_CODE(t_status_prohibit_r15.unpack(bref));
  HANDLE_CODE(bref.unpack(extended_rlc_li_field_r15, 1));

  return SRSASN_SUCCESS;
}
void dl_am_rlc_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("t-Reordering-r15", t_reordering_r15.to_string());
  j.write_str("t-StatusProhibit-r15", t_status_prohibit_r15.to_string());
  j.write_bool("extended-RLC-LI-Field-r15", extended_rlc_li_field_r15);
  j.end_obj();
}

// DL-UM-RLC-r15 ::= SEQUENCE
SRSASN_CODE dl_um_rlc_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sn_field_len_r15.pack(bref));
  HANDLE_CODE(t_reordering_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_um_rlc_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sn_field_len_r15.unpack(bref));
  HANDLE_CODE(t_reordering_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_um_rlc_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sn-FieldLength-r15", sn_field_len_r15.to_string());
  j.write_str("t-Reordering-r15", t_reordering_r15.to_string());
  j.end_obj();
}

// MeasSubframePattern-r10 ::= CHOICE
void meas_sf_pattern_r10_c::destroy_()
{
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      c.destroy<fixed_bitstring<40> >();
      break;
    case types::sf_pattern_tdd_r10:
      c.destroy<sf_pattern_tdd_r10_c_>();
      break;
    default:
      break;
  }
}
void meas_sf_pattern_r10_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      c.init<fixed_bitstring<40> >();
      break;
    case types::sf_pattern_tdd_r10:
      c.init<sf_pattern_tdd_r10_c_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c");
  }
}
meas_sf_pattern_r10_c::meas_sf_pattern_r10_c(const meas_sf_pattern_r10_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      c.init(other.c.get<fixed_bitstring<40> >());
      break;
    case types::sf_pattern_tdd_r10:
      c.init(other.c.get<sf_pattern_tdd_r10_c_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c");
  }
}
meas_sf_pattern_r10_c& meas_sf_pattern_r10_c::operator=(const meas_sf_pattern_r10_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      c.set(other.c.get<fixed_bitstring<40> >());
      break;
    case types::sf_pattern_tdd_r10:
      c.set(other.c.get<sf_pattern_tdd_r10_c_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c");
  }

  return *this;
}
void meas_sf_pattern_r10_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      j.write_str("subframePatternFDD-r10", c.get<fixed_bitstring<40> >().to_string());
      break;
    case types::sf_pattern_tdd_r10:
      j.write_fieldname("subframePatternTDD-r10");
      c.get<sf_pattern_tdd_r10_c_>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c");
  }
  j.end_obj();
}
SRSASN_CODE meas_sf_pattern_r10_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().pack(bref));
      break;
    case types::sf_pattern_tdd_r10:
      HANDLE_CODE(c.get<sf_pattern_tdd_r10_c_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE meas_sf_pattern_r10_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sf_pattern_fdd_r10:
      HANDLE_CODE(c.get<fixed_bitstring<40> >().unpack(bref));
      break;
    case types::sf_pattern_tdd_r10:
      HANDLE_CODE(c.get<sf_pattern_tdd_r10_c_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::destroy_()
{
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      c.destroy<fixed_bitstring<20> >();
      break;
    case types::sf_cfg0_r10:
      c.destroy<fixed_bitstring<70> >();
      break;
    case types::sf_cfg6_r10:
      c.destroy<fixed_bitstring<60> >();
      break;
    default:
      break;
  }
}
void meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      c.init<fixed_bitstring<20> >();
      break;
    case types::sf_cfg0_r10:
      c.init<fixed_bitstring<70> >();
      break;
    case types::sf_cfg6_r10:
      c.init<fixed_bitstring<60> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_");
  }
}
meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::sf_pattern_tdd_r10_c_(
    const meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      c.init(other.c.get<fixed_bitstring<20> >());
      break;
    case types::sf_cfg0_r10:
      c.init(other.c.get<fixed_bitstring<70> >());
      break;
    case types::sf_cfg6_r10:
      c.init(other.c.get<fixed_bitstring<60> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_");
  }
}
meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_& meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::
                                              operator=(const meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      c.set(other.c.get<fixed_bitstring<20> >());
      break;
    case types::sf_cfg0_r10:
      c.set(other.c.get<fixed_bitstring<70> >());
      break;
    case types::sf_cfg6_r10:
      c.set(other.c.get<fixed_bitstring<60> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_");
  }

  return *this;
}
void meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      j.write_str("subframeConfig1-5-r10", c.get<fixed_bitstring<20> >().to_string());
      break;
    case types::sf_cfg0_r10:
      j.write_str("subframeConfig0-r10", c.get<fixed_bitstring<70> >().to_string());
      break;
    case types::sf_cfg6_r10:
      j.write_str("subframeConfig6-r10", c.get<fixed_bitstring<60> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_");
  }
  j.end_obj();
}
SRSASN_CODE meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      HANDLE_CODE(c.get<fixed_bitstring<20> >().pack(bref));
      break;
    case types::sf_cfg0_r10:
      HANDLE_CODE(c.get<fixed_bitstring<70> >().pack(bref));
      break;
    case types::sf_cfg6_r10:
      HANDLE_CODE(c.get<fixed_bitstring<60> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::sf_cfg1_minus5_r10:
      HANDLE_CODE(c.get<fixed_bitstring<20> >().unpack(bref));
      break;
    case types::sf_cfg0_r10:
      HANDLE_CODE(c.get<fixed_bitstring<70> >().unpack(bref));
      break;
    case types::sf_cfg6_r10:
      HANDLE_CODE(c.get<fixed_bitstring<60> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "meas_sf_pattern_r10_c::sf_pattern_tdd_r10_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SPDCCH-Elements-r15 ::= CHOICE
void spdcch_elems_r15_c::set(types::options e)
{
  type_ = e;
}
void spdcch_elems_r15_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.spdcch_set_cfg_id_r15_present) {
        j.write_int("spdcch-SetConfigId-r15", c.spdcch_set_cfg_id_r15);
      }
      if (c.spdcch_set_ref_sig_r15_present) {
        j.write_str("spdcch-SetReferenceSig-r15", c.spdcch_set_ref_sig_r15.to_string());
      }
      if (c.tx_type_r15_present) {
        j.write_str("transmissionType-r15", c.tx_type_r15.to_string());
      }
      if (c.spdcch_no_of_symbols_r15_present) {
        j.write_int("spdcch-NoOfSymbols-r15", c.spdcch_no_of_symbols_r15);
      }
      if (c.dmrs_scrambling_seq_int_r15_present) {
        j.write_int("dmrs-ScramblingSequenceInt-r15", c.dmrs_scrambling_seq_int_r15);
      }
      if (c.dci7_candidates_per_al_pdcch_r15_present) {
        j.start_array("dci7-CandidatesPerAL-PDCCH-r15");
        for (const auto& e1 : c.dci7_candidates_per_al_pdcch_r15) {
          j.write_int(e1);
        }
        j.end_array();
      }
      if (c.dci7_candidate_sets_per_al_spdcch_r15_present) {
        j.start_array("dci7-CandidateSetsPerAL-SPDCCH-r15");
        for (const auto& e1 : c.dci7_candidate_sets_per_al_spdcch_r15) {
          j.start_array();
          for (const auto& e2 : e1) {
            j.write_int(e2);
          }
          j.end_array();
        }
        j.end_array();
      }
      if (c.res_block_assign_r15_present) {
        j.write_fieldname("resourceBlockAssignment-r15");
        j.start_obj();
        j.write_int("numberRB-InFreq-domain-r15", c.res_block_assign_r15.num_rb_in_freq_domain_r15);
        j.write_str("resourceBlockAssignment-r15", c.res_block_assign_r15.res_block_assign_r15.to_string());
        j.end_obj();
      }
      if (c.subslot_applicability_r15_present) {
        j.write_str("subslotApplicability-r15", c.subslot_applicability_r15.to_string());
      }
      if (c.al_start_point_spdcch_r15_present) {
        j.start_array("al-StartingPointSPDCCH-r15");
        for (const auto& e1 : c.al_start_point_spdcch_r15) {
          j.write_int(e1);
        }
        j.end_array();
      }
      if (c.sf_type_r15_present) {
        j.write_str("subframeType-r15", c.sf_type_r15.to_string());
      }
      if (c.rate_matching_mode_r15_present) {
        j.write_str("rateMatchingMode-r15", c.rate_matching_mode_r15.to_string());
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "spdcch_elems_r15_c");
  }
  j.end_obj();
}
SRSASN_CODE spdcch_elems_r15_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      bref.pack(c.ext, 1);
      HANDLE_CODE(bref.pack(c.spdcch_set_cfg_id_r15_present, 1));
      HANDLE_CODE(bref.pack(c.spdcch_set_ref_sig_r15_present, 1));
      HANDLE_CODE(bref.pack(c.tx_type_r15_present, 1));
      HANDLE_CODE(bref.pack(c.spdcch_no_of_symbols_r15_present, 1));
      HANDLE_CODE(bref.pack(c.dmrs_scrambling_seq_int_r15_present, 1));
      HANDLE_CODE(bref.pack(c.dci7_candidates_per_al_pdcch_r15_present, 1));
      HANDLE_CODE(bref.pack(c.dci7_candidate_sets_per_al_spdcch_r15_present, 1));
      HANDLE_CODE(bref.pack(c.res_block_assign_r15_present, 1));
      HANDLE_CODE(bref.pack(c.subslot_applicability_r15_present, 1));
      HANDLE_CODE(bref.pack(c.al_start_point_spdcch_r15_present, 1));
      HANDLE_CODE(bref.pack(c.sf_type_r15_present, 1));
      HANDLE_CODE(bref.pack(c.rate_matching_mode_r15_present, 1));
      if (c.spdcch_set_cfg_id_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.spdcch_set_cfg_id_r15, (uint8_t)0u, (uint8_t)3u));
      }
      if (c.spdcch_set_ref_sig_r15_present) {
        HANDLE_CODE(c.spdcch_set_ref_sig_r15.pack(bref));
      }
      if (c.tx_type_r15_present) {
        HANDLE_CODE(c.tx_type_r15.pack(bref));
      }
      if (c.spdcch_no_of_symbols_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.spdcch_no_of_symbols_r15, (uint8_t)1u, (uint8_t)2u));
      }
      if (c.dmrs_scrambling_seq_int_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.dmrs_scrambling_seq_int_r15, (uint16_t)0u, (uint16_t)503u));
      }
      if (c.dci7_candidates_per_al_pdcch_r15_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.dci7_candidates_per_al_pdcch_r15, 1, 4, integer_packer<uint8_t>(0, 6)));
      }
      if (c.dci7_candidate_sets_per_al_spdcch_r15_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref,
                                    c.dci7_candidate_sets_per_al_spdcch_r15,
                                    1,
                                    2,
                                    SeqOfPacker<integer_packer<uint8_t> >(1, 4, integer_packer<uint8_t>(0, 6))));
      }
      if (c.res_block_assign_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.res_block_assign_r15.num_rb_in_freq_domain_r15, (uint8_t)2u, (uint8_t)100u));
        HANDLE_CODE(c.res_block_assign_r15.res_block_assign_r15.pack(bref));
      }
      if (c.subslot_applicability_r15_present) {
        HANDLE_CODE(c.subslot_applicability_r15.pack(bref));
      }
      if (c.al_start_point_spdcch_r15_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.al_start_point_spdcch_r15, 1, 4, integer_packer<uint8_t>(0, 49)));
      }
      if (c.sf_type_r15_present) {
        HANDLE_CODE(c.sf_type_r15.pack(bref));
      }
      if (c.rate_matching_mode_r15_present) {
        HANDLE_CODE(c.rate_matching_mode_r15.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "spdcch_elems_r15_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE spdcch_elems_r15_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      bref.unpack(c.ext, 1);
      HANDLE_CODE(bref.unpack(c.spdcch_set_cfg_id_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.spdcch_set_ref_sig_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.tx_type_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.spdcch_no_of_symbols_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.dmrs_scrambling_seq_int_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.dci7_candidates_per_al_pdcch_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.dci7_candidate_sets_per_al_spdcch_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.res_block_assign_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.subslot_applicability_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.al_start_point_spdcch_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.sf_type_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.rate_matching_mode_r15_present, 1));
      if (c.spdcch_set_cfg_id_r15_present) {
        HANDLE_CODE(unpack_integer(c.spdcch_set_cfg_id_r15, bref, (uint8_t)0u, (uint8_t)3u));
      }
      if (c.spdcch_set_ref_sig_r15_present) {
        HANDLE_CODE(c.spdcch_set_ref_sig_r15.unpack(bref));
      }
      if (c.tx_type_r15_present) {
        HANDLE_CODE(c.tx_type_r15.unpack(bref));
      }
      if (c.spdcch_no_of_symbols_r15_present) {
        HANDLE_CODE(unpack_integer(c.spdcch_no_of_symbols_r15, bref, (uint8_t)1u, (uint8_t)2u));
      }
      if (c.dmrs_scrambling_seq_int_r15_present) {
        HANDLE_CODE(unpack_integer(c.dmrs_scrambling_seq_int_r15, bref, (uint16_t)0u, (uint16_t)503u));
      }
      if (c.dci7_candidates_per_al_pdcch_r15_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.dci7_candidates_per_al_pdcch_r15, bref, 1, 4, integer_packer<uint8_t>(0, 6)));
      }
      if (c.dci7_candidate_sets_per_al_spdcch_r15_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.dci7_candidate_sets_per_al_spdcch_r15,
                                      bref,
                                      1,
                                      2,
                                      SeqOfPacker<integer_packer<uint8_t> >(1, 4, integer_packer<uint8_t>(0, 6))));
      }
      if (c.res_block_assign_r15_present) {
        HANDLE_CODE(unpack_integer(c.res_block_assign_r15.num_rb_in_freq_domain_r15, bref, (uint8_t)2u, (uint8_t)100u));
        HANDLE_CODE(c.res_block_assign_r15.res_block_assign_r15.unpack(bref));
      }
      if (c.subslot_applicability_r15_present) {
        HANDLE_CODE(c.subslot_applicability_r15.unpack(bref));
      }
      if (c.al_start_point_spdcch_r15_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.al_start_point_spdcch_r15, bref, 1, 4, integer_packer<uint8_t>(0, 49)));
      }
      if (c.sf_type_r15_present) {
        HANDLE_CODE(c.sf_type_r15.unpack(bref));
      }
      if (c.rate_matching_mode_r15_present) {
        HANDLE_CODE(c.rate_matching_mode_r15.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "spdcch_elems_r15_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// SPUCCH-Elements-r15 ::= CHOICE
void spucch_elems_r15_c::set(types::options e)
{
  type_ = e;
}
void spucch_elems_r15_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.n1_subslot_spucch_an_list_r15_present) {
        j.start_array("n1SubslotSPUCCH-AN-List-r15");
        for (const auto& e1 : c.n1_subslot_spucch_an_list_r15) {
          j.write_int(e1);
        }
        j.end_array();
      }
      if (c.n1_slot_spucch_fh_an_list_r15_present) {
        j.write_int("n1SlotSPUCCH-FH-AN-List-r15", c.n1_slot_spucch_fh_an_list_r15);
      }
      if (c.n1_slot_spucch_no_fh_an_list_r15_present) {
        j.write_int("n1SlotSPUCCH-NoFH-AN-List-r15", c.n1_slot_spucch_no_fh_an_list_r15);
      }
      if (c.n3_spucch_an_list_r15_present) {
        j.write_int("n3SPUCCH-AN-List-r15", c.n3_spucch_an_list_r15);
      }
      if (c.n4_spucch_slot_res_r15_present) {
        j.start_array("n4SPUCCHSlot-Resource-r15");
        for (const auto& e1 : c.n4_spucch_slot_res_r15) {
          e1.to_json(j);
        }
        j.end_array();
      }
      if (c.n4_spucch_subslot_res_r15_present) {
        j.start_array("n4SPUCCHSubslot-Resource-r15");
        for (const auto& e1 : c.n4_spucch_subslot_res_r15) {
          e1.to_json(j);
        }
        j.end_array();
      }
      if (c.n4max_coderate_slot_pucch_r15_present) {
        j.write_int("n4maxCoderateSlotPUCCH-r15", c.n4max_coderate_slot_pucch_r15);
      }
      if (c.n4max_coderate_subslot_pucch_r15_present) {
        j.write_int("n4maxCoderateSubslotPUCCH-r15", c.n4max_coderate_subslot_pucch_r15);
      }
      if (c.n4max_coderate_multi_res_slot_pucch_r15_present) {
        j.write_int("n4maxCoderateMultiResourceSlotPUCCH-r15", c.n4max_coderate_multi_res_slot_pucch_r15);
      }
      if (c.n4max_coderate_multi_res_subslot_pucch_r15_present) {
        j.write_int("n4maxCoderateMultiResourceSubslotPUCCH-r15", c.n4max_coderate_multi_res_subslot_pucch_r15);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "spucch_elems_r15_c");
  }
  j.end_obj();
}
SRSASN_CODE spucch_elems_r15_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.n1_subslot_spucch_an_list_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n1_slot_spucch_fh_an_list_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n1_slot_spucch_no_fh_an_list_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n3_spucch_an_list_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n4_spucch_slot_res_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n4_spucch_subslot_res_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n4max_coderate_slot_pucch_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n4max_coderate_subslot_pucch_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n4max_coderate_multi_res_slot_pucch_r15_present, 1));
      HANDLE_CODE(bref.pack(c.n4max_coderate_multi_res_subslot_pucch_r15_present, 1));
      if (c.n1_subslot_spucch_an_list_r15_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.n1_subslot_spucch_an_list_r15, 1, 4, integer_packer<uint16_t>(0, 1319)));
      }
      if (c.n1_slot_spucch_fh_an_list_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n1_slot_spucch_fh_an_list_r15, (uint16_t)0u, (uint16_t)1319u));
      }
      if (c.n1_slot_spucch_no_fh_an_list_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n1_slot_spucch_no_fh_an_list_r15, (uint16_t)0u, (uint16_t)3959u));
      }
      if (c.n3_spucch_an_list_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n3_spucch_an_list_r15, (uint16_t)0u, (uint16_t)549u));
      }
      if (c.n4_spucch_slot_res_r15_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.n4_spucch_slot_res_r15, 1, 2));
      }
      if (c.n4_spucch_subslot_res_r15_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.n4_spucch_subslot_res_r15, 1, 2));
      }
      if (c.n4max_coderate_slot_pucch_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n4max_coderate_slot_pucch_r15, (uint8_t)0u, (uint8_t)7u));
      }
      if (c.n4max_coderate_subslot_pucch_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n4max_coderate_subslot_pucch_r15, (uint8_t)0u, (uint8_t)7u));
      }
      if (c.n4max_coderate_multi_res_slot_pucch_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n4max_coderate_multi_res_slot_pucch_r15, (uint8_t)0u, (uint8_t)7u));
      }
      if (c.n4max_coderate_multi_res_subslot_pucch_r15_present) {
        HANDLE_CODE(pack_integer(bref, c.n4max_coderate_multi_res_subslot_pucch_r15, (uint8_t)0u, (uint8_t)7u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "spucch_elems_r15_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE spucch_elems_r15_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.n1_subslot_spucch_an_list_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n1_slot_spucch_fh_an_list_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n1_slot_spucch_no_fh_an_list_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n3_spucch_an_list_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n4_spucch_slot_res_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n4_spucch_subslot_res_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n4max_coderate_slot_pucch_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n4max_coderate_subslot_pucch_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n4max_coderate_multi_res_slot_pucch_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.n4max_coderate_multi_res_subslot_pucch_r15_present, 1));
      if (c.n1_subslot_spucch_an_list_r15_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.n1_subslot_spucch_an_list_r15, bref, 1, 4, integer_packer<uint16_t>(0, 1319)));
      }
      if (c.n1_slot_spucch_fh_an_list_r15_present) {
        HANDLE_CODE(unpack_integer(c.n1_slot_spucch_fh_an_list_r15, bref, (uint16_t)0u, (uint16_t)1319u));
      }
      if (c.n1_slot_spucch_no_fh_an_list_r15_present) {
        HANDLE_CODE(unpack_integer(c.n1_slot_spucch_no_fh_an_list_r15, bref, (uint16_t)0u, (uint16_t)3959u));
      }
      if (c.n3_spucch_an_list_r15_present) {
        HANDLE_CODE(unpack_integer(c.n3_spucch_an_list_r15, bref, (uint16_t)0u, (uint16_t)549u));
      }
      if (c.n4_spucch_slot_res_r15_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.n4_spucch_slot_res_r15, bref, 1, 2));
      }
      if (c.n4_spucch_subslot_res_r15_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.n4_spucch_subslot_res_r15, bref, 1, 2));
      }
      if (c.n4max_coderate_slot_pucch_r15_present) {
        HANDLE_CODE(unpack_integer(c.n4max_coderate_slot_pucch_r15, bref, (uint8_t)0u, (uint8_t)7u));
      }
      if (c.n4max_coderate_subslot_pucch_r15_present) {
        HANDLE_CODE(unpack_integer(c.n4max_coderate_subslot_pucch_r15, bref, (uint8_t)0u, (uint8_t)7u));
      }
      if (c.n4max_coderate_multi_res_slot_pucch_r15_present) {
        HANDLE_CODE(unpack_integer(c.n4max_coderate_multi_res_slot_pucch_r15, bref, (uint8_t)0u, (uint8_t)7u));
      }
      if (c.n4max_coderate_multi_res_subslot_pucch_r15_present) {
        HANDLE_CODE(unpack_integer(c.n4max_coderate_multi_res_subslot_pucch_r15, bref, (uint8_t)0u, (uint8_t)7u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "spucch_elems_r15_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// TPC-Index ::= CHOICE
void tpc_idx_c::destroy_() {}
void tpc_idx_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
tpc_idx_c::tpc_idx_c(const tpc_idx_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::idx_of_format3:
      c.init(other.c.get<uint8_t>());
      break;
    case types::idx_of_format3_a:
      c.init(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "tpc_idx_c");
  }
}
tpc_idx_c& tpc_idx_c::operator=(const tpc_idx_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::idx_of_format3:
      c.set(other.c.get<uint8_t>());
      break;
    case types::idx_of_format3_a:
      c.set(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "tpc_idx_c");
  }

  return *this;
}
void tpc_idx_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::idx_of_format3:
      j.write_int("indexOfFormat3", c.get<uint8_t>());
      break;
    case types::idx_of_format3_a:
      j.write_int("indexOfFormat3A", c.get<uint8_t>());
      break;
    default:
      log_invalid_choice_id(type_, "tpc_idx_c");
  }
  j.end_obj();
}
SRSASN_CODE tpc_idx_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::idx_of_format3:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)15u));
      break;
    case types::idx_of_format3_a:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)31u));
      break;
    default:
      log_invalid_choice_id(type_, "tpc_idx_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE tpc_idx_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::idx_of_format3:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)15u));
      break;
    case types::idx_of_format3_a:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)31u));
      break;
    default:
      log_invalid_choice_id(type_, "tpc_idx_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// UL-AM-RLC-r15 ::= SEQUENCE
SRSASN_CODE ul_am_rlc_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(t_poll_retx_r15.pack(bref));
  HANDLE_CODE(poll_pdu_r15.pack(bref));
  HANDLE_CODE(poll_byte_r15.pack(bref));
  HANDLE_CODE(max_retx_thres_r15.pack(bref));
  HANDLE_CODE(bref.pack(extended_rlc_li_field_r15, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_am_rlc_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(t_poll_retx_r15.unpack(bref));
  HANDLE_CODE(poll_pdu_r15.unpack(bref));
  HANDLE_CODE(poll_byte_r15.unpack(bref));
  HANDLE_CODE(max_retx_thres_r15.unpack(bref));
  HANDLE_CODE(bref.unpack(extended_rlc_li_field_r15, 1));

  return SRSASN_SUCCESS;
}
void ul_am_rlc_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("t-PollRetransmit-r15", t_poll_retx_r15.to_string());
  j.write_str("pollPDU-r15", poll_pdu_r15.to_string());
  j.write_str("pollByte-r15", poll_byte_r15.to_string());
  j.write_str("maxRetxThreshold-r15", max_retx_thres_r15.to_string());
  j.write_bool("extended-RLC-LI-Field-r15", extended_rlc_li_field_r15);
  j.end_obj();
}

// UL-UM-RLC ::= SEQUENCE
SRSASN_CODE ul_um_rlc_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sn_field_len.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_um_rlc_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sn_field_len.unpack(bref));

  return SRSASN_SUCCESS;
}
void ul_um_rlc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sn-FieldLength", sn_field_len.to_string());
  j.end_obj();
}

// ZeroTxPowerCSI-RS-Conf-r12 ::= CHOICE
void zero_tx_pwr_csi_rs_conf_r12_c::set(types::options e)
{
  type_ = e;
}
void zero_tx_pwr_csi_rs_conf_r12_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "zero_tx_pwr_csi_rs_conf_r12_c");
  }
  j.end_obj();
}
SRSASN_CODE zero_tx_pwr_csi_rs_conf_r12_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "zero_tx_pwr_csi_rs_conf_r12_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE zero_tx_pwr_csi_rs_conf_r12_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "zero_tx_pwr_csi_rs_conf_r12_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportConfig-r10 ::= SEQUENCE
SRSASN_CODE cqi_report_cfg_r10_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cqi_report_aperiodic_r10_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_periodic_r10_present, 1));
  HANDLE_CODE(bref.pack(pmi_ri_report_r9_present, 1));
  HANDLE_CODE(bref.pack(csi_sf_pattern_cfg_r10_present, 1));

  if (cqi_report_aperiodic_r10_present) {
    HANDLE_CODE(cqi_report_aperiodic_r10.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, nom_pdsch_rs_epre_offset, (int8_t)-1, (int8_t)6));
  if (cqi_report_periodic_r10_present) {
    HANDLE_CODE(cqi_report_periodic_r10.pack(bref));
  }
  if (csi_sf_pattern_cfg_r10_present) {
    HANDLE_CODE(csi_sf_pattern_cfg_r10.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_r10_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cqi_report_aperiodic_r10_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_periodic_r10_present, 1));
  HANDLE_CODE(bref.unpack(pmi_ri_report_r9_present, 1));
  HANDLE_CODE(bref.unpack(csi_sf_pattern_cfg_r10_present, 1));

  if (cqi_report_aperiodic_r10_present) {
    HANDLE_CODE(cqi_report_aperiodic_r10.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(nom_pdsch_rs_epre_offset, bref, (int8_t)-1, (int8_t)6));
  if (cqi_report_periodic_r10_present) {
    HANDLE_CODE(cqi_report_periodic_r10.unpack(bref));
  }
  if (csi_sf_pattern_cfg_r10_present) {
    HANDLE_CODE(csi_sf_pattern_cfg_r10.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_cfg_r10_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cqi_report_aperiodic_r10_present) {
    j.write_fieldname("cqi-ReportAperiodic-r10");
    cqi_report_aperiodic_r10.to_json(j);
  }
  j.write_int("nomPDSCH-RS-EPRE-Offset", nom_pdsch_rs_epre_offset);
  if (cqi_report_periodic_r10_present) {
    j.write_fieldname("cqi-ReportPeriodic-r10");
    cqi_report_periodic_r10.to_json(j);
  }
  if (pmi_ri_report_r9_present) {
    j.write_str("pmi-RI-Report-r9", "setup");
  }
  if (csi_sf_pattern_cfg_r10_present) {
    j.write_fieldname("csi-SubframePatternConfig-r10");
    csi_sf_pattern_cfg_r10.to_json(j);
  }
  j.end_obj();
}

void cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_::set(types::options e)
{
  type_ = e;
}
void cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_fieldname("csi-MeasSubframeSet1-r10");
      c.csi_meas_sf_set1_r10.to_json(j);
      j.write_fieldname("csi-MeasSubframeSet2-r10");
      c.csi_meas_sf_set2_r10.to_json(j);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.csi_meas_sf_set1_r10.pack(bref));
      HANDLE_CODE(c.csi_meas_sf_set2_r10.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.csi_meas_sf_set1_r10.unpack(bref));
      HANDLE_CODE(c.csi_meas_sf_set2_r10.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_r10_s::csi_sf_pattern_cfg_r10_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportConfig-v1130 ::= SEQUENCE
SRSASN_CODE cqi_report_cfg_v1130_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(cqi_report_periodic_v1130.pack(bref));
  HANDLE_CODE(cqi_report_both_r11.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_v1130_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(cqi_report_periodic_v1130.unpack(bref));
  HANDLE_CODE(cqi_report_both_r11.unpack(bref));

  return SRSASN_SUCCESS;
}
void cqi_report_cfg_v1130_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cqi-ReportPeriodic-v1130");
  cqi_report_periodic_v1130.to_json(j);
  j.write_fieldname("cqi-ReportBoth-r11");
  cqi_report_both_r11.to_json(j);
  j.end_obj();
}

// CQI-ReportConfig-v1250 ::= SEQUENCE
SRSASN_CODE cqi_report_cfg_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_sf_pattern_cfg_r12_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_both_v1250_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_aperiodic_v1250_present, 1));
  HANDLE_CODE(bref.pack(alt_cqi_table_r12_present, 1));

  if (csi_sf_pattern_cfg_r12_present) {
    HANDLE_CODE(csi_sf_pattern_cfg_r12.pack(bref));
  }
  if (cqi_report_both_v1250_present) {
    HANDLE_CODE(cqi_report_both_v1250.pack(bref));
  }
  if (cqi_report_aperiodic_v1250_present) {
    HANDLE_CODE(cqi_report_aperiodic_v1250.pack(bref));
  }
  if (alt_cqi_table_r12_present) {
    HANDLE_CODE(alt_cqi_table_r12.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_sf_pattern_cfg_r12_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_both_v1250_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_aperiodic_v1250_present, 1));
  HANDLE_CODE(bref.unpack(alt_cqi_table_r12_present, 1));

  if (csi_sf_pattern_cfg_r12_present) {
    HANDLE_CODE(csi_sf_pattern_cfg_r12.unpack(bref));
  }
  if (cqi_report_both_v1250_present) {
    HANDLE_CODE(cqi_report_both_v1250.unpack(bref));
  }
  if (cqi_report_aperiodic_v1250_present) {
    HANDLE_CODE(cqi_report_aperiodic_v1250.unpack(bref));
  }
  if (alt_cqi_table_r12_present) {
    HANDLE_CODE(alt_cqi_table_r12.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_cfg_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_sf_pattern_cfg_r12_present) {
    j.write_fieldname("csi-SubframePatternConfig-r12");
    csi_sf_pattern_cfg_r12.to_json(j);
  }
  if (cqi_report_both_v1250_present) {
    j.write_fieldname("cqi-ReportBoth-v1250");
    cqi_report_both_v1250.to_json(j);
  }
  if (cqi_report_aperiodic_v1250_present) {
    j.write_fieldname("cqi-ReportAperiodic-v1250");
    cqi_report_aperiodic_v1250.to_json(j);
  }
  if (alt_cqi_table_r12_present) {
    j.write_str("altCQI-Table-r12", alt_cqi_table_r12.to_string());
  }
  j.end_obj();
}

void cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_::set(types::options e)
{
  type_ = e;
}
void cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("csi-MeasSubframeSets-r12", c.csi_meas_sf_sets_r12.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.csi_meas_sf_sets_r12.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.csi_meas_sf_sets_r12.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_v1250_s::csi_sf_pattern_cfg_r12_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportConfig-v1310 ::= SEQUENCE
SRSASN_CODE cqi_report_cfg_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cqi_report_both_v1310_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_aperiodic_v1310_present, 1));
  HANDLE_CODE(bref.pack(cqi_report_periodic_v1310_present, 1));

  if (cqi_report_both_v1310_present) {
    HANDLE_CODE(cqi_report_both_v1310.pack(bref));
  }
  if (cqi_report_aperiodic_v1310_present) {
    HANDLE_CODE(cqi_report_aperiodic_v1310.pack(bref));
  }
  if (cqi_report_periodic_v1310_present) {
    HANDLE_CODE(cqi_report_periodic_v1310.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cqi_report_both_v1310_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_aperiodic_v1310_present, 1));
  HANDLE_CODE(bref.unpack(cqi_report_periodic_v1310_present, 1));

  if (cqi_report_both_v1310_present) {
    HANDLE_CODE(cqi_report_both_v1310.unpack(bref));
  }
  if (cqi_report_aperiodic_v1310_present) {
    HANDLE_CODE(cqi_report_aperiodic_v1310.unpack(bref));
  }
  if (cqi_report_periodic_v1310_present) {
    HANDLE_CODE(cqi_report_periodic_v1310.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_cfg_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cqi_report_both_v1310_present) {
    j.write_fieldname("cqi-ReportBoth-v1310");
    cqi_report_both_v1310.to_json(j);
  }
  if (cqi_report_aperiodic_v1310_present) {
    j.write_fieldname("cqi-ReportAperiodic-v1310");
    cqi_report_aperiodic_v1310.to_json(j);
  }
  if (cqi_report_periodic_v1310_present) {
    j.write_fieldname("cqi-ReportPeriodic-v1310");
    cqi_report_periodic_v1310.to_json(j);
  }
  j.end_obj();
}

// CQI-ReportConfig-v1320 ::= SEQUENCE
SRSASN_CODE cqi_report_cfg_v1320_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cqi_report_periodic_v1320_present, 1));

  if (cqi_report_periodic_v1320_present) {
    HANDLE_CODE(cqi_report_periodic_v1320.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_v1320_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cqi_report_periodic_v1320_present, 1));

  if (cqi_report_periodic_v1320_present) {
    HANDLE_CODE(cqi_report_periodic_v1320.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_cfg_v1320_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cqi_report_periodic_v1320_present) {
    j.write_fieldname("cqi-ReportPeriodic-v1320");
    cqi_report_periodic_v1320.to_json(j);
  }
  j.end_obj();
}

// CQI-ReportConfig-v1430 ::= SEQUENCE
SRSASN_CODE cqi_report_cfg_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(cqi_report_aperiodic_hybrid_r14_present, 1));

  if (cqi_report_aperiodic_hybrid_r14_present) {
    HANDLE_CODE(cqi_report_aperiodic_hybrid_r14.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(cqi_report_aperiodic_hybrid_r14_present, 1));

  if (cqi_report_aperiodic_hybrid_r14_present) {
    HANDLE_CODE(cqi_report_aperiodic_hybrid_r14.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cqi_report_cfg_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (cqi_report_aperiodic_hybrid_r14_present) {
    j.write_fieldname("cqi-ReportAperiodicHybrid-r14");
    cqi_report_aperiodic_hybrid_r14.to_json(j);
  }
  j.end_obj();
}

// CSI-RS-Config-r10 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_r10_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(csi_rs_r10_present, 1));
  HANDLE_CODE(bref.pack(zero_tx_pwr_csi_rs_r10_present, 1));

  if (csi_rs_r10_present) {
    HANDLE_CODE(csi_rs_r10.pack(bref));
  }
  if (zero_tx_pwr_csi_rs_r10_present) {
    HANDLE_CODE(zero_tx_pwr_csi_rs_r10.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_r10_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(csi_rs_r10_present, 1));
  HANDLE_CODE(bref.unpack(zero_tx_pwr_csi_rs_r10_present, 1));

  if (csi_rs_r10_present) {
    HANDLE_CODE(csi_rs_r10.unpack(bref));
  }
  if (zero_tx_pwr_csi_rs_r10_present) {
    HANDLE_CODE(zero_tx_pwr_csi_rs_r10.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_r10_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (csi_rs_r10_present) {
    j.write_fieldname("csi-RS-r10");
    csi_rs_r10.to_json(j);
  }
  if (zero_tx_pwr_csi_rs_r10_present) {
    j.write_fieldname("zeroTxPowerCSI-RS-r10");
    zero_tx_pwr_csi_rs_r10.to_json(j);
  }
  j.end_obj();
}

void csi_rs_cfg_r10_s::csi_rs_r10_c_::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_r10_s::csi_rs_r10_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("antennaPortsCount-r10", c.ant_ports_count_r10.to_string());
      j.write_int("resourceConfig-r10", c.res_cfg_r10);
      j.write_int("subframeConfig-r10", c.sf_cfg_r10);
      j.write_int("p-C-r10", c.p_c_r10);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_r10_s::csi_rs_r10_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_r10_s::csi_rs_r10_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.ant_ports_count_r10.pack(bref));
      HANDLE_CODE(pack_integer(bref, c.res_cfg_r10, (uint8_t)0u, (uint8_t)31u));
      HANDLE_CODE(pack_integer(bref, c.sf_cfg_r10, (uint8_t)0u, (uint8_t)154u));
      HANDLE_CODE(pack_integer(bref, c.p_c_r10, (int8_t)-8, (int8_t)15));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_r10_s::csi_rs_r10_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_r10_s::csi_rs_r10_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.ant_ports_count_r10.unpack(bref));
      HANDLE_CODE(unpack_integer(c.res_cfg_r10, bref, (uint8_t)0u, (uint8_t)31u));
      HANDLE_CODE(unpack_integer(c.sf_cfg_r10, bref, (uint8_t)0u, (uint8_t)154u));
      HANDLE_CODE(unpack_integer(c.p_c_r10, bref, (int8_t)-8, (int8_t)15));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_r10_s::csi_rs_r10_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-Config-v1250 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_v1250_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(zero_tx_pwr_csi_rs2_r12_present, 1));
  HANDLE_CODE(bref.pack(ds_zero_tx_pwr_csi_rs_r12_present, 1));

  if (zero_tx_pwr_csi_rs2_r12_present) {
    HANDLE_CODE(zero_tx_pwr_csi_rs2_r12.pack(bref));
  }
  if (ds_zero_tx_pwr_csi_rs_r12_present) {
    HANDLE_CODE(ds_zero_tx_pwr_csi_rs_r12.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_v1250_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(zero_tx_pwr_csi_rs2_r12_present, 1));
  HANDLE_CODE(bref.unpack(ds_zero_tx_pwr_csi_rs_r12_present, 1));

  if (zero_tx_pwr_csi_rs2_r12_present) {
    HANDLE_CODE(zero_tx_pwr_csi_rs2_r12.unpack(bref));
  }
  if (ds_zero_tx_pwr_csi_rs_r12_present) {
    HANDLE_CODE(ds_zero_tx_pwr_csi_rs_r12.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_v1250_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (zero_tx_pwr_csi_rs2_r12_present) {
    j.write_fieldname("zeroTxPowerCSI-RS2-r12");
    zero_tx_pwr_csi_rs2_r12.to_json(j);
  }
  if (ds_zero_tx_pwr_csi_rs_r12_present) {
    j.write_fieldname("ds-ZeroTxPowerCSI-RS-r12");
    ds_zero_tx_pwr_csi_rs_r12.to_json(j);
  }
  j.end_obj();
}

void csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("zeroTxPowerCSI-RS-List-r12");
      for (const auto& e1 : c.zero_tx_pwr_csi_rs_list_r12) {
        e1.to_json(j);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.zero_tx_pwr_csi_rs_list_r12, 1, 5));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.zero_tx_pwr_csi_rs_list_r12, bref, 1, 5));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_v1250_s::ds_zero_tx_pwr_csi_rs_r12_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-Config-v1310 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(emimo_type_r13_present, 1));

  if (emimo_type_r13_present) {
    HANDLE_CODE(emimo_type_r13.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(emimo_type_r13_present, 1));

  if (emimo_type_r13_present) {
    HANDLE_CODE(emimo_type_r13.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (emimo_type_r13_present) {
    j.write_fieldname("eMIMO-Type-r13");
    emimo_type_r13.to_json(j);
  }
  j.end_obj();
}

// CSI-RS-Config-v1430 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_v1430_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(dummy_present, 1));
  HANDLE_CODE(bref.pack(emimo_hybrid_r14_present, 1));
  HANDLE_CODE(bref.pack(advanced_codebook_enabled_r14_present, 1));

  if (dummy_present) {
    HANDLE_CODE(dummy.pack(bref));
  }
  if (emimo_hybrid_r14_present) {
    HANDLE_CODE(emimo_hybrid_r14.pack(bref));
  }
  if (advanced_codebook_enabled_r14_present) {
    HANDLE_CODE(bref.pack(advanced_codebook_enabled_r14, 1));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_v1430_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(dummy_present, 1));
  HANDLE_CODE(bref.unpack(emimo_hybrid_r14_present, 1));
  HANDLE_CODE(bref.unpack(advanced_codebook_enabled_r14_present, 1));

  if (dummy_present) {
    HANDLE_CODE(dummy.unpack(bref));
  }
  if (emimo_hybrid_r14_present) {
    HANDLE_CODE(emimo_hybrid_r14.unpack(bref));
  }
  if (advanced_codebook_enabled_r14_present) {
    HANDLE_CODE(bref.unpack(advanced_codebook_enabled_r14, 1));
  }

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_v1430_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dummy_present) {
    j.write_fieldname("dummy");
    dummy.to_json(j);
  }
  if (emimo_hybrid_r14_present) {
    j.write_fieldname("eMIMO-Hybrid-r14");
    emimo_hybrid_r14.to_json(j);
  }
  if (advanced_codebook_enabled_r14_present) {
    j.write_bool("advancedCodebookEnabled-r14", advanced_codebook_enabled_r14);
  }
  j.end_obj();
}

// CSI-RS-ConfigZP-r11 ::= SEQUENCE
SRSASN_CODE csi_rs_cfg_zp_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, csi_rs_cfg_zp_id_r11, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(res_cfg_list_r11.pack(bref));
  HANDLE_CODE(pack_integer(bref, sf_cfg_r11, (uint8_t)0u, (uint8_t)154u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_zp_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(csi_rs_cfg_zp_id_r11, bref, (uint8_t)1u, (uint8_t)4u));
  HANDLE_CODE(res_cfg_list_r11.unpack(bref));
  HANDLE_CODE(unpack_integer(sf_cfg_r11, bref, (uint8_t)0u, (uint8_t)154u));

  return SRSASN_SUCCESS;
}
void csi_rs_cfg_zp_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("csi-RS-ConfigZPId-r11", csi_rs_cfg_zp_id_r11);
  j.write_str("resourceConfigList-r11", res_cfg_list_r11.to_string());
  j.write_int("subframeConfig-r11", sf_cfg_r11);
  j.end_obj();
}

// DL-AM-RLC ::= SEQUENCE
SRSASN_CODE dl_am_rlc_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(t_reordering.pack(bref));
  HANDLE_CODE(t_status_prohibit.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_am_rlc_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(t_reordering.unpack(bref));
  HANDLE_CODE(t_status_prohibit.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_am_rlc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("t-Reordering", t_reordering.to_string());
  j.write_str("t-StatusProhibit", t_status_prohibit.to_string());
  j.end_obj();
}

// DL-UM-RLC ::= SEQUENCE
SRSASN_CODE dl_um_rlc_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(sn_field_len.pack(bref));
  HANDLE_CODE(t_reordering.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_um_rlc_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(sn_field_len.unpack(bref));
  HANDLE_CODE(t_reordering.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_um_rlc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sn-FieldLength", sn_field_len.to_string());
  j.write_str("t-Reordering", t_reordering.to_string());
  j.end_obj();
}

// DeltaTxD-OffsetListSPUCCH-r15 ::= SEQUENCE
SRSASN_CODE delta_tx_d_offset_list_spucch_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(delta_tx_d_offset_spucch_format1_r15.pack(bref));
  HANDLE_CODE(delta_tx_d_offset_spucch_format1a_r15.pack(bref));
  HANDLE_CODE(delta_tx_d_offset_spucch_format1b_r15.pack(bref));
  HANDLE_CODE(delta_tx_d_offset_spucch_format3_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE delta_tx_d_offset_list_spucch_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(delta_tx_d_offset_spucch_format1_r15.unpack(bref));
  HANDLE_CODE(delta_tx_d_offset_spucch_format1a_r15.unpack(bref));
  HANDLE_CODE(delta_tx_d_offset_spucch_format1b_r15.unpack(bref));
  HANDLE_CODE(delta_tx_d_offset_spucch_format3_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void delta_tx_d_offset_list_spucch_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("deltaTxD-OffsetSPUCCH-Format1-r15", delta_tx_d_offset_spucch_format1_r15.to_string());
  j.write_str("deltaTxD-OffsetSPUCCH-Format1a-r15", delta_tx_d_offset_spucch_format1a_r15.to_string());
  j.write_str("deltaTxD-OffsetSPUCCH-Format1b-r15", delta_tx_d_offset_spucch_format1b_r15.to_string());
  j.write_str("deltaTxD-OffsetSPUCCH-Format3-r15", delta_tx_d_offset_spucch_format3_r15.to_string());
  j.end_obj();
}

// EPDCCH-SetConfig-r11 ::= SEQUENCE
SRSASN_CODE epdcch_set_cfg_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(re_map_qcl_cfg_id_r11_present, 1));

  HANDLE_CODE(pack_integer(bref, set_cfg_id_r11, (uint8_t)0u, (uint8_t)1u));
  HANDLE_CODE(tx_type_r11.pack(bref));
  HANDLE_CODE(res_block_assign_r11.num_prb_pairs_r11.pack(bref));
  HANDLE_CODE(res_block_assign_r11.res_block_assign_r11.pack(bref));
  HANDLE_CODE(pack_integer(bref, dmrs_scrambling_seq_int_r11, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(pack_integer(bref, pucch_res_start_offset_r11, (uint16_t)0u, (uint16_t)2047u));
  if (re_map_qcl_cfg_id_r11_present) {
    HANDLE_CODE(pack_integer(bref, re_map_qcl_cfg_id_r11, (uint8_t)1u, (uint8_t)4u));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= csi_rs_cfg_zp_id2_r12.is_present();
    group_flags[1] |= num_prb_pairs_v1310.is_present();
    group_flags[1] |= mpdcch_cfg_r13.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(csi_rs_cfg_zp_id2_r12.is_present(), 1));
      if (csi_rs_cfg_zp_id2_r12.is_present()) {
        HANDLE_CODE(csi_rs_cfg_zp_id2_r12->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(num_prb_pairs_v1310.is_present(), 1));
      HANDLE_CODE(bref.pack(mpdcch_cfg_r13.is_present(), 1));
      if (num_prb_pairs_v1310.is_present()) {
        HANDLE_CODE(num_prb_pairs_v1310->pack(bref));
      }
      if (mpdcch_cfg_r13.is_present()) {
        HANDLE_CODE(mpdcch_cfg_r13->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE epdcch_set_cfg_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(re_map_qcl_cfg_id_r11_present, 1));

  HANDLE_CODE(unpack_integer(set_cfg_id_r11, bref, (uint8_t)0u, (uint8_t)1u));
  HANDLE_CODE(tx_type_r11.unpack(bref));
  HANDLE_CODE(res_block_assign_r11.num_prb_pairs_r11.unpack(bref));
  HANDLE_CODE(res_block_assign_r11.res_block_assign_r11.unpack(bref));
  HANDLE_CODE(unpack_integer(dmrs_scrambling_seq_int_r11, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(unpack_integer(pucch_res_start_offset_r11, bref, (uint16_t)0u, (uint16_t)2047u));
  if (re_map_qcl_cfg_id_r11_present) {
    HANDLE_CODE(unpack_integer(re_map_qcl_cfg_id_r11, bref, (uint8_t)1u, (uint8_t)4u));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool csi_rs_cfg_zp_id2_r12_present;
      HANDLE_CODE(bref.unpack(csi_rs_cfg_zp_id2_r12_present, 1));
      csi_rs_cfg_zp_id2_r12.set_present(csi_rs_cfg_zp_id2_r12_present);
      if (csi_rs_cfg_zp_id2_r12.is_present()) {
        HANDLE_CODE(csi_rs_cfg_zp_id2_r12->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool num_prb_pairs_v1310_present;
      HANDLE_CODE(bref.unpack(num_prb_pairs_v1310_present, 1));
      num_prb_pairs_v1310.set_present(num_prb_pairs_v1310_present);
      bool mpdcch_cfg_r13_present;
      HANDLE_CODE(bref.unpack(mpdcch_cfg_r13_present, 1));
      mpdcch_cfg_r13.set_present(mpdcch_cfg_r13_present);
      if (num_prb_pairs_v1310.is_present()) {
        HANDLE_CODE(num_prb_pairs_v1310->unpack(bref));
      }
      if (mpdcch_cfg_r13.is_present()) {
        HANDLE_CODE(mpdcch_cfg_r13->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void epdcch_set_cfg_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("setConfigId-r11", set_cfg_id_r11);
  j.write_str("transmissionType-r11", tx_type_r11.to_string());
  j.write_fieldname("resourceBlockAssignment-r11");
  j.start_obj();
  j.write_str("numberPRB-Pairs-r11", res_block_assign_r11.num_prb_pairs_r11.to_string());
  j.write_str("resourceBlockAssignment-r11", res_block_assign_r11.res_block_assign_r11.to_string());
  j.end_obj();
  j.write_int("dmrs-ScramblingSequenceInt-r11", dmrs_scrambling_seq_int_r11);
  j.write_int("pucch-ResourceStartOffset-r11", pucch_res_start_offset_r11);
  if (re_map_qcl_cfg_id_r11_present) {
    j.write_int("re-MappingQCL-ConfigId-r11", re_map_qcl_cfg_id_r11);
  }
  if (ext) {
    if (csi_rs_cfg_zp_id2_r12.is_present()) {
      j.write_fieldname("csi-RS-ConfigZPId2-r12");
      csi_rs_cfg_zp_id2_r12->to_json(j);
    }
    if (num_prb_pairs_v1310.is_present()) {
      j.write_fieldname("numberPRB-Pairs-v1310");
      num_prb_pairs_v1310->to_json(j);
    }
    if (mpdcch_cfg_r13.is_present()) {
      j.write_fieldname("mpdcch-config-r13");
      mpdcch_cfg_r13->to_json(j);
    }
  }
  j.end_obj();
}

void epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_::set(types::options e)
{
  type_ = e;
}
void epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_int("setup", c);
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_");
  }
  j.end_obj();
}
SRSASN_CODE epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_integer(bref, c, (uint8_t)1u, (uint8_t)4u));
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_integer(c, bref, (uint8_t)1u, (uint8_t)4u));
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::csi_rs_cfg_zp_id2_r12_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void epdcch_set_cfg_r11_s::num_prb_pairs_v1310_c_::set(types::options e)
{
  type_ = e;
}
void epdcch_set_cfg_r11_s::num_prb_pairs_v1310_c_::to_json(json_writer& j) const
{
  j.start_obj();
  j.end_obj();
}
SRSASN_CODE epdcch_set_cfg_r11_s::num_prb_pairs_v1310_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  return SRSASN_SUCCESS;
}
SRSASN_CODE epdcch_set_cfg_r11_s::num_prb_pairs_v1310_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  return SRSASN_SUCCESS;
}

void epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::set(types::options e)
{
  type_ = e;
}
void epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("csi-NumRepetitionCE-r13", c.csi_num_repeat_ce_r13.to_string());
      j.write_str("mpdcch-pdsch-HoppingConfig-r13", c.mpdcch_pdsch_hop_cfg_r13.to_string());
      j.write_fieldname("mpdcch-StartSF-UESS-r13");
      c.mpdcch_start_sf_uess_r13.to_json(j);
      j.write_str("mpdcch-NumRepetition-r13", c.mpdcch_num_repeat_r13.to_string());
      j.write_int("mpdcch-Narrowband-r13", c.mpdcch_nb_r13);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.csi_num_repeat_ce_r13.pack(bref));
      HANDLE_CODE(c.mpdcch_pdsch_hop_cfg_r13.pack(bref));
      HANDLE_CODE(c.mpdcch_start_sf_uess_r13.pack(bref));
      HANDLE_CODE(c.mpdcch_num_repeat_r13.pack(bref));
      HANDLE_CODE(pack_integer(bref, c.mpdcch_nb_r13, (uint8_t)1u, (uint8_t)16u));
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.csi_num_repeat_ce_r13.unpack(bref));
      HANDLE_CODE(c.mpdcch_pdsch_hop_cfg_r13.unpack(bref));
      HANDLE_CODE(c.mpdcch_start_sf_uess_r13.unpack(bref));
      HANDLE_CODE(c.mpdcch_num_repeat_r13.unpack(bref));
      HANDLE_CODE(unpack_integer(c.mpdcch_nb_r13, bref, (uint8_t)1u, (uint8_t)16u));
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::destroy_() {}
void epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::set(types::options e)
{
  destroy_();
  type_ = e;
}
epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::mpdcch_start_sf_uess_r13_c_(
    const epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::fdd_r13:
      c.init(other.c.get<fdd_r13_e_>());
      break;
    case types::tdd_r13:
      c.init(other.c.get<tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_");
  }
}
epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_&
epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::
operator=(const epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::fdd_r13:
      c.set(other.c.get<fdd_r13_e_>());
      break;
    case types::tdd_r13:
      c.set(other.c.get<tdd_r13_e_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_");
  }

  return *this;
}
void epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::fdd_r13:
      j.write_str("fdd-r13", c.get<fdd_r13_e_>().to_string());
      break;
    case types::tdd_r13:
      j.write_str("tdd-r13", c.get<tdd_r13_e_>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::fdd_r13:
      HANDLE_CODE(c.get<fdd_r13_e_>().pack(bref));
      break;
    case types::tdd_r13:
      HANDLE_CODE(c.get<tdd_r13_e_>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::fdd_r13:
      HANDLE_CODE(c.get<fdd_r13_e_>().unpack(bref));
      break;
    case types::tdd_r13:
      HANDLE_CODE(c.get<tdd_r13_e_>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "epdcch_set_cfg_r11_s::mpdcch_cfg_r13_c_::setup_s_::mpdcch_start_sf_uess_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// Enable256QAM-r14 ::= CHOICE
void enable256_qam_r14_c::set(types::options e)
{
  type_ = e;
}
void enable256_qam_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      c.to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE enable256_qam_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enable256_qam_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void enable256_qam_r14_c::setup_c_::destroy_()
{
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      c.destroy<tpc_sf_set_cfgured_r14_s_>();
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      c.destroy<tpc_sf_set_not_cfgured_r14_s_>();
      break;
    default:
      break;
  }
}
void enable256_qam_r14_c::setup_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      c.init<tpc_sf_set_cfgured_r14_s_>();
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      c.init<tpc_sf_set_not_cfgured_r14_s_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c::setup_c_");
  }
}
enable256_qam_r14_c::setup_c_::setup_c_(const enable256_qam_r14_c::setup_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      c.init(other.c.get<tpc_sf_set_cfgured_r14_s_>());
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      c.init(other.c.get<tpc_sf_set_not_cfgured_r14_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c::setup_c_");
  }
}
enable256_qam_r14_c::setup_c_& enable256_qam_r14_c::setup_c_::operator=(const enable256_qam_r14_c::setup_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      c.set(other.c.get<tpc_sf_set_cfgured_r14_s_>());
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      c.set(other.c.get<tpc_sf_set_not_cfgured_r14_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c::setup_c_");
  }

  return *this;
}
void enable256_qam_r14_c::setup_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      j.write_fieldname("tpc-SubframeSet-Configured-r14");
      j.start_obj();
      j.write_bool("subframeSet1-DCI-Format0-r14", c.get<tpc_sf_set_cfgured_r14_s_>().sf_set1_dci_format0_r14);
      j.write_bool("subframeSet1-DCI-Format4-r14", c.get<tpc_sf_set_cfgured_r14_s_>().sf_set1_dci_format4_r14);
      j.write_bool("subframeSet2-DCI-Format0-r14", c.get<tpc_sf_set_cfgured_r14_s_>().sf_set2_dci_format0_r14);
      j.write_bool("subframeSet2-DCI-Format4-r14", c.get<tpc_sf_set_cfgured_r14_s_>().sf_set2_dci_format4_r14);
      j.end_obj();
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      j.write_fieldname("tpc-SubframeSet-NotConfigured-r14");
      j.start_obj();
      j.write_bool("dci-Format0-r14", c.get<tpc_sf_set_not_cfgured_r14_s_>().dci_format0_r14);
      j.write_bool("dci-Format4-r14", c.get<tpc_sf_set_not_cfgured_r14_s_>().dci_format4_r14);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c::setup_c_");
  }
  j.end_obj();
}
SRSASN_CODE enable256_qam_r14_c::setup_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      HANDLE_CODE(bref.pack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set1_dci_format0_r14, 1));
      HANDLE_CODE(bref.pack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set1_dci_format4_r14, 1));
      HANDLE_CODE(bref.pack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set2_dci_format0_r14, 1));
      HANDLE_CODE(bref.pack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set2_dci_format4_r14, 1));
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      HANDLE_CODE(bref.pack(c.get<tpc_sf_set_not_cfgured_r14_s_>().dci_format0_r14, 1));
      HANDLE_CODE(bref.pack(c.get<tpc_sf_set_not_cfgured_r14_s_>().dci_format4_r14, 1));
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c::setup_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enable256_qam_r14_c::setup_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::tpc_sf_set_cfgured_r14:
      HANDLE_CODE(bref.unpack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set1_dci_format0_r14, 1));
      HANDLE_CODE(bref.unpack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set1_dci_format4_r14, 1));
      HANDLE_CODE(bref.unpack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set2_dci_format0_r14, 1));
      HANDLE_CODE(bref.unpack(c.get<tpc_sf_set_cfgured_r14_s_>().sf_set2_dci_format4_r14, 1));
      break;
    case types::tpc_sf_set_not_cfgured_r14:
      HANDLE_CODE(bref.unpack(c.get<tpc_sf_set_not_cfgured_r14_s_>().dci_format0_r14, 1));
      HANDLE_CODE(bref.unpack(c.get<tpc_sf_set_not_cfgured_r14_s_>().dci_format4_r14, 1));
      break;
    default:
      log_invalid_choice_id(type_, "enable256_qam_r14_c::setup_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// LogicalChannelConfig ::= SEQUENCE
SRSASN_CODE lc_ch_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ul_specific_params_present, 1));

  if (ul_specific_params_present) {
    HANDLE_CODE(bref.pack(ul_specific_params.lc_ch_group_present, 1));
    HANDLE_CODE(pack_integer(bref, ul_specific_params.prio, (uint8_t)1u, (uint8_t)16u));
    HANDLE_CODE(ul_specific_params.prioritised_bit_rate.pack(bref));
    HANDLE_CODE(ul_specific_params.bucket_size_dur.pack(bref));
    if (ul_specific_params.lc_ch_group_present) {
      HANDLE_CODE(pack_integer(bref, ul_specific_params.lc_ch_group, (uint8_t)0u, (uint8_t)3u));
    }
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= lc_ch_sr_mask_r9_present;
    group_flags[1] |= lc_ch_sr_prohibit_r12_present;
    group_flags[2] |= laa_ul_allowed_r14_present;
    group_flags[2] |= bit_rate_query_prohibit_timer_r14_present;
    group_flags[3] |= allowed_tti_lens_r15.is_present();
    group_flags[3] |= lc_ch_sr_restrict_r15.is_present();
    group_flags[3] |= chl_access_prio_r15.is_present();
    group_flags[3] |= lch_cell_restrict_r15_present;
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(lc_ch_sr_mask_r9_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(lc_ch_sr_prohibit_r12_present, 1));
      if (lc_ch_sr_prohibit_r12_present) {
        HANDLE_CODE(bref.pack(lc_ch_sr_prohibit_r12, 1));
      }
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(laa_ul_allowed_r14_present, 1));
      HANDLE_CODE(bref.pack(bit_rate_query_prohibit_timer_r14_present, 1));
      if (laa_ul_allowed_r14_present) {
        HANDLE_CODE(bref.pack(laa_ul_allowed_r14, 1));
      }
      if (bit_rate_query_prohibit_timer_r14_present) {
        HANDLE_CODE(bit_rate_query_prohibit_timer_r14.pack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(allowed_tti_lens_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(lc_ch_sr_restrict_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(chl_access_prio_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(lch_cell_restrict_r15_present, 1));
      if (allowed_tti_lens_r15.is_present()) {
        HANDLE_CODE(allowed_tti_lens_r15->pack(bref));
      }
      if (lc_ch_sr_restrict_r15.is_present()) {
        HANDLE_CODE(lc_ch_sr_restrict_r15->pack(bref));
      }
      if (chl_access_prio_r15.is_present()) {
        HANDLE_CODE(chl_access_prio_r15->pack(bref));
      }
      if (lch_cell_restrict_r15_present) {
        HANDLE_CODE(lch_cell_restrict_r15.pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE lc_ch_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ul_specific_params_present, 1));

  if (ul_specific_params_present) {
    HANDLE_CODE(bref.unpack(ul_specific_params.lc_ch_group_present, 1));
    HANDLE_CODE(unpack_integer(ul_specific_params.prio, bref, (uint8_t)1u, (uint8_t)16u));
    HANDLE_CODE(ul_specific_params.prioritised_bit_rate.unpack(bref));
    HANDLE_CODE(ul_specific_params.bucket_size_dur.unpack(bref));
    if (ul_specific_params.lc_ch_group_present) {
      HANDLE_CODE(unpack_integer(ul_specific_params.lc_ch_group, bref, (uint8_t)0u, (uint8_t)3u));
    }
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(4);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(lc_ch_sr_mask_r9_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(lc_ch_sr_prohibit_r12_present, 1));
      if (lc_ch_sr_prohibit_r12_present) {
        HANDLE_CODE(bref.unpack(lc_ch_sr_prohibit_r12, 1));
      }
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(laa_ul_allowed_r14_present, 1));
      HANDLE_CODE(bref.unpack(bit_rate_query_prohibit_timer_r14_present, 1));
      if (laa_ul_allowed_r14_present) {
        HANDLE_CODE(bref.unpack(laa_ul_allowed_r14, 1));
      }
      if (bit_rate_query_prohibit_timer_r14_present) {
        HANDLE_CODE(bit_rate_query_prohibit_timer_r14.unpack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool allowed_tti_lens_r15_present;
      HANDLE_CODE(bref.unpack(allowed_tti_lens_r15_present, 1));
      allowed_tti_lens_r15.set_present(allowed_tti_lens_r15_present);
      bool lc_ch_sr_restrict_r15_present;
      HANDLE_CODE(bref.unpack(lc_ch_sr_restrict_r15_present, 1));
      lc_ch_sr_restrict_r15.set_present(lc_ch_sr_restrict_r15_present);
      bool chl_access_prio_r15_present;
      HANDLE_CODE(bref.unpack(chl_access_prio_r15_present, 1));
      chl_access_prio_r15.set_present(chl_access_prio_r15_present);
      HANDLE_CODE(bref.unpack(lch_cell_restrict_r15_present, 1));
      if (allowed_tti_lens_r15.is_present()) {
        HANDLE_CODE(allowed_tti_lens_r15->unpack(bref));
      }
      if (lc_ch_sr_restrict_r15.is_present()) {
        HANDLE_CODE(lc_ch_sr_restrict_r15->unpack(bref));
      }
      if (chl_access_prio_r15.is_present()) {
        HANDLE_CODE(chl_access_prio_r15->unpack(bref));
      }
      if (lch_cell_restrict_r15_present) {
        HANDLE_CODE(lch_cell_restrict_r15.unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void lc_ch_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ul_specific_params_present) {
    j.write_fieldname("ul-SpecificParameters");
    j.start_obj();
    j.write_int("priority", ul_specific_params.prio);
    j.write_str("prioritisedBitRate", ul_specific_params.prioritised_bit_rate.to_string());
    j.write_str("bucketSizeDuration", ul_specific_params.bucket_size_dur.to_string());
    if (ul_specific_params.lc_ch_group_present) {
      j.write_int("logicalChannelGroup", ul_specific_params.lc_ch_group);
    }
    j.end_obj();
  }
  if (ext) {
    if (lc_ch_sr_mask_r9_present) {
      j.write_str("logicalChannelSR-Mask-r9", "setup");
    }
    if (lc_ch_sr_prohibit_r12_present) {
      j.write_bool("logicalChannelSR-Prohibit-r12", lc_ch_sr_prohibit_r12);
    }
    if (laa_ul_allowed_r14_present) {
      j.write_bool("laa-UL-Allowed-r14", laa_ul_allowed_r14);
    }
    if (bit_rate_query_prohibit_timer_r14_present) {
      j.write_str("bitRateQueryProhibitTimer-r14", bit_rate_query_prohibit_timer_r14.to_string());
    }
    if (allowed_tti_lens_r15.is_present()) {
      j.write_fieldname("allowedTTI-Lengths-r15");
      allowed_tti_lens_r15->to_json(j);
    }
    if (lc_ch_sr_restrict_r15.is_present()) {
      j.write_fieldname("logicalChannelSR-Restriction-r15");
      lc_ch_sr_restrict_r15->to_json(j);
    }
    if (chl_access_prio_r15.is_present()) {
      j.write_fieldname("channellAccessPriority-r15");
      chl_access_prio_r15->to_json(j);
    }
    if (lch_cell_restrict_r15_present) {
      j.write_str("lch-CellRestriction-r15", lch_cell_restrict_r15.to_string());
    }
  }
  j.end_obj();
}

void lc_ch_cfg_s::allowed_tti_lens_r15_c_::set(types::options e)
{
  type_ = e;
}
void lc_ch_cfg_s::allowed_tti_lens_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_bool("shortTTI-r15", c.short_tti_r15);
      j.write_bool("subframeTTI-r15", c.sf_tti_r15);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::allowed_tti_lens_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE lc_ch_cfg_s::allowed_tti_lens_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.short_tti_r15, 1));
      HANDLE_CODE(bref.pack(c.sf_tti_r15, 1));
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::allowed_tti_lens_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE lc_ch_cfg_s::allowed_tti_lens_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.short_tti_r15, 1));
      HANDLE_CODE(bref.unpack(c.sf_tti_r15, 1));
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::allowed_tti_lens_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_::set(types::options e)
{
  type_ = e;
}
void lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_str("setup", c.to_string());
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::lc_ch_sr_restrict_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void lc_ch_cfg_s::chl_access_prio_r15_c_::set(types::options e)
{
  type_ = e;
}
void lc_ch_cfg_s::chl_access_prio_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_int("setup", c);
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::chl_access_prio_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE lc_ch_cfg_s::chl_access_prio_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_integer(bref, c, (uint8_t)1u, (uint8_t)4u));
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::chl_access_prio_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE lc_ch_cfg_s::chl_access_prio_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_integer(c, bref, (uint8_t)1u, (uint8_t)4u));
      break;
    default:
      log_invalid_choice_id(type_, "lc_ch_cfg_s::chl_access_prio_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// PDSCH-RE-MappingQCL-Config-r11 ::= SEQUENCE
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(optional_set_of_fields_r11_present, 1));
  HANDLE_CODE(bref.pack(qcl_csi_rs_cfg_nzp_id_r11_present, 1));

  HANDLE_CODE(pack_integer(bref, pdsch_re_map_qcl_cfg_id_r11, (uint8_t)1u, (uint8_t)4u));
  if (optional_set_of_fields_r11_present) {
    HANDLE_CODE(bref.pack(optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11_present, 1));
    HANDLE_CODE(optional_set_of_fields_r11.crs_ports_count_r11.pack(bref));
    HANDLE_CODE(pack_integer(bref, optional_set_of_fields_r11.crs_freq_shift_r11, (uint8_t)0u, (uint8_t)5u));
    if (optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11_present) {
      HANDLE_CODE(optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11.pack(bref));
    }
    HANDLE_CODE(optional_set_of_fields_r11.pdsch_start_r11.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, csi_rs_cfg_zp_id_r11, (uint8_t)1u, (uint8_t)4u));
  if (qcl_csi_rs_cfg_nzp_id_r11_present) {
    HANDLE_CODE(pack_integer(bref, qcl_csi_rs_cfg_nzp_id_r11, (uint8_t)1u, (uint8_t)3u));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= mbsfn_sf_cfg_list_v1430.is_present();
    group_flags[1] |= codeword_one_cfg_v1530.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_v1430.is_present(), 1));
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(mbsfn_sf_cfg_list_v1430->pack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(codeword_one_cfg_v1530.is_present(), 1));
      if (codeword_one_cfg_v1530.is_present()) {
        HANDLE_CODE(codeword_one_cfg_v1530->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(optional_set_of_fields_r11_present, 1));
  HANDLE_CODE(bref.unpack(qcl_csi_rs_cfg_nzp_id_r11_present, 1));

  HANDLE_CODE(unpack_integer(pdsch_re_map_qcl_cfg_id_r11, bref, (uint8_t)1u, (uint8_t)4u));
  if (optional_set_of_fields_r11_present) {
    HANDLE_CODE(bref.unpack(optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11_present, 1));
    HANDLE_CODE(optional_set_of_fields_r11.crs_ports_count_r11.unpack(bref));
    HANDLE_CODE(unpack_integer(optional_set_of_fields_r11.crs_freq_shift_r11, bref, (uint8_t)0u, (uint8_t)5u));
    if (optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11_present) {
      HANDLE_CODE(optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11.unpack(bref));
    }
    HANDLE_CODE(optional_set_of_fields_r11.pdsch_start_r11.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(csi_rs_cfg_zp_id_r11, bref, (uint8_t)1u, (uint8_t)4u));
  if (qcl_csi_rs_cfg_nzp_id_r11_present) {
    HANDLE_CODE(unpack_integer(qcl_csi_rs_cfg_nzp_id_r11, bref, (uint8_t)1u, (uint8_t)3u));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(2);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbsfn_sf_cfg_list_v1430_present;
      HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_v1430_present, 1));
      mbsfn_sf_cfg_list_v1430.set_present(mbsfn_sf_cfg_list_v1430_present);
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(mbsfn_sf_cfg_list_v1430->unpack(bref));
      }
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool codeword_one_cfg_v1530_present;
      HANDLE_CODE(bref.unpack(codeword_one_cfg_v1530_present, 1));
      codeword_one_cfg_v1530.set_present(codeword_one_cfg_v1530_present);
      if (codeword_one_cfg_v1530.is_present()) {
        HANDLE_CODE(codeword_one_cfg_v1530->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void pdsch_re_map_qcl_cfg_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pdsch-RE-MappingQCL-ConfigId-r11", pdsch_re_map_qcl_cfg_id_r11);
  if (optional_set_of_fields_r11_present) {
    j.write_fieldname("optionalSetOfFields-r11");
    j.start_obj();
    j.write_str("crs-PortsCount-r11", optional_set_of_fields_r11.crs_ports_count_r11.to_string());
    j.write_int("crs-FreqShift-r11", optional_set_of_fields_r11.crs_freq_shift_r11);
    if (optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11_present) {
      j.write_fieldname("mbsfn-SubframeConfigList-r11");
      optional_set_of_fields_r11.mbsfn_sf_cfg_list_r11.to_json(j);
    }
    j.write_str("pdsch-Start-r11", optional_set_of_fields_r11.pdsch_start_r11.to_string());
    j.end_obj();
  }
  j.write_int("csi-RS-ConfigZPId-r11", csi_rs_cfg_zp_id_r11);
  if (qcl_csi_rs_cfg_nzp_id_r11_present) {
    j.write_int("qcl-CSI-RS-ConfigNZPId-r11", qcl_csi_rs_cfg_nzp_id_r11);
  }
  if (ext) {
    if (mbsfn_sf_cfg_list_v1430.is_present()) {
      j.write_fieldname("mbsfn-SubframeConfigList-v1430");
      mbsfn_sf_cfg_list_v1430->to_json(j);
    }
    if (codeword_one_cfg_v1530.is_present()) {
      j.write_fieldname("codewordOneConfig-v1530");
      codeword_one_cfg_v1530->to_json(j);
    }
  }
  j.end_obj();
}

void pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_::set(types::options e)
{
  type_ = e;
}
void pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("subframeConfigList");
      for (const auto& e1 : c.sf_cfg_list) {
        e1.to_json(j);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_,
                            "pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_");
  }
  j.end_obj();
}
SRSASN_CODE
pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.sf_cfg_list, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_,
                            "pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.sf_cfg_list, bref, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_,
                            "pdsch_re_map_qcl_cfg_r11_s::optional_set_of_fields_r11_s_::mbsfn_sf_cfg_list_r11_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_::set(types::options e)
{
  type_ = e;
}
void pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("subframeConfigList-v1430");
      for (const auto& e1 : c.sf_cfg_list_v1430) {
        e1.to_json(j);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_");
  }
  j.end_obj();
}
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.sf_cfg_list_v1430, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.sf_cfg_list_v1430, bref, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "pdsch_re_map_qcl_cfg_r11_s::mbsfn_sf_cfg_list_v1430_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_::set(types::options e)
{
  type_ = e;
}
void pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("crs-PortsCount-v1530", c.crs_ports_count_v1530.to_string());
      j.write_int("crs-FreqShift-v1530", c.crs_freq_shift_v1530);
      if (c.mbsfn_sf_cfg_list_v1530_present) {
        j.start_array("mbsfn-SubframeConfigList-v1530");
        for (const auto& e1 : c.mbsfn_sf_cfg_list_v1530) {
          e1.to_json(j);
        }
        j.end_array();
      }
      if (c.mbsfn_sf_cfg_list_ext_v1530_present) {
        j.start_array("mbsfn-SubframeConfigListExt-v1530");
        for (const auto& e1 : c.mbsfn_sf_cfg_list_ext_v1530) {
          e1.to_json(j);
        }
        j.end_array();
      }
      j.write_str("pdsch-Start-v1530", c.pdsch_start_v1530.to_string());
      j.write_int("csi-RS-ConfigZPId-v1530", c.csi_rs_cfg_zp_id_v1530);
      if (c.qcl_csi_rs_cfg_nzp_id_v1530_present) {
        j.write_int("qcl-CSI-RS-ConfigNZPId-v1530", c.qcl_csi_rs_cfg_nzp_id_v1530);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_");
  }
  j.end_obj();
}
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.mbsfn_sf_cfg_list_v1530_present, 1));
      HANDLE_CODE(bref.pack(c.mbsfn_sf_cfg_list_ext_v1530_present, 1));
      HANDLE_CODE(bref.pack(c.qcl_csi_rs_cfg_nzp_id_v1530_present, 1));
      HANDLE_CODE(c.crs_ports_count_v1530.pack(bref));
      HANDLE_CODE(pack_integer(bref, c.crs_freq_shift_v1530, (uint8_t)0u, (uint8_t)5u));
      if (c.mbsfn_sf_cfg_list_v1530_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.mbsfn_sf_cfg_list_v1530, 1, 8));
      }
      if (c.mbsfn_sf_cfg_list_ext_v1530_present) {
        HANDLE_CODE(pack_dyn_seq_of(bref, c.mbsfn_sf_cfg_list_ext_v1530, 1, 8));
      }
      HANDLE_CODE(c.pdsch_start_v1530.pack(bref));
      HANDLE_CODE(pack_integer(bref, c.csi_rs_cfg_zp_id_v1530, (uint8_t)1u, (uint8_t)4u));
      if (c.qcl_csi_rs_cfg_nzp_id_v1530_present) {
        HANDLE_CODE(pack_integer(bref, c.qcl_csi_rs_cfg_nzp_id_v1530, (uint8_t)1u, (uint8_t)3u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.mbsfn_sf_cfg_list_v1530_present, 1));
      HANDLE_CODE(bref.unpack(c.mbsfn_sf_cfg_list_ext_v1530_present, 1));
      HANDLE_CODE(bref.unpack(c.qcl_csi_rs_cfg_nzp_id_v1530_present, 1));
      HANDLE_CODE(c.crs_ports_count_v1530.unpack(bref));
      HANDLE_CODE(unpack_integer(c.crs_freq_shift_v1530, bref, (uint8_t)0u, (uint8_t)5u));
      if (c.mbsfn_sf_cfg_list_v1530_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.mbsfn_sf_cfg_list_v1530, bref, 1, 8));
      }
      if (c.mbsfn_sf_cfg_list_ext_v1530_present) {
        HANDLE_CODE(unpack_dyn_seq_of(c.mbsfn_sf_cfg_list_ext_v1530, bref, 1, 8));
      }
      HANDLE_CODE(c.pdsch_start_v1530.unpack(bref));
      HANDLE_CODE(unpack_integer(c.csi_rs_cfg_zp_id_v1530, bref, (uint8_t)1u, (uint8_t)4u));
      if (c.qcl_csi_rs_cfg_nzp_id_v1530_present) {
        HANDLE_CODE(unpack_integer(c.qcl_csi_rs_cfg_nzp_id_v1530, bref, (uint8_t)1u, (uint8_t)3u));
      }
      break;
    default:
      log_invalid_choice_id(type_, "pdsch_re_map_qcl_cfg_r11_s::codeword_one_cfg_v1530_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// RLC-Config-r15 ::= SEQUENCE
SRSASN_CODE rlc_cfg_r15_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(reestablish_rlc_r15_present, 1));
  HANDLE_CODE(bref.pack(rlc_out_of_order_delivery_r15_present, 1));

  HANDLE_CODE(mode_r15.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE rlc_cfg_r15_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(reestablish_rlc_r15_present, 1));
  HANDLE_CODE(bref.unpack(rlc_out_of_order_delivery_r15_present, 1));

  HANDLE_CODE(mode_r15.unpack(bref));

  return SRSASN_SUCCESS;
}
void rlc_cfg_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("mode-r15");
  mode_r15.to_json(j);
  if (reestablish_rlc_r15_present) {
    j.write_str("reestablishRLC-r15", "true");
  }
  if (rlc_out_of_order_delivery_r15_present) {
    j.write_str("rlc-OutOfOrderDelivery-r15", "true");
  }
  j.end_obj();
}

void rlc_cfg_r15_s::mode_r15_c_::destroy_()
{
  switch (type_) {
    case types::am_r15:
      c.destroy<am_r15_s_>();
      break;
    case types::um_bi_dir_r15:
      c.destroy<um_bi_dir_r15_s_>();
      break;
    case types::um_uni_dir_ul_r15:
      c.destroy<um_uni_dir_ul_r15_s_>();
      break;
    case types::um_uni_dir_dl_r15:
      c.destroy<um_uni_dir_dl_r15_s_>();
      break;
    default:
      break;
  }
}
void rlc_cfg_r15_s::mode_r15_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::am_r15:
      c.init<am_r15_s_>();
      break;
    case types::um_bi_dir_r15:
      c.init<um_bi_dir_r15_s_>();
      break;
    case types::um_uni_dir_ul_r15:
      c.init<um_uni_dir_ul_r15_s_>();
      break;
    case types::um_uni_dir_dl_r15:
      c.init<um_uni_dir_dl_r15_s_>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rlc_cfg_r15_s::mode_r15_c_");
  }
}
rlc_cfg_r15_s::mode_r15_c_::mode_r15_c_(const rlc_cfg_r15_s::mode_r15_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::am_r15:
      c.init(other.c.get<am_r15_s_>());
      break;
    case types::um_bi_dir_r15:
      c.init(other.c.get<um_bi_dir_r15_s_>());
      break;
    case types::um_uni_dir_ul_r15:
      c.init(other.c.get<um_uni_dir_ul_r15_s_>());
      break;
    case types::um_uni_dir_dl_r15:
      c.init(other.c.get<um_uni_dir_dl_r15_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rlc_cfg_r15_s::mode_r15_c_");
  }
}
rlc_cfg_r15_s::mode_r15_c_& rlc_cfg_r15_s::mode_r15_c_::operator=(const rlc_cfg_r15_s::mode_r15_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::am_r15:
      c.set(other.c.get<am_r15_s_>());
      break;
    case types::um_bi_dir_r15:
      c.set(other.c.get<um_bi_dir_r15_s_>());
      break;
    case types::um_uni_dir_ul_r15:
      c.set(other.c.get<um_uni_dir_ul_r15_s_>());
      break;
    case types::um_uni_dir_dl_r15:
      c.set(other.c.get<um_uni_dir_dl_r15_s_>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rlc_cfg_r15_s::mode_r15_c_");
  }

  return *this;
}
void rlc_cfg_r15_s::mode_r15_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::am_r15:
      j.write_fieldname("am-r15");
      j.start_obj();
      j.write_fieldname("ul-AM-RLC-r15");
      c.get<am_r15_s_>().ul_am_rlc_r15.to_json(j);
      j.write_fieldname("dl-AM-RLC-r15");
      c.get<am_r15_s_>().dl_am_rlc_r15.to_json(j);
      j.end_obj();
      break;
    case types::um_bi_dir_r15:
      j.write_fieldname("um-Bi-Directional-r15");
      j.start_obj();
      j.write_fieldname("ul-UM-RLC-r15");
      c.get<um_bi_dir_r15_s_>().ul_um_rlc_r15.to_json(j);
      j.write_fieldname("dl-UM-RLC-r15");
      c.get<um_bi_dir_r15_s_>().dl_um_rlc_r15.to_json(j);
      j.end_obj();
      break;
    case types::um_uni_dir_ul_r15:
      j.write_fieldname("um-Uni-Directional-UL-r15");
      j.start_obj();
      j.write_fieldname("ul-UM-RLC-r15");
      c.get<um_uni_dir_ul_r15_s_>().ul_um_rlc_r15.to_json(j);
      j.end_obj();
      break;
    case types::um_uni_dir_dl_r15:
      j.write_fieldname("um-Uni-Directional-DL-r15");
      j.start_obj();
      j.write_fieldname("dl-UM-RLC-r15");
      c.get<um_uni_dir_dl_r15_s_>().dl_um_rlc_r15.to_json(j);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "rlc_cfg_r15_s::mode_r15_c_");
  }
  j.end_obj();
}
SRSASN_CODE rlc_cfg_r15_s::mode_r15_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::am_r15:
      HANDLE_CODE(c.get<am_r15_s_>().ul_am_rlc_r15.pack(bref));
      HANDLE_CODE(c.get<am_r15_s_>().dl_am_rlc_r15.pack(bref));
      break;
    case types::um_bi_dir_r15:
      HANDLE_CODE(c.get<um_bi_dir_r15_s_>().ul_um_rlc_r15.pack(bref));
      HANDLE_CODE(c.get<um_bi_dir_r15_s_>().dl_um_rlc_r15.pack(bref));
      break;
    case types::um_uni_dir_ul_r15:
      HANDLE_CODE(c.get<um_uni_dir_ul_r15_s_>().ul_um_rlc_r15.pack(bref));
      break;
    case types::um_uni_dir_dl_r15:
      HANDLE_CODE(c.get<um_uni_dir_dl_r15_s_>().dl_um_rlc_r15.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "rlc_cfg_r15_s::mode_r15_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rlc_cfg_r15_s::mode_r15_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::am_r15:
      HANDLE_CODE(c.get<am_r15_s_>().ul_am_rlc_r15.unpack(bref));
      HANDLE_CODE(c.get<am_r15_s_>().dl_am_rlc_r15.unpack(bref));
      break;
    case types::um_bi_dir_r15:
      HANDLE_CODE(c.get<um_bi_dir_r15_s_>().ul_um_rlc_r15.unpack(bref));
      HANDLE_CODE(c.get<um_bi_dir_r15_s_>().dl_um_rlc_r15.unpack(bref));
      break;
    case types::um_uni_dir_ul_r15:
      HANDLE_CODE(c.get<um_uni_dir_ul_r15_s_>().ul_um_rlc_r15.unpack(bref));
      break;
    case types::um_uni_dir_dl_r15:
      HANDLE_CODE(c.get<um_uni_dir_dl_r15_s_>().dl_um_rlc_r15.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "rlc_cfg_r15_s::mode_r15_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// TPC-PDCCH-Config ::= CHOICE
void tpc_pdcch_cfg_c::set(types::options e)
{
  type_ = e;
}
void tpc_pdcch_cfg_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("tpc-RNTI", c.tpc_rnti.to_string());
      j.write_fieldname("tpc-Index");
      c.tpc_idx.to_json(j);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "tpc_pdcch_cfg_c");
  }
  j.end_obj();
}
SRSASN_CODE tpc_pdcch_cfg_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.tpc_rnti.pack(bref));
      HANDLE_CODE(c.tpc_idx.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "tpc_pdcch_cfg_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE tpc_pdcch_cfg_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.tpc_rnti.unpack(bref));
      HANDLE_CODE(c.tpc_idx.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "tpc_pdcch_cfg_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// UL-AM-RLC ::= SEQUENCE
SRSASN_CODE ul_am_rlc_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(t_poll_retx.pack(bref));
  HANDLE_CODE(poll_pdu.pack(bref));
  HANDLE_CODE(poll_byte.pack(bref));
  HANDLE_CODE(max_retx_thres.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_am_rlc_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(t_poll_retx.unpack(bref));
  HANDLE_CODE(poll_pdu.unpack(bref));
  HANDLE_CODE(poll_byte.unpack(bref));
  HANDLE_CODE(max_retx_thres.unpack(bref));

  return SRSASN_SUCCESS;
}
void ul_am_rlc_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("t-PollRetransmit", t_poll_retx.to_string());
  j.write_str("pollPDU", poll_pdu.to_string());
  j.write_str("pollByte", poll_byte.to_string());
  j.write_str("maxRetxThreshold", max_retx_thres.to_string());
  j.end_obj();
}

// AntennaInfoDedicatedSTTI-r15 ::= CHOICE
void ant_info_ded_stti_r15_c::set(types::options e)
{
  type_ = e;
}
void ant_info_ded_stti_r15_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.tx_mode_dl_mbsfn_r15_present) {
        j.write_str("transmissionModeDL-MBSFN-r15", c.tx_mode_dl_mbsfn_r15.to_string());
      }
      if (c.tx_mode_dl_non_mbsfn_r15_present) {
        j.write_str("transmissionModeDL-nonMBSFN-r15", c.tx_mode_dl_non_mbsfn_r15.to_string());
      }
      if (c.codebook_subset_restrict_present) {
        j.write_fieldname("codebookSubsetRestriction");
        c.codebook_subset_restrict.to_json(j);
      }
      if (c.max_layers_mimo_stti_r15_present) {
        j.write_str("maxLayersMIMO-STTI-r15", c.max_layers_mimo_stti_r15.to_string());
      }
      j.write_bool("slotSubslotPDSCH-TxDiv-2Layer-r15", c.slot_subslot_pdsch_tx_div_minus2_layer_r15);
      j.write_bool("slotSubslotPDSCH-TxDiv-4Layer-r15", c.slot_subslot_pdsch_tx_div_minus4_layer_r15);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c");
  }
  j.end_obj();
}
SRSASN_CODE ant_info_ded_stti_r15_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.tx_mode_dl_mbsfn_r15_present, 1));
      HANDLE_CODE(bref.pack(c.tx_mode_dl_non_mbsfn_r15_present, 1));
      HANDLE_CODE(bref.pack(c.codebook_subset_restrict_present, 1));
      HANDLE_CODE(bref.pack(c.max_layers_mimo_stti_r15_present, 1));
      if (c.tx_mode_dl_mbsfn_r15_present) {
        HANDLE_CODE(c.tx_mode_dl_mbsfn_r15.pack(bref));
      }
      if (c.tx_mode_dl_non_mbsfn_r15_present) {
        HANDLE_CODE(c.tx_mode_dl_non_mbsfn_r15.pack(bref));
      }
      if (c.codebook_subset_restrict_present) {
        HANDLE_CODE(c.codebook_subset_restrict.pack(bref));
      }
      if (c.max_layers_mimo_stti_r15_present) {
        HANDLE_CODE(c.max_layers_mimo_stti_r15.pack(bref));
      }
      HANDLE_CODE(bref.pack(c.slot_subslot_pdsch_tx_div_minus2_layer_r15, 1));
      HANDLE_CODE(bref.pack(c.slot_subslot_pdsch_tx_div_minus4_layer_r15, 1));
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ant_info_ded_stti_r15_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.tx_mode_dl_mbsfn_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.tx_mode_dl_non_mbsfn_r15_present, 1));
      HANDLE_CODE(bref.unpack(c.codebook_subset_restrict_present, 1));
      HANDLE_CODE(bref.unpack(c.max_layers_mimo_stti_r15_present, 1));
      if (c.tx_mode_dl_mbsfn_r15_present) {
        HANDLE_CODE(c.tx_mode_dl_mbsfn_r15.unpack(bref));
      }
      if (c.tx_mode_dl_non_mbsfn_r15_present) {
        HANDLE_CODE(c.tx_mode_dl_non_mbsfn_r15.unpack(bref));
      }
      if (c.codebook_subset_restrict_present) {
        HANDLE_CODE(c.codebook_subset_restrict.unpack(bref));
      }
      if (c.max_layers_mimo_stti_r15_present) {
        HANDLE_CODE(c.max_layers_mimo_stti_r15.unpack(bref));
      }
      HANDLE_CODE(bref.unpack(c.slot_subslot_pdsch_tx_div_minus2_layer_r15, 1));
      HANDLE_CODE(bref.unpack(c.slot_subslot_pdsch_tx_div_minus4_layer_r15, 1));
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::destroy_()
{
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      c.destroy<fixed_bitstring<2> >();
      break;
    case types::n4_tx_ant_tm3_r15:
      c.destroy<fixed_bitstring<4> >();
      break;
    case types::n2_tx_ant_tm4_r15:
      c.destroy<fixed_bitstring<6> >();
      break;
    case types::n4_tx_ant_tm4_r15:
      c.destroy<fixed_bitstring<64> >();
      break;
    case types::n2_tx_ant_tm5_r15:
      c.destroy<fixed_bitstring<4> >();
      break;
    case types::n4_tx_ant_tm5_r15:
      c.destroy<fixed_bitstring<16> >();
      break;
    case types::n2_tx_ant_tm6_r15:
      c.destroy<fixed_bitstring<4> >();
      break;
    case types::n4_tx_ant_tm6_r15:
      c.destroy<fixed_bitstring<16> >();
      break;
    case types::n2_tx_ant_tm8_r15:
      c.destroy<fixed_bitstring<6> >();
      break;
    case types::n4_tx_ant_tm8_r15:
      c.destroy<fixed_bitstring<64> >();
      break;
    case types::n2_tx_ant_tm9and10_r15:
      c.destroy<fixed_bitstring<6> >();
      break;
    case types::n4_tx_ant_tm9and10_r15:
      c.destroy<fixed_bitstring<96> >();
      break;
    case types::n8_tx_ant_tm9and10_r15:
      c.destroy<fixed_bitstring<109> >();
      break;
    default:
      break;
  }
}
void ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      c.init<fixed_bitstring<2> >();
      break;
    case types::n4_tx_ant_tm3_r15:
      c.init<fixed_bitstring<4> >();
      break;
    case types::n2_tx_ant_tm4_r15:
      c.init<fixed_bitstring<6> >();
      break;
    case types::n4_tx_ant_tm4_r15:
      c.init<fixed_bitstring<64> >();
      break;
    case types::n2_tx_ant_tm5_r15:
      c.init<fixed_bitstring<4> >();
      break;
    case types::n4_tx_ant_tm5_r15:
      c.init<fixed_bitstring<16> >();
      break;
    case types::n2_tx_ant_tm6_r15:
      c.init<fixed_bitstring<4> >();
      break;
    case types::n4_tx_ant_tm6_r15:
      c.init<fixed_bitstring<16> >();
      break;
    case types::n2_tx_ant_tm8_r15:
      c.init<fixed_bitstring<6> >();
      break;
    case types::n4_tx_ant_tm8_r15:
      c.init<fixed_bitstring<64> >();
      break;
    case types::n2_tx_ant_tm9and10_r15:
      c.init<fixed_bitstring<6> >();
      break;
    case types::n4_tx_ant_tm9and10_r15:
      c.init<fixed_bitstring<96> >();
      break;
    case types::n8_tx_ant_tm9and10_r15:
      c.init<fixed_bitstring<109> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_");
  }
}
ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::codebook_subset_restrict_c_(
    const ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_& other)
{
  type_ = other.type();
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      c.init(other.c.get<fixed_bitstring<2> >());
      break;
    case types::n4_tx_ant_tm3_r15:
      c.init(other.c.get<fixed_bitstring<4> >());
      break;
    case types::n2_tx_ant_tm4_r15:
      c.init(other.c.get<fixed_bitstring<6> >());
      break;
    case types::n4_tx_ant_tm4_r15:
      c.init(other.c.get<fixed_bitstring<64> >());
      break;
    case types::n2_tx_ant_tm5_r15:
      c.init(other.c.get<fixed_bitstring<4> >());
      break;
    case types::n4_tx_ant_tm5_r15:
      c.init(other.c.get<fixed_bitstring<16> >());
      break;
    case types::n2_tx_ant_tm6_r15:
      c.init(other.c.get<fixed_bitstring<4> >());
      break;
    case types::n4_tx_ant_tm6_r15:
      c.init(other.c.get<fixed_bitstring<16> >());
      break;
    case types::n2_tx_ant_tm8_r15:
      c.init(other.c.get<fixed_bitstring<6> >());
      break;
    case types::n4_tx_ant_tm8_r15:
      c.init(other.c.get<fixed_bitstring<64> >());
      break;
    case types::n2_tx_ant_tm9and10_r15:
      c.init(other.c.get<fixed_bitstring<6> >());
      break;
    case types::n4_tx_ant_tm9and10_r15:
      c.init(other.c.get<fixed_bitstring<96> >());
      break;
    case types::n8_tx_ant_tm9and10_r15:
      c.init(other.c.get<fixed_bitstring<109> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_");
  }
}
ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_&
ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::
operator=(const ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      c.set(other.c.get<fixed_bitstring<2> >());
      break;
    case types::n4_tx_ant_tm3_r15:
      c.set(other.c.get<fixed_bitstring<4> >());
      break;
    case types::n2_tx_ant_tm4_r15:
      c.set(other.c.get<fixed_bitstring<6> >());
      break;
    case types::n4_tx_ant_tm4_r15:
      c.set(other.c.get<fixed_bitstring<64> >());
      break;
    case types::n2_tx_ant_tm5_r15:
      c.set(other.c.get<fixed_bitstring<4> >());
      break;
    case types::n4_tx_ant_tm5_r15:
      c.set(other.c.get<fixed_bitstring<16> >());
      break;
    case types::n2_tx_ant_tm6_r15:
      c.set(other.c.get<fixed_bitstring<4> >());
      break;
    case types::n4_tx_ant_tm6_r15:
      c.set(other.c.get<fixed_bitstring<16> >());
      break;
    case types::n2_tx_ant_tm8_r15:
      c.set(other.c.get<fixed_bitstring<6> >());
      break;
    case types::n4_tx_ant_tm8_r15:
      c.set(other.c.get<fixed_bitstring<64> >());
      break;
    case types::n2_tx_ant_tm9and10_r15:
      c.set(other.c.get<fixed_bitstring<6> >());
      break;
    case types::n4_tx_ant_tm9and10_r15:
      c.set(other.c.get<fixed_bitstring<96> >());
      break;
    case types::n8_tx_ant_tm9and10_r15:
      c.set(other.c.get<fixed_bitstring<109> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_");
  }

  return *this;
}
void ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      j.write_str("n2TxAntenna-tm3-r15", c.get<fixed_bitstring<2> >().to_string());
      break;
    case types::n4_tx_ant_tm3_r15:
      j.write_str("n4TxAntenna-tm3-r15", c.get<fixed_bitstring<4> >().to_string());
      break;
    case types::n2_tx_ant_tm4_r15:
      j.write_str("n2TxAntenna-tm4-r15", c.get<fixed_bitstring<6> >().to_string());
      break;
    case types::n4_tx_ant_tm4_r15:
      j.write_str("n4TxAntenna-tm4-r15", c.get<fixed_bitstring<64> >().to_string());
      break;
    case types::n2_tx_ant_tm5_r15:
      j.write_str("n2TxAntenna-tm5-r15", c.get<fixed_bitstring<4> >().to_string());
      break;
    case types::n4_tx_ant_tm5_r15:
      j.write_str("n4TxAntenna-tm5-r15", c.get<fixed_bitstring<16> >().to_string());
      break;
    case types::n2_tx_ant_tm6_r15:
      j.write_str("n2TxAntenna-tm6-r15", c.get<fixed_bitstring<4> >().to_string());
      break;
    case types::n4_tx_ant_tm6_r15:
      j.write_str("n4TxAntenna-tm6-r15", c.get<fixed_bitstring<16> >().to_string());
      break;
    case types::n2_tx_ant_tm8_r15:
      j.write_str("n2TxAntenna-tm8-r15", c.get<fixed_bitstring<6> >().to_string());
      break;
    case types::n4_tx_ant_tm8_r15:
      j.write_str("n4TxAntenna-tm8-r15", c.get<fixed_bitstring<64> >().to_string());
      break;
    case types::n2_tx_ant_tm9and10_r15:
      j.write_str("n2TxAntenna-tm9and10-r15", c.get<fixed_bitstring<6> >().to_string());
      break;
    case types::n4_tx_ant_tm9and10_r15:
      j.write_str("n4TxAntenna-tm9and10-r15", c.get<fixed_bitstring<96> >().to_string());
      break;
    case types::n8_tx_ant_tm9and10_r15:
      j.write_str("n8TxAntenna-tm9and10-r15", c.get<fixed_bitstring<109> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_");
  }
  j.end_obj();
}
SRSASN_CODE ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      HANDLE_CODE(c.get<fixed_bitstring<2> >().pack(bref));
      break;
    case types::n4_tx_ant_tm3_r15:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().pack(bref));
      break;
    case types::n2_tx_ant_tm4_r15:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().pack(bref));
      break;
    case types::n4_tx_ant_tm4_r15:
      HANDLE_CODE(c.get<fixed_bitstring<64> >().pack(bref));
      break;
    case types::n2_tx_ant_tm5_r15:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().pack(bref));
      break;
    case types::n4_tx_ant_tm5_r15:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().pack(bref));
      break;
    case types::n2_tx_ant_tm6_r15:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().pack(bref));
      break;
    case types::n4_tx_ant_tm6_r15:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().pack(bref));
      break;
    case types::n2_tx_ant_tm8_r15:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().pack(bref));
      break;
    case types::n4_tx_ant_tm8_r15:
      HANDLE_CODE(c.get<fixed_bitstring<64> >().pack(bref));
      break;
    case types::n2_tx_ant_tm9and10_r15:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().pack(bref));
      break;
    case types::n4_tx_ant_tm9and10_r15:
      HANDLE_CODE(c.get<fixed_bitstring<96> >().pack(bref));
      break;
    case types::n8_tx_ant_tm9and10_r15:
      HANDLE_CODE(c.get<fixed_bitstring<109> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::n2_tx_ant_tm3_r15:
      HANDLE_CODE(c.get<fixed_bitstring<2> >().unpack(bref));
      break;
    case types::n4_tx_ant_tm3_r15:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().unpack(bref));
      break;
    case types::n2_tx_ant_tm4_r15:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().unpack(bref));
      break;
    case types::n4_tx_ant_tm4_r15:
      HANDLE_CODE(c.get<fixed_bitstring<64> >().unpack(bref));
      break;
    case types::n2_tx_ant_tm5_r15:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().unpack(bref));
      break;
    case types::n4_tx_ant_tm5_r15:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().unpack(bref));
      break;
    case types::n2_tx_ant_tm6_r15:
      HANDLE_CODE(c.get<fixed_bitstring<4> >().unpack(bref));
      break;
    case types::n4_tx_ant_tm6_r15:
      HANDLE_CODE(c.get<fixed_bitstring<16> >().unpack(bref));
      break;
    case types::n2_tx_ant_tm8_r15:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().unpack(bref));
      break;
    case types::n4_tx_ant_tm8_r15:
      HANDLE_CODE(c.get<fixed_bitstring<64> >().unpack(bref));
      break;
    case types::n2_tx_ant_tm9and10_r15:
      HANDLE_CODE(c.get<fixed_bitstring<6> >().unpack(bref));
      break;
    case types::n4_tx_ant_tm9and10_r15:
      HANDLE_CODE(c.get<fixed_bitstring<96> >().unpack(bref));
      break;
    case types::n8_tx_ant_tm9and10_r15:
      HANDLE_CODE(c.get<fixed_bitstring<109> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ant_info_ded_stti_r15_c::setup_s_::codebook_subset_restrict_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// AntennaInfoUL-STTI-r15 ::= SEQUENCE
SRSASN_CODE ant_info_ul_stti_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(tx_mode_ul_stti_r15_present, 1));

  if (tx_mode_ul_stti_r15_present) {
    HANDLE_CODE(tx_mode_ul_stti_r15.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ant_info_ul_stti_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(tx_mode_ul_stti_r15_present, 1));

  if (tx_mode_ul_stti_r15_present) {
    HANDLE_CODE(tx_mode_ul_stti_r15.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ant_info_ul_stti_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (tx_mode_ul_stti_r15_present) {
    j.write_str("transmissionModeUL-STTI-r15", tx_mode_ul_stti_r15.to_string());
  }
  j.end_obj();
}

// CQI-ReportConfig-r15 ::= CHOICE
void cqi_report_cfg_r15_c::set(types::options e)
{
  type_ = e;
}
void cqi_report_cfg_r15_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.cqi_report_cfg_r10_present) {
        j.write_fieldname("cqi-ReportConfig-r10");
        c.cqi_report_cfg_r10.to_json(j);
      }
      if (c.cqi_report_cfg_v1130_present) {
        j.write_fieldname("cqi-ReportConfig-v1130");
        c.cqi_report_cfg_v1130.to_json(j);
      }
      if (c.cqi_report_cfg_pcell_v1250_present) {
        j.write_fieldname("cqi-ReportConfigPCell-v1250");
        c.cqi_report_cfg_pcell_v1250.to_json(j);
      }
      if (c.cqi_report_cfg_v1310_present) {
        j.write_fieldname("cqi-ReportConfig-v1310");
        c.cqi_report_cfg_v1310.to_json(j);
      }
      if (c.cqi_report_cfg_v1320_present) {
        j.write_fieldname("cqi-ReportConfig-v1320");
        c.cqi_report_cfg_v1320.to_json(j);
      }
      if (c.cqi_report_cfg_v1430_present) {
        j.write_fieldname("cqi-ReportConfig-v1430");
        c.cqi_report_cfg_v1430.to_json(j);
      }
      if (c.alt_cqi_table_minus1024_qam_r15_present) {
        j.write_str("altCQI-Table-1024QAM-r15", c.alt_cqi_table_minus1024_qam_r15.to_string());
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_r15_c");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_cfg_r15_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.cqi_report_cfg_r10_present, 1));
      HANDLE_CODE(bref.pack(c.cqi_report_cfg_v1130_present, 1));
      HANDLE_CODE(bref.pack(c.cqi_report_cfg_pcell_v1250_present, 1));
      HANDLE_CODE(bref.pack(c.cqi_report_cfg_v1310_present, 1));
      HANDLE_CODE(bref.pack(c.cqi_report_cfg_v1320_present, 1));
      HANDLE_CODE(bref.pack(c.cqi_report_cfg_v1430_present, 1));
      HANDLE_CODE(bref.pack(c.alt_cqi_table_minus1024_qam_r15_present, 1));
      if (c.cqi_report_cfg_r10_present) {
        HANDLE_CODE(c.cqi_report_cfg_r10.pack(bref));
      }
      if (c.cqi_report_cfg_v1130_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1130.pack(bref));
      }
      if (c.cqi_report_cfg_pcell_v1250_present) {
        HANDLE_CODE(c.cqi_report_cfg_pcell_v1250.pack(bref));
      }
      if (c.cqi_report_cfg_v1310_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1310.pack(bref));
      }
      if (c.cqi_report_cfg_v1320_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1320.pack(bref));
      }
      if (c.cqi_report_cfg_v1430_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1430.pack(bref));
      }
      if (c.alt_cqi_table_minus1024_qam_r15_present) {
        HANDLE_CODE(c.alt_cqi_table_minus1024_qam_r15.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_r15_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_cfg_r15_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.cqi_report_cfg_r10_present, 1));
      HANDLE_CODE(bref.unpack(c.cqi_report_cfg_v1130_present, 1));
      HANDLE_CODE(bref.unpack(c.cqi_report_cfg_pcell_v1250_present, 1));
      HANDLE_CODE(bref.unpack(c.cqi_report_cfg_v1310_present, 1));
      HANDLE_CODE(bref.unpack(c.cqi_report_cfg_v1320_present, 1));
      HANDLE_CODE(bref.unpack(c.cqi_report_cfg_v1430_present, 1));
      HANDLE_CODE(bref.unpack(c.alt_cqi_table_minus1024_qam_r15_present, 1));
      if (c.cqi_report_cfg_r10_present) {
        HANDLE_CODE(c.cqi_report_cfg_r10.unpack(bref));
      }
      if (c.cqi_report_cfg_v1130_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1130.unpack(bref));
      }
      if (c.cqi_report_cfg_pcell_v1250_present) {
        HANDLE_CODE(c.cqi_report_cfg_pcell_v1250.unpack(bref));
      }
      if (c.cqi_report_cfg_v1310_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1310.unpack(bref));
      }
      if (c.cqi_report_cfg_v1320_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1320.unpack(bref));
      }
      if (c.cqi_report_cfg_v1430_present) {
        HANDLE_CODE(c.cqi_report_cfg_v1430.unpack(bref));
      }
      if (c.alt_cqi_table_minus1024_qam_r15_present) {
        HANDLE_CODE(c.alt_cqi_table_minus1024_qam_r15.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_cfg_r15_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CQI-ReportPeriodic ::= CHOICE
void cqi_report_periodic_c::set(types::options e)
{
  type_ = e;
}
void cqi_report_periodic_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("cqi-PUCCH-ResourceIndex", c.cqi_pucch_res_idx);
      j.write_int("cqi-pmi-ConfigIndex", c.cqi_pmi_cfg_idx);
      j.write_fieldname("cqi-FormatIndicatorPeriodic");
      c.cqi_format_ind_periodic.to_json(j);
      if (c.ri_cfg_idx_present) {
        j.write_int("ri-ConfigIndex", c.ri_cfg_idx);
      }
      j.write_bool("simultaneousAckNackAndCQI", c.simul_ack_nack_and_cqi);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_c");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.ri_cfg_idx_present, 1));
      HANDLE_CODE(pack_integer(bref, c.cqi_pucch_res_idx, (uint16_t)0u, (uint16_t)1185u));
      HANDLE_CODE(pack_integer(bref, c.cqi_pmi_cfg_idx, (uint16_t)0u, (uint16_t)1023u));
      HANDLE_CODE(c.cqi_format_ind_periodic.pack(bref));
      if (c.ri_cfg_idx_present) {
        HANDLE_CODE(pack_integer(bref, c.ri_cfg_idx, (uint16_t)0u, (uint16_t)1023u));
      }
      HANDLE_CODE(bref.pack(c.simul_ack_nack_and_cqi, 1));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.ri_cfg_idx_present, 1));
      HANDLE_CODE(unpack_integer(c.cqi_pucch_res_idx, bref, (uint16_t)0u, (uint16_t)1185u));
      HANDLE_CODE(unpack_integer(c.cqi_pmi_cfg_idx, bref, (uint16_t)0u, (uint16_t)1023u));
      HANDLE_CODE(c.cqi_format_ind_periodic.unpack(bref));
      if (c.ri_cfg_idx_present) {
        HANDLE_CODE(unpack_integer(c.ri_cfg_idx, bref, (uint16_t)0u, (uint16_t)1023u));
      }
      HANDLE_CODE(bref.unpack(c.simul_ack_nack_and_cqi, 1));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_::set(types::options e)
{
  type_ = e;
}
void cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::wideband_cqi:
      break;
    case types::subband_cqi:
      j.write_fieldname("subbandCQI");
      j.start_obj();
      j.write_int("k", c.k);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_");
  }
  j.end_obj();
}
SRSASN_CODE cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::wideband_cqi:
      break;
    case types::subband_cqi:
      HANDLE_CODE(pack_integer(bref, c.k, (uint8_t)1u, (uint8_t)4u));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::wideband_cqi:
      break;
    case types::subband_cqi:
      HANDLE_CODE(unpack_integer(c.k, bref, (uint8_t)1u, (uint8_t)4u));
      break;
    default:
      log_invalid_choice_id(type_, "cqi_report_periodic_c::setup_s_::cqi_format_ind_periodic_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CRS-AssistanceInfo-r11 ::= SEQUENCE
SRSASN_CODE crs_assist_info_r11_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, pci_r11, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(ant_ports_count_r11.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_sf_cfg_list_r11, 1, 8));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= mbsfn_sf_cfg_list_v1430.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_v1430.is_present(), 1));
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *mbsfn_sf_cfg_list_v1430, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE crs_assist_info_r11_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(pci_r11, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(ant_ports_count_r11.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(mbsfn_sf_cfg_list_r11, bref, 1, 8));

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbsfn_sf_cfg_list_v1430_present;
      HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_v1430_present, 1));
      mbsfn_sf_cfg_list_v1430.set_present(mbsfn_sf_cfg_list_v1430_present);
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*mbsfn_sf_cfg_list_v1430, bref, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void crs_assist_info_r11_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId-r11", pci_r11);
  j.write_str("antennaPortsCount-r11", ant_ports_count_r11.to_string());
  j.start_array("mbsfn-SubframeConfigList-r11");
  for (const auto& e1 : mbsfn_sf_cfg_list_r11) {
    e1.to_json(j);
  }
  j.end_array();
  if (ext) {
    if (mbsfn_sf_cfg_list_v1430.is_present()) {
      j.start_array("mbsfn-SubframeConfigList-v1430");
      for (const auto& e1 : *mbsfn_sf_cfg_list_v1430) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// CRS-AssistanceInfo-r13 ::= SEQUENCE
SRSASN_CODE crs_assist_info_r13_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_r13_present, 1));

  HANDLE_CODE(pack_integer(bref, pci_r13, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(ant_ports_count_r13.pack(bref));
  if (mbsfn_sf_cfg_list_r13_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_sf_cfg_list_r13, 1, 8));
  }

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= mbsfn_sf_cfg_list_v1430.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(mbsfn_sf_cfg_list_v1430.is_present(), 1));
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(pack_dyn_seq_of(bref, *mbsfn_sf_cfg_list_v1430, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE crs_assist_info_r13_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_r13_present, 1));

  HANDLE_CODE(unpack_integer(pci_r13, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(ant_ports_count_r13.unpack(bref));
  if (mbsfn_sf_cfg_list_r13_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbsfn_sf_cfg_list_r13, bref, 1, 8));
  }

  if (ext) {
    ext_groups_unpacker_guard group_flags(1);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool mbsfn_sf_cfg_list_v1430_present;
      HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_list_v1430_present, 1));
      mbsfn_sf_cfg_list_v1430.set_present(mbsfn_sf_cfg_list_v1430_present);
      if (mbsfn_sf_cfg_list_v1430.is_present()) {
        HANDLE_CODE(unpack_dyn_seq_of(*mbsfn_sf_cfg_list_v1430, bref, 1, 8));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void crs_assist_info_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId-r13", pci_r13);
  j.write_str("antennaPortsCount-r13", ant_ports_count_r13.to_string());
  if (mbsfn_sf_cfg_list_r13_present) {
    j.start_array("mbsfn-SubframeConfigList-r13");
    for (const auto& e1 : mbsfn_sf_cfg_list_r13) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ext) {
    if (mbsfn_sf_cfg_list_v1430.is_present()) {
      j.start_array("mbsfn-SubframeConfigList-v1430");
      for (const auto& e1 : *mbsfn_sf_cfg_list_v1430) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// CRS-AssistanceInfo-r15 ::= SEQUENCE
SRSASN_CODE crs_assist_info_r15_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(crs_intf_mitig_enabled_minus15_present, 1));

  HANDLE_CODE(pack_integer(bref, pci_r15, (uint16_t)0u, (uint16_t)503u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE crs_assist_info_r15_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(crs_intf_mitig_enabled_minus15_present, 1));

  HANDLE_CODE(unpack_integer(pci_r15, bref, (uint16_t)0u, (uint16_t)503u));

  return SRSASN_SUCCESS;
}
void crs_assist_info_r15_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId-r15", pci_r15);
  if (crs_intf_mitig_enabled_minus15_present) {
    j.write_str("crs-IntfMitigEnabled-15", "enabled");
  }
  j.end_obj();
}

// CSI-RS-Config-r15 ::= CHOICE
void csi_rs_cfg_r15_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_r15_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.csi_rs_cfg_r10_present) {
        j.write_fieldname("csi-RS-Config-r10");
        c.csi_rs_cfg_r10.to_json(j);
      }
      if (c.csi_rs_cfg_v1250_present) {
        j.write_fieldname("csi-RS-Config-v1250");
        c.csi_rs_cfg_v1250.to_json(j);
      }
      if (c.csi_rs_cfg_v1310_present) {
        j.write_fieldname("csi-RS-Config-v1310");
        c.csi_rs_cfg_v1310.to_json(j);
      }
      if (c.csi_rs_cfg_v1430_present) {
        j.write_fieldname("csi-RS-Config-v1430");
        c.csi_rs_cfg_v1430.to_json(j);
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_r15_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_r15_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.pack(c.csi_rs_cfg_r10_present, 1));
      HANDLE_CODE(bref.pack(c.csi_rs_cfg_v1250_present, 1));
      HANDLE_CODE(bref.pack(c.csi_rs_cfg_v1310_present, 1));
      HANDLE_CODE(bref.pack(c.csi_rs_cfg_v1430_present, 1));
      if (c.csi_rs_cfg_r10_present) {
        HANDLE_CODE(c.csi_rs_cfg_r10.pack(bref));
      }
      if (c.csi_rs_cfg_v1250_present) {
        HANDLE_CODE(c.csi_rs_cfg_v1250.pack(bref));
      }
      if (c.csi_rs_cfg_v1310_present) {
        HANDLE_CODE(c.csi_rs_cfg_v1310.pack(bref));
      }
      if (c.csi_rs_cfg_v1430_present) {
        HANDLE_CODE(c.csi_rs_cfg_v1430.pack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_r15_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_r15_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(bref.unpack(c.csi_rs_cfg_r10_present, 1));
      HANDLE_CODE(bref.unpack(c.csi_rs_cfg_v1250_present, 1));
      HANDLE_CODE(bref.unpack(c.csi_rs_cfg_v1310_present, 1));
      HANDLE_CODE(bref.unpack(c.csi_rs_cfg_v1430_present, 1));
      if (c.csi_rs_cfg_r10_present) {
        HANDLE_CODE(c.csi_rs_cfg_r10.unpack(bref));
      }
      if (c.csi_rs_cfg_v1250_present) {
        HANDLE_CODE(c.csi_rs_cfg_v1250.unpack(bref));
      }
      if (c.csi_rs_cfg_v1310_present) {
        HANDLE_CODE(c.csi_rs_cfg_v1310.unpack(bref));
      }
      if (c.csi_rs_cfg_v1430_present) {
        HANDLE_CODE(c.csi_rs_cfg_v1430.unpack(bref));
      }
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_r15_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// CSI-RS-ConfigZP-ApList-r14 ::= CHOICE
void csi_rs_cfg_zp_ap_list_r14_c::set(types::options e)
{
  type_ = e;
}
void csi_rs_cfg_zp_ap_list_r14_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.start_array("setup");
      for (const auto& e1 : c) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_zp_ap_list_r14_c");
  }
  j.end_obj();
}
SRSASN_CODE csi_rs_cfg_zp_ap_list_r14_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 4));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_zp_ap_list_r14_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE csi_rs_cfg_zp_ap_list_r14_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 4));
      break;
    default:
      log_invalid_choice_id(type_, "csi_rs_cfg_zp_ap_list_r14_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// DMRS-Config-r11 ::= CHOICE
void dmrs_cfg_r11_c::set(types::options e)
{
  type_ = e;
}
void dmrs_cfg_r11_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("scramblingIdentity-r11", c.scrambling_id_r11);
      j.write_int("scramblingIdentity2-r11", c.scrambling_id2_r11);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "dmrs_cfg_r11_c");
  }
  j.end_obj();
}
SRSASN_CODE dmrs_cfg_r11_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_integer(bref, c.scrambling_id_r11, (uint16_t)0u, (uint16_t)503u));
      HANDLE_CODE(pack_integer(bref, c.scrambling_id2_r11, (uint16_t)0u, (uint16_t)503u));
      break;
    default:
      log_invalid_choice_id(type_, "dmrs_cfg_r11_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dmrs_cfg_r11_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_integer(c.scrambling_id_r11, bref, (uint16_t)0u, (uint16_t)503u));
      HANDLE_CODE(unpack_integer(c.scrambling_id2_r11, bref, (uint16_t)0u, (uint16_t)503u));
      break;
    default:
      log_invalid_choice_id(type_, "dmrs_cfg_r11_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// DMRS-Config-v1310 ::= SEQUENCE
SRSASN_CODE dmrs_cfg_v1310_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(dmrs_table_alt_r13_present, 1));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dmrs_cfg_v1310_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(dmrs_table_alt_r13_present, 1));

  return SRSASN_SUCCESS;
}
void dmrs_cfg_v1310_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dmrs_table_alt_r13_present) {
    j.write_str("dmrs-tableAlt-r13", "true");
  }
  j.end_obj();
}

// DeltaTxD-OffsetListPUCCH-r10 ::= SEQUENCE
SRSASN_CODE delta_tx_d_offset_list_pucch_r10_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(delta_tx_d_offset_pucch_format1_r10.pack(bref));
  HANDLE_CODE(delta_tx_d_offset_pucch_format1a1b_r10.pack(bref));
  HANDLE_CODE(delta_tx_d_offset_pucch_format22a2b_r10.pack(bref));
  HANDLE_CODE(delta_tx_d_offset_pucch_format3_r10.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE delta_tx_d_offset_list_pucch_r10_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(delta_tx_d_offset_pucch_format1_r10.unpack(bref));
  HANDLE_CODE(delta_tx_d_offset_pucch_format1a1b_r10.unpack(bref));
  HANDLE_CODE(delta_tx_d_offset_pucch_format22a2b_r10.unpack(bref));
  HANDLE_CODE(delta_tx_d_offset_pucch_format3_r10.unpack(bref));

  return SRSASN_SUCCESS;
}
void delta_tx_d_offset_list_pucch_r10_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("deltaTxD-OffsetPUCCH-Format1-r10", delta_tx_d_offset_pucch_format1_r10.to_string());
  j.write_str("deltaTxD-OffsetPUCCH-Format1a1b-r10", delta_tx_d_offset_pucch_format1a1b_r10.to_string());
  j.write_str("deltaTxD-OffsetPUCCH-Format22a2b-r10", delta_tx_d_offset_pucch_format22a2b_r10.to_string());
  j.write_str("deltaTxD-OffsetPUCCH-Format3-r10", delta_tx_d_offset_pucch_format3_r10.to_string());
  j.end_obj();
}

// DeltaTxD-OffsetListPUCCH-v1130 ::= SEQUENCE
SRSASN_CODE delta_tx_d_offset_list_pucch_v1130_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(delta_tx_d_offset_pucch_format1b_cs_r11.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE delta_tx_d_offset_list_pucch_v1130_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(delta_tx_d_offset_pucch_format1b_cs_r11.unpack(bref));

  return SRSASN_SUCCESS;
}
void delta_tx_d_offset_list_pucch_v1130_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("deltaTxD-OffsetPUCCH-Format1bCS-r11", delta_tx_d_offset_pucch_format1b_cs_r11.to_string());
  j.end_obj();
}

// EIMTA-MainConfig-r12 ::= CHOICE
void eimta_main_cfg_r12_c::set(types::options e)
{
  type_ = e;
}
void eimta_main_cfg_r12_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_str("eimta-RNTI-r12", c.eimta_rnti_r12.to_string());
      j.write_str("eimta-CommandPeriodicity-r12", c.eimta_cmd_periodicity_r12.to_string());
      j.write_str("eimta-CommandSubframeSet-r12", c.eimta_cmd_sf_set_r12.to_string());
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_r12_c");
  }
  j.end_obj();
}
SRSASN_CODE eimta_main_cfg_r12_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.eimta_rnti_r12.pack(bref));
      HANDLE_CODE(c.eimta_cmd_periodicity_r12.pack(bref));
      HANDLE_CODE(c.eimta_cmd_sf_set_r12.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_r12_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE eimta_main_cfg_r12_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.eimta_rnti_r12.unpack(bref));
      HANDLE_CODE(c.eimta_cmd_periodicity_r12.unpack(bref));
      HANDLE_CODE(c.eimta_cmd_sf_set_r12.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_r12_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// EIMTA-MainConfigServCell-r12 ::= CHOICE
void eimta_main_cfg_serv_cell_r12_c::set(types::options e)
{
  type_ = e;
}
void eimta_main_cfg_serv_cell_r12_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.write_int("eimta-UL-DL-ConfigIndex-r12", c.eimta_ul_dl_cfg_idx_r12);
      j.write_str("eimta-HARQ-ReferenceConfig-r12", c.eimta_harq_ref_cfg_r12.to_string());
      j.write_fieldname("mbsfn-SubframeConfigList-v1250");
      c.mbsfn_sf_cfg_list_v1250.to_json(j);
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_serv_cell_r12_c");
  }
  j.end_obj();
}
SRSASN_CODE eimta_main_cfg_serv_cell_r12_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_integer(bref, c.eimta_ul_dl_cfg_idx_r12, (uint8_t)1u, (uint8_t)5u));
      HANDLE_CODE(c.eimta_harq_ref_cfg_r12.pack(bref));
      HANDLE_CODE(c.mbsfn_sf_cfg_list_v1250.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_serv_cell_r12_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE eimta_main_cfg_serv_cell_r12_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_integer(c.eimta_ul_dl_cfg_idx_r12, bref, (uint8_t)1u, (uint8_t)5u));
      HANDLE_CODE(c.eimta_harq_ref_cfg_r12.unpack(bref));
      HANDLE_CODE(c.mbsfn_sf_cfg_list_v1250.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_serv_cell_r12_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_::set(types::options e)
{
  type_ = e;
}
void eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      j.start_array("subframeConfigList-r12");
      for (const auto& e1 : c.sf_cfg_list_r12) {
        e1.to_json(j);
      }
      j.end_array();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_");
  }
  j.end_obj();
}
SRSASN_CODE eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.sf_cfg_list_r12, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(unpack_dyn_seq_of(c.sf_cfg_list_r12, bref, 1, 8));
      break;
    default:
      log_invalid_choice_id(type_, "eimta_main_cfg_serv_cell_r12_c::setup_s_::mbsfn_sf_cfg_list_v1250_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

// Format4-resource-r13 ::= SEQUENCE
SRSASN_CODE format4_res_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, start_prb_format4_r13, (uint8_t)0u, (uint8_t)109u));
  HANDLE_CODE(pack_integer(bref, nof_prb_format4_r13, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE format4_res_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(start_prb_format4_r13, bref, (uint8_t)0u, (uint8_t)109u));
  HANDLE_CODE(unpack_integer(nof_prb_format4_r13, bref, (uint8_t)0u, (uint8_t)7u));

  return SRSASN_SUCCESS;
}
void format4_res_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("startingPRB-format4-r13", start_prb_format4_r13);
  j.write_int("numberOfPRB-format4-r13", nof_prb_format4_r13);
  j.end_obj();
}

// Format5-resource-r13 ::= SEQUENCE
SRSASN_CODE format5_res_r13_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, start_prb_format5_r13, (uint8_t)0u, (uint8_t)109u));
  HANDLE_CODE(pack_integer(bref, cdm_idx_format5_r13, (uint8_t)0u, (uint8_t)1u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE format5_res_r13_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(start_prb_format5_r13, bref, (uint8_t)0u, (uint8_t)109u));
  HANDLE_CODE(unpack_integer(cdm_idx_format5_r13, bref, (uint8_t)0u, (uint8_t)1u));

  return SRSASN_SUCCESS;
}
void format5_res_r13_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("startingPRB-format5-r13", start_prb_format5_r13);
  j.write_int("cdm-index-format5-r13", cdm_idx_format5_r13);
  j.end_obj();
}

// NeighCellsInfo-r12 ::= SEQUENCE
SRSASN_CODE neigh_cells_info_r12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbsfn_sf_cfg_r12_present, 1));

  HANDLE_CODE(pack_integer(bref, pci_r12, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(pack_integer(bref, p_b_r12, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(crs_ports_count_r12.pack(bref));
  if (mbsfn_sf_cfg_r12_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_sf_cfg_r12, 1, 8));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, p_a_list_r12, 1, 3));
  HANDLE_CODE(tx_mode_list_r12.pack(bref));
  HANDLE_CODE(pack_integer(bref, res_alloc_granularity_r12, (uint8_t)1u, (uint8_t)4u));

  return SRSASN_SUCCESS;
}
SRSASN_CODE neigh_cells_info_r12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbsfn_sf_cfg_r12_present, 1));

  HANDLE_CODE(unpack_integer(pci_r12, bref, (uint16_t)0u, (uint16_t)503u));
  HANDLE_CODE(unpack_integer(p_b_r12, bref, (uint8_t)0u, (uint8_t)3u));
  HANDLE_CODE(crs_ports_count_r12.unpack(bref));
  if (mbsfn_sf_cfg_r12_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbsfn_sf_cfg_r12, bref, 1, 8));
  }
  HANDLE_CODE(unpack_dyn_seq_of(p_a_list_r12, bref, 1, 3));
  HANDLE_CODE(tx_mode_list_r12.unpack(bref));
  HANDLE_CODE(unpack_integer(res_alloc_granularity_r12, bref, (uint8_t)1u, (uint8_t)4u));

  return SRSASN_SUCCESS;
}
void neigh_cells_info_r12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("physCellId-r12", pci_r12);
  j.write_int("p-b-r12", p_b_r12);
  j.write_str("crs-PortsCount-r12", crs_ports_count_r12.to_string());
  if (mbsfn_sf_cfg_r12_present) {
    j.start_array("mbsfn-SubframeConfig-r12");
    for (const auto& e1 : mbsfn_sf_cfg_r12) {
      e1.to_json(j);
    }
    j.end_array();
  }
  j.start_array("p-aList-r12");
  for (const auto& e1 : p_a_list_r12) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.write_str("transmissionModeList-r12", tx_mode_list_r12.to_string());
  j.write_int("resAllocGranularity-r12", res_alloc_granularity_r12);
  j.end_obj();
}

// PDCP-Config ::= SEQUENCE
SRSASN_CODE pdcp_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(discard_timer_present, 1));
  HANDLE_CODE(bref.pack(rlc_am_present, 1));
  HANDLE_CODE(bref.pack(rlc_um_present, 1));

  if (discard_timer_present) {
    HANDLE_CODE(discard_timer.pack(bref));
  }
  if (rlc_am_present) {
    HANDLE_CODE(bref.pack(rlc_am.status_report_required, 1));
  }
  if (rlc_um_present) {
    HANDLE_CODE(rlc_um.pdcp_sn_size.pack(bref));
  }
  HANDLE_CODE(hdr_compress.pack(bref));

  if (ext) {
    ext_groups_packer_guard group_flags;
    group_flags[0] |= rn_integrity_protection_r10_present;
    group_flags[1] |= pdcp_sn_size_v1130_present;
    group_flags[2] |= ul_data_split_drb_via_scg_r12_present;
    group_flags[2] |= t_reordering_r12_present;
    group_flags[3] |= ul_data_split_thres_r13.is_present();
    group_flags[3] |= pdcp_sn_size_v1310_present;
    group_flags[3] |= status_feedback_r13.is_present();
    group_flags[4] |= ul_lwa_cfg_r14.is_present();
    group_flags[4] |= ul_only_hdr_compress_r14.is_present();
    group_flags[5] |= ul_data_compress_r15.is_present();
    group_flags[5] |= pdcp_dupl_cfg_r15.is_present();
    group_flags.pack(bref);

    if (group_flags[0]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(rn_integrity_protection_r10_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(pdcp_sn_size_v1130_present, 1));
    }
    if (group_flags[2]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ul_data_split_drb_via_scg_r12_present, 1));
      HANDLE_CODE(bref.pack(t_reordering_r12_present, 1));
      if (ul_data_split_drb_via_scg_r12_present) {
        HANDLE_CODE(bref.pack(ul_data_split_drb_via_scg_r12, 1));
      }
      if (t_reordering_r12_present) {
        HANDLE_CODE(t_reordering_r12.pack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ul_data_split_thres_r13.is_present(), 1));
      HANDLE_CODE(bref.pack(pdcp_sn_size_v1310_present, 1));
      HANDLE_CODE(bref.pack(status_feedback_r13.is_present(), 1));
      if (ul_data_split_thres_r13.is_present()) {
        HANDLE_CODE(ul_data_split_thres_r13->pack(bref));
      }
      if (status_feedback_r13.is_present()) {
        HANDLE_CODE(status_feedback_r13->pack(bref));
      }
    }
    if (group_flags[4]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ul_lwa_cfg_r14.is_present(), 1));
      HANDLE_CODE(bref.pack(ul_only_hdr_compress_r14.is_present(), 1));
      if (ul_lwa_cfg_r14.is_present()) {
        HANDLE_CODE(ul_lwa_cfg_r14->pack(bref));
      }
      if (ul_only_hdr_compress_r14.is_present()) {
        HANDLE_CODE(ul_only_hdr_compress_r14->pack(bref));
      }
    }
    if (group_flags[5]) {
      varlength_field_pack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.pack(ul_data_compress_r15.is_present(), 1));
      HANDLE_CODE(bref.pack(pdcp_dupl_cfg_r15.is_present(), 1));
      if (ul_data_compress_r15.is_present()) {
        bref.pack(ul_data_compress_r15->ext, 1);
        HANDLE_CODE(bref.pack(ul_data_compress_r15->dictionary_r15_present, 1));
        HANDLE_CODE(ul_data_compress_r15->buffer_size_r15.pack(bref));
        if (ul_data_compress_r15->dictionary_r15_present) {
          HANDLE_CODE(ul_data_compress_r15->dictionary_r15.pack(bref));
        }
      }
      if (pdcp_dupl_cfg_r15.is_present()) {
        HANDLE_CODE(pdcp_dupl_cfg_r15->pack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdcp_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(discard_timer_present, 1));
  HANDLE_CODE(bref.unpack(rlc_am_present, 1));
  HANDLE_CODE(bref.unpack(rlc_um_present, 1));

  if (discard_timer_present) {
    HANDLE_CODE(discard_timer.unpack(bref));
  }
  if (rlc_am_present) {
    HANDLE_CODE(bref.unpack(rlc_am.status_report_required, 1));
  }
  if (rlc_um_present) {
    HANDLE_CODE(rlc_um.pdcp_sn_size.unpack(bref));
  }
  HANDLE_CODE(hdr_compress.unpack(bref));

  if (ext) {
    ext_groups_unpacker_guard group_flags(6);
    group_flags.unpack(bref);

    if (group_flags[0]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(rn_integrity_protection_r10_present, 1));
    }
    if (group_flags[1]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(pdcp_sn_size_v1130_present, 1));
    }
    if (group_flags[2]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      HANDLE_CODE(bref.unpack(ul_data_split_drb_via_scg_r12_present, 1));
      HANDLE_CODE(bref.unpack(t_reordering_r12_present, 1));
      if (ul_data_split_drb_via_scg_r12_present) {
        HANDLE_CODE(bref.unpack(ul_data_split_drb_via_scg_r12, 1));
      }
      if (t_reordering_r12_present) {
        HANDLE_CODE(t_reordering_r12.unpack(bref));
      }
    }
    if (group_flags[3]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool ul_data_split_thres_r13_present;
      HANDLE_CODE(bref.unpack(ul_data_split_thres_r13_present, 1));
      ul_data_split_thres_r13.set_present(ul_data_split_thres_r13_present);
      HANDLE_CODE(bref.unpack(pdcp_sn_size_v1310_present, 1));
      bool status_feedback_r13_present;
      HANDLE_CODE(bref.unpack(status_feedback_r13_present, 1));
      status_feedback_r13.set_present(status_feedback_r13_present);
      if (ul_data_split_thres_r13.is_present()) {
        HANDLE_CODE(ul_data_split_thres_r13->unpack(bref));
      }
      if (status_feedback_r13.is_present()) {
        HANDLE_CODE(status_feedback_r13->unpack(bref));
      }
    }
    if (group_flags[4]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool ul_lwa_cfg_r14_present;
      HANDLE_CODE(bref.unpack(ul_lwa_cfg_r14_present, 1));
      ul_lwa_cfg_r14.set_present(ul_lwa_cfg_r14_present);
      bool ul_only_hdr_compress_r14_present;
      HANDLE_CODE(bref.unpack(ul_only_hdr_compress_r14_present, 1));
      ul_only_hdr_compress_r14.set_present(ul_only_hdr_compress_r14_present);
      if (ul_lwa_cfg_r14.is_present()) {
        HANDLE_CODE(ul_lwa_cfg_r14->unpack(bref));
      }
      if (ul_only_hdr_compress_r14.is_present()) {
        HANDLE_CODE(ul_only_hdr_compress_r14->unpack(bref));
      }
    }
    if (group_flags[5]) {
      varlength_field_unpack_guard varlen_scope(bref, false);

      bool ul_data_compress_r15_present;
      HANDLE_CODE(bref.unpack(ul_data_compress_r15_present, 1));
      ul_data_compress_r15.set_present(ul_data_compress_r15_present);
      bool pdcp_dupl_cfg_r15_present;
      HANDLE_CODE(bref.unpack(pdcp_dupl_cfg_r15_present, 1));
      pdcp_dupl_cfg_r15.set_present(pdcp_dupl_cfg_r15_present);
      if (ul_data_compress_r15.is_present()) {
        bref.unpack(ul_data_compress_r15->ext, 1);
        HANDLE_CODE(bref.unpack(ul_data_compress_r15->dictionary_r15_present, 1));
        HANDLE_CODE(ul_data_compress_r15->buffer_size_r15.unpack(bref));
        if (ul_data_compress_r15->dictionary_r15_present) {
          HANDLE_CODE(ul_data_compress_r15->dictionary_r15.unpack(bref));
        }
      }
      if (pdcp_dupl_cfg_r15.is_present()) {
        HANDLE_CODE(pdcp_dupl_cfg_r15->unpack(bref));
      }
    }
  }
  return SRSASN_SUCCESS;
}
void pdcp_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (discard_timer_present) {
    j.write_str("discardTimer", discard_timer.to_string());
  }
  if (rlc_am_present) {
    j.write_fieldname("rlc-AM");
    j.start_obj();
    j.write_bool("statusReportRequired", rlc_am.status_report_required);
    j.end_obj();
  }
  if (rlc_um_present) {
    j.write_fieldname("rlc-UM");
    j.start_obj();
    j.write_str("pdcp-SN-Size", rlc_um.pdcp_sn_size.to_string());
    j.end_obj();
  }
  j.write_fieldname("headerCompression");
  hdr_compress.to_json(j);
  if (ext) {
    if (rn_integrity_protection_r10_present) {
      j.write_str("rn-IntegrityProtection-r10", "enabled");
    }
    if (pdcp_sn_size_v1130_present) {
      j.write_str("pdcp-SN-Size-v1130", "len15bits");
    }
    if (ul_data_split_drb_via_scg_r12_present) {
      j.write_bool("ul-DataSplitDRB-ViaSCG-r12", ul_data_split_drb_via_scg_r12);
    }
    if (t_reordering_r12_present) {
      j.write_str("t-Reordering-r12", t_reordering_r12.to_string());
    }
    if (ul_data_split_thres_r13.is_present()) {
      j.write_fieldname("ul-DataSplitThreshold-r13");
      ul_data_split_thres_r13->to_json(j);
    }
    if (pdcp_sn_size_v1310_present) {
      j.write_str("pdcp-SN-Size-v1310", "len18bits");
    }
    if (status_feedback_r13.is_present()) {
      j.write_fieldname("statusFeedback-r13");
      status_feedback_r13->to_json(j);
    }
    if (ul_lwa_cfg_r14.is_present()) {
      j.write_fieldname("ul-LWA-Config-r14");
      ul_lwa_cfg_r14->to_json(j);
    }
    if (ul_only_hdr_compress_r14.is_present()) {
      j.write_fieldname("uplinkOnlyHeaderCompression-r14");
      ul_only_hdr_compress_r14->to_json(j);
    }
    if (ul_data_compress_r15.is_present()) {
      j.write_fieldname("uplinkDataCompression-r15");
      j.start_obj();
      j.write_str("bufferSize-r15", ul_data_compress_r15->buffer_size_r15.to_string());
      if (ul_data_compress_r15->dictionary_r15_present) {
        j.write_str("dictionary-r15", ul_data_compress_r15->dictionary_r15.to_string());
      }
      j.end_obj();
    }
    if (pdcp_dupl_cfg_r15.is_present()) {
      j.write_fieldname("pdcp-DuplicationConfig-r15");
      pdcp_dupl_cfg_r15->to_json(j);
    }
  }
  j.end_obj();
}

void pdcp_cfg_s::hdr_compress_c_::set(types::options e)
{
  type_ = e;
}
void pdcp_cfg_s::hdr_compress_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::not_used:
      break;
    case types::rohc:
      j.write_fieldname("rohc");
      j.start_obj();
      if (c.max_cid_present) {
        j.write_int("maxCID", c.max_cid);
      }
      j.write_fieldname("profiles");
      j.start_obj();
      j.write_bool("profile0x0001", c.profiles.profile0x0001);
      j.write_bool("profile0x0002", c.profiles.profile0x0002);
      j.write_bool("profile0x0003", c.profiles.profile0x0003);
      j.write_bool("profile0x0004", c.profiles.profile0x0004);
      j.write_bool("profile0x0006", c.profiles.profile0x0006);
      j.write_bool("profile0x0101", c.profiles.profile0x0101);
      j.write_bool("profile0x0102", c.profiles.profile0x0102);
      j.write_bool("profile0x0103", c.profiles.profile0x0103);
      j.write_bool("profile0x0104", c.profiles.profile0x0104);
      j.end_obj();
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::hdr_compress_c_");
  }
  j.end_obj();
}
SRSASN_CODE pdcp_cfg_s::hdr_compress_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::not_used:
      break;
    case types::rohc:
      bref.pack(c.ext, 1);
      HANDLE_CODE(bref.pack(c.max_cid_present, 1));
      if (c.max_cid_present) {
        HANDLE_CODE(pack_integer(bref, c.max_cid, (uint16_t)1u, (uint16_t)16383u));
      }
      HANDLE_CODE(bref.pack(c.profiles.profile0x0001, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0002, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0003, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0004, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0006, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0101, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0102, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0103, 1));
      HANDLE_CODE(bref.pack(c.profiles.profile0x0104, 1));
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::hdr_compress_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdcp_cfg_s::hdr_compress_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::not_used:
      break;
    case types::rohc:
      bref.unpack(c.ext, 1);
      HANDLE_CODE(bref.unpack(c.max_cid_present, 1));
      if (c.max_cid_present) {
        HANDLE_CODE(unpack_integer(c.max_cid, bref, (uint16_t)1u, (uint16_t)16383u));
      }
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0001, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0002, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0003, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0004, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0006, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0101, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0102, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0103, 1));
      HANDLE_CODE(bref.unpack(c.profiles.profile0x0104, 1));
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::hdr_compress_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void pdcp_cfg_s::ul_data_split_thres_r13_c_::set(types::options e)
{
  type_ = e;
}
void pdcp_cfg_s::ul_data_split_thres_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_str("setup", c.to_string());
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::ul_data_split_thres_r13_c_");
  }
  j.end_obj();
}
SRSASN_CODE pdcp_cfg_s::ul_data_split_thres_r13_c_::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::ul_data_split_thres_r13_c_");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdcp_cfg_s::ul_data_split_thres_r13_c_::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      HANDLE_CODE(c.unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::ul_data_split_thres_r13_c_");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

void pdcp_cfg_s::status_feedback_r13_c_::set(types::options e)
{
  type_ = e;
}
void pdcp_cfg_s::status_feedback_r13_c_::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::release:
      break;
    case types::setup:
      j.write_fieldname("setup");
      j.start_obj();
      if (c.status_pdu_type_for_polling_r13_present) {
        j.write_str("statusPDU-TypeForPolling-r13", c.status_pdu_type_for_polling_r13.to_string());
      }
      if (c.status_pdu_periodicity_type1_r13_present) {
        j.write_str("statusPDU-Periodicity-Type1-r13", c.status_pdu_periodicity_type1_r13.to_string());
      }
      if (c.status_pdu_periodicity_type2_r13_present) {
        j.write_str("statusPDU-Periodicity-Type2-r13", c.status_pdu_periodicity_type2_r13.to_string());
      }
      if (c.status_pdu_periodicity_offset_r13_present) {
        j.write_str("statusPDU-Periodicity-Offset-r13", c.status_pdu_periodicity_offset_r13.to_string());
      }
      j.end_obj();
      break;
    default:
      log_invalid_choice_id(type_, "pdcp_cfg_s::status_feedback_r13_c_");
  }
  j.