/*
 * Copyright 2013-2020 Software Radio Systems Limited
 *
 * This file is part of srsLTE.
 *
 * srsLTE is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsLTE is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

#include "srslte/asn1/s1ap_asn1.h"
#include <sstream>

using namespace asn1;
using namespace asn1::s1ap;

/*******************************************************************************
 *                              Logging Utilities
 ******************************************************************************/

void asn1::s1ap::log_invalid_access_choice_id(uint32_t val, uint32_t choice_id)
{
  asn1::log_error("The access choice id is invalid (%d!=%d)\n", val, choice_id);
}

void asn1::s1ap::assert_choice_type(uint32_t val, uint32_t choice_id)
{
  if (val != choice_id) {
    log_invalid_access_choice_id(val, choice_id);
  }
}

void asn1::s1ap::assert_choice_type(const std::string& access_type,
                                    const std::string& current_type,
                                    const std::string& choice_type)
{
  if (access_type != current_type) {
    asn1::log_error("Invalid field access for choice type \"%s\" (\"%s\"!=\"%s\")\n",
                    choice_type.c_str(),
                    access_type.c_str(),
                    current_type.c_str());
  }
}

const char*
asn1::s1ap::convert_enum_idx(const char* array[], uint32_t nof_types, uint32_t enum_val, const char* enum_type)
{
  if (enum_val >= nof_types) {
    if (enum_val == nof_types) {
      asn1::log_error("The enum of type %s was not initialized.\n", enum_type);
    } else {
      asn1::log_error("The enum value=%d of type %s is not valid.\n", enum_val, enum_type);
    }
    return "";
  }
  return array[enum_val];
}

template <class ItemType>
ItemType asn1::s1ap::map_enum_number(ItemType* array, uint32_t nof_types, uint32_t enum_val, const char* enum_type)
{
  if (enum_val >= nof_types) {
    if (enum_val == nof_types) {
      asn1::log_error("The enum of type %s is not initialized.\n", enum_type);
    } else {
      asn1::log_error("The enum value=%d of type %s cannot be converted to a number.\n", enum_val, enum_type);
    }
    return 0;
  }
  return array[enum_val];
}
template const uint8_t  asn1::s1ap::map_enum_number<const uint8_t>(const uint8_t* array,
                                                                  uint32_t       nof_types,
                                                                  uint32_t       enum_val,
                                                                  const char*    enum_type);
template const uint16_t asn1::s1ap::map_enum_number<const uint16_t>(const uint16_t* array,
                                                                    uint32_t        nof_types,
                                                                    uint32_t        enum_val,
                                                                    const char*     enum_type);
template const uint32_t asn1::s1ap::map_enum_number<const uint32_t>(const uint32_t* array,
                                                                    uint32_t        nof_types,
                                                                    uint32_t        enum_val,
                                                                    const char*     enum_type);
template const uint64_t asn1::s1ap::map_enum_number<const uint64_t>(const uint64_t* array,
                                                                    uint32_t        nof_types,
                                                                    uint32_t        enum_val,
                                                                    const char*     enum_type);
template const int8_t   asn1::s1ap::map_enum_number<const int8_t>(const int8_t* array,
                                                                uint32_t      nof_types,
                                                                uint32_t      enum_val,
                                                                const char*   enum_type);
template const int16_t  asn1::s1ap::map_enum_number<const int16_t>(const int16_t* array,
                                                                  uint32_t       nof_types,
                                                                  uint32_t       enum_val,
                                                                  const char*    enum_type);
template const int32_t  asn1::s1ap::map_enum_number<const int32_t>(const int32_t* array,
                                                                  uint32_t       nof_types,
                                                                  uint32_t       enum_val,
                                                                  const char*    enum_type);
template const int64_t  asn1::s1ap::map_enum_number<const int64_t>(const int64_t* array,
                                                                  uint32_t       nof_types,
                                                                  uint32_t       enum_val,
                                                                  const char*    enum_type);
template const float    asn1::s1ap::map_enum_number<const float>(const float* array,
                                                              uint32_t     nof_types,
                                                              uint32_t     enum_val,
                                                              const char*  enum_type);

void s1ap_asn1_warn_assert(bool cond, const char* filename, int lineno)
{
  if (cond) {
    asn1::log_warning("Assertion in [%s][%d] failed.\n", filename, lineno);
  }
}
static void log_invalid_choice_id(uint32_t val, const char* choice_type)
{
  asn1::log_error("Invalid choice id=%d for choice type %s\n", val, choice_type);
}

static void invalid_enum_number(int value, const char* name)
{
  asn1::log_error("The provided enum value=%d of type %s cannot be translated into a number\n", value, name);
}

/*******************************************************************************
 *                                Struct Methods
 ******************************************************************************/

// Criticality ::= ENUMERATED
std::string crit_opts::to_string() const
{
  static const char* options[] = {"reject", "ignore", "notify"};
  return convert_enum_idx(options, 3, value, "crit_e");
}

// Presence ::= ENUMERATED
std::string presence_opts::to_string() const
{
  static const char* options[] = {"optional", "conditional", "mandatory"};
  return convert_enum_idx(options, 3, value, "presence_e");
}

// PrivateIE-ID ::= CHOICE
void private_ie_id_c::destroy_() {}
void private_ie_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
private_ie_id_c::private_ie_id_c(const private_ie_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::local:
      c.init(other.c.get<uint32_t>());
      break;
    case types::global:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
  }
}
private_ie_id_c& private_ie_id_c::operator=(const private_ie_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::local:
      c.set(other.c.get<uint32_t>());
      break;
    case types::global:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
  }

  return *this;
}
void private_ie_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::local:
      j.write_int("local", c.get<uint32_t>());
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
  }
  j.end_obj();
}
SRSASN_CODE private_ie_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::local:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE private_ie_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::local:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string private_ie_id_c::types_opts::to_string() const
{
  static const char* options[] = {"local", "global"};
  return convert_enum_idx(options, 2, value, "private_ie_id_c::types");
}

// PrivateIE-Field{S1AP-PRIVATE-IES : IEsSetParam} ::= SEQUENCE{{S1AP-PRIVATE-IES}}
template <class ies_set_paramT_>
SRSASN_CODE private_ie_field_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(id.pack(bref));
  HANDLE_CODE(crit.pack(bref));
  HANDLE_CODE(value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE private_ie_field_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(id.unpack(bref));
  HANDLE_CODE(crit.unpack(bref));
  HANDLE_CODE(value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void private_ie_field_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("id");
  id.to_json(j);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}

// ProtocolExtensionField{S1AP-PROTOCOL-EXTENSION : ExtensionSetParam} ::= SEQUENCE{{S1AP-PROTOCOL-EXTENSION}}
template <class ext_set_paramT_>
SRSASN_CODE protocol_ext_field_s<ext_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  s1ap_asn1_warn_assert(crit != ext_set_paramT_::get_crit(id), __func__, __LINE__);
  HANDLE_CODE(crit.pack(bref));
  HANDLE_CODE(ext_value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ext_set_paramT_>
SRSASN_CODE protocol_ext_field_s<ext_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.unpack(bref));
  ext_value = ext_set_paramT_::get_ext(id);
  HANDLE_CODE(ext_value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ext_set_paramT_>
void protocol_ext_field_s<ext_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}
template <class ext_set_paramT_>
bool protocol_ext_field_s<ext_set_paramT_>::load_info_obj(const uint32_t& id_)
{
  if (not ext_set_paramT_::is_id_valid(id_)) {
    return false;
  }
  id        = id_;
  crit      = ext_set_paramT_::get_crit(id);
  ext_value = ext_set_paramT_::get_ext(id);
  return ext_value.type().value != ext_set_paramT_::ext_c::types_opts::nulltype;
}

// ProtocolIE-Field{S1AP-PROTOCOL-IES : IEsSetParam} ::= SEQUENCE{{S1AP-PROTOCOL-IES}}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_field_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  s1ap_asn1_warn_assert(crit != ies_set_paramT_::get_crit(id), __func__, __LINE__);
  HANDLE_CODE(crit.pack(bref));
  HANDLE_CODE(value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_field_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.unpack(bref));
  value = ies_set_paramT_::get_value(id);
  HANDLE_CODE(value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void protocol_ie_field_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}
template <class ies_set_paramT_>
bool protocol_ie_field_s<ies_set_paramT_>::load_info_obj(const uint32_t& id_)
{
  if (not ies_set_paramT_::is_id_valid(id_)) {
    return false;
  }
  id    = id_;
  crit  = ies_set_paramT_::get_crit(id);
  value = ies_set_paramT_::get_value(id);
  return value.type().value != ies_set_paramT_::value_c::types_opts::nulltype;
}

// ProtocolIE-SingleContainer{S1AP-PROTOCOL-IES : IEsSetParam} ::= SEQUENCE{{S1AP-PROTOCOL-IES}}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_single_container_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  s1ap_asn1_warn_assert(crit != ies_set_paramT_::get_crit(id), __func__, __LINE__);
  HANDLE_CODE(crit.pack(bref));
  HANDLE_CODE(value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_single_container_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.unpack(bref));
  value = ies_set_paramT_::get_value(id);
  HANDLE_CODE(value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void protocol_ie_single_container_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}
template <class ies_set_paramT_>
bool protocol_ie_single_container_s<ies_set_paramT_>::load_info_obj(const uint32_t& id_)
{
  if (not ies_set_paramT_::is_id_valid(id_)) {
    return false;
  }
  id    = id_;
  crit  = ies_set_paramT_::get_crit(id);
  value = ies_set_paramT_::get_value(id);
  return value.type().value != ies_set_paramT_::value_c::types_opts::nulltype;
}

// ProtocolIE-FieldPair{S1AP-PROTOCOL-IES-PAIR : IEsSetParam} ::= SEQUENCE{{S1AP-PROTOCOL-IES-PAIR}}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_field_pair_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  s1ap_asn1_warn_assert(first_crit != ies_set_paramT_::get_first_crit(id), __func__, __LINE__);
  HANDLE_CODE(first_crit.pack(bref));
  HANDLE_CODE(first_value.pack(bref));
  s1ap_asn1_warn_assert(second_crit != ies_set_paramT_::get_second_crit(id), __func__, __LINE__);
  HANDLE_CODE(second_crit.pack(bref));
  HANDLE_CODE(second_value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE protocol_ie_field_pair_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(first_crit.unpack(bref));
  first_value = ies_set_paramT_::get_first_value(id);
  HANDLE_CODE(first_value.unpack(bref));
  HANDLE_CODE(second_crit.unpack(bref));
  second_value = ies_set_paramT_::get_second_value(id);
  HANDLE_CODE(second_value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void protocol_ie_field_pair_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("firstCriticality", first_crit.to_string());
  j.write_str("secondCriticality", second_crit.to_string());
  j.end_obj();
}
template <class ies_set_paramT_>
bool protocol_ie_field_pair_s<ies_set_paramT_>::load_info_obj(const uint32_t& id_)
{
  if (not ies_set_paramT_::is_id_valid(id_)) {
    return false;
  }
  id           = id_;
  first_crit   = ies_set_paramT_::get_first_crit(id);
  first_value  = ies_set_paramT_::get_first_value(id);
  second_crit  = ies_set_paramT_::get_second_crit(id);
  second_value = ies_set_paramT_::get_second_value(id);
  return first_value.type().value != ies_set_paramT_::first_value_c::types_opts::nulltype and
         second_value.type().value != ies_set_paramT_::second_value_c::types_opts::nulltype;
}

// ActivatedCellsList-Item ::= SEQUENCE
SRSASN_CODE activ_cells_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE activ_cells_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));

  return SRSASN_SUCCESS;
}
void activ_cells_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.end_obj();
}

uint32_t s1ap_protocol_ext_empty_o::idx_to_id(uint32_t idx)
{
  asn1::log_error("object set is empty\n");
  return 0;
}
bool s1ap_protocol_ext_empty_o::is_id_valid(const uint32_t& id)
{
  asn1::log_error("object set is empty\n");
  return false;
}
crit_e s1ap_protocol_ext_empty_o::get_crit(const uint32_t& id)
{
  return {};
}
s1ap_protocol_ext_empty_o::ext_c s1ap_protocol_ext_empty_o::get_ext(const uint32_t& id)
{
  return {};
}
presence_e s1ap_protocol_ext_empty_o::get_presence(const uint32_t& id)
{
  return {};
}

// Extension ::= OPEN TYPE
void s1ap_protocol_ext_empty_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.end_obj();
}
SRSASN_CODE s1ap_protocol_ext_empty_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  return SRSASN_SUCCESS;
}
SRSASN_CODE s1ap_protocol_ext_empty_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  return SRSASN_SUCCESS;
}

std::string s1ap_protocol_ext_empty_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {};
  return convert_enum_idx(options, 0, value, "s1ap_protocol_ext_empty_o::ext_c::types");
}

template <class extT_>
protocol_ext_container_item_s<extT_>::protocol_ext_container_item_s(uint32_t id_, crit_e crit_) : id(id_), crit(crit_)

{}
template <class extT_>
SRSASN_CODE protocol_ext_container_item_s<extT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.pack(bref));
  {
    varlength_field_pack_guard varlen_scope(bref, true);
    HANDLE_CODE(ext.pack(bref));
  }
  return SRSASN_SUCCESS;
}
template <class extT_>
SRSASN_CODE protocol_ext_container_item_s<extT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.unpack(bref));
  {
    varlength_field_unpack_guard varlen_scope(bref, true);
    HANDLE_CODE(ext.unpack(bref));
  }
  return SRSASN_SUCCESS;
}
template <class extT_>
void protocol_ext_container_item_s<extT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}

SRSASN_CODE protocol_ext_container_empty_l::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  return SRSASN_SUCCESS;
}
SRSASN_CODE protocol_ext_container_empty_l::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);
  if (nof_ies > 0) {
    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void protocol_ext_container_empty_l::to_json(json_writer& j) const
{
  j.start_obj();
  j.end_obj();
}

// GUMMEI ::= SEQUENCE
SRSASN_CODE gummei_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(mme_group_id.pack(bref));
  HANDLE_CODE(mme_code.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gummei_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(mme_group_id.unpack(bref));
  HANDLE_CODE(mme_code.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gummei_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.write_str("mME-Group-ID", mme_group_id.to_string());
  j.write_str("mME-Code", mme_code.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Additional-GUTI ::= SEQUENCE
SRSASN_CODE add_guti_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(gummei.pack(bref));
  HANDLE_CODE(m_tmsi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE add_guti_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(gummei.unpack(bref));
  HANDLE_CODE(m_tmsi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void add_guti_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("gUMMEI");
  gummei.to_json(j);
  j.write_str("m-TMSI", m_tmsi.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Pre-emptionCapability ::= ENUMERATED
std::string pre_emption_cap_opts::to_string() const
{
  static const char* options[] = {"shall-not-trigger-pre-emption", "may-trigger-pre-emption"};
  return convert_enum_idx(options, 2, value, "pre_emption_cap_e");
}

// Pre-emptionVulnerability ::= ENUMERATED
std::string pre_emption_vulnerability_opts::to_string() const
{
  static const char* options[] = {"not-pre-emptable", "pre-emptable"};
  return convert_enum_idx(options, 2, value, "pre_emption_vulnerability_e");
}

// AllocationAndRetentionPriority ::= SEQUENCE
SRSASN_CODE alloc_and_retention_prio_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, prio_level, (uint8_t)0u, (uint8_t)15u, false, true));
  HANDLE_CODE(pre_emption_cap.pack(bref));
  HANDLE_CODE(pre_emption_vulnerability.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE alloc_and_retention_prio_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(prio_level, bref, (uint8_t)0u, (uint8_t)15u, false, true));
  HANDLE_CODE(pre_emption_cap.unpack(bref));
  HANDLE_CODE(pre_emption_vulnerability.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void alloc_and_retention_prio_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("priorityLevel", prio_level);
  j.write_str("pre-emptionCapability", pre_emption_cap.to_string());
  j.write_str("pre-emptionVulnerability", pre_emption_vulnerability.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EUTRAN-CGI ::= SEQUENCE
SRSASN_CODE eutran_cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(cell_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutran_cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(cell_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void eutran_cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("cell-ID", cell_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAI ::= SEQUENCE
SRSASN_CODE tai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(tac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(tac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("tAC", tac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellBasedMDT ::= SEQUENCE
SRSASN_CODE cell_based_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, cell_id_listfor_mdt, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_based_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(cell_id_listfor_mdt, bref, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_based_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("cellIdListforMDT");
  for (const auto& e1 : cell_id_listfor_mdt) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TABasedMDT ::= SEQUENCE
SRSASN_CODE ta_based_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, talistfor_mdt, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ta_based_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(talistfor_mdt, bref, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ta_based_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("tAListforMDT");
  for (const auto& e1 : talistfor_mdt) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIBasedMDT ::= SEQUENCE
SRSASN_CODE tai_based_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, tai_listfor_mdt, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_based_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(tai_listfor_mdt, bref, 1, 8, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_based_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("tAIListforMDT");
  for (const auto& e1 : tai_listfor_mdt) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaScopeOfMDT ::= CHOICE
void area_scope_of_mdt_c::destroy_()
{
  switch (type_) {
    case types::cell_based:
      c.destroy<cell_based_mdt_s>();
      break;
    case types::tabased:
      c.destroy<ta_based_mdt_s>();
      break;
    case types::tai_based:
      c.destroy<tai_based_mdt_s>();
      break;
    default:
      break;
  }
}
void area_scope_of_mdt_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_based:
      c.init<cell_based_mdt_s>();
      break;
    case types::tabased:
      c.init<ta_based_mdt_s>();
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      c.init<tai_based_mdt_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }
}
area_scope_of_mdt_c::area_scope_of_mdt_c(const area_scope_of_mdt_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_based:
      c.init(other.c.get<cell_based_mdt_s>());
      break;
    case types::tabased:
      c.init(other.c.get<ta_based_mdt_s>());
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      c.init(other.c.get<tai_based_mdt_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }
}
area_scope_of_mdt_c& area_scope_of_mdt_c::operator=(const area_scope_of_mdt_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_based:
      c.set(other.c.get<cell_based_mdt_s>());
      break;
    case types::tabased:
      c.set(other.c.get<ta_based_mdt_s>());
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      c.set(other.c.get<tai_based_mdt_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }

  return *this;
}
void area_scope_of_mdt_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_based:
      j.write_fieldname("cellBased");
      c.get<cell_based_mdt_s>().to_json(j);
      break;
    case types::tabased:
      j.write_fieldname("tABased");
      c.get<ta_based_mdt_s>().to_json(j);
      break;
    case types::plmn_wide:
      break;
    case types::tai_based:
      j.write_fieldname("tAIBased");
      c.get<tai_based_mdt_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
  }
  j.end_obj();
}
SRSASN_CODE area_scope_of_mdt_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_based:
      HANDLE_CODE(c.get<cell_based_mdt_s>().pack(bref));
      break;
    case types::tabased:
      HANDLE_CODE(c.get<ta_based_mdt_s>().pack(bref));
      break;
    case types::plmn_wide:
      break;
    case types::tai_based: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<tai_based_mdt_s>().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE area_scope_of_mdt_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_based:
      HANDLE_CODE(c.get<cell_based_mdt_s>().unpack(bref));
      break;
    case types::tabased:
      HANDLE_CODE(c.get<ta_based_mdt_s>().unpack(bref));
      break;
    case types::plmn_wide:
      break;
    case types::tai_based: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<tai_based_mdt_s>().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "area_scope_of_mdt_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string area_scope_of_mdt_c::types_opts::to_string() const
{
  static const char* options[] = {"cellBased", "tABased", "pLMNWide", "tAIBased"};
  return convert_enum_idx(options, 4, value, "area_scope_of_mdt_c::types");
}

// CellIdentifierAndCELevelForCECapableUEs ::= SEQUENCE
SRSASN_CODE cell_id_and_ce_level_for_ce_capable_ues_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_cell_id.pack(bref));
  HANDLE_CODE(celevel.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_and_ce_level_for_ce_capable_ues_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_cell_id.unpack(bref));
  HANDLE_CODE(celevel.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_and_ce_level_for_ce_capable_ues_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-Cell-ID");
  global_cell_id.to_json(j);
  j.write_str("cELevel", celevel.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForCECapableUEs ::= SEQUENCE
SRSASN_CODE assist_data_for_ce_capable_ues_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cell_id_and_ce_level_for_ce_capable_ues.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_ce_capable_ues_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cell_id_and_ce_level_for_ce_capable_ues.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_ce_capable_ues_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellIdentifierAndCELevelForCECapableUEs");
  cell_id_and_ce_level_for_ce_capable_ues.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedCellItem ::= SEQUENCE
SRSASN_CODE recommended_cell_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.pack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(pack_integer(bref, time_stayed_in_cell, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cell_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.unpack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(unpack_integer(time_stayed_in_cell, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_cell_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRAN-CGI");
  eutran_cgi.to_json(j);
  if (time_stayed_in_cell_present) {
    j.write_int("timeStayedInCell", time_stayed_in_cell);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedCellItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t recommended_cell_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {214};
  return map_enum_number(options, 1, idx, "id");
}
bool recommended_cell_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 214 == id;
}
crit_e recommended_cell_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 214) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
recommended_cell_item_ies_o::value_c recommended_cell_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 214) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e recommended_cell_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 214) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void recommended_cell_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("RecommendedCellItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE recommended_cell_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cell_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string recommended_cell_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"RecommendedCellItem"};
  return convert_enum_idx(options, 1, value, "recommended_cell_item_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<recommended_cell_item_ies_o>;

// NextPagingAreaScope ::= ENUMERATED
std::string next_paging_area_scope_opts::to_string() const
{
  static const char* options[] = {"same", "changed"};
  return convert_enum_idx(options, 2, value, "next_paging_area_scope_e");
}

// RecommendedCellsForPaging ::= SEQUENCE
SRSASN_CODE recommended_cells_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, recommended_cell_list, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cells_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(recommended_cell_list, bref, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_cells_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("recommendedCellList");
  for (const auto& e1 : recommended_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForRecommendedCells ::= SEQUENCE
SRSASN_CODE assist_data_for_recommended_cells_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_recommended_cells_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_recommended_cells_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("recommendedCellsForPaging");
  recommended_cells_for_paging.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PagingAttemptInformation ::= SEQUENCE
SRSASN_CODE paging_attempt_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(next_paging_area_scope_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, paging_attempt_count, (uint8_t)1u, (uint8_t)16u, true, true));
  HANDLE_CODE(pack_integer(bref, intended_nof_paging_attempts, (uint8_t)1u, (uint8_t)16u, true, true));
  if (next_paging_area_scope_present) {
    HANDLE_CODE(next_paging_area_scope.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE paging_attempt_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(next_paging_area_scope_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(paging_attempt_count, bref, (uint8_t)1u, (uint8_t)16u, true, true));
  HANDLE_CODE(unpack_integer(intended_nof_paging_attempts, bref, (uint8_t)1u, (uint8_t)16u, true, true));
  if (next_paging_area_scope_present) {
    HANDLE_CODE(next_paging_area_scope.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void paging_attempt_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pagingAttemptCount", paging_attempt_count);
  j.write_int("intendedNumberOfPagingAttempts", intended_nof_paging_attempts);
  if (next_paging_area_scope_present) {
    j.write_str("nextPagingAreaScope", next_paging_area_scope.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForPaging ::= SEQUENCE
SRSASN_CODE assist_data_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(assist_data_for_recommended_cells_present, 1));
  HANDLE_CODE(bref.pack(assist_data_for_ce_capable_ues_present, 1));
  HANDLE_CODE(bref.pack(paging_attempt_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (assist_data_for_recommended_cells_present) {
    HANDLE_CODE(assist_data_for_recommended_cells.pack(bref));
  }
  if (assist_data_for_ce_capable_ues_present) {
    HANDLE_CODE(assist_data_for_ce_capable_ues.pack(bref));
  }
  if (paging_attempt_info_present) {
    HANDLE_CODE(paging_attempt_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(assist_data_for_recommended_cells_present, 1));
  HANDLE_CODE(bref.unpack(assist_data_for_ce_capable_ues_present, 1));
  HANDLE_CODE(bref.unpack(paging_attempt_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (assist_data_for_recommended_cells_present) {
    HANDLE_CODE(assist_data_for_recommended_cells.unpack(bref));
  }
  if (assist_data_for_ce_capable_ues_present) {
    HANDLE_CODE(assist_data_for_ce_capable_ues.unpack(bref));
  }
  if (paging_attempt_info_present) {
    HANDLE_CODE(paging_attempt_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (assist_data_for_recommended_cells_present) {
    j.write_fieldname("assistanceDataForRecommendedCells");
    assist_data_for_recommended_cells.to_json(j);
  }
  if (assist_data_for_ce_capable_ues_present) {
    j.write_fieldname("assistanceDataForCECapableUEs");
    assist_data_for_ce_capable_ues.to_json(j);
  }
  if (paging_attempt_info_present) {
    j.write_fieldname("pagingAttemptInformation");
    paging_attempt_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// COUNTValueExtended ::= SEQUENCE
SRSASN_CODE count_value_extended_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_sn_extended, (uint16_t)0u, (uint16_t)32767u, false, true));
  HANDLE_CODE(pack_integer(bref, hfn_modified, (uint32_t)0u, (uint32_t)131071u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE count_value_extended_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_sn_extended, bref, (uint16_t)0u, (uint16_t)32767u, false, true));
  HANDLE_CODE(unpack_integer(hfn_modified, bref, (uint32_t)0u, (uint32_t)131071u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void count_value_extended_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SNExtended", pdcp_sn_extended);
  j.write_int("hFNModified", hfn_modified);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// COUNTvaluePDCP-SNlength18 ::= SEQUENCE
SRSASN_CODE coun_tvalue_pdcp_snlen18_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_snlen18, (uint32_t)0u, (uint32_t)262143u, false, true));
  HANDLE_CODE(pack_integer(bref, hfnfor_pdcp_snlen18, (uint16_t)0u, (uint16_t)16383u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE coun_tvalue_pdcp_snlen18_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_snlen18, bref, (uint32_t)0u, (uint32_t)262143u, false, true));
  HANDLE_CODE(unpack_integer(hfnfor_pdcp_snlen18, bref, (uint16_t)0u, (uint16_t)16383u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void coun_tvalue_pdcp_snlen18_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SNlength18", pdcp_snlen18);
  j.write_int("hFNforPDCP-SNlength18", hfnfor_pdcp_snlen18);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Bearers-SubjectToStatusTransfer-ItemExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t bearers_subject_to_status_transfer_item_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {179, 180, 181, 217, 218, 219};
  return map_enum_number(options, 6, idx, "id");
}
bool bearers_subject_to_status_transfer_item_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {179, 180, 181, 217, 218, 219};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e bearers_subject_to_status_transfer_item_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 179:
      return crit_e::ignore;
    case 180:
      return crit_e::ignore;
    case 181:
      return crit_e::ignore;
    case 217:
      return crit_e::ignore;
    case 218:
      return crit_e::ignore;
    case 219:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c
bearers_subject_to_status_transfer_item_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 179:
      ret.set(ext_c::types::ulcount_value_extended);
      break;
    case 180:
      ret.set(ext_c::types::dlcount_value_extended);
      break;
    case 181:
      ret.set(ext_c::types::receive_status_of_ulpdcpsdus_extended);
      break;
    case 217:
      ret.set(ext_c::types::ulcount_value_pdcp_snlen18);
      break;
    case 218:
      ret.set(ext_c::types::dlcount_value_pdcp_snlen18);
      break;
    case 219:
      ret.set(ext_c::types::receive_status_of_ulpdcpsdus_pdcp_snlen18);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e bearers_subject_to_status_transfer_item_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 179:
      return presence_e::optional;
    case 180:
      return presence_e::optional;
    case 181:
      return presence_e::optional;
    case 217:
      return presence_e::optional;
    case 218:
      return presence_e::optional;
    case 219:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_extended()
{
  assert_choice_type("COUNTValueExtended", type_.to_string(), "Extension");
  return c.get<count_value_extended_s>();
}
count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_extended()
{
  assert_choice_type("COUNTValueExtended", type_.to_string(), "Extension");
  return c.get<count_value_extended_s>();
}
bounded_bitstring<1, 16384, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_extended()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Extension");
  return c.get<bounded_bitstring<1, 16384, false, true> >();
}
coun_tvalue_pdcp_snlen18_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_pdcp_snlen18()
{
  assert_choice_type("COUNTvaluePDCP-SNlength18", type_.to_string(), "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
coun_tvalue_pdcp_snlen18_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_pdcp_snlen18()
{
  assert_choice_type("COUNTvaluePDCP-SNlength18", type_.to_string(), "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
bounded_bitstring<1, 131072, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_pdcp_snlen18()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Extension");
  return c.get<bounded_bitstring<1, 131072, false, true> >();
}
const count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_extended() const
{
  assert_choice_type("COUNTValueExtended", type_.to_string(), "Extension");
  return c.get<count_value_extended_s>();
}
const count_value_extended_s& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_extended() const
{
  assert_choice_type("COUNTValueExtended", type_.to_string(), "Extension");
  return c.get<count_value_extended_s>();
}
const bounded_bitstring<1, 16384, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_extended() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Extension");
  return c.get<bounded_bitstring<1, 16384, false, true> >();
}
const coun_tvalue_pdcp_snlen18_s&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ulcount_value_pdcp_snlen18() const
{
  assert_choice_type("COUNTvaluePDCP-SNlength18", type_.to_string(), "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
const coun_tvalue_pdcp_snlen18_s&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::dlcount_value_pdcp_snlen18() const
{
  assert_choice_type("COUNTvaluePDCP-SNlength18", type_.to_string(), "Extension");
  return c.get<coun_tvalue_pdcp_snlen18_s>();
}
const bounded_bitstring<1, 131072, false, true>&
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::receive_status_of_ulpdcpsdus_pdcp_snlen18() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Extension");
  return c.get<bounded_bitstring<1, 131072, false, true> >();
}
void bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::ulcount_value_extended:
      c.destroy<count_value_extended_s>();
      break;
    case types::dlcount_value_extended:
      c.destroy<count_value_extended_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.destroy<bounded_bitstring<1, 16384, false, true> >();
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.destroy<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.destroy<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.destroy<bounded_bitstring<1, 131072, false, true> >();
      break;
    default:
      break;
  }
}
void bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ulcount_value_extended:
      c.init<count_value_extended_s>();
      break;
    case types::dlcount_value_extended:
      c.init<count_value_extended_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.init<bounded_bitstring<1, 16384, false, true> >();
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.init<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.init<coun_tvalue_pdcp_snlen18_s>();
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.init<bounded_bitstring<1, 131072, false, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }
}
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::ext_c(
    const bearers_subject_to_status_transfer_item_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ulcount_value_extended:
      c.init(other.c.get<count_value_extended_s>());
      break;
    case types::dlcount_value_extended:
      c.init(other.c.get<count_value_extended_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.init(other.c.get<bounded_bitstring<1, 16384, false, true> >());
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.init(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.init(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.init(other.c.get<bounded_bitstring<1, 131072, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }
}
bearers_subject_to_status_transfer_item_ext_ies_o::ext_c& bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::
                                                          operator=(const bearers_subject_to_status_transfer_item_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ulcount_value_extended:
      c.set(other.c.get<count_value_extended_s>());
      break;
    case types::dlcount_value_extended:
      c.set(other.c.get<count_value_extended_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      c.set(other.c.get<bounded_bitstring<1, 16384, false, true> >());
      break;
    case types::ulcount_value_pdcp_snlen18:
      c.set(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::dlcount_value_pdcp_snlen18:
      c.set(other.c.get<coun_tvalue_pdcp_snlen18_s>());
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      c.set(other.c.get<bounded_bitstring<1, 131072, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }

  return *this;
}
void bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ulcount_value_extended:
      j.write_fieldname("COUNTValueExtended");
      c.get<count_value_extended_s>().to_json(j);
      break;
    case types::dlcount_value_extended:
      j.write_fieldname("COUNTValueExtended");
      c.get<count_value_extended_s>().to_json(j);
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 16384, false, true> >().to_string());
      break;
    case types::ulcount_value_pdcp_snlen18:
      j.write_fieldname("COUNTvaluePDCP-SNlength18");
      c.get<coun_tvalue_pdcp_snlen18_s>().to_json(j);
      break;
    case types::dlcount_value_pdcp_snlen18:
      j.write_fieldname("COUNTvaluePDCP-SNlength18");
      c.get<coun_tvalue_pdcp_snlen18_s>().to_json(j);
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 131072, false, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ulcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().pack(bref));
      break;
    case types::dlcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().pack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      HANDLE_CODE((c.get<bounded_bitstring<1, 16384, false, true> >().pack(bref)));
      break;
    case types::ulcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().pack(bref));
      break;
    case types::dlcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().pack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      HANDLE_CODE((c.get<bounded_bitstring<1, 131072, false, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ulcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().unpack(bref));
      break;
    case types::dlcount_value_extended:
      HANDLE_CODE(c.get<count_value_extended_s>().unpack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_extended:
      HANDLE_CODE((c.get<bounded_bitstring<1, 16384, false, true> >().unpack(bref)));
      break;
    case types::ulcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().unpack(bref));
      break;
    case types::dlcount_value_pdcp_snlen18:
      HANDLE_CODE(c.get<coun_tvalue_pdcp_snlen18_s>().unpack(bref));
      break;
    case types::receive_status_of_ulpdcpsdus_pdcp_snlen18:
      HANDLE_CODE((c.get<bounded_bitstring<1, 131072, false, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"COUNTValueExtended",
                                  "COUNTValueExtended",
                                  "BIT STRING",
                                  "COUNTvaluePDCP-SNlength18",
                                  "COUNTvaluePDCP-SNlength18",
                                  "BIT STRING"};
  return convert_enum_idx(options, 6, value, "bearers_subject_to_status_transfer_item_ext_ies_o::ext_c::types");
}

// COUNTvalue ::= SEQUENCE
SRSASN_CODE coun_tvalue_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_sn, (uint16_t)0u, (uint16_t)4095u, false, true));
  HANDLE_CODE(pack_integer(bref, hfn, (uint32_t)0u, (uint32_t)1048575u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE coun_tvalue_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_sn, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  HANDLE_CODE(unpack_integer(hfn, bref, (uint32_t)0u, (uint32_t)1048575u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void coun_tvalue_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SN", pdcp_sn);
  j.write_int("hFN", hfn);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::s1ap::protocol_ext_field_s<bearers_subject_to_status_transfer_item_ext_ies_o>;

bearers_subject_to_status_transfer_item_ext_ies_container::bearers_subject_to_status_transfer_item_ext_ies_container() :
  ulcount_value_extended(179, crit_e::ignore),
  dlcount_value_extended(180, crit_e::ignore),
  receive_status_of_ulpdcpsdus_extended(181, crit_e::ignore),
  ulcount_value_pdcp_snlen18(217, crit_e::ignore),
  dlcount_value_pdcp_snlen18(218, crit_e::ignore),
  receive_status_of_ulpdcpsdus_pdcp_snlen18(219, crit_e::ignore)
{}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += ulcount_value_extended_present ? 1 : 0;
  nof_ies += dlcount_value_extended_present ? 1 : 0;
  nof_ies += receive_status_of_ulpdcpsdus_extended_present ? 1 : 0;
  nof_ies += ulcount_value_pdcp_snlen18_present ? 1 : 0;
  nof_ies += dlcount_value_pdcp_snlen18_present ? 1 : 0;
  nof_ies += receive_status_of_ulpdcpsdus_pdcp_snlen18_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (ulcount_value_extended_present) {
    HANDLE_CODE(ulcount_value_extended.pack(bref));
  }
  if (dlcount_value_extended_present) {
    HANDLE_CODE(dlcount_value_extended.pack(bref));
  }
  if (receive_status_of_ulpdcpsdus_extended_present) {
    HANDLE_CODE(receive_status_of_ulpdcpsdus_extended.pack(bref));
  }
  if (ulcount_value_pdcp_snlen18_present) {
    HANDLE_CODE(ulcount_value_pdcp_snlen18.pack(bref));
  }
  if (dlcount_value_pdcp_snlen18_present) {
    HANDLE_CODE(dlcount_value_pdcp_snlen18.pack(bref));
  }
  if (receive_status_of_ulpdcpsdus_pdcp_snlen18_present) {
    HANDLE_CODE(receive_status_of_ulpdcpsdus_pdcp_snlen18.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<bearers_subject_to_status_transfer_item_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 179:
        ulcount_value_extended_present = true;
        ulcount_value_extended.id      = c.id;
        ulcount_value_extended.crit    = c.crit;
        ulcount_value_extended.ext     = c.ext_value.ulcount_value_extended();
        break;
      case 180:
        dlcount_value_extended_present = true;
        dlcount_value_extended.id      = c.id;
        dlcount_value_extended.crit    = c.crit;
        dlcount_value_extended.ext     = c.ext_value.dlcount_value_extended();
        break;
      case 181:
        receive_status_of_ulpdcpsdus_extended_present = true;
        receive_status_of_ulpdcpsdus_extended.id      = c.id;
        receive_status_of_ulpdcpsdus_extended.crit    = c.crit;
        receive_status_of_ulpdcpsdus_extended.ext     = c.ext_value.receive_status_of_ulpdcpsdus_extended();
        break;
      case 217:
        ulcount_value_pdcp_snlen18_present = true;
        ulcount_value_pdcp_snlen18.id      = c.id;
        ulcount_value_pdcp_snlen18.crit    = c.crit;
        ulcount_value_pdcp_snlen18.ext     = c.ext_value.ulcount_value_pdcp_snlen18();
        break;
      case 218:
        dlcount_value_pdcp_snlen18_present = true;
        dlcount_value_pdcp_snlen18.id      = c.id;
        dlcount_value_pdcp_snlen18.crit    = c.crit;
        dlcount_value_pdcp_snlen18.ext     = c.ext_value.dlcount_value_pdcp_snlen18();
        break;
      case 219:
        receive_status_of_ulpdcpsdus_pdcp_snlen18_present = true;
        receive_status_of_ulpdcpsdus_pdcp_snlen18.id      = c.id;
        receive_status_of_ulpdcpsdus_pdcp_snlen18.crit    = c.crit;
        receive_status_of_ulpdcpsdus_pdcp_snlen18.ext     = c.ext_value.receive_status_of_ulpdcpsdus_pdcp_snlen18();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void bearers_subject_to_status_transfer_item_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (ulcount_value_extended_present) {
    j.write_fieldname("");
    ulcount_value_extended.to_json(j);
  }
  if (dlcount_value_extended_present) {
    j.write_fieldname("");
    dlcount_value_extended.to_json(j);
  }
  if (receive_status_of_ulpdcpsdus_extended_present) {
    j.write_fieldname("");
    receive_status_of_ulpdcpsdus_extended.to_json(j);
  }
  if (ulcount_value_pdcp_snlen18_present) {
    j.write_fieldname("");
    ulcount_value_pdcp_snlen18.to_json(j);
  }
  if (dlcount_value_pdcp_snlen18_present) {
    j.write_fieldname("");
    dlcount_value_pdcp_snlen18.to_json(j);
  }
  if (receive_status_of_ulpdcpsdus_pdcp_snlen18_present) {
    j.write_fieldname("");
    receive_status_of_ulpdcpsdus_pdcp_snlen18.to_json(j);
  }
  j.end_obj();
}

// Bearers-SubjectToStatusTransfer-Item ::= SEQUENCE
SRSASN_CODE bearers_subject_to_status_transfer_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(receive_statusof_ulpdcpsdus_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(ul_coun_tvalue.pack(bref));
  HANDLE_CODE(dl_coun_tvalue.pack(bref));
  if (receive_statusof_ulpdcpsdus_present) {
    HANDLE_CODE(receive_statusof_ulpdcpsdus.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(receive_statusof_ulpdcpsdus_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(ul_coun_tvalue.unpack(bref));
  HANDLE_CODE(dl_coun_tvalue.unpack(bref));
  if (receive_statusof_ulpdcpsdus_present) {
    HANDLE_CODE(receive_statusof_ulpdcpsdus.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void bearers_subject_to_status_transfer_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("uL-COUNTvalue");
  ul_coun_tvalue.to_json(j);
  j.write_fieldname("dL-COUNTvalue");
  dl_coun_tvalue.to_json(j);
  if (receive_statusof_ulpdcpsdus_present) {
    j.write_str("receiveStatusofULPDCPSDUs", receive_statusof_ulpdcpsdus.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Bearers-SubjectToStatusTransfer-ItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t bearers_subject_to_status_transfer_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {89};
  return map_enum_number(options, 1, idx, "id");
}
bool bearers_subject_to_status_transfer_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 89 == id;
}
crit_e bearers_subject_to_status_transfer_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 89) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
bearers_subject_to_status_transfer_item_ies_o::value_c
bearers_subject_to_status_transfer_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 89) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e bearers_subject_to_status_transfer_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 89) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void bearers_subject_to_status_transfer_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("Bearers-SubjectToStatusTransfer-Item");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE bearers_subject_to_status_transfer_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string bearers_subject_to_status_transfer_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Bearers-SubjectToStatusTransfer-Item"};
  return convert_enum_idx(options, 1, value, "bearers_subject_to_status_transfer_item_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<bearers_subject_to_status_transfer_item_ies_o>;

// CancelledCellinEAI-Item ::= SEQUENCE
SRSASN_CODE cancelled_cellin_eai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cellin_eai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cellin_eai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CancelledCellinTAI-Item ::= SEQUENCE
SRSASN_CODE cancelled_cellin_tai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cellin_tai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cellin_tai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellID-Cancelled-Item ::= SEQUENCE
SRSASN_CODE cell_id_cancelled_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_cancelled_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_cancelled_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaID-Cancelled-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_cancelled_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cellin_eai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_cancelled_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cellin_eai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_cancelled_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("cancelledCellinEAI");
  for (const auto& e1 : cancelled_cellin_eai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAI-Cancelled-Item ::= SEQUENCE
SRSASN_CODE tai_cancelled_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cellin_tai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_cancelled_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cellin_tai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_cancelled_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("cancelledCellinTAI");
  for (const auto& e1 : cancelled_cellin_tai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// BroadcastCancelledAreaList ::= CHOICE
void broadcast_cancelled_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_cancelled:
      c.destroy<cell_id_cancelled_l>();
      break;
    case types::tai_cancelled:
      c.destroy<tai_cancelled_l>();
      break;
    case types::emergency_area_id_cancelled:
      c.destroy<emergency_area_id_cancelled_l>();
      break;
    default:
      break;
  }
}
void broadcast_cancelled_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_cancelled:
      c.init<cell_id_cancelled_l>();
      break;
    case types::tai_cancelled:
      c.init<tai_cancelled_l>();
      break;
    case types::emergency_area_id_cancelled:
      c.init<emergency_area_id_cancelled_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
}
broadcast_cancelled_area_list_c::broadcast_cancelled_area_list_c(const broadcast_cancelled_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_cancelled:
      c.init(other.c.get<cell_id_cancelled_l>());
      break;
    case types::tai_cancelled:
      c.init(other.c.get<tai_cancelled_l>());
      break;
    case types::emergency_area_id_cancelled:
      c.init(other.c.get<emergency_area_id_cancelled_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
}
broadcast_cancelled_area_list_c& broadcast_cancelled_area_list_c::
                                 operator=(const broadcast_cancelled_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_cancelled:
      c.set(other.c.get<cell_id_cancelled_l>());
      break;
    case types::tai_cancelled:
      c.set(other.c.get<tai_cancelled_l>());
      break;
    case types::emergency_area_id_cancelled:
      c.set(other.c.get<emergency_area_id_cancelled_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }

  return *this;
}
void broadcast_cancelled_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_cancelled:
      j.start_array("cellID-Cancelled");
      for (const auto& e1 : c.get<cell_id_cancelled_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_cancelled:
      j.start_array("tAI-Cancelled");
      for (const auto& e1 : c.get<tai_cancelled_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_cancelled:
      j.start_array("emergencyAreaID-Cancelled");
      for (const auto& e1 : c.get<emergency_area_id_cancelled_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE broadcast_cancelled_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_cancelled:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_cancelled_l>(), 1, 65535, true));
      break;
    case types::tai_cancelled:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_cancelled_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_cancelled_l>(), 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_cancelled_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_cancelled:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_cancelled_l>(), bref, 1, 65535, true));
      break;
    case types::tai_cancelled:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_cancelled_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_cancelled_l>(), bref, 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string broadcast_cancelled_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"cellID-Cancelled", "tAI-Cancelled", "emergencyAreaID-Cancelled"};
  return convert_enum_idx(options, 3, value, "broadcast_cancelled_area_list_c::types");
}

// CompletedCellinEAI-Item ::= SEQUENCE
SRSASN_CODE completed_cellin_eai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cellin_eai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cellin_eai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CompletedCellinTAI-Item ::= SEQUENCE
SRSASN_CODE completed_cellin_tai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cellin_tai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cellin_tai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellID-Broadcast-Item ::= SEQUENCE
SRSASN_CODE cell_id_broadcast_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_broadcast_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ecgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_broadcast_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eCGI");
  ecgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaID-Broadcast-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_broadcast_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cellin_eai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_broadcast_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cellin_eai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_broadcast_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("completedCellinEAI");
  for (const auto& e1 : completed_cellin_eai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAI-Broadcast-Item ::= SEQUENCE
SRSASN_CODE tai_broadcast_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cellin_tai, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_broadcast_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cellin_tai, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_broadcast_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("completedCellinTAI");
  for (const auto& e1 : completed_cellin_tai) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// BroadcastCompletedAreaList ::= CHOICE
void broadcast_completed_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_broadcast:
      c.destroy<cell_id_broadcast_l>();
      break;
    case types::tai_broadcast:
      c.destroy<tai_broadcast_l>();
      break;
    case types::emergency_area_id_broadcast:
      c.destroy<emergency_area_id_broadcast_l>();
      break;
    default:
      break;
  }
}
void broadcast_completed_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_broadcast:
      c.init<cell_id_broadcast_l>();
      break;
    case types::tai_broadcast:
      c.init<tai_broadcast_l>();
      break;
    case types::emergency_area_id_broadcast:
      c.init<emergency_area_id_broadcast_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
}
broadcast_completed_area_list_c::broadcast_completed_area_list_c(const broadcast_completed_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_broadcast:
      c.init(other.c.get<cell_id_broadcast_l>());
      break;
    case types::tai_broadcast:
      c.init(other.c.get<tai_broadcast_l>());
      break;
    case types::emergency_area_id_broadcast:
      c.init(other.c.get<emergency_area_id_broadcast_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
}
broadcast_completed_area_list_c& broadcast_completed_area_list_c::
                                 operator=(const broadcast_completed_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_broadcast:
      c.set(other.c.get<cell_id_broadcast_l>());
      break;
    case types::tai_broadcast:
      c.set(other.c.get<tai_broadcast_l>());
      break;
    case types::emergency_area_id_broadcast:
      c.set(other.c.get<emergency_area_id_broadcast_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }

  return *this;
}
void broadcast_completed_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_broadcast:
      j.start_array("cellID-Broadcast");
      for (const auto& e1 : c.get<cell_id_broadcast_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_broadcast:
      j.start_array("tAI-Broadcast");
      for (const auto& e1 : c.get<tai_broadcast_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_broadcast:
      j.start_array("emergencyAreaID-Broadcast");
      for (const auto& e1 : c.get<emergency_area_id_broadcast_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE broadcast_completed_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_broadcast:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_broadcast_l>(), 1, 65535, true));
      break;
    case types::tai_broadcast:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_broadcast_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_broadcast_l>(), 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_completed_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_broadcast:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_broadcast_l>(), bref, 1, 65535, true));
      break;
    case types::tai_broadcast:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_broadcast_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_broadcast_l>(), bref, 1, 65535, true));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string broadcast_completed_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"cellID-Broadcast", "tAI-Broadcast", "emergencyAreaID-Broadcast"};
  return convert_enum_idx(options, 3, value, "broadcast_completed_area_list_c::types");
}

// CGI ::= SEQUENCE
SRSASN_CODE cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(rac_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(lac.pack(bref));
  HANDLE_CODE(ci.pack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(rac_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(lac.unpack(bref));
  HANDLE_CODE(ci.unpack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("lAC", lac.to_string());
  j.write_str("cI", ci.to_string());
  if (rac_present) {
    j.write_str("rAC", rac.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CSG-IdList-Item ::= SEQUENCE
SRSASN_CODE csg_id_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(csg_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csg_id_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(csg_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csg_id_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cSG-Id", csg_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CSGMembershipStatus ::= ENUMERATED
std::string csg_membership_status_opts::to_string() const
{
  static const char* options[] = {"member", "not-member"};
  return convert_enum_idx(options, 2, value, "csg_membership_status_e");
}

// CellAccessMode ::= ENUMERATED
std::string cell_access_mode_opts::to_string() const
{
  static const char* options[] = {"hybrid"};
  return convert_enum_idx(options, 1, value, "cell_access_mode_e");
}

// CSGMembershipInfo ::= SEQUENCE
SRSASN_CODE csg_membership_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(cell_access_mode_present, 1));
  HANDLE_CODE(bref.pack(plm_nid_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(csg_membership_status.pack(bref));
  HANDLE_CODE(csg_id.pack(bref));
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (plm_nid_present) {
    HANDLE_CODE(plm_nid.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE csg_membership_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(cell_access_mode_present, 1));
  HANDLE_CODE(bref.unpack(plm_nid_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(csg_membership_status.unpack(bref));
  HANDLE_CODE(csg_id.unpack(bref));
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.unpack(bref));
  }
  if (plm_nid_present) {
    HANDLE_CODE(plm_nid.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void csg_membership_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cSGMembershipStatus", csg_membership_status.to_string());
  j.write_str("cSG-Id", csg_id.to_string());
  if (cell_access_mode_present) {
    j.write_str("cellAccessMode", "hybrid");
  }
  if (plm_nid_present) {
    j.write_str("pLMNidentity", plm_nid.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// IRAT-Cell-ID ::= CHOICE
void irat_cell_id_c::destroy_()
{
  switch (type_) {
    case types::eutran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.destroy<fixed_octstring<16, true> >();
      break;
    default:
      break;
  }
}
void irat_cell_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.init<fixed_octstring<16, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }
}
irat_cell_id_c::irat_cell_id_c(const irat_cell_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.init(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }
}
irat_cell_id_c& irat_cell_id_c::operator=(const irat_cell_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.set(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }

  return *this;
}
void irat_cell_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutran:
      j.write_str("eUTRAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::utran:
      j.write_str("uTRAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::geran:
      j.write_str("gERAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ehrpd:
      j.write_str("eHRPD", c.get<fixed_octstring<16, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
  }
  j.end_obj();
}
SRSASN_CODE irat_cell_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ehrpd: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().pack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE irat_cell_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ehrpd: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().unpack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "irat_cell_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string irat_cell_id_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRAN", "uTRAN", "gERAN", "eHRPD"};
  return convert_enum_idx(options, 4, value, "irat_cell_id_c::types");
}

// CandidatePCI ::= SEQUENCE
SRSASN_CODE candidate_pci_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, pci, (uint16_t)0u, (uint16_t)503u, false, true));
  HANDLE_CODE(earfcn.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE candidate_pci_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(pci, bref, (uint16_t)0u, (uint16_t)503u, false, true));
  HANDLE_CODE(earfcn.unpack(bref));

  return SRSASN_SUCCESS;
}
void candidate_pci_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pCI", pci);
  j.write_str("eARFCN", earfcn.to_string());
  j.end_obj();
}

// CauseMisc ::= ENUMERATED
std::string cause_misc_opts::to_string() const
{
  static const char* options[] = {"control-processing-overload",
                                  "not-enough-user-plane-processing-resources",
                                  "hardware-failure",
                                  "om-intervention",
                                  "unspecified",
                                  "unknown-PLMN"};
  return convert_enum_idx(options, 6, value, "cause_misc_e");
}

// CauseNas ::= ENUMERATED
std::string cause_nas_opts::to_string() const
{
  static const char* options[] = {
      "normal-release", "authentication-failure", "detach", "unspecified", "csg-subscription-expiry"};
  return convert_enum_idx(options, 5, value, "cause_nas_e");
}

// CauseProtocol ::= ENUMERATED
std::string cause_protocol_opts::to_string() const
{
  static const char* options[] = {"transfer-syntax-error",
                                  "abstract-syntax-error-reject",
                                  "abstract-syntax-error-ignore-and-notify",
                                  "message-not-compatible-with-receiver-state",
                                  "semantic-error",
                                  "abstract-syntax-error-falsely-constructed-message",
                                  "unspecified"};
  return convert_enum_idx(options, 7, value, "cause_protocol_e");
}

// CauseRadioNetwork ::= ENUMERATED
std::string cause_radio_network_opts::to_string() const
{
  static const char* options[] = {"unspecified",
                                  "tx2relocoverall-expiry",
                                  "successful-handover",
                                  "release-due-to-eutran-generated-reason",
                                  "handover-cancelled",
                                  "partial-handover",
                                  "ho-failure-in-target-EPC-eNB-or-target-system",
                                  "ho-target-not-allowed",
                                  "tS1relocoverall-expiry",
                                  "tS1relocprep-expiry",
                                  "cell-not-available",
                                  "unknown-targetID",
                                  "no-radio-resources-available-in-target-cell",
                                  "unknown-mme-ue-s1ap-id",
                                  "unknown-enb-ue-s1ap-id",
                                  "unknown-pair-ue-s1ap-id",
                                  "handover-desirable-for-radio-reason",
                                  "time-critical-handover",
                                  "resource-optimisation-handover",
                                  "reduce-load-in-serving-cell",
                                  "user-inactivity",
                                  "radio-connection-with-ue-lost",
                                  "load-balancing-tau-required",
                                  "cs-fallback-triggered",
                                  "ue-not-available-for-ps-service",
                                  "radio-resources-not-available",
                                  "failure-in-radio-interface-procedure",
                                  "invalid-qos-combination",
                                  "interrat-redirection",
                                  "interaction-with-other-procedure",
                                  "unknown-E-RAB-ID",
                                  "multiple-E-RAB-ID-instances",
                                  "encryption-and-or-integrity-protection-algorithms-not-supported",
                                  "s1-intra-system-handover-triggered",
                                  "s1-inter-system-handover-triggered",
                                  "x2-handover-triggered",
                                  "redirection-towards-1xRTT",
                                  "not-supported-QCI-value",
                                  "invalid-CSG-Id",
                                  "release-due-to-pre-emption"};
  return convert_enum_idx(options, 40, value, "cause_radio_network_e");
}

// CauseTransport ::= ENUMERATED
std::string cause_transport_opts::to_string() const
{
  static const char* options[] = {"transport-resource-unavailable", "unspecified"};
  return convert_enum_idx(options, 2, value, "cause_transport_e");
}

// Cause ::= CHOICE
void cause_c::destroy_() {}
void cause_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
cause_c::cause_c(const cause_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::radio_network:
      c.init(other.c.get<cause_radio_network_e>());
      break;
    case types::transport:
      c.init(other.c.get<cause_transport_e>());
      break;
    case types::nas:
      c.init(other.c.get<cause_nas_e>());
      break;
    case types::protocol:
      c.init(other.c.get<cause_protocol_e>());
      break;
    case types::misc:
      c.init(other.c.get<cause_misc_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
}
cause_c& cause_c::operator=(const cause_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::radio_network:
      c.set(other.c.get<cause_radio_network_e>());
      break;
    case types::transport:
      c.set(other.c.get<cause_transport_e>());
      break;
    case types::nas:
      c.set(other.c.get<cause_nas_e>());
      break;
    case types::protocol:
      c.set(other.c.get<cause_protocol_e>());
      break;
    case types::misc:
      c.set(other.c.get<cause_misc_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }

  return *this;
}
void cause_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::radio_network:
      j.write_str("radioNetwork", c.get<cause_radio_network_e>().to_string());
      break;
    case types::transport:
      j.write_str("transport", c.get<cause_transport_e>().to_string());
      break;
    case types::nas:
      j.write_str("nas", c.get<cause_nas_e>().to_string());
      break;
    case types::protocol:
      j.write_str("protocol", c.get<cause_protocol_e>().to_string());
      break;
    case types::misc:
      j.write_str("misc", c.get<cause_misc_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
  j.end_obj();
}
SRSASN_CODE cause_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::radio_network:
      HANDLE_CODE(c.get<cause_radio_network_e>().pack(bref));
      break;
    case types::transport:
      HANDLE_CODE(c.get<cause_transport_e>().pack(bref));
      break;
    case types::nas:
      HANDLE_CODE(c.get<cause_nas_e>().pack(bref));
      break;
    case types::protocol:
      HANDLE_CODE(c.get<cause_protocol_e>().pack(bref));
      break;
    case types::misc:
      HANDLE_CODE(c.get<cause_misc_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cause_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::radio_network:
      HANDLE_CODE(c.get<cause_radio_network_e>().unpack(bref));
      break;
    case types::transport:
      HANDLE_CODE(c.get<cause_transport_e>().unpack(bref));
      break;
    case types::nas:
      HANDLE_CODE(c.get<cause_nas_e>().unpack(bref));
      break;
    case types::protocol:
      HANDLE_CODE(c.get<cause_protocol_e>().unpack(bref));
      break;
    case types::misc:
      HANDLE_CODE(c.get<cause_misc_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string cause_c::types_opts::to_string() const
{
  static const char* options[] = {"radioNetwork", "transport", "nas", "protocol", "misc"};
  return convert_enum_idx(options, 5, value, "cause_c::types");
}
uint8_t cause_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "cause_c::types");
}

// Cdma2000OneXSRVCCInfo ::= SEQUENCE
SRSASN_CODE cdma2000_one_xsrvcc_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cdma2000_one_xmeid.pack(bref));
  HANDLE_CODE(cdma2000_one_xmsi.pack(bref));
  HANDLE_CODE(cdma2000_one_xpilot.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cdma2000_one_xsrvcc_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cdma2000_one_xmeid.unpack(bref));
  HANDLE_CODE(cdma2000_one_xmsi.unpack(bref));
  HANDLE_CODE(cdma2000_one_xpilot.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cdma2000_one_xsrvcc_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cdma2000OneXMEID", cdma2000_one_xmeid.to_string());
  j.write_str("cdma2000OneXMSI", cdma2000_one_xmsi.to_string());
  j.write_str("cdma2000OneXPilot", cdma2000_one_xpilot.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellsToActivateList-Item ::= SEQUENCE
SRSASN_CODE cells_to_activ_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cells_to_activ_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));

  return SRSASN_SUCCESS;
}
void cells_to_activ_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.end_obj();
}

// CellActivationRequest ::= SEQUENCE
SRSASN_CODE cell_activation_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(minimum_activation_time_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, cells_to_activ_list, 1, 256, true));
  if (minimum_activation_time_present) {
    HANDLE_CODE(pack_integer(bref, minimum_activation_time, (uint8_t)1u, (uint8_t)60u, false, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_activation_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(minimum_activation_time_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(cells_to_activ_list, bref, 1, 256, true));
  if (minimum_activation_time_present) {
    HANDLE_CODE(unpack_integer(minimum_activation_time, bref, (uint8_t)1u, (uint8_t)60u, false, true));
  }

  return SRSASN_SUCCESS;
}
void cell_activation_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("cellsToActivateList");
  for (const auto& e1 : cells_to_activ_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (minimum_activation_time_present) {
    j.write_int("minimumActivationTime", minimum_activation_time);
  }
  j.end_obj();
}

// CellActivationResponse ::= SEQUENCE
SRSASN_CODE cell_activation_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, activ_cells_list, 0, 256, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_activation_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(activ_cells_list, bref, 0, 256, true));

  return SRSASN_SUCCESS;
}
void cell_activation_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("activatedCellsList");
  for (const auto& e1 : activ_cells_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// EHRPDCompositeAvailableCapacity ::= SEQUENCE
SRSASN_CODE ehrpd_composite_available_capacity_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_integer(bref, ehrpd_sector_capacity_class_value, (uint8_t)1u, (uint8_t)100u, true, true));
  HANDLE_CODE(pack_integer(bref, ehrpd_capacity_value, (uint8_t)0u, (uint8_t)100u, false, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ehrpd_composite_available_capacity_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_integer(ehrpd_sector_capacity_class_value, bref, (uint8_t)1u, (uint8_t)100u, true, true));
  HANDLE_CODE(unpack_integer(ehrpd_capacity_value, bref, (uint8_t)0u, (uint8_t)100u, false, true));

  return SRSASN_SUCCESS;
}
void ehrpd_composite_available_capacity_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("eHRPDSectorCapacityClassValue", ehrpd_sector_capacity_class_value);
  j.write_int("eHRPDCapacityValue", ehrpd_capacity_value);
  j.end_obj();
}

// EHRPDSectorLoadReportingResponse ::= SEQUENCE
SRSASN_CODE ehrpd_sector_load_report_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(dl_ehrpd_composite_available_capacity.pack(bref));
  HANDLE_CODE(ul_ehrpd_composite_available_capacity.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ehrpd_sector_load_report_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(dl_ehrpd_composite_available_capacity.unpack(bref));
  HANDLE_CODE(ul_ehrpd_composite_available_capacity.unpack(bref));

  return SRSASN_SUCCESS;
}
void ehrpd_sector_load_report_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("dL-EHRPD-CompositeAvailableCapacity");
  dl_ehrpd_composite_available_capacity.to_json(j);
  j.write_fieldname("uL-EHRPD-CompositeAvailableCapacity");
  ul_ehrpd_composite_available_capacity.to_json(j);
  j.end_obj();
}

// EUTRANcellLoadReportingResponse ::= SEQUENCE
SRSASN_CODE eutra_ncell_load_report_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(composite_available_capacity_group.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutra_ncell_load_report_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(composite_available_capacity_group.unpack(bref));

  return SRSASN_SUCCESS;
}
void eutra_ncell_load_report_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("compositeAvailableCapacityGroup", composite_available_capacity_group.to_string());
  j.end_obj();
}

// CellLoadReportingResponse ::= CHOICE
void cell_load_report_resp_c::destroy_()
{
  switch (type_) {
    case types::eutran:
      c.destroy<eutra_ncell_load_report_resp_s>();
      break;
    case types::utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.destroy<ehrpd_sector_load_report_resp_s>();
      break;
    default:
      break;
  }
}
void cell_load_report_resp_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutran:
      c.init<eutra_ncell_load_report_resp_s>();
      break;
    case types::utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::geran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ehrpd:
      c.init<ehrpd_sector_load_report_resp_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }
}
cell_load_report_resp_c::cell_load_report_resp_c(const cell_load_report_resp_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutran:
      c.init(other.c.get<eutra_ncell_load_report_resp_s>());
      break;
    case types::utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.init(other.c.get<ehrpd_sector_load_report_resp_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }
}
cell_load_report_resp_c& cell_load_report_resp_c::operator=(const cell_load_report_resp_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutran:
      c.set(other.c.get<eutra_ncell_load_report_resp_s>());
      break;
    case types::utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::geran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ehrpd:
      c.set(other.c.get<ehrpd_sector_load_report_resp_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }

  return *this;
}
void cell_load_report_resp_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutran:
      j.write_fieldname("eUTRAN");
      c.get<eutra_ncell_load_report_resp_s>().to_json(j);
      break;
    case types::utran:
      j.write_str("uTRAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::geran:
      j.write_str("gERAN", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ehrpd:
      j.write_fieldname("eHRPD");
      c.get<ehrpd_sector_load_report_resp_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_load_report_resp_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<eutra_ncell_load_report_resp_s>().pack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ehrpd: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<ehrpd_sector_load_report_resp_s>().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_load_report_resp_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutran:
      HANDLE_CODE(c.get<eutra_ncell_load_report_resp_s>().unpack(bref));
      break;
    case types::utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::geran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ehrpd: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<ehrpd_sector_load_report_resp_s>().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "cell_load_report_resp_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string cell_load_report_resp_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRAN", "uTRAN", "gERAN", "eHRPD"};
  return convert_enum_idx(options, 4, value, "cell_load_report_resp_c::types");
}

// NotifyFlag ::= ENUMERATED
std::string notify_flag_opts::to_string() const
{
  static const char* options[] = {"activated", "deactivated"};
  return convert_enum_idx(options, 2, value, "notify_flag_e");
}

// NotificationCellList-Item ::= SEQUENCE
SRSASN_CODE notif_cell_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));
  HANDLE_CODE(notify_flag.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE notif_cell_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));
  HANDLE_CODE(notify_flag.unpack(bref));

  return SRSASN_SUCCESS;
}
void notif_cell_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.write_str("notifyFlag", notify_flag.to_string());
  j.end_obj();
}

// CellStateIndication ::= SEQUENCE
SRSASN_CODE cell_state_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, notif_cell_list, 1, 256, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_state_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(notif_cell_list, bref, 1, 256, true));

  return SRSASN_SUCCESS;
}
void cell_state_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("notificationCellList");
  for (const auto& e1 : notif_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}

// PrivacyIndicator ::= ENUMERATED
std::string privacy_ind_opts::to_string() const
{
  static const char* options[] = {"immediate-MDT", "logged-MDT"};
  return convert_enum_idx(options, 2, value, "privacy_ind_e");
}

// CellTrafficTraceIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t cell_traffic_trace_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 86, 100, 131, 166};
  return map_enum_number(options, 6, idx, "id");
}
bool cell_traffic_trace_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 86, 100, 131, 166};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e cell_traffic_trace_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 86:
      return crit_e::ignore;
    case 100:
      return crit_e::ignore;
    case 131:
      return crit_e::ignore;
    case 166:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
cell_traffic_trace_ies_o::value_c cell_traffic_trace_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 86:
      ret.set(value_c::types::e_utran_trace_id);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 131:
      ret.set(value_c::types::trace_collection_entity_ip_address);
      break;
    case 166:
      ret.set(value_c::types::privacy_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e cell_traffic_trace_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 86:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 131:
      return presence_e::mandatory;
    case 166:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& cell_traffic_trace_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& cell_traffic_trace_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
fixed_octstring<8, true>& cell_traffic_trace_ies_o::value_c::e_utran_trace_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<fixed_octstring<8, true> >();
}
eutran_cgi_s& cell_traffic_trace_ies_o::value_c::eutran_cgi()
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
bounded_bitstring<1, 160, true, true>& cell_traffic_trace_ies_o::value_c::trace_collection_entity_ip_address()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
privacy_ind_e& cell_traffic_trace_ies_o::value_c::privacy_ind()
{
  assert_choice_type("PrivacyIndicator", type_.to_string(), "Value");
  return c.get<privacy_ind_e>();
}
const uint64_t& cell_traffic_trace_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& cell_traffic_trace_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const fixed_octstring<8, true>& cell_traffic_trace_ies_o::value_c::e_utran_trace_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<fixed_octstring<8, true> >();
}
const eutran_cgi_s& cell_traffic_trace_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
const bounded_bitstring<1, 160, true, true>&
cell_traffic_trace_ies_o::value_c::trace_collection_entity_ip_address() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
const privacy_ind_e& cell_traffic_trace_ies_o::value_c::privacy_ind() const
{
  assert_choice_type("PrivacyIndicator", type_.to_string(), "Value");
  return c.get<privacy_ind_e>();
}
void cell_traffic_trace_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::e_utran_trace_id:
      c.destroy<fixed_octstring<8, true> >();
      break;
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::trace_collection_entity_ip_address:
      c.destroy<bounded_bitstring<1, 160, true, true> >();
      break;
    default:
      break;
  }
}
void cell_traffic_trace_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::e_utran_trace_id:
      c.init<fixed_octstring<8, true> >();
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::trace_collection_entity_ip_address:
      c.init<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::privacy_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
}
cell_traffic_trace_ies_o::value_c::value_c(const cell_traffic_trace_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.init(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::trace_collection_entity_ip_address:
      c.init(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::privacy_ind:
      c.init(other.c.get<privacy_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
}
cell_traffic_trace_ies_o::value_c& cell_traffic_trace_ies_o::value_c::
                                   operator=(const cell_traffic_trace_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.set(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::trace_collection_entity_ip_address:
      c.set(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::privacy_ind:
      c.set(other.c.get<privacy_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }

  return *this;
}
void cell_traffic_trace_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<8, true> >().to_string());
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::trace_collection_entity_ip_address:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 160, true, true> >().to_string());
      break;
    case types::privacy_ind:
      j.write_str("PrivacyIndicator", c.get<privacy_ind_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_traffic_trace_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().pack(bref)));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::trace_collection_entity_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().pack(bref)));
      break;
    case types::privacy_ind:
      HANDLE_CODE(c.get<privacy_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().unpack(bref)));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::trace_collection_entity_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().unpack(bref)));
      break;
    case types::privacy_ind:
      HANDLE_CODE(c.get<privacy_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string cell_traffic_trace_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "EUTRAN-CGI",
                                  "BIT STRING",
                                  "PrivacyIndicator"};
  return convert_enum_idx(options, 6, value, "cell_traffic_trace_ies_o::value_c::types");
}

template <class valueT_>
protocol_ie_container_item_s<valueT_>::protocol_ie_container_item_s(uint32_t id_, crit_e crit_) : id(id_), crit(crit_)

{}
template <class valueT_>
SRSASN_CODE protocol_ie_container_item_s<valueT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(pack_integer(bref, id, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.pack(bref));
  {
    varlength_field_pack_guard varlen_scope(bref, true);
    HANDLE_CODE(value.pack(bref));
  }
  return SRSASN_SUCCESS;
}
template <class valueT_>
SRSASN_CODE protocol_ie_container_item_s<valueT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(crit.unpack(bref));
  {
    varlength_field_unpack_guard varlen_scope(bref, true);
    HANDLE_CODE(value.unpack(bref));
  }
  return SRSASN_SUCCESS;
}
template <class valueT_>
void protocol_ie_container_item_s<valueT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("id", id);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}

template struct asn1::s1ap::protocol_ie_field_s<cell_traffic_trace_ies_o>;

cell_traffic_trace_ies_container::cell_traffic_trace_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  e_utran_trace_id(86, crit_e::ignore),
  eutran_cgi(100, crit_e::ignore),
  trace_collection_entity_ip_address(131, crit_e::ignore),
  privacy_ind(166, crit_e::ignore)
{}
SRSASN_CODE cell_traffic_trace_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  nof_ies += privacy_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(e_utran_trace_id.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.pack(bref));
  if (privacy_ind_present) {
    HANDLE_CODE(privacy_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<cell_traffic_trace_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 86:
        nof_mandatory_ies--;
        e_utran_trace_id.id    = c.id;
        e_utran_trace_id.crit  = c.crit;
        e_utran_trace_id.value = c.value.e_utran_trace_id();
        break;
      case 100:
        nof_mandatory_ies--;
        eutran_cgi.id    = c.id;
        eutran_cgi.crit  = c.crit;
        eutran_cgi.value = c.value.eutran_cgi();
        break;
      case 131:
        nof_mandatory_ies--;
        trace_collection_entity_ip_address.id    = c.id;
        trace_collection_entity_ip_address.crit  = c.crit;
        trace_collection_entity_ip_address.value = c.value.trace_collection_entity_ip_address();
        break;
      case 166:
        privacy_ind_present = true;
        privacy_ind.id      = c.id;
        privacy_ind.crit    = c.crit;
        privacy_ind.value   = c.value.privacy_ind();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void cell_traffic_trace_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  e_utran_trace_id.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  trace_collection_entity_ip_address.to_json(j);
  if (privacy_ind_present) {
    j.write_fieldname("");
    privacy_ind.to_json(j);
  }
  j.end_obj();
}

// CellTrafficTrace ::= SEQUENCE
SRSASN_CODE cell_traffic_trace_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void cell_traffic_trace_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// Cell-Size ::= ENUMERATED
std::string cell_size_opts::to_string() const
{
  static const char* options[] = {"verysmall", "small", "medium", "large"};
  return convert_enum_idx(options, 4, value, "cell_size_e");
}

// CellType ::= SEQUENCE
SRSASN_CODE cell_type_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cell_size.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_type_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cell_size.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_type_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-Size", cell_size.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CE-ModeBRestricted ::= ENUMERATED
std::string ce_mode_brestricted_opts::to_string() const
{
  static const char* options[] = {"restricted", "not-restricted"};
  return convert_enum_idx(options, 2, value, "ce_mode_brestricted_e");
}

// DL-CP-SecurityInformation ::= SEQUENCE
SRSASN_CODE dl_cp_security_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(dl_nas_mac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_cp_security_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(dl_nas_mac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void dl_cp_security_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("dl-NAS-MAC", dl_nas_mac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EnhancedCoverageRestricted ::= ENUMERATED
std::string enhanced_coverage_restricted_opts::to_string() const
{
  static const char* options[] = {"restricted"};
  return convert_enum_idx(options, 1, value, "enhanced_coverage_restricted_e");
}

// ConnectionEstablishmentIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t conn_establishment_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 74, 251, 253, 271};
  return map_enum_number(options, 6, idx, "id");
}
bool conn_establishment_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 74, 251, 253, 271};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e conn_establishment_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 74:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 253:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
conn_establishment_ind_ies_o::value_c conn_establishment_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 74:
      ret.set(value_c::types::ue_radio_cap);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 253:
      ret.set(value_c::types::dl_cp_security_info);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e conn_establishment_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 74:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 253:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& conn_establishment_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& conn_establishment_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
unbounded_octstring<true>& conn_establishment_ind_ies_o::value_c::ue_radio_cap()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
enhanced_coverage_restricted_e& conn_establishment_ind_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
dl_cp_security_info_s& conn_establishment_ind_ies_o::value_c::dl_cp_security_info()
{
  assert_choice_type("DL-CP-SecurityInformation", type_.to_string(), "Value");
  return c.get<dl_cp_security_info_s>();
}
ce_mode_brestricted_e& conn_establishment_ind_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
const uint64_t& conn_establishment_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& conn_establishment_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const unbounded_octstring<true>& conn_establishment_ind_ies_o::value_c::ue_radio_cap() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const enhanced_coverage_restricted_e& conn_establishment_ind_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const dl_cp_security_info_s& conn_establishment_ind_ies_o::value_c::dl_cp_security_info() const
{
  assert_choice_type("DL-CP-SecurityInformation", type_.to_string(), "Value");
  return c.get<dl_cp_security_info_s>();
}
const ce_mode_brestricted_e& conn_establishment_ind_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
void conn_establishment_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ue_radio_cap:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::dl_cp_security_info:
      c.destroy<dl_cp_security_info_s>();
      break;
    default:
      break;
  }
}
void conn_establishment_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ue_radio_cap:
      c.init<unbounded_octstring<true> >();
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::dl_cp_security_info:
      c.init<dl_cp_security_info_s>();
      break;
    case types::ce_mode_brestricted:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }
}
conn_establishment_ind_ies_o::value_c::value_c(const conn_establishment_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ue_radio_cap:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::dl_cp_security_info:
      c.init(other.c.get<dl_cp_security_info_s>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }
}
conn_establishment_ind_ies_o::value_c& conn_establishment_ind_ies_o::value_c::
                                       operator=(const conn_establishment_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ue_radio_cap:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::dl_cp_security_info:
      c.set(other.c.get<dl_cp_security_info_s>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }

  return *this;
}
void conn_establishment_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ue_radio_cap:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::dl_cp_security_info:
      j.write_fieldname("DL-CP-SecurityInformation");
      c.get<dl_cp_security_info_s>().to_json(j);
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE conn_establishment_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::dl_cp_security_info:
      HANDLE_CODE(c.get<dl_cp_security_info_s>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE conn_establishment_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::dl_cp_security_info:
      HANDLE_CODE(c.get<dl_cp_security_info_s>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "conn_establishment_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string conn_establishment_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "EnhancedCoverageRestricted",
                                  "DL-CP-SecurityInformation",
                                  "CE-ModeBRestricted"};
  return convert_enum_idx(options, 6, value, "conn_establishment_ind_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<conn_establishment_ind_ies_o>;

conn_establishment_ind_ies_container::conn_establishment_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  ue_radio_cap(74, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  dl_cp_security_info(253, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore)
{}
SRSASN_CODE conn_establishment_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += ue_radio_cap_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += dl_cp_security_info_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ue_radio_cap_present) {
    HANDLE_CODE(ue_radio_cap.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (dl_cp_security_info_present) {
    HANDLE_CODE(dl_cp_security_info.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE conn_establishment_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<conn_establishment_ind_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 74:
        ue_radio_cap_present = true;
        ue_radio_cap.id      = c.id;
        ue_radio_cap.crit    = c.crit;
        ue_radio_cap.value   = c.value.ue_radio_cap();
        break;
      case 251:
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = c.id;
        enhanced_coverage_restricted.crit    = c.crit;
        enhanced_coverage_restricted.value   = c.value.enhanced_coverage_restricted();
        break;
      case 253:
        dl_cp_security_info_present = true;
        dl_cp_security_info.id      = c.id;
        dl_cp_security_info.crit    = c.crit;
        dl_cp_security_info.value   = c.value.dl_cp_security_info();
        break;
      case 271:
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = c.id;
        ce_mode_brestricted.crit    = c.crit;
        ce_mode_brestricted.value   = c.value.ce_mode_brestricted();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void conn_establishment_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ue_radio_cap_present) {
    j.write_fieldname("");
    ue_radio_cap.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (dl_cp_security_info_present) {
    j.write_fieldname("");
    dl_cp_security_info.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  j.end_obj();
}

// ConnectionEstablishmentIndication ::= SEQUENCE
SRSASN_CODE conn_establishment_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE conn_establishment_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void conn_establishment_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// TypeOfError ::= ENUMERATED
std::string type_of_error_opts::to_string() const
{
  static const char* options[] = {"not-understood", "missing"};
  return convert_enum_idx(options, 2, value, "type_of_error_e");
}

// CriticalityDiagnostics-IE-Item ::= SEQUENCE
SRSASN_CODE crit_diagnostics_ie_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(iecrit.pack(bref));
  HANDLE_CODE(pack_integer(bref, ie_id, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(type_of_error.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE crit_diagnostics_ie_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(iecrit.unpack(bref));
  HANDLE_CODE(unpack_integer(ie_id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(type_of_error.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void crit_diagnostics_ie_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("iECriticality", iecrit.to_string());
  j.write_int("iE-ID", ie_id);
  j.write_str("typeOfError", type_of_error.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TriggeringMessage ::= ENUMERATED
std::string trigger_msg_opts::to_string() const
{
  static const char* options[] = {"initiating-message", "successful-outcome", "unsuccessfull-outcome"};
  return convert_enum_idx(options, 3, value, "trigger_msg_e");
}

// CriticalityDiagnostics ::= SEQUENCE
SRSASN_CODE crit_diagnostics_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(proc_code_present, 1));
  HANDLE_CODE(bref.pack(trigger_msg_present, 1));
  HANDLE_CODE(bref.pack(proc_crit_present, 1));
  HANDLE_CODE(bref.pack(ies_crit_diagnostics_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (proc_code_present) {
    HANDLE_CODE(pack_integer(bref, proc_code, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (trigger_msg_present) {
    HANDLE_CODE(trigger_msg.pack(bref));
  }
  if (proc_crit_present) {
    HANDLE_CODE(proc_crit.pack(bref));
  }
  if (ies_crit_diagnostics_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ies_crit_diagnostics, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE crit_diagnostics_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(proc_code_present, 1));
  HANDLE_CODE(bref.unpack(trigger_msg_present, 1));
  HANDLE_CODE(bref.unpack(proc_crit_present, 1));
  HANDLE_CODE(bref.unpack(ies_crit_diagnostics_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (proc_code_present) {
    HANDLE_CODE(unpack_integer(proc_code, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (trigger_msg_present) {
    HANDLE_CODE(trigger_msg.unpack(bref));
  }
  if (proc_crit_present) {
    HANDLE_CODE(proc_crit.unpack(bref));
  }
  if (ies_crit_diagnostics_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ies_crit_diagnostics, bref, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void crit_diagnostics_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (proc_code_present) {
    j.write_int("procedureCode", proc_code);
  }
  if (trigger_msg_present) {
    j.write_str("triggeringMessage", trigger_msg.to_string());
  }
  if (proc_crit_present) {
    j.write_str("procedureCriticality", proc_crit.to_string());
  }
  if (ies_crit_diagnostics_present) {
    j.start_array("iEsCriticalityDiagnostics");
    for (const auto& e1 : ies_crit_diagnostics) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DeactivateTraceIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t deactiv_trace_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 86};
  return map_enum_number(options, 3, idx, "id");
}
bool deactiv_trace_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 86};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e deactiv_trace_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 86:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
deactiv_trace_ies_o::value_c deactiv_trace_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 86:
      ret.set(value_c::types::e_utran_trace_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e deactiv_trace_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 86:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& deactiv_trace_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& deactiv_trace_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
fixed_octstring<8, true>& deactiv_trace_ies_o::value_c::e_utran_trace_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<fixed_octstring<8, true> >();
}
const uint64_t& deactiv_trace_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& deactiv_trace_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const fixed_octstring<8, true>& deactiv_trace_ies_o::value_c::e_utran_trace_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<fixed_octstring<8, true> >();
}
void deactiv_trace_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::e_utran_trace_id:
      c.destroy<fixed_octstring<8, true> >();
      break;
    default:
      break;
  }
}
void deactiv_trace_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::e_utran_trace_id:
      c.init<fixed_octstring<8, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
}
deactiv_trace_ies_o::value_c::value_c(const deactiv_trace_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.init(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
}
deactiv_trace_ies_o::value_c& deactiv_trace_ies_o::value_c::operator=(const deactiv_trace_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      c.set(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }

  return *this;
}
void deactiv_trace_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::e_utran_trace_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<8, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE deactiv_trace_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::e_utran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string deactiv_trace_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "INTEGER (0..16777215)", "OCTET STRING"};
  return convert_enum_idx(options, 3, value, "deactiv_trace_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<deactiv_trace_ies_o>;

deactiv_trace_ies_container::deactiv_trace_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  e_utran_trace_id(86, crit_e::ignore)
{}
SRSASN_CODE deactiv_trace_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(e_utran_trace_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<deactiv_trace_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 86:
        nof_mandatory_ies--;
        e_utran_trace_id.id    = c.id;
        e_utran_trace_id.crit  = c.crit;
        e_utran_trace_id.value = c.value.e_utran_trace_id();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void deactiv_trace_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  e_utran_trace_id.to_json(j);
  j.end_obj();
}

// DeactivateTrace ::= SEQUENCE
SRSASN_CODE deactiv_trace_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void deactiv_trace_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// ForbiddenLAs-Item ::= SEQUENCE
SRSASN_CODE forbidden_las_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_lacs, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE forbidden_las_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(forbidden_lacs, bref, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void forbidden_las_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.start_array("forbiddenLACs");
  for (const auto& e1 : forbidden_lacs) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ForbiddenTAs-Item ::= SEQUENCE
SRSASN_CODE forbidden_tas_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_tacs, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE forbidden_tas_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(forbidden_tacs, bref, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void forbidden_tas_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMN-Identity", plmn_id.to_string());
  j.start_array("forbiddenTACs");
  for (const auto& e1 : forbidden_tacs) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ForbiddenInterRATs ::= ENUMERATED
std::string forbidden_inter_rats_opts::to_string() const
{
  static const char* options[] = {"all", "geran", "utran", "cdma2000", "geranandutran", "cdma2000andutran"};
  return convert_enum_idx(options, 6, value, "forbidden_inter_rats_e");
}

// DLNASPDUDeliveryAckRequest ::= ENUMERATED
std::string dlnaspdu_delivery_ack_request_opts::to_string() const
{
  static const char* options[] = {"requested"};
  return convert_enum_idx(options, 1, value, "dlnaspdu_delivery_ack_request_e");
}

// HandoverRestrictionList ::= SEQUENCE
SRSASN_CODE ho_restrict_list_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(equivalent_plmns_present, 1));
  HANDLE_CODE(bref.pack(forbidden_tas_present, 1));
  HANDLE_CODE(bref.pack(forbidden_las_present, 1));
  HANDLE_CODE(bref.pack(forbidden_inter_rats_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(serving_plmn.pack(bref));
  if (equivalent_plmns_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, equivalent_plmns, 1, 15, true));
  }
  if (forbidden_tas_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_tas, 1, 16, true));
  }
  if (forbidden_las_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_las, 1, 16, true));
  }
  if (forbidden_inter_rats_present) {
    HANDLE_CODE(forbidden_inter_rats.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_restrict_list_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(equivalent_plmns_present, 1));
  HANDLE_CODE(bref.unpack(forbidden_tas_present, 1));
  HANDLE_CODE(bref.unpack(forbidden_las_present, 1));
  HANDLE_CODE(bref.unpack(forbidden_inter_rats_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(serving_plmn.unpack(bref));
  if (equivalent_plmns_present) {
    HANDLE_CODE(unpack_dyn_seq_of(equivalent_plmns, bref, 1, 15, true));
  }
  if (forbidden_tas_present) {
    HANDLE_CODE(unpack_dyn_seq_of(forbidden_tas, bref, 1, 16, true));
  }
  if (forbidden_las_present) {
    HANDLE_CODE(unpack_dyn_seq_of(forbidden_las, bref, 1, 16, true));
  }
  if (forbidden_inter_rats_present) {
    HANDLE_CODE(forbidden_inter_rats.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ho_restrict_list_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("servingPLMN", serving_plmn.to_string());
  if (equivalent_plmns_present) {
    j.start_array("equivalentPLMNs");
    for (const auto& e1 : equivalent_plmns) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (forbidden_tas_present) {
    j.start_array("forbiddenTAs");
    for (const auto& e1 : forbidden_tas) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (forbidden_las_present) {
    j.start_array("forbiddenLAs");
    for (const auto& e1 : forbidden_las) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (forbidden_inter_rats_present) {
    j.write_str("forbiddenInterRATs", forbidden_inter_rats.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PendingDataIndication ::= ENUMERATED
std::string pending_data_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "pending_data_ind_e");
}

// SRVCCOperationPossible ::= ENUMERATED
std::string srvcc_operation_possible_opts::to_string() const
{
  static const char* options[] = {"possible"};
  return convert_enum_idx(options, 1, value, "srvcc_operation_possible_e");
}

// DownlinkNASTransport-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_nas_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 26, 41, 106, 124, 74, 249, 251, 271, 283};
  return map_enum_number(options, 11, idx, "id");
}
bool dl_nas_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 26, 41, 106, 124, 74, 249, 251, 271, 283};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_nas_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 26:
      return crit_e::reject;
    case 41:
      return crit_e::ignore;
    case 106:
      return crit_e::ignore;
    case 124:
      return crit_e::ignore;
    case 74:
      return crit_e::ignore;
    case 249:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    case 283:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_nas_transport_ies_o::value_c dl_nas_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 26:
      ret.set(value_c::types::nas_pdu);
      break;
    case 41:
      ret.set(value_c::types::ho_restrict_list);
      break;
    case 106:
      ret.set(value_c::types::subscriber_profile_idfor_rfp);
      break;
    case 124:
      ret.set(value_c::types::srvcc_operation_possible);
      break;
    case 74:
      ret.set(value_c::types::ue_radio_cap);
      break;
    case 249:
      ret.set(value_c::types::dlnaspdu_delivery_ack_request);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    case 283:
      ret.set(value_c::types::pending_data_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_nas_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 26:
      return presence_e::mandatory;
    case 41:
      return presence_e::optional;
    case 106:
      return presence_e::optional;
    case 124:
      return presence_e::optional;
    case 74:
      return presence_e::optional;
    case 249:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    case 283:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& dl_nas_transport_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& dl_nas_transport_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::nas_pdu()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
ho_restrict_list_s& dl_nas_transport_ies_o::value_c::ho_restrict_list()
{
  assert_choice_type("HandoverRestrictionList", type_.to_string(), "Value");
  return c.get<ho_restrict_list_s>();
}
uint16_t& dl_nas_transport_ies_o::value_c::subscriber_profile_idfor_rfp()
{
  assert_choice_type("INTEGER (1..256)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
srvcc_operation_possible_e& dl_nas_transport_ies_o::value_c::srvcc_operation_possible()
{
  assert_choice_type("SRVCCOperationPossible", type_.to_string(), "Value");
  return c.get<srvcc_operation_possible_e>();
}
unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::ue_radio_cap()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
dlnaspdu_delivery_ack_request_e& dl_nas_transport_ies_o::value_c::dlnaspdu_delivery_ack_request()
{
  assert_choice_type("DLNASPDUDeliveryAckRequest", type_.to_string(), "Value");
  return c.get<dlnaspdu_delivery_ack_request_e>();
}
enhanced_coverage_restricted_e& dl_nas_transport_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
ce_mode_brestricted_e& dl_nas_transport_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
pending_data_ind_e& dl_nas_transport_ies_o::value_c::pending_data_ind()
{
  assert_choice_type("PendingDataIndication", type_.to_string(), "Value");
  return c.get<pending_data_ind_e>();
}
const uint64_t& dl_nas_transport_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& dl_nas_transport_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::nas_pdu() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const ho_restrict_list_s& dl_nas_transport_ies_o::value_c::ho_restrict_list() const
{
  assert_choice_type("HandoverRestrictionList", type_.to_string(), "Value");
  return c.get<ho_restrict_list_s>();
}
const uint16_t& dl_nas_transport_ies_o::value_c::subscriber_profile_idfor_rfp() const
{
  assert_choice_type("INTEGER (1..256)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
const srvcc_operation_possible_e& dl_nas_transport_ies_o::value_c::srvcc_operation_possible() const
{
  assert_choice_type("SRVCCOperationPossible", type_.to_string(), "Value");
  return c.get<srvcc_operation_possible_e>();
}
const unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::ue_radio_cap() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const dlnaspdu_delivery_ack_request_e& dl_nas_transport_ies_o::value_c::dlnaspdu_delivery_ack_request() const
{
  assert_choice_type("DLNASPDUDeliveryAckRequest", type_.to_string(), "Value");
  return c.get<dlnaspdu_delivery_ack_request_e>();
}
const enhanced_coverage_restricted_e& dl_nas_transport_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const ce_mode_brestricted_e& dl_nas_transport_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
const pending_data_ind_e& dl_nas_transport_ies_o::value_c::pending_data_ind() const
{
  assert_choice_type("PendingDataIndication", type_.to_string(), "Value");
  return c.get<pending_data_ind_e>();
}
void dl_nas_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ho_restrict_list:
      c.destroy<ho_restrict_list_s>();
      break;
    case types::ue_radio_cap:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_nas_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ho_restrict_list:
      c.init<ho_restrict_list_s>();
      break;
    case types::subscriber_profile_idfor_rfp:
      break;
    case types::srvcc_operation_possible:
      break;
    case types::ue_radio_cap:
      c.init<unbounded_octstring<true> >();
      break;
    case types::dlnaspdu_delivery_ack_request:
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::ce_mode_brestricted:
      break;
    case types::pending_data_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
}
dl_nas_transport_ies_o::value_c::value_c(const dl_nas_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ho_restrict_list:
      c.init(other.c.get<ho_restrict_list_s>());
      break;
    case types::subscriber_profile_idfor_rfp:
      c.init(other.c.get<uint16_t>());
      break;
    case types::srvcc_operation_possible:
      c.init(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::ue_radio_cap:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::dlnaspdu_delivery_ack_request:
      c.init(other.c.get<dlnaspdu_delivery_ack_request_e>());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::pending_data_ind:
      c.init(other.c.get<pending_data_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
}
dl_nas_transport_ies_o::value_c& dl_nas_transport_ies_o::value_c::
                                 operator=(const dl_nas_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ho_restrict_list:
      c.set(other.c.get<ho_restrict_list_s>());
      break;
    case types::subscriber_profile_idfor_rfp:
      c.set(other.c.get<uint16_t>());
      break;
    case types::srvcc_operation_possible:
      c.set(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::ue_radio_cap:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::dlnaspdu_delivery_ack_request:
      c.set(other.c.get<dlnaspdu_delivery_ack_request_e>());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::pending_data_ind:
      c.set(other.c.get<pending_data_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }

  return *this;
}
void dl_nas_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ho_restrict_list:
      j.write_fieldname("HandoverRestrictionList");
      c.get<ho_restrict_list_s>().to_json(j);
      break;
    case types::subscriber_profile_idfor_rfp:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::srvcc_operation_possible:
      j.write_str("SRVCCOperationPossible", "possible");
      break;
    case types::ue_radio_cap:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::dlnaspdu_delivery_ack_request:
      j.write_str("DLNASPDUDeliveryAckRequest", "requested");
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    case types::pending_data_ind:
      j.write_str("PendingDataIndication", "true");
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_nas_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().pack(bref));
      break;
    case types::subscriber_profile_idfor_rfp:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().pack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::dlnaspdu_delivery_ack_request:
      HANDLE_CODE(c.get<dlnaspdu_delivery_ack_request_e>().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().unpack(bref));
      break;
    case types::subscriber_profile_idfor_rfp:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().unpack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::dlnaspdu_delivery_ack_request:
      HANDLE_CODE(c.get<dlnaspdu_delivery_ack_request_e>().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string dl_nas_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "HandoverRestrictionList",
                                  "INTEGER (1..256)",
                                  "SRVCCOperationPossible",
                                  "OCTET STRING",
                                  "DLNASPDUDeliveryAckRequest",
                                  "EnhancedCoverageRestricted",
                                  "CE-ModeBRestricted",
                                  "PendingDataIndication"};
  return convert_enum_idx(options, 11, value, "dl_nas_transport_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<dl_nas_transport_ies_o>;

dl_nas_transport_ies_container::dl_nas_transport_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  nas_pdu(26, crit_e::reject),
  ho_restrict_list(41, crit_e::ignore),
  subscriber_profile_idfor_rfp(106, crit_e::ignore),
  srvcc_operation_possible(124, crit_e::ignore),
  ue_radio_cap(74, crit_e::ignore),
  dlnaspdu_delivery_ack_request(249, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore),
  pending_data_ind(283, crit_e::ignore)
{}
SRSASN_CODE dl_nas_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ho_restrict_list_present ? 1 : 0;
  nof_ies += subscriber_profile_idfor_rfp_present ? 1 : 0;
  nof_ies += srvcc_operation_possible_present ? 1 : 0;
  nof_ies += ue_radio_cap_present ? 1 : 0;
  nof_ies += dlnaspdu_delivery_ack_request_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  nof_ies += pending_data_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  if (ho_restrict_list_present) {
    HANDLE_CODE(ho_restrict_list.pack(bref));
  }
  if (subscriber_profile_idfor_rfp_present) {
    HANDLE_CODE(subscriber_profile_idfor_rfp.pack(bref));
  }
  if (srvcc_operation_possible_present) {
    HANDLE_CODE(srvcc_operation_possible.pack(bref));
  }
  if (ue_radio_cap_present) {
    HANDLE_CODE(ue_radio_cap.pack(bref));
  }
  if (dlnaspdu_delivery_ack_request_present) {
    HANDLE_CODE(dlnaspdu_delivery_ack_request.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }
  if (pending_data_ind_present) {
    HANDLE_CODE(pending_data_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<dl_nas_transport_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 26:
        nof_mandatory_ies--;
        nas_pdu.id    = c.id;
        nas_pdu.crit  = c.crit;
        nas_pdu.value = c.value.nas_pdu();
        break;
      case 41:
        ho_restrict_list_present = true;
        ho_restrict_list.id      = c.id;
        ho_restrict_list.crit    = c.crit;
        ho_restrict_list.value   = c.value.ho_restrict_list();
        break;
      case 106:
        subscriber_profile_idfor_rfp_present = true;
        subscriber_profile_idfor_rfp.id      = c.id;
        subscriber_profile_idfor_rfp.crit    = c.crit;
        subscriber_profile_idfor_rfp.value   = c.value.subscriber_profile_idfor_rfp();
        break;
      case 124:
        srvcc_operation_possible_present = true;
        srvcc_operation_possible.id      = c.id;
        srvcc_operation_possible.crit    = c.crit;
        srvcc_operation_possible.value   = c.value.srvcc_operation_possible();
        break;
      case 74:
        ue_radio_cap_present = true;
        ue_radio_cap.id      = c.id;
        ue_radio_cap.crit    = c.crit;
        ue_radio_cap.value   = c.value.ue_radio_cap();
        break;
      case 249:
        dlnaspdu_delivery_ack_request_present = true;
        dlnaspdu_delivery_ack_request.id      = c.id;
        dlnaspdu_delivery_ack_request.crit    = c.crit;
        dlnaspdu_delivery_ack_request.value   = c.value.dlnaspdu_delivery_ack_request();
        break;
      case 251:
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = c.id;
        enhanced_coverage_restricted.crit    = c.crit;
        enhanced_coverage_restricted.value   = c.value.enhanced_coverage_restricted();
        break;
      case 271:
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = c.id;
        ce_mode_brestricted.crit    = c.crit;
        ce_mode_brestricted.value   = c.value.ce_mode_brestricted();
        break;
      case 283:
        pending_data_ind_present = true;
        pending_data_ind.id      = c.id;
        pending_data_ind.crit    = c.crit;
        pending_data_ind.value   = c.value.pending_data_ind();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_nas_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  nas_pdu.to_json(j);
  if (ho_restrict_list_present) {
    j.write_fieldname("");
    ho_restrict_list.to_json(j);
  }
  if (subscriber_profile_idfor_rfp_present) {
    j.write_fieldname("");
    subscriber_profile_idfor_rfp.to_json(j);
  }
  if (srvcc_operation_possible_present) {
    j.write_fieldname("");
    srvcc_operation_possible.to_json(j);
  }
  if (ue_radio_cap_present) {
    j.write_fieldname("");
    ue_radio_cap.to_json(j);
  }
  if (dlnaspdu_delivery_ack_request_present) {
    j.write_fieldname("");
    dlnaspdu_delivery_ack_request.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  if (pending_data_ind_present) {
    j.write_fieldname("");
    pending_data_ind.to_json(j);
  }
  j.end_obj();
}

// DownlinkNASTransport ::= SEQUENCE
SRSASN_CODE dl_nas_transport_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_nas_transport_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// DownlinkNonUEAssociatedLPPaTransport-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_non_ueassociated_lp_pa_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {148, 147};
  return map_enum_number(options, 2, idx, "id");
}
bool dl_non_ueassociated_lp_pa_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {148, 147};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_non_ueassociated_lp_pa_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 148:
      return crit_e::reject;
    case 147:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_non_ueassociated_lp_pa_transport_ies_o::value_c
dl_non_ueassociated_lp_pa_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 148:
      ret.set(value_c::types::routing_id);
      break;
    case 147:
      ret.set(value_c::types::lp_pa_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_non_ueassociated_lp_pa_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 148:
      return presence_e::mandatory;
    case 147:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint16_t& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::routing_id()
{
  assert_choice_type("INTEGER (0..255)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint16_t& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::routing_id() const
{
  assert_choice_type("INTEGER (0..255)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_non_ueassociated_lp_pa_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::lp_pa_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_non_ueassociated_lp_pa_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::routing_id:
      break;
    case types::lp_pa_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_non_ueassociated_lp_pa_transport_ies_o::value_c::value_c(
    const dl_non_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::routing_id:
      c.init(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_non_ueassociated_lp_pa_transport_ies_o::value_c& dl_non_ueassociated_lp_pa_transport_ies_o::value_c::
                                                    operator=(const dl_non_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::routing_id:
      c.set(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }

  return *this;
}
void dl_non_ueassociated_lp_pa_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::routing_id:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::routing_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::routing_id:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..255)", "OCTET STRING"};
  return convert_enum_idx(options, 2, value, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types");
}
uint8_t dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "dl_non_ueassociated_lp_pa_transport_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<dl_non_ueassociated_lp_pa_transport_ies_o>;

dl_non_ueassociated_lp_pa_transport_ies_container::dl_non_ueassociated_lp_pa_transport_ies_container() :
  routing_id(148, crit_e::reject),
  lp_pa_pdu(147, crit_e::reject)
{}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(routing_id.pack(bref));
  HANDLE_CODE(lp_pa_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<dl_non_ueassociated_lp_pa_transport_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 148:
        nof_mandatory_ies--;
        routing_id.id    = c.id;
        routing_id.crit  = c.crit;
        routing_id.value = c.value.routing_id();
        break;
      case 147:
        nof_mandatory_ies--;
        lp_pa_pdu.id    = c.id;
        lp_pa_pdu.crit  = c.crit;
        lp_pa_pdu.value = c.value.lp_pa_pdu();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_non_ueassociated_lp_pa_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  routing_id.to_json(j);
  j.write_fieldname("");
  lp_pa_pdu.to_json(j);
  j.end_obj();
}

// DownlinkNonUEAssociatedLPPaTransport ::= SEQUENCE
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_lp_pa_transport_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_non_ueassociated_lp_pa_transport_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABDataForwardingItem ::= SEQUENCE
SRSASN_CODE erab_data_forwarding_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.pack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.pack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.pack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.pack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_data_forwarding_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.unpack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.unpack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.unpack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_data_forwarding_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (dl_transport_layer_address_present) {
    j.write_str("dL-transportLayerAddress", dl_transport_layer_address.to_string());
  }
  if (dl_g_tp_teid_present) {
    j.write_str("dL-gTP-TEID", dl_g_tp_teid.to_string());
  }
  if (ul_transport_layer_address_present) {
    j.write_str("uL-TransportLayerAddress", ul_transport_layer_address.to_string());
  }
  if (ul_gtp_teid_present) {
    j.write_str("uL-GTP-TEID", ul_gtp_teid.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABDataForwardingItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_data_forwarding_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {14};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_data_forwarding_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 14 == id;
}
crit_e erab_data_forwarding_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 14) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_data_forwarding_item_ies_o::value_c erab_data_forwarding_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 14) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_data_forwarding_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 14) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_data_forwarding_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABDataForwardingItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_data_forwarding_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_data_forwarding_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_data_forwarding_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABDataForwardingItem"};
  return convert_enum_idx(options, 1, value, "erab_data_forwarding_item_ies_o::value_c::types");
}

// Cdma2000HOStatus ::= ENUMERATED
std::string cdma2000_ho_status_opts::to_string() const
{
  static const char* options[] = {"hOSuccess", "hOFailure"};
  return convert_enum_idx(options, 2, value, "cdma2000_ho_status_e");
}

// Cdma2000RATType ::= ENUMERATED
std::string cdma2000_rat_type_opts::to_string() const
{
  static const char* options[] = {"hRPD", "onexRTT"};
  return convert_enum_idx(options, 2, value, "cdma2000_rat_type_e");
}
uint8_t cdma2000_rat_type_opts::to_number() const
{
  if (value == onex_rtt) {
    return 1;
  }
  invalid_enum_number(value, "cdma2000_rat_type_e");
  return 0;
}

// DownlinkS1cdma2000tunnellingIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_s1cdma2000tunnelling_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 12, 83, 71, 70};
  return map_enum_number(options, 6, idx, "id");
}
bool dl_s1cdma2000tunnelling_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 12, 83, 71, 70};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_s1cdma2000tunnelling_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 12:
      return crit_e::ignore;
    case 83:
      return crit_e::ignore;
    case 71:
      return crit_e::reject;
    case 70:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_s1cdma2000tunnelling_ies_o::value_c dl_s1cdma2000tunnelling_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 12:
      ret.set(value_c::types::erab_subjectto_data_forwarding_list);
      break;
    case 83:
      ret.set(value_c::types::cdma2000_ho_status);
      break;
    case 71:
      ret.set(value_c::types::cdma2000_rat_type);
      break;
    case 70:
      ret.set(value_c::types::cdma2000_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_s1cdma2000tunnelling_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 12:
      return presence_e::optional;
    case 83:
      return presence_e::optional;
    case 71:
      return presence_e::mandatory;
    case 70:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& dl_s1cdma2000tunnelling_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& dl_s1cdma2000tunnelling_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_ie_container_list_l<erab_data_forwarding_item_ies_o>&
dl_s1cdma2000tunnelling_ies_o::value_c::erab_subjectto_data_forwarding_list()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
cdma2000_ho_status_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_ho_status()
{
  assert_choice_type("Cdma2000HOStatus", type_.to_string(), "Value");
  return c.get<cdma2000_ho_status_e>();
}
cdma2000_rat_type_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_rat_type()
{
  assert_choice_type("Cdma2000RATType", type_.to_string(), "Value");
  return c.get<cdma2000_rat_type_e>();
}
unbounded_octstring<true>& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_pdu()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& dl_s1cdma2000tunnelling_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& dl_s1cdma2000tunnelling_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_ie_container_list_l<erab_data_forwarding_item_ies_o>&
dl_s1cdma2000tunnelling_ies_o::value_c::erab_subjectto_data_forwarding_list() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
const cdma2000_ho_status_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_ho_status() const
{
  assert_choice_type("Cdma2000HOStatus", type_.to_string(), "Value");
  return c.get<cdma2000_ho_status_e>();
}
const cdma2000_rat_type_e& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_rat_type() const
{
  assert_choice_type("Cdma2000RATType", type_.to_string(), "Value");
  return c.get<cdma2000_rat_type_e>();
}
const unbounded_octstring<true>& dl_s1cdma2000tunnelling_ies_o::value_c::cdma2000_pdu() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_s1cdma2000tunnelling_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_subjectto_data_forwarding_list:
      c.destroy<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::cdma2000_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_s1cdma2000tunnelling_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::cdma2000_ho_status:
      break;
    case types::cdma2000_rat_type:
      break;
    case types::cdma2000_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }
}
dl_s1cdma2000tunnelling_ies_o::value_c::value_c(const dl_s1cdma2000tunnelling_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::cdma2000_ho_status:
      c.init(other.c.get<cdma2000_ho_status_e>());
      break;
    case types::cdma2000_rat_type:
      c.init(other.c.get<cdma2000_rat_type_e>());
      break;
    case types::cdma2000_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }
}
dl_s1cdma2000tunnelling_ies_o::value_c& dl_s1cdma2000tunnelling_ies_o::value_c::
                                        operator=(const dl_s1cdma2000tunnelling_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.set(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::cdma2000_ho_status:
      c.set(other.c.get<cdma2000_ho_status_e>());
      break;
    case types::cdma2000_rat_type:
      c.set(other.c.get<cdma2000_rat_type_e>());
      break;
    case types::cdma2000_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }

  return *this;
}
void dl_s1cdma2000tunnelling_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_subjectto_data_forwarding_list:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}");
      break;
    case types::cdma2000_ho_status:
      j.write_str("Cdma2000HOStatus", c.get<cdma2000_ho_status_e>().to_string());
      break;
    case types::cdma2000_rat_type:
      j.write_str("Cdma2000RATType", c.get<cdma2000_rat_type_e>().to_string());
      break;
    case types::cdma2000_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), 1, 256, true));
      break;
    case types::cdma2000_ho_status:
      HANDLE_CODE(c.get<cdma2000_ho_status_e>().pack(bref));
      break;
    case types::cdma2000_rat_type:
      HANDLE_CODE(c.get<cdma2000_rat_type_e>().pack(bref));
      break;
    case types::cdma2000_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          unpack_dyn_seq_of(c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), bref, 1, 256, true));
      break;
    case types::cdma2000_ho_status:
      HANDLE_CODE(c.get<cdma2000_ho_status_e>().unpack(bref));
      break;
    case types::cdma2000_rat_type:
      HANDLE_CODE(c.get<cdma2000_rat_type_e>().unpack(bref));
      break;
    case types::cdma2000_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_s1cdma2000tunnelling_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string dl_s1cdma2000tunnelling_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}",
                                  "Cdma2000HOStatus",
                                  "Cdma2000RATType",
                                  "OCTET STRING"};
  return convert_enum_idx(options, 6, value, "dl_s1cdma2000tunnelling_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<dl_s1cdma2000tunnelling_ies_o>;

dl_s1cdma2000tunnelling_ies_container::dl_s1cdma2000tunnelling_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  erab_subjectto_data_forwarding_list(12, crit_e::ignore),
  cdma2000_ho_status(83, crit_e::ignore),
  cdma2000_rat_type(71, crit_e::reject),
  cdma2000_pdu(70, crit_e::reject)
{}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += erab_subjectto_data_forwarding_list_present ? 1 : 0;
  nof_ies += cdma2000_ho_status_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_subjectto_data_forwarding_list_present) {
    HANDLE_CODE(erab_subjectto_data_forwarding_list.pack(bref));
  }
  if (cdma2000_ho_status_present) {
    HANDLE_CODE(cdma2000_ho_status.pack(bref));
  }
  HANDLE_CODE(cdma2000_rat_type.pack(bref));
  HANDLE_CODE(cdma2000_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_s1cdma2000tunnelling_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<dl_s1cdma2000tunnelling_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 12:
        erab_subjectto_data_forwarding_list_present = true;
        erab_subjectto_data_forwarding_list.id      = c.id;
        erab_subjectto_data_forwarding_list.crit    = c.crit;
        erab_subjectto_data_forwarding_list.value   = c.value.erab_subjectto_data_forwarding_list();
        break;
      case 83:
        cdma2000_ho_status_present = true;
        cdma2000_ho_status.id      = c.id;
        cdma2000_ho_status.crit    = c.crit;
        cdma2000_ho_status.value   = c.value.cdma2000_ho_status();
        break;
      case 71:
        nof_mandatory_ies--;
        cdma2000_rat_type.id    = c.id;
        cdma2000_rat_type.crit  = c.crit;
        cdma2000_rat_type.value = c.value.cdma2000_rat_type();
        break;
      case 70:
        nof_mandatory_ies--;
        cdma2000_pdu.id    = c.id;
        cdma2000_pdu.crit  = c.crit;
        cdma2000_pdu.value = c.value.cdma2000_pdu();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_s1cdma2000tunnelling_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_subjectto_data_forwarding_list_present) {
    j.write_fieldname("");
    erab_subjectto_data_forwarding_list.to_json(j);
  }
  if (cdma2000_ho_status_present) {
    j.write_fieldname("");
    cdma2000_ho_status.to_json(j);
  }
  j.write_fieldname("");
  cdma2000_rat_type.to_json(j);
  j.write_fieldname("");
  cdma2000_pdu.to_json(j);
  j.end_obj();
}

// DownlinkS1cdma2000tunnelling ::= SEQUENCE
SRSASN_CODE dl_s1cdma2000tunnelling_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_s1cdma2000tunnelling_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_s1cdma2000tunnelling_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// DownlinkUEAssociatedLPPaTransport-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t dl_ueassociated_lp_pa_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 148, 147};
  return map_enum_number(options, 4, idx, "id");
}
bool dl_ueassociated_lp_pa_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 148, 147};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_ueassociated_lp_pa_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 148:
      return crit_e::reject;
    case 147:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_ueassociated_lp_pa_transport_ies_o::value_c dl_ueassociated_lp_pa_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 148:
      ret.set(value_c::types::routing_id);
      break;
    case 147:
      ret.set(value_c::types::lp_pa_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_ueassociated_lp_pa_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 148:
      return presence_e::mandatory;
    case 147:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
uint16_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::routing_id()
{
  assert_choice_type("INTEGER (0..255)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& dl_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const uint16_t& dl_ueassociated_lp_pa_transport_ies_o::value_c::routing_id() const
{
  assert_choice_type("INTEGER (0..255)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& dl_ueassociated_lp_pa_transport_ies_o::value_c::lp_pa_pdu() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_ueassociated_lp_pa_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::lp_pa_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_ueassociated_lp_pa_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::routing_id:
      break;
    case types::lp_pa_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_ueassociated_lp_pa_transport_ies_o::value_c::value_c(const dl_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::routing_id:
      c.init(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }
}
dl_ueassociated_lp_pa_transport_ies_o::value_c& dl_ueassociated_lp_pa_transport_ies_o::value_c::
                                                operator=(const dl_ueassociated_lp_pa_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::routing_id:
      c.set(other.c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }

  return *this;
}
void dl_ueassociated_lp_pa_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::routing_id:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::lp_pa_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::routing_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::routing_id:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::lp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_lp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string dl_ueassociated_lp_pa_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "INTEGER (0..255)", "OCTET STRING"};
  return convert_enum_idx(options, 4, value, "dl_ueassociated_lp_pa_transport_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<dl_ueassociated_lp_pa_transport_ies_o>;

dl_ueassociated_lp_pa_transport_ies_container::dl_ueassociated_lp_pa_transport_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  routing_id(148, crit_e::reject),
  lp_pa_pdu(147, crit_e::reject)
{}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(routing_id.pack(bref));
  HANDLE_CODE(lp_pa_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<dl_ueassociated_lp_pa_transport_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 148:
        nof_mandatory_ies--;
        routing_id.id    = c.id;
        routing_id.crit  = c.crit;
        routing_id.value = c.value.routing_id();
        break;
      case 147:
        nof_mandatory_ies--;
        lp_pa_pdu.id    = c.id;
        lp_pa_pdu.crit  = c.crit;
        lp_pa_pdu.value = c.value.lp_pa_pdu();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_ueassociated_lp_pa_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  routing_id.to_json(j);
  j.write_fieldname("");
  lp_pa_pdu.to_json(j);
  j.end_obj();
}

// DownlinkUEAssociatedLPPaTransport ::= SEQUENCE
SRSASN_CODE dl_ueassociated_lp_pa_transport_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_lp_pa_transport_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void dl_ueassociated_lp_pa_transport_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABAdmittedItem ::= SEQUENCE
SRSASN_CODE erab_admitted_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.pack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.pack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.pack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.pack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.pack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_admitted_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dl_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(dl_g_tp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ul_transport_layer_address_present, 1));
  HANDLE_CODE(bref.unpack(ul_gtp_teid_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (dl_transport_layer_address_present) {
    HANDLE_CODE(dl_transport_layer_address.unpack(bref));
  }
  if (dl_g_tp_teid_present) {
    HANDLE_CODE(dl_g_tp_teid.unpack(bref));
  }
  if (ul_transport_layer_address_present) {
    HANDLE_CODE(ul_transport_layer_address.unpack(bref));
  }
  if (ul_gtp_teid_present) {
    HANDLE_CODE(ul_gtp_teid.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_admitted_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (dl_transport_layer_address_present) {
    j.write_str("dL-transportLayerAddress", dl_transport_layer_address.to_string());
  }
  if (dl_g_tp_teid_present) {
    j.write_str("dL-gTP-TEID", dl_g_tp_teid.to_string());
  }
  if (ul_transport_layer_address_present) {
    j.write_str("uL-TransportLayerAddress", ul_transport_layer_address.to_string());
  }
  if (ul_gtp_teid_present) {
    j.write_str("uL-GTP-TEID", ul_gtp_teid.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABAdmittedItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_admitted_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {20};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_admitted_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 20 == id;
}
crit_e erab_admitted_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 20) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_admitted_item_ies_o::value_c erab_admitted_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 20) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_admitted_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 20) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_admitted_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABAdmittedItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_admitted_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_admitted_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_admitted_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABAdmittedItem"};
  return convert_enum_idx(options, 1, value, "erab_admitted_item_ies_o::value_c::types");
}

// E-RABFailedToResumeItemResumeReq ::= SEQUENCE
SRSASN_CODE erab_failed_to_resume_item_resume_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_failed_to_resume_item_resume_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABFailedToResumeItemResumeReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_failed_to_resume_item_resume_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {236};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_failed_to_resume_item_resume_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 236 == id;
}
crit_e erab_failed_to_resume_item_resume_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 236) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_failed_to_resume_item_resume_req_ies_o::value_c
erab_failed_to_resume_item_resume_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 236) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_failed_to_resume_item_resume_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 236) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_failed_to_resume_item_resume_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABFailedToResumeItemResumeReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_failed_to_resume_item_resume_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_failed_to_resume_item_resume_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABFailedToResumeItemResumeReq"};
  return convert_enum_idx(options, 1, value, "erab_failed_to_resume_item_resume_req_ies_o::value_c::types");
}

// E-RABFailedToResumeItemResumeRes ::= SEQUENCE
SRSASN_CODE erab_failed_to_resume_item_resume_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_failed_to_resume_item_resume_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABFailedToResumeItemResumeResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_failed_to_resume_item_resume_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {238};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_failed_to_resume_item_resume_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 238 == id;
}
crit_e erab_failed_to_resume_item_resume_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 238) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_failed_to_resume_item_resume_res_ies_o::value_c
erab_failed_to_resume_item_resume_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 238) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_failed_to_resume_item_resume_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 238) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_failed_to_resume_item_resume_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABFailedToResumeItemResumeRes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_failed_to_resume_item_resume_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_resume_item_resume_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_failed_to_resume_item_resume_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABFailedToResumeItemResumeRes"};
  return convert_enum_idx(options, 1, value, "erab_failed_to_resume_item_resume_res_ies_o::value_c::types");
}

// E-RABFailedToSetupItemHOReqAck ::= SEQUENCE
SRSASN_CODE erab_failed_to_setup_item_ho_req_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failed_to_setup_item_ho_req_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_failed_to_setup_item_ho_req_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABFailedtoSetupItemHOReqAckIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_failedto_setup_item_ho_req_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {21};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_failedto_setup_item_ho_req_ack_ies_o::is_id_valid(const uint32_t& id)
{
  return 21 == id;
}
crit_e erab_failedto_setup_item_ho_req_ack_ies_o::get_crit(const uint32_t& id)
{
  if (id == 21) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_failedto_setup_item_ho_req_ack_ies_o::value_c
erab_failedto_setup_item_ho_req_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 21) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_failedto_setup_item_ho_req_ack_ies_o::get_presence(const uint32_t& id)
{
  if (id == 21) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_failedto_setup_item_ho_req_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABFailedToSetupItemHOReqAck");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_failedto_setup_item_ho_req_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_failedto_setup_item_ho_req_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_failedto_setup_item_ho_req_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABFailedToSetupItemHOReqAck"};
  return convert_enum_idx(options, 1, value, "erab_failedto_setup_item_ho_req_ack_ies_o::value_c::types");
}

// DL-Forwarding ::= ENUMERATED
std::string dl_forwarding_opts::to_string() const
{
  static const char* options[] = {"dL-Forwarding-proposed"};
  return convert_enum_idx(options, 1, value, "dl_forwarding_e");
}

// E-RABInformationListItem ::= SEQUENCE
SRSASN_CODE erab_info_list_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dl_forwarding_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_forwarding_present) {
    HANDLE_CODE(dl_forwarding.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_info_list_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dl_forwarding_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dl_forwarding_present) {
    HANDLE_CODE(dl_forwarding.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_info_list_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (dl_forwarding_present) {
    j.write_str("dL-Forwarding", "dL-Forwarding-proposed");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABInformationListIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_info_list_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {78};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_info_list_ies_o::is_id_valid(const uint32_t& id)
{
  return 78 == id;
}
crit_e erab_info_list_ies_o::get_crit(const uint32_t& id)
{
  if (id == 78) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_info_list_ies_o::value_c erab_info_list_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 78) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_info_list_ies_o::get_presence(const uint32_t& id)
{
  if (id == 78) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_info_list_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABInformationListItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_info_list_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_info_list_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_info_list_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABInformationListItem"};
  return convert_enum_idx(options, 1, value, "erab_info_list_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_info_list_ies_o>;

// E-RABItem ::= SEQUENCE
SRSASN_CODE erab_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {35};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 35 == id;
}
crit_e erab_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 35) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_item_ies_o::value_c erab_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 35) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 35) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABItem"};
  return convert_enum_idx(options, 1, value, "erab_item_ies_o::value_c::types");
}

// GBR-QosInformation ::= SEQUENCE
SRSASN_CODE gbr_qos_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_maximum_bitrate_dl, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, erab_maximum_bitrate_ul, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, erab_guaranteed_bitrate_dl, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, erab_guaranteed_bitrate_ul, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gbr_qos_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_maximum_bitrate_dl, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(erab_maximum_bitrate_ul, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(erab_guaranteed_bitrate_dl, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(erab_guaranteed_bitrate_ul, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gbr_qos_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-MaximumBitrateDL", erab_maximum_bitrate_dl);
  j.write_int("e-RAB-MaximumBitrateUL", erab_maximum_bitrate_ul);
  j.write_int("e-RAB-GuaranteedBitrateDL", erab_guaranteed_bitrate_dl);
  j.write_int("e-RAB-GuaranteedBitrateUL", erab_guaranteed_bitrate_ul);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABLevelQoSParameters ::= SEQUENCE
SRSASN_CODE erab_level_qos_params_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(gbr_qos_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, qci, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(alloc_retention_prio.pack(bref));
  if (gbr_qos_info_present) {
    HANDLE_CODE(gbr_qos_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_level_qos_params_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(gbr_qos_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(qci, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(alloc_retention_prio.unpack(bref));
  if (gbr_qos_info_present) {
    HANDLE_CODE(gbr_qos_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_level_qos_params_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("qCI", qci);
  j.write_fieldname("allocationRetentionPriority");
  alloc_retention_prio.to_json(j);
  if (gbr_qos_info_present) {
    j.write_fieldname("gbrQosInformation");
    gbr_qos_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_item_ies_o>;

// E-RABModifyItemBearerModConf ::= SEQUENCE
SRSASN_CODE erab_modify_item_bearer_mod_conf_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_conf_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_modify_item_bearer_mod_conf_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyItemBearerModConfIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_item_bearer_mod_conf_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {204};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_modify_item_bearer_mod_conf_ies_o::is_id_valid(const uint32_t& id)
{
  return 204 == id;
}
crit_e erab_modify_item_bearer_mod_conf_ies_o::get_crit(const uint32_t& id)
{
  if (id == 204) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_modify_item_bearer_mod_conf_ies_o::value_c erab_modify_item_bearer_mod_conf_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 204) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_item_bearer_mod_conf_ies_o::get_presence(const uint32_t& id)
{
  if (id == 204) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_modify_item_bearer_mod_conf_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABModifyItemBearerModConf");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_modify_item_bearer_mod_conf_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_conf_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_modify_item_bearer_mod_conf_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABModifyItemBearerModConf"};
  return convert_enum_idx(options, 1, value, "erab_modify_item_bearer_mod_conf_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_modify_item_bearer_mod_conf_ies_o>;

// E-RABModificationConfirmIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_mod_confirm_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 203, 205, 210, 58, 146};
  return map_enum_number(options, 7, idx, "id");
}
bool erab_mod_confirm_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 203, 205, 210, 58, 146};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_mod_confirm_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 203:
      return crit_e::ignore;
    case 205:
      return crit_e::ignore;
    case 210:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 146:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_mod_confirm_ies_o::value_c erab_mod_confirm_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 203:
      ret.set(value_c::types::erab_modify_list_bearer_mod_conf);
      break;
    case 205:
      ret.set(value_c::types::erab_failed_to_modify_list_bearer_mod_conf);
      break;
    case 210:
      ret.set(value_c::types::erab_to_be_released_list_bearer_mod_conf);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 146:
      ret.set(value_c::types::csg_membership_status);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_mod_confirm_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 203:
      return presence_e::optional;
    case 205:
      return presence_e::optional;
    case 210:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_mod_confirm_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_mod_confirm_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_modify_list_bearer_mod_conf_l& erab_mod_confirm_ies_o::value_c::erab_modify_list_bearer_mod_conf()
{
  assert_choice_type("E-RABModifyListBearerModConf", type_.to_string(), "Value");
  return c.get<erab_modify_list_bearer_mod_conf_l>();
}
erab_list_l& erab_mod_confirm_ies_o::value_c::erab_failed_to_modify_list_bearer_mod_conf()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
erab_list_l& erab_mod_confirm_ies_o::value_c::erab_to_be_released_list_bearer_mod_conf()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_mod_confirm_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
csg_membership_status_e& erab_mod_confirm_ies_o::value_c::csg_membership_status()
{
  assert_choice_type("CSGMembershipStatus", type_.to_string(), "Value");
  return c.get<csg_membership_status_e>();
}
const uint64_t& erab_mod_confirm_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_mod_confirm_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_modify_list_bearer_mod_conf_l& erab_mod_confirm_ies_o::value_c::erab_modify_list_bearer_mod_conf() const
{
  assert_choice_type("E-RABModifyListBearerModConf", type_.to_string(), "Value");
  return c.get<erab_modify_list_bearer_mod_conf_l>();
}
const erab_list_l& erab_mod_confirm_ies_o::value_c::erab_failed_to_modify_list_bearer_mod_conf() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const erab_list_l& erab_mod_confirm_ies_o::value_c::erab_to_be_released_list_bearer_mod_conf() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_mod_confirm_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const csg_membership_status_e& erab_mod_confirm_ies_o::value_c::csg_membership_status() const
{
  assert_choice_type("CSGMembershipStatus", type_.to_string(), "Value");
  return c.get<csg_membership_status_e>();
}
void erab_mod_confirm_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_modify_list_bearer_mod_conf:
      c.destroy<erab_modify_list_bearer_mod_conf_l>();
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.destroy<erab_list_l>();
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void erab_mod_confirm_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_modify_list_bearer_mod_conf:
      c.init<erab_modify_list_bearer_mod_conf_l>();
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.init<erab_list_l>();
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::csg_membership_status:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }
}
erab_mod_confirm_ies_o::value_c::value_c(const erab_mod_confirm_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_conf:
      c.init(other.c.get<erab_modify_list_bearer_mod_conf_l>());
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::csg_membership_status:
      c.init(other.c.get<csg_membership_status_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }
}
erab_mod_confirm_ies_o::value_c& erab_mod_confirm_ies_o::value_c::
                                 operator=(const erab_mod_confirm_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_conf:
      c.set(other.c.get<erab_modify_list_bearer_mod_conf_l>());
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::csg_membership_status:
      c.set(other.c.get<csg_membership_status_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }

  return *this;
}
void erab_mod_confirm_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_conf:
      j.start_array("E-RABModifyListBearerModConf");
      for (const auto& e1 : c.get<erab_modify_list_bearer_mod_conf_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::csg_membership_status:
      j.write_str("CSGMembershipStatus", c.get<csg_membership_status_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_mod_confirm_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_conf:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_modify_list_bearer_mod_conf_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_confirm_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_conf:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_modify_list_bearer_mod_conf_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_modify_list_bearer_mod_conf:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::erab_to_be_released_list_bearer_mod_conf:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_confirm_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_mod_confirm_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABModifyListBearerModConf",
                                  "E-RABList",
                                  "E-RABList",
                                  "CriticalityDiagnostics",
                                  "CSGMembershipStatus"};
  return convert_enum_idx(options, 7, value, "erab_mod_confirm_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_mod_confirm_ies_o>;

erab_mod_confirm_ies_container::erab_mod_confirm_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_modify_list_bearer_mod_conf(203, crit_e::ignore),
  erab_failed_to_modify_list_bearer_mod_conf(205, crit_e::ignore),
  erab_to_be_released_list_bearer_mod_conf(210, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  csg_membership_status(146, crit_e::ignore)
{}
SRSASN_CODE erab_mod_confirm_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_modify_list_bearer_mod_conf_present ? 1 : 0;
  nof_ies += erab_failed_to_modify_list_bearer_mod_conf_present ? 1 : 0;
  nof_ies += erab_to_be_released_list_bearer_mod_conf_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += csg_membership_status_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_modify_list_bearer_mod_conf_present) {
    HANDLE_CODE(erab_modify_list_bearer_mod_conf.pack(bref));
  }
  if (erab_failed_to_modify_list_bearer_mod_conf_present) {
    HANDLE_CODE(erab_failed_to_modify_list_bearer_mod_conf.pack(bref));
  }
  if (erab_to_be_released_list_bearer_mod_conf_present) {
    HANDLE_CODE(erab_to_be_released_list_bearer_mod_conf.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (csg_membership_status_present) {
    HANDLE_CODE(csg_membership_status.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_confirm_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_mod_confirm_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 203:
        erab_modify_list_bearer_mod_conf_present = true;
        erab_modify_list_bearer_mod_conf.id      = c.id;
        erab_modify_list_bearer_mod_conf.crit    = c.crit;
        erab_modify_list_bearer_mod_conf.value   = c.value.erab_modify_list_bearer_mod_conf();
        break;
      case 205:
        erab_failed_to_modify_list_bearer_mod_conf_present = true;
        erab_failed_to_modify_list_bearer_mod_conf.id      = c.id;
        erab_failed_to_modify_list_bearer_mod_conf.crit    = c.crit;
        erab_failed_to_modify_list_bearer_mod_conf.value   = c.value.erab_failed_to_modify_list_bearer_mod_conf();
        break;
      case 210:
        erab_to_be_released_list_bearer_mod_conf_present = true;
        erab_to_be_released_list_bearer_mod_conf.id      = c.id;
        erab_to_be_released_list_bearer_mod_conf.crit    = c.crit;
        erab_to_be_released_list_bearer_mod_conf.value   = c.value.erab_to_be_released_list_bearer_mod_conf();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      case 146:
        csg_membership_status_present = true;
        csg_membership_status.id      = c.id;
        csg_membership_status.crit    = c.crit;
        csg_membership_status.value   = c.value.csg_membership_status();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_mod_confirm_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_modify_list_bearer_mod_conf_present) {
    j.write_fieldname("");
    erab_modify_list_bearer_mod_conf.to_json(j);
  }
  if (erab_failed_to_modify_list_bearer_mod_conf_present) {
    j.write_fieldname("");
    erab_failed_to_modify_list_bearer_mod_conf.to_json(j);
  }
  if (erab_to_be_released_list_bearer_mod_conf_present) {
    j.write_fieldname("");
    erab_to_be_released_list_bearer_mod_conf.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (csg_membership_status_present) {
    j.write_fieldname("");
    csg_membership_status.to_json(j);
  }
  j.end_obj();
}

// E-RABModificationConfirm ::= SEQUENCE
SRSASN_CODE erab_mod_confirm_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_confirm_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_mod_confirm_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABNotToBeModifiedItemBearerModInd ::= SEQUENCE
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(dl_gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(dl_gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_not_to_be_modified_item_bearer_mod_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("dL-GTP-TEID", dl_gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeModifiedItemBearerModInd ::= SEQUENCE
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(dl_gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(dl_gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_modified_item_bearer_mod_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("dL-GTP-TEID", dl_gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABNotToBeModifiedItemBearerModIndIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_not_to_be_modified_item_bearer_mod_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {202};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_not_to_be_modified_item_bearer_mod_ind_ies_o::is_id_valid(const uint32_t& id)
{
  return 202 == id;
}
crit_e erab_not_to_be_modified_item_bearer_mod_ind_ies_o::get_crit(const uint32_t& id)
{
  if (id == 202) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c
erab_not_to_be_modified_item_bearer_mod_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 202) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_not_to_be_modified_item_bearer_mod_ind_ies_o::get_presence(const uint32_t& id)
{
  if (id == 202) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABNotToBeModifiedItemBearerModInd");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABNotToBeModifiedItemBearerModInd"};
  return convert_enum_idx(options, 1, value, "erab_not_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types");
}

// E-RABToBeModifiedItemBearerModIndIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_modified_item_bearer_mod_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {200};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_modified_item_bearer_mod_ind_ies_o::is_id_valid(const uint32_t& id)
{
  return 200 == id;
}
crit_e erab_to_be_modified_item_bearer_mod_ind_ies_o::get_crit(const uint32_t& id)
{
  if (id == 200) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c
erab_to_be_modified_item_bearer_mod_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 200) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_modified_item_bearer_mod_ind_ies_o::get_presence(const uint32_t& id)
{
  if (id == 200) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeModifiedItemBearerModInd");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeModifiedItemBearerModInd"};
  return convert_enum_idx(options, 1, value, "erab_to_be_modified_item_bearer_mod_ind_ies_o::value_c::types");
}

// TunnelInformation ::= SEQUENCE
SRSASN_CODE tunnel_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(udp_port_num_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(transport_layer_address.pack(bref));
  if (udp_port_num_present) {
    HANDLE_CODE(udp_port_num.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tunnel_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(udp_port_num_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(transport_layer_address.unpack(bref));
  if (udp_port_num_present) {
    HANDLE_CODE(udp_port_num.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tunnel_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  if (udp_port_num_present) {
    j.write_str("uDP-Port-Number", udp_port_num.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModificationIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_mod_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 199, 201, 226, 176};
  return map_enum_number(options, 6, idx, "id");
}
bool erab_mod_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 199, 201, 226, 176};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_mod_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 199:
      return crit_e::reject;
    case 201:
      return crit_e::reject;
    case 226:
      return crit_e::reject;
    case 176:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_mod_ind_ies_o::value_c erab_mod_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 199:
      ret.set(value_c::types::erab_to_be_modified_list_bearer_mod_ind);
      break;
    case 201:
      ret.set(value_c::types::erab_not_to_be_modified_list_bearer_mod_ind);
      break;
    case 226:
      ret.set(value_c::types::csg_membership_info);
      break;
    case 176:
      ret.set(value_c::types::tunnel_info_for_bbf);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_mod_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 199:
      return presence_e::mandatory;
    case 201:
      return presence_e::optional;
    case 226:
      return presence_e::optional;
    case 176:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_mod_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_mod_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_to_be_modified_list_bearer_mod_ind()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABToBeModifiedItemBearerModIndIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_not_to_be_modified_list_bearer_mod_ind()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABNotToBeModifiedItemBearerModIndIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
csg_membership_info_s& erab_mod_ind_ies_o::value_c::csg_membership_info()
{
  assert_choice_type("CSGMembershipInfo", type_.to_string(), "Value");
  return c.get<csg_membership_info_s>();
}
tunnel_info_s& erab_mod_ind_ies_o::value_c::tunnel_info_for_bbf()
{
  assert_choice_type("TunnelInformation", type_.to_string(), "Value");
  return c.get<tunnel_info_s>();
}
const uint64_t& erab_mod_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_mod_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_to_be_modified_list_bearer_mod_ind() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABToBeModifiedItemBearerModIndIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
const erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o>&
erab_mod_ind_ies_o::value_c::erab_not_to_be_modified_list_bearer_mod_ind() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABNotToBeModifiedItemBearerModIndIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
}
const csg_membership_info_s& erab_mod_ind_ies_o::value_c::csg_membership_info() const
{
  assert_choice_type("CSGMembershipInfo", type_.to_string(), "Value");
  return c.get<csg_membership_info_s>();
}
const tunnel_info_s& erab_mod_ind_ies_o::value_c::tunnel_info_for_bbf() const
{
  assert_choice_type("TunnelInformation", type_.to_string(), "Value");
  return c.get<tunnel_info_s>();
}
void erab_mod_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.destroy<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.destroy<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::csg_membership_info:
      c.destroy<csg_membership_info_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.destroy<tunnel_info_s>();
      break;
    default:
      break;
  }
}
void erab_mod_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.init<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.init<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >();
      break;
    case types::csg_membership_info:
      c.init<csg_membership_info_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.init<tunnel_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }
}
erab_mod_ind_ies_o::value_c::value_c(const erab_mod_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.init(other.c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.init(other.c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::csg_membership_info:
      c.init(other.c.get<csg_membership_info_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.init(other.c.get<tunnel_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }
}
erab_mod_ind_ies_o::value_c& erab_mod_ind_ies_o::value_c::operator=(const erab_mod_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      c.set(other.c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      c.set(other.c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >());
      break;
    case types::csg_membership_info:
      c.set(other.c.get<csg_membership_info_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.set(other.c.get<tunnel_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }

  return *this;
}
void erab_mod_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABToBeModifiedItemBearerModIndIEs}}");
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABNotToBeModifiedItemBearerModIndIEs}}");
      break;
    case types::csg_membership_info:
      j.write_fieldname("CSGMembershipInfo");
      c.get<csg_membership_info_s>().to_json(j);
      break;
    case types::tunnel_info_for_bbf:
      j.write_fieldname("TunnelInformation");
      c.get<tunnel_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_mod_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(pack_dyn_seq_of(
          bref, c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >(), 1, 256, true));
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(pack_dyn_seq_of(
          bref, c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >(), 1, 256, true));
      break;
    case types::csg_membership_info:
      HANDLE_CODE(c.get<csg_membership_info_s>().pack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_to_be_modified_item_bearer_mod_ind_ies_o> >(), bref, 1, 256, true));
      break;
    case types::erab_not_to_be_modified_list_bearer_mod_ind:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_not_to_be_modified_item_bearer_mod_ind_ies_o> >(), bref, 1, 256, true));
      break;
    case types::csg_membership_info:
      HANDLE_CODE(c.get<csg_membership_info_s>().unpack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_mod_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_mod_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RAB-IE-ContainerList{{E-RABToBeModifiedItemBearerModIndIEs}}",
                                  "E-RAB-IE-ContainerList{{E-RABNotToBeModifiedItemBearerModIndIEs}}",
                                  "CSGMembershipInfo",
                                  "TunnelInformation"};
  return convert_enum_idx(options, 6, value, "erab_mod_ind_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_mod_ind_ies_o>;

erab_mod_ind_ies_container::erab_mod_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  erab_to_be_modified_list_bearer_mod_ind(199, crit_e::reject),
  erab_not_to_be_modified_list_bearer_mod_ind(201, crit_e::reject),
  csg_membership_info(226, crit_e::reject),
  tunnel_info_for_bbf(176, crit_e::ignore)
{}
SRSASN_CODE erab_mod_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += erab_not_to_be_modified_list_bearer_mod_ind_present ? 1 : 0;
  nof_ies += csg_membership_info_present ? 1 : 0;
  nof_ies += tunnel_info_for_bbf_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_to_be_modified_list_bearer_mod_ind.pack(bref));
  if (erab_not_to_be_modified_list_bearer_mod_ind_present) {
    HANDLE_CODE(erab_not_to_be_modified_list_bearer_mod_ind.pack(bref));
  }
  if (csg_membership_info_present) {
    HANDLE_CODE(csg_membership_info.pack(bref));
  }
  if (tunnel_info_for_bbf_present) {
    HANDLE_CODE(tunnel_info_for_bbf.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_mod_ind_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 199:
        nof_mandatory_ies--;
        erab_to_be_modified_list_bearer_mod_ind.id    = c.id;
        erab_to_be_modified_list_bearer_mod_ind.crit  = c.crit;
        erab_to_be_modified_list_bearer_mod_ind.value = c.value.erab_to_be_modified_list_bearer_mod_ind();
        break;
      case 201:
        erab_not_to_be_modified_list_bearer_mod_ind_present = true;
        erab_not_to_be_modified_list_bearer_mod_ind.id      = c.id;
        erab_not_to_be_modified_list_bearer_mod_ind.crit    = c.crit;
        erab_not_to_be_modified_list_bearer_mod_ind.value   = c.value.erab_not_to_be_modified_list_bearer_mod_ind();
        break;
      case 226:
        csg_membership_info_present = true;
        csg_membership_info.id      = c.id;
        csg_membership_info.crit    = c.crit;
        csg_membership_info.value   = c.value.csg_membership_info();
        break;
      case 176:
        tunnel_info_for_bbf_present = true;
        tunnel_info_for_bbf.id      = c.id;
        tunnel_info_for_bbf.crit    = c.crit;
        tunnel_info_for_bbf.value   = c.value.tunnel_info_for_bbf();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_mod_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_to_be_modified_list_bearer_mod_ind.to_json(j);
  if (erab_not_to_be_modified_list_bearer_mod_ind_present) {
    j.write_fieldname("");
    erab_not_to_be_modified_list_bearer_mod_ind.to_json(j);
  }
  if (csg_membership_info_present) {
    j.write_fieldname("");
    csg_membership_info.to_json(j);
  }
  if (tunnel_info_for_bbf_present) {
    j.write_fieldname("");
    tunnel_info_for_bbf.to_json(j);
  }
  j.end_obj();
}

// E-RABModificationIndication ::= SEQUENCE
SRSASN_CODE erab_mod_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_mod_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_mod_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABModifyItemBearerModRes ::= SEQUENCE
SRSASN_CODE erab_modify_item_bearer_mod_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_modify_item_bearer_mod_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyItemBearerModResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_item_bearer_mod_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {37};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_modify_item_bearer_mod_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 37 == id;
}
crit_e erab_modify_item_bearer_mod_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 37) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_modify_item_bearer_mod_res_ies_o::value_c erab_modify_item_bearer_mod_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 37) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_item_bearer_mod_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 37) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_modify_item_bearer_mod_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABModifyItemBearerModRes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_modify_item_bearer_mod_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_item_bearer_mod_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_modify_item_bearer_mod_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABModifyItemBearerModRes"};
  return convert_enum_idx(options, 1, value, "erab_modify_item_bearer_mod_res_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_modify_item_bearer_mod_res_ies_o>;

// TransportInformation ::= SEQUENCE
SRSASN_CODE transport_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(ul_gtp_teid.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE transport_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(ul_gtp_teid.unpack(bref));

  return SRSASN_SUCCESS;
}
void transport_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("uL-GTP-TEID", ul_gtp_teid.to_string());
  j.end_obj();
}

// E-RABToBeModifyItemBearerModReqExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_modify_item_bearer_mod_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {185};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_modify_item_bearer_mod_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 185 == id;
}
crit_e erab_to_be_modify_item_bearer_mod_req_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 185) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c
erab_to_be_modify_item_bearer_mod_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 185) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_modify_item_bearer_mod_req_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 185) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("TransportInformation");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"TransportInformation"};
  return convert_enum_idx(options, 1, value, "erab_to_be_modify_item_bearer_mod_req_ext_ies_o::ext_c::types");
}

// E-RABToBeModifiedItemBearerModReq ::= SEQUENCE
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  HANDLE_CODE(nas_pdu.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_modified_item_bearer_mod_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("e-RABLevelQoSParameters");
  erab_level_qos_params.to_json(j);
  j.write_str("nAS-PDU", nas_pdu.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// E-RABToBeModifiedItemBearerModReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_modified_item_bearer_mod_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {36};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_modified_item_bearer_mod_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 36 == id;
}
crit_e erab_to_be_modified_item_bearer_mod_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 36) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_modified_item_bearer_mod_req_ies_o::value_c
erab_to_be_modified_item_bearer_mod_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 36) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_modified_item_bearer_mod_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 36) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeModifiedItemBearerModReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeModifiedItemBearerModReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_modified_item_bearer_mod_req_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_to_be_modified_item_bearer_mod_req_ies_o>;

// UEAggregateMaximumBitrate ::= SEQUENCE
SRSASN_CODE ue_aggregate_maximum_bitrate_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, ueaggregate_maximum_bit_rate_dl, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(pack_integer(bref, ueaggregate_maximum_bit_rate_ul, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_aggregate_maximum_bitrate_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(ueaggregate_maximum_bit_rate_dl, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  HANDLE_CODE(unpack_integer(ueaggregate_maximum_bit_rate_ul, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_aggregate_maximum_bitrate_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("uEaggregateMaximumBitRateDL", ueaggregate_maximum_bit_rate_dl);
  j.write_int("uEaggregateMaximumBitRateUL", ueaggregate_maximum_bit_rate_ul);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 66, 30};
  return map_enum_number(options, 4, idx, "id");
}
bool erab_modify_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 66, 30};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_modify_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 30:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_modify_request_ies_o::value_c erab_modify_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 30:
      ret.set(value_c::types::erab_to_be_modified_list_bearer_mod_req);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 30:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_modify_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_modify_request_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& erab_modify_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_to_be_modified_list_bearer_mod_req_l& erab_modify_request_ies_o::value_c::erab_to_be_modified_list_bearer_mod_req()
{
  assert_choice_type("E-RABToBeModifiedListBearerModReq", type_.to_string(), "Value");
  return c.get<erab_to_be_modified_list_bearer_mod_req_l>();
}
const uint64_t& erab_modify_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_modify_request_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& erab_modify_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_to_be_modified_list_bearer_mod_req_l&
erab_modify_request_ies_o::value_c::erab_to_be_modified_list_bearer_mod_req() const
{
  assert_choice_type("E-RABToBeModifiedListBearerModReq", type_.to_string(), "Value");
  return c.get<erab_to_be_modified_list_bearer_mod_req_l>();
}
void erab_modify_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.destroy<erab_to_be_modified_list_bearer_mod_req_l>();
      break;
    default:
      break;
  }
}
void erab_modify_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.init<erab_to_be_modified_list_bearer_mod_req_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }
}
erab_modify_request_ies_o::value_c::value_c(const erab_modify_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.init(other.c.get<erab_to_be_modified_list_bearer_mod_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }
}
erab_modify_request_ies_o::value_c& erab_modify_request_ies_o::value_c::
                                    operator=(const erab_modify_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      c.set(other.c.get<erab_to_be_modified_list_bearer_mod_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }

  return *this;
}
void erab_modify_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      j.start_array("E-RABToBeModifiedListBearerModReq");
      for (const auto& e1 : c.get<erab_to_be_modified_list_bearer_mod_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_modify_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_to_be_modified_list_bearer_mod_req_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_modified_list_bearer_mod_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_to_be_modified_list_bearer_mod_req_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_modify_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "UEAggregateMaximumBitrate",
                                  "E-RABToBeModifiedListBearerModReq"};
  return convert_enum_idx(options, 4, value, "erab_modify_request_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_modify_request_ies_o>;

erab_modify_request_ies_container::erab_modify_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_modified_list_bearer_mod_req(30, crit_e::reject)
{}
SRSASN_CODE erab_modify_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ueaggregate_maximum_bitrate_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ueaggregate_maximum_bitrate_present) {
    HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  }
  HANDLE_CODE(erab_to_be_modified_list_bearer_mod_req.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_modify_request_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 66:
        ueaggregate_maximum_bitrate_present = true;
        ueaggregate_maximum_bitrate.id      = c.id;
        ueaggregate_maximum_bitrate.crit    = c.crit;
        ueaggregate_maximum_bitrate.value   = c.value.ueaggregate_maximum_bitrate();
        break;
      case 30:
        nof_mandatory_ies--;
        erab_to_be_modified_list_bearer_mod_req.id    = c.id;
        erab_to_be_modified_list_bearer_mod_req.crit  = c.crit;
        erab_to_be_modified_list_bearer_mod_req.value = c.value.erab_to_be_modified_list_bearer_mod_req();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_modify_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ueaggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ueaggregate_maximum_bitrate.to_json(j);
  }
  j.write_fieldname("");
  erab_to_be_modified_list_bearer_mod_req.to_json(j);
  j.end_obj();
}

// E-RABModifyRequest ::= SEQUENCE
SRSASN_CODE erab_modify_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_modify_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABModifyResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_modify_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 31, 32, 58};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_modify_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 31, 32, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_modify_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 31:
      return crit_e::ignore;
    case 32:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_modify_resp_ies_o::value_c erab_modify_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 31:
      ret.set(value_c::types::erab_modify_list_bearer_mod_res);
      break;
    case 32:
      ret.set(value_c::types::erab_failed_to_modify_list);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_modify_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 31:
      return presence_e::optional;
    case 32:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_modify_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_modify_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_modify_list_bearer_mod_res_l& erab_modify_resp_ies_o::value_c::erab_modify_list_bearer_mod_res()
{
  assert_choice_type("E-RABModifyListBearerModRes", type_.to_string(), "Value");
  return c.get<erab_modify_list_bearer_mod_res_l>();
}
erab_list_l& erab_modify_resp_ies_o::value_c::erab_failed_to_modify_list()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_modify_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& erab_modify_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_modify_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_modify_list_bearer_mod_res_l& erab_modify_resp_ies_o::value_c::erab_modify_list_bearer_mod_res() const
{
  assert_choice_type("E-RABModifyListBearerModRes", type_.to_string(), "Value");
  return c.get<erab_modify_list_bearer_mod_res_l>();
}
const erab_list_l& erab_modify_resp_ies_o::value_c::erab_failed_to_modify_list() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_modify_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void erab_modify_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_modify_list_bearer_mod_res:
      c.destroy<erab_modify_list_bearer_mod_res_l>();
      break;
    case types::erab_failed_to_modify_list:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void erab_modify_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_modify_list_bearer_mod_res:
      c.init<erab_modify_list_bearer_mod_res_l>();
      break;
    case types::erab_failed_to_modify_list:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }
}
erab_modify_resp_ies_o::value_c::value_c(const erab_modify_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_res:
      c.init(other.c.get<erab_modify_list_bearer_mod_res_l>());
      break;
    case types::erab_failed_to_modify_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }
}
erab_modify_resp_ies_o::value_c& erab_modify_resp_ies_o::value_c::
                                 operator=(const erab_modify_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_res:
      c.set(other.c.get<erab_modify_list_bearer_mod_res_l>());
      break;
    case types::erab_failed_to_modify_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }

  return *this;
}
void erab_modify_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_modify_list_bearer_mod_res:
      j.start_array("E-RABModifyListBearerModRes");
      for (const auto& e1 : c.get<erab_modify_list_bearer_mod_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_modify_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_modify_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_modify_list_bearer_mod_res_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_modify_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_modify_list_bearer_mod_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_modify_list_bearer_mod_res_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_modify_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_modify_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_modify_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABModifyListBearerModRes",
                                  "E-RABList",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "erab_modify_resp_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_modify_resp_ies_o>;

erab_modify_resp_ies_container::erab_modify_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_modify_list_bearer_mod_res(31, crit_e::ignore),
  erab_failed_to_modify_list(32, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE erab_modify_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_modify_list_bearer_mod_res_present ? 1 : 0;
  nof_ies += erab_failed_to_modify_list_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_modify_list_bearer_mod_res_present) {
    HANDLE_CODE(erab_modify_list_bearer_mod_res.pack(bref));
  }
  if (erab_failed_to_modify_list_present) {
    HANDLE_CODE(erab_failed_to_modify_list.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_modify_resp_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 31:
        erab_modify_list_bearer_mod_res_present = true;
        erab_modify_list_bearer_mod_res.id      = c.id;
        erab_modify_list_bearer_mod_res.crit    = c.crit;
        erab_modify_list_bearer_mod_res.value   = c.value.erab_modify_list_bearer_mod_res();
        break;
      case 32:
        erab_failed_to_modify_list_present = true;
        erab_failed_to_modify_list.id      = c.id;
        erab_failed_to_modify_list.crit    = c.crit;
        erab_failed_to_modify_list.value   = c.value.erab_failed_to_modify_list();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_modify_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_modify_list_bearer_mod_res_present) {
    j.write_fieldname("");
    erab_modify_list_bearer_mod_res.to_json(j);
  }
  if (erab_failed_to_modify_list_present) {
    j.write_fieldname("");
    erab_failed_to_modify_list.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// E-RABModifyResponse ::= SEQUENCE
SRSASN_CODE erab_modify_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_modify_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_modify_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABReleaseCommandIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_cmd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 66, 33, 26};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_release_cmd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 66, 33, 26};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_release_cmd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 33:
      return crit_e::ignore;
    case 26:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_release_cmd_ies_o::value_c erab_release_cmd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 33:
      ret.set(value_c::types::erab_to_be_released_list);
      break;
    case 26:
      ret.set(value_c::types::nas_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_cmd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 33:
      return presence_e::mandatory;
    case 26:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_release_cmd_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_release_cmd_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& erab_release_cmd_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_list_l& erab_release_cmd_ies_o::value_c::erab_to_be_released_list()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
unbounded_octstring<true>& erab_release_cmd_ies_o::value_c::nas_pdu()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& erab_release_cmd_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_release_cmd_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& erab_release_cmd_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_list_l& erab_release_cmd_ies_o::value_c::erab_to_be_released_list() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const unbounded_octstring<true>& erab_release_cmd_ies_o::value_c::nas_pdu() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
void erab_release_cmd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_released_list:
      c.destroy<erab_list_l>();
      break;
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void erab_release_cmd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_released_list:
      c.init<erab_list_l>();
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }
}
erab_release_cmd_ies_o::value_c::value_c(const erab_release_cmd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_released_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }
}
erab_release_cmd_ies_o::value_c& erab_release_cmd_ies_o::value_c::
                                 operator=(const erab_release_cmd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_released_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }

  return *this;
}
void erab_release_cmd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_released_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_release_cmd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_released_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_cmd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_released_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_cmd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_release_cmd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "UEAggregateMaximumBitrate", "E-RABList", "OCTET STRING"};
  return convert_enum_idx(options, 5, value, "erab_release_cmd_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_release_cmd_ies_o>;

erab_release_cmd_ies_container::erab_release_cmd_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_released_list(33, crit_e::ignore),
  nas_pdu(26, crit_e::ignore)
{}
SRSASN_CODE erab_release_cmd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ueaggregate_maximum_bitrate_present ? 1 : 0;
  nof_ies += nas_pdu_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ueaggregate_maximum_bitrate_present) {
    HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  }
  HANDLE_CODE(erab_to_be_released_list.pack(bref));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_cmd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_release_cmd_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 66:
        ueaggregate_maximum_bitrate_present = true;
        ueaggregate_maximum_bitrate.id      = c.id;
        ueaggregate_maximum_bitrate.crit    = c.crit;
        ueaggregate_maximum_bitrate.value   = c.value.ueaggregate_maximum_bitrate();
        break;
      case 33:
        nof_mandatory_ies--;
        erab_to_be_released_list.id    = c.id;
        erab_to_be_released_list.crit  = c.crit;
        erab_to_be_released_list.value = c.value.erab_to_be_released_list();
        break;
      case 26:
        nas_pdu_present = true;
        nas_pdu.id      = c.id;
        nas_pdu.crit    = c.crit;
        nas_pdu.value   = c.value.nas_pdu();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_release_cmd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ueaggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ueaggregate_maximum_bitrate.to_json(j);
  }
  j.write_fieldname("");
  erab_to_be_released_list.to_json(j);
  if (nas_pdu_present) {
    j.write_fieldname("");
    nas_pdu.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseCommand ::= SEQUENCE
SRSASN_CODE erab_release_cmd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_cmd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_release_cmd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// UserLocationInformation ::= SEQUENCE
SRSASN_CODE user_location_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutran_cgi.unpack(bref));
  HANDLE_CODE(tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void user_location_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eutran-cgi");
  eutran_cgi.to_json(j);
  j.write_fieldname("tai");
  tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 110, 189};
  return map_enum_number(options, 4, idx, "id");
}
bool erab_release_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 110, 189};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_release_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 110:
      return crit_e::ignore;
    case 189:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_release_ind_ies_o::value_c erab_release_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 110:
      ret.set(value_c::types::erab_released_list);
      break;
    case 189:
      ret.set(value_c::types::user_location_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 110:
      return presence_e::mandatory;
    case 189:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_release_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_release_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_list_l& erab_release_ind_ies_o::value_c::erab_released_list()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
user_location_info_s& erab_release_ind_ies_o::value_c::user_location_info()
{
  assert_choice_type("UserLocationInformation", type_.to_string(), "Value");
  return c.get<user_location_info_s>();
}
const uint64_t& erab_release_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_release_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_list_l& erab_release_ind_ies_o::value_c::erab_released_list() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const user_location_info_s& erab_release_ind_ies_o::value_c::user_location_info() const
{
  assert_choice_type("UserLocationInformation", type_.to_string(), "Value");
  return c.get<user_location_info_s>();
}
void erab_release_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_released_list:
      c.destroy<erab_list_l>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_s>();
      break;
    default:
      break;
  }
}
void erab_release_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_released_list:
      c.init<erab_list_l>();
      break;
    case types::user_location_info:
      c.init<user_location_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }
}
erab_release_ind_ies_o::value_c::value_c(const erab_release_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_released_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }
}
erab_release_ind_ies_o::value_c& erab_release_ind_ies_o::value_c::
                                 operator=(const erab_release_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_released_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }

  return *this;
}
void erab_release_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_released_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_release_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_released_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_released_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_release_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "E-RABList", "UserLocationInformation"};
  return convert_enum_idx(options, 4, value, "erab_release_ind_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_release_ind_ies_o>;

erab_release_ind_ies_container::erab_release_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  erab_released_list(110, crit_e::ignore),
  user_location_info(189, crit_e::ignore)
{}
SRSASN_CODE erab_release_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += user_location_info_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_released_list.pack(bref));
  if (user_location_info_present) {
    HANDLE_CODE(user_location_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_release_ind_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 110:
        nof_mandatory_ies--;
        erab_released_list.id    = c.id;
        erab_released_list.crit  = c.crit;
        erab_released_list.value = c.value.erab_released_list();
        break;
      case 189:
        user_location_info_present = true;
        user_location_info.id      = c.id;
        user_location_info.crit    = c.crit;
        user_location_info.value   = c.value.user_location_info();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_release_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_released_list.to_json(j);
  if (user_location_info_present) {
    j.write_fieldname("");
    user_location_info.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseIndication ::= SEQUENCE
SRSASN_CODE erab_release_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_release_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABReleaseItemBearerRelComp ::= SEQUENCE
SRSASN_CODE erab_release_item_bearer_rel_comp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_item_bearer_rel_comp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_release_item_bearer_rel_comp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseItemBearerRelCompIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_item_bearer_rel_comp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {15};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_release_item_bearer_rel_comp_ies_o::is_id_valid(const uint32_t& id)
{
  return 15 == id;
}
crit_e erab_release_item_bearer_rel_comp_ies_o::get_crit(const uint32_t& id)
{
  if (id == 15) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_release_item_bearer_rel_comp_ies_o::value_c erab_release_item_bearer_rel_comp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 15) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_item_bearer_rel_comp_ies_o::get_presence(const uint32_t& id)
{
  if (id == 15) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_release_item_bearer_rel_comp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABReleaseItemBearerRelComp");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_release_item_bearer_rel_comp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_item_bearer_rel_comp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_release_item_bearer_rel_comp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABReleaseItemBearerRelComp"};
  return convert_enum_idx(options, 1, value, "erab_release_item_bearer_rel_comp_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_release_item_bearer_rel_comp_ies_o>;

// E-RABReleaseResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_release_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 69, 34, 58, 189};
  return map_enum_number(options, 6, idx, "id");
}
bool erab_release_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 69, 34, 58, 189};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_release_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 69:
      return crit_e::ignore;
    case 34:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 189:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_release_resp_ies_o::value_c erab_release_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 69:
      ret.set(value_c::types::erab_release_list_bearer_rel_comp);
      break;
    case 34:
      ret.set(value_c::types::erab_failed_to_release_list);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 189:
      ret.set(value_c::types::user_location_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_release_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 69:
      return presence_e::optional;
    case 34:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 189:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_release_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_release_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_release_list_bearer_rel_comp_l& erab_release_resp_ies_o::value_c::erab_release_list_bearer_rel_comp()
{
  assert_choice_type("E-RABReleaseListBearerRelComp", type_.to_string(), "Value");
  return c.get<erab_release_list_bearer_rel_comp_l>();
}
erab_list_l& erab_release_resp_ies_o::value_c::erab_failed_to_release_list()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_release_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
user_location_info_s& erab_release_resp_ies_o::value_c::user_location_info()
{
  assert_choice_type("UserLocationInformation", type_.to_string(), "Value");
  return c.get<user_location_info_s>();
}
const uint64_t& erab_release_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_release_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_release_list_bearer_rel_comp_l& erab_release_resp_ies_o::value_c::erab_release_list_bearer_rel_comp() const
{
  assert_choice_type("E-RABReleaseListBearerRelComp", type_.to_string(), "Value");
  return c.get<erab_release_list_bearer_rel_comp_l>();
}
const erab_list_l& erab_release_resp_ies_o::value_c::erab_failed_to_release_list() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_release_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const user_location_info_s& erab_release_resp_ies_o::value_c::user_location_info() const
{
  assert_choice_type("UserLocationInformation", type_.to_string(), "Value");
  return c.get<user_location_info_s>();
}
void erab_release_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_release_list_bearer_rel_comp:
      c.destroy<erab_release_list_bearer_rel_comp_l>();
      break;
    case types::erab_failed_to_release_list:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_s>();
      break;
    default:
      break;
  }
}
void erab_release_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_release_list_bearer_rel_comp:
      c.init<erab_release_list_bearer_rel_comp_l>();
      break;
    case types::erab_failed_to_release_list:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::user_location_info:
      c.init<user_location_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }
}
erab_release_resp_ies_o::value_c::value_c(const erab_release_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_release_list_bearer_rel_comp:
      c.init(other.c.get<erab_release_list_bearer_rel_comp_l>());
      break;
    case types::erab_failed_to_release_list:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }
}
erab_release_resp_ies_o::value_c& erab_release_resp_ies_o::value_c::
                                  operator=(const erab_release_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_release_list_bearer_rel_comp:
      c.set(other.c.get<erab_release_list_bearer_rel_comp_l>());
      break;
    case types::erab_failed_to_release_list:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }

  return *this;
}
void erab_release_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_release_list_bearer_rel_comp:
      j.start_array("E-RABReleaseListBearerRelComp");
      for (const auto& e1 : c.get<erab_release_list_bearer_rel_comp_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_release_list:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_release_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_release_list_bearer_rel_comp:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_release_list_bearer_rel_comp_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_release_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_release_list_bearer_rel_comp:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_release_list_bearer_rel_comp_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_release_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_release_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_release_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABReleaseListBearerRelComp",
                                  "E-RABList",
                                  "CriticalityDiagnostics",
                                  "UserLocationInformation"};
  return convert_enum_idx(options, 6, value, "erab_release_resp_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_release_resp_ies_o>;

erab_release_resp_ies_container::erab_release_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_release_list_bearer_rel_comp(69, crit_e::ignore),
  erab_failed_to_release_list(34, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  user_location_info(189, crit_e::ignore)
{}
SRSASN_CODE erab_release_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_release_list_bearer_rel_comp_present ? 1 : 0;
  nof_ies += erab_failed_to_release_list_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += user_location_info_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_release_list_bearer_rel_comp_present) {
    HANDLE_CODE(erab_release_list_bearer_rel_comp.pack(bref));
  }
  if (erab_failed_to_release_list_present) {
    HANDLE_CODE(erab_failed_to_release_list.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (user_location_info_present) {
    HANDLE_CODE(user_location_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_release_resp_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 69:
        erab_release_list_bearer_rel_comp_present = true;
        erab_release_list_bearer_rel_comp.id      = c.id;
        erab_release_list_bearer_rel_comp.crit    = c.crit;
        erab_release_list_bearer_rel_comp.value   = c.value.erab_release_list_bearer_rel_comp();
        break;
      case 34:
        erab_failed_to_release_list_present = true;
        erab_failed_to_release_list.id      = c.id;
        erab_failed_to_release_list.crit    = c.crit;
        erab_failed_to_release_list.value   = c.value.erab_failed_to_release_list();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      case 189:
        user_location_info_present = true;
        user_location_info.id      = c.id;
        user_location_info.crit    = c.crit;
        user_location_info.value   = c.value.user_location_info();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_release_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_release_list_bearer_rel_comp_present) {
    j.write_fieldname("");
    erab_release_list_bearer_rel_comp.to_json(j);
  }
  if (erab_failed_to_release_list_present) {
    j.write_fieldname("");
    erab_failed_to_release_list.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (user_location_info_present) {
    j.write_fieldname("");
    user_location_info.to_json(j);
  }
  j.end_obj();
}

// E-RABReleaseResponse ::= SEQUENCE
SRSASN_CODE erab_release_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_release_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_release_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABSetupItemBearerSURes ::= SEQUENCE
SRSASN_CODE erab_setup_item_bearer_su_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_bearer_su_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_setup_item_bearer_su_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABSetupItemBearerSUResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_item_bearer_su_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {39};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_setup_item_bearer_su_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 39 == id;
}
crit_e erab_setup_item_bearer_su_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 39) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_setup_item_bearer_su_res_ies_o::value_c erab_setup_item_bearer_su_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 39) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_item_bearer_su_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 39) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_setup_item_bearer_su_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABSetupItemBearerSURes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_setup_item_bearer_su_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_bearer_su_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_setup_item_bearer_su_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABSetupItemBearerSURes"};
  return convert_enum_idx(options, 1, value, "erab_setup_item_bearer_su_res_ies_o::value_c::types");
}

// E-RABSetupItemCtxtSURes ::= SEQUENCE
SRSASN_CODE erab_setup_item_ctxt_su_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_ctxt_su_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_setup_item_ctxt_su_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABSetupItemCtxtSUResIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_item_ctxt_su_res_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {50};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_setup_item_ctxt_su_res_ies_o::is_id_valid(const uint32_t& id)
{
  return 50 == id;
}
crit_e erab_setup_item_ctxt_su_res_ies_o::get_crit(const uint32_t& id)
{
  if (id == 50) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_setup_item_ctxt_su_res_ies_o::value_c erab_setup_item_ctxt_su_res_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 50) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_item_ctxt_su_res_ies_o::get_presence(const uint32_t& id)
{
  if (id == 50) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_setup_item_ctxt_su_res_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABSetupItemCtxtSURes");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_setup_item_ctxt_su_res_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_item_ctxt_su_res_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_setup_item_ctxt_su_res_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABSetupItemCtxtSURes"};
  return convert_enum_idx(options, 1, value, "erab_setup_item_ctxt_su_res_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_setup_item_bearer_su_res_ies_o>;

template struct asn1::s1ap::protocol_ie_single_container_s<erab_setup_item_ctxt_su_res_ies_o>;

// BearerType ::= ENUMERATED
std::string bearer_type_opts::to_string() const
{
  static const char* options[] = {"non-IP"};
  return convert_enum_idx(options, 1, value, "bearer_type_e");
}

// E-RABToBeSetupItemBearerSUReqExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_setup_item_bearer_su_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {156, 183, 233};
  return map_enum_number(options, 3, idx, "id");
}
bool erab_to_be_setup_item_bearer_su_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {156, 183, 233};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_to_be_setup_item_bearer_su_req_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 156:
      return crit_e::ignore;
    case 183:
      return crit_e::ignore;
    case 233:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c
erab_to_be_setup_item_bearer_su_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 156:
      ret.set(ext_c::types::correlation_id);
      break;
    case 183:
      ret.set(ext_c::types::sipto_correlation_id);
      break;
    case 233:
      ret.set(ext_c::types::bearer_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_bearer_su_req_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 156:
      return presence_e::optional;
    case 183:
      return presence_e::optional;
    case 233:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::correlation_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::sipto_correlation_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
bearer_type_e& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::bearer_type()
{
  assert_choice_type("BearerType", type_.to_string(), "Extension");
  return c.get<bearer_type_e>();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::correlation_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::sipto_correlation_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const bearer_type_e& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::bearer_type() const
{
  assert_choice_type("BearerType", type_.to_string(), "Extension");
  return c.get<bearer_type_e>();
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    default:
      break;
  }
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::bearer_type:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::ext_c(
    const erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.init(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c& erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::
                                                      operator=(const erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.set(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }

  return *this;
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::sipto_correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::bearer_type:
      j.write_str("BearerType", "non-IP");
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING", "OCTET STRING", "BearerType"};
  return convert_enum_idx(options, 3, value, "erab_to_be_setup_item_bearer_su_req_ext_ies_o::ext_c::types");
}

template struct asn1::s1ap::protocol_ext_field_s<erab_to_be_setup_item_bearer_su_req_ext_ies_o>;

erab_to_be_setup_item_bearer_su_req_ext_ies_container::erab_to_be_setup_item_bearer_su_req_ext_ies_container() :
  correlation_id(156, crit_e::ignore),
  sipto_correlation_id(183, crit_e::ignore),
  bearer_type(233, crit_e::reject)
{}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += correlation_id_present ? 1 : 0;
  nof_ies += sipto_correlation_id_present ? 1 : 0;
  nof_ies += bearer_type_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (correlation_id_present) {
    HANDLE_CODE(correlation_id.pack(bref));
  }
  if (sipto_correlation_id_present) {
    HANDLE_CODE(sipto_correlation_id.pack(bref));
  }
  if (bearer_type_present) {
    HANDLE_CODE(bearer_type.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<erab_to_be_setup_item_bearer_su_req_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 156:
        correlation_id_present = true;
        correlation_id.id      = c.id;
        correlation_id.crit    = c.crit;
        correlation_id.ext     = c.ext_value.correlation_id();
        break;
      case 183:
        sipto_correlation_id_present = true;
        sipto_correlation_id.id      = c.id;
        sipto_correlation_id.crit    = c.crit;
        sipto_correlation_id.ext     = c.ext_value.sipto_correlation_id();
        break;
      case 233:
        bearer_type_present = true;
        bearer_type.id      = c.id;
        bearer_type.crit    = c.crit;
        bearer_type.ext     = c.ext_value.bearer_type();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_bearer_su_req_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (correlation_id_present) {
    j.write_fieldname("");
    correlation_id.to_json(j);
  }
  if (sipto_correlation_id_present) {
    j.write_fieldname("");
    sipto_correlation_id.to_json(j);
  }
  if (bearer_type_present) {
    j.write_fieldname("");
    bearer_type.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemBearerSUReq ::= SEQUENCE
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  HANDLE_CODE(nas_pdu.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_bearer_su_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("e-RABlevelQoSParameters");
  erab_level_qos_params.to_json(j);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  j.write_str("nAS-PDU", nas_pdu.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemBearerSUReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_setup_item_bearer_su_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {17};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_setup_item_bearer_su_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 17 == id;
}
crit_e erab_to_be_setup_item_bearer_su_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 17) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_setup_item_bearer_su_req_ies_o::value_c
erab_to_be_setup_item_bearer_su_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 17) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_bearer_su_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 17) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_setup_item_bearer_su_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSetupItemBearerSUReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_bearer_su_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_setup_item_bearer_su_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSetupItemBearerSUReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_setup_item_bearer_su_req_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_to_be_setup_item_bearer_su_req_ies_o>;

// E-RABSetupRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 66, 16};
  return map_enum_number(options, 4, idx, "id");
}
bool erab_setup_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 66, 16};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_setup_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 16:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_setup_request_ies_o::value_c erab_setup_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 16:
      ret.set(value_c::types::erab_to_be_setup_list_bearer_su_req);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 16:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_setup_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_setup_request_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& erab_setup_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_to_be_setup_list_bearer_su_req_l& erab_setup_request_ies_o::value_c::erab_to_be_setup_list_bearer_su_req()
{
  assert_choice_type("E-RABToBeSetupListBearerSUReq", type_.to_string(), "Value");
  return c.get<erab_to_be_setup_list_bearer_su_req_l>();
}
const uint64_t& erab_setup_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_setup_request_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& erab_setup_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_to_be_setup_list_bearer_su_req_l&
erab_setup_request_ies_o::value_c::erab_to_be_setup_list_bearer_su_req() const
{
  assert_choice_type("E-RABToBeSetupListBearerSUReq", type_.to_string(), "Value");
  return c.get<erab_to_be_setup_list_bearer_su_req_l>();
}
void erab_setup_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.destroy<erab_to_be_setup_list_bearer_su_req_l>();
      break;
    default:
      break;
  }
}
void erab_setup_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.init<erab_to_be_setup_list_bearer_su_req_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }
}
erab_setup_request_ies_o::value_c::value_c(const erab_setup_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.init(other.c.get<erab_to_be_setup_list_bearer_su_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }
}
erab_setup_request_ies_o::value_c& erab_setup_request_ies_o::value_c::
                                   operator=(const erab_setup_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      c.set(other.c.get<erab_to_be_setup_list_bearer_su_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }

  return *this;
}
void erab_setup_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      j.start_array("E-RABToBeSetupListBearerSUReq");
      for (const auto& e1 : c.get<erab_to_be_setup_list_bearer_su_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_setup_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_to_be_setup_list_bearer_su_req_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_setup_list_bearer_su_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_to_be_setup_list_bearer_su_req_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_setup_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "UEAggregateMaximumBitrate", "E-RABToBeSetupListBearerSUReq"};
  return convert_enum_idx(options, 4, value, "erab_setup_request_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_setup_request_ies_o>;

erab_setup_request_ies_container::erab_setup_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_setup_list_bearer_su_req(16, crit_e::reject)
{}
SRSASN_CODE erab_setup_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += ueaggregate_maximum_bitrate_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (ueaggregate_maximum_bitrate_present) {
    HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  }
  HANDLE_CODE(erab_to_be_setup_list_bearer_su_req.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_setup_request_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 66:
        ueaggregate_maximum_bitrate_present = true;
        ueaggregate_maximum_bitrate.id      = c.id;
        ueaggregate_maximum_bitrate.crit    = c.crit;
        ueaggregate_maximum_bitrate.value   = c.value.ueaggregate_maximum_bitrate();
        break;
      case 16:
        nof_mandatory_ies--;
        erab_to_be_setup_list_bearer_su_req.id    = c.id;
        erab_to_be_setup_list_bearer_su_req.crit  = c.crit;
        erab_to_be_setup_list_bearer_su_req.value = c.value.erab_to_be_setup_list_bearer_su_req();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_setup_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (ueaggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ueaggregate_maximum_bitrate.to_json(j);
  }
  j.write_fieldname("");
  erab_to_be_setup_list_bearer_su_req.to_json(j);
  j.end_obj();
}

// E-RABSetupRequest ::= SEQUENCE
SRSASN_CODE erab_setup_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_setup_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABSetupResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_setup_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 28, 29, 58};
  return map_enum_number(options, 5, idx, "id");
}
bool erab_setup_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 28, 29, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_setup_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 28:
      return crit_e::ignore;
    case 29:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_setup_resp_ies_o::value_c erab_setup_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 28:
      ret.set(value_c::types::erab_setup_list_bearer_su_res);
      break;
    case 29:
      ret.set(value_c::types::erab_failed_to_setup_list_bearer_su_res);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_setup_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 28:
      return presence_e::optional;
    case 29:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& erab_setup_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& erab_setup_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_setup_list_bearer_su_res_l& erab_setup_resp_ies_o::value_c::erab_setup_list_bearer_su_res()
{
  assert_choice_type("E-RABSetupListBearerSURes", type_.to_string(), "Value");
  return c.get<erab_setup_list_bearer_su_res_l>();
}
erab_list_l& erab_setup_resp_ies_o::value_c::erab_failed_to_setup_list_bearer_su_res()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& erab_setup_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& erab_setup_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& erab_setup_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_setup_list_bearer_su_res_l& erab_setup_resp_ies_o::value_c::erab_setup_list_bearer_su_res() const
{
  assert_choice_type("E-RABSetupListBearerSURes", type_.to_string(), "Value");
  return c.get<erab_setup_list_bearer_su_res_l>();
}
const erab_list_l& erab_setup_resp_ies_o::value_c::erab_failed_to_setup_list_bearer_su_res() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& erab_setup_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void erab_setup_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_setup_list_bearer_su_res:
      c.destroy<erab_setup_list_bearer_su_res_l>();
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void erab_setup_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_setup_list_bearer_su_res:
      c.init<erab_setup_list_bearer_su_res_l>();
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }
}
erab_setup_resp_ies_o::value_c::value_c(const erab_setup_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_setup_list_bearer_su_res:
      c.init(other.c.get<erab_setup_list_bearer_su_res_l>());
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }
}
erab_setup_resp_ies_o::value_c& erab_setup_resp_ies_o::value_c::operator=(const erab_setup_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_setup_list_bearer_su_res:
      c.set(other.c.get<erab_setup_list_bearer_su_res_l>());
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }

  return *this;
}
void erab_setup_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_setup_list_bearer_su_res:
      j.start_array("E-RABSetupListBearerSURes");
      for (const auto& e1 : c.get<erab_setup_list_bearer_su_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_setup_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_setup_list_bearer_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_setup_list_bearer_su_res_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_setup_list_bearer_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_setup_list_bearer_su_res_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_bearer_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_setup_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABSetupListBearerSURes",
                                  "E-RABList",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "erab_setup_resp_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<erab_setup_resp_ies_o>;

erab_setup_resp_ies_container::erab_setup_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_setup_list_bearer_su_res(28, crit_e::ignore),
  erab_failed_to_setup_list_bearer_su_res(29, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE erab_setup_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += erab_setup_list_bearer_su_res_present ? 1 : 0;
  nof_ies += erab_failed_to_setup_list_bearer_su_res_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (erab_setup_list_bearer_su_res_present) {
    HANDLE_CODE(erab_setup_list_bearer_su_res.pack(bref));
  }
  if (erab_failed_to_setup_list_bearer_su_res_present) {
    HANDLE_CODE(erab_failed_to_setup_list_bearer_su_res.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<erab_setup_resp_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 28:
        erab_setup_list_bearer_su_res_present = true;
        erab_setup_list_bearer_su_res.id      = c.id;
        erab_setup_list_bearer_su_res.crit    = c.crit;
        erab_setup_list_bearer_su_res.value   = c.value.erab_setup_list_bearer_su_res();
        break;
      case 29:
        erab_failed_to_setup_list_bearer_su_res_present = true;
        erab_failed_to_setup_list_bearer_su_res.id      = c.id;
        erab_failed_to_setup_list_bearer_su_res.crit    = c.crit;
        erab_failed_to_setup_list_bearer_su_res.value   = c.value.erab_failed_to_setup_list_bearer_su_res();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void erab_setup_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (erab_setup_list_bearer_su_res_present) {
    j.write_fieldname("");
    erab_setup_list_bearer_su_res.to_json(j);
  }
  if (erab_failed_to_setup_list_bearer_su_res_present) {
    j.write_fieldname("");
    erab_failed_to_setup_list_bearer_su_res.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// E-RABSetupResponse ::= SEQUENCE
SRSASN_CODE erab_setup_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_setup_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void erab_setup_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// E-RABToBeSetupItemCtxtSUReqExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_setup_item_ctxt_su_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {156, 183, 233};
  return map_enum_number(options, 3, idx, "id");
}
bool erab_to_be_setup_item_ctxt_su_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {156, 183, 233};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_to_be_setup_item_ctxt_su_req_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 156:
      return crit_e::ignore;
    case 183:
      return crit_e::ignore;
    case 233:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 156:
      ret.set(ext_c::types::correlation_id);
      break;
    case 183:
      ret.set(ext_c::types::sipto_correlation_id);
      break;
    case 233:
      ret.set(ext_c::types::bearer_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ctxt_su_req_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 156:
      return presence_e::optional;
    case 183:
      return presence_e::optional;
    case 233:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::correlation_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::sipto_correlation_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
bearer_type_e& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::bearer_type()
{
  assert_choice_type("BearerType", type_.to_string(), "Extension");
  return c.get<bearer_type_e>();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::correlation_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const fixed_octstring<4, true>& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::sipto_correlation_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Extension");
  return c.get<fixed_octstring<4, true> >();
}
const bearer_type_e& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::bearer_type() const
{
  assert_choice_type("BearerType", type_.to_string(), "Extension");
  return c.get<bearer_type_e>();
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.destroy<fixed_octstring<4, true> >();
      break;
    default:
      break;
  }
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::sipto_correlation_id:
      c.init<fixed_octstring<4, true> >();
      break;
    case types::bearer_type:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::ext_c(
    const erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.init(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.init(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c& erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::
                                                    operator=(const erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::sipto_correlation_id:
      c.set(other.c.get<fixed_octstring<4, true> >());
      break;
    case types::bearer_type:
      c.set(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }

  return *this;
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::sipto_correlation_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<4, true> >().to_string());
      break;
    case types::bearer_type:
      j.write_str("BearerType", "non-IP");
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().pack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::sipto_correlation_id:
      HANDLE_CODE((c.get<fixed_octstring<4, true> >().unpack(bref)));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING", "OCTET STRING", "BearerType"};
  return convert_enum_idx(options, 3, value, "erab_to_be_setup_item_ctxt_su_req_ext_ies_o::ext_c::types");
}

template struct asn1::s1ap::protocol_ext_field_s<erab_to_be_setup_item_ctxt_su_req_ext_ies_o>;

erab_to_be_setup_item_ctxt_su_req_ext_ies_container::erab_to_be_setup_item_ctxt_su_req_ext_ies_container() :
  correlation_id(156, crit_e::ignore),
  sipto_correlation_id(183, crit_e::ignore),
  bearer_type(233, crit_e::reject)
{}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += correlation_id_present ? 1 : 0;
  nof_ies += sipto_correlation_id_present ? 1 : 0;
  nof_ies += bearer_type_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (correlation_id_present) {
    HANDLE_CODE(correlation_id.pack(bref));
  }
  if (sipto_correlation_id_present) {
    HANDLE_CODE(sipto_correlation_id.pack(bref));
  }
  if (bearer_type_present) {
    HANDLE_CODE(bearer_type.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<erab_to_be_setup_item_ctxt_su_req_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 156:
        correlation_id_present = true;
        correlation_id.id      = c.id;
        correlation_id.crit    = c.crit;
        correlation_id.ext     = c.ext_value.correlation_id();
        break;
      case 183:
        sipto_correlation_id_present = true;
        sipto_correlation_id.id      = c.id;
        sipto_correlation_id.crit    = c.crit;
        sipto_correlation_id.ext     = c.ext_value.sipto_correlation_id();
        break;
      case 233:
        bearer_type_present = true;
        bearer_type.id      = c.id;
        bearer_type.crit    = c.crit;
        bearer_type.ext     = c.ext_value.bearer_type();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ctxt_su_req_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (correlation_id_present) {
    j.write_fieldname("");
    correlation_id.to_json(j);
  }
  if (sipto_correlation_id_present) {
    j.write_fieldname("");
    sipto_correlation_id.to_json(j);
  }
  if (bearer_type_present) {
    j.write_fieldname("");
    bearer_type.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemCtxtSUReq ::= SEQUENCE
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(nas_pdu_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(nas_pdu_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ctxt_su_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_fieldname("e-RABlevelQoSParameters");
  erab_level_qos_params.to_json(j);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (nas_pdu_present) {
    j.write_str("nAS-PDU", nas_pdu.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemCtxtSUReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_setup_item_ctxt_su_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {52};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_setup_item_ctxt_su_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 52 == id;
}
crit_e erab_to_be_setup_item_ctxt_su_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 52) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_setup_item_ctxt_su_req_ies_o::value_c erab_to_be_setup_item_ctxt_su_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 52) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ctxt_su_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 52) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSetupItemCtxtSUReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSetupItemCtxtSUReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_setup_item_ctxt_su_req_ies_o::value_c::types");
}

// Data-Forwarding-Not-Possible ::= ENUMERATED
std::string data_forwarding_not_possible_opts::to_string() const
{
  static const char* options[] = {"data-Forwarding-not-Possible"};
  return convert_enum_idx(options, 1, value, "data_forwarding_not_possible_e");
}

// E-RABToBeSetupItemHOReq-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t erab_to_be_setup_item_ho_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {143, 233};
  return map_enum_number(options, 2, idx, "id");
}
bool erab_to_be_setup_item_ho_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {143, 233};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e erab_to_be_setup_item_ho_req_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 143:
      return crit_e::ignore;
    case 233:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c erab_to_be_setup_item_ho_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 143:
      ret.set(ext_c::types::data_forwarding_not_possible);
      break;
    case 233:
      ret.set(ext_c::types::bearer_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ho_req_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 143:
      return presence_e::optional;
    case 233:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
data_forwarding_not_possible_e& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::data_forwarding_not_possible()
{
  assert_choice_type("Data-Forwarding-Not-Possible", type_.to_string(), "Extension");
  return c.get<data_forwarding_not_possible_e>();
}
bearer_type_e& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::bearer_type()
{
  assert_choice_type("BearerType", type_.to_string(), "Extension");
  return c.get<bearer_type_e>();
}
const data_forwarding_not_possible_e&
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::data_forwarding_not_possible() const
{
  assert_choice_type("Data-Forwarding-Not-Possible", type_.to_string(), "Extension");
  return c.get<data_forwarding_not_possible_e>();
}
const bearer_type_e& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::bearer_type() const
{
  assert_choice_type("BearerType", type_.to_string(), "Extension");
  return c.get<bearer_type_e>();
}
void erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::destroy_() {}
void erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::ext_c(const erab_to_be_setup_item_ho_req_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::data_forwarding_not_possible:
      c.init(other.c.get<data_forwarding_not_possible_e>());
      break;
    case types::bearer_type:
      c.init(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
  }
}
erab_to_be_setup_item_ho_req_ext_ies_o::ext_c& erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::
                                               operator=(const erab_to_be_setup_item_ho_req_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::data_forwarding_not_possible:
      c.set(other.c.get<data_forwarding_not_possible_e>());
      break;
    case types::bearer_type:
      c.set(other.c.get<bearer_type_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
  }

  return *this;
}
void erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::data_forwarding_not_possible:
      j.write_str("Data-Forwarding-Not-Possible", "data-Forwarding-not-Possible");
      break;
    case types::bearer_type:
      j.write_str("BearerType", "non-IP");
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::data_forwarding_not_possible:
      HANDLE_CODE(c.get<data_forwarding_not_possible_e>().pack(bref));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::data_forwarding_not_possible:
      HANDLE_CODE(c.get<data_forwarding_not_possible_e>().unpack(bref));
      break;
    case types::bearer_type:
      HANDLE_CODE(c.get<bearer_type_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"Data-Forwarding-Not-Possible", "BearerType"};
  return convert_enum_idx(options, 2, value, "erab_to_be_setup_item_ho_req_ext_ies_o::ext_c::types");
}

template struct asn1::s1ap::protocol_ext_field_s<erab_to_be_setup_item_ho_req_ext_ies_o>;

erab_to_be_setup_item_ho_req_ext_ies_container::erab_to_be_setup_item_ho_req_ext_ies_container() :
  data_forwarding_not_possible(143, crit_e::ignore),
  bearer_type(233, crit_e::reject)
{}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += data_forwarding_not_possible_present ? 1 : 0;
  nof_ies += bearer_type_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (data_forwarding_not_possible_present) {
    HANDLE_CODE(data_forwarding_not_possible.pack(bref));
  }
  if (bearer_type_present) {
    HANDLE_CODE(bearer_type.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<erab_to_be_setup_item_ho_req_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 143:
        data_forwarding_not_possible_present = true;
        data_forwarding_not_possible.id      = c.id;
        data_forwarding_not_possible.crit    = c.crit;
        data_forwarding_not_possible.ext     = c.ext_value.data_forwarding_not_possible();
        break;
      case 233:
        bearer_type_present = true;
        bearer_type.id      = c.id;
        bearer_type.crit    = c.crit;
        bearer_type.ext     = c.ext_value.bearer_type();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ho_req_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (data_forwarding_not_possible_present) {
    j.write_fieldname("");
    data_forwarding_not_possible.to_json(j);
  }
  if (bearer_type_present) {
    j.write_fieldname("");
    bearer_type.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemHOReq ::= SEQUENCE
SRSASN_CODE erab_to_be_setup_item_ho_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  HANDLE_CODE(erab_level_qos_params.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  HANDLE_CODE(erab_level_qos_params.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_setup_item_ho_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  j.write_fieldname("e-RABlevelQosParameters");
  erab_level_qos_params.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSetupItemHOReqIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_setup_item_ho_req_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {27};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_setup_item_ho_req_ies_o::is_id_valid(const uint32_t& id)
{
  return 27 == id;
}
crit_e erab_to_be_setup_item_ho_req_ies_o::get_crit(const uint32_t& id)
{
  if (id == 27) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_setup_item_ho_req_ies_o::value_c erab_to_be_setup_item_ho_req_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 27) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_setup_item_ho_req_ies_o::get_presence(const uint32_t& id)
{
  if (id == 27) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_setup_item_ho_req_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSetupItemHOReq");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_setup_item_ho_req_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_setup_item_ho_req_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSetupItemHOReq"};
  return convert_enum_idx(options, 1, value, "erab_to_be_setup_item_ho_req_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<erab_to_be_setup_item_ctxt_su_req_ies_o>;

// E-RABToBeSwitchedDLItem ::= SEQUENCE
SRSASN_CODE erab_to_be_switched_dl_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_dl_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_switched_dl_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSwitchedDLItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_switched_dl_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {23};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_switched_dl_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 23 == id;
}
crit_e erab_to_be_switched_dl_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 23) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_switched_dl_item_ies_o::value_c erab_to_be_switched_dl_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 23) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_switched_dl_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 23) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_switched_dl_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSwitchedDLItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_switched_dl_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_dl_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_switched_dl_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSwitchedDLItem"};
  return convert_enum_idx(options, 1, value, "erab_to_be_switched_dl_item_ies_o::value_c::types");
}

// E-RABToBeSwitchedULItem ::= SEQUENCE
SRSASN_CODE erab_to_be_switched_ul_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_ul_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_to_be_switched_ul_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// E-RABToBeSwitchedULItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t erab_to_be_switched_ul_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {94};
  return map_enum_number(options, 1, idx, "id");
}
bool erab_to_be_switched_ul_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 94 == id;
}
crit_e erab_to_be_switched_ul_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 94) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
erab_to_be_switched_ul_item_ies_o::value_c erab_to_be_switched_ul_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 94) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e erab_to_be_switched_ul_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 94) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void erab_to_be_switched_ul_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("E-RABToBeSwitchedULItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE erab_to_be_switched_ul_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_to_be_switched_ul_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string erab_to_be_switched_ul_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"E-RABToBeSwitchedULItem"};
  return convert_enum_idx(options, 1, value, "erab_to_be_switched_ul_item_ies_o::value_c::types");
}

// EHRPDMultiSectorLoadReportingResponseItem ::= SEQUENCE
SRSASN_CODE ehrpd_multi_sector_load_report_resp_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(ehrpd_sector_id.pack(bref));
  HANDLE_CODE(ehrpd_sector_load_report_resp.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ehrpd_multi_sector_load_report_resp_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(ehrpd_sector_id.unpack(bref));
  HANDLE_CODE(ehrpd_sector_load_report_resp.unpack(bref));

  return SRSASN_SUCCESS;
}
void ehrpd_multi_sector_load_report_resp_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("eHRPD-Sector-ID", ehrpd_sector_id.to_string());
  j.write_fieldname("eHRPDSectorLoadReportingResponse");
  ehrpd_sector_load_report_resp.to_json(j);
  j.end_obj();
}

// ENB-StatusTransfer-TransparentContainer ::= SEQUENCE
SRSASN_CODE enb_status_transfer_transparent_container_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, bearers_subject_to_status_transfer_list, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_transparent_container_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(bearers_subject_to_status_transfer_list, bref, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void enb_status_transfer_transparent_container_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("bearers-SubjectToStatusTransferList");
  for (const auto& e1 : bearers_subject_to_status_transfer_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// S-TMSI ::= SEQUENCE
SRSASN_CODE s_tmsi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(mmec.pack(bref));
  HANDLE_CODE(m_tmsi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE s_tmsi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(mmec.unpack(bref));
  HANDLE_CODE(m_tmsi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void s_tmsi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("mMEC", mmec.to_string());
  j.write_str("m-TMSI", m_tmsi.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UL-CP-SecurityInformation ::= SEQUENCE
SRSASN_CODE ul_cp_security_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ul_nas_mac.pack(bref));
  HANDLE_CODE(ul_nas_count.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ul_cp_security_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ul_nas_mac.unpack(bref));
  HANDLE_CODE(ul_nas_count.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ul_cp_security_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("ul-NAS-MAC", ul_nas_mac.to_string());
  j.write_str("ul-NAS-Count", ul_nas_count.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ENBCPRelocationIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enbcp_relocation_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {8, 96, 100, 67, 254};
  return map_enum_number(options, 5, idx, "id");
}
bool enbcp_relocation_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {8, 96, 100, 67, 254};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enbcp_relocation_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 8:
      return crit_e::reject;
    case 96:
      return crit_e::reject;
    case 100:
      return crit_e::ignore;
    case 67:
      return crit_e::ignore;
    case 254:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enbcp_relocation_ind_ies_o::value_c enbcp_relocation_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 96:
      ret.set(value_c::types::s_tmsi);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 67:
      ret.set(value_c::types::tai);
      break;
    case 254:
      ret.set(value_c::types::ul_cp_security_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enbcp_relocation_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 8:
      return presence_e::mandatory;
    case 96:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 67:
      return presence_e::mandatory;
    case 254:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint32_t& enbcp_relocation_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
s_tmsi_s& enbcp_relocation_ind_ies_o::value_c::s_tmsi()
{
  assert_choice_type("S-TMSI", type_.to_string(), "Value");
  return c.get<s_tmsi_s>();
}
eutran_cgi_s& enbcp_relocation_ind_ies_o::value_c::eutran_cgi()
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
tai_s& enbcp_relocation_ind_ies_o::value_c::tai()
{
  assert_choice_type("TAI", type_.to_string(), "Value");
  return c.get<tai_s>();
}
ul_cp_security_info_s& enbcp_relocation_ind_ies_o::value_c::ul_cp_security_info()
{
  assert_choice_type("UL-CP-SecurityInformation", type_.to_string(), "Value");
  return c.get<ul_cp_security_info_s>();
}
const uint32_t& enbcp_relocation_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const s_tmsi_s& enbcp_relocation_ind_ies_o::value_c::s_tmsi() const
{
  assert_choice_type("S-TMSI", type_.to_string(), "Value");
  return c.get<s_tmsi_s>();
}
const eutran_cgi_s& enbcp_relocation_ind_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
const tai_s& enbcp_relocation_ind_ies_o::value_c::tai() const
{
  assert_choice_type("TAI", type_.to_string(), "Value");
  return c.get<tai_s>();
}
const ul_cp_security_info_s& enbcp_relocation_ind_ies_o::value_c::ul_cp_security_info() const
{
  assert_choice_type("UL-CP-SecurityInformation", type_.to_string(), "Value");
  return c.get<ul_cp_security_info_s>();
}
void enbcp_relocation_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::s_tmsi:
      c.destroy<s_tmsi_s>();
      break;
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    case types::ul_cp_security_info:
      c.destroy<ul_cp_security_info_s>();
      break;
    default:
      break;
  }
}
void enbcp_relocation_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enb_ue_s1ap_id:
      break;
    case types::s_tmsi:
      c.init<s_tmsi_s>();
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::ul_cp_security_info:
      c.init<ul_cp_security_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }
}
enbcp_relocation_ind_ies_o::value_c::value_c(const enbcp_relocation_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::s_tmsi:
      c.init(other.c.get<s_tmsi_s>());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::ul_cp_security_info:
      c.init(other.c.get<ul_cp_security_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }
}
enbcp_relocation_ind_ies_o::value_c& enbcp_relocation_ind_ies_o::value_c::
                                     operator=(const enbcp_relocation_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::s_tmsi:
      c.set(other.c.get<s_tmsi_s>());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::ul_cp_security_info:
      c.set(other.c.get<ul_cp_security_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }

  return *this;
}
void enbcp_relocation_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::s_tmsi:
      j.write_fieldname("S-TMSI");
      c.get<s_tmsi_s>().to_json(j);
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::tai:
      j.write_fieldname("TAI");
      c.get<tai_s>().to_json(j);
      break;
    case types::ul_cp_security_info:
      j.write_fieldname("UL-CP-SecurityInformation");
      c.get<ul_cp_security_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enbcp_relocation_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().pack(bref));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    case types::ul_cp_security_info:
      HANDLE_CODE(c.get<ul_cp_security_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enbcp_relocation_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().unpack(bref));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    case types::ul_cp_security_info:
      HANDLE_CODE(c.get<ul_cp_security_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enbcp_relocation_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string enbcp_relocation_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..16777215)", "S-TMSI", "EUTRAN-CGI", "TAI", "UL-CP-SecurityInformation"};
  return convert_enum_idx(options, 5, value, "enbcp_relocation_ind_ies_o::value_c::types");
}
uint8_t enbcp_relocation_ind_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "enbcp_relocation_ind_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<enbcp_relocation_ind_ies_o>;

enbcp_relocation_ind_ies_container::enbcp_relocation_ind_ies_container() :
  enb_ue_s1ap_id(8, crit_e::reject),
  s_tmsi(96, crit_e::reject),
  eutran_cgi(100, crit_e::ignore),
  tai(67, crit_e::ignore),
  ul_cp_security_info(254, crit_e::reject)
{}
SRSASN_CODE enbcp_relocation_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(s_tmsi.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(ul_cp_security_info.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enbcp_relocation_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<enbcp_relocation_ind_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 96:
        nof_mandatory_ies--;
        s_tmsi.id    = c.id;
        s_tmsi.crit  = c.crit;
        s_tmsi.value = c.value.s_tmsi();
        break;
      case 100:
        nof_mandatory_ies--;
        eutran_cgi.id    = c.id;
        eutran_cgi.crit  = c.crit;
        eutran_cgi.value = c.value.eutran_cgi();
        break;
      case 67:
        nof_mandatory_ies--;
        tai.id    = c.id;
        tai.crit  = c.crit;
        tai.value = c.value.tai();
        break;
      case 254:
        nof_mandatory_ies--;
        ul_cp_security_info.id    = c.id;
        ul_cp_security_info.crit  = c.crit;
        ul_cp_security_info.value = c.value.ul_cp_security_info();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void enbcp_relocation_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  s_tmsi.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  tai.to_json(j);
  j.write_fieldname("");
  ul_cp_security_info.to_json(j);
  j.end_obj();
}

// ENBCPRelocationIndication ::= SEQUENCE
SRSASN_CODE enbcp_relocation_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enbcp_relocation_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void enbcp_relocation_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// ENBX2ExtTLA ::= SEQUENCE
SRSASN_CODE enbx2_ext_tla_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ipsec_tla_present, 1));
  HANDLE_CODE(bref.pack(gtptl_aa_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (ipsec_tla_present) {
    HANDLE_CODE(ipsec_tla.pack(bref));
  }
  if (gtptl_aa_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, gtptl_aa, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enbx2_ext_tla_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ipsec_tla_present, 1));
  HANDLE_CODE(bref.unpack(gtptl_aa_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (ipsec_tla_present) {
    HANDLE_CODE(ipsec_tla.unpack(bref));
  }
  if (gtptl_aa_present) {
    HANDLE_CODE(unpack_dyn_seq_of(gtptl_aa, bref, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void enbx2_ext_tla_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ipsec_tla_present) {
    j.write_str("iPsecTLA", ipsec_tla.to_string());
  }
  if (gtptl_aa_present) {
    j.start_array("gTPTLAa");
    for (const auto& e1 : gtptl_aa) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MutingAvailabilityIndication ::= ENUMERATED
std::string muting_availability_ind_opts::to_string() const
{
  static const char* options[] = {"available", "unavailable"};
  return convert_enum_idx(options, 2, value, "muting_availability_ind_e");
}

// RLFReportInformation ::= SEQUENCE
SRSASN_CODE rlf_report_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ue_rlf_report_container_for_extended_bands_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ue_rlf_report_container.pack(bref));
  if (ue_rlf_report_container_for_extended_bands_present) {
    HANDLE_CODE(ue_rlf_report_container_for_extended_bands.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE rlf_report_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ue_rlf_report_container_for_extended_bands_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ue_rlf_report_container.unpack(bref));
  if (ue_rlf_report_container_for_extended_bands_present) {
    HANDLE_CODE(ue_rlf_report_container_for_extended_bands.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void rlf_report_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("uE-RLF-Report-Container", ue_rlf_report_container.to_string());
  if (ue_rlf_report_container_for_extended_bands_present) {
    j.write_str("uE-RLF-Report-Container-for-extended-bands", ue_rlf_report_container_for_extended_bands.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SynchronisationStatus ::= ENUMERATED
std::string synchronisation_status_opts::to_string() const
{
  static const char* options[] = {"synchronous", "asynchronous"};
  return convert_enum_idx(options, 2, value, "synchronisation_status_e");
}

// TimeSynchronisationInfo-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t time_synchronisation_info_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {207};
  return map_enum_number(options, 1, idx, "id");
}
bool time_synchronisation_info_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 207 == id;
}
crit_e time_synchronisation_info_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 207) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
time_synchronisation_info_ext_ies_o::ext_c time_synchronisation_info_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 207) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e time_synchronisation_info_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 207) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void time_synchronisation_info_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("MutingAvailabilityIndication", c.to_string());
  j.end_obj();
}
SRSASN_CODE time_synchronisation_info_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE time_synchronisation_info_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string time_synchronisation_info_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"MutingAvailabilityIndication"};
  return convert_enum_idx(options, 1, value, "time_synchronisation_info_ext_ies_o::ext_c::types");
}

// MutingPatternInformation ::= SEQUENCE
SRSASN_CODE muting_pattern_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(muting_pattern_offset_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(muting_pattern_period.pack(bref));
  if (muting_pattern_offset_present) {
    HANDLE_CODE(pack_integer(bref, muting_pattern_offset, (uint16_t)0u, (uint16_t)10239u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE muting_pattern_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(muting_pattern_offset_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(muting_pattern_period.unpack(bref));
  if (muting_pattern_offset_present) {
    HANDLE_CODE(unpack_integer(muting_pattern_offset, bref, (uint16_t)0u, (uint16_t)10239u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void muting_pattern_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("muting-pattern-period", muting_pattern_period.to_string());
  if (muting_pattern_offset_present) {
    j.write_int("muting-pattern-offset", muting_pattern_offset);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

std::string muting_pattern_info_s::muting_pattern_period_opts::to_string() const
{
  static const char* options[] = {"ms0", "ms1280", "ms2560", "ms5120", "ms10240"};
  return convert_enum_idx(options, 5, value, "muting_pattern_info_s::muting_pattern_period_e_");
}
uint16_t muting_pattern_info_s::muting_pattern_period_opts::to_number() const
{
  static const uint16_t options[] = {0, 1280, 2560, 5120, 10240};
  return map_enum_number(options, 5, value, "muting_pattern_info_s::muting_pattern_period_e_");
}

// SONInformationReport ::= CHOICE
void son_info_report_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("rLFReportInformation");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE son_info_report_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_report_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "son_info_report_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string son_info_report_c::types_opts::to_string() const
{
  static const char* options[] = {"rLFReportInformation"};
  return convert_enum_idx(options, 1, value, "son_info_report_c::types");
}

// TimeSynchronisationInfo ::= SEQUENCE
SRSASN_CODE time_synchronisation_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, stratum_level, (uint8_t)0u, (uint8_t)3u, true, true));
  HANDLE_CODE(synchronisation_status.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE time_synchronisation_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(stratum_level, bref, (uint8_t)0u, (uint8_t)3u, true, true));
  HANDLE_CODE(synchronisation_status.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void time_synchronisation_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("stratumLevel", stratum_level);
  j.write_str("synchronisationStatus", synchronisation_status.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// X2TNLConfigurationInfo-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t x2_tnl_cfg_info_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {153, 193};
  return map_enum_number(options, 2, idx, "id");
}
bool x2_tnl_cfg_info_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {153, 193};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e x2_tnl_cfg_info_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 153:
      return crit_e::ignore;
    case 193:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
x2_tnl_cfg_info_ext_ies_o::ext_c x2_tnl_cfg_info_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 153:
      ret.set(ext_c::types::enbx2_extended_transport_layer_addresses);
      break;
    case 193:
      ret.set(ext_c::types::enb_indirect_x2_transport_layer_addresses);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e x2_tnl_cfg_info_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 153:
      return presence_e::optional;
    case 193:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
enbx2_ext_tlas_l& x2_tnl_cfg_info_ext_ies_o::ext_c::enbx2_extended_transport_layer_addresses()
{
  assert_choice_type("ENBX2ExtTLAs", type_.to_string(), "Extension");
  return c.get<enbx2_ext_tlas_l>();
}
enb_indirect_x2_transport_layer_addresses_l&
x2_tnl_cfg_info_ext_ies_o::ext_c::enb_indirect_x2_transport_layer_addresses()
{
  assert_choice_type("ENBIndirectX2TransportLayerAddresses", type_.to_string(), "Extension");
  return c.get<enb_indirect_x2_transport_layer_addresses_l>();
}
const enbx2_ext_tlas_l& x2_tnl_cfg_info_ext_ies_o::ext_c::enbx2_extended_transport_layer_addresses() const
{
  assert_choice_type("ENBX2ExtTLAs", type_.to_string(), "Extension");
  return c.get<enbx2_ext_tlas_l>();
}
const enb_indirect_x2_transport_layer_addresses_l&
x2_tnl_cfg_info_ext_ies_o::ext_c::enb_indirect_x2_transport_layer_addresses() const
{
  assert_choice_type("ENBIndirectX2TransportLayerAddresses", type_.to_string(), "Extension");
  return c.get<enb_indirect_x2_transport_layer_addresses_l>();
}
void x2_tnl_cfg_info_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.destroy<enbx2_ext_tlas_l>();
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.destroy<enb_indirect_x2_transport_layer_addresses_l>();
      break;
    default:
      break;
  }
}
void x2_tnl_cfg_info_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.init<enbx2_ext_tlas_l>();
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.init<enb_indirect_x2_transport_layer_addresses_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }
}
x2_tnl_cfg_info_ext_ies_o::ext_c::ext_c(const x2_tnl_cfg_info_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.init(other.c.get<enbx2_ext_tlas_l>());
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.init(other.c.get<enb_indirect_x2_transport_layer_addresses_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }
}
x2_tnl_cfg_info_ext_ies_o::ext_c& x2_tnl_cfg_info_ext_ies_o::ext_c::
                                  operator=(const x2_tnl_cfg_info_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      c.set(other.c.get<enbx2_ext_tlas_l>());
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      c.set(other.c.get<enb_indirect_x2_transport_layer_addresses_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }

  return *this;
}
void x2_tnl_cfg_info_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      j.start_array("ENBX2ExtTLAs");
      for (const auto& e1 : c.get<enbx2_ext_tlas_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      j.start_array("ENBIndirectX2TransportLayerAddresses");
      for (const auto& e1 : c.get<enb_indirect_x2_transport_layer_addresses_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<enbx2_ext_tlas_l>(), 1, 16, true));
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<enb_indirect_x2_transport_layer_addresses_l>(), 1, 2, true));
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbx2_extended_transport_layer_addresses:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<enbx2_ext_tlas_l>(), bref, 1, 16, true));
      break;
    case types::enb_indirect_x2_transport_layer_addresses:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<enb_indirect_x2_transport_layer_addresses_l>(), bref, 1, 2, true));
      break;
    default:
      log_invalid_choice_id(type_, "x2_tnl_cfg_info_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string x2_tnl_cfg_info_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"ENBX2ExtTLAs", "ENBIndirectX2TransportLayerAddresses"};
  return convert_enum_idx(options, 2, value, "x2_tnl_cfg_info_ext_ies_o::ext_c::types");
}

// ENB-ID ::= CHOICE
void enb_id_c::destroy_()
{
  switch (type_) {
    case types::macro_enb_id:
      c.destroy<fixed_bitstring<20, false, true> >();
      break;
    case types::home_enb_id:
      c.destroy<fixed_bitstring<28, false, true> >();
      break;
    case types::short_macro_enb_id:
      c.destroy<fixed_bitstring<18, false, true> >();
      break;
    case types::long_macro_enb_id:
      c.destroy<fixed_bitstring<21, false, true> >();
      break;
    default:
      break;
  }
}
void enb_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::macro_enb_id:
      c.init<fixed_bitstring<20, false, true> >();
      break;
    case types::home_enb_id:
      c.init<fixed_bitstring<28, false, true> >();
      break;
    case types::short_macro_enb_id:
      c.init<fixed_bitstring<18, false, true> >();
      break;
    case types::long_macro_enb_id:
      c.init<fixed_bitstring<21, false, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }
}
enb_id_c::enb_id_c(const enb_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::macro_enb_id:
      c.init(other.c.get<fixed_bitstring<20, false, true> >());
      break;
    case types::home_enb_id:
      c.init(other.c.get<fixed_bitstring<28, false, true> >());
      break;
    case types::short_macro_enb_id:
      c.init(other.c.get<fixed_bitstring<18, false, true> >());
      break;
    case types::long_macro_enb_id:
      c.init(other.c.get<fixed_bitstring<21, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }
}
enb_id_c& enb_id_c::operator=(const enb_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::macro_enb_id:
      c.set(other.c.get<fixed_bitstring<20, false, true> >());
      break;
    case types::home_enb_id:
      c.set(other.c.get<fixed_bitstring<28, false, true> >());
      break;
    case types::short_macro_enb_id:
      c.set(other.c.get<fixed_bitstring<18, false, true> >());
      break;
    case types::long_macro_enb_id:
      c.set(other.c.get<fixed_bitstring<21, false, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }

  return *this;
}
void enb_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::macro_enb_id:
      j.write_str("macroENB-ID", c.get<fixed_bitstring<20, false, true> >().to_string());
      break;
    case types::home_enb_id:
      j.write_str("homeENB-ID", c.get<fixed_bitstring<28, false, true> >().to_string());
      break;
    case types::short_macro_enb_id:
      j.write_str("short-macroENB-ID", c.get<fixed_bitstring<18, false, true> >().to_string());
      break;
    case types::long_macro_enb_id:
      j.write_str("long-macroENB-ID", c.get<fixed_bitstring<21, false, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::macro_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<20, false, true> >().pack(bref)));
      break;
    case types::home_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<28, false, true> >().pack(bref)));
      break;
    case types::short_macro_enb_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<18, false, true> >().pack(bref)));
    } break;
    case types::long_macro_enb_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<21, false, true> >().pack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::macro_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<20, false, true> >().unpack(bref)));
      break;
    case types::home_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<28, false, true> >().unpack(bref)));
      break;
    case types::short_macro_enb_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<18, false, true> >().unpack(bref)));
    } break;
    case types::long_macro_enb_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_bitstring<21, false, true> >().unpack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "enb_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string enb_id_c::types_opts::to_string() const
{
  static const char* options[] = {"macroENB-ID", "homeENB-ID", "short-macroENB-ID", "long-macroENB-ID"};
  return convert_enum_idx(options, 4, value, "enb_id_c::types");
}

// ListeningSubframePattern ::= SEQUENCE
SRSASN_CODE listening_sf_pattern_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pattern_period.pack(bref));
  HANDLE_CODE(pack_integer(bref, pattern_offset, (uint16_t)0u, (uint16_t)10239u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE listening_sf_pattern_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(pattern_period.unpack(bref));
  HANDLE_CODE(unpack_integer(pattern_offset, bref, (uint16_t)0u, (uint16_t)10239u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void listening_sf_pattern_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pattern-period", pattern_period.to_string());
  j.write_int("pattern-offset", pattern_offset);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

std::string listening_sf_pattern_s::pattern_period_opts::to_string() const
{
  static const char* options[] = {"ms1280", "ms2560", "ms5120", "ms10240"};
  return convert_enum_idx(options, 4, value, "listening_sf_pattern_s::pattern_period_e_");
}
uint16_t listening_sf_pattern_s::pattern_period_opts::to_number() const
{
  static const uint16_t options[] = {1280, 2560, 5120, 10240};
  return map_enum_number(options, 4, value, "listening_sf_pattern_s::pattern_period_e_");
}

// SONInformation-ExtensionIE ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t son_info_ext_ie_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {206};
  return map_enum_number(options, 1, idx, "id");
}
bool son_info_ext_ie_o::is_id_valid(const uint32_t& id)
{
  return 206 == id;
}
crit_e son_info_ext_ie_o::get_crit(const uint32_t& id)
{
  if (id == 206) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
son_info_ext_ie_o::value_c son_info_ext_ie_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 206) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e son_info_ext_ie_o::get_presence(const uint32_t& id)
{
  if (id == 206) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void son_info_ext_ie_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("SONInformationReport");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE son_info_ext_ie_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_ext_ie_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string son_info_ext_ie_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"SONInformationReport"};
  return convert_enum_idx(options, 1, value, "son_info_ext_ie_o::value_c::types");
}

// SONInformationReply-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t son_info_reply_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {149};
  return map_enum_number(options, 1, idx, "id");
}
bool son_info_reply_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 149 == id;
}
crit_e son_info_reply_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 149) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
son_info_reply_ext_ies_o::ext_c son_info_reply_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 149) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e son_info_reply_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 149) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void son_info_reply_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("TimeSynchronisationInfo");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE son_info_reply_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_reply_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string son_info_reply_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"TimeSynchronisationInfo"};
  return convert_enum_idx(options, 1, value, "son_info_reply_ext_ies_o::ext_c::types");
}

template struct asn1::s1ap::protocol_ext_field_s<x2_tnl_cfg_info_ext_ies_o>;

x2_tnl_cfg_info_ext_ies_container::x2_tnl_cfg_info_ext_ies_container() :
  enbx2_extended_transport_layer_addresses(153, crit_e::ignore),
  enb_indirect_x2_transport_layer_addresses(193, crit_e::ignore)
{}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += enbx2_extended_transport_layer_addresses_present ? 1 : 0;
  nof_ies += enb_indirect_x2_transport_layer_addresses_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (enbx2_extended_transport_layer_addresses_present) {
    HANDLE_CODE(enbx2_extended_transport_layer_addresses.pack(bref));
  }
  if (enb_indirect_x2_transport_layer_addresses_present) {
    HANDLE_CODE(enb_indirect_x2_transport_layer_addresses.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE x2_tnl_cfg_info_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<x2_tnl_cfg_info_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 153:
        enbx2_extended_transport_layer_addresses_present = true;
        enbx2_extended_transport_layer_addresses.id      = c.id;
        enbx2_extended_transport_layer_addresses.crit    = c.crit;
        enbx2_extended_transport_layer_addresses.ext     = c.ext_value.enbx2_extended_transport_layer_addresses();
        break;
      case 193:
        enb_indirect_x2_transport_layer_addresses_present = true;
        enb_indirect_x2_transport_layer_addresses.id      = c.id;
        enb_indirect_x2_transport_layer_addresses.crit    = c.crit;
        enb_indirect_x2_transport_layer_addresses.ext     = c.ext_value.enb_indirect_x2_transport_layer_addresses();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void x2_tnl_cfg_info_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (enbx2_extended_transport_layer_addresses_present) {
    j.write_fieldname("");
    enbx2_extended_transport_layer_addresses.to_json(j);
  }
  if (enb_indirect_x2_transport_layer_addresses_present) {
    j.write_fieldname("");
    enb_indirect_x2_transport_layer_addresses.to_json(j);
  }
  j.end_obj();
}

// X2TNLConfigurationInfo ::= SEQUENCE
SRSASN_CODE x2_tnl_cfg_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, enbx2_transport_layer_addresses, 1, 2, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE x2_tnl_cfg_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(enbx2_transport_layer_addresses, bref, 1, 2, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void x2_tnl_cfg_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("eNBX2TransportLayerAddresses");
  for (const auto& e1 : enbx2_transport_layer_addresses) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Global-ENB-ID ::= SEQUENCE
SRSASN_CODE global_enb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(enb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_enb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(enb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_enb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_fieldname("eNB-ID");
  enb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::s1ap::protocol_ie_single_container_s<son_info_ext_ie_o>;

// SONInformationReply ::= SEQUENCE
SRSASN_CODE son_info_reply_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(x2_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_reply_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(x2_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void son_info_reply_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (x2_tnl_cfg_info_present) {
    j.write_fieldname("x2TNLConfigurationInfo");
    x2_tnl_cfg_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// SONInformationRequest ::= ENUMERATED
std::string son_info_request_opts::to_string() const
{
  static const char* options[] = {
      "x2TNL-Configuration-Info", "time-Synchronisation-Info", "activate-Muting", "deactivate-Muting"};
  return convert_enum_idx(options, 4, value, "son_info_request_e");
}
uint8_t son_info_request_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "son_info_request_e");
}

// SynchronisationInformation ::= SEQUENCE
SRSASN_CODE synchronisation_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(source_stratum_level_present, 1));
  HANDLE_CODE(bref.pack(listening_sf_pattern_present, 1));
  HANDLE_CODE(bref.pack(aggressore_cgi_list_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (source_stratum_level_present) {
    HANDLE_CODE(pack_integer(bref, source_stratum_level, (uint8_t)0u, (uint8_t)3u, true, true));
  }
  if (listening_sf_pattern_present) {
    HANDLE_CODE(listening_sf_pattern.pack(bref));
  }
  if (aggressore_cgi_list_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, aggressore_cgi_list, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE synchronisation_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(source_stratum_level_present, 1));
  HANDLE_CODE(bref.unpack(listening_sf_pattern_present, 1));
  HANDLE_CODE(bref.unpack(aggressore_cgi_list_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (source_stratum_level_present) {
    HANDLE_CODE(unpack_integer(source_stratum_level, bref, (uint8_t)0u, (uint8_t)3u, true, true));
  }
  if (listening_sf_pattern_present) {
    HANDLE_CODE(listening_sf_pattern.unpack(bref));
  }
  if (aggressore_cgi_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(aggressore_cgi_list, bref, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void synchronisation_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (source_stratum_level_present) {
    j.write_int("sourceStratumLevel", source_stratum_level);
  }
  if (listening_sf_pattern_present) {
    j.write_fieldname("listeningSubframePattern");
    listening_sf_pattern.to_json(j);
  }
  if (aggressore_cgi_list_present) {
    j.start_array("aggressoreCGI-List");
    for (const auto& e1 : aggressore_cgi_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SONConfigurationTransfer-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t son_cfg_transfer_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {152, 209};
  return map_enum_number(options, 2, idx, "id");
}
bool son_cfg_transfer_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {152, 209};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e son_cfg_transfer_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 152:
      return crit_e::ignore;
    case 209:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
son_cfg_transfer_ext_ies_o::ext_c son_cfg_transfer_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 152:
      ret.set(ext_c::types::x2_tnl_cfg_info);
      break;
    case 209:
      ret.set(ext_c::types::synchronisation_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e son_cfg_transfer_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 152:
      return presence_e::conditional;
    case 209:
      return presence_e::conditional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
x2_tnl_cfg_info_s& son_cfg_transfer_ext_ies_o::ext_c::x2_tnl_cfg_info()
{
  assert_choice_type("X2TNLConfigurationInfo", type_.to_string(), "Extension");
  return c.get<x2_tnl_cfg_info_s>();
}
synchronisation_info_s& son_cfg_transfer_ext_ies_o::ext_c::synchronisation_info()
{
  assert_choice_type("SynchronisationInformation", type_.to_string(), "Extension");
  return c.get<synchronisation_info_s>();
}
const x2_tnl_cfg_info_s& son_cfg_transfer_ext_ies_o::ext_c::x2_tnl_cfg_info() const
{
  assert_choice_type("X2TNLConfigurationInfo", type_.to_string(), "Extension");
  return c.get<x2_tnl_cfg_info_s>();
}
const synchronisation_info_s& son_cfg_transfer_ext_ies_o::ext_c::synchronisation_info() const
{
  assert_choice_type("SynchronisationInformation", type_.to_string(), "Extension");
  return c.get<synchronisation_info_s>();
}
void son_cfg_transfer_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.destroy<x2_tnl_cfg_info_s>();
      break;
    case types::synchronisation_info:
      c.destroy<synchronisation_info_s>();
      break;
    default:
      break;
  }
}
void son_cfg_transfer_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.init<x2_tnl_cfg_info_s>();
      break;
    case types::synchronisation_info:
      c.init<synchronisation_info_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }
}
son_cfg_transfer_ext_ies_o::ext_c::ext_c(const son_cfg_transfer_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.init(other.c.get<x2_tnl_cfg_info_s>());
      break;
    case types::synchronisation_info:
      c.init(other.c.get<synchronisation_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }
}
son_cfg_transfer_ext_ies_o::ext_c& son_cfg_transfer_ext_ies_o::ext_c::
                                   operator=(const son_cfg_transfer_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::x2_tnl_cfg_info:
      c.set(other.c.get<x2_tnl_cfg_info_s>());
      break;
    case types::synchronisation_info:
      c.set(other.c.get<synchronisation_info_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }

  return *this;
}
void son_cfg_transfer_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::x2_tnl_cfg_info:
      j.write_fieldname("X2TNLConfigurationInfo");
      c.get<x2_tnl_cfg_info_s>().to_json(j);
      break;
    case types::synchronisation_info:
      j.write_fieldname("SynchronisationInformation");
      c.get<synchronisation_info_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE son_cfg_transfer_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::x2_tnl_cfg_info:
      HANDLE_CODE(c.get<x2_tnl_cfg_info_s>().pack(bref));
      break;
    case types::synchronisation_info:
      HANDLE_CODE(c.get<synchronisation_info_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::x2_tnl_cfg_info:
      HANDLE_CODE(c.get<x2_tnl_cfg_info_s>().unpack(bref));
      break;
    case types::synchronisation_info:
      HANDLE_CODE(c.get<synchronisation_info_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "son_cfg_transfer_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string son_cfg_transfer_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"X2TNLConfigurationInfo", "SynchronisationInformation"};
  return convert_enum_idx(options, 2, value, "son_cfg_transfer_ext_ies_o::ext_c::types");
}
uint8_t son_cfg_transfer_ext_ies_o::ext_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "son_cfg_transfer_ext_ies_o::ext_c::types");
}

// SONInformation ::= CHOICE
void son_info_c::destroy_()
{
  switch (type_) {
    case types::son_info_reply:
      c.destroy<son_info_reply_s>();
      break;
    case types::son_info_ext:
      c.destroy<protocol_ie_single_container_s<son_info_ext_ie_o> >();
      break;
    default:
      break;
  }
}
void son_info_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::son_info_request:
      break;
    case types::son_info_reply:
      c.init<son_info_reply_s>();
      break;
    case types::son_info_ext:
      c.init<protocol_ie_single_container_s<son_info_ext_ie_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
}
son_info_c::son_info_c(const son_info_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::son_info_request:
      c.init(other.c.get<son_info_request_e>());
      break;
    case types::son_info_reply:
      c.init(other.c.get<son_info_reply_s>());
      break;
    case types::son_info_ext:
      c.init(other.c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
}
son_info_c& son_info_c::operator=(const son_info_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::son_info_request:
      c.set(other.c.get<son_info_request_e>());
      break;
    case types::son_info_reply:
      c.set(other.c.get<son_info_reply_s>());
      break;
    case types::son_info_ext:
      c.set(other.c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }

  return *this;
}
void son_info_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::son_info_request:
      j.write_str("sONInformationRequest", c.get<son_info_request_e>().to_string());
      break;
    case types::son_info_reply:
      j.write_fieldname("sONInformationReply");
      c.get<son_info_reply_s>().to_json(j);
      break;
    case types::son_info_ext:
      j.write_fieldname("sONInformation-Extension");
      c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
  j.end_obj();
}
SRSASN_CODE son_info_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::son_info_request:
      HANDLE_CODE(c.get<son_info_request_e>().pack(bref));
      break;
    case types::son_info_reply:
      HANDLE_CODE(c.get<son_info_reply_s>().pack(bref));
      break;
    case types::son_info_ext: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::son_info_request:
      HANDLE_CODE(c.get<son_info_request_e>().unpack(bref));
      break;
    case types::son_info_reply:
      HANDLE_CODE(c.get<son_info_reply_s>().unpack(bref));
      break;
    case types::son_info_ext: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<son_info_ext_ie_o> >().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string son_info_c::types_opts::to_string() const
{
  static const char* options[] = {"sONInformationRequest", "sONInformationReply", "sONInformation-Extension"};
  return convert_enum_idx(options, 3, value, "son_info_c::types");
}

// SourceeNB-ID ::= SEQUENCE
SRSASN_CODE sourceenb_id_s::pack(bit_ref& bref) const
{
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE sourceenb_id_s::unpack(cbit_ref& bref)
{
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void sourceenb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-ENB-ID");
  global_enb_id.to_json(j);
  j.write_fieldname("selected-TAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TargeteNB-ID ::= SEQUENCE
SRSASN_CODE targetenb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE targetenb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void targetenb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("global-ENB-ID");
  global_enb_id.to_json(j);
  j.write_fieldname("selected-TAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::s1ap::protocol_ext_field_s<son_cfg_transfer_ext_ies_o>;

son_cfg_transfer_ext_ies_container::son_cfg_transfer_ext_ies_container() :
  x2_tnl_cfg_info(152, crit_e::ignore),
  synchronisation_info(209, crit_e::ignore)
{}
SRSASN_CODE son_cfg_transfer_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += x2_tnl_cfg_info_present ? 1 : 0;
  nof_ies += synchronisation_info_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (x2_tnl_cfg_info_present) {
    HANDLE_CODE(x2_tnl_cfg_info.pack(bref));
  }
  if (synchronisation_info_present) {
    HANDLE_CODE(synchronisation_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<son_cfg_transfer_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 152:
        x2_tnl_cfg_info_present = true;
        x2_tnl_cfg_info.id      = c.id;
        x2_tnl_cfg_info.crit    = c.crit;
        x2_tnl_cfg_info.ext     = c.ext_value.x2_tnl_cfg_info();
        break;
      case 209:
        synchronisation_info_present = true;
        synchronisation_info.id      = c.id;
        synchronisation_info.crit    = c.crit;
        synchronisation_info.ext     = c.ext_value.synchronisation_info();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void son_cfg_transfer_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (x2_tnl_cfg_info_present) {
    j.write_fieldname("");
    x2_tnl_cfg_info.to_json(j);
  }
  if (synchronisation_info_present) {
    j.write_fieldname("");
    synchronisation_info.to_json(j);
  }
  j.end_obj();
}

// SONConfigurationTransfer ::= SEQUENCE
SRSASN_CODE son_cfg_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(targetenb_id.pack(bref));
  HANDLE_CODE(sourceenb_id.pack(bref));
  HANDLE_CODE(son_info.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(targetenb_id.unpack(bref));
  HANDLE_CODE(sourceenb_id.unpack(bref));
  HANDLE_CODE(son_info.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void son_cfg_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("targeteNB-ID");
  targetenb_id.to_json(j);
  j.write_fieldname("sourceeNB-ID");
  sourceenb_id.to_json(j);
  j.write_fieldname("sONInformation");
  son_info.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ENBConfigurationTransferIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {129};
  return map_enum_number(options, 1, idx, "id");
}
bool enb_cfg_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  return 129 == id;
}
crit_e enb_cfg_transfer_ies_o::get_crit(const uint32_t& id)
{
  if (id == 129) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
enb_cfg_transfer_ies_o::value_c enb_cfg_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 129) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_transfer_ies_o::get_presence(const uint32_t& id)
{
  if (id == 129) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void enb_cfg_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("SONConfigurationTransfer");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE enb_cfg_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string enb_cfg_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"SONConfigurationTransfer"};
  return convert_enum_idx(options, 1, value, "enb_cfg_transfer_ies_o::value_c::types");
}

// ENBConfigurationTransfer ::= SEQUENCE
SRSASN_CODE enb_cfg_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, protocol_ies, 0, 65535, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(protocol_ies, bref, 0, 65535, true));

  return SRSASN_SUCCESS;
}
void enb_cfg_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  j.end_obj();
}

// RAT-Type ::= ENUMERATED
std::string rat_type_opts::to_string() const
{
  static const char* options[] = {"nbiot"};
  return convert_enum_idx(options, 1, value, "rat_type_e");
}

// SupportedTAs-Item-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t supported_tas_item_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {232};
  return map_enum_number(options, 1, idx, "id");
}
bool supported_tas_item_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 232 == id;
}
crit_e supported_tas_item_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 232) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
supported_tas_item_ext_ies_o::ext_c supported_tas_item_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 232) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e supported_tas_item_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 232) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void supported_tas_item_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("RAT-Type", "nbiot");
  j.end_obj();
}
SRSASN_CODE supported_tas_item_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE supported_tas_item_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string supported_tas_item_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"RAT-Type"};
  return convert_enum_idx(options, 1, value, "supported_tas_item_ext_ies_o::ext_c::types");
}

// SupportedTAs-Item ::= SEQUENCE
SRSASN_CODE supported_tas_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tac.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, broadcast_plmns, 1, 6, true));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE supported_tas_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tac.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(broadcast_plmns, bref, 1, 6, true));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void supported_tas_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("tAC", tac.to_string());
  j.start_array("broadcastPLMNs");
  for (const auto& e1 : broadcast_plmns) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// NB-IoT-DefaultPagingDRX ::= ENUMERATED
std::string nb_io_t_default_paging_drx_opts::to_string() const
{
  static const char* options[] = {"v128", "v256", "v512", "v1024"};
  return convert_enum_idx(options, 4, value, "nb_io_t_default_paging_drx_e");
}
uint16_t nb_io_t_default_paging_drx_opts::to_number() const
{
  static const uint16_t options[] = {128, 256, 512, 1024};
  return map_enum_number(options, 4, value, "nb_io_t_default_paging_drx_e");
}

// PagingDRX ::= ENUMERATED
std::string paging_drx_opts::to_string() const
{
  static const char* options[] = {"v32", "v64", "v128", "v256"};
  return convert_enum_idx(options, 4, value, "paging_drx_e");
}
uint16_t paging_drx_opts::to_number() const
{
  static const uint16_t options[] = {32, 64, 128, 256};
  return map_enum_number(options, 4, value, "paging_drx_e");
}

// ENBConfigurationUpdateIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_upd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {60, 64, 128, 137, 234};
  return map_enum_number(options, 5, idx, "id");
}
bool enb_cfg_upd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {60, 64, 128, 137, 234};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_cfg_upd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 60:
      return crit_e::ignore;
    case 64:
      return crit_e::reject;
    case 128:
      return crit_e::reject;
    case 137:
      return crit_e::ignore;
    case 234:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_cfg_upd_ies_o::value_c enb_cfg_upd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 60:
      ret.set(value_c::types::enbname);
      break;
    case 64:
      ret.set(value_c::types::supported_tas);
      break;
    case 128:
      ret.set(value_c::types::csg_id_list);
      break;
    case 137:
      ret.set(value_c::types::default_paging_drx);
      break;
    case 234:
      ret.set(value_c::types::nb_io_t_default_paging_drx);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_upd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 60:
      return presence_e::optional;
    case 64:
      return presence_e::optional;
    case 128:
      return presence_e::optional;
    case 137:
      return presence_e::optional;
    case 234:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
printable_string<1, 150, true, true>& enb_cfg_upd_ies_o::value_c::enbname()
{
  assert_choice_type("PrintableString", type_.to_string(), "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
supported_tas_l& enb_cfg_upd_ies_o::value_c::supported_tas()
{
  assert_choice_type("SupportedTAs", type_.to_string(), "Value");
  return c.get<supported_tas_l>();
}
csg_id_list_l& enb_cfg_upd_ies_o::value_c::csg_id_list()
{
  assert_choice_type("CSG-IdList", type_.to_string(), "Value");
  return c.get<csg_id_list_l>();
}
paging_drx_e& enb_cfg_upd_ies_o::value_c::default_paging_drx()
{
  assert_choice_type("PagingDRX", type_.to_string(), "Value");
  return c.get<paging_drx_e>();
}
nb_io_t_default_paging_drx_e& enb_cfg_upd_ies_o::value_c::nb_io_t_default_paging_drx()
{
  assert_choice_type("NB-IoT-DefaultPagingDRX", type_.to_string(), "Value");
  return c.get<nb_io_t_default_paging_drx_e>();
}
const printable_string<1, 150, true, true>& enb_cfg_upd_ies_o::value_c::enbname() const
{
  assert_choice_type("PrintableString", type_.to_string(), "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const supported_tas_l& enb_cfg_upd_ies_o::value_c::supported_tas() const
{
  assert_choice_type("SupportedTAs", type_.to_string(), "Value");
  return c.get<supported_tas_l>();
}
const csg_id_list_l& enb_cfg_upd_ies_o::value_c::csg_id_list() const
{
  assert_choice_type("CSG-IdList", type_.to_string(), "Value");
  return c.get<csg_id_list_l>();
}
const paging_drx_e& enb_cfg_upd_ies_o::value_c::default_paging_drx() const
{
  assert_choice_type("PagingDRX", type_.to_string(), "Value");
  return c.get<paging_drx_e>();
}
const nb_io_t_default_paging_drx_e& enb_cfg_upd_ies_o::value_c::nb_io_t_default_paging_drx() const
{
  assert_choice_type("NB-IoT-DefaultPagingDRX", type_.to_string(), "Value");
  return c.get<nb_io_t_default_paging_drx_e>();
}
void enb_cfg_upd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::enbname:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::supported_tas:
      c.destroy<supported_tas_l>();
      break;
    case types::csg_id_list:
      c.destroy<csg_id_list_l>();
      break;
    default:
      break;
  }
}
void enb_cfg_upd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enbname:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::supported_tas:
      c.init<supported_tas_l>();
      break;
    case types::csg_id_list:
      c.init<csg_id_list_l>();
      break;
    case types::default_paging_drx:
      break;
    case types::nb_io_t_default_paging_drx:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }
}
enb_cfg_upd_ies_o::value_c::value_c(const enb_cfg_upd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enbname:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_tas:
      c.init(other.c.get<supported_tas_l>());
      break;
    case types::csg_id_list:
      c.init(other.c.get<csg_id_list_l>());
      break;
    case types::default_paging_drx:
      c.init(other.c.get<paging_drx_e>());
      break;
    case types::nb_io_t_default_paging_drx:
      c.init(other.c.get<nb_io_t_default_paging_drx_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }
}
enb_cfg_upd_ies_o::value_c& enb_cfg_upd_ies_o::value_c::operator=(const enb_cfg_upd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enbname:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_tas:
      c.set(other.c.get<supported_tas_l>());
      break;
    case types::csg_id_list:
      c.set(other.c.get<csg_id_list_l>());
      break;
    case types::default_paging_drx:
      c.set(other.c.get<paging_drx_e>());
      break;
    case types::nb_io_t_default_paging_drx:
      c.set(other.c.get<nb_io_t_default_paging_drx_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }

  return *this;
}
void enb_cfg_upd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enbname:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::supported_tas:
      j.start_array("SupportedTAs");
      for (const auto& e1 : c.get<supported_tas_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::csg_id_list:
      j.start_array("CSG-IdList");
      for (const auto& e1 : c.get<csg_id_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::default_paging_drx:
      j.write_str("PagingDRX", c.get<paging_drx_e>().to_string());
      break;
    case types::nb_io_t_default_paging_drx:
      j.write_str("NB-IoT-DefaultPagingDRX", c.get<nb_io_t_default_paging_drx_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_cfg_upd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbname:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::supported_tas:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<supported_tas_l>(), 1, 256, true));
      break;
    case types::csg_id_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<csg_id_list_l>(), 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().pack(bref));
      break;
    case types::nb_io_t_default_paging_drx:
      HANDLE_CODE(c.get<nb_io_t_default_paging_drx_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enbname:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::supported_tas:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<supported_tas_l>(), bref, 1, 256, true));
      break;
    case types::csg_id_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<csg_id_list_l>(), bref, 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().unpack(bref));
      break;
    case types::nb_io_t_default_paging_drx:
      HANDLE_CODE(c.get<nb_io_t_default_paging_drx_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string enb_cfg_upd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "PrintableString", "SupportedTAs", "CSG-IdList", "PagingDRX", "NB-IoT-DefaultPagingDRX"};
  return convert_enum_idx(options, 5, value, "enb_cfg_upd_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<enb_cfg_upd_ies_o>;

enb_cfg_upd_ies_container::enb_cfg_upd_ies_container() :
  enbname(60, crit_e::ignore),
  supported_tas(64, crit_e::reject),
  csg_id_list(128, crit_e::reject),
  default_paging_drx(137, crit_e::ignore),
  nb_io_t_default_paging_drx(234, crit_e::ignore)
{}
SRSASN_CODE enb_cfg_upd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += enbname_present ? 1 : 0;
  nof_ies += supported_tas_present ? 1 : 0;
  nof_ies += csg_id_list_present ? 1 : 0;
  nof_ies += default_paging_drx_present ? 1 : 0;
  nof_ies += nb_io_t_default_paging_drx_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (enbname_present) {
    HANDLE_CODE(enbname.pack(bref));
  }
  if (supported_tas_present) {
    HANDLE_CODE(supported_tas.pack(bref));
  }
  if (csg_id_list_present) {
    HANDLE_CODE(csg_id_list.pack(bref));
  }
  if (default_paging_drx_present) {
    HANDLE_CODE(default_paging_drx.pack(bref));
  }
  if (nb_io_t_default_paging_drx_present) {
    HANDLE_CODE(nb_io_t_default_paging_drx.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<enb_cfg_upd_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 60:
        enbname_present = true;
        enbname.id      = c.id;
        enbname.crit    = c.crit;
        enbname.value   = c.value.enbname();
        break;
      case 64:
        supported_tas_present = true;
        supported_tas.id      = c.id;
        supported_tas.crit    = c.crit;
        supported_tas.value   = c.value.supported_tas();
        break;
      case 128:
        csg_id_list_present = true;
        csg_id_list.id      = c.id;
        csg_id_list.crit    = c.crit;
        csg_id_list.value   = c.value.csg_id_list();
        break;
      case 137:
        default_paging_drx_present = true;
        default_paging_drx.id      = c.id;
        default_paging_drx.crit    = c.crit;
        default_paging_drx.value   = c.value.default_paging_drx();
        break;
      case 234:
        nb_io_t_default_paging_drx_present = true;
        nb_io_t_default_paging_drx.id      = c.id;
        nb_io_t_default_paging_drx.crit    = c.crit;
        nb_io_t_default_paging_drx.value   = c.value.nb_io_t_default_paging_drx();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void enb_cfg_upd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (enbname_present) {
    j.write_fieldname("");
    enbname.to_json(j);
  }
  if (supported_tas_present) {
    j.write_fieldname("");
    supported_tas.to_json(j);
  }
  if (csg_id_list_present) {
    j.write_fieldname("");
    csg_id_list.to_json(j);
  }
  if (default_paging_drx_present) {
    j.write_fieldname("");
    default_paging_drx.to_json(j);
  }
  if (nb_io_t_default_paging_drx_present) {
    j.write_fieldname("");
    nb_io_t_default_paging_drx.to_json(j);
  }
  j.end_obj();
}

// ENBConfigurationUpdate ::= SEQUENCE
SRSASN_CODE enb_cfg_upd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void enb_cfg_upd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// ENBConfigurationUpdateAcknowledgeIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_upd_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {58};
  return map_enum_number(options, 1, idx, "id");
}
bool enb_cfg_upd_ack_ies_o::is_id_valid(const uint32_t& id)
{
  return 58 == id;
}
crit_e enb_cfg_upd_ack_ies_o::get_crit(const uint32_t& id)
{
  if (id == 58) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
enb_cfg_upd_ack_ies_o::value_c enb_cfg_upd_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 58) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_upd_ack_ies_o::get_presence(const uint32_t& id)
{
  if (id == 58) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void enb_cfg_upd_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("CriticalityDiagnostics");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE enb_cfg_upd_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string enb_cfg_upd_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"CriticalityDiagnostics"};
  return convert_enum_idx(options, 1, value, "enb_cfg_upd_ack_ies_o::value_c::types");
}

// ENBConfigurationUpdateAcknowledge ::= SEQUENCE
SRSASN_CODE enb_cfg_upd_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, protocol_ies, 0, 65535, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(protocol_ies, bref, 0, 65535, true));

  return SRSASN_SUCCESS;
}
void enb_cfg_upd_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  j.end_obj();
}

// TimeToWait ::= ENUMERATED
std::string time_to_wait_opts::to_string() const
{
  static const char* options[] = {"v1s", "v2s", "v5s", "v10s", "v20s", "v60s"};
  return convert_enum_idx(options, 6, value, "time_to_wait_e");
}
uint8_t time_to_wait_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 5, 10, 20, 60};
  return map_enum_number(options, 6, value, "time_to_wait_e");
}

// ENBConfigurationUpdateFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_cfg_upd_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {2, 65, 58};
  return map_enum_number(options, 3, idx, "id");
}
bool enb_cfg_upd_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {2, 65, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_cfg_upd_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 2:
      return crit_e::ignore;
    case 65:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_cfg_upd_fail_ies_o::value_c enb_cfg_upd_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 65:
      ret.set(value_c::types::time_to_wait);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_cfg_upd_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 2:
      return presence_e::mandatory;
    case 65:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
cause_c& enb_cfg_upd_fail_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
time_to_wait_e& enb_cfg_upd_fail_ies_o::value_c::time_to_wait()
{
  assert_choice_type("TimeToWait", type_.to_string(), "Value");
  return c.get<time_to_wait_e>();
}
crit_diagnostics_s& enb_cfg_upd_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const cause_c& enb_cfg_upd_fail_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const time_to_wait_e& enb_cfg_upd_fail_ies_o::value_c::time_to_wait() const
{
  assert_choice_type("TimeToWait", type_.to_string(), "Value");
  return c.get<time_to_wait_e>();
}
const crit_diagnostics_s& enb_cfg_upd_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void enb_cfg_upd_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void enb_cfg_upd_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cause:
      c.init<cause_c>();
      break;
    case types::time_to_wait:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }
}
enb_cfg_upd_fail_ies_o::value_c::value_c(const enb_cfg_upd_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.init(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }
}
enb_cfg_upd_fail_ies_o::value_c& enb_cfg_upd_fail_ies_o::value_c::
                                 operator=(const enb_cfg_upd_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.set(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }

  return *this;
}
void enb_cfg_upd_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::time_to_wait:
      j.write_str("TimeToWait", c.get<time_to_wait_e>().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_cfg_upd_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string enb_cfg_upd_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Cause", "TimeToWait", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "enb_cfg_upd_fail_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<enb_cfg_upd_fail_ies_o>;

enb_cfg_upd_fail_ies_container::enb_cfg_upd_fail_ies_container() :
  cause(2, crit_e::ignore),
  time_to_wait(65, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE enb_cfg_upd_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 1;
  nof_ies += time_to_wait_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(cause.pack(bref));
  if (time_to_wait_present) {
    HANDLE_CODE(time_to_wait.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 1;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<enb_cfg_upd_fail_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      case 65:
        time_to_wait_present = true;
        time_to_wait.id      = c.id;
        time_to_wait.crit    = c.crit;
        time_to_wait.value   = c.value.time_to_wait();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void enb_cfg_upd_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  cause.to_json(j);
  if (time_to_wait_present) {
    j.write_fieldname("");
    time_to_wait.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// ENBConfigurationUpdateFailure ::= SEQUENCE
SRSASN_CODE enb_cfg_upd_fail_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_cfg_upd_fail_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void enb_cfg_upd_fail_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// LAI ::= SEQUENCE
SRSASN_CODE lai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.pack(bref));
  HANDLE_CODE(lac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE lai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plm_nid.unpack(bref));
  HANDLE_CODE(lac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void lai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNidentity", plm_nid.to_string());
  j.write_str("lAC", lac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GERAN-Cell-ID ::= SEQUENCE
SRSASN_CODE geran_cell_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(lai.pack(bref));
  HANDLE_CODE(rac.pack(bref));
  HANDLE_CODE(ci.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE geran_cell_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(lai.unpack(bref));
  HANDLE_CODE(rac.unpack(bref));
  HANDLE_CODE(ci.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void geran_cell_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("lAI");
  lai.to_json(j);
  j.write_str("rAC", rac.to_string());
  j.write_str("cI", ci.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TargetRNC-ID ::= SEQUENCE
SRSASN_CODE target_rnc_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(rac_present, 1));
  HANDLE_CODE(bref.pack(extended_rnc_id_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(lai.pack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, rnc_id, (uint16_t)0u, (uint16_t)4095u, false, true));
  if (extended_rnc_id_present) {
    HANDLE_CODE(pack_integer(bref, extended_rnc_id, (uint32_t)4096u, (uint32_t)65535u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE target_rnc_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(rac_present, 1));
  HANDLE_CODE(bref.unpack(extended_rnc_id_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(lai.unpack(bref));
  if (rac_present) {
    HANDLE_CODE(rac.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(rnc_id, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  if (extended_rnc_id_present) {
    HANDLE_CODE(unpack_integer(extended_rnc_id, bref, (uint32_t)4096u, (uint32_t)65535u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void target_rnc_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("lAI");
  lai.to_json(j);
  if (rac_present) {
    j.write_str("rAC", rac.to_string());
  }
  j.write_int("rNC-ID", rnc_id);
  if (extended_rnc_id_present) {
    j.write_int("extendedRNC-ID", extended_rnc_id);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RIMRoutingAddress ::= CHOICE
void rim_routing_address_c::destroy_()
{
  switch (type_) {
    case types::geran_cell_id:
      c.destroy<geran_cell_id_s>();
      break;
    case types::target_rnc_id:
      c.destroy<target_rnc_id_s>();
      break;
    case types::ehrpd_sector_id:
      c.destroy<fixed_octstring<16, true> >();
      break;
    default:
      break;
  }
}
void rim_routing_address_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::geran_cell_id:
      c.init<geran_cell_id_s>();
      break;
    case types::target_rnc_id:
      c.init<target_rnc_id_s>();
      break;
    case types::ehrpd_sector_id:
      c.init<fixed_octstring<16, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }
}
rim_routing_address_c::rim_routing_address_c(const rim_routing_address_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::geran_cell_id:
      c.init(other.c.get<geran_cell_id_s>());
      break;
    case types::target_rnc_id:
      c.init(other.c.get<target_rnc_id_s>());
      break;
    case types::ehrpd_sector_id:
      c.init(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }
}
rim_routing_address_c& rim_routing_address_c::operator=(const rim_routing_address_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::geran_cell_id:
      c.set(other.c.get<geran_cell_id_s>());
      break;
    case types::target_rnc_id:
      c.set(other.c.get<target_rnc_id_s>());
      break;
    case types::ehrpd_sector_id:
      c.set(other.c.get<fixed_octstring<16, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }

  return *this;
}
void rim_routing_address_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::geran_cell_id:
      j.write_fieldname("gERAN-Cell-ID");
      c.get<geran_cell_id_s>().to_json(j);
      break;
    case types::target_rnc_id:
      j.write_fieldname("targetRNC-ID");
      c.get<target_rnc_id_s>().to_json(j);
      break;
    case types::ehrpd_sector_id:
      j.write_str("eHRPD-Sector-ID", c.get<fixed_octstring<16, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
  }
  j.end_obj();
}
SRSASN_CODE rim_routing_address_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::geran_cell_id:
      HANDLE_CODE(c.get<geran_cell_id_s>().pack(bref));
      break;
    case types::target_rnc_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<target_rnc_id_s>().pack(bref));
    } break;
    case types::ehrpd_sector_id: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().pack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rim_routing_address_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::geran_cell_id:
      HANDLE_CODE(c.get<geran_cell_id_s>().unpack(bref));
      break;
    case types::target_rnc_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<target_rnc_id_s>().unpack(bref));
    } break;
    case types::ehrpd_sector_id: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE((c.get<fixed_octstring<16, true> >().unpack(bref)));
    } break;
    default:
      log_invalid_choice_id(type_, "rim_routing_address_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string rim_routing_address_c::types_opts::to_string() const
{
  static const char* options[] = {"gERAN-Cell-ID", "targetRNC-ID", "eHRPD-Sector-ID"};
  return convert_enum_idx(options, 3, value, "rim_routing_address_c::types");
}

// RIMTransfer ::= SEQUENCE
SRSASN_CODE rim_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(rim_routing_address_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(rim_info.pack(bref));
  if (rim_routing_address_present) {
    HANDLE_CODE(rim_routing_address.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE rim_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(rim_routing_address_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(rim_info.unpack(bref));
  if (rim_routing_address_present) {
    HANDLE_CODE(rim_routing_address.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void rim_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("rIMInformation", rim_info.to_string());
  if (rim_routing_address_present) {
    j.write_fieldname("rIMRoutingAddress");
    rim_routing_address.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Inter-SystemInformationTransferType ::= CHOICE
void inter_sys_info_transfer_type_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("rIMTransfer");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE inter_sys_info_transfer_type_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE inter_sys_info_transfer_type_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "inter_sys_info_transfer_type_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string inter_sys_info_transfer_type_c::types_opts::to_string() const
{
  static const char* options[] = {"rIMTransfer"};
  return convert_enum_idx(options, 1, value, "inter_sys_info_transfer_type_c::types");
}

// ENBDirectInformationTransferIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_direct_info_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {121};
  return map_enum_number(options, 1, idx, "id");
}
bool enb_direct_info_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  return 121 == id;
}
crit_e enb_direct_info_transfer_ies_o::get_crit(const uint32_t& id)
{
  if (id == 121) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
enb_direct_info_transfer_ies_o::value_c enb_direct_info_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 121) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_direct_info_transfer_ies_o::get_presence(const uint32_t& id)
{
  if (id == 121) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void enb_direct_info_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("Inter-SystemInformationTransferType");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE enb_direct_info_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_direct_info_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string enb_direct_info_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Inter-SystemInformationTransferType"};
  return convert_enum_idx(options, 1, value, "enb_direct_info_transfer_ies_o::value_c::types");
}

// ENBDirectInformationTransfer ::= SEQUENCE
SRSASN_CODE enb_direct_info_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(pack_dyn_seq_of(bref, protocol_ies, 0, 65535, true));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_direct_info_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(unpack_dyn_seq_of(protocol_ies, bref, 0, 65535, true));

  return SRSASN_SUCCESS;
}
void enb_direct_info_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  j.end_obj();
}

// ENBStatusTransferIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t enb_status_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 90};
  return map_enum_number(options, 3, idx, "id");
}
bool enb_status_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 90};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e enb_status_transfer_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 90:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
enb_status_transfer_ies_o::value_c enb_status_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 90:
      ret.set(value_c::types::enb_status_transfer_transparent_container);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e enb_status_transfer_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 90:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& enb_status_transfer_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& enb_status_transfer_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
enb_status_transfer_transparent_container_s&
enb_status_transfer_ies_o::value_c::enb_status_transfer_transparent_container()
{
  assert_choice_type("ENB-StatusTransfer-TransparentContainer", type_.to_string(), "Value");
  return c.get<enb_status_transfer_transparent_container_s>();
}
const uint64_t& enb_status_transfer_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& enb_status_transfer_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const enb_status_transfer_transparent_container_s&
enb_status_transfer_ies_o::value_c::enb_status_transfer_transparent_container() const
{
  assert_choice_type("ENB-StatusTransfer-TransparentContainer", type_.to_string(), "Value");
  return c.get<enb_status_transfer_transparent_container_s>();
}
void enb_status_transfer_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::enb_status_transfer_transparent_container:
      c.destroy<enb_status_transfer_transparent_container_s>();
      break;
    default:
      break;
  }
}
void enb_status_transfer_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::enb_status_transfer_transparent_container:
      c.init<enb_status_transfer_transparent_container_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }
}
enb_status_transfer_ies_o::value_c::value_c(const enb_status_transfer_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::enb_status_transfer_transparent_container:
      c.init(other.c.get<enb_status_transfer_transparent_container_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }
}
enb_status_transfer_ies_o::value_c& enb_status_transfer_ies_o::value_c::
                                    operator=(const enb_status_transfer_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::enb_status_transfer_transparent_container:
      c.set(other.c.get<enb_status_transfer_transparent_container_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }

  return *this;
}
void enb_status_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::enb_status_transfer_transparent_container:
      j.write_fieldname("ENB-StatusTransfer-TransparentContainer");
      c.get<enb_status_transfer_transparent_container_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE enb_status_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::enb_status_transfer_transparent_container:
      HANDLE_CODE(c.get<enb_status_transfer_transparent_container_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::enb_status_transfer_transparent_container:
      HANDLE_CODE(c.get<enb_status_transfer_transparent_container_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "enb_status_transfer_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string enb_status_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "ENB-StatusTransfer-TransparentContainer"};
  return convert_enum_idx(options, 3, value, "enb_status_transfer_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<enb_status_transfer_ies_o>;

enb_status_transfer_ies_container::enb_status_transfer_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  enb_status_transfer_transparent_container(90, crit_e::reject)
{}
SRSASN_CODE enb_status_transfer_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_status_transfer_transparent_container.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<enb_status_transfer_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 90:
        nof_mandatory_ies--;
        enb_status_transfer_transparent_container.id    = c.id;
        enb_status_transfer_transparent_container.crit  = c.crit;
        enb_status_transfer_transparent_container.value = c.value.enb_status_transfer_transparent_container();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void enb_status_transfer_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_status_transfer_transparent_container.to_json(j);
  j.end_obj();
}

// ENBStatusTransfer ::= SEQUENCE
SRSASN_CODE enb_status_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE enb_status_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void enb_status_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// EUTRANResponse ::= SEQUENCE
SRSASN_CODE eutran_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(cell_id.pack(bref));
  HANDLE_CODE(eutra_ncell_load_report_resp.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutran_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(cell_id.unpack(bref));
  HANDLE_CODE(eutra_ncell_load_report_resp.unpack(bref));

  return SRSASN_SUCCESS;
}
void eutran_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cell-ID", cell_id.to_string());
  j.write_fieldname("eUTRANcellLoadReportingResponse");
  eutra_ncell_load_report_resp.to_json(j);
  j.end_obj();
}

// ErrorIndicationIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t error_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  return map_enum_number(options, 4, idx, "id");
}
bool error_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e error_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
error_ind_ies_o::value_c error_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e error_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::optional;
    case 8:
      return presence_e::optional;
    case 2:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& error_ind_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& error_ind_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
cause_c& error_ind_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& error_ind_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& error_ind_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& error_ind_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const cause_c& error_ind_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& error_ind_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void error_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void error_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
}
error_ind_ies_o::value_c::value_c(const error_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
}
error_ind_ies_o::value_c& error_ind_ies_o::value_c::operator=(const error_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }

  return *this;
}
void error_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE error_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string error_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "error_ind_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<error_ind_ies_o>;

error_ind_ies_container::error_ind_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  cause(2, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE error_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += mme_ue_s1ap_id_present ? 1 : 0;
  nof_ies += enb_ue_s1ap_id_present ? 1 : 0;
  nof_ies += cause_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (mme_ue_s1ap_id_present) {
    HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  }
  if (enb_ue_s1ap_id_present) {
    HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  }
  if (cause_present) {
    HANDLE_CODE(cause.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<error_ind_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        mme_ue_s1ap_id_present = true;
        mme_ue_s1ap_id.id      = c.id;
        mme_ue_s1ap_id.crit    = c.crit;
        mme_ue_s1ap_id.value   = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        enb_ue_s1ap_id_present = true;
        enb_ue_s1ap_id.id      = c.id;
        enb_ue_s1ap_id.crit    = c.crit;
        enb_ue_s1ap_id.value   = c.value.enb_ue_s1ap_id();
        break;
      case 2:
        cause_present = true;
        cause.id      = c.id;
        cause.crit    = c.crit;
        cause.value   = c.value.cause();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void error_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (mme_ue_s1ap_id_present) {
    j.write_fieldname("");
    mme_ue_s1ap_id.to_json(j);
  }
  if (enb_ue_s1ap_id_present) {
    j.write_fieldname("");
    enb_ue_s1ap_id.to_json(j);
  }
  if (cause_present) {
    j.write_fieldname("");
    cause.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// ErrorIndication ::= SEQUENCE
SRSASN_CODE error_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void error_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// NumberOfMeasurementReportingLevels ::= ENUMERATED
std::string nof_meas_report_levels_opts::to_string() const
{
  static const char* options[] = {"rl2", "rl3", "rl4", "rl5", "rl10"};
  return convert_enum_idx(options, 5, value, "nof_meas_report_levels_e");
}
uint8_t nof_meas_report_levels_opts::to_number() const
{
  static const uint8_t options[] = {2, 3, 4, 5, 10};
  return map_enum_number(options, 5, value, "nof_meas_report_levels_e");
}

// EventTriggeredCellLoadReportingRequest ::= SEQUENCE
SRSASN_CODE event_triggered_cell_load_report_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(nof_meas_report_levels.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE event_triggered_cell_load_report_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(nof_meas_report_levels.unpack(bref));

  return SRSASN_SUCCESS;
}
void event_triggered_cell_load_report_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("numberOfMeasurementReportingLevels", nof_meas_report_levels.to_string());
  j.end_obj();
}

// OverloadFlag ::= ENUMERATED
std::string overload_flag_opts::to_string() const
{
  static const char* options[] = {"overload"};
  return convert_enum_idx(options, 1, value, "overload_flag_e");
}

// EventTriggeredCellLoadReportingResponse ::= SEQUENCE
SRSASN_CODE event_triggered_cell_load_report_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(overload_flag_present, 1));

  HANDLE_CODE(cell_load_report_resp.pack(bref));
  if (overload_flag_present) {
    HANDLE_CODE(overload_flag.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE event_triggered_cell_load_report_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(overload_flag_present, 1));

  HANDLE_CODE(cell_load_report_resp.unpack(bref));
  if (overload_flag_present) {
    HANDLE_CODE(overload_flag.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void event_triggered_cell_load_report_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cellLoadReportingResponse");
  cell_load_report_resp.to_json(j);
  if (overload_flag_present) {
    j.write_str("overloadFlag", "overload");
  }
  j.end_obj();
}

// SourceOfUEActivityBehaviourInformation ::= ENUMERATED
std::string source_of_ue_activity_behaviour_info_opts::to_string() const
{
  static const char* options[] = {"subscription-information", "statistics"};
  return convert_enum_idx(options, 2, value, "source_of_ue_activity_behaviour_info_e");
}

// ExpectedUEActivityBehaviour ::= SEQUENCE
SRSASN_CODE expected_ue_activity_behaviour_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(expected_activity_period_present, 1));
  HANDLE_CODE(bref.pack(expected_idle_period_present, 1));
  HANDLE_CODE(bref.pack(sourceof_ue_activity_behaviour_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (expected_activity_period_present) {
    HANDLE_CODE(pack_integer(bref, expected_activity_period, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (expected_idle_period_present) {
    HANDLE_CODE(pack_integer(bref, expected_idle_period, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (sourceof_ue_activity_behaviour_info_present) {
    HANDLE_CODE(sourceof_ue_activity_behaviour_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_activity_behaviour_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(expected_activity_period_present, 1));
  HANDLE_CODE(bref.unpack(expected_idle_period_present, 1));
  HANDLE_CODE(bref.unpack(sourceof_ue_activity_behaviour_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (expected_activity_period_present) {
    HANDLE_CODE(unpack_integer(expected_activity_period, bref, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (expected_idle_period_present) {
    HANDLE_CODE(unpack_integer(expected_idle_period, bref, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (sourceof_ue_activity_behaviour_info_present) {
    HANDLE_CODE(sourceof_ue_activity_behaviour_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_activity_behaviour_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (expected_activity_period_present) {
    j.write_int("expectedActivityPeriod", expected_activity_period);
  }
  if (expected_idle_period_present) {
    j.write_int("expectedIdlePeriod", expected_idle_period);
  }
  if (sourceof_ue_activity_behaviour_info_present) {
    j.write_str("sourceofUEActivityBehaviourInformation", sourceof_ue_activity_behaviour_info.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ExpectedHOInterval ::= ENUMERATED
std::string expected_ho_interv_opts::to_string() const
{
  static const char* options[] = {"sec15", "sec30", "sec60", "sec90", "sec120", "sec180", "long-time"};
  return convert_enum_idx(options, 7, value, "expected_ho_interv_e");
}
uint8_t expected_ho_interv_opts::to_number() const
{
  static const uint8_t options[] = {15, 30, 60, 90, 120, 180};
  return map_enum_number(options, 6, value, "expected_ho_interv_e");
}

// ExpectedUEBehaviour ::= SEQUENCE
SRSASN_CODE expected_ue_behaviour_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(expected_activity_present, 1));
  HANDLE_CODE(bref.pack(expected_ho_interv_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (expected_activity_present) {
    HANDLE_CODE(expected_activity.pack(bref));
  }
  if (expected_ho_interv_present) {
    HANDLE_CODE(expected_ho_interv.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_behaviour_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(expected_activity_present, 1));
  HANDLE_CODE(bref.unpack(expected_ho_interv_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (expected_activity_present) {
    HANDLE_CODE(expected_activity.unpack(bref));
  }
  if (expected_ho_interv_present) {
    HANDLE_CODE(expected_ho_interv.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_behaviour_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (expected_activity_present) {
    j.write_fieldname("expectedActivity");
    expected_activity.to_json(j);
  }
  if (expected_ho_interv_present) {
    j.write_str("expectedHOInterval", expected_ho_interv.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TooEarlyInterRATHOReportReportFromEUTRAN ::= SEQUENCE
SRSASN_CODE too_early_inter_ratho_report_report_from_eutran_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mob_info_present, 1));

  HANDLE_CODE(uerlf_report_container.pack(bref));
  if (mob_info_present) {
    HANDLE_CODE(mob_info.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE too_early_inter_ratho_report_report_from_eutran_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mob_info_present, 1));

  HANDLE_CODE(uerlf_report_container.unpack(bref));
  if (mob_info_present) {
    HANDLE_CODE(mob_info.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void too_early_inter_ratho_report_report_from_eutran_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("uERLFReportContainer", uerlf_report_container.to_string());
  if (mob_info_present) {
    j.write_str("mobilityInformation", mob_info.to_string());
  }
  j.end_obj();
}

// FailureEventReport ::= CHOICE
void fail_event_report_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tooEarlyInterRATHOReportFromEUTRAN");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE fail_event_report_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE fail_event_report_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "fail_event_report_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string fail_event_report_c::types_opts::to_string() const
{
  static const char* options[] = {"tooEarlyInterRATHOReportFromEUTRAN"};
  return convert_enum_idx(options, 1, value, "fail_event_report_c::types");
}

// HoReportType ::= ENUMERATED
std::string ho_report_type_opts::to_string() const
{
  static const char* options[] = {"unnecessaryhotoanotherrat", "earlyirathandover"};
  return convert_enum_idx(options, 2, value, "ho_report_type_e");
}

// HoType ::= ENUMERATED
std::string ho_type_opts::to_string() const
{
  static const char* options[] = {"ltetoutran", "ltetogeran"};
  return convert_enum_idx(options, 2, value, "ho_type_e");
}

// HOReport ::= SEQUENCE
SRSASN_CODE ho_report_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(ho_type.pack(bref));
  HANDLE_CODE(ho_report_type.pack(bref));
  HANDLE_CODE(hosource_id.pack(bref));
  HANDLE_CODE(ho_target_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, candidate_cell_list, 1, 16, true));

  if (ext) {
    HANDLE_CODE(bref.pack(candidate_pci_list.is_present(), 1));

    if (candidate_pci_list.is_present()) {
      HANDLE_CODE(pack_dyn_seq_of(bref, *candidate_pci_list, 1, 16, true));
    }
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_report_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(ho_type.unpack(bref));
  HANDLE_CODE(ho_report_type.unpack(bref));
  HANDLE_CODE(hosource_id.unpack(bref));
  HANDLE_CODE(ho_target_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(candidate_cell_list, bref, 1, 16, true));

  if (ext) {
    bool candidate_pci_list_present;
    HANDLE_CODE(bref.unpack(candidate_pci_list_present, 1));
    candidate_pci_list.set_present(candidate_pci_list_present);

    if (candidate_pci_list.is_present()) {
      HANDLE_CODE(unpack_dyn_seq_of(*candidate_pci_list, bref, 1, 16, true));
    }
  }
  return SRSASN_SUCCESS;
}
void ho_report_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("hoType", ho_type.to_string());
  j.write_str("hoReportType", ho_report_type.to_string());
  j.write_fieldname("hosourceID");
  hosource_id.to_json(j);
  j.write_fieldname("hoTargetID");
  ho_target_id.to_json(j);
  j.start_array("candidateCellList");
  for (const auto& e1 : candidate_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ext) {
    if (candidate_pci_list.is_present()) {
      j.start_array("candidatePCIList");
      for (const auto& e1 : *candidate_pci_list) {
        e1.to_json(j);
      }
      j.end_array();
    }
  }
  j.end_obj();
}

// HandoverCancelIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_cancel_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_cancel_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cancel_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 2:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cancel_ies_o::value_c ho_cancel_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cancel_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_cancel_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_cancel_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
cause_c& ho_cancel_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const uint64_t& ho_cancel_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_cancel_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const cause_c& ho_cancel_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
void ho_cancel_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    default:
      break;
  }
}
void ho_cancel_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
}
ho_cancel_ies_o::value_c::value_c(const ho_cancel_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
}
ho_cancel_ies_o::value_c& ho_cancel_ies_o::value_c::operator=(const ho_cancel_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }

  return *this;
}
void ho_cancel_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cancel_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_cancel_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause"};
  return convert_enum_idx(options, 3, value, "ho_cancel_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_cancel_ies_o>;

ho_cancel_ies_container::ho_cancel_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  cause(2, crit_e::ignore)
{}
SRSASN_CODE ho_cancel_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_cancel_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cancel_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.end_obj();
}

// HandoverCancel ::= SEQUENCE
SRSASN_CODE ho_cancel_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_cancel_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// HandoverCancelAcknowledgeIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_cancel_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 58};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_cancel_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cancel_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cancel_ack_ies_o::value_c ho_cancel_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cancel_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_cancel_ack_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_cancel_ack_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
crit_diagnostics_s& ho_cancel_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_cancel_ack_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_cancel_ack_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const crit_diagnostics_s& ho_cancel_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_cancel_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_cancel_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
}
ho_cancel_ack_ies_o::value_c::value_c(const ho_cancel_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
}
ho_cancel_ack_ies_o::value_c& ho_cancel_ack_ies_o::value_c::operator=(const ho_cancel_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }

  return *this;
}
void ho_cancel_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cancel_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_cancel_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "INTEGER (0..16777215)", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ho_cancel_ack_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_cancel_ack_ies_o>;

ho_cancel_ack_ies_container::ho_cancel_ack_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_cancel_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_cancel_ack_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cancel_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverCancelAcknowledge ::= SEQUENCE
SRSASN_CODE ho_cancel_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_cancel_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// HandoverType ::= ENUMERATED
std::string handov_type_opts::to_string() const
{
  static const char* options[] = {"intralte", "ltetoutran", "ltetogeran", "utrantolte", "gerantolte"};
  return convert_enum_idx(options, 5, value, "handov_type_e");
}

// HandoverCommandIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_cmd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 1, 135, 12, 13, 123, 139, 58};
  return map_enum_number(options, 9, idx, "id");
}
bool ho_cmd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 1, 135, 12, 13, 123, 139, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cmd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 1:
      return crit_e::reject;
    case 135:
      return crit_e::reject;
    case 12:
      return crit_e::ignore;
    case 13:
      return crit_e::ignore;
    case 123:
      return crit_e::reject;
    case 139:
      return crit_e::reject;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cmd_ies_o::value_c ho_cmd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 1:
      ret.set(value_c::types::handov_type);
      break;
    case 135:
      ret.set(value_c::types::nas_security_paramsfrom_e_utran);
      break;
    case 12:
      ret.set(value_c::types::erab_subjectto_data_forwarding_list);
      break;
    case 13:
      ret.set(value_c::types::erab_to_release_list_ho_cmd);
      break;
    case 123:
      ret.set(value_c::types::target_to_source_transparent_container);
      break;
    case 139:
      ret.set(value_c::types::target_to_source_transparent_container_secondary);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cmd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 1:
      return presence_e::mandatory;
    case 135:
      return presence_e::conditional;
    case 12:
      return presence_e::optional;
    case 13:
      return presence_e::optional;
    case 123:
      return presence_e::mandatory;
    case 139:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_cmd_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_cmd_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
handov_type_e& ho_cmd_ies_o::value_c::handov_type()
{
  assert_choice_type("HandoverType", type_.to_string(), "Value");
  return c.get<handov_type_e>();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::nas_security_paramsfrom_e_utran()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
erab_ie_container_list_l<erab_data_forwarding_item_ies_o>& ho_cmd_ies_o::value_c::erab_subjectto_data_forwarding_list()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
erab_list_l& ho_cmd_ies_o::value_c::erab_to_release_list_ho_cmd()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container_secondary()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
crit_diagnostics_s& ho_cmd_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_cmd_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_cmd_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const handov_type_e& ho_cmd_ies_o::value_c::handov_type() const
{
  assert_choice_type("HandoverType", type_.to_string(), "Value");
  return c.get<handov_type_e>();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::nas_security_paramsfrom_e_utran() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const erab_ie_container_list_l<erab_data_forwarding_item_ies_o>&
ho_cmd_ies_o::value_c::erab_subjectto_data_forwarding_list() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
}
const erab_list_l& ho_cmd_ies_o::value_c::erab_to_release_list_ho_cmd() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container_secondary() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const crit_diagnostics_s& ho_cmd_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_cmd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_security_paramsfrom_e_utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.destroy<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::erab_to_release_list_ho_cmd:
      c.destroy<erab_list_l>();
      break;
    case types::target_to_source_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::target_to_source_transparent_container_secondary:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_cmd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::handov_type:
      break;
    case types::nas_security_paramsfrom_e_utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >();
      break;
    case types::erab_to_release_list_ho_cmd:
      c.init<erab_list_l>();
      break;
    case types::target_to_source_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::target_to_source_transparent_container_secondary:
      c.init<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
}
ho_cmd_ies_o::value_c::value_c(const ho_cmd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::nas_security_paramsfrom_e_utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.init(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::erab_to_release_list_ho_cmd:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::target_to_source_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::target_to_source_transparent_container_secondary:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
}
ho_cmd_ies_o::value_c& ho_cmd_ies_o::value_c::operator=(const ho_cmd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::nas_security_paramsfrom_e_utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::erab_subjectto_data_forwarding_list:
      c.set(other.c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >());
      break;
    case types::erab_to_release_list_ho_cmd:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::target_to_source_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::target_to_source_transparent_container_secondary:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }

  return *this;
}
void ho_cmd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::nas_security_paramsfrom_e_utran:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::erab_subjectto_data_forwarding_list:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}");
      break;
    case types::erab_to_release_list_ho_cmd:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::target_to_source_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::target_to_source_transparent_container_secondary:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cmd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::nas_security_paramsfrom_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), 1, 256, true));
      break;
    case types::erab_to_release_list_ho_cmd:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::target_to_source_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::nas_security_paramsfrom_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::erab_subjectto_data_forwarding_list:
      HANDLE_CODE(
          unpack_dyn_seq_of(c.get<erab_ie_container_list_l<erab_data_forwarding_item_ies_o> >(), bref, 1, 256, true));
      break;
    case types::erab_to_release_list_ho_cmd:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::target_to_source_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_cmd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "HandoverType",
                                  "OCTET STRING",
                                  "E-RAB-IE-ContainerList{{E-RABDataForwardingItemIEs}}",
                                  "E-RABList",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 9, value, "ho_cmd_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_cmd_ies_o>;

ho_cmd_ies_container::ho_cmd_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  handov_type(1, crit_e::reject),
  nas_security_paramsfrom_e_utran(135, crit_e::reject),
  erab_subjectto_data_forwarding_list(12, crit_e::ignore),
  erab_to_release_list_ho_cmd(13, crit_e::ignore),
  target_to_source_transparent_container(123, crit_e::reject),
  target_to_source_transparent_container_secondary(139, crit_e::reject),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_cmd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += nas_security_paramsfrom_e_utran_present ? 1 : 0;
  nof_ies += erab_subjectto_data_forwarding_list_present ? 1 : 0;
  nof_ies += erab_to_release_list_ho_cmd_present ? 1 : 0;
  nof_ies += target_to_source_transparent_container_secondary_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  if (nas_security_paramsfrom_e_utran_present) {
    HANDLE_CODE(nas_security_paramsfrom_e_utran.pack(bref));
  }
  if (erab_subjectto_data_forwarding_list_present) {
    HANDLE_CODE(erab_subjectto_data_forwarding_list.pack(bref));
  }
  if (erab_to_release_list_ho_cmd_present) {
    HANDLE_CODE(erab_to_release_list_ho_cmd.pack(bref));
  }
  HANDLE_CODE(target_to_source_transparent_container.pack(bref));
  if (target_to_source_transparent_container_secondary_present) {
    HANDLE_CODE(target_to_source_transparent_container_secondary.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_cmd_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 1:
        nof_mandatory_ies--;
        handov_type.id    = c.id;
        handov_type.crit  = c.crit;
        handov_type.value = c.value.handov_type();
        break;
      case 135:
        nas_security_paramsfrom_e_utran_present = true;
        nas_security_paramsfrom_e_utran.id      = c.id;
        nas_security_paramsfrom_e_utran.crit    = c.crit;
        nas_security_paramsfrom_e_utran.value   = c.value.nas_security_paramsfrom_e_utran();
        break;
      case 12:
        erab_subjectto_data_forwarding_list_present = true;
        erab_subjectto_data_forwarding_list.id      = c.id;
        erab_subjectto_data_forwarding_list.crit    = c.crit;
        erab_subjectto_data_forwarding_list.value   = c.value.erab_subjectto_data_forwarding_list();
        break;
      case 13:
        erab_to_release_list_ho_cmd_present = true;
        erab_to_release_list_ho_cmd.id      = c.id;
        erab_to_release_list_ho_cmd.crit    = c.crit;
        erab_to_release_list_ho_cmd.value   = c.value.erab_to_release_list_ho_cmd();
        break;
      case 123:
        nof_mandatory_ies--;
        target_to_source_transparent_container.id    = c.id;
        target_to_source_transparent_container.crit  = c.crit;
        target_to_source_transparent_container.value = c.value.target_to_source_transparent_container();
        break;
      case 139:
        target_to_source_transparent_container_secondary_present = true;
        target_to_source_transparent_container_secondary.id      = c.id;
        target_to_source_transparent_container_secondary.crit    = c.crit;
        target_to_source_transparent_container_secondary.value =
            c.value.target_to_source_transparent_container_secondary();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cmd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  if (nas_security_paramsfrom_e_utran_present) {
    j.write_fieldname("");
    nas_security_paramsfrom_e_utran.to_json(j);
  }
  if (erab_subjectto_data_forwarding_list_present) {
    j.write_fieldname("");
    erab_subjectto_data_forwarding_list.to_json(j);
  }
  if (erab_to_release_list_ho_cmd_present) {
    j.write_fieldname("");
    erab_to_release_list_ho_cmd.to_json(j);
  }
  j.write_fieldname("");
  target_to_source_transparent_container.to_json(j);
  if (target_to_source_transparent_container_secondary_present) {
    j.write_fieldname("");
    target_to_source_transparent_container_secondary.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverCommand ::= SEQUENCE
SRSASN_CODE ho_cmd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_cmd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// HandoverFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 2, 58};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 2, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_fail_ies_o::value_c ho_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_fail_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
cause_c& ho_fail_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& ho_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_fail_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const cause_c& ho_fail_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& ho_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
}
ho_fail_ies_o::value_c::value_c(const ho_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
}
ho_fail_ies_o::value_c& ho_fail_ies_o::value_c::operator=(const ho_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }

  return *this;
}
void ho_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ho_fail_ies_o::value_c::types");
}
uint8_t ho_fail_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "ho_fail_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_fail_ies_o>;

ho_fail_ies_container::ho_fail_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  cause(2, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_fail_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverFailure ::= SEQUENCE
SRSASN_CODE ho_fail_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_fail_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// HandoverNotifyIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_notify_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 100, 67, 176, 186};
  return map_enum_number(options, 6, idx, "id");
}
bool ho_notify_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 100, 67, 176, 186};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_notify_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 100:
      return crit_e::ignore;
    case 67:
      return crit_e::ignore;
    case 176:
      return crit_e::ignore;
    case 186:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_notify_ies_o::value_c ho_notify_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 67:
      ret.set(value_c::types::tai);
      break;
    case 176:
      ret.set(value_c::types::tunnel_info_for_bbf);
      break;
    case 186:
      ret.set(value_c::types::lhn_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_notify_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 67:
      return presence_e::mandatory;
    case 176:
      return presence_e::optional;
    case 186:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_notify_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_notify_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
eutran_cgi_s& ho_notify_ies_o::value_c::eutran_cgi()
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
tai_s& ho_notify_ies_o::value_c::tai()
{
  assert_choice_type("TAI", type_.to_string(), "Value");
  return c.get<tai_s>();
}
tunnel_info_s& ho_notify_ies_o::value_c::tunnel_info_for_bbf()
{
  assert_choice_type("TunnelInformation", type_.to_string(), "Value");
  return c.get<tunnel_info_s>();
}
unbounded_octstring<true>& ho_notify_ies_o::value_c::lhn_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& ho_notify_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_notify_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const eutran_cgi_s& ho_notify_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
const tai_s& ho_notify_ies_o::value_c::tai() const
{
  assert_choice_type("TAI", type_.to_string(), "Value");
  return c.get<tai_s>();
}
const tunnel_info_s& ho_notify_ies_o::value_c::tunnel_info_for_bbf() const
{
  assert_choice_type("TunnelInformation", type_.to_string(), "Value");
  return c.get<tunnel_info_s>();
}
const unbounded_octstring<true>& ho_notify_ies_o::value_c::lhn_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
void ho_notify_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.destroy<tunnel_info_s>();
      break;
    case types::lhn_id:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void ho_notify_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::tunnel_info_for_bbf:
      c.init<tunnel_info_s>();
      break;
    case types::lhn_id:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
}
ho_notify_ies_o::value_c::value_c(const ho_notify_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.init(other.c.get<tunnel_info_s>());
      break;
    case types::lhn_id:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
}
ho_notify_ies_o::value_c& ho_notify_ies_o::value_c::operator=(const ho_notify_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::tunnel_info_for_bbf:
      c.set(other.c.get<tunnel_info_s>());
      break;
    case types::lhn_id:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }

  return *this;
}
void ho_notify_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::tai:
      j.write_fieldname("TAI");
      c.get<tai_s>().to_json(j);
      break;
    case types::tunnel_info_for_bbf:
      j.write_fieldname("TunnelInformation");
      c.get<tunnel_info_s>().to_json(j);
      break;
    case types::lhn_id:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_notify_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().pack(bref));
      break;
    case types::lhn_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().unpack(bref));
      break;
    case types::lhn_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_notify_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "EUTRAN-CGI", "TAI", "TunnelInformation", "OCTET STRING"};
  return convert_enum_idx(options, 6, value, "ho_notify_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_notify_ies_o>;

ho_notify_ies_container::ho_notify_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  eutran_cgi(100, crit_e::ignore),
  tai(67, crit_e::ignore),
  tunnel_info_for_bbf(176, crit_e::ignore),
  lhn_id(186, crit_e::ignore)
{}
SRSASN_CODE ho_notify_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += tunnel_info_for_bbf_present ? 1 : 0;
  nof_ies += lhn_id_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  if (tunnel_info_for_bbf_present) {
    HANDLE_CODE(tunnel_info_for_bbf.pack(bref));
  }
  if (lhn_id_present) {
    HANDLE_CODE(lhn_id.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_notify_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 100:
        nof_mandatory_ies--;
        eutran_cgi.id    = c.id;
        eutran_cgi.crit  = c.crit;
        eutran_cgi.value = c.value.eutran_cgi();
        break;
      case 67:
        nof_mandatory_ies--;
        tai.id    = c.id;
        tai.crit  = c.crit;
        tai.value = c.value.tai();
        break;
      case 176:
        tunnel_info_for_bbf_present = true;
        tunnel_info_for_bbf.id      = c.id;
        tunnel_info_for_bbf.crit    = c.crit;
        tunnel_info_for_bbf.value   = c.value.tunnel_info_for_bbf();
        break;
      case 186:
        lhn_id_present = true;
        lhn_id.id      = c.id;
        lhn_id.crit    = c.crit;
        lhn_id.value   = c.value.lhn_id();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_notify_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  tai.to_json(j);
  if (tunnel_info_for_bbf_present) {
    j.write_fieldname("");
    tunnel_info_for_bbf.to_json(j);
  }
  if (lhn_id_present) {
    j.write_fieldname("");
    lhn_id.to_json(j);
  }
  j.end_obj();
}

// HandoverNotify ::= SEQUENCE
SRSASN_CODE ho_notify_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_notify_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// HandoverPreparationFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_prep_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  return map_enum_number(options, 4, idx, "id");
}
bool ho_prep_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_prep_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_prep_fail_ies_o::value_c ho_prep_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_prep_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_prep_fail_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_prep_fail_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
cause_c& ho_prep_fail_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& ho_prep_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_prep_fail_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_prep_fail_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const cause_c& ho_prep_fail_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& ho_prep_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_prep_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_prep_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
}
ho_prep_fail_ies_o::value_c::value_c(const ho_prep_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
}
ho_prep_fail_ies_o::value_c& ho_prep_fail_ies_o::value_c::operator=(const ho_prep_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }

  return *this;
}
void ho_prep_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_prep_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_prep_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "ho_prep_fail_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_prep_fail_ies_o>;

ho_prep_fail_ies_container::ho_prep_fail_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  cause(2, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE ho_prep_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_prep_fail_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_prep_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverPreparationFailure ::= SEQUENCE
SRSASN_CODE ho_prep_fail_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_prep_fail_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// MBSFN-ResultToLogInfo ::= SEQUENCE
SRSASN_CODE mbsfn_result_to_log_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbsfn_area_id_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (mbsfn_area_id_present) {
    HANDLE_CODE(pack_integer(bref, mbsfn_area_id, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  HANDLE_CODE(pack_integer(bref, carrier_freq, (uint32_t)0u, (uint32_t)262143u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mbsfn_result_to_log_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbsfn_area_id_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (mbsfn_area_id_present) {
    HANDLE_CODE(unpack_integer(mbsfn_area_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  HANDLE_CODE(unpack_integer(carrier_freq, bref, (uint32_t)0u, (uint32_t)262143u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void mbsfn_result_to_log_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (mbsfn_area_id_present) {
    j.write_int("mBSFN-AreaId", mbsfn_area_id);
  }
  j.write_int("carrierFreq", carrier_freq);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Links-to-log ::= ENUMERATED
std::string links_to_log_opts::to_string() const
{
  static const char* options[] = {"uplink", "downlink", "both-uplink-and-downlink"};
  return convert_enum_idx(options, 3, value, "links_to_log_e");
}

// LoggingDuration ::= ENUMERATED
std::string logging_dur_opts::to_string() const
{
  static const char* options[] = {"m10", "m20", "m40", "m60", "m90", "m120"};
  return convert_enum_idx(options, 6, value, "logging_dur_e");
}
uint8_t logging_dur_opts::to_number() const
{
  static const uint8_t options[] = {10, 20, 40, 60, 90, 120};
  return map_enum_number(options, 6, value, "logging_dur_e");
}

// LoggingInterval ::= ENUMERATED
std::string logging_interv_opts::to_string() const
{
  static const char* options[] = {"ms128", "ms256", "ms512", "ms1024", "ms2048", "ms3072", "ms4096", "ms6144"};
  return convert_enum_idx(options, 8, value, "logging_interv_e");
}
uint16_t logging_interv_opts::to_number() const
{
  static const uint16_t options[] = {128, 256, 512, 1024, 2048, 3072, 4096, 6144};
  return map_enum_number(options, 8, value, "logging_interv_e");
}

// M3period ::= ENUMERATED
std::string m3period_opts::to_string() const
{
  static const char* options[] = {"ms100", "ms1000", "ms10000"};
  return convert_enum_idx(options, 3, value, "m3period_e");
}
uint16_t m3period_opts::to_number() const
{
  static const uint16_t options[] = {100, 1000, 10000};
  return map_enum_number(options, 3, value, "m3period_e");
}

// M4period ::= ENUMERATED
std::string m4period_opts::to_string() const
{
  static const char* options[] = {"ms1024", "ms2048", "ms5120", "ms10240", "min1"};
  return convert_enum_idx(options, 5, value, "m4period_e");
}
uint16_t m4period_opts::to_number() const
{
  static const uint16_t options[] = {1024, 2048, 5120, 10240, 1};
  return map_enum_number(options, 5, value, "m4period_e");
}

// M5period ::= ENUMERATED
std::string m5period_opts::to_string() const
{
  static const char* options[] = {"ms1024", "ms2048", "ms5120", "ms10240", "min1"};
  return convert_enum_idx(options, 5, value, "m5period_e");
}
uint16_t m5period_opts::to_number() const
{
  static const uint16_t options[] = {1024, 2048, 5120, 10240, 1};
  return map_enum_number(options, 5, value, "m5period_e");
}

// M6delay-threshold ::= ENUMERATED
std::string m6delay_thres_opts::to_string() const
{
  static const char* options[] = {
      "ms30", "ms40", "ms50", "ms60", "ms70", "ms80", "ms90", "ms100", "ms150", "ms300", "ms500", "ms750"};
  return convert_enum_idx(options, 12, value, "m6delay_thres_e");
}
uint16_t m6delay_thres_opts::to_number() const
{
  static const uint16_t options[] = {30, 40, 50, 60, 70, 80, 90, 100, 150, 300, 500, 750};
  return map_enum_number(options, 12, value, "m6delay_thres_e");
}

// M6report-Interval ::= ENUMERATED
std::string m6report_interv_opts::to_string() const
{
  static const char* options[] = {"ms1024", "ms2048", "ms5120", "ms10240"};
  return convert_enum_idx(options, 4, value, "m6report_interv_e");
}
uint16_t m6report_interv_opts::to_number() const
{
  static const uint16_t options[] = {1024, 2048, 5120, 10240};
  return map_enum_number(options, 4, value, "m6report_interv_e");
}

// LoggedMBSFNMDT ::= SEQUENCE
SRSASN_CODE logged_mbsfnmdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mbsfn_result_to_log_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.pack(bref));
  HANDLE_CODE(logging_dur.pack(bref));
  if (mbsfn_result_to_log_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, mbsfn_result_to_log, 1, 8, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE logged_mbsfnmdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mbsfn_result_to_log_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.unpack(bref));
  HANDLE_CODE(logging_dur.unpack(bref));
  if (mbsfn_result_to_log_present) {
    HANDLE_CODE(unpack_dyn_seq_of(mbsfn_result_to_log, bref, 1, 8, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void logged_mbsfnmdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("loggingInterval", logging_interv.to_string());
  j.write_str("loggingDuration", logging_dur.to_string());
  if (mbsfn_result_to_log_present) {
    j.start_array("mBSFN-ResultToLog");
    for (const auto& e1 : mbsfn_result_to_log) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M3Configuration ::= SEQUENCE
SRSASN_CODE m3_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m3period.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m3_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m3period.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m3_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m3period", m3period.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M4Configuration ::= SEQUENCE
SRSASN_CODE m4_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m4period.pack(bref));
  HANDLE_CODE(m4_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m4_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m4period.unpack(bref));
  HANDLE_CODE(m4_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m4_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m4period", m4period.to_string());
  j.write_str("m4-links-to-log", m4_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M5Configuration ::= SEQUENCE
SRSASN_CODE m5_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m5period.pack(bref));
  HANDLE_CODE(m5_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m5_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m5period.unpack(bref));
  HANDLE_CODE(m5_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m5_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m5period", m5period.to_string());
  j.write_str("m5-links-to-log", m5_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M6Configuration ::= SEQUENCE
SRSASN_CODE m6_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(m6delay_thres_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(m6report_interv.pack(bref));
  if (m6delay_thres_present) {
    HANDLE_CODE(m6delay_thres.pack(bref));
  }
  HANDLE_CODE(m6_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m6_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(m6delay_thres_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(m6report_interv.unpack(bref));
  if (m6delay_thres_present) {
    HANDLE_CODE(m6delay_thres.unpack(bref));
  }
  HANDLE_CODE(m6_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m6_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("m6report-Interval", m6report_interv.to_string());
  if (m6delay_thres_present) {
    j.write_str("m6delay-threshold", m6delay_thres.to_string());
  }
  j.write_str("m6-links-to-log", m6_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M7Configuration ::= SEQUENCE
SRSASN_CODE m7_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, m7period, (uint8_t)1u, (uint8_t)60u, true, true));
  HANDLE_CODE(m7_links_to_log.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m7_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(m7period, bref, (uint8_t)1u, (uint8_t)60u, true, true));
  HANDLE_CODE(m7_links_to_log.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m7_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("m7period", m7period);
  j.write_str("m7-links-to-log", m7_links_to_log.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MeasurementThresholdA2 ::= CHOICE
void meas_thres_a2_c::destroy_() {}
void meas_thres_a2_c::set(types::options e)
{
  destroy_();
  type_ = e;
}
meas_thres_a2_c::meas_thres_a2_c(const meas_thres_a2_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::thres_rsrp:
      c.init(other.c.get<uint8_t>());
      break;
    case types::thres_rsrq:
      c.init(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
  }
}
meas_thres_a2_c& meas_thres_a2_c::operator=(const meas_thres_a2_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::thres_rsrp:
      c.set(other.c.get<uint8_t>());
      break;
    case types::thres_rsrq:
      c.set(other.c.get<uint8_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
  }

  return *this;
}
void meas_thres_a2_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::thres_rsrp:
      j.write_int("threshold-RSRP", c.get<uint8_t>());
      break;
    case types::thres_rsrq:
      j.write_int("threshold-RSRQ", c.get<uint8_t>());
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
  }
  j.end_obj();
}
SRSASN_CODE meas_thres_a2_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::thres_rsrp:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)97u, false, true));
      break;
    case types::thres_rsrq:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)0u, (uint8_t)34u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE meas_thres_a2_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::thres_rsrp:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)97u, false, true));
      break;
    case types::thres_rsrq:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)0u, (uint8_t)34u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "meas_thres_a2_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string meas_thres_a2_c::types_opts::to_string() const
{
  static const char* options[] = {"threshold-RSRP", "threshold-RSRQ"};
  return convert_enum_idx(options, 2, value, "meas_thres_a2_c::types");
}

// ReportAmountMDT ::= ENUMERATED
std::string report_amount_mdt_opts::to_string() const
{
  static const char* options[] = {"r1", "r2", "r4", "r8", "r16", "r32", "r64", "rinfinity"};
  return convert_enum_idx(options, 8, value, "report_amount_mdt_e");
}
int8_t report_amount_mdt_opts::to_number() const
{
  static const int8_t options[] = {1, 2, 4, 8, 16, 32, 64, -1};
  return map_enum_number(options, 8, value, "report_amount_mdt_e");
}

// ReportIntervalMDT ::= ENUMERATED
std::string report_interv_mdt_opts::to_string() const
{
  static const char* options[] = {"ms120",
                                  "ms240",
                                  "ms480",
                                  "ms640",
                                  "ms1024",
                                  "ms2048",
                                  "ms5120",
                                  "ms10240",
                                  "min1",
                                  "min6",
                                  "min12",
                                  "min30",
                                  "min60"};
  return convert_enum_idx(options, 13, value, "report_interv_mdt_e");
}
uint16_t report_interv_mdt_opts::to_number() const
{
  static const uint16_t options[] = {120, 240, 480, 640, 1024, 2048, 5120, 10240, 1, 6, 12, 30, 60};
  return map_enum_number(options, 13, value, "report_interv_mdt_e");
}

// ImmediateMDT-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t immediate_mdt_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {171, 172, 173, 174, 220, 221};
  return map_enum_number(options, 6, idx, "id");
}
bool immediate_mdt_ext_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {171, 172, 173, 174, 220, 221};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e immediate_mdt_ext_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 171:
      return crit_e::ignore;
    case 172:
      return crit_e::ignore;
    case 173:
      return crit_e::ignore;
    case 174:
      return crit_e::ignore;
    case 220:
      return crit_e::ignore;
    case 221:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
immediate_mdt_ext_ies_o::ext_c immediate_mdt_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  switch (id) {
    case 171:
      ret.set(ext_c::types::m3_cfg);
      break;
    case 172:
      ret.set(ext_c::types::m4_cfg);
      break;
    case 173:
      ret.set(ext_c::types::m5_cfg);
      break;
    case 174:
      ret.set(ext_c::types::mdt_location_info);
      break;
    case 220:
      ret.set(ext_c::types::m6_cfg);
      break;
    case 221:
      ret.set(ext_c::types::m7_cfg);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e immediate_mdt_ext_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 171:
      return presence_e::conditional;
    case 172:
      return presence_e::conditional;
    case 173:
      return presence_e::conditional;
    case 174:
      return presence_e::optional;
    case 220:
      return presence_e::conditional;
    case 221:
      return presence_e::conditional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Extension ::= OPEN TYPE
m3_cfg_s& immediate_mdt_ext_ies_o::ext_c::m3_cfg()
{
  assert_choice_type("M3Configuration", type_.to_string(), "Extension");
  return c.get<m3_cfg_s>();
}
m4_cfg_s& immediate_mdt_ext_ies_o::ext_c::m4_cfg()
{
  assert_choice_type("M4Configuration", type_.to_string(), "Extension");
  return c.get<m4_cfg_s>();
}
m5_cfg_s& immediate_mdt_ext_ies_o::ext_c::m5_cfg()
{
  assert_choice_type("M5Configuration", type_.to_string(), "Extension");
  return c.get<m5_cfg_s>();
}
fixed_bitstring<8, false, true>& immediate_mdt_ext_ies_o::ext_c::mdt_location_info()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Extension");
  return c.get<fixed_bitstring<8, false, true> >();
}
m6_cfg_s& immediate_mdt_ext_ies_o::ext_c::m6_cfg()
{
  assert_choice_type("M6Configuration", type_.to_string(), "Extension");
  return c.get<m6_cfg_s>();
}
m7_cfg_s& immediate_mdt_ext_ies_o::ext_c::m7_cfg()
{
  assert_choice_type("M7Configuration", type_.to_string(), "Extension");
  return c.get<m7_cfg_s>();
}
const m3_cfg_s& immediate_mdt_ext_ies_o::ext_c::m3_cfg() const
{
  assert_choice_type("M3Configuration", type_.to_string(), "Extension");
  return c.get<m3_cfg_s>();
}
const m4_cfg_s& immediate_mdt_ext_ies_o::ext_c::m4_cfg() const
{
  assert_choice_type("M4Configuration", type_.to_string(), "Extension");
  return c.get<m4_cfg_s>();
}
const m5_cfg_s& immediate_mdt_ext_ies_o::ext_c::m5_cfg() const
{
  assert_choice_type("M5Configuration", type_.to_string(), "Extension");
  return c.get<m5_cfg_s>();
}
const fixed_bitstring<8, false, true>& immediate_mdt_ext_ies_o::ext_c::mdt_location_info() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Extension");
  return c.get<fixed_bitstring<8, false, true> >();
}
const m6_cfg_s& immediate_mdt_ext_ies_o::ext_c::m6_cfg() const
{
  assert_choice_type("M6Configuration", type_.to_string(), "Extension");
  return c.get<m6_cfg_s>();
}
const m7_cfg_s& immediate_mdt_ext_ies_o::ext_c::m7_cfg() const
{
  assert_choice_type("M7Configuration", type_.to_string(), "Extension");
  return c.get<m7_cfg_s>();
}
void immediate_mdt_ext_ies_o::ext_c::destroy_()
{
  switch (type_) {
    case types::m3_cfg:
      c.destroy<m3_cfg_s>();
      break;
    case types::m4_cfg:
      c.destroy<m4_cfg_s>();
      break;
    case types::m5_cfg:
      c.destroy<m5_cfg_s>();
      break;
    case types::mdt_location_info:
      c.destroy<fixed_bitstring<8, false, true> >();
      break;
    case types::m6_cfg:
      c.destroy<m6_cfg_s>();
      break;
    case types::m7_cfg:
      c.destroy<m7_cfg_s>();
      break;
    default:
      break;
  }
}
void immediate_mdt_ext_ies_o::ext_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::m3_cfg:
      c.init<m3_cfg_s>();
      break;
    case types::m4_cfg:
      c.init<m4_cfg_s>();
      break;
    case types::m5_cfg:
      c.init<m5_cfg_s>();
      break;
    case types::mdt_location_info:
      c.init<fixed_bitstring<8, false, true> >();
      break;
    case types::m6_cfg:
      c.init<m6_cfg_s>();
      break;
    case types::m7_cfg:
      c.init<m7_cfg_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }
}
immediate_mdt_ext_ies_o::ext_c::ext_c(const immediate_mdt_ext_ies_o::ext_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::m3_cfg:
      c.init(other.c.get<m3_cfg_s>());
      break;
    case types::m4_cfg:
      c.init(other.c.get<m4_cfg_s>());
      break;
    case types::m5_cfg:
      c.init(other.c.get<m5_cfg_s>());
      break;
    case types::mdt_location_info:
      c.init(other.c.get<fixed_bitstring<8, false, true> >());
      break;
    case types::m6_cfg:
      c.init(other.c.get<m6_cfg_s>());
      break;
    case types::m7_cfg:
      c.init(other.c.get<m7_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }
}
immediate_mdt_ext_ies_o::ext_c& immediate_mdt_ext_ies_o::ext_c::operator=(const immediate_mdt_ext_ies_o::ext_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::m3_cfg:
      c.set(other.c.get<m3_cfg_s>());
      break;
    case types::m4_cfg:
      c.set(other.c.get<m4_cfg_s>());
      break;
    case types::m5_cfg:
      c.set(other.c.get<m5_cfg_s>());
      break;
    case types::mdt_location_info:
      c.set(other.c.get<fixed_bitstring<8, false, true> >());
      break;
    case types::m6_cfg:
      c.set(other.c.get<m6_cfg_s>());
      break;
    case types::m7_cfg:
      c.set(other.c.get<m7_cfg_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }

  return *this;
}
void immediate_mdt_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::m3_cfg:
      j.write_fieldname("M3Configuration");
      c.get<m3_cfg_s>().to_json(j);
      break;
    case types::m4_cfg:
      j.write_fieldname("M4Configuration");
      c.get<m4_cfg_s>().to_json(j);
      break;
    case types::m5_cfg:
      j.write_fieldname("M5Configuration");
      c.get<m5_cfg_s>().to_json(j);
      break;
    case types::mdt_location_info:
      j.write_str("BIT STRING", c.get<fixed_bitstring<8, false, true> >().to_string());
      break;
    case types::m6_cfg:
      j.write_fieldname("M6Configuration");
      c.get<m6_cfg_s>().to_json(j);
      break;
    case types::m7_cfg:
      j.write_fieldname("M7Configuration");
      c.get<m7_cfg_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
  }
  j.end_obj();
}
SRSASN_CODE immediate_mdt_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::m3_cfg:
      HANDLE_CODE(c.get<m3_cfg_s>().pack(bref));
      break;
    case types::m4_cfg:
      HANDLE_CODE(c.get<m4_cfg_s>().pack(bref));
      break;
    case types::m5_cfg:
      HANDLE_CODE(c.get<m5_cfg_s>().pack(bref));
      break;
    case types::mdt_location_info:
      HANDLE_CODE((c.get<fixed_bitstring<8, false, true> >().pack(bref)));
      break;
    case types::m6_cfg:
      HANDLE_CODE(c.get<m6_cfg_s>().pack(bref));
      break;
    case types::m7_cfg:
      HANDLE_CODE(c.get<m7_cfg_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE immediate_mdt_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::m3_cfg:
      HANDLE_CODE(c.get<m3_cfg_s>().unpack(bref));
      break;
    case types::m4_cfg:
      HANDLE_CODE(c.get<m4_cfg_s>().unpack(bref));
      break;
    case types::m5_cfg:
      HANDLE_CODE(c.get<m5_cfg_s>().unpack(bref));
      break;
    case types::mdt_location_info:
      HANDLE_CODE((c.get<fixed_bitstring<8, false, true> >().unpack(bref)));
      break;
    case types::m6_cfg:
      HANDLE_CODE(c.get<m6_cfg_s>().unpack(bref));
      break;
    case types::m7_cfg:
      HANDLE_CODE(c.get<m7_cfg_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "immediate_mdt_ext_ies_o::ext_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string immediate_mdt_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {
      "M3Configuration", "M4Configuration", "M5Configuration", "BIT STRING", "M6Configuration", "M7Configuration"};
  return convert_enum_idx(options, 6, value, "immediate_mdt_ext_ies_o::ext_c::types");
}
uint8_t immediate_mdt_ext_ies_o::ext_c::types_opts::to_number() const
{
  switch (value) {
    case m3_cfg:
      return 3;
    case m4_cfg:
      return 4;
    case m5_cfg:
      return 5;
    case m6_cfg:
      return 6;
    case m7_cfg:
      return 7;
    default:
      invalid_enum_number(value, "immediate_mdt_ext_ies_o::ext_c::types");
  }
  return 0;
}

// M1PeriodicReporting ::= SEQUENCE
SRSASN_CODE m1_periodic_report_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(report_interv.pack(bref));
  HANDLE_CODE(report_amount.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m1_periodic_report_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(report_interv.unpack(bref));
  HANDLE_CODE(report_amount.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m1_periodic_report_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("reportInterval", report_interv.to_string());
  j.write_str("reportAmount", report_amount.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// M1ReportingTrigger ::= ENUMERATED
std::string m1_report_trigger_opts::to_string() const
{
  static const char* options[] = {"periodic", "a2eventtriggered", "a2eventtriggered-periodic"};
  return convert_enum_idx(options, 3, value, "m1_report_trigger_e");
}

// M1ThresholdEventA2 ::= SEQUENCE
SRSASN_CODE m1_thres_event_a2_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(meas_thres.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE m1_thres_event_a2_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(meas_thres.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void m1_thres_event_a2_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("measurementThreshold");
  meas_thres.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MDTMode-ExtensionIE ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t mdt_mode_ext_ie_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {197};
  return map_enum_number(options, 1, idx, "id");
}
bool mdt_mode_ext_ie_o::is_id_valid(const uint32_t& id)
{
  return 197 == id;
}
crit_e mdt_mode_ext_ie_o::get_crit(const uint32_t& id)
{
  if (id == 197) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
mdt_mode_ext_ie_o::value_c mdt_mode_ext_ie_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 197) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e mdt_mode_ext_ie_o::get_presence(const uint32_t& id)
{
  if (id == 197) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void mdt_mode_ext_ie_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("LoggedMBSFNMDT");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE mdt_mode_ext_ie_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_mode_ext_ie_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string mdt_mode_ext_ie_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"LoggedMBSFNMDT"};
  return convert_enum_idx(options, 1, value, "mdt_mode_ext_ie_o::value_c::types");
}

template struct asn1::s1ap::protocol_ext_field_s<immediate_mdt_ext_ies_o>;

immediate_mdt_ext_ies_container::immediate_mdt_ext_ies_container() :
  m3_cfg(171, crit_e::ignore),
  m4_cfg(172, crit_e::ignore),
  m5_cfg(173, crit_e::ignore),
  mdt_location_info(174, crit_e::ignore),
  m6_cfg(220, crit_e::ignore),
  m7_cfg(221, crit_e::ignore)
{}
SRSASN_CODE immediate_mdt_ext_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += m3_cfg_present ? 1 : 0;
  nof_ies += m4_cfg_present ? 1 : 0;
  nof_ies += m5_cfg_present ? 1 : 0;
  nof_ies += mdt_location_info_present ? 1 : 0;
  nof_ies += m6_cfg_present ? 1 : 0;
  nof_ies += m7_cfg_present ? 1 : 0;
  pack_length(bref, nof_ies, 1u, 65535u, true);

  if (m3_cfg_present) {
    HANDLE_CODE(m3_cfg.pack(bref));
  }
  if (m4_cfg_present) {
    HANDLE_CODE(m4_cfg.pack(bref));
  }
  if (m5_cfg_present) {
    HANDLE_CODE(m5_cfg.pack(bref));
  }
  if (mdt_location_info_present) {
    HANDLE_CODE(mdt_location_info.pack(bref));
  }
  if (m6_cfg_present) {
    HANDLE_CODE(m6_cfg.pack(bref));
  }
  if (m7_cfg_present) {
    HANDLE_CODE(m7_cfg.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE immediate_mdt_ext_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 1u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    protocol_ext_field_s<immediate_mdt_ext_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 171:
        m3_cfg_present = true;
        m3_cfg.id      = c.id;
        m3_cfg.crit    = c.crit;
        m3_cfg.ext     = c.ext_value.m3_cfg();
        break;
      case 172:
        m4_cfg_present = true;
        m4_cfg.id      = c.id;
        m4_cfg.crit    = c.crit;
        m4_cfg.ext     = c.ext_value.m4_cfg();
        break;
      case 173:
        m5_cfg_present = true;
        m5_cfg.id      = c.id;
        m5_cfg.crit    = c.crit;
        m5_cfg.ext     = c.ext_value.m5_cfg();
        break;
      case 174:
        mdt_location_info_present = true;
        mdt_location_info.id      = c.id;
        mdt_location_info.crit    = c.crit;
        mdt_location_info.ext     = c.ext_value.mdt_location_info();
        break;
      case 220:
        m6_cfg_present = true;
        m6_cfg.id      = c.id;
        m6_cfg.crit    = c.crit;
        m6_cfg.ext     = c.ext_value.m6_cfg();
        break;
      case 221:
        m7_cfg_present = true;
        m7_cfg.id      = c.id;
        m7_cfg.crit    = c.crit;
        m7_cfg.ext     = c.ext_value.m7_cfg();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void immediate_mdt_ext_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (m3_cfg_present) {
    j.write_fieldname("");
    m3_cfg.to_json(j);
  }
  if (m4_cfg_present) {
    j.write_fieldname("");
    m4_cfg.to_json(j);
  }
  if (m5_cfg_present) {
    j.write_fieldname("");
    m5_cfg.to_json(j);
  }
  if (mdt_location_info_present) {
    j.write_fieldname("");
    mdt_location_info.to_json(j);
  }
  if (m6_cfg_present) {
    j.write_fieldname("");
    m6_cfg.to_json(j);
  }
  if (m7_cfg_present) {
    j.write_fieldname("");
    m7_cfg.to_json(j);
  }
  j.end_obj();
}

// ImmediateMDT ::= SEQUENCE
SRSASN_CODE immediate_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(m1thresevent_a2_present, 1));
  HANDLE_CODE(bref.pack(m1periodic_report_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(meass_to_activ.pack(bref));
  HANDLE_CODE(m1report_trigger.pack(bref));
  if (m1thresevent_a2_present) {
    HANDLE_CODE(m1thresevent_a2.pack(bref));
  }
  if (m1periodic_report_present) {
    HANDLE_CODE(m1periodic_report.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE immediate_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(m1thresevent_a2_present, 1));
  HANDLE_CODE(bref.unpack(m1periodic_report_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(meass_to_activ.unpack(bref));
  HANDLE_CODE(m1report_trigger.unpack(bref));
  if (m1thresevent_a2_present) {
    HANDLE_CODE(m1thresevent_a2.unpack(bref));
  }
  if (m1periodic_report_present) {
    HANDLE_CODE(m1periodic_report.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void immediate_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("measurementsToActivate", meass_to_activ.to_string());
  j.write_str("m1reportingTrigger", m1report_trigger.to_string());
  if (m1thresevent_a2_present) {
    j.write_fieldname("m1thresholdeventA2");
    m1thresevent_a2.to_json(j);
  }
  if (m1periodic_report_present) {
    j.write_fieldname("m1periodicReporting");
    m1periodic_report.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// LoggedMDT ::= SEQUENCE
SRSASN_CODE logged_mdt_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.pack(bref));
  HANDLE_CODE(logging_dur.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE logged_mdt_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(logging_interv.unpack(bref));
  HANDLE_CODE(logging_dur.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void logged_mdt_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("loggingInterval", logging_interv.to_string());
  j.write_str("loggingDuration", logging_dur.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

template struct asn1::s1ap::protocol_ie_single_container_s<mdt_mode_ext_ie_o>;

// MDT-Activation ::= ENUMERATED
std::string mdt_activation_opts::to_string() const
{
  static const char* options[] = {
      "immediate-MDT-only", "immediate-MDT-and-Trace", "logged-MDT-only", "logged-MBSFN-MDT"};
  return convert_enum_idx(options, 4, value, "mdt_activation_e");
}

// MDT-Configuration-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t mdt_cfg_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {178};
  return map_enum_number(options, 1, idx, "id");
}
bool mdt_cfg_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 178 == id;
}
crit_e mdt_cfg_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 178) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
mdt_cfg_ext_ies_o::ext_c mdt_cfg_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 178) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e mdt_cfg_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 178) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void mdt_cfg_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("MDTPLMNList");
  for (const auto& e1 : c) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.end_obj();
}
SRSASN_CODE mdt_cfg_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 16, true));
  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_cfg_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 16, true));
  return SRSASN_SUCCESS;
}

std::string mdt_cfg_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"MDTPLMNList"};
  return convert_enum_idx(options, 1, value, "mdt_cfg_ext_ies_o::ext_c::types");
}

// MDTMode ::= CHOICE
void mdt_mode_c::destroy_()
{
  switch (type_) {
    case types::immediate_mdt:
      c.destroy<immediate_mdt_s>();
      break;
    case types::logged_mdt:
      c.destroy<logged_mdt_s>();
      break;
    case types::mdt_mode_ext:
      c.destroy<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >();
      break;
    default:
      break;
  }
}
void mdt_mode_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::immediate_mdt:
      c.init<immediate_mdt_s>();
      break;
    case types::logged_mdt:
      c.init<logged_mdt_s>();
      break;
    case types::mdt_mode_ext:
      c.init<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }
}
mdt_mode_c::mdt_mode_c(const mdt_mode_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::immediate_mdt:
      c.init(other.c.get<immediate_mdt_s>());
      break;
    case types::logged_mdt:
      c.init(other.c.get<logged_mdt_s>());
      break;
    case types::mdt_mode_ext:
      c.init(other.c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }
}
mdt_mode_c& mdt_mode_c::operator=(const mdt_mode_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::immediate_mdt:
      c.set(other.c.get<immediate_mdt_s>());
      break;
    case types::logged_mdt:
      c.set(other.c.get<logged_mdt_s>());
      break;
    case types::mdt_mode_ext:
      c.set(other.c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }

  return *this;
}
void mdt_mode_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::immediate_mdt:
      j.write_fieldname("immediateMDT");
      c.get<immediate_mdt_s>().to_json(j);
      break;
    case types::logged_mdt:
      j.write_fieldname("loggedMDT");
      c.get<logged_mdt_s>().to_json(j);
      break;
    case types::mdt_mode_ext:
      j.write_fieldname("mDTMode-Extension");
      c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
  }
  j.end_obj();
}
SRSASN_CODE mdt_mode_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::immediate_mdt:
      HANDLE_CODE(c.get<immediate_mdt_s>().pack(bref));
      break;
    case types::logged_mdt:
      HANDLE_CODE(c.get<logged_mdt_s>().pack(bref));
      break;
    case types::mdt_mode_ext: {
      varlength_field_pack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >().pack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_mode_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::immediate_mdt:
      HANDLE_CODE(c.get<immediate_mdt_s>().unpack(bref));
      break;
    case types::logged_mdt:
      HANDLE_CODE(c.get<logged_mdt_s>().unpack(bref));
      break;
    case types::mdt_mode_ext: {
      varlength_field_unpack_guard varlen_scope(bref, true);
      HANDLE_CODE(c.get<protocol_ie_single_container_s<mdt_mode_ext_ie_o> >().unpack(bref));
    } break;
    default:
      log_invalid_choice_id(type_, "mdt_mode_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string mdt_mode_c::types_opts::to_string() const
{
  static const char* options[] = {"immediateMDT", "loggedMDT", "mDTMode-Extension"};
  return convert_enum_idx(options, 3, value, "mdt_mode_c::types");
}

// MDT-Configuration ::= SEQUENCE
SRSASN_CODE mdt_cfg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(mdt_activation.pack(bref));
  HANDLE_CODE(area_scope_of_mdt.pack(bref));
  HANDLE_CODE(mdt_mode.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mdt_cfg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(mdt_activation.unpack(bref));
  HANDLE_CODE(area_scope_of_mdt.unpack(bref));
  HANDLE_CODE(mdt_mode.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void mdt_cfg_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("mdt-Activation", mdt_activation.to_string());
  j.write_fieldname("areaScopeOfMDT");
  area_scope_of_mdt.to_json(j);
  j.write_fieldname("mDTMode");
  mdt_mode.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// ProSeUEtoNetworkRelaying ::= ENUMERATED
std::string pro_se_ueto_network_relaying_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pro_se_ueto_network_relaying_e");
}

// EventType ::= ENUMERATED
std::string event_type_opts::to_string() const
{
  static const char* options[] = {"direct", "change-of-serve-cell", "stop-change-of-serve-cell"};
  return convert_enum_idx(options, 3, value, "event_type_e");
}

// PedestrianUE ::= ENUMERATED
std::string pedestrian_ue_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pedestrian_ue_e");
}

// ProSeAuthorized-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t pro_se_authorized_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {216};
  return map_enum_number(options, 1, idx, "id");
}
bool pro_se_authorized_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 216 == id;
}
crit_e pro_se_authorized_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 216) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
pro_se_authorized_ext_ies_o::ext_c pro_se_authorized_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 216) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pro_se_authorized_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 216) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void pro_se_authorized_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("ProSeUEtoNetworkRelaying", c.to_string());
  j.end_obj();
}
SRSASN_CODE pro_se_authorized_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE pro_se_authorized_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string pro_se_authorized_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"ProSeUEtoNetworkRelaying"};
  return convert_enum_idx(options, 1, value, "pro_se_authorized_ext_ies_o::ext_c::types");
}
uint8_t pro_se_authorized_ext_ies_o::ext_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "pro_se_authorized_ext_ies_o::ext_c::types");
}

// ProSeDirectCommunication ::= ENUMERATED
std::string pro_se_direct_communication_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pro_se_direct_communication_e");
}

// ProSeDirectDiscovery ::= ENUMERATED
std::string pro_se_direct_discovery_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "pro_se_direct_discovery_e");
}

// ReportArea ::= ENUMERATED
std::string report_area_opts::to_string() const
{
  static const char* options[] = {"ecgi"};
  return convert_enum_idx(options, 1, value, "report_area_e");
}

// TraceActivation-ExtIEs ::= OBJECT SET OF S1AP-PROTOCOL-EXTENSION
uint32_t trace_activation_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {162};
  return map_enum_number(options, 1, idx, "id");
}
bool trace_activation_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 162 == id;
}
crit_e trace_activation_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 162) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
trace_activation_ext_ies_o::ext_c trace_activation_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 162) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e trace_activation_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 162) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void trace_activation_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("MDT-Configuration");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE trace_activation_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE trace_activation_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string trace_activation_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"MDT-Configuration"};
  return convert_enum_idx(options, 1, value, "trace_activation_ext_ies_o::ext_c::types");
}

// TraceDepth ::= ENUMERATED
std::string trace_depth_opts::to_string() const
{
  static const char* options[] = {"minimum",
                                  "medium",
                                  "maximum",
                                  "minimumWithoutVendorSpecificExtension",
                                  "mediumWithoutVendorSpecificExtension",
                                  "maximumWithoutVendorSpecificExtension"};
  return convert_enum_idx(options, 6, value, "trace_depth_e");
}

// VehicleUE ::= ENUMERATED
std::string vehicle_ue_opts::to_string() const
{
  static const char* options[] = {"authorized", "not-authorized"};
  return convert_enum_idx(options, 2, value, "vehicle_ue_e");
}

// ManagementBasedMDTAllowed ::= ENUMERATED
std::string management_based_mdt_allowed_opts::to_string() const
{
  static const char* options[] = {"allowed"};
  return convert_enum_idx(options, 1, value, "management_based_mdt_allowed_e");
}

// ProSeAuthorized ::= SEQUENCE
SRSASN_CODE pro_se_authorized_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(pro_se_direct_discovery_present, 1));
  HANDLE_CODE(bref.pack(pro_se_direct_communication_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (pro_se_direct_discovery_present) {
    HANDLE_CODE(pro_se_direct_discovery.pack(bref));
  }
  if (pro_se_direct_communication_present) {
    HANDLE_CODE(pro_se_direct_communication.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pro_se_authorized_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(pro_se_direct_discovery_present, 1));
  HANDLE_CODE(bref.unpack(pro_se_direct_communication_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (pro_se_direct_discovery_present) {
    HANDLE_CODE(pro_se_direct_discovery.unpack(bref));
  }
  if (pro_se_direct_communication_present) {
    HANDLE_CODE(pro_se_direct_communication.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void pro_se_authorized_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (pro_se_direct_discovery_present) {
    j.write_str("proSeDirectDiscovery", pro_se_direct_discovery.to_string());
  }
  if (pro_se_direct_communication_present) {
    j.write_str("proSeDirectCommunication", pro_se_direct_communication.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// RequestType ::= SEQUENCE
SRSASN_CODE request_type_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(event_type.pack(bref));
  HANDLE_CODE(report_area.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE request_type_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(event_type.unpack(bref));
  HANDLE_CODE(report_area.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void request_type_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("eventType", event_type.to_string());
  j.write_str("reportArea", "ecgi");
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SecurityContext ::= SEQUENCE
SRSASN_CODE security_context_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, next_hop_chaining_count, (uint8_t)0u, (uint8_t)7u, false, true));
  HANDLE_CODE(next_hop_param.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE security_context_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(next_hop_chaining_count, bref, (uint8_t)0u, (uint8_t)7u, false, true));
  HANDLE_CODE(next_hop_param.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void security_context_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("nextHopChainingCount", next_hop_chaining_count);
  j.write_str("nextHopParameter", next_hop_param.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TraceActivation ::= SEQUENCE
SRSASN_CODE trace_activation_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(e_utran_trace_id.pack(bref));
  HANDLE_CODE(interfaces_to_trace.pack(bref));
  HANDLE_CODE(trace_depth.pack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE trace_activation_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(e_utran_trace_id.unpack(bref));
  HANDLE_CODE(interfaces_to_trace.unpack(bref));
  HANDLE_CODE(trace_depth.unpack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void trace_activation_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("e-UTRAN-Trace-ID", e_utran_trace_id.to_string());
  j.write_str("interfacesToTrace", interfaces_to_trace.to_string());
  j.write_str("traceDepth", trace_depth.to_string());
  j.write_str("traceCollectionEntityIPAddress", trace_collection_entity_ip_address.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// UESecurityCapabilities ::= SEQUENCE
SRSASN_CODE ue_security_cap_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(encryption_algorithms.pack(bref));
  HANDLE_CODE(integrity_protection_algorithms.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_security_cap_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(encryption_algorithms.unpack(bref));
  HANDLE_CODE(integrity_protection_algorithms.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_security_cap_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("encryptionAlgorithms", encryption_algorithms.to_string());
  j.write_str("integrityProtectionAlgorithms", integrity_protection_algorithms.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UESidelinkAggregateMaximumBitrate ::= SEQUENCE
SRSASN_CODE ue_sidelink_aggregate_maximum_bitrate_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(
      pack_integer(bref, uesidelink_aggregate_maximum_bit_rate, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_sidelink_aggregate_maximum_bitrate_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(
      unpack_integer(uesidelink_aggregate_maximum_bit_rate, bref, (uint64_t)0u, (uint64_t)10000000000u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_sidelink_aggregate_maximum_bitrate_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("uESidelinkAggregateMaximumBitRate", uesidelink_aggregate_maximum_bit_rate);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UEUserPlaneCIoTSupportIndicator ::= ENUMERATED
std::string ueuser_plane_cio_tsupport_ind_opts::to_string() const
{
  static const char* options[] = {"supported"};
  return convert_enum_idx(options, 1, value, "ueuser_plane_cio_tsupport_ind_e");
}

// V2XServicesAuthorized ::= SEQUENCE
SRSASN_CODE v2xservices_authorized_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(vehicle_ue_present, 1));
  HANDLE_CODE(bref.pack(pedestrian_ue_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (vehicle_ue_present) {
    HANDLE_CODE(vehicle_ue.pack(bref));
  }
  if (pedestrian_ue_present) {
    HANDLE_CODE(pedestrian_ue.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE v2xservices_authorized_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(vehicle_ue_present, 1));
  HANDLE_CODE(bref.unpack(pedestrian_ue_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (vehicle_ue_present) {
    HANDLE_CODE(vehicle_ue.unpack(bref));
  }
  if (pedestrian_ue_present) {
    HANDLE_CODE(pedestrian_ue.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void v2xservices_authorized_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (vehicle_ue_present) {
    j.write_str("vehicleUE", vehicle_ue.to_string());
  }
  if (pedestrian_ue_present) {
    j.write_str("pedestrianUE", pedestrian_ue.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0,   1,  2,   66,  53,  104, 107, 41,  25,  98,  124, 40,  136, 127,
                                     146, 75, 158, 165, 177, 192, 196, 195, 241, 240, 248, 251, 271, 283};
  return map_enum_number(options, 28, idx, "id");
}
bool ho_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0,   1,  2,   66,  53,  104, 107, 41,  25,  98,  124, 40,  136, 127,
                                     146, 75, 158, 165, 177, 192, 196, 195, 241, 240, 248, 251, 271, 283};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 1:
      return crit_e::reject;
    case 2:
      return crit_e::ignore;
    case 66:
      return crit_e::reject;
    case 53:
      return crit_e::reject;
    case 104:
      return crit_e::reject;
    case 107:
      return crit_e::reject;
    case 41:
      return crit_e::ignore;
    case 25:
      return crit_e::ignore;
    case 98:
      return crit_e::ignore;
    case 124:
      return crit_e::ignore;
    case 40:
      return crit_e::reject;
    case 136:
      return crit_e::reject;
    case 127:
      return crit_e::reject;
    case 146:
      return crit_e::ignore;
    case 75:
      return crit_e::ignore;
    case 158:
      return crit_e::ignore;
    case 165:
      return crit_e::ignore;
    case 177:
      return crit_e::ignore;
    case 192:
      return crit_e::ignore;
    case 196:
      return crit_e::ignore;
    case 195:
      return crit_e::ignore;
    case 241:
      return crit_e::ignore;
    case 240:
      return crit_e::ignore;
    case 248:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    case 283:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_request_ies_o::value_c ho_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 1:
      ret.set(value_c::types::handov_type);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 53:
      ret.set(value_c::types::erab_to_be_setup_list_ho_req);
      break;
    case 104:
      ret.set(value_c::types::source_to_target_transparent_container);
      break;
    case 107:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 41:
      ret.set(value_c::types::ho_restrict_list);
      break;
    case 25:
      ret.set(value_c::types::trace_activation);
      break;
    case 98:
      ret.set(value_c::types::request_type);
      break;
    case 124:
      ret.set(value_c::types::srvcc_operation_possible);
      break;
    case 40:
      ret.set(value_c::types::security_context);
      break;
    case 136:
      ret.set(value_c::types::nas_security_paramsto_e_utran);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 146:
      ret.set(value_c::types::csg_membership_status);
      break;
    case 75:
      ret.set(value_c::types::gummei_id);
      break;
    case 158:
      ret.set(value_c::types::mme_ue_s1ap_id_minus2);
      break;
    case 165:
      ret.set(value_c::types::management_based_mdt_allowed);
      break;
    case 177:
      ret.set(value_c::types::management_based_mdtplmn_list);
      break;
    case 192:
      ret.set(value_c::types::masked_imeisv);
      break;
    case 196:
      ret.set(value_c::types::expected_ue_behaviour);
      break;
    case 195:
      ret.set(value_c::types::pro_se_authorized);
      break;
    case 241:
      ret.set(value_c::types::ueuser_plane_cio_tsupport_ind);
      break;
    case 240:
      ret.set(value_c::types::v2xservices_authorized);
      break;
    case 248:
      ret.set(value_c::types::ue_sidelink_aggregate_maximum_bitrate);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    case 283:
      ret.set(value_c::types::pending_data_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 1:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 66:
      return presence_e::mandatory;
    case 53:
      return presence_e::mandatory;
    case 104:
      return presence_e::mandatory;
    case 107:
      return presence_e::mandatory;
    case 41:
      return presence_e::optional;
    case 25:
      return presence_e::optional;
    case 98:
      return presence_e::optional;
    case 124:
      return presence_e::optional;
    case 40:
      return presence_e::mandatory;
    case 136:
      return presence_e::conditional;
    case 127:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    case 75:
      return presence_e::optional;
    case 158:
      return presence_e::optional;
    case 165:
      return presence_e::optional;
    case 177:
      return presence_e::optional;
    case 192:
      return presence_e::optional;
    case 196:
      return presence_e::optional;
    case 195:
      return presence_e::optional;
    case 241:
      return presence_e::optional;
    case 240:
      return presence_e::optional;
    case 248:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    case 283:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
handov_type_e& ho_request_ies_o::value_c::handov_type()
{
  assert_choice_type("HandoverType", type_.to_string(), "Value");
  return c.get<handov_type_e>();
}
cause_c& ho_request_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
ue_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o>& ho_request_ies_o::value_c::erab_to_be_setup_list_ho_req()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABToBeSetupItemHOReqIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
}
unbounded_octstring<true>& ho_request_ies_o::value_c::source_to_target_transparent_container()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
ue_security_cap_s& ho_request_ies_o::value_c::ue_security_cap()
{
  assert_choice_type("UESecurityCapabilities", type_.to_string(), "Value");
  return c.get<ue_security_cap_s>();
}
ho_restrict_list_s& ho_request_ies_o::value_c::ho_restrict_list()
{
  assert_choice_type("HandoverRestrictionList", type_.to_string(), "Value");
  return c.get<ho_restrict_list_s>();
}
trace_activation_s& ho_request_ies_o::value_c::trace_activation()
{
  assert_choice_type("TraceActivation", type_.to_string(), "Value");
  return c.get<trace_activation_s>();
}
request_type_s& ho_request_ies_o::value_c::request_type()
{
  assert_choice_type("RequestType", type_.to_string(), "Value");
  return c.get<request_type_s>();
}
srvcc_operation_possible_e& ho_request_ies_o::value_c::srvcc_operation_possible()
{
  assert_choice_type("SRVCCOperationPossible", type_.to_string(), "Value");
  return c.get<srvcc_operation_possible_e>();
}
security_context_s& ho_request_ies_o::value_c::security_context()
{
  assert_choice_type("SecurityContext", type_.to_string(), "Value");
  return c.get<security_context_s>();
}
unbounded_octstring<true>& ho_request_ies_o::value_c::nas_security_paramsto_e_utran()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<27, false, true>& ho_request_ies_o::value_c::csg_id()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
csg_membership_status_e& ho_request_ies_o::value_c::csg_membership_status()
{
  assert_choice_type("CSGMembershipStatus", type_.to_string(), "Value");
  return c.get<csg_membership_status_e>();
}
gummei_s& ho_request_ies_o::value_c::gummei_id()
{
  assert_choice_type("GUMMEI", type_.to_string(), "Value");
  return c.get<gummei_s>();
}
uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id_minus2()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
management_based_mdt_allowed_e& ho_request_ies_o::value_c::management_based_mdt_allowed()
{
  assert_choice_type("ManagementBasedMDTAllowed", type_.to_string(), "Value");
  return c.get<management_based_mdt_allowed_e>();
}
mdtplmn_list_l& ho_request_ies_o::value_c::management_based_mdtplmn_list()
{
  assert_choice_type("MDTPLMNList", type_.to_string(), "Value");
  return c.get<mdtplmn_list_l>();
}
fixed_bitstring<64, false, true>& ho_request_ies_o::value_c::masked_imeisv()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
expected_ue_behaviour_s& ho_request_ies_o::value_c::expected_ue_behaviour()
{
  assert_choice_type("ExpectedUEBehaviour", type_.to_string(), "Value");
  return c.get<expected_ue_behaviour_s>();
}
pro_se_authorized_s& ho_request_ies_o::value_c::pro_se_authorized()
{
  assert_choice_type("ProSeAuthorized", type_.to_string(), "Value");
  return c.get<pro_se_authorized_s>();
}
ueuser_plane_cio_tsupport_ind_e& ho_request_ies_o::value_c::ueuser_plane_cio_tsupport_ind()
{
  assert_choice_type("UEUserPlaneCIoTSupportIndicator", type_.to_string(), "Value");
  return c.get<ueuser_plane_cio_tsupport_ind_e>();
}
v2xservices_authorized_s& ho_request_ies_o::value_c::v2xservices_authorized()
{
  assert_choice_type("V2XServicesAuthorized", type_.to_string(), "Value");
  return c.get<v2xservices_authorized_s>();
}
ue_sidelink_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ue_sidelink_aggregate_maximum_bitrate()
{
  assert_choice_type("UESidelinkAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_sidelink_aggregate_maximum_bitrate_s>();
}
enhanced_coverage_restricted_e& ho_request_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
ce_mode_brestricted_e& ho_request_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
pending_data_ind_e& ho_request_ies_o::value_c::pending_data_ind()
{
  assert_choice_type("PendingDataIndication", type_.to_string(), "Value");
  return c.get<pending_data_ind_e>();
}
const uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const handov_type_e& ho_request_ies_o::value_c::handov_type() const
{
  assert_choice_type("HandoverType", type_.to_string(), "Value");
  return c.get<handov_type_e>();
}
const cause_c& ho_request_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const ue_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o>&
ho_request_ies_o::value_c::erab_to_be_setup_list_ho_req() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABToBeSetupItemHOReqIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
}
const unbounded_octstring<true>& ho_request_ies_o::value_c::source_to_target_transparent_container() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const ue_security_cap_s& ho_request_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type("UESecurityCapabilities", type_.to_string(), "Value");
  return c.get<ue_security_cap_s>();
}
const ho_restrict_list_s& ho_request_ies_o::value_c::ho_restrict_list() const
{
  assert_choice_type("HandoverRestrictionList", type_.to_string(), "Value");
  return c.get<ho_restrict_list_s>();
}
const trace_activation_s& ho_request_ies_o::value_c::trace_activation() const
{
  assert_choice_type("TraceActivation", type_.to_string(), "Value");
  return c.get<trace_activation_s>();
}
const request_type_s& ho_request_ies_o::value_c::request_type() const
{
  assert_choice_type("RequestType", type_.to_string(), "Value");
  return c.get<request_type_s>();
}
const srvcc_operation_possible_e& ho_request_ies_o::value_c::srvcc_operation_possible() const
{
  assert_choice_type("SRVCCOperationPossible", type_.to_string(), "Value");
  return c.get<srvcc_operation_possible_e>();
}
const security_context_s& ho_request_ies_o::value_c::security_context() const
{
  assert_choice_type("SecurityContext", type_.to_string(), "Value");
  return c.get<security_context_s>();
}
const unbounded_octstring<true>& ho_request_ies_o::value_c::nas_security_paramsto_e_utran() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<27, false, true>& ho_request_ies_o::value_c::csg_id() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const csg_membership_status_e& ho_request_ies_o::value_c::csg_membership_status() const
{
  assert_choice_type("CSGMembershipStatus", type_.to_string(), "Value");
  return c.get<csg_membership_status_e>();
}
const gummei_s& ho_request_ies_o::value_c::gummei_id() const
{
  assert_choice_type("GUMMEI", type_.to_string(), "Value");
  return c.get<gummei_s>();
}
const uint64_t& ho_request_ies_o::value_c::mme_ue_s1ap_id_minus2() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const management_based_mdt_allowed_e& ho_request_ies_o::value_c::management_based_mdt_allowed() const
{
  assert_choice_type("ManagementBasedMDTAllowed", type_.to_string(), "Value");
  return c.get<management_based_mdt_allowed_e>();
}
const mdtplmn_list_l& ho_request_ies_o::value_c::management_based_mdtplmn_list() const
{
  assert_choice_type("MDTPLMNList", type_.to_string(), "Value");
  return c.get<mdtplmn_list_l>();
}
const fixed_bitstring<64, false, true>& ho_request_ies_o::value_c::masked_imeisv() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
const expected_ue_behaviour_s& ho_request_ies_o::value_c::expected_ue_behaviour() const
{
  assert_choice_type("ExpectedUEBehaviour", type_.to_string(), "Value");
  return c.get<expected_ue_behaviour_s>();
}
const pro_se_authorized_s& ho_request_ies_o::value_c::pro_se_authorized() const
{
  assert_choice_type("ProSeAuthorized", type_.to_string(), "Value");
  return c.get<pro_se_authorized_s>();
}
const ueuser_plane_cio_tsupport_ind_e& ho_request_ies_o::value_c::ueuser_plane_cio_tsupport_ind() const
{
  assert_choice_type("UEUserPlaneCIoTSupportIndicator", type_.to_string(), "Value");
  return c.get<ueuser_plane_cio_tsupport_ind_e>();
}
const v2xservices_authorized_s& ho_request_ies_o::value_c::v2xservices_authorized() const
{
  assert_choice_type("V2XServicesAuthorized", type_.to_string(), "Value");
  return c.get<v2xservices_authorized_s>();
}
const ue_sidelink_aggregate_maximum_bitrate_s& ho_request_ies_o::value_c::ue_sidelink_aggregate_maximum_bitrate() const
{
  assert_choice_type("UESidelinkAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_sidelink_aggregate_maximum_bitrate_s>();
}
const enhanced_coverage_restricted_e& ho_request_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const ce_mode_brestricted_e& ho_request_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
const pending_data_ind_e& ho_request_ies_o::value_c::pending_data_ind() const
{
  assert_choice_type("PendingDataIndication", type_.to_string(), "Value");
  return c.get<pending_data_ind_e>();
}
void ho_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.destroy<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
      break;
    case types::source_to_target_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::ho_restrict_list:
      c.destroy<ho_restrict_list_s>();
      break;
    case types::trace_activation:
      c.destroy<trace_activation_s>();
      break;
    case types::request_type:
      c.destroy<request_type_s>();
      break;
    case types::security_context:
      c.destroy<security_context_s>();
      break;
    case types::nas_security_paramsto_e_utran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    case types::gummei_id:
      c.destroy<gummei_s>();
      break;
    case types::management_based_mdtplmn_list:
      c.destroy<mdtplmn_list_l>();
      break;
    case types::masked_imeisv:
      c.destroy<fixed_bitstring<64, false, true> >();
      break;
    case types::expected_ue_behaviour:
      c.destroy<expected_ue_behaviour_s>();
      break;
    case types::pro_se_authorized:
      c.destroy<pro_se_authorized_s>();
      break;
    case types::v2xservices_authorized:
      c.destroy<v2xservices_authorized_s>();
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.destroy<ue_sidelink_aggregate_maximum_bitrate_s>();
      break;
    default:
      break;
  }
}
void ho_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::handov_type:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.init<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >();
      break;
    case types::source_to_target_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::ho_restrict_list:
      c.init<ho_restrict_list_s>();
      break;
    case types::trace_activation:
      c.init<trace_activation_s>();
      break;
    case types::request_type:
      c.init<request_type_s>();
      break;
    case types::srvcc_operation_possible:
      break;
    case types::security_context:
      c.init<security_context_s>();
      break;
    case types::nas_security_paramsto_e_utran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::csg_membership_status:
      break;
    case types::gummei_id:
      c.init<gummei_s>();
      break;
    case types::mme_ue_s1ap_id_minus2:
      break;
    case types::management_based_mdt_allowed:
      break;
    case types::management_based_mdtplmn_list:
      c.init<mdtplmn_list_l>();
      break;
    case types::masked_imeisv:
      c.init<fixed_bitstring<64, false, true> >();
      break;
    case types::expected_ue_behaviour:
      c.init<expected_ue_behaviour_s>();
      break;
    case types::pro_se_authorized:
      c.init<pro_se_authorized_s>();
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      break;
    case types::v2xservices_authorized:
      c.init<v2xservices_authorized_s>();
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.init<ue_sidelink_aggregate_maximum_bitrate_s>();
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::ce_mode_brestricted:
      break;
    case types::pending_data_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
}
ho_request_ies_o::value_c::value_c(const ho_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.init(other.c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >());
      break;
    case types::source_to_target_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::ho_restrict_list:
      c.init(other.c.get<ho_restrict_list_s>());
      break;
    case types::trace_activation:
      c.init(other.c.get<trace_activation_s>());
      break;
    case types::request_type:
      c.init(other.c.get<request_type_s>());
      break;
    case types::srvcc_operation_possible:
      c.init(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::security_context:
      c.init(other.c.get<security_context_s>());
      break;
    case types::nas_security_paramsto_e_utran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::csg_membership_status:
      c.init(other.c.get<csg_membership_status_e>());
      break;
    case types::gummei_id:
      c.init(other.c.get<gummei_s>());
      break;
    case types::mme_ue_s1ap_id_minus2:
      c.init(other.c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      c.init(other.c.get<management_based_mdt_allowed_e>());
      break;
    case types::management_based_mdtplmn_list:
      c.init(other.c.get<mdtplmn_list_l>());
      break;
    case types::masked_imeisv:
      c.init(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::expected_ue_behaviour:
      c.init(other.c.get<expected_ue_behaviour_s>());
      break;
    case types::pro_se_authorized:
      c.init(other.c.get<pro_se_authorized_s>());
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      c.init(other.c.get<ueuser_plane_cio_tsupport_ind_e>());
      break;
    case types::v2xservices_authorized:
      c.init(other.c.get<v2xservices_authorized_s>());
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.init(other.c.get<ue_sidelink_aggregate_maximum_bitrate_s>());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::pending_data_ind:
      c.init(other.c.get<pending_data_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
}
ho_request_ies_o::value_c& ho_request_ies_o::value_c::operator=(const ho_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_ho_req:
      c.set(other.c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >());
      break;
    case types::source_to_target_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::ho_restrict_list:
      c.set(other.c.get<ho_restrict_list_s>());
      break;
    case types::trace_activation:
      c.set(other.c.get<trace_activation_s>());
      break;
    case types::request_type:
      c.set(other.c.get<request_type_s>());
      break;
    case types::srvcc_operation_possible:
      c.set(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::security_context:
      c.set(other.c.get<security_context_s>());
      break;
    case types::nas_security_paramsto_e_utran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::csg_membership_status:
      c.set(other.c.get<csg_membership_status_e>());
      break;
    case types::gummei_id:
      c.set(other.c.get<gummei_s>());
      break;
    case types::mme_ue_s1ap_id_minus2:
      c.set(other.c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      c.set(other.c.get<management_based_mdt_allowed_e>());
      break;
    case types::management_based_mdtplmn_list:
      c.set(other.c.get<mdtplmn_list_l>());
      break;
    case types::masked_imeisv:
      c.set(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::expected_ue_behaviour:
      c.set(other.c.get<expected_ue_behaviour_s>());
      break;
    case types::pro_se_authorized:
      c.set(other.c.get<pro_se_authorized_s>());
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      c.set(other.c.get<ueuser_plane_cio_tsupport_ind_e>());
      break;
    case types::v2xservices_authorized:
      c.set(other.c.get<v2xservices_authorized_s>());
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.set(other.c.get<ue_sidelink_aggregate_maximum_bitrate_s>());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::pending_data_ind:
      c.set(other.c.get<pending_data_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }

  return *this;
}
void ho_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_setup_list_ho_req:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABToBeSetupItemHOReqIEs}}");
      break;
    case types::source_to_target_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::ho_restrict_list:
      j.write_fieldname("HandoverRestrictionList");
      c.get<ho_restrict_list_s>().to_json(j);
      break;
    case types::trace_activation:
      j.write_fieldname("TraceActivation");
      c.get<trace_activation_s>().to_json(j);
      break;
    case types::request_type:
      j.write_fieldname("RequestType");
      c.get<request_type_s>().to_json(j);
      break;
    case types::srvcc_operation_possible:
      j.write_str("SRVCCOperationPossible", "possible");
      break;
    case types::security_context:
      j.write_fieldname("SecurityContext");
      c.get<security_context_s>().to_json(j);
      break;
    case types::nas_security_paramsto_e_utran:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::csg_membership_status:
      j.write_str("CSGMembershipStatus", c.get<csg_membership_status_e>().to_string());
      break;
    case types::gummei_id:
      j.write_fieldname("GUMMEI");
      c.get<gummei_s>().to_json(j);
      break;
    case types::mme_ue_s1ap_id_minus2:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      j.write_str("ManagementBasedMDTAllowed", "allowed");
      break;
    case types::management_based_mdtplmn_list:
      j.start_array("MDTPLMNList");
      for (const auto& e1 : c.get<mdtplmn_list_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    case types::masked_imeisv:
      j.write_str("BIT STRING", c.get<fixed_bitstring<64, false, true> >().to_string());
      break;
    case types::expected_ue_behaviour:
      j.write_fieldname("ExpectedUEBehaviour");
      c.get<expected_ue_behaviour_s>().to_json(j);
      break;
    case types::pro_se_authorized:
      j.write_fieldname("ProSeAuthorized");
      c.get<pro_se_authorized_s>().to_json(j);
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      j.write_str("UEUserPlaneCIoTSupportIndicator", "supported");
      break;
    case types::v2xservices_authorized:
      j.write_fieldname("V2XServicesAuthorized");
      c.get<v2xservices_authorized_s>().to_json(j);
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      j.write_fieldname("UESidelinkAggregateMaximumBitrate");
      c.get<ue_sidelink_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    case types::pending_data_ind:
      j.write_str("PendingDataIndication", "true");
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_setup_list_ho_req:
      HANDLE_CODE(
          pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >(), 1, 256, true));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().pack(bref));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().pack(bref));
      break;
    case types::request_type:
      HANDLE_CODE(c.get<request_type_s>().pack(bref));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().pack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().pack(bref));
      break;
    case types::nas_security_paramsto_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().pack(bref));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().pack(bref));
      break;
    case types::mme_ue_s1ap_id_minus2:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::management_based_mdt_allowed:
      HANDLE_CODE(c.get<management_based_mdt_allowed_e>().pack(bref));
      break;
    case types::management_based_mdtplmn_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<mdtplmn_list_l>(), 1, 16, true));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().pack(bref)));
      break;
    case types::expected_ue_behaviour:
      HANDLE_CODE(c.get<expected_ue_behaviour_s>().pack(bref));
      break;
    case types::pro_se_authorized:
      HANDLE_CODE(c.get<pro_se_authorized_s>().pack(bref));
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      HANDLE_CODE(c.get<ueuser_plane_cio_tsupport_ind_e>().pack(bref));
      break;
    case types::v2xservices_authorized:
      HANDLE_CODE(c.get<v2xservices_authorized_s>().pack(bref));
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_sidelink_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_setup_list_ho_req:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_to_be_setup_item_ho_req_ies_o> >(), bref, 1, 256, true));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().unpack(bref));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().unpack(bref));
      break;
    case types::request_type:
      HANDLE_CODE(c.get<request_type_s>().unpack(bref));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().unpack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().unpack(bref));
      break;
    case types::nas_security_paramsto_e_utran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().unpack(bref));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().unpack(bref));
      break;
    case types::mme_ue_s1ap_id_minus2:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::management_based_mdt_allowed:
      HANDLE_CODE(c.get<management_based_mdt_allowed_e>().unpack(bref));
      break;
    case types::management_based_mdtplmn_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<mdtplmn_list_l>(), bref, 1, 16, true));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().unpack(bref)));
      break;
    case types::expected_ue_behaviour:
      HANDLE_CODE(c.get<expected_ue_behaviour_s>().unpack(bref));
      break;
    case types::pro_se_authorized:
      HANDLE_CODE(c.get<pro_se_authorized_s>().unpack(bref));
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      HANDLE_CODE(c.get<ueuser_plane_cio_tsupport_ind_e>().unpack(bref));
      break;
    case types::v2xservices_authorized:
      HANDLE_CODE(c.get<v2xservices_authorized_s>().unpack(bref));
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_sidelink_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "HandoverType",
                                  "Cause",
                                  "UEAggregateMaximumBitrate",
                                  "E-RAB-IE-ContainerList{{E-RABToBeSetupItemHOReqIEs}}",
                                  "OCTET STRING",
                                  "UESecurityCapabilities",
                                  "HandoverRestrictionList",
                                  "TraceActivation",
                                  "RequestType",
                                  "SRVCCOperationPossible",
                                  "SecurityContext",
                                  "OCTET STRING",
                                  "BIT STRING",
                                  "CSGMembershipStatus",
                                  "GUMMEI",
                                  "INTEGER (0..4294967295)",
                                  "ManagementBasedMDTAllowed",
                                  "MDTPLMNList",
                                  "BIT STRING",
                                  "ExpectedUEBehaviour",
                                  "ProSeAuthorized",
                                  "UEUserPlaneCIoTSupportIndicator",
                                  "V2XServicesAuthorized",
                                  "UESidelinkAggregateMaximumBitrate",
                                  "EnhancedCoverageRestricted",
                                  "CE-ModeBRestricted",
                                  "PendingDataIndication"};
  return convert_enum_idx(options, 28, value, "ho_request_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_request_ies_o>;

ho_request_ies_container::ho_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  handov_type(1, crit_e::reject),
  cause(2, crit_e::ignore),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_setup_list_ho_req(53, crit_e::reject),
  source_to_target_transparent_container(104, crit_e::reject),
  ue_security_cap(107, crit_e::reject),
  ho_restrict_list(41, crit_e::ignore),
  trace_activation(25, crit_e::ignore),
  request_type(98, crit_e::ignore),
  srvcc_operation_possible(124, crit_e::ignore),
  security_context(40, crit_e::reject),
  nas_security_paramsto_e_utran(136, crit_e::reject),
  csg_id(127, crit_e::reject),
  csg_membership_status(146, crit_e::ignore),
  gummei_id(75, crit_e::ignore),
  mme_ue_s1ap_id_minus2(158, crit_e::ignore),
  management_based_mdt_allowed(165, crit_e::ignore),
  management_based_mdtplmn_list(177, crit_e::ignore),
  masked_imeisv(192, crit_e::ignore),
  expected_ue_behaviour(196, crit_e::ignore),
  pro_se_authorized(195, crit_e::ignore),
  ueuser_plane_cio_tsupport_ind(241, crit_e::ignore),
  v2xservices_authorized(240, crit_e::ignore),
  ue_sidelink_aggregate_maximum_bitrate(248, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore),
  pending_data_ind(283, crit_e::ignore)
{}
SRSASN_CODE ho_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 8;
  nof_ies += ho_restrict_list_present ? 1 : 0;
  nof_ies += trace_activation_present ? 1 : 0;
  nof_ies += request_type_present ? 1 : 0;
  nof_ies += srvcc_operation_possible_present ? 1 : 0;
  nof_ies += nas_security_paramsto_e_utran_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += csg_membership_status_present ? 1 : 0;
  nof_ies += gummei_id_present ? 1 : 0;
  nof_ies += mme_ue_s1ap_id_minus2_present ? 1 : 0;
  nof_ies += management_based_mdt_allowed_present ? 1 : 0;
  nof_ies += management_based_mdtplmn_list_present ? 1 : 0;
  nof_ies += masked_imeisv_present ? 1 : 0;
  nof_ies += expected_ue_behaviour_present ? 1 : 0;
  nof_ies += pro_se_authorized_present ? 1 : 0;
  nof_ies += ueuser_plane_cio_tsupport_ind_present ? 1 : 0;
  nof_ies += v2xservices_authorized_present ? 1 : 0;
  nof_ies += ue_sidelink_aggregate_maximum_bitrate_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  nof_ies += pending_data_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  HANDLE_CODE(erab_to_be_setup_list_ho_req.pack(bref));
  HANDLE_CODE(source_to_target_transparent_container.pack(bref));
  HANDLE_CODE(ue_security_cap.pack(bref));
  if (ho_restrict_list_present) {
    HANDLE_CODE(ho_restrict_list.pack(bref));
  }
  if (trace_activation_present) {
    HANDLE_CODE(trace_activation.pack(bref));
  }
  if (request_type_present) {
    HANDLE_CODE(request_type.pack(bref));
  }
  if (srvcc_operation_possible_present) {
    HANDLE_CODE(srvcc_operation_possible.pack(bref));
  }
  HANDLE_CODE(security_context.pack(bref));
  if (nas_security_paramsto_e_utran_present) {
    HANDLE_CODE(nas_security_paramsto_e_utran.pack(bref));
  }
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (csg_membership_status_present) {
    HANDLE_CODE(csg_membership_status.pack(bref));
  }
  if (gummei_id_present) {
    HANDLE_CODE(gummei_id.pack(bref));
  }
  if (mme_ue_s1ap_id_minus2_present) {
    HANDLE_CODE(mme_ue_s1ap_id_minus2.pack(bref));
  }
  if (management_based_mdt_allowed_present) {
    HANDLE_CODE(management_based_mdt_allowed.pack(bref));
  }
  if (management_based_mdtplmn_list_present) {
    HANDLE_CODE(management_based_mdtplmn_list.pack(bref));
  }
  if (masked_imeisv_present) {
    HANDLE_CODE(masked_imeisv.pack(bref));
  }
  if (expected_ue_behaviour_present) {
    HANDLE_CODE(expected_ue_behaviour.pack(bref));
  }
  if (pro_se_authorized_present) {
    HANDLE_CODE(pro_se_authorized.pack(bref));
  }
  if (ueuser_plane_cio_tsupport_ind_present) {
    HANDLE_CODE(ueuser_plane_cio_tsupport_ind.pack(bref));
  }
  if (v2xservices_authorized_present) {
    HANDLE_CODE(v2xservices_authorized.pack(bref));
  }
  if (ue_sidelink_aggregate_maximum_bitrate_present) {
    HANDLE_CODE(ue_sidelink_aggregate_maximum_bitrate.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }
  if (pending_data_ind_present) {
    HANDLE_CODE(pending_data_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 8;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_request_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 1:
        nof_mandatory_ies--;
        handov_type.id    = c.id;
        handov_type.crit  = c.crit;
        handov_type.value = c.value.handov_type();
        break;
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      case 66:
        nof_mandatory_ies--;
        ueaggregate_maximum_bitrate.id    = c.id;
        ueaggregate_maximum_bitrate.crit  = c.crit;
        ueaggregate_maximum_bitrate.value = c.value.ueaggregate_maximum_bitrate();
        break;
      case 53:
        nof_mandatory_ies--;
        erab_to_be_setup_list_ho_req.id    = c.id;
        erab_to_be_setup_list_ho_req.crit  = c.crit;
        erab_to_be_setup_list_ho_req.value = c.value.erab_to_be_setup_list_ho_req();
        break;
      case 104:
        nof_mandatory_ies--;
        source_to_target_transparent_container.id    = c.id;
        source_to_target_transparent_container.crit  = c.crit;
        source_to_target_transparent_container.value = c.value.source_to_target_transparent_container();
        break;
      case 107:
        nof_mandatory_ies--;
        ue_security_cap.id    = c.id;
        ue_security_cap.crit  = c.crit;
        ue_security_cap.value = c.value.ue_security_cap();
        break;
      case 41:
        ho_restrict_list_present = true;
        ho_restrict_list.id      = c.id;
        ho_restrict_list.crit    = c.crit;
        ho_restrict_list.value   = c.value.ho_restrict_list();
        break;
      case 25:
        trace_activation_present = true;
        trace_activation.id      = c.id;
        trace_activation.crit    = c.crit;
        trace_activation.value   = c.value.trace_activation();
        break;
      case 98:
        request_type_present = true;
        request_type.id      = c.id;
        request_type.crit    = c.crit;
        request_type.value   = c.value.request_type();
        break;
      case 124:
        srvcc_operation_possible_present = true;
        srvcc_operation_possible.id      = c.id;
        srvcc_operation_possible.crit    = c.crit;
        srvcc_operation_possible.value   = c.value.srvcc_operation_possible();
        break;
      case 40:
        nof_mandatory_ies--;
        security_context.id    = c.id;
        security_context.crit  = c.crit;
        security_context.value = c.value.security_context();
        break;
      case 136:
        nas_security_paramsto_e_utran_present = true;
        nas_security_paramsto_e_utran.id      = c.id;
        nas_security_paramsto_e_utran.crit    = c.crit;
        nas_security_paramsto_e_utran.value   = c.value.nas_security_paramsto_e_utran();
        break;
      case 127:
        csg_id_present = true;
        csg_id.id      = c.id;
        csg_id.crit    = c.crit;
        csg_id.value   = c.value.csg_id();
        break;
      case 146:
        csg_membership_status_present = true;
        csg_membership_status.id      = c.id;
        csg_membership_status.crit    = c.crit;
        csg_membership_status.value   = c.value.csg_membership_status();
        break;
      case 75:
        gummei_id_present = true;
        gummei_id.id      = c.id;
        gummei_id.crit    = c.crit;
        gummei_id.value   = c.value.gummei_id();
        break;
      case 158:
        mme_ue_s1ap_id_minus2_present = true;
        mme_ue_s1ap_id_minus2.id      = c.id;
        mme_ue_s1ap_id_minus2.crit    = c.crit;
        mme_ue_s1ap_id_minus2.value   = c.value.mme_ue_s1ap_id_minus2();
        break;
      case 165:
        management_based_mdt_allowed_present = true;
        management_based_mdt_allowed.id      = c.id;
        management_based_mdt_allowed.crit    = c.crit;
        management_based_mdt_allowed.value   = c.value.management_based_mdt_allowed();
        break;
      case 177:
        management_based_mdtplmn_list_present = true;
        management_based_mdtplmn_list.id      = c.id;
        management_based_mdtplmn_list.crit    = c.crit;
        management_based_mdtplmn_list.value   = c.value.management_based_mdtplmn_list();
        break;
      case 192:
        masked_imeisv_present = true;
        masked_imeisv.id      = c.id;
        masked_imeisv.crit    = c.crit;
        masked_imeisv.value   = c.value.masked_imeisv();
        break;
      case 196:
        expected_ue_behaviour_present = true;
        expected_ue_behaviour.id      = c.id;
        expected_ue_behaviour.crit    = c.crit;
        expected_ue_behaviour.value   = c.value.expected_ue_behaviour();
        break;
      case 195:
        pro_se_authorized_present = true;
        pro_se_authorized.id      = c.id;
        pro_se_authorized.crit    = c.crit;
        pro_se_authorized.value   = c.value.pro_se_authorized();
        break;
      case 241:
        ueuser_plane_cio_tsupport_ind_present = true;
        ueuser_plane_cio_tsupport_ind.id      = c.id;
        ueuser_plane_cio_tsupport_ind.crit    = c.crit;
        ueuser_plane_cio_tsupport_ind.value   = c.value.ueuser_plane_cio_tsupport_ind();
        break;
      case 240:
        v2xservices_authorized_present = true;
        v2xservices_authorized.id      = c.id;
        v2xservices_authorized.crit    = c.crit;
        v2xservices_authorized.value   = c.value.v2xservices_authorized();
        break;
      case 248:
        ue_sidelink_aggregate_maximum_bitrate_present = true;
        ue_sidelink_aggregate_maximum_bitrate.id      = c.id;
        ue_sidelink_aggregate_maximum_bitrate.crit    = c.crit;
        ue_sidelink_aggregate_maximum_bitrate.value   = c.value.ue_sidelink_aggregate_maximum_bitrate();
        break;
      case 251:
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = c.id;
        enhanced_coverage_restricted.crit    = c.crit;
        enhanced_coverage_restricted.value   = c.value.enhanced_coverage_restricted();
        break;
      case 271:
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = c.id;
        ce_mode_brestricted.crit    = c.crit;
        ce_mode_brestricted.value   = c.value.ce_mode_brestricted();
        break;
      case 283:
        pending_data_ind_present = true;
        pending_data_ind.id      = c.id;
        pending_data_ind.crit    = c.crit;
        pending_data_ind.value   = c.value.pending_data_ind();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.write_fieldname("");
  ueaggregate_maximum_bitrate.to_json(j);
  j.write_fieldname("");
  erab_to_be_setup_list_ho_req.to_json(j);
  j.write_fieldname("");
  source_to_target_transparent_container.to_json(j);
  j.write_fieldname("");
  ue_security_cap.to_json(j);
  if (ho_restrict_list_present) {
    j.write_fieldname("");
    ho_restrict_list.to_json(j);
  }
  if (trace_activation_present) {
    j.write_fieldname("");
    trace_activation.to_json(j);
  }
  if (request_type_present) {
    j.write_fieldname("");
    request_type.to_json(j);
  }
  if (srvcc_operation_possible_present) {
    j.write_fieldname("");
    srvcc_operation_possible.to_json(j);
  }
  j.write_fieldname("");
  security_context.to_json(j);
  if (nas_security_paramsto_e_utran_present) {
    j.write_fieldname("");
    nas_security_paramsto_e_utran.to_json(j);
  }
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (csg_membership_status_present) {
    j.write_fieldname("");
    csg_membership_status.to_json(j);
  }
  if (gummei_id_present) {
    j.write_fieldname("");
    gummei_id.to_json(j);
  }
  if (mme_ue_s1ap_id_minus2_present) {
    j.write_fieldname("");
    mme_ue_s1ap_id_minus2.to_json(j);
  }
  if (management_based_mdt_allowed_present) {
    j.write_fieldname("");
    management_based_mdt_allowed.to_json(j);
  }
  if (management_based_mdtplmn_list_present) {
    j.write_fieldname("");
    management_based_mdtplmn_list.to_json(j);
  }
  if (masked_imeisv_present) {
    j.write_fieldname("");
    masked_imeisv.to_json(j);
  }
  if (expected_ue_behaviour_present) {
    j.write_fieldname("");
    expected_ue_behaviour.to_json(j);
  }
  if (pro_se_authorized_present) {
    j.write_fieldname("");
    pro_se_authorized.to_json(j);
  }
  if (ueuser_plane_cio_tsupport_ind_present) {
    j.write_fieldname("");
    ueuser_plane_cio_tsupport_ind.to_json(j);
  }
  if (v2xservices_authorized_present) {
    j.write_fieldname("");
    v2xservices_authorized.to_json(j);
  }
  if (ue_sidelink_aggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ue_sidelink_aggregate_maximum_bitrate.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  if (pending_data_ind_present) {
    j.write_fieldname("");
    pending_data_ind.to_json(j);
  }
  j.end_obj();
}

// HandoverRequest ::= SEQUENCE
SRSASN_CODE ho_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// CE-mode-B-SupportIndicator ::= ENUMERATED
std::string ce_mode_b_support_ind_opts::to_string() const
{
  static const char* options[] = {"supported"};
  return convert_enum_idx(options, 1, value, "ce_mode_b_support_ind_e");
}

// HandoverRequestAcknowledgeIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_request_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 18, 19, 123, 127, 58, 145, 242};
  return map_enum_number(options, 9, idx, "id");
}
bool ho_request_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 18, 19, 123, 127, 58, 145, 242};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_request_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 18:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    case 123:
      return crit_e::reject;
    case 127:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 145:
      return crit_e::ignore;
    case 242:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_request_ack_ies_o::value_c ho_request_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 18:
      ret.set(value_c::types::erab_admitted_list);
      break;
    case 19:
      ret.set(value_c::types::erab_failed_to_setup_list_ho_req_ack);
      break;
    case 123:
      ret.set(value_c::types::target_to_source_transparent_container);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 145:
      ret.set(value_c::types::cell_access_mode);
      break;
    case 242:
      ret.set(value_c::types::ce_mode_b_support_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 18:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    case 123:
      return presence_e::mandatory;
    case 127:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 145:
      return presence_e::optional;
    case 242:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_request_ack_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_request_ack_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_ie_container_list_l<erab_admitted_item_ies_o>& ho_request_ack_ies_o::value_c::erab_admitted_list()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABAdmittedItemIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
}
erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o>&
ho_request_ack_ies_o::value_c::erab_failed_to_setup_list_ho_req_ack()
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABFailedtoSetupItemHOReqAckIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
}
unbounded_octstring<true>& ho_request_ack_ies_o::value_c::target_to_source_transparent_container()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<27, false, true>& ho_request_ack_ies_o::value_c::csg_id()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
crit_diagnostics_s& ho_request_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
cell_access_mode_e& ho_request_ack_ies_o::value_c::cell_access_mode()
{
  assert_choice_type("CellAccessMode", type_.to_string(), "Value");
  return c.get<cell_access_mode_e>();
}
ce_mode_b_support_ind_e& ho_request_ack_ies_o::value_c::ce_mode_b_support_ind()
{
  assert_choice_type("CE-mode-B-SupportIndicator", type_.to_string(), "Value");
  return c.get<ce_mode_b_support_ind_e>();
}
const uint64_t& ho_request_ack_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_request_ack_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_ie_container_list_l<erab_admitted_item_ies_o>& ho_request_ack_ies_o::value_c::erab_admitted_list() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABAdmittedItemIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
}
const erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o>&
ho_request_ack_ies_o::value_c::erab_failed_to_setup_list_ho_req_ack() const
{
  assert_choice_type("E-RAB-IE-ContainerList{{E-RABFailedtoSetupItemHOReqAckIEs}}", type_.to_string(), "Value");
  return c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
}
const unbounded_octstring<true>& ho_request_ack_ies_o::value_c::target_to_source_transparent_container() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<27, false, true>& ho_request_ack_ies_o::value_c::csg_id() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const crit_diagnostics_s& ho_request_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const cell_access_mode_e& ho_request_ack_ies_o::value_c::cell_access_mode() const
{
  assert_choice_type("CellAccessMode", type_.to_string(), "Value");
  return c.get<cell_access_mode_e>();
}
const ce_mode_b_support_ind_e& ho_request_ack_ies_o::value_c::ce_mode_b_support_ind() const
{
  assert_choice_type("CE-mode-B-SupportIndicator", type_.to_string(), "Value");
  return c.get<ce_mode_b_support_ind_e>();
}
void ho_request_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_admitted_list:
      c.destroy<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.destroy<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
      break;
    case types::target_to_source_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_request_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_admitted_list:
      c.init<erab_ie_container_list_l<erab_admitted_item_ies_o> >();
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.init<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >();
      break;
    case types::target_to_source_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::cell_access_mode:
      break;
    case types::ce_mode_b_support_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
}
ho_request_ack_ies_o::value_c::value_c(const ho_request_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_admitted_list:
      c.init(other.c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >());
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.init(other.c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >());
      break;
    case types::target_to_source_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::cell_access_mode:
      c.init(other.c.get<cell_access_mode_e>());
      break;
    case types::ce_mode_b_support_ind:
      c.init(other.c.get<ce_mode_b_support_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
}
ho_request_ack_ies_o::value_c& ho_request_ack_ies_o::value_c::operator=(const ho_request_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_admitted_list:
      c.set(other.c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >());
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      c.set(other.c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >());
      break;
    case types::target_to_source_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::cell_access_mode:
      c.set(other.c.get<cell_access_mode_e>());
      break;
    case types::ce_mode_b_support_ind:
      c.set(other.c.get<ce_mode_b_support_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }

  return *this;
}
void ho_request_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_admitted_list:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABAdmittedItemIEs}}");
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      j.write_fieldname("E-RAB-IE-ContainerList{{E-RABFailedtoSetupItemHOReqAckIEs}}");
      break;
    case types::target_to_source_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::cell_access_mode:
      j.write_str("CellAccessMode", "hybrid");
      break;
    case types::ce_mode_b_support_ind:
      j.write_str("CE-mode-B-SupportIndicator", "supported");
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_request_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_admitted_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >(), 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      HANDLE_CODE(pack_dyn_seq_of(
          bref, c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >(), 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().pack(bref));
      break;
    case types::ce_mode_b_support_ind:
      HANDLE_CODE(c.get<ce_mode_b_support_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_admitted_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_ie_container_list_l<erab_admitted_item_ies_o> >(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_ho_req_ack:
      HANDLE_CODE(unpack_dyn_seq_of(
          c.get<erab_ie_container_list_l<erab_failedto_setup_item_ho_req_ack_ies_o> >(), bref, 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().unpack(bref));
      break;
    case types::ce_mode_b_support_ind:
      HANDLE_CODE(c.get<ce_mode_b_support_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_request_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RAB-IE-ContainerList{{E-RABAdmittedItemIEs}}",
                                  "E-RAB-IE-ContainerList{{E-RABFailedtoSetupItemHOReqAckIEs}}",
                                  "OCTET STRING",
                                  "BIT STRING",
                                  "CriticalityDiagnostics",
                                  "CellAccessMode",
                                  "CE-mode-B-SupportIndicator"};
  return convert_enum_idx(options, 9, value, "ho_request_ack_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_request_ack_ies_o>;

ho_request_ack_ies_container::ho_request_ack_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_admitted_list(18, crit_e::ignore),
  erab_failed_to_setup_list_ho_req_ack(19, crit_e::ignore),
  target_to_source_transparent_container(123, crit_e::reject),
  csg_id(127, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore),
  cell_access_mode(145, crit_e::ignore),
  ce_mode_b_support_ind(242, crit_e::ignore)
{}
SRSASN_CODE ho_request_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += erab_failed_to_setup_list_ho_req_ack_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  nof_ies += cell_access_mode_present ? 1 : 0;
  nof_ies += ce_mode_b_support_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_admitted_list.pack(bref));
  if (erab_failed_to_setup_list_ho_req_ack_present) {
    HANDLE_CODE(erab_failed_to_setup_list_ho_req_ack.pack(bref));
  }
  HANDLE_CODE(target_to_source_transparent_container.pack(bref));
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (ce_mode_b_support_ind_present) {
    HANDLE_CODE(ce_mode_b_support_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_request_ack_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 18:
        nof_mandatory_ies--;
        erab_admitted_list.id    = c.id;
        erab_admitted_list.crit  = c.crit;
        erab_admitted_list.value = c.value.erab_admitted_list();
        break;
      case 19:
        erab_failed_to_setup_list_ho_req_ack_present = true;
        erab_failed_to_setup_list_ho_req_ack.id      = c.id;
        erab_failed_to_setup_list_ho_req_ack.crit    = c.crit;
        erab_failed_to_setup_list_ho_req_ack.value   = c.value.erab_failed_to_setup_list_ho_req_ack();
        break;
      case 123:
        nof_mandatory_ies--;
        target_to_source_transparent_container.id    = c.id;
        target_to_source_transparent_container.crit  = c.crit;
        target_to_source_transparent_container.value = c.value.target_to_source_transparent_container();
        break;
      case 127:
        csg_id_present = true;
        csg_id.id      = c.id;
        csg_id.crit    = c.crit;
        csg_id.value   = c.value.csg_id();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      case 145:
        cell_access_mode_present = true;
        cell_access_mode.id      = c.id;
        cell_access_mode.crit    = c.crit;
        cell_access_mode.value   = c.value.cell_access_mode();
        break;
      case 242:
        ce_mode_b_support_ind_present = true;
        ce_mode_b_support_ind.id      = c.id;
        ce_mode_b_support_ind.crit    = c.crit;
        ce_mode_b_support_ind.value   = c.value.ce_mode_b_support_ind();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_request_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_admitted_list.to_json(j);
  if (erab_failed_to_setup_list_ho_req_ack_present) {
    j.write_fieldname("");
    erab_failed_to_setup_list_ho_req_ack.to_json(j);
  }
  j.write_fieldname("");
  target_to_source_transparent_container.to_json(j);
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  if (cell_access_mode_present) {
    j.write_fieldname("");
    cell_access_mode.to_json(j);
  }
  if (ce_mode_b_support_ind_present) {
    j.write_fieldname("");
    ce_mode_b_support_ind.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestAcknowledge ::= SEQUENCE
SRSASN_CODE ho_request_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_request_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// Direct-Forwarding-Path-Availability ::= ENUMERATED
std::string direct_forwarding_path_availability_opts::to_string() const
{
  static const char* options[] = {"directPathAvailable"};
  return convert_enum_idx(options, 1, value, "direct_forwarding_path_availability_e");
}

// PS-ServiceNotAvailable ::= ENUMERATED
std::string ps_service_not_available_opts::to_string() const
{
  static const char* options[] = {"ps-service-not-available"};
  return convert_enum_idx(options, 1, value, "ps_service_not_available_e");
}

// SRVCCHOIndication ::= ENUMERATED
std::string srvccho_ind_opts::to_string() const
{
  static const char* options[] = {"pSandCS", "cSonly"};
  return convert_enum_idx(options, 2, value, "srvccho_ind_e");
}

// TargetID ::= CHOICE
void target_id_c::destroy_()
{
  switch (type_) {
    case types::targetenb_id:
      c.destroy<targetenb_id_s>();
      break;
    case types::target_rnc_id:
      c.destroy<target_rnc_id_s>();
      break;
    case types::cgi:
      c.destroy<cgi_s>();
      break;
    default:
      break;
  }
}
void target_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::targetenb_id:
      c.init<targetenb_id_s>();
      break;
    case types::target_rnc_id:
      c.init<target_rnc_id_s>();
      break;
    case types::cgi:
      c.init<cgi_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
}
target_id_c::target_id_c(const target_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::targetenb_id:
      c.init(other.c.get<targetenb_id_s>());
      break;
    case types::target_rnc_id:
      c.init(other.c.get<target_rnc_id_s>());
      break;
    case types::cgi:
      c.init(other.c.get<cgi_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
}
target_id_c& target_id_c::operator=(const target_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::targetenb_id:
      c.set(other.c.get<targetenb_id_s>());
      break;
    case types::target_rnc_id:
      c.set(other.c.get<target_rnc_id_s>());
      break;
    case types::cgi:
      c.set(other.c.get<cgi_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }

  return *this;
}
void target_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::targetenb_id:
      j.write_fieldname("targeteNB-ID");
      c.get<targetenb_id_s>().to_json(j);
      break;
    case types::target_rnc_id:
      j.write_fieldname("targetRNC-ID");
      c.get<target_rnc_id_s>().to_json(j);
      break;
    case types::cgi:
      j.write_fieldname("cGI");
      c.get<cgi_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
  j.end_obj();
}
SRSASN_CODE target_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::targetenb_id:
      HANDLE_CODE(c.get<targetenb_id_s>().pack(bref));
      break;
    case types::target_rnc_id:
      HANDLE_CODE(c.get<target_rnc_id_s>().pack(bref));
      break;
    case types::cgi:
      HANDLE_CODE(c.get<cgi_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE target_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::targetenb_id:
      HANDLE_CODE(c.get<targetenb_id_s>().unpack(bref));
      break;
    case types::target_rnc_id:
      HANDLE_CODE(c.get<target_rnc_id_s>().unpack(bref));
      break;
    case types::cgi:
      HANDLE_CODE(c.get<cgi_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string target_id_c::types_opts::to_string() const
{
  static const char* options[] = {"targeteNB-ID", "targetRNC-ID", "cGI"};
  return convert_enum_idx(options, 3, value, "target_id_c::types");
}

// HandoverRequiredIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ho_required_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 1, 2, 4, 79, 125, 104, 138, 132, 133, 127, 145, 150};
  return map_enum_number(options, 14, idx, "id");
}
bool ho_required_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 1, 2, 4, 79, 125, 104, 138, 132, 133, 127, 145, 150};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_required_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 1:
      return crit_e::reject;
    case 2:
      return crit_e::ignore;
    case 4:
      return crit_e::reject;
    case 79:
      return crit_e::ignore;
    case 125:
      return crit_e::reject;
    case 104:
      return crit_e::reject;
    case 138:
      return crit_e::reject;
    case 132:
      return crit_e::reject;
    case 133:
      return crit_e::ignore;
    case 127:
      return crit_e::reject;
    case 145:
      return crit_e::reject;
    case 150:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_required_ies_o::value_c ho_required_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 1:
      ret.set(value_c::types::handov_type);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 4:
      ret.set(value_c::types::target_id);
      break;
    case 79:
      ret.set(value_c::types::direct_forwarding_path_availability);
      break;
    case 125:
      ret.set(value_c::types::srvccho_ind);
      break;
    case 104:
      ret.set(value_c::types::source_to_target_transparent_container);
      break;
    case 138:
      ret.set(value_c::types::source_to_target_transparent_container_secondary);
      break;
    case 132:
      ret.set(value_c::types::ms_classmark2);
      break;
    case 133:
      ret.set(value_c::types::ms_classmark3);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 145:
      ret.set(value_c::types::cell_access_mode);
      break;
    case 150:
      ret.set(value_c::types::ps_service_not_available);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_required_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 1:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 4:
      return presence_e::mandatory;
    case 79:
      return presence_e::optional;
    case 125:
      return presence_e::optional;
    case 104:
      return presence_e::mandatory;
    case 138:
      return presence_e::optional;
    case 132:
      return presence_e::conditional;
    case 133:
      return presence_e::conditional;
    case 127:
      return presence_e::optional;
    case 145:
      return presence_e::optional;
    case 150:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& ho_required_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& ho_required_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
handov_type_e& ho_required_ies_o::value_c::handov_type()
{
  assert_choice_type("HandoverType", type_.to_string(), "Value");
  return c.get<handov_type_e>();
}
cause_c& ho_required_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
target_id_c& ho_required_ies_o::value_c::target_id()
{
  assert_choice_type("TargetID", type_.to_string(), "Value");
  return c.get<target_id_c>();
}
direct_forwarding_path_availability_e& ho_required_ies_o::value_c::direct_forwarding_path_availability()
{
  assert_choice_type("Direct-Forwarding-Path-Availability", type_.to_string(), "Value");
  return c.get<direct_forwarding_path_availability_e>();
}
srvccho_ind_e& ho_required_ies_o::value_c::srvccho_ind()
{
  assert_choice_type("SRVCCHOIndication", type_.to_string(), "Value");
  return c.get<srvccho_ind_e>();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container_secondary()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark2()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark3()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<27, false, true>& ho_required_ies_o::value_c::csg_id()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
cell_access_mode_e& ho_required_ies_o::value_c::cell_access_mode()
{
  assert_choice_type("CellAccessMode", type_.to_string(), "Value");
  return c.get<cell_access_mode_e>();
}
ps_service_not_available_e& ho_required_ies_o::value_c::ps_service_not_available()
{
  assert_choice_type("PS-ServiceNotAvailable", type_.to_string(), "Value");
  return c.get<ps_service_not_available_e>();
}
const uint64_t& ho_required_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& ho_required_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const handov_type_e& ho_required_ies_o::value_c::handov_type() const
{
  assert_choice_type("HandoverType", type_.to_string(), "Value");
  return c.get<handov_type_e>();
}
const cause_c& ho_required_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const target_id_c& ho_required_ies_o::value_c::target_id() const
{
  assert_choice_type("TargetID", type_.to_string(), "Value");
  return c.get<target_id_c>();
}
const direct_forwarding_path_availability_e& ho_required_ies_o::value_c::direct_forwarding_path_availability() const
{
  assert_choice_type("Direct-Forwarding-Path-Availability", type_.to_string(), "Value");
  return c.get<direct_forwarding_path_availability_e>();
}
const srvccho_ind_e& ho_required_ies_o::value_c::srvccho_ind() const
{
  assert_choice_type("SRVCCHOIndication", type_.to_string(), "Value");
  return c.get<srvccho_ind_e>();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container_secondary() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark2() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::ms_classmark3() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<27, false, true>& ho_required_ies_o::value_c::csg_id() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const cell_access_mode_e& ho_required_ies_o::value_c::cell_access_mode() const
{
  assert_choice_type("CellAccessMode", type_.to_string(), "Value");
  return c.get<cell_access_mode_e>();
}
const ps_service_not_available_e& ho_required_ies_o::value_c::ps_service_not_available() const
{
  assert_choice_type("PS-ServiceNotAvailable", type_.to_string(), "Value");
  return c.get<ps_service_not_available_e>();
}
void ho_required_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::target_id:
      c.destroy<target_id_c>();
      break;
    case types::source_to_target_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::source_to_target_transparent_container_secondary:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ms_classmark2:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::ms_classmark3:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    default:
      break;
  }
}
void ho_required_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::handov_type:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::target_id:
      c.init<target_id_c>();
      break;
    case types::direct_forwarding_path_availability:
      break;
    case types::srvccho_ind:
      break;
    case types::source_to_target_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::source_to_target_transparent_container_secondary:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ms_classmark2:
      c.init<unbounded_octstring<true> >();
      break;
    case types::ms_classmark3:
      c.init<unbounded_octstring<true> >();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::cell_access_mode:
      break;
    case types::ps_service_not_available:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
}
ho_required_ies_o::value_c::value_c(const ho_required_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::target_id:
      c.init(other.c.get<target_id_c>());
      break;
    case types::direct_forwarding_path_availability:
      c.init(other.c.get<direct_forwarding_path_availability_e>());
      break;
    case types::srvccho_ind:
      c.init(other.c.get<srvccho_ind_e>());
      break;
    case types::source_to_target_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::source_to_target_transparent_container_secondary:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark2:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark3:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::cell_access_mode:
      c.init(other.c.get<cell_access_mode_e>());
      break;
    case types::ps_service_not_available:
      c.init(other.c.get<ps_service_not_available_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
}
ho_required_ies_o::value_c& ho_required_ies_o::value_c::operator=(const ho_required_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::target_id:
      c.set(other.c.get<target_id_c>());
      break;
    case types::direct_forwarding_path_availability:
      c.set(other.c.get<direct_forwarding_path_availability_e>());
      break;
    case types::srvccho_ind:
      c.set(other.c.get<srvccho_ind_e>());
      break;
    case types::source_to_target_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::source_to_target_transparent_container_secondary:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark2:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::ms_classmark3:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::cell_access_mode:
      c.set(other.c.get<cell_access_mode_e>());
      break;
    case types::ps_service_not_available:
      c.set(other.c.get<ps_service_not_available_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }

  return *this;
}
void ho_required_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::target_id:
      j.write_fieldname("TargetID");
      c.get<target_id_c>().to_json(j);
      break;
    case types::direct_forwarding_path_availability:
      j.write_str("Direct-Forwarding-Path-Availability", "directPathAvailable");
      break;
    case types::srvccho_ind:
      j.write_str("SRVCCHOIndication", c.get<srvccho_ind_e>().to_string());
      break;
    case types::source_to_target_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::source_to_target_transparent_container_secondary:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ms_classmark2:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::ms_classmark3:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::cell_access_mode:
      j.write_str("CellAccessMode", "hybrid");
      break;
    case types::ps_service_not_available:
      j.write_str("PS-ServiceNotAvailable", "ps-service-not-available");
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_required_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::target_id:
      HANDLE_CODE(c.get<target_id_c>().pack(bref));
      break;
    case types::direct_forwarding_path_availability:
      HANDLE_CODE(c.get<direct_forwarding_path_availability_e>().pack(bref));
      break;
    case types::srvccho_ind:
      HANDLE_CODE(c.get<srvccho_ind_e>().pack(bref));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::source_to_target_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ms_classmark2:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::ms_classmark3:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().pack(bref));
      break;
    case types::ps_service_not_available:
      HANDLE_CODE(c.get<ps_service_not_available_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::target_id:
      HANDLE_CODE(c.get<target_id_c>().unpack(bref));
      break;
    case types::direct_forwarding_path_availability:
      HANDLE_CODE(c.get<direct_forwarding_path_availability_e>().unpack(bref));
      break;
    case types::srvccho_ind:
      HANDLE_CODE(c.get<srvccho_ind_e>().unpack(bref));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::source_to_target_transparent_container_secondary:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ms_classmark2:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::ms_classmark3:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().unpack(bref));
      break;
    case types::ps_service_not_available:
      HANDLE_CODE(c.get<ps_service_not_available_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ho_required_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "HandoverType",
                                  "Cause",
                                  "TargetID",
                                  "Direct-Forwarding-Path-Availability",
                                  "SRVCCHOIndication",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "BIT STRING",
                                  "CellAccessMode",
                                  "PS-ServiceNotAvailable"};
  return convert_enum_idx(options, 14, value, "ho_required_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<ho_required_ies_o>;

ho_required_ies_container::ho_required_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  handov_type(1, crit_e::reject),
  cause(2, crit_e::ignore),
  target_id(4, crit_e::reject),
  direct_forwarding_path_availability(79, crit_e::ignore),
  srvccho_ind(125, crit_e::reject),
  source_to_target_transparent_container(104, crit_e::reject),
  source_to_target_transparent_container_secondary(138, crit_e::reject),
  ms_classmark2(132, crit_e::reject),
  ms_classmark3(133, crit_e::ignore),
  csg_id(127, crit_e::reject),
  cell_access_mode(145, crit_e::reject),
  ps_service_not_available(150, crit_e::ignore)
{}
SRSASN_CODE ho_required_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 6;
  nof_ies += direct_forwarding_path_availability_present ? 1 : 0;
  nof_ies += srvccho_ind_present ? 1 : 0;
  nof_ies += source_to_target_transparent_container_secondary_present ? 1 : 0;
  nof_ies += ms_classmark2_present ? 1 : 0;
  nof_ies += ms_classmark3_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += cell_access_mode_present ? 1 : 0;
  nof_ies += ps_service_not_available_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  HANDLE_CODE(target_id.pack(bref));
  if (direct_forwarding_path_availability_present) {
    HANDLE_CODE(direct_forwarding_path_availability.pack(bref));
  }
  if (srvccho_ind_present) {
    HANDLE_CODE(srvccho_ind.pack(bref));
  }
  HANDLE_CODE(source_to_target_transparent_container.pack(bref));
  if (source_to_target_transparent_container_secondary_present) {
    HANDLE_CODE(source_to_target_transparent_container_secondary.pack(bref));
  }
  if (ms_classmark2_present) {
    HANDLE_CODE(ms_classmark2.pack(bref));
  }
  if (ms_classmark3_present) {
    HANDLE_CODE(ms_classmark3.pack(bref));
  }
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (ps_service_not_available_present) {
    HANDLE_CODE(ps_service_not_available.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 6;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<ho_required_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 1:
        nof_mandatory_ies--;
        handov_type.id    = c.id;
        handov_type.crit  = c.crit;
        handov_type.value = c.value.handov_type();
        break;
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      case 4:
        nof_mandatory_ies--;
        target_id.id    = c.id;
        target_id.crit  = c.crit;
        target_id.value = c.value.target_id();
        break;
      case 79:
        direct_forwarding_path_availability_present = true;
        direct_forwarding_path_availability.id      = c.id;
        direct_forwarding_path_availability.crit    = c.crit;
        direct_forwarding_path_availability.value   = c.value.direct_forwarding_path_availability();
        break;
      case 125:
        srvccho_ind_present = true;
        srvccho_ind.id      = c.id;
        srvccho_ind.crit    = c.crit;
        srvccho_ind.value   = c.value.srvccho_ind();
        break;
      case 104:
        nof_mandatory_ies--;
        source_to_target_transparent_container.id    = c.id;
        source_to_target_transparent_container.crit  = c.crit;
        source_to_target_transparent_container.value = c.value.source_to_target_transparent_container();
        break;
      case 138:
        source_to_target_transparent_container_secondary_present = true;
        source_to_target_transparent_container_secondary.id      = c.id;
        source_to_target_transparent_container_secondary.crit    = c.crit;
        source_to_target_transparent_container_secondary.value =
            c.value.source_to_target_transparent_container_secondary();
        break;
      case 132:
        ms_classmark2_present = true;
        ms_classmark2.id      = c.id;
        ms_classmark2.crit    = c.crit;
        ms_classmark2.value   = c.value.ms_classmark2();
        break;
      case 133:
        ms_classmark3_present = true;
        ms_classmark3.id      = c.id;
        ms_classmark3.crit    = c.crit;
        ms_classmark3.value   = c.value.ms_classmark3();
        break;
      case 127:
        csg_id_present = true;
        csg_id.id      = c.id;
        csg_id.crit    = c.crit;
        csg_id.value   = c.value.csg_id();
        break;
      case 145:
        cell_access_mode_present = true;
        cell_access_mode.id      = c.id;
        cell_access_mode.crit    = c.crit;
        cell_access_mode.value   = c.value.cell_access_mode();
        break;
      case 150:
        ps_service_not_available_present = true;
        ps_service_not_available.id      = c.id;
        ps_service_not_available.crit    = c.crit;
        ps_service_not_available.value   = c.value.ps_service_not_available();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_required_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.write_fieldname("");
  target_id.to_json(j);
  if (direct_forwarding_path_availability_present) {
    j.write_fieldname("");
    direct_forwarding_path_availability.to_json(j);
  }
  if (srvccho_ind_present) {
    j.write_fieldname("");
    srvccho_ind.to_json(j);
  }
  j.write_fieldname("");
  source_to_target_transparent_container.to_json(j);
  if (source_to_target_transparent_container_secondary_present) {
    j.write_fieldname("");
    source_to_target_transparent_container_secondary.to_json(j);
  }
  if (ms_classmark2_present) {
    j.write_fieldname("");
    ms_classmark2.to_json(j);
  }
  if (ms_classmark3_present) {
    j.write_fieldname("");
    ms_classmark3.to_json(j);
  }
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (cell_access_mode_present) {
    j.write_fieldname("");
    cell_access_mode.to_json(j);
  }
  if (ps_service_not_available_present) {
    j.write_fieldname("");
    ps_service_not_available.to_json(j);
  }
  j.end_obj();
}

// HandoverRequired ::= SEQUENCE
SRSASN_CODE ho_required_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void ho_required_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// MMEPagingTarget ::= CHOICE
void mme_paging_target_c::destroy_()
{
  switch (type_) {
    case types::global_enb_id:
      c.destroy<global_enb_id_s>();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    default:
      break;
  }
}
void mme_paging_target_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::global_enb_id:
      c.init<global_enb_id_s>();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mme_paging_target_c");
  }
}
mme_paging_target_c::mme_paging_target_c(const mme_paging_target_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::global_enb_id:
      c.init(other.c.get<global_enb_id_s>());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mme_paging_target_c");
  }
}
mme_paging_target_c& mme_paging_target_c::operator=(const mme_paging_target_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::global_enb_id:
      c.set(other.c.get<global_enb_id_s>());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "mme_paging_target_c");
  }

  return *this;
}
void mme_paging_target_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::global_enb_id:
      j.write_fieldname("global-ENB-ID");
      c.get<global_enb_id_s>().to_json(j);
      break;
    case types::tai:
      j.write_fieldname("tAI");
      c.get<tai_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "mme_paging_target_c");
  }
  j.end_obj();
}
SRSASN_CODE mme_paging_target_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::global_enb_id:
      HANDLE_CODE(c.get<global_enb_id_s>().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "mme_paging_target_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE mme_paging_target_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::global_enb_id:
      HANDLE_CODE(c.get<global_enb_id_s>().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "mme_paging_target_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string mme_paging_target_c::types_opts::to_string() const
{
  static const char* options[] = {"global-ENB-ID", "tAI"};
  return convert_enum_idx(options, 2, value, "mme_paging_target_c::types");
}

// RecommendedENBItem ::= SEQUENCE
SRSASN_CODE recommended_enb_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(mme_paging_target.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_enb_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(mme_paging_target.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_enb_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("mMEPagingTarget");
  mme_paging_target.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedENBItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t recommended_enb_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {215};
  return map_enum_number(options, 1, idx, "id");
}
bool recommended_enb_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 215 == id;
}
crit_e recommended_enb_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 215) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
recommended_enb_item_ies_o::value_c recommended_enb_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 215) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e recommended_enb_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 215) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void recommended_enb_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("RecommendedENBItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE recommended_enb_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_enb_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string recommended_enb_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"RecommendedENBItem"};
  return convert_enum_idx(options, 1, value, "recommended_enb_item_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<recommended_enb_item_ies_o>;

// RecommendedENBsForPaging ::= SEQUENCE
SRSASN_CODE recommended_enbs_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, recommended_enb_list, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_enbs_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(recommended_enb_list, bref, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_enbs_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("recommendedENBList");
  for (const auto& e1 : recommended_enb_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// InformationOnRecommendedCellsAndENBsForPaging ::= SEQUENCE
SRSASN_CODE info_on_recommended_cells_and_enbs_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.pack(bref));
  HANDLE_CODE(recommend_enbs_for_paging.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE info_on_recommended_cells_and_enbs_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.unpack(bref));
  HANDLE_CODE(recommend_enbs_for_paging.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void info_on_recommended_cells_and_enbs_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("recommendedCellsForPaging");
  recommended_cells_for_paging.to_json(j);
  j.write_fieldname("recommendENBsForPaging");
  recommend_enbs_for_paging.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupFailureIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t init_context_setup_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  return map_enum_number(options, 4, idx, "id");
}
bool init_context_setup_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 2, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_context_setup_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 2:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_context_setup_fail_ies_o::value_c init_context_setup_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 2:
      ret.set(value_c::types::cause);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_context_setup_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 2:
      return presence_e::mandatory;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& init_context_setup_fail_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& init_context_setup_fail_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
cause_c& init_context_setup_fail_ies_o::value_c::cause()
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& init_context_setup_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& init_context_setup_fail_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& init_context_setup_fail_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const cause_c& init_context_setup_fail_ies_o::value_c::cause() const
{
  assert_choice_type("Cause", type_.to_string(), "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& init_context_setup_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void init_context_setup_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void init_context_setup_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }
}
init_context_setup_fail_ies_o::value_c::value_c(const init_context_setup_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }
}
init_context_setup_fail_ies_o::value_c& init_context_setup_fail_ies_o::value_c::
                                        operator=(const init_context_setup_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }

  return *this;
}
void init_context_setup_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_context_setup_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string init_context_setup_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..16777215)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "init_context_setup_fail_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<init_context_setup_fail_ies_o>;

init_context_setup_fail_ies_container::init_context_setup_fail_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  cause(2, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE init_context_setup_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<init_context_setup_fail_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 2:
        nof_mandatory_ies--;
        cause.id    = c.id;
        cause.crit  = c.crit;
        cause.value = c.value.cause();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_context_setup_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupFailure ::= SEQUENCE
SRSASN_CODE init_context_setup_fail_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_fail_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void init_context_setup_fail_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// AdditionalCSFallbackIndicator ::= ENUMERATED
std::string add_cs_fallback_ind_opts::to_string() const
{
  static const char* options[] = {"no-restriction", "restriction"};
  return convert_enum_idx(options, 2, value, "add_cs_fallback_ind_e");
}

// CSFallbackIndicator ::= ENUMERATED
std::string cs_fallback_ind_opts::to_string() const
{
  static const char* options[] = {"cs-fallback-required", "cs-fallback-high-priority"};
  return convert_enum_idx(options, 2, value, "cs_fallback_ind_e");
}

// InitialContextSetupRequestIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t init_context_setup_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0,  8,   66,  24,  107, 73,  25,  41,  74,  106, 108, 124, 146, 159,
                                     75, 158, 165, 177, 187, 192, 196, 195, 241, 240, 248, 251, 271, 283};
  return map_enum_number(options, 28, idx, "id");
}
bool init_context_setup_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0,  8,   66,  24,  107, 73,  25,  41,  74,  106, 108, 124, 146, 159,
                                     75, 158, 165, 177, 187, 192, 196, 195, 241, 240, 248, 251, 271, 283};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_context_setup_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::reject;
    case 8:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 24:
      return crit_e::reject;
    case 107:
      return crit_e::reject;
    case 73:
      return crit_e::reject;
    case 25:
      return crit_e::ignore;
    case 41:
      return crit_e::ignore;
    case 74:
      return crit_e::ignore;
    case 106:
      return crit_e::ignore;
    case 108:
      return crit_e::reject;
    case 124:
      return crit_e::ignore;
    case 146:
      return crit_e::ignore;
    case 159:
      return crit_e::ignore;
    case 75:
      return crit_e::ignore;
    case 158:
      return crit_e::ignore;
    case 165:
      return crit_e::ignore;
    case 177:
      return crit_e::ignore;
    case 187:
      return crit_e::ignore;
    case 192:
      return crit_e::ignore;
    case 196:
      return crit_e::ignore;
    case 195:
      return crit_e::ignore;
    case 241:
      return crit_e::ignore;
    case 240:
      return crit_e::ignore;
    case 248:
      return crit_e::ignore;
    case 251:
      return crit_e::ignore;
    case 271:
      return crit_e::ignore;
    case 283:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_context_setup_request_ies_o::value_c init_context_setup_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 66:
      ret.set(value_c::types::ueaggregate_maximum_bitrate);
      break;
    case 24:
      ret.set(value_c::types::erab_to_be_setup_list_ctxt_su_req);
      break;
    case 107:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 73:
      ret.set(value_c::types::security_key);
      break;
    case 25:
      ret.set(value_c::types::trace_activation);
      break;
    case 41:
      ret.set(value_c::types::ho_restrict_list);
      break;
    case 74:
      ret.set(value_c::types::ue_radio_cap);
      break;
    case 106:
      ret.set(value_c::types::subscriber_profile_idfor_rfp);
      break;
    case 108:
      ret.set(value_c::types::cs_fallback_ind);
      break;
    case 124:
      ret.set(value_c::types::srvcc_operation_possible);
      break;
    case 146:
      ret.set(value_c::types::csg_membership_status);
      break;
    case 159:
      ret.set(value_c::types::registered_lai);
      break;
    case 75:
      ret.set(value_c::types::gummei_id);
      break;
    case 158:
      ret.set(value_c::types::mme_ue_s1ap_id_minus2);
      break;
    case 165:
      ret.set(value_c::types::management_based_mdt_allowed);
      break;
    case 177:
      ret.set(value_c::types::management_based_mdtplmn_list);
      break;
    case 187:
      ret.set(value_c::types::add_cs_fallback_ind);
      break;
    case 192:
      ret.set(value_c::types::masked_imeisv);
      break;
    case 196:
      ret.set(value_c::types::expected_ue_behaviour);
      break;
    case 195:
      ret.set(value_c::types::pro_se_authorized);
      break;
    case 241:
      ret.set(value_c::types::ueuser_plane_cio_tsupport_ind);
      break;
    case 240:
      ret.set(value_c::types::v2xservices_authorized);
      break;
    case 248:
      ret.set(value_c::types::ue_sidelink_aggregate_maximum_bitrate);
      break;
    case 251:
      ret.set(value_c::types::enhanced_coverage_restricted);
      break;
    case 271:
      ret.set(value_c::types::ce_mode_brestricted);
      break;
    case 283:
      ret.set(value_c::types::pending_data_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_context_setup_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 66:
      return presence_e::mandatory;
    case 24:
      return presence_e::mandatory;
    case 107:
      return presence_e::mandatory;
    case 73:
      return presence_e::mandatory;
    case 25:
      return presence_e::optional;
    case 41:
      return presence_e::optional;
    case 74:
      return presence_e::optional;
    case 106:
      return presence_e::optional;
    case 108:
      return presence_e::optional;
    case 124:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    case 159:
      return presence_e::optional;
    case 75:
      return presence_e::optional;
    case 158:
      return presence_e::optional;
    case 165:
      return presence_e::optional;
    case 177:
      return presence_e::optional;
    case 187:
      return presence_e::conditional;
    case 192:
      return presence_e::optional;
    case 196:
      return presence_e::optional;
    case 195:
      return presence_e::optional;
    case 241:
      return presence_e::optional;
    case 240:
      return presence_e::optional;
    case 248:
      return presence_e::optional;
    case 251:
      return presence_e::optional;
    case 271:
      return presence_e::optional;
    case 283:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& init_context_setup_request_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& init_context_setup_request_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
ue_aggregate_maximum_bitrate_s& init_context_setup_request_ies_o::value_c::ueaggregate_maximum_bitrate()
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
erab_to_be_setup_list_ctxt_su_req_l& init_context_setup_request_ies_o::value_c::erab_to_be_setup_list_ctxt_su_req()
{
  assert_choice_type("E-RABToBeSetupListCtxtSUReq", type_.to_string(), "Value");
  return c.get<erab_to_be_setup_list_ctxt_su_req_l>();
}
ue_security_cap_s& init_context_setup_request_ies_o::value_c::ue_security_cap()
{
  assert_choice_type("UESecurityCapabilities", type_.to_string(), "Value");
  return c.get<ue_security_cap_s>();
}
fixed_bitstring<256, false, true>& init_context_setup_request_ies_o::value_c::security_key()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<256, false, true> >();
}
trace_activation_s& init_context_setup_request_ies_o::value_c::trace_activation()
{
  assert_choice_type("TraceActivation", type_.to_string(), "Value");
  return c.get<trace_activation_s>();
}
ho_restrict_list_s& init_context_setup_request_ies_o::value_c::ho_restrict_list()
{
  assert_choice_type("HandoverRestrictionList", type_.to_string(), "Value");
  return c.get<ho_restrict_list_s>();
}
unbounded_octstring<true>& init_context_setup_request_ies_o::value_c::ue_radio_cap()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
uint16_t& init_context_setup_request_ies_o::value_c::subscriber_profile_idfor_rfp()
{
  assert_choice_type("INTEGER (1..256)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
cs_fallback_ind_e& init_context_setup_request_ies_o::value_c::cs_fallback_ind()
{
  assert_choice_type("CSFallbackIndicator", type_.to_string(), "Value");
  return c.get<cs_fallback_ind_e>();
}
srvcc_operation_possible_e& init_context_setup_request_ies_o::value_c::srvcc_operation_possible()
{
  assert_choice_type("SRVCCOperationPossible", type_.to_string(), "Value");
  return c.get<srvcc_operation_possible_e>();
}
csg_membership_status_e& init_context_setup_request_ies_o::value_c::csg_membership_status()
{
  assert_choice_type("CSGMembershipStatus", type_.to_string(), "Value");
  return c.get<csg_membership_status_e>();
}
lai_s& init_context_setup_request_ies_o::value_c::registered_lai()
{
  assert_choice_type("LAI", type_.to_string(), "Value");
  return c.get<lai_s>();
}
gummei_s& init_context_setup_request_ies_o::value_c::gummei_id()
{
  assert_choice_type("GUMMEI", type_.to_string(), "Value");
  return c.get<gummei_s>();
}
uint64_t& init_context_setup_request_ies_o::value_c::mme_ue_s1ap_id_minus2()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
management_based_mdt_allowed_e& init_context_setup_request_ies_o::value_c::management_based_mdt_allowed()
{
  assert_choice_type("ManagementBasedMDTAllowed", type_.to_string(), "Value");
  return c.get<management_based_mdt_allowed_e>();
}
mdtplmn_list_l& init_context_setup_request_ies_o::value_c::management_based_mdtplmn_list()
{
  assert_choice_type("MDTPLMNList", type_.to_string(), "Value");
  return c.get<mdtplmn_list_l>();
}
add_cs_fallback_ind_e& init_context_setup_request_ies_o::value_c::add_cs_fallback_ind()
{
  assert_choice_type("AdditionalCSFallbackIndicator", type_.to_string(), "Value");
  return c.get<add_cs_fallback_ind_e>();
}
fixed_bitstring<64, false, true>& init_context_setup_request_ies_o::value_c::masked_imeisv()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
expected_ue_behaviour_s& init_context_setup_request_ies_o::value_c::expected_ue_behaviour()
{
  assert_choice_type("ExpectedUEBehaviour", type_.to_string(), "Value");
  return c.get<expected_ue_behaviour_s>();
}
pro_se_authorized_s& init_context_setup_request_ies_o::value_c::pro_se_authorized()
{
  assert_choice_type("ProSeAuthorized", type_.to_string(), "Value");
  return c.get<pro_se_authorized_s>();
}
ueuser_plane_cio_tsupport_ind_e& init_context_setup_request_ies_o::value_c::ueuser_plane_cio_tsupport_ind()
{
  assert_choice_type("UEUserPlaneCIoTSupportIndicator", type_.to_string(), "Value");
  return c.get<ueuser_plane_cio_tsupport_ind_e>();
}
v2xservices_authorized_s& init_context_setup_request_ies_o::value_c::v2xservices_authorized()
{
  assert_choice_type("V2XServicesAuthorized", type_.to_string(), "Value");
  return c.get<v2xservices_authorized_s>();
}
ue_sidelink_aggregate_maximum_bitrate_s&
init_context_setup_request_ies_o::value_c::ue_sidelink_aggregate_maximum_bitrate()
{
  assert_choice_type("UESidelinkAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_sidelink_aggregate_maximum_bitrate_s>();
}
enhanced_coverage_restricted_e& init_context_setup_request_ies_o::value_c::enhanced_coverage_restricted()
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
ce_mode_brestricted_e& init_context_setup_request_ies_o::value_c::ce_mode_brestricted()
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
pending_data_ind_e& init_context_setup_request_ies_o::value_c::pending_data_ind()
{
  assert_choice_type("PendingDataIndication", type_.to_string(), "Value");
  return c.get<pending_data_ind_e>();
}
const uint64_t& init_context_setup_request_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& init_context_setup_request_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const ue_aggregate_maximum_bitrate_s& init_context_setup_request_ies_o::value_c::ueaggregate_maximum_bitrate() const
{
  assert_choice_type("UEAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_aggregate_maximum_bitrate_s>();
}
const erab_to_be_setup_list_ctxt_su_req_l&
init_context_setup_request_ies_o::value_c::erab_to_be_setup_list_ctxt_su_req() const
{
  assert_choice_type("E-RABToBeSetupListCtxtSUReq", type_.to_string(), "Value");
  return c.get<erab_to_be_setup_list_ctxt_su_req_l>();
}
const ue_security_cap_s& init_context_setup_request_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type("UESecurityCapabilities", type_.to_string(), "Value");
  return c.get<ue_security_cap_s>();
}
const fixed_bitstring<256, false, true>& init_context_setup_request_ies_o::value_c::security_key() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<256, false, true> >();
}
const trace_activation_s& init_context_setup_request_ies_o::value_c::trace_activation() const
{
  assert_choice_type("TraceActivation", type_.to_string(), "Value");
  return c.get<trace_activation_s>();
}
const ho_restrict_list_s& init_context_setup_request_ies_o::value_c::ho_restrict_list() const
{
  assert_choice_type("HandoverRestrictionList", type_.to_string(), "Value");
  return c.get<ho_restrict_list_s>();
}
const unbounded_octstring<true>& init_context_setup_request_ies_o::value_c::ue_radio_cap() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint16_t& init_context_setup_request_ies_o::value_c::subscriber_profile_idfor_rfp() const
{
  assert_choice_type("INTEGER (1..256)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
const cs_fallback_ind_e& init_context_setup_request_ies_o::value_c::cs_fallback_ind() const
{
  assert_choice_type("CSFallbackIndicator", type_.to_string(), "Value");
  return c.get<cs_fallback_ind_e>();
}
const srvcc_operation_possible_e& init_context_setup_request_ies_o::value_c::srvcc_operation_possible() const
{
  assert_choice_type("SRVCCOperationPossible", type_.to_string(), "Value");
  return c.get<srvcc_operation_possible_e>();
}
const csg_membership_status_e& init_context_setup_request_ies_o::value_c::csg_membership_status() const
{
  assert_choice_type("CSGMembershipStatus", type_.to_string(), "Value");
  return c.get<csg_membership_status_e>();
}
const lai_s& init_context_setup_request_ies_o::value_c::registered_lai() const
{
  assert_choice_type("LAI", type_.to_string(), "Value");
  return c.get<lai_s>();
}
const gummei_s& init_context_setup_request_ies_o::value_c::gummei_id() const
{
  assert_choice_type("GUMMEI", type_.to_string(), "Value");
  return c.get<gummei_s>();
}
const uint64_t& init_context_setup_request_ies_o::value_c::mme_ue_s1ap_id_minus2() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const management_based_mdt_allowed_e& init_context_setup_request_ies_o::value_c::management_based_mdt_allowed() const
{
  assert_choice_type("ManagementBasedMDTAllowed", type_.to_string(), "Value");
  return c.get<management_based_mdt_allowed_e>();
}
const mdtplmn_list_l& init_context_setup_request_ies_o::value_c::management_based_mdtplmn_list() const
{
  assert_choice_type("MDTPLMNList", type_.to_string(), "Value");
  return c.get<mdtplmn_list_l>();
}
const add_cs_fallback_ind_e& init_context_setup_request_ies_o::value_c::add_cs_fallback_ind() const
{
  assert_choice_type("AdditionalCSFallbackIndicator", type_.to_string(), "Value");
  return c.get<add_cs_fallback_ind_e>();
}
const fixed_bitstring<64, false, true>& init_context_setup_request_ies_o::value_c::masked_imeisv() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
const expected_ue_behaviour_s& init_context_setup_request_ies_o::value_c::expected_ue_behaviour() const
{
  assert_choice_type("ExpectedUEBehaviour", type_.to_string(), "Value");
  return c.get<expected_ue_behaviour_s>();
}
const pro_se_authorized_s& init_context_setup_request_ies_o::value_c::pro_se_authorized() const
{
  assert_choice_type("ProSeAuthorized", type_.to_string(), "Value");
  return c.get<pro_se_authorized_s>();
}
const ueuser_plane_cio_tsupport_ind_e& init_context_setup_request_ies_o::value_c::ueuser_plane_cio_tsupport_ind() const
{
  assert_choice_type("UEUserPlaneCIoTSupportIndicator", type_.to_string(), "Value");
  return c.get<ueuser_plane_cio_tsupport_ind_e>();
}
const v2xservices_authorized_s& init_context_setup_request_ies_o::value_c::v2xservices_authorized() const
{
  assert_choice_type("V2XServicesAuthorized", type_.to_string(), "Value");
  return c.get<v2xservices_authorized_s>();
}
const ue_sidelink_aggregate_maximum_bitrate_s&
init_context_setup_request_ies_o::value_c::ue_sidelink_aggregate_maximum_bitrate() const
{
  assert_choice_type("UESidelinkAggregateMaximumBitrate", type_.to_string(), "Value");
  return c.get<ue_sidelink_aggregate_maximum_bitrate_s>();
}
const enhanced_coverage_restricted_e& init_context_setup_request_ies_o::value_c::enhanced_coverage_restricted() const
{
  assert_choice_type("EnhancedCoverageRestricted", type_.to_string(), "Value");
  return c.get<enhanced_coverage_restricted_e>();
}
const ce_mode_brestricted_e& init_context_setup_request_ies_o::value_c::ce_mode_brestricted() const
{
  assert_choice_type("CE-ModeBRestricted", type_.to_string(), "Value");
  return c.get<ce_mode_brestricted_e>();
}
const pending_data_ind_e& init_context_setup_request_ies_o::value_c::pending_data_ind() const
{
  assert_choice_type("PendingDataIndication", type_.to_string(), "Value");
  return c.get<pending_data_ind_e>();
}
void init_context_setup_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ueaggregate_maximum_bitrate:
      c.destroy<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      c.destroy<erab_to_be_setup_list_ctxt_su_req_l>();
      break;
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::security_key:
      c.destroy<fixed_bitstring<256, false, true> >();
      break;
    case types::trace_activation:
      c.destroy<trace_activation_s>();
      break;
    case types::ho_restrict_list:
      c.destroy<ho_restrict_list_s>();
      break;
    case types::ue_radio_cap:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::registered_lai:
      c.destroy<lai_s>();
      break;
    case types::gummei_id:
      c.destroy<gummei_s>();
      break;
    case types::management_based_mdtplmn_list:
      c.destroy<mdtplmn_list_l>();
      break;
    case types::masked_imeisv:
      c.destroy<fixed_bitstring<64, false, true> >();
      break;
    case types::expected_ue_behaviour:
      c.destroy<expected_ue_behaviour_s>();
      break;
    case types::pro_se_authorized:
      c.destroy<pro_se_authorized_s>();
      break;
    case types::v2xservices_authorized:
      c.destroy<v2xservices_authorized_s>();
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.destroy<ue_sidelink_aggregate_maximum_bitrate_s>();
      break;
    default:
      break;
  }
}
void init_context_setup_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init<ue_aggregate_maximum_bitrate_s>();
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      c.init<erab_to_be_setup_list_ctxt_su_req_l>();
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::security_key:
      c.init<fixed_bitstring<256, false, true> >();
      break;
    case types::trace_activation:
      c.init<trace_activation_s>();
      break;
    case types::ho_restrict_list:
      c.init<ho_restrict_list_s>();
      break;
    case types::ue_radio_cap:
      c.init<unbounded_octstring<true> >();
      break;
    case types::subscriber_profile_idfor_rfp:
      break;
    case types::cs_fallback_ind:
      break;
    case types::srvcc_operation_possible:
      break;
    case types::csg_membership_status:
      break;
    case types::registered_lai:
      c.init<lai_s>();
      break;
    case types::gummei_id:
      c.init<gummei_s>();
      break;
    case types::mme_ue_s1ap_id_minus2:
      break;
    case types::management_based_mdt_allowed:
      break;
    case types::management_based_mdtplmn_list:
      c.init<mdtplmn_list_l>();
      break;
    case types::add_cs_fallback_ind:
      break;
    case types::masked_imeisv:
      c.init<fixed_bitstring<64, false, true> >();
      break;
    case types::expected_ue_behaviour:
      c.init<expected_ue_behaviour_s>();
      break;
    case types::pro_se_authorized:
      c.init<pro_se_authorized_s>();
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      break;
    case types::v2xservices_authorized:
      c.init<v2xservices_authorized_s>();
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.init<ue_sidelink_aggregate_maximum_bitrate_s>();
      break;
    case types::enhanced_coverage_restricted:
      break;
    case types::ce_mode_brestricted:
      break;
    case types::pending_data_ind:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }
}
init_context_setup_request_ies_o::value_c::value_c(const init_context_setup_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.init(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      c.init(other.c.get<erab_to_be_setup_list_ctxt_su_req_l>());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::security_key:
      c.init(other.c.get<fixed_bitstring<256, false, true> >());
      break;
    case types::trace_activation:
      c.init(other.c.get<trace_activation_s>());
      break;
    case types::ho_restrict_list:
      c.init(other.c.get<ho_restrict_list_s>());
      break;
    case types::ue_radio_cap:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::subscriber_profile_idfor_rfp:
      c.init(other.c.get<uint16_t>());
      break;
    case types::cs_fallback_ind:
      c.init(other.c.get<cs_fallback_ind_e>());
      break;
    case types::srvcc_operation_possible:
      c.init(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::csg_membership_status:
      c.init(other.c.get<csg_membership_status_e>());
      break;
    case types::registered_lai:
      c.init(other.c.get<lai_s>());
      break;
    case types::gummei_id:
      c.init(other.c.get<gummei_s>());
      break;
    case types::mme_ue_s1ap_id_minus2:
      c.init(other.c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      c.init(other.c.get<management_based_mdt_allowed_e>());
      break;
    case types::management_based_mdtplmn_list:
      c.init(other.c.get<mdtplmn_list_l>());
      break;
    case types::add_cs_fallback_ind:
      c.init(other.c.get<add_cs_fallback_ind_e>());
      break;
    case types::masked_imeisv:
      c.init(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::expected_ue_behaviour:
      c.init(other.c.get<expected_ue_behaviour_s>());
      break;
    case types::pro_se_authorized:
      c.init(other.c.get<pro_se_authorized_s>());
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      c.init(other.c.get<ueuser_plane_cio_tsupport_ind_e>());
      break;
    case types::v2xservices_authorized:
      c.init(other.c.get<v2xservices_authorized_s>());
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.init(other.c.get<ue_sidelink_aggregate_maximum_bitrate_s>());
      break;
    case types::enhanced_coverage_restricted:
      c.init(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::ce_mode_brestricted:
      c.init(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::pending_data_ind:
      c.init(other.c.get<pending_data_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }
}
init_context_setup_request_ies_o::value_c& init_context_setup_request_ies_o::value_c::
                                           operator=(const init_context_setup_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      c.set(other.c.get<ue_aggregate_maximum_bitrate_s>());
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      c.set(other.c.get<erab_to_be_setup_list_ctxt_su_req_l>());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::security_key:
      c.set(other.c.get<fixed_bitstring<256, false, true> >());
      break;
    case types::trace_activation:
      c.set(other.c.get<trace_activation_s>());
      break;
    case types::ho_restrict_list:
      c.set(other.c.get<ho_restrict_list_s>());
      break;
    case types::ue_radio_cap:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::subscriber_profile_idfor_rfp:
      c.set(other.c.get<uint16_t>());
      break;
    case types::cs_fallback_ind:
      c.set(other.c.get<cs_fallback_ind_e>());
      break;
    case types::srvcc_operation_possible:
      c.set(other.c.get<srvcc_operation_possible_e>());
      break;
    case types::csg_membership_status:
      c.set(other.c.get<csg_membership_status_e>());
      break;
    case types::registered_lai:
      c.set(other.c.get<lai_s>());
      break;
    case types::gummei_id:
      c.set(other.c.get<gummei_s>());
      break;
    case types::mme_ue_s1ap_id_minus2:
      c.set(other.c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      c.set(other.c.get<management_based_mdt_allowed_e>());
      break;
    case types::management_based_mdtplmn_list:
      c.set(other.c.get<mdtplmn_list_l>());
      break;
    case types::add_cs_fallback_ind:
      c.set(other.c.get<add_cs_fallback_ind_e>());
      break;
    case types::masked_imeisv:
      c.set(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::expected_ue_behaviour:
      c.set(other.c.get<expected_ue_behaviour_s>());
      break;
    case types::pro_se_authorized:
      c.set(other.c.get<pro_se_authorized_s>());
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      c.set(other.c.get<ueuser_plane_cio_tsupport_ind_e>());
      break;
    case types::v2xservices_authorized:
      c.set(other.c.get<v2xservices_authorized_s>());
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      c.set(other.c.get<ue_sidelink_aggregate_maximum_bitrate_s>());
      break;
    case types::enhanced_coverage_restricted:
      c.set(other.c.get<enhanced_coverage_restricted_e>());
      break;
    case types::ce_mode_brestricted:
      c.set(other.c.get<ce_mode_brestricted_e>());
      break;
    case types::pending_data_ind:
      c.set(other.c.get<pending_data_ind_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }

  return *this;
}
void init_context_setup_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::ueaggregate_maximum_bitrate:
      j.write_fieldname("UEAggregateMaximumBitrate");
      c.get<ue_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      j.start_array("E-RABToBeSetupListCtxtSUReq");
      for (const auto& e1 : c.get<erab_to_be_setup_list_ctxt_su_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::security_key:
      j.write_str("BIT STRING", c.get<fixed_bitstring<256, false, true> >().to_string());
      break;
    case types::trace_activation:
      j.write_fieldname("TraceActivation");
      c.get<trace_activation_s>().to_json(j);
      break;
    case types::ho_restrict_list:
      j.write_fieldname("HandoverRestrictionList");
      c.get<ho_restrict_list_s>().to_json(j);
      break;
    case types::ue_radio_cap:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::subscriber_profile_idfor_rfp:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::cs_fallback_ind:
      j.write_str("CSFallbackIndicator", c.get<cs_fallback_ind_e>().to_string());
      break;
    case types::srvcc_operation_possible:
      j.write_str("SRVCCOperationPossible", "possible");
      break;
    case types::csg_membership_status:
      j.write_str("CSGMembershipStatus", c.get<csg_membership_status_e>().to_string());
      break;
    case types::registered_lai:
      j.write_fieldname("LAI");
      c.get<lai_s>().to_json(j);
      break;
    case types::gummei_id:
      j.write_fieldname("GUMMEI");
      c.get<gummei_s>().to_json(j);
      break;
    case types::mme_ue_s1ap_id_minus2:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::management_based_mdt_allowed:
      j.write_str("ManagementBasedMDTAllowed", "allowed");
      break;
    case types::management_based_mdtplmn_list:
      j.start_array("MDTPLMNList");
      for (const auto& e1 : c.get<mdtplmn_list_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    case types::add_cs_fallback_ind:
      j.write_str("AdditionalCSFallbackIndicator", c.get<add_cs_fallback_ind_e>().to_string());
      break;
    case types::masked_imeisv:
      j.write_str("BIT STRING", c.get<fixed_bitstring<64, false, true> >().to_string());
      break;
    case types::expected_ue_behaviour:
      j.write_fieldname("ExpectedUEBehaviour");
      c.get<expected_ue_behaviour_s>().to_json(j);
      break;
    case types::pro_se_authorized:
      j.write_fieldname("ProSeAuthorized");
      c.get<pro_se_authorized_s>().to_json(j);
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      j.write_str("UEUserPlaneCIoTSupportIndicator", "supported");
      break;
    case types::v2xservices_authorized:
      j.write_fieldname("V2XServicesAuthorized");
      c.get<v2xservices_authorized_s>().to_json(j);
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      j.write_fieldname("UESidelinkAggregateMaximumBitrate");
      c.get<ue_sidelink_aggregate_maximum_bitrate_s>().to_json(j);
      break;
    case types::enhanced_coverage_restricted:
      j.write_str("EnhancedCoverageRestricted", "restricted");
      break;
    case types::ce_mode_brestricted:
      j.write_str("CE-ModeBRestricted", c.get<ce_mode_brestricted_e>().to_string());
      break;
    case types::pending_data_ind:
      j.write_str("PendingDataIndication", "true");
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_context_setup_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_to_be_setup_list_ctxt_su_req_l>(), 1, 256, true));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::security_key:
      HANDLE_CODE((c.get<fixed_bitstring<256, false, true> >().pack(bref)));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().pack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().pack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::subscriber_profile_idfor_rfp:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::cs_fallback_ind:
      HANDLE_CODE(c.get<cs_fallback_ind_e>().pack(bref));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().pack(bref));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().pack(bref));
      break;
    case types::registered_lai:
      HANDLE_CODE(c.get<lai_s>().pack(bref));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().pack(bref));
      break;
    case types::mme_ue_s1ap_id_minus2:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::management_based_mdt_allowed:
      HANDLE_CODE(c.get<management_based_mdt_allowed_e>().pack(bref));
      break;
    case types::management_based_mdtplmn_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<mdtplmn_list_l>(), 1, 16, true));
      break;
    case types::add_cs_fallback_ind:
      HANDLE_CODE(c.get<add_cs_fallback_ind_e>().pack(bref));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().pack(bref)));
      break;
    case types::expected_ue_behaviour:
      HANDLE_CODE(c.get<expected_ue_behaviour_s>().pack(bref));
      break;
    case types::pro_se_authorized:
      HANDLE_CODE(c.get<pro_se_authorized_s>().pack(bref));
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      HANDLE_CODE(c.get<ueuser_plane_cio_tsupport_ind_e>().pack(bref));
      break;
    case types::v2xservices_authorized:
      HANDLE_CODE(c.get<v2xservices_authorized_s>().pack(bref));
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_sidelink_aggregate_maximum_bitrate_s>().pack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().pack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().pack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::ueaggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::erab_to_be_setup_list_ctxt_su_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_to_be_setup_list_ctxt_su_req_l>(), bref, 1, 256, true));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::security_key:
      HANDLE_CODE((c.get<fixed_bitstring<256, false, true> >().unpack(bref)));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().unpack(bref));
      break;
    case types::ho_restrict_list:
      HANDLE_CODE(c.get<ho_restrict_list_s>().unpack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::subscriber_profile_idfor_rfp:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::cs_fallback_ind:
      HANDLE_CODE(c.get<cs_fallback_ind_e>().unpack(bref));
      break;
    case types::srvcc_operation_possible:
      HANDLE_CODE(c.get<srvcc_operation_possible_e>().unpack(bref));
      break;
    case types::csg_membership_status:
      HANDLE_CODE(c.get<csg_membership_status_e>().unpack(bref));
      break;
    case types::registered_lai:
      HANDLE_CODE(c.get<lai_s>().unpack(bref));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().unpack(bref));
      break;
    case types::mme_ue_s1ap_id_minus2:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::management_based_mdt_allowed:
      HANDLE_CODE(c.get<management_based_mdt_allowed_e>().unpack(bref));
      break;
    case types::management_based_mdtplmn_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<mdtplmn_list_l>(), bref, 1, 16, true));
      break;
    case types::add_cs_fallback_ind:
      HANDLE_CODE(c.get<add_cs_fallback_ind_e>().unpack(bref));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().unpack(bref)));
      break;
    case types::expected_ue_behaviour:
      HANDLE_CODE(c.get<expected_ue_behaviour_s>().unpack(bref));
      break;
    case types::pro_se_authorized:
      HANDLE_CODE(c.get<pro_se_authorized_s>().unpack(bref));
      break;
    case types::ueuser_plane_cio_tsupport_ind:
      HANDLE_CODE(c.get<ueuser_plane_cio_tsupport_ind_e>().unpack(bref));
      break;
    case types::v2xservices_authorized:
      HANDLE_CODE(c.get<v2xservices_authorized_s>().unpack(bref));
      break;
    case types::ue_sidelink_aggregate_maximum_bitrate:
      HANDLE_CODE(c.get<ue_sidelink_aggregate_maximum_bitrate_s>().unpack(bref));
      break;
    case types::enhanced_coverage_restricted:
      HANDLE_CODE(c.get<enhanced_coverage_restricted_e>().unpack(bref));
      break;
    case types::ce_mode_brestricted:
      HANDLE_CODE(c.get<ce_mode_brestricted_e>().unpack(bref));
      break;
    case types::pending_data_ind:
      HANDLE_CODE(c.get<pending_data_ind_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string init_context_setup_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "UEAggregateMaximumBitrate",
                                  "E-RABToBeSetupListCtxtSUReq",
                                  "UESecurityCapabilities",
                                  "BIT STRING",
                                  "TraceActivation",
                                  "HandoverRestrictionList",
                                  "OCTET STRING",
                                  "INTEGER (1..256)",
                                  "CSFallbackIndicator",
                                  "SRVCCOperationPossible",
                                  "CSGMembershipStatus",
                                  "LAI",
                                  "GUMMEI",
                                  "INTEGER (0..4294967295)",
                                  "ManagementBasedMDTAllowed",
                                  "MDTPLMNList",
                                  "AdditionalCSFallbackIndicator",
                                  "BIT STRING",
                                  "ExpectedUEBehaviour",
                                  "ProSeAuthorized",
                                  "UEUserPlaneCIoTSupportIndicator",
                                  "V2XServicesAuthorized",
                                  "UESidelinkAggregateMaximumBitrate",
                                  "EnhancedCoverageRestricted",
                                  "CE-ModeBRestricted",
                                  "PendingDataIndication"};
  return convert_enum_idx(options, 28, value, "init_context_setup_request_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<init_context_setup_request_ies_o>;

init_context_setup_request_ies_container::init_context_setup_request_ies_container() :
  mme_ue_s1ap_id(0, crit_e::reject),
  enb_ue_s1ap_id(8, crit_e::reject),
  ueaggregate_maximum_bitrate(66, crit_e::reject),
  erab_to_be_setup_list_ctxt_su_req(24, crit_e::reject),
  ue_security_cap(107, crit_e::reject),
  security_key(73, crit_e::reject),
  trace_activation(25, crit_e::ignore),
  ho_restrict_list(41, crit_e::ignore),
  ue_radio_cap(74, crit_e::ignore),
  subscriber_profile_idfor_rfp(106, crit_e::ignore),
  cs_fallback_ind(108, crit_e::reject),
  srvcc_operation_possible(124, crit_e::ignore),
  csg_membership_status(146, crit_e::ignore),
  registered_lai(159, crit_e::ignore),
  gummei_id(75, crit_e::ignore),
  mme_ue_s1ap_id_minus2(158, crit_e::ignore),
  management_based_mdt_allowed(165, crit_e::ignore),
  management_based_mdtplmn_list(177, crit_e::ignore),
  add_cs_fallback_ind(187, crit_e::ignore),
  masked_imeisv(192, crit_e::ignore),
  expected_ue_behaviour(196, crit_e::ignore),
  pro_se_authorized(195, crit_e::ignore),
  ueuser_plane_cio_tsupport_ind(241, crit_e::ignore),
  v2xservices_authorized(240, crit_e::ignore),
  ue_sidelink_aggregate_maximum_bitrate(248, crit_e::ignore),
  enhanced_coverage_restricted(251, crit_e::ignore),
  ce_mode_brestricted(271, crit_e::ignore),
  pending_data_ind(283, crit_e::ignore)
{}
SRSASN_CODE init_context_setup_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 6;
  nof_ies += trace_activation_present ? 1 : 0;
  nof_ies += ho_restrict_list_present ? 1 : 0;
  nof_ies += ue_radio_cap_present ? 1 : 0;
  nof_ies += subscriber_profile_idfor_rfp_present ? 1 : 0;
  nof_ies += cs_fallback_ind_present ? 1 : 0;
  nof_ies += srvcc_operation_possible_present ? 1 : 0;
  nof_ies += csg_membership_status_present ? 1 : 0;
  nof_ies += registered_lai_present ? 1 : 0;
  nof_ies += gummei_id_present ? 1 : 0;
  nof_ies += mme_ue_s1ap_id_minus2_present ? 1 : 0;
  nof_ies += management_based_mdt_allowed_present ? 1 : 0;
  nof_ies += management_based_mdtplmn_list_present ? 1 : 0;
  nof_ies += add_cs_fallback_ind_present ? 1 : 0;
  nof_ies += masked_imeisv_present ? 1 : 0;
  nof_ies += expected_ue_behaviour_present ? 1 : 0;
  nof_ies += pro_se_authorized_present ? 1 : 0;
  nof_ies += ueuser_plane_cio_tsupport_ind_present ? 1 : 0;
  nof_ies += v2xservices_authorized_present ? 1 : 0;
  nof_ies += ue_sidelink_aggregate_maximum_bitrate_present ? 1 : 0;
  nof_ies += enhanced_coverage_restricted_present ? 1 : 0;
  nof_ies += ce_mode_brestricted_present ? 1 : 0;
  nof_ies += pending_data_ind_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(ueaggregate_maximum_bitrate.pack(bref));
  HANDLE_CODE(erab_to_be_setup_list_ctxt_su_req.pack(bref));
  HANDLE_CODE(ue_security_cap.pack(bref));
  HANDLE_CODE(security_key.pack(bref));
  if (trace_activation_present) {
    HANDLE_CODE(trace_activation.pack(bref));
  }
  if (ho_restrict_list_present) {
    HANDLE_CODE(ho_restrict_list.pack(bref));
  }
  if (ue_radio_cap_present) {
    HANDLE_CODE(ue_radio_cap.pack(bref));
  }
  if (subscriber_profile_idfor_rfp_present) {
    HANDLE_CODE(subscriber_profile_idfor_rfp.pack(bref));
  }
  if (cs_fallback_ind_present) {
    HANDLE_CODE(cs_fallback_ind.pack(bref));
  }
  if (srvcc_operation_possible_present) {
    HANDLE_CODE(srvcc_operation_possible.pack(bref));
  }
  if (csg_membership_status_present) {
    HANDLE_CODE(csg_membership_status.pack(bref));
  }
  if (registered_lai_present) {
    HANDLE_CODE(registered_lai.pack(bref));
  }
  if (gummei_id_present) {
    HANDLE_CODE(gummei_id.pack(bref));
  }
  if (mme_ue_s1ap_id_minus2_present) {
    HANDLE_CODE(mme_ue_s1ap_id_minus2.pack(bref));
  }
  if (management_based_mdt_allowed_present) {
    HANDLE_CODE(management_based_mdt_allowed.pack(bref));
  }
  if (management_based_mdtplmn_list_present) {
    HANDLE_CODE(management_based_mdtplmn_list.pack(bref));
  }
  if (add_cs_fallback_ind_present) {
    HANDLE_CODE(add_cs_fallback_ind.pack(bref));
  }
  if (masked_imeisv_present) {
    HANDLE_CODE(masked_imeisv.pack(bref));
  }
  if (expected_ue_behaviour_present) {
    HANDLE_CODE(expected_ue_behaviour.pack(bref));
  }
  if (pro_se_authorized_present) {
    HANDLE_CODE(pro_se_authorized.pack(bref));
  }
  if (ueuser_plane_cio_tsupport_ind_present) {
    HANDLE_CODE(ueuser_plane_cio_tsupport_ind.pack(bref));
  }
  if (v2xservices_authorized_present) {
    HANDLE_CODE(v2xservices_authorized.pack(bref));
  }
  if (ue_sidelink_aggregate_maximum_bitrate_present) {
    HANDLE_CODE(ue_sidelink_aggregate_maximum_bitrate.pack(bref));
  }
  if (enhanced_coverage_restricted_present) {
    HANDLE_CODE(enhanced_coverage_restricted.pack(bref));
  }
  if (ce_mode_brestricted_present) {
    HANDLE_CODE(ce_mode_brestricted.pack(bref));
  }
  if (pending_data_ind_present) {
    HANDLE_CODE(pending_data_ind.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 6;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<init_context_setup_request_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 66:
        nof_mandatory_ies--;
        ueaggregate_maximum_bitrate.id    = c.id;
        ueaggregate_maximum_bitrate.crit  = c.crit;
        ueaggregate_maximum_bitrate.value = c.value.ueaggregate_maximum_bitrate();
        break;
      case 24:
        nof_mandatory_ies--;
        erab_to_be_setup_list_ctxt_su_req.id    = c.id;
        erab_to_be_setup_list_ctxt_su_req.crit  = c.crit;
        erab_to_be_setup_list_ctxt_su_req.value = c.value.erab_to_be_setup_list_ctxt_su_req();
        break;
      case 107:
        nof_mandatory_ies--;
        ue_security_cap.id    = c.id;
        ue_security_cap.crit  = c.crit;
        ue_security_cap.value = c.value.ue_security_cap();
        break;
      case 73:
        nof_mandatory_ies--;
        security_key.id    = c.id;
        security_key.crit  = c.crit;
        security_key.value = c.value.security_key();
        break;
      case 25:
        trace_activation_present = true;
        trace_activation.id      = c.id;
        trace_activation.crit    = c.crit;
        trace_activation.value   = c.value.trace_activation();
        break;
      case 41:
        ho_restrict_list_present = true;
        ho_restrict_list.id      = c.id;
        ho_restrict_list.crit    = c.crit;
        ho_restrict_list.value   = c.value.ho_restrict_list();
        break;
      case 74:
        ue_radio_cap_present = true;
        ue_radio_cap.id      = c.id;
        ue_radio_cap.crit    = c.crit;
        ue_radio_cap.value   = c.value.ue_radio_cap();
        break;
      case 106:
        subscriber_profile_idfor_rfp_present = true;
        subscriber_profile_idfor_rfp.id      = c.id;
        subscriber_profile_idfor_rfp.crit    = c.crit;
        subscriber_profile_idfor_rfp.value   = c.value.subscriber_profile_idfor_rfp();
        break;
      case 108:
        cs_fallback_ind_present = true;
        cs_fallback_ind.id      = c.id;
        cs_fallback_ind.crit    = c.crit;
        cs_fallback_ind.value   = c.value.cs_fallback_ind();
        break;
      case 124:
        srvcc_operation_possible_present = true;
        srvcc_operation_possible.id      = c.id;
        srvcc_operation_possible.crit    = c.crit;
        srvcc_operation_possible.value   = c.value.srvcc_operation_possible();
        break;
      case 146:
        csg_membership_status_present = true;
        csg_membership_status.id      = c.id;
        csg_membership_status.crit    = c.crit;
        csg_membership_status.value   = c.value.csg_membership_status();
        break;
      case 159:
        registered_lai_present = true;
        registered_lai.id      = c.id;
        registered_lai.crit    = c.crit;
        registered_lai.value   = c.value.registered_lai();
        break;
      case 75:
        gummei_id_present = true;
        gummei_id.id      = c.id;
        gummei_id.crit    = c.crit;
        gummei_id.value   = c.value.gummei_id();
        break;
      case 158:
        mme_ue_s1ap_id_minus2_present = true;
        mme_ue_s1ap_id_minus2.id      = c.id;
        mme_ue_s1ap_id_minus2.crit    = c.crit;
        mme_ue_s1ap_id_minus2.value   = c.value.mme_ue_s1ap_id_minus2();
        break;
      case 165:
        management_based_mdt_allowed_present = true;
        management_based_mdt_allowed.id      = c.id;
        management_based_mdt_allowed.crit    = c.crit;
        management_based_mdt_allowed.value   = c.value.management_based_mdt_allowed();
        break;
      case 177:
        management_based_mdtplmn_list_present = true;
        management_based_mdtplmn_list.id      = c.id;
        management_based_mdtplmn_list.crit    = c.crit;
        management_based_mdtplmn_list.value   = c.value.management_based_mdtplmn_list();
        break;
      case 187:
        add_cs_fallback_ind_present = true;
        add_cs_fallback_ind.id      = c.id;
        add_cs_fallback_ind.crit    = c.crit;
        add_cs_fallback_ind.value   = c.value.add_cs_fallback_ind();
        break;
      case 192:
        masked_imeisv_present = true;
        masked_imeisv.id      = c.id;
        masked_imeisv.crit    = c.crit;
        masked_imeisv.value   = c.value.masked_imeisv();
        break;
      case 196:
        expected_ue_behaviour_present = true;
        expected_ue_behaviour.id      = c.id;
        expected_ue_behaviour.crit    = c.crit;
        expected_ue_behaviour.value   = c.value.expected_ue_behaviour();
        break;
      case 195:
        pro_se_authorized_present = true;
        pro_se_authorized.id      = c.id;
        pro_se_authorized.crit    = c.crit;
        pro_se_authorized.value   = c.value.pro_se_authorized();
        break;
      case 241:
        ueuser_plane_cio_tsupport_ind_present = true;
        ueuser_plane_cio_tsupport_ind.id      = c.id;
        ueuser_plane_cio_tsupport_ind.crit    = c.crit;
        ueuser_plane_cio_tsupport_ind.value   = c.value.ueuser_plane_cio_tsupport_ind();
        break;
      case 240:
        v2xservices_authorized_present = true;
        v2xservices_authorized.id      = c.id;
        v2xservices_authorized.crit    = c.crit;
        v2xservices_authorized.value   = c.value.v2xservices_authorized();
        break;
      case 248:
        ue_sidelink_aggregate_maximum_bitrate_present = true;
        ue_sidelink_aggregate_maximum_bitrate.id      = c.id;
        ue_sidelink_aggregate_maximum_bitrate.crit    = c.crit;
        ue_sidelink_aggregate_maximum_bitrate.value   = c.value.ue_sidelink_aggregate_maximum_bitrate();
        break;
      case 251:
        enhanced_coverage_restricted_present = true;
        enhanced_coverage_restricted.id      = c.id;
        enhanced_coverage_restricted.crit    = c.crit;
        enhanced_coverage_restricted.value   = c.value.enhanced_coverage_restricted();
        break;
      case 271:
        ce_mode_brestricted_present = true;
        ce_mode_brestricted.id      = c.id;
        ce_mode_brestricted.crit    = c.crit;
        ce_mode_brestricted.value   = c.value.ce_mode_brestricted();
        break;
      case 283:
        pending_data_ind_present = true;
        pending_data_ind.id      = c.id;
        pending_data_ind.crit    = c.crit;
        pending_data_ind.value   = c.value.pending_data_ind();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_context_setup_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  ueaggregate_maximum_bitrate.to_json(j);
  j.write_fieldname("");
  erab_to_be_setup_list_ctxt_su_req.to_json(j);
  j.write_fieldname("");
  ue_security_cap.to_json(j);
  j.write_fieldname("");
  security_key.to_json(j);
  if (trace_activation_present) {
    j.write_fieldname("");
    trace_activation.to_json(j);
  }
  if (ho_restrict_list_present) {
    j.write_fieldname("");
    ho_restrict_list.to_json(j);
  }
  if (ue_radio_cap_present) {
    j.write_fieldname("");
    ue_radio_cap.to_json(j);
  }
  if (subscriber_profile_idfor_rfp_present) {
    j.write_fieldname("");
    subscriber_profile_idfor_rfp.to_json(j);
  }
  if (cs_fallback_ind_present) {
    j.write_fieldname("");
    cs_fallback_ind.to_json(j);
  }
  if (srvcc_operation_possible_present) {
    j.write_fieldname("");
    srvcc_operation_possible.to_json(j);
  }
  if (csg_membership_status_present) {
    j.write_fieldname("");
    csg_membership_status.to_json(j);
  }
  if (registered_lai_present) {
    j.write_fieldname("");
    registered_lai.to_json(j);
  }
  if (gummei_id_present) {
    j.write_fieldname("");
    gummei_id.to_json(j);
  }
  if (mme_ue_s1ap_id_minus2_present) {
    j.write_fieldname("");
    mme_ue_s1ap_id_minus2.to_json(j);
  }
  if (management_based_mdt_allowed_present) {
    j.write_fieldname("");
    management_based_mdt_allowed.to_json(j);
  }
  if (management_based_mdtplmn_list_present) {
    j.write_fieldname("");
    management_based_mdtplmn_list.to_json(j);
  }
  if (add_cs_fallback_ind_present) {
    j.write_fieldname("");
    add_cs_fallback_ind.to_json(j);
  }
  if (masked_imeisv_present) {
    j.write_fieldname("");
    masked_imeisv.to_json(j);
  }
  if (expected_ue_behaviour_present) {
    j.write_fieldname("");
    expected_ue_behaviour.to_json(j);
  }
  if (pro_se_authorized_present) {
    j.write_fieldname("");
    pro_se_authorized.to_json(j);
  }
  if (ueuser_plane_cio_tsupport_ind_present) {
    j.write_fieldname("");
    ueuser_plane_cio_tsupport_ind.to_json(j);
  }
  if (v2xservices_authorized_present) {
    j.write_fieldname("");
    v2xservices_authorized.to_json(j);
  }
  if (ue_sidelink_aggregate_maximum_bitrate_present) {
    j.write_fieldname("");
    ue_sidelink_aggregate_maximum_bitrate.to_json(j);
  }
  if (enhanced_coverage_restricted_present) {
    j.write_fieldname("");
    enhanced_coverage_restricted.to_json(j);
  }
  if (ce_mode_brestricted_present) {
    j.write_fieldname("");
    ce_mode_brestricted.to_json(j);
  }
  if (pending_data_ind_present) {
    j.write_fieldname("");
    pending_data_ind.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupRequest ::= SEQUENCE
SRSASN_CODE init_context_setup_request_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_request_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void init_context_setup_request_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// InitialContextSetupResponseIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t init_context_setup_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {0, 8, 51, 48, 58};
  return map_enum_number(options, 5, idx, "id");
}
bool init_context_setup_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {0, 8, 51, 48, 58};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_context_setup_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 0:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    case 51:
      return crit_e::ignore;
    case 48:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_context_setup_resp_ies_o::value_c init_context_setup_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 0:
      ret.set(value_c::types::mme_ue_s1ap_id);
      break;
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 51:
      ret.set(value_c::types::erab_setup_list_ctxt_su_res);
      break;
    case 48:
      ret.set(value_c::types::erab_failed_to_setup_list_ctxt_su_res);
      break;
    case 58:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_context_setup_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 0:
      return presence_e::mandatory;
    case 8:
      return presence_e::mandatory;
    case 51:
      return presence_e::mandatory;
    case 48:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint64_t& init_context_setup_resp_ies_o::value_c::mme_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
uint32_t& init_context_setup_resp_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
erab_setup_list_ctxt_su_res_l& init_context_setup_resp_ies_o::value_c::erab_setup_list_ctxt_su_res()
{
  assert_choice_type("E-RABSetupListCtxtSURes", type_.to_string(), "Value");
  return c.get<erab_setup_list_ctxt_su_res_l>();
}
erab_list_l& init_context_setup_resp_ies_o::value_c::erab_failed_to_setup_list_ctxt_su_res()
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
crit_diagnostics_s& init_context_setup_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& init_context_setup_resp_ies_o::value_c::mme_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..4294967295)", type_.to_string(), "Value");
  return c.get<uint64_t>();
}
const uint32_t& init_context_setup_resp_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const erab_setup_list_ctxt_su_res_l& init_context_setup_resp_ies_o::value_c::erab_setup_list_ctxt_su_res() const
{
  assert_choice_type("E-RABSetupListCtxtSURes", type_.to_string(), "Value");
  return c.get<erab_setup_list_ctxt_su_res_l>();
}
const erab_list_l& init_context_setup_resp_ies_o::value_c::erab_failed_to_setup_list_ctxt_su_res() const
{
  assert_choice_type("E-RABList", type_.to_string(), "Value");
  return c.get<erab_list_l>();
}
const crit_diagnostics_s& init_context_setup_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type("CriticalityDiagnostics", type_.to_string(), "Value");
  return c.get<crit_diagnostics_s>();
}
void init_context_setup_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::erab_setup_list_ctxt_su_res:
      c.destroy<erab_setup_list_ctxt_su_res_l>();
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      c.destroy<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void init_context_setup_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::mme_ue_s1ap_id:
      break;
    case types::enb_ue_s1ap_id:
      break;
    case types::erab_setup_list_ctxt_su_res:
      c.init<erab_setup_list_ctxt_su_res_l>();
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      c.init<erab_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }
}
init_context_setup_resp_ies_o::value_c::value_c(const init_context_setup_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::erab_setup_list_ctxt_su_res:
      c.init(other.c.get<erab_setup_list_ctxt_su_res_l>());
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      c.init(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }
}
init_context_setup_resp_ies_o::value_c& init_context_setup_resp_ies_o::value_c::
                                        operator=(const init_context_setup_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::erab_setup_list_ctxt_su_res:
      c.set(other.c.get<erab_setup_list_ctxt_su_res_l>());
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      c.set(other.c.get<erab_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }

  return *this;
}
void init_context_setup_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::mme_ue_s1ap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::erab_setup_list_ctxt_su_res:
      j.start_array("E-RABSetupListCtxtSURes");
      for (const auto& e1 : c.get<erab_setup_list_ctxt_su_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      j.start_array("E-RABList");
      for (const auto& e1 : c.get<erab_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_context_setup_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_setup_list_ctxt_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_setup_list_ctxt_su_res_l>(), 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<erab_list_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::erab_setup_list_ctxt_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_setup_list_ctxt_su_res_l>(), bref, 1, 256, true));
      break;
    case types::erab_failed_to_setup_list_ctxt_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<erab_list_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string init_context_setup_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..16777215)",
                                  "E-RABSetupListCtxtSURes",
                                  "E-RABList",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "init_context_setup_resp_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<init_context_setup_resp_ies_o>;

init_context_setup_resp_ies_container::init_context_setup_resp_ies_container() :
  mme_ue_s1ap_id(0, crit_e::ignore),
  enb_ue_s1ap_id(8, crit_e::ignore),
  erab_setup_list_ctxt_su_res(51, crit_e::ignore),
  erab_failed_to_setup_list_ctxt_su_res(48, crit_e::ignore),
  crit_diagnostics(58, crit_e::ignore)
{}
SRSASN_CODE init_context_setup_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += erab_failed_to_setup_list_ctxt_su_res_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(mme_ue_s1ap_id.pack(bref));
  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(erab_setup_list_ctxt_su_res.pack(bref));
  if (erab_failed_to_setup_list_ctxt_su_res_present) {
    HANDLE_CODE(erab_failed_to_setup_list_ctxt_su_res.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<init_context_setup_resp_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 0:
        nof_mandatory_ies--;
        mme_ue_s1ap_id.id    = c.id;
        mme_ue_s1ap_id.crit  = c.crit;
        mme_ue_s1ap_id.value = c.value.mme_ue_s1ap_id();
        break;
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 51:
        nof_mandatory_ies--;
        erab_setup_list_ctxt_su_res.id    = c.id;
        erab_setup_list_ctxt_su_res.crit  = c.crit;
        erab_setup_list_ctxt_su_res.value = c.value.erab_setup_list_ctxt_su_res();
        break;
      case 48:
        erab_failed_to_setup_list_ctxt_su_res_present = true;
        erab_failed_to_setup_list_ctxt_su_res.id      = c.id;
        erab_failed_to_setup_list_ctxt_su_res.crit    = c.crit;
        erab_failed_to_setup_list_ctxt_su_res.value   = c.value.erab_failed_to_setup_list_ctxt_su_res();
        break;
      case 58:
        crit_diagnostics_present = true;
        crit_diagnostics.id      = c.id;
        crit_diagnostics.crit    = c.crit;
        crit_diagnostics.value   = c.value.crit_diagnostics();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_context_setup_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  mme_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  erab_setup_list_ctxt_su_res.to_json(j);
  if (erab_failed_to_setup_list_ctxt_su_res_present) {
    j.write_fieldname("");
    erab_failed_to_setup_list_ctxt_su_res.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupResponse ::= SEQUENCE
SRSASN_CODE init_context_setup_resp_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_resp_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  return SRSASN_SUCCESS;
}
void init_context_setup_resp_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
}

// Coverage-Level ::= ENUMERATED
std::string coverage_level_opts::to_string() const
{
  static const char* options[] = {"extendedcoverage"};
  return convert_enum_idx(options, 1, value, "coverage_level_e");
}

// GUMMEIType ::= ENUMERATED
std::string gummei_type_opts::to_string() const
{
  static const char* options[] = {"native", "mapped"};
  return convert_enum_idx(options, 2, value, "gummei_type_e");
}

// RRC-Establishment-Cause ::= ENUMERATED
std::string rrc_establishment_cause_opts::to_string() const
{
  static const char* options[] = {"emergency",
                                  "highPriorityAccess",
                                  "mt-Access",
                                  "mo-Signalling",
                                  "mo-Data",
                                  "delay-TolerantAccess",
                                  "mo-VoiceCall",
                                  "mo-ExceptionData"};
  return convert_enum_idx(options, 8, value, "rrc_establishment_cause_e");
}

// RelayNode-Indicator ::= ENUMERATED
std::string relay_node_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "relay_node_ind_e");
}

// InitialUEMessage-IEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t init_ue_msg_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {8,   26,  67,  100, 134, 96,  127, 75,  145, 155,
                                     160, 170, 176, 184, 186, 223, 230, 242, 246, 250};
  return map_enum_number(options, 20, idx, "id");
}
bool init_ue_msg_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {8,   26,  67,  100, 134, 96,  127, 75,  145, 155,
                                     160, 170, 176, 184, 186, 223, 230, 242, 246, 250};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_ue_msg_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 8:
      return crit_e::reject;
    case 26:
      return crit_e::reject;
    case 67:
      return crit_e::reject;
    case 100:
      return crit_e::ignore;
    case 134:
      return crit_e::ignore;
    case 96:
      return crit_e::reject;
    case 127:
      return crit_e::reject;
    case 75:
      return crit_e::reject;
    case 145:
      return crit_e::reject;
    case 155:
      return crit_e::ignore;
    case 160:
      return crit_e::reject;
    case 170:
      return crit_e::ignore;
    case 176:
      return crit_e::ignore;
    case 184:
      return crit_e::ignore;
    case 186:
      return crit_e::ignore;
    case 223:
      return crit_e::ignore;
    case 230:
      return crit_e::ignore;
    case 242:
      return crit_e::ignore;
    case 246:
      return crit_e::ignore;
    case 250:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_ue_msg_ies_o::value_c init_ue_msg_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 8:
      ret.set(value_c::types::enb_ue_s1ap_id);
      break;
    case 26:
      ret.set(value_c::types::nas_pdu);
      break;
    case 67:
      ret.set(value_c::types::tai);
      break;
    case 100:
      ret.set(value_c::types::eutran_cgi);
      break;
    case 134:
      ret.set(value_c::types::rrc_establishment_cause);
      break;
    case 96:
      ret.set(value_c::types::s_tmsi);
      break;
    case 127:
      ret.set(value_c::types::csg_id);
      break;
    case 75:
      ret.set(value_c::types::gummei_id);
      break;
    case 145:
      ret.set(value_c::types::cell_access_mode);
      break;
    case 155:
      ret.set(value_c::types::gw_transport_layer_address);
      break;
    case 160:
      ret.set(value_c::types::relay_node_ind);
      break;
    case 170:
      ret.set(value_c::types::gummei_type);
      break;
    case 176:
      ret.set(value_c::types::tunnel_info_for_bbf);
      break;
    case 184:
      ret.set(value_c::types::sipto_l_gw_transport_layer_address);
      break;
    case 186:
      ret.set(value_c::types::lhn_id);
      break;
    case 223:
      ret.set(value_c::types::mme_group_id);
      break;
    case 230:
      ret.set(value_c::types::ue_usage_type);
      break;
    case 242:
      ret.set(value_c::types::ce_mode_b_support_ind);
      break;
    case 246:
      ret.set(value_c::types::dcn_id);
      break;
    case 250:
      ret.set(value_c::types::coverage_level);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_ue_msg_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 8:
      return presence_e::mandatory;
    case 26:
      return presence_e::mandatory;
    case 67:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 134:
      return presence_e::mandatory;
    case 96:
      return presence_e::optional;
    case 127:
      return presence_e::optional;
    case 75:
      return presence_e::optional;
    case 145:
      return presence_e::optional;
    case 155:
      return presence_e::optional;
    case 160:
      return presence_e::optional;
    case 170:
      return presence_e::optional;
    case 176:
      return presence_e::optional;
    case 184:
      return presence_e::optional;
    case 186:
      return presence_e::optional;
    case 223:
      return presence_e::optional;
    case 230:
      return presence_e::optional;
    case 242:
      return presence_e::optional;
    case 246:
      return presence_e::optional;
    case 250:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
uint32_t& init_ue_msg_ies_o::value_c::enb_ue_s1ap_id()
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
unbounded_octstring<true>& init_ue_msg_ies_o::value_c::nas_pdu()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
tai_s& init_ue_msg_ies_o::value_c::tai()
{
  assert_choice_type("TAI", type_.to_string(), "Value");
  return c.get<tai_s>();
}
eutran_cgi_s& init_ue_msg_ies_o::value_c::eutran_cgi()
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
rrc_establishment_cause_e& init_ue_msg_ies_o::value_c::rrc_establishment_cause()
{
  assert_choice_type("RRC-Establishment-Cause", type_.to_string(), "Value");
  return c.get<rrc_establishment_cause_e>();
}
s_tmsi_s& init_ue_msg_ies_o::value_c::s_tmsi()
{
  assert_choice_type("S-TMSI", type_.to_string(), "Value");
  return c.get<s_tmsi_s>();
}
fixed_bitstring<27, false, true>& init_ue_msg_ies_o::value_c::csg_id()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
gummei_s& init_ue_msg_ies_o::value_c::gummei_id()
{
  assert_choice_type("GUMMEI", type_.to_string(), "Value");
  return c.get<gummei_s>();
}
cell_access_mode_e& init_ue_msg_ies_o::value_c::cell_access_mode()
{
  assert_choice_type("CellAccessMode", type_.to_string(), "Value");
  return c.get<cell_access_mode_e>();
}
bounded_bitstring<1, 160, true, true>& init_ue_msg_ies_o::value_c::gw_transport_layer_address()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
relay_node_ind_e& init_ue_msg_ies_o::value_c::relay_node_ind()
{
  assert_choice_type("RelayNode-Indicator", type_.to_string(), "Value");
  return c.get<relay_node_ind_e>();
}
gummei_type_e& init_ue_msg_ies_o::value_c::gummei_type()
{
  assert_choice_type("GUMMEIType", type_.to_string(), "Value");
  return c.get<gummei_type_e>();
}
tunnel_info_s& init_ue_msg_ies_o::value_c::tunnel_info_for_bbf()
{
  assert_choice_type("TunnelInformation", type_.to_string(), "Value");
  return c.get<tunnel_info_s>();
}
bounded_bitstring<1, 160, true, true>& init_ue_msg_ies_o::value_c::sipto_l_gw_transport_layer_address()
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
unbounded_octstring<true>& init_ue_msg_ies_o::value_c::lhn_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_octstring<2, true>& init_ue_msg_ies_o::value_c::mme_group_id()
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<fixed_octstring<2, true> >();
}
uint16_t& init_ue_msg_ies_o::value_c::ue_usage_type()
{
  assert_choice_type("INTEGER (0..255)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
ce_mode_b_support_ind_e& init_ue_msg_ies_o::value_c::ce_mode_b_support_ind()
{
  assert_choice_type("CE-mode-B-SupportIndicator", type_.to_string(), "Value");
  return c.get<ce_mode_b_support_ind_e>();
}
uint32_t& init_ue_msg_ies_o::value_c::dcn_id()
{
  assert_choice_type("INTEGER (0..65535)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
coverage_level_e& init_ue_msg_ies_o::value_c::coverage_level()
{
  assert_choice_type("Coverage-Level", type_.to_string(), "Value");
  return c.get<coverage_level_e>();
}
const uint32_t& init_ue_msg_ies_o::value_c::enb_ue_s1ap_id() const
{
  assert_choice_type("INTEGER (0..16777215)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const unbounded_octstring<true>& init_ue_msg_ies_o::value_c::nas_pdu() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const tai_s& init_ue_msg_ies_o::value_c::tai() const
{
  assert_choice_type("TAI", type_.to_string(), "Value");
  return c.get<tai_s>();
}
const eutran_cgi_s& init_ue_msg_ies_o::value_c::eutran_cgi() const
{
  assert_choice_type("EUTRAN-CGI", type_.to_string(), "Value");
  return c.get<eutran_cgi_s>();
}
const rrc_establishment_cause_e& init_ue_msg_ies_o::value_c::rrc_establishment_cause() const
{
  assert_choice_type("RRC-Establishment-Cause", type_.to_string(), "Value");
  return c.get<rrc_establishment_cause_e>();
}
const s_tmsi_s& init_ue_msg_ies_o::value_c::s_tmsi() const
{
  assert_choice_type("S-TMSI", type_.to_string(), "Value");
  return c.get<s_tmsi_s>();
}
const fixed_bitstring<27, false, true>& init_ue_msg_ies_o::value_c::csg_id() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<fixed_bitstring<27, false, true> >();
}
const gummei_s& init_ue_msg_ies_o::value_c::gummei_id() const
{
  assert_choice_type("GUMMEI", type_.to_string(), "Value");
  return c.get<gummei_s>();
}
const cell_access_mode_e& init_ue_msg_ies_o::value_c::cell_access_mode() const
{
  assert_choice_type("CellAccessMode", type_.to_string(), "Value");
  return c.get<cell_access_mode_e>();
}
const bounded_bitstring<1, 160, true, true>& init_ue_msg_ies_o::value_c::gw_transport_layer_address() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
const relay_node_ind_e& init_ue_msg_ies_o::value_c::relay_node_ind() const
{
  assert_choice_type("RelayNode-Indicator", type_.to_string(), "Value");
  return c.get<relay_node_ind_e>();
}
const gummei_type_e& init_ue_msg_ies_o::value_c::gummei_type() const
{
  assert_choice_type("GUMMEIType", type_.to_string(), "Value");
  return c.get<gummei_type_e>();
}
const tunnel_info_s& init_ue_msg_ies_o::value_c::tunnel_info_for_bbf() const
{
  assert_choice_type("TunnelInformation", type_.to_string(), "Value");
  return c.get<tunnel_info_s>();
}
const bounded_bitstring<1, 160, true, true>& init_ue_msg_ies_o::value_c::sipto_l_gw_transport_layer_address() const
{
  assert_choice_type("BIT STRING", type_.to_string(), "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
const unbounded_octstring<true>& init_ue_msg_ies_o::value_c::lhn_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_octstring<2, true>& init_ue_msg_ies_o::value_c::mme_group_id() const
{
  assert_choice_type("OCTET STRING", type_.to_string(), "Value");
  return c.get<fixed_octstring<2, true> >();
}
const uint16_t& init_ue_msg_ies_o::value_c::ue_usage_type() const
{
  assert_choice_type("INTEGER (0..255)", type_.to_string(), "Value");
  return c.get<uint16_t>();
}
const ce_mode_b_support_ind_e& init_ue_msg_ies_o::value_c::ce_mode_b_support_ind() const
{
  assert_choice_type("CE-mode-B-SupportIndicator", type_.to_string(), "Value");
  return c.get<ce_mode_b_support_ind_e>();
}
const uint32_t& init_ue_msg_ies_o::value_c::dcn_id() const
{
  assert_choice_type("INTEGER (0..65535)", type_.to_string(), "Value");
  return c.get<uint32_t>();
}
const coverage_level_e& init_ue_msg_ies_o::value_c::coverage_level() const
{
  assert_choice_type("Coverage-Level", type_.to_string(), "Value");
  return c.get<coverage_level_e>();
}
void init_ue_msg_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    case types::eutran_cgi:
      c.destroy<eutran_cgi_s>();
      break;
    case types::s_tmsi:
      c.destroy<s_tmsi_s>();
      break;
    case types::csg_id:
      c.destroy<fixed_bitstring<27, false, true> >();
      break;
    case types::gummei_id:
      c.destroy<gummei_s>();
      break;
    case types::gw_transport_layer_address:
      c.destroy<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::tunnel_info_for_bbf:
      c.destroy<tunnel_info_s>();
      break;
    case types::sipto_l_gw_transport_layer_address:
      c.destroy<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::lhn_id:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::mme_group_id:
      c.destroy<fixed_octstring<2, true> >();
      break;
    default:
      break;
  }
}
void init_ue_msg_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::enb_ue_s1ap_id:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::eutran_cgi:
      c.init<eutran_cgi_s>();
      break;
    case types::rrc_establishment_cause:
      break;
    case types::s_tmsi:
      c.init<s_tmsi_s>();
      break;
    case types::csg_id:
      c.init<fixed_bitstring<27, false, true> >();
      break;
    case types::gummei_id:
      c.init<gummei_s>();
      break;
    case types::cell_access_mode:
      break;
    case types::gw_transport_layer_address:
      c.init<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::relay_node_ind:
      break;
    case types::gummei_type:
      break;
    case types::tunnel_info_for_bbf:
      c.init<tunnel_info_s>();
      break;
    case types::sipto_l_gw_transport_layer_address:
      c.init<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::lhn_id:
      c.init<unbounded_octstring<true> >();
      break;
    case types::mme_group_id:
      c.init<fixed_octstring<2, true> >();
      break;
    case types::ue_usage_type:
      break;
    case types::ce_mode_b_support_ind:
      break;
    case types::dcn_id:
      break;
    case types::coverage_level:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }
}
init_ue_msg_ies_o::value_c::value_c(const init_ue_msg_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::enb_ue_s1ap_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::eutran_cgi:
      c.init(other.c.get<eutran_cgi_s>());
      break;
    case types::rrc_establishment_cause:
      c.init(other.c.get<rrc_establishment_cause_e>());
      break;
    case types::s_tmsi:
      c.init(other.c.get<s_tmsi_s>());
      break;
    case types::csg_id:
      c.init(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::gummei_id:
      c.init(other.c.get<gummei_s>());
      break;
    case types::cell_access_mode:
      c.init(other.c.get<cell_access_mode_e>());
      break;
    case types::gw_transport_layer_address:
      c.init(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::relay_node_ind:
      c.init(other.c.get<relay_node_ind_e>());
      break;
    case types::gummei_type:
      c.init(other.c.get<gummei_type_e>());
      break;
    case types::tunnel_info_for_bbf:
      c.init(other.c.get<tunnel_info_s>());
      break;
    case types::sipto_l_gw_transport_layer_address:
      c.init(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::lhn_id:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::mme_group_id:
      c.init(other.c.get<fixed_octstring<2, true> >());
      break;
    case types::ue_usage_type:
      c.init(other.c.get<uint16_t>());
      break;
    case types::ce_mode_b_support_ind:
      c.init(other.c.get<ce_mode_b_support_ind_e>());
      break;
    case types::dcn_id:
      c.init(other.c.get<uint32_t>());
      break;
    case types::coverage_level:
      c.init(other.c.get<coverage_level_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }
}
init_ue_msg_ies_o::value_c& init_ue_msg_ies_o::value_c::operator=(const init_ue_msg_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::enb_ue_s1ap_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::eutran_cgi:
      c.set(other.c.get<eutran_cgi_s>());
      break;
    case types::rrc_establishment_cause:
      c.set(other.c.get<rrc_establishment_cause_e>());
      break;
    case types::s_tmsi:
      c.set(other.c.get<s_tmsi_s>());
      break;
    case types::csg_id:
      c.set(other.c.get<fixed_bitstring<27, false, true> >());
      break;
    case types::gummei_id:
      c.set(other.c.get<gummei_s>());
      break;
    case types::cell_access_mode:
      c.set(other.c.get<cell_access_mode_e>());
      break;
    case types::gw_transport_layer_address:
      c.set(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::relay_node_ind:
      c.set(other.c.get<relay_node_ind_e>());
      break;
    case types::gummei_type:
      c.set(other.c.get<gummei_type_e>());
      break;
    case types::tunnel_info_for_bbf:
      c.set(other.c.get<tunnel_info_s>());
      break;
    case types::sipto_l_gw_transport_layer_address:
      c.set(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::lhn_id:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::mme_group_id:
      c.set(other.c.get<fixed_octstring<2, true> >());
      break;
    case types::ue_usage_type:
      c.set(other.c.get<uint16_t>());
      break;
    case types::ce_mode_b_support_ind:
      c.set(other.c.get<ce_mode_b_support_ind_e>());
      break;
    case types::dcn_id:
      c.set(other.c.get<uint32_t>());
      break;
    case types::coverage_level:
      c.set(other.c.get<coverage_level_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }

  return *this;
}
void init_ue_msg_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::enb_ue_s1ap_id:
      j.write_int("INTEGER (0..16777215)", c.get<uint32_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::tai:
      j.write_fieldname("TAI");
      c.get<tai_s>().to_json(j);
      break;
    case types::eutran_cgi:
      j.write_fieldname("EUTRAN-CGI");
      c.get<eutran_cgi_s>().to_json(j);
      break;
    case types::rrc_establishment_cause:
      j.write_str("RRC-Establishment-Cause", c.get<rrc_establishment_cause_e>().to_string());
      break;
    case types::s_tmsi:
      j.write_fieldname("S-TMSI");
      c.get<s_tmsi_s>().to_json(j);
      break;
    case types::csg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<27, false, true> >().to_string());
      break;
    case types::gummei_id:
      j.write_fieldname("GUMMEI");
      c.get<gummei_s>().to_json(j);
      break;
    case types::cell_access_mode:
      j.write_str("CellAccessMode", "hybrid");
      break;
    case types::gw_transport_layer_address:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 160, true, true> >().to_string());
      break;
    case types::relay_node_ind:
      j.write_str("RelayNode-Indicator", "true");
      break;
    case types::gummei_type:
      j.write_str("GUMMEIType", c.get<gummei_type_e>().to_string());
      break;
    case types::tunnel_info_for_bbf:
      j.write_fieldname("TunnelInformation");
      c.get<tunnel_info_s>().to_json(j);
      break;
    case types::sipto_l_gw_transport_layer_address:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 160, true, true> >().to_string());
      break;
    case types::lhn_id:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::mme_group_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<2, true> >().to_string());
      break;
    case types::ue_usage_type:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::ce_mode_b_support_ind:
      j.write_str("CE-mode-B-SupportIndicator", "supported");
      break;
    case types::dcn_id:
      j.write_int("INTEGER (0..65535)", c.get<uint32_t>());
      break;
    case types::coverage_level:
      j.write_str("Coverage-Level", "extendedcoverage");
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_ue_msg_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().pack(bref));
      break;
    case types::rrc_establishment_cause:
      HANDLE_CODE(c.get<rrc_establishment_cause_e>().pack(bref));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().pack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().pack(bref)));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().pack(bref));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().pack(bref));
      break;
    case types::gw_transport_layer_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().pack(bref)));
      break;
    case types::relay_node_ind:
      HANDLE_CODE(c.get<relay_node_ind_e>().pack(bref));
      break;
    case types::gummei_type:
      HANDLE_CODE(c.get<gummei_type_e>().pack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().pack(bref));
      break;
    case types::sipto_l_gw_transport_layer_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().pack(bref)));
      break;
    case types::lhn_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::mme_group_id:
      HANDLE_CODE((c.get<fixed_octstring<2, true> >().pack(bref)));
      break;
    case types::ue_usage_type:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::ce_mode_b_support_ind:
      HANDLE_CODE(c.get<ce_mode_b_support_ind_e>().pack(bref));
      break;
    case types::dcn_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint32_t>(), (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::coverage_level:
      HANDLE_CODE(c.get<coverage_level_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_ue_msg_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::enb_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    case types::eutran_cgi:
      HANDLE_CODE(c.get<eutran_cgi_s>().unpack(bref));
      break;
    case types::rrc_establishment_cause:
      HANDLE_CODE(c.get<rrc_establishment_cause_e>().unpack(bref));
      break;
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().unpack(bref));
      break;
    case types::csg_id:
      HANDLE_CODE((c.get<fixed_bitstring<27, false, true> >().unpack(bref)));
      break;
    case types::gummei_id:
      HANDLE_CODE(c.get<gummei_s>().unpack(bref));
      break;
    case types::cell_access_mode:
      HANDLE_CODE(c.get<cell_access_mode_e>().unpack(bref));
      break;
    case types::gw_transport_layer_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().unpack(bref)));
      break;
    case types::relay_node_ind:
      HANDLE_CODE(c.get<relay_node_ind_e>().unpack(bref));
      break;
    case types::gummei_type:
      HANDLE_CODE(c.get<gummei_type_e>().unpack(bref));
      break;
    case types::tunnel_info_for_bbf:
      HANDLE_CODE(c.get<tunnel_info_s>().unpack(bref));
      break;
    case types::sipto_l_gw_transport_layer_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().unpack(bref)));
      break;
    case types::lhn_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::mme_group_id:
      HANDLE_CODE((c.get<fixed_octstring<2, true> >().unpack(bref)));
      break;
    case types::ue_usage_type:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::ce_mode_b_support_ind:
      HANDLE_CODE(c.get<ce_mode_b_support_ind_e>().unpack(bref));
      break;
    case types::dcn_id:
      HANDLE_CODE(unpack_integer(c.get<uint32_t>(), bref, (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::coverage_level:
      HANDLE_CODE(c.get<coverage_level_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string init_ue_msg_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..16777215)",
                                  "OCTET STRING",
                                  "TAI",
                                  "EUTRAN-CGI",
                                  "RRC-Establishment-Cause",
                                  "S-TMSI",
                                  "BIT STRING",
                                  "GUMMEI",
                                  "CellAccessMode",
                                  "BIT STRING",
                                  "RelayNode-Indicator",
                                  "GUMMEIType",
                                  "TunnelInformation",
                                  "BIT STRING",
                                  "OCTET STRING",
                                  "OCTET STRING",
                                  "INTEGER (0..255)",
                                  "CE-mode-B-SupportIndicator",
                                  "INTEGER (0..65535)",
                                  "Coverage-Level"};
  return convert_enum_idx(options, 20, value, "init_ue_msg_ies_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_field_s<init_ue_msg_ies_o>;

init_ue_msg_ies_container::init_ue_msg_ies_container() :
  enb_ue_s1ap_id(8, crit_e::reject),
  nas_pdu(26, crit_e::reject),
  tai(67, crit_e::reject),
  eutran_cgi(100, crit_e::ignore),
  rrc_establishment_cause(134, crit_e::ignore),
  s_tmsi(96, crit_e::reject),
  csg_id(127, crit_e::reject),
  gummei_id(75, crit_e::reject),
  cell_access_mode(145, crit_e::reject),
  gw_transport_layer_address(155, crit_e::ignore),
  relay_node_ind(160, crit_e::reject),
  gummei_type(170, crit_e::ignore),
  tunnel_info_for_bbf(176, crit_e::ignore),
  sipto_l_gw_transport_layer_address(184, crit_e::ignore),
  lhn_id(186, crit_e::ignore),
  mme_group_id(223, crit_e::ignore),
  ue_usage_type(230, crit_e::ignore),
  ce_mode_b_support_ind(242, crit_e::ignore),
  dcn_id(246, crit_e::ignore),
  coverage_level(250, crit_e::ignore)
{}
SRSASN_CODE init_ue_msg_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  nof_ies += s_tmsi_present ? 1 : 0;
  nof_ies += csg_id_present ? 1 : 0;
  nof_ies += gummei_id_present ? 1 : 0;
  nof_ies += cell_access_mode_present ? 1 : 0;
  nof_ies += gw_transport_layer_address_present ? 1 : 0;
  nof_ies += relay_node_ind_present ? 1 : 0;
  nof_ies += gummei_type_present ? 1 : 0;
  nof_ies += tunnel_info_for_bbf_present ? 1 : 0;
  nof_ies += sipto_l_gw_transport_layer_address_present ? 1 : 0;
  nof_ies += lhn_id_present ? 1 : 0;
  nof_ies += mme_group_id_present ? 1 : 0;
  nof_ies += ue_usage_type_present ? 1 : 0;
  nof_ies += ce_mode_b_support_ind_present ? 1 : 0;
  nof_ies += dcn_id_present ? 1 : 0;
  nof_ies += coverage_level_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(enb_ue_s1ap_id.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(eutran_cgi.pack(bref));
  HANDLE_CODE(rrc_establishment_cause.pack(bref));
  if (s_tmsi_present) {
    HANDLE_CODE(s_tmsi.pack(bref));
  }
  if (csg_id_present) {
    HANDLE_CODE(csg_id.pack(bref));
  }
  if (gummei_id_present) {
    HANDLE_CODE(gummei_id.pack(bref));
  }
  if (cell_access_mode_present) {
    HANDLE_CODE(cell_access_mode.pack(bref));
  }
  if (gw_transport_layer_address_present) {
    HANDLE_CODE(gw_transport_layer_address.pack(bref));
  }
  if (relay_node_ind_present) {
    HANDLE_CODE(relay_node_ind.pack(bref));
  }
  if (gummei_type_present) {
    HANDLE_CODE(gummei_type.pack(bref));
  }
  if (tunnel_info_for_bbf_present) {
    HANDLE_CODE(tunnel_info_for_bbf.pack(bref));
  }
  if (sipto_l_gw_transport_layer_address_present) {
    HANDLE_CODE(sipto_l_gw_transport_layer_address.pack(bref));
  }
  if (lhn_id_present) {
    HANDLE_CODE(lhn_id.pack(bref));
  }
  if (mme_group_id_present) {
    HANDLE_CODE(mme_group_id.pack(bref));
  }
  if (ue_usage_type_present) {
    HANDLE_CODE(ue_usage_type.pack(bref));
  }
  if (ce_mode_b_support_ind_present) {
    HANDLE_CODE(ce_mode_b_support_ind.pack(bref));
  }
  if (dcn_id_present) {
    HANDLE_CODE(dcn_id.pack(bref));
  }
  if (coverage_level_present) {
    HANDLE_CODE(coverage_level.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_ue_msg_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    protocol_ie_field_s<init_ue_msg_ies_o> c;
    HANDLE_CODE(c.unpack(bref));
    switch (c.id) {
      case 8:
        nof_mandatory_ies--;
        enb_ue_s1ap_id.id    = c.id;
        enb_ue_s1ap_id.crit  = c.crit;
        enb_ue_s1ap_id.value = c.value.enb_ue_s1ap_id();
        break;
      case 26:
        nof_mandatory_ies--;
        nas_pdu.id    = c.id;
        nas_pdu.crit  = c.crit;
        nas_pdu.value = c.value.nas_pdu();
        break;
      case 67:
        nof_mandatory_ies--;
        tai.id    = c.id;
        tai.crit  = c.crit;
        tai.value = c.value.tai();
        break;
      case 100:
        nof_mandatory_ies--;
        eutran_cgi.id    = c.id;
        eutran_cgi.crit  = c.crit;
        eutran_cgi.value = c.value.eutran_cgi();
        break;
      case 134:
        nof_mandatory_ies--;
        rrc_establishment_cause.id    = c.id;
        rrc_establishment_cause.crit  = c.crit;
        rrc_establishment_cause.value = c.value.rrc_establishment_cause();
        break;
      case 96:
        s_tmsi_present = true;
        s_tmsi.id      = c.id;
        s_tmsi.crit    = c.crit;
        s_tmsi.value   = c.value.s_tmsi();
        break;
      case 127:
        csg_id_present = true;
        csg_id.id      = c.id;
        csg_id.crit    = c.crit;
        csg_id.value   = c.value.csg_id();
        break;
      case 75:
        gummei_id_present = true;
        gummei_id.id      = c.id;
        gummei_id.crit    = c.crit;
        gummei_id.value   = c.value.gummei_id();
        break;
      case 145:
        cell_access_mode_present = true;
        cell_access_mode.id      = c.id;
        cell_access_mode.crit    = c.crit;
        cell_access_mode.value   = c.value.cell_access_mode();
        break;
      case 155:
        gw_transport_layer_address_present = true;
        gw_transport_layer_address.id      = c.id;
        gw_transport_layer_address.crit    = c.crit;
        gw_transport_layer_address.value   = c.value.gw_transport_layer_address();
        break;
      case 160:
        relay_node_ind_present = true;
        relay_node_ind.id      = c.id;
        relay_node_ind.crit    = c.crit;
        relay_node_ind.value   = c.value.relay_node_ind();
        break;
      case 170:
        gummei_type_present = true;
        gummei_type.id      = c.id;
        gummei_type.crit    = c.crit;
        gummei_type.value   = c.value.gummei_type();
        break;
      case 176:
        tunnel_info_for_bbf_present = true;
        tunnel_info_for_bbf.id      = c.id;
        tunnel_info_for_bbf.crit    = c.crit;
        tunnel_info_for_bbf.value   = c.value.tunnel_info_for_bbf();
        break;
      case 184:
        sipto_l_gw_transport_layer_address_present = true;
        sipto_l_gw_transport_layer_address.id      = c.id;
        sipto_l_gw_transport_layer_address.crit    = c.crit;
        sipto_l_gw_transport_layer_address.value   = c.value.sipto_l_gw_transport_layer_address();
        break;
      case 186:
        lhn_id_present = true;
        lhn_id.id      = c.id;
        lhn_id.crit    = c.crit;
        lhn_id.value   = c.value.lhn_id();
        break;
      case 223:
        mme_group_id_present = true;
        mme_group_id.id      = c.id;
        mme_group_id.crit    = c.crit;
        mme_group_id.value   = c.value.mme_group_id();
        break;
      case 230:
        ue_usage_type_present = true;
        ue_usage_type.id      = c.id;
        ue_usage_type.crit    = c.crit;
        ue_usage_type.value   = c.value.ue_usage_type();
        break;
      case 242:
        ce_mode_b_support_ind_present = true;
        ce_mode_b_support_ind.id      = c.id;
        ce_mode_b_support_ind.crit    = c.crit;
        ce_mode_b_support_ind.value   = c.value.ce_mode_b_support_ind();
        break;
      case 246:
        dcn_id_present = true;
        dcn_id.id      = c.id;
        dcn_id.crit    = c.crit;
        dcn_id.value   = c.value.dcn_id();
        break;
      case 250:
        coverage_level_present = true;
        coverage_level.id      = c.id;
        coverage_level.crit    = c.crit;
        coverage_level.value   = c.value.coverage_level();
        break;
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", c.id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_ue_msg_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  enb_ue_s1ap_id.to_json(j);
  j.write_fieldname("");
  nas_pdu.to_json(j);
  j.write_fieldname("");
  tai.to_json(j);
  j.write_fieldname("");
  eutran_cgi.to_json(j);
  j.write_fieldname("");
  rrc_establishment_cause.to_json(j);
  if (s_tmsi_present) {
    j.write_fieldname("");
    s_tmsi.to_json(j);
  }
  if (csg_id_present) {
    j.write_fieldname("");
    csg_id.to_json(j);
  }
  if (gummei_id_present) {
    j.write_fieldname("");
    gummei_id.to_json(j);
  }
  if (cell_access_mode_present) {
    j.write_fieldname("");
    cell_access_mode.to_json(j);
  }
  if (gw_transport_layer_address_present) {
    j.write_fieldname("");
    gw_transport_layer_address.to_json(j);
  }
  if (relay_node_ind_present) {
    j.write_fieldname("");
    relay_node_ind.to_json(j);
  }
  if (gummei_type_present) {
    j.write_fieldname("");
    gummei_type.to_json(j);
  }
  if (tunnel_info_for_bbf_present) {
    j.write_fieldname("");
    tunnel_info_for_bbf.to_json(j);
  }
  if (sipto_l_gw_transport_layer_address_present) {
    j.write_fieldname("");
    sipto_l_gw_transport_layer_address.to_json(j);
  }
  if (lhn_id_present) {
    j.write_fieldname("");
    lhn_id.to_json(j);
  }
  if (mme_group_id_present) {
    j.write_fieldname("");
    mme_group_id.to_json(j);
  }
  if (ue_usage_type_present) {
    j.write_fieldname("");
    ue_usage_type.to_json(j);
  }
  if (ce_mode_b_support_ind_present) {
    j.write_fieldname("");
    ce_mode_b_support_ind.to_json(j);
  }
  if (dcn_id_present) {
    j.write_fieldname("");
    dcn_id.to_json(j);
  }
  if (coverage_level_present) {
    j.write_fieldname("");
    coverage_level.to_json(j);
  }
  j.end_obj();
}

// InitialUEMessage ::= SEQUENCE
SRSASN_CODE init_ue_msg_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(protocol_ies.pack(bref));

  bref.align_bytes_zero();

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_ue_msg_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(protocol_ies.unpack(bref));

  bref.align_bytes();

  return SRSASN_SUCCESS;
}
void init_ue_msg_s::to_json(json_writer& j) const
{
  j.start_array();
  j.start_obj();
  j.start_obj("InitialUEMessage");
  j.write_fieldname("protocolIEs");
  protocol_ies.to_json(j);
  j.end_obj();
  j.end_obj();
  j.end_array();
}

// UE-associatedLogicalS1-ConnectionItem ::= SEQUENCE
SRSASN_CODE ue_associated_lc_s1_conn_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(mme_ue_s1ap_id_present, 1));
  HANDLE_CODE(bref.pack(enb_ue_s1ap_id_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (mme_ue_s1ap_id_present) {
    HANDLE_CODE(pack_integer(bref, mme_ue_s1ap_id, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  }
  if (enb_ue_s1ap_id_present) {
    HANDLE_CODE(pack_integer(bref, enb_ue_s1ap_id, (uint32_t)0u, (uint32_t)16777215u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_associated_lc_s1_conn_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(mme_ue_s1ap_id_present, 1));
  HANDLE_CODE(bref.unpack(enb_ue_s1ap_id_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (mme_ue_s1ap_id_present) {
    HANDLE_CODE(unpack_integer(mme_ue_s1ap_id, bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  }
  if (enb_ue_s1ap_id_present) {
    HANDLE_CODE(unpack_integer(enb_ue_s1ap_id, bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_associated_lc_s1_conn_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (mme_ue_s1ap_id_present) {
    j.write_int("mME-UE-S1AP-ID", mme_ue_s1ap_id);
  }
  if (enb_ue_s1ap_id_present) {
    j.write_int("eNB-UE-S1AP-ID", enb_ue_s1ap_id);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIItem ::= SEQUENCE
SRSASN_CODE tai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UE-associatedLogicalS1-ConnectionItemRes ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ue_associated_lc_s1_conn_item_res_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {91};
  return map_enum_number(options, 1, idx, "id");
}
bool ue_associated_lc_s1_conn_item_res_o::is_id_valid(const uint32_t& id)
{
  return 91 == id;
}
crit_e ue_associated_lc_s1_conn_item_res_o::get_crit(const uint32_t& id)
{
  if (id == 91) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
ue_associated_lc_s1_conn_item_res_o::value_c ue_associated_lc_s1_conn_item_res_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 91) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ue_associated_lc_s1_conn_item_res_o::get_presence(const uint32_t& id)
{
  if (id == 91) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void ue_associated_lc_s1_conn_item_res_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("UE-associatedLogicalS1-ConnectionItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE ue_associated_lc_s1_conn_item_res_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_associated_lc_s1_conn_item_res_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string ue_associated_lc_s1_conn_item_res_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"UE-associatedLogicalS1-ConnectionItem"};
  return convert_enum_idx(options, 1, value, "ue_associated_lc_s1_conn_item_res_o::value_c::types");
}
uint8_t ue_associated_lc_s1_conn_item_res_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {1};
  return map_enum_number(options, 1, value, "ue_associated_lc_s1_conn_item_res_o::value_c::types");
}

// NB-IoT-Paging-eDRX-Cycle ::= ENUMERATED
std::string nb_io_t_paging_e_drx_cycle_opts::to_string() const
{
  static const char* options[] = {
      "hf2", "hf4", "hf6", "hf8", "hf10", "hf12", "hf14", "hf16", "hf32", "hf64", "hf128", "hf256", "hf512", "hf1024"};
  return convert_enum_idx(options, 14, value, "nb_io_t_paging_e_drx_cycle_e");
}
uint16_t nb_io_t_paging_e_drx_cycle_opts::to_number() const
{
  static const uint16_t options[] = {2, 4, 6, 8, 10, 12, 14, 16, 32, 64, 128, 256, 512, 1024};
  return map_enum_number(options, 14, value, "nb_io_t_paging_e_drx_cycle_e");
}

// NB-IoT-PagingTimeWindow ::= ENUMERATED
std::string nb_io_t_paging_time_win_opts::to_string() const
{
  static const char* options[] = {
      "s1", "s2", "s3", "s4", "s5", "s6", "s7", "s8", "s9", "s10", "s11", "s12", "s13", "s14", "s15", "s16"};
  return convert_enum_idx(options, 16, value, "nb_io_t_paging_time_win_e");
}
uint8_t nb_io_t_paging_time_win_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16};
  return map_enum_number(options, 16, value, "nb_io_t_paging_time_win_e");
}

// OverloadAction ::= ENUMERATED
std::string overload_action_opts::to_string() const
{
  static const char* options[] = {
      "reject-non-emergency-mo-dt",
      "reject-rrc-cr-signalling",
      "permit-emergency-sessions-and-mobile-terminated-services-only",
      "permit-high-priority-sessions-and-mobile-terminated-services-only",
      "reject-delay-tolerant-access",
      "permit-high-priority-sessions-and-exception-reporting-and-mobile-terminated-services-only",
      "not-accept-mo-data-or-delay-tolerant-access-from-CP-CIoT"};
  return convert_enum_idx(options, 7, value, "overload_action_e");
}

// Paging-eDRX-Cycle ::= ENUMERATED
std::string paging_e_drx_cycle_opts::to_string() const
{
  static const char* options[] = {
      "hfhalf", "hf1", "hf2", "hf4", "hf6", "hf8", "hf10", "hf12", "hf14", "hf16", "hf32", "hf64", "hf128", "hf256"};
  return convert_enum_idx(options, 14, value, "paging_e_drx_cycle_e");
}
float paging_e_drx_cycle_opts::to_number() const
{
  static const float options[] = {0.5, 1.0, 2.0, 4.0, 6.0, 8.0, 10.0, 12.0, 14.0, 16.0, 32.0, 64.0, 128.0, 256.0};
  return map_enum_number(options, 14, value, "paging_e_drx_cycle_e");
}
std::string paging_e_drx_cycle_opts::to_number_string() const
{
  static const char* options[] = {"0.5", "1", "2", "4", "6", "8", "10", "12", "14", "16", "32", "64", "128", "256"};
  return convert_enum_idx(options, 14, value, "paging_e_drx_cycle_e");
}

// PagingTimeWindow ::= ENUMERATED
std::string paging_time_win_opts::to_string() const
{
  static const char* options[] = {
      "s1", "s2", "s3", "s4", "s5", "s6", "s7", "s8", "s9", "s10", "s11", "s12", "s13", "s14", "s15", "s16"};
  return convert_enum_idx(options, 16, value, "paging_time_win_e");
}
uint8_t paging_time_win_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16};
  return map_enum_number(options, 16, value, "paging_time_win_e");
}

// ResetAll ::= ENUMERATED
std::string reset_all_opts::to_string() const
{
  static const char* options[] = {"reset-all"};
  return convert_enum_idx(options, 1, value, "reset_all_e");
}

// ServedDCNsItem ::= SEQUENCE
SRSASN_CODE served_dcns_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, dcn_id, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(pack_integer(bref, relative_dcn_capacity, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE served_dcns_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(dcn_id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(unpack_integer(relative_dcn_capacity, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void served_dcns_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dCN-ID", dcn_id);
  j.write_int("relativeDCNCapacity", relative_dcn_capacity);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ServedGUMMEIsItem ::= SEQUENCE
SRSASN_CODE served_gummeis_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, served_plmns, 1, 32, true));
  HANDLE_CODE(pack_dyn_seq_of(bref, served_group_ids, 1, 65535, true));
  HANDLE_CODE(pack_dyn_seq_of(bref, served_mmecs, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE served_gummeis_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(served_plmns, bref, 1, 32, true));
  HANDLE_CODE(unpack_dyn_seq_of(served_group_ids, bref, 1, 65535, true));
  HANDLE_CODE(unpack_dyn_seq_of(served_mmecs, bref, 1, 256, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void served_gummeis_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("servedPLMNs");
  for (const auto& e1 : served_plmns) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.start_array("servedGroupIDs");
  for (const auto& e1 : served_group_ids) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  j.start_array("servedMMECs");
  for (const auto& e1 : served_mmecs) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIItemIEs ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t tai_item_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {47};
  return map_enum_number(options, 1, idx, "id");
}
bool tai_item_ies_o::is_id_valid(const uint32_t& id)
{
  return 47 == id;
}
crit_e tai_item_ies_o::get_crit(const uint32_t& id)
{
  if (id == 47) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
tai_item_ies_o::value_c tai_item_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 47) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e tai_item_ies_o::get_presence(const uint32_t& id)
{
  if (id == 47) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void tai_item_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("TAIItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE tai_item_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_item_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string tai_item_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"TAIItem"};
  return convert_enum_idx(options, 1, value, "tai_item_ies_o::value_c::types");
}

// UE-S1AP-ID-pair ::= SEQUENCE
SRSASN_CODE ue_s1ap_id_pair_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, mme_ue_s1ap_id, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  HANDLE_CODE(pack_integer(bref, enb_ue_s1ap_id, (uint32_t)0u, (uint32_t)16777215u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_s1ap_id_pair_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(mme_ue_s1ap_id, bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  HANDLE_CODE(unpack_integer(enb_ue_s1ap_id, bref, (uint32_t)0u, (uint32_t)16777215u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_s1ap_id_pair_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("mME-UE-S1AP-ID", mme_ue_s1ap_id);
  j.write_int("eNB-UE-S1AP-ID", enb_ue_s1ap_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UE-associatedLogicalS1-ConnectionItemResAck ::= OBJECT SET OF S1AP-PROTOCOL-IES
uint32_t ue_associated_lc_s1_conn_item_res_ack_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {91};
  return map_enum_number(options, 1, idx, "id");
}
bool ue_associated_lc_s1_conn_item_res_ack_o::is_id_valid(const uint32_t& id)
{
  return 91 == id;
}
crit_e ue_associated_lc_s1_conn_item_res_ack_o::get_crit(const uint32_t& id)
{
  if (id == 91) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
ue_associated_lc_s1_conn_item_res_ack_o::value_c ue_associated_lc_s1_conn_item_res_ack_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 91) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ue_associated_lc_s1_conn_item_res_ack_o::get_presence(const uint32_t& id)
{
  if (id == 91) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void ue_associated_lc_s1_conn_item_res_ack_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("UE-associatedLogicalS1-ConnectionItem");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE ue_associated_lc_s1_conn_item_res_ack_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_associated_lc_s1_conn_item_res_ack_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string ue_associated_lc_s1_conn_item_res_ack_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"UE-associatedLogicalS1-ConnectionItem"};
  return convert_enum_idx(options, 1, value, "ue_associated_lc_s1_conn_item_res_ack_o::value_c::types");
}
uint8_t ue_associated_lc_s1_conn_item_res_ack_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {1};
  return map_enum_number(options, 1, value, "ue_associated_lc_s1_conn_item_res_ack_o::value_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<ue_associated_lc_s1_conn_item_res_o>;

// CNDomain ::= ENUMERATED
std::string cn_domain_opts::to_string() const
{
  static const char* options[] = {"ps", "cs"};
  return convert_enum_idx(options, 2, value, "cn_domain_e");
}

// Cdma2000HORequiredIndication ::= ENUMERATED
std::string cdma2000_ho_required_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "cdma2000_ho_required_ind_e");
}

// ConcurrentWarningMessageIndicator ::= ENUMERATED
std::string concurrent_warning_msg_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "concurrent_warning_msg_ind_e");
}

// GWContextReleaseIndication ::= ENUMERATED
std::string gw_context_release_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "gw_context_release_ind_e");
}

// KillAllWarningMessages ::= ENUMERATED
std::string kill_all_warning_msgs_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "kill_all_warning_msgs_e");
}

// MMERelaySupportIndicator ::= ENUMERATED
std::string mme_relay_support_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "mme_relay_support_ind_e");
}

// NB-IoT-Paging-eDRXInformation ::= SEQUENCE
SRSASN_CODE nb_io_t_paging_e_drx_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(nb_io_t_paging_time_win_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nb_io_t_paging_e_drx_cycle.pack(bref));
  if (nb_io_t_paging_time_win_present) {
    HANDLE_CODE(nb_io_t_paging_time_win.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE nb_io_t_paging_e_drx_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(nb_io_t_paging_time_win_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nb_io_t_paging_e_drx_cycle.unpack(bref));
  if (nb_io_t_paging_time_win_present) {
    HANDLE_CODE(nb_io_t_paging_time_win.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void nb_io_t_paging_e_drx_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("nB-IoT-paging-eDRX-Cycle", nb_io_t_paging_e_drx_cycle.to_string());
  if (nb_io_t_paging_time_win_present) {
    j.write_str("nB-IoT-pagingTimeWindow", nb_io_t_paging_time_win.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// OverloadResponse ::= CHOICE
void overload_resp_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("overloadAction", c.to_string());
  j.end_obj();
}
SRSASN_CODE overload_resp_c::pack(bit_ref& bref) const
{
  pack_enum(bref, type());
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE overload_resp_c::unpack(cbit_ref& bref)
{
  types e;
  unpack_enum(e, bref);
  if (e != type()) {
    log_invalid_choice_id(e, "overload_resp_c");
    return SRSASN_ERROR_DECODE_FAIL;
  }
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

std::string overload_resp_c::types_opts::to_string() const
{
  static const char* options[] = {"overloadAction"};
  return convert_enum_idx(options, 1, value, "overload_resp_c::types");
}

// Paging-eDRXInformation ::= SEQUENCE
SRSASN_CODE paging_e_drx_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(paging_time_win_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(paging_e_drx_cycle.pack(bref));
  if (paging_time_win_present) {
    HANDLE_CODE(paging_time_win.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE paging_e_drx_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(paging_time_win_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(paging_e_drx_cycle.unpack(bref));
  if (paging_time_win_present) {
    HANDLE_CODE(paging_time_win.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void paging_e_drx_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("paging-eDRX-Cycle", paging_e_drx_cycle.to_string());
  if (paging_time_win_present) {
    j.write_str("pagingTimeWindow", paging_time_win.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PagingPriority ::= ENUMERATED
std::string paging_prio_opts::to_string() const
{
  static const char* options[] = {
      "priolevel1", "priolevel2", "priolevel3", "priolevel4", "priolevel5", "priolevel6", "priolevel7", "priolevel8"};
  return convert_enum_idx(options, 8, value, "paging_prio_e");
}
uint8_t paging_prio_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 3, 4, 5, 6, 7, 8};
  return map_enum_number(options, 8, value, "paging_prio_e");
}

// ResetType ::= CHOICE
void reset_type_c::destroy_()
{
  switch (type_) {
    case types::part_of_s1_interface:
      c.destroy<ue_associated_lc_s1_conn_list_res_l>();
      break;
    default:
      break;
  }
}
void reset_type_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::s1_interface:
      break;
    case types::part_of_s1_interface:
      c.init<ue_associated_lc_s1_conn_list_res_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }
}
reset_type_c::reset_type_c(const reset_type_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::s1_interface:
      c.init(other.c.get<reset_all_e>());
      break;
    case types::part_of_s1_interface:
      c.init(other.c.get<ue_associated_lc_s1_conn_list_res_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }
}
reset_type_c& reset_type_c::operator=(const reset_type_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::s1_interface:
      c.set(other.c.get<reset_all_e>());
      break;
    case types::part_of_s1_interface:
      c.set(other.c.get<ue_associated_lc_s1_conn_list_res_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }

  return *this;
}
void reset_type_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::s1_interface:
      j.write_str("s1-Interface", "reset-all");
      break;
    case types::part_of_s1_interface:
      j.start_array("partOfS1-Interface");
      for (const auto& e1 : c.get<ue_associated_lc_s1_conn_list_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }
  j.end_obj();
}
SRSASN_CODE reset_type_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::s1_interface:
      HANDLE_CODE(c.get<reset_all_e>().pack(bref));
      break;
    case types::part_of_s1_interface:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<ue_associated_lc_s1_conn_list_res_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE reset_type_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::s1_interface:
      HANDLE_CODE(c.get<reset_all_e>().unpack(bref));
      break;
    case types::part_of_s1_interface:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<ue_associated_lc_s1_conn_list_res_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string reset_type_c::types_opts::to_string() const
{
  static const char* options[] = {"s1-Interface", "partOfS1-Interface"};
  return convert_enum_idx(options, 2, value, "reset_type_c::types");
}

// SRVCCOperationNotPossible ::= ENUMERATED
std::string srvcc_operation_not_possible_opts::to_string() const
{
  static const char* options[] = {"notPossible"};
  return convert_enum_idx(options, 1, value, "srvcc_operation_not_possible_e");
}

template struct asn1::s1ap::protocol_ie_single_container_s<tai_item_ies_o>;

// UE-RetentionInformation ::= ENUMERATED
std::string ue_retention_info_opts::to_string() const
{
  static const char* options[] = {"ues-retained"};
  return convert_enum_idx(options, 1, value, "ue_retention_info_e");
}

// UE-S1AP-IDs ::= CHOICE
void ue_s1ap_ids_c::destroy_()
{
  switch (type_) {
    case types::ue_s1ap_id_pair:
      c.destroy<ue_s1ap_id_pair_s>();
      break;
    default:
      break;
  }
}
void ue_s1ap_ids_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ue_s1ap_id_pair:
      c.init<ue_s1ap_id_pair_s>();
      break;
    case types::mme_ue_s1ap_id:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_s1ap_ids_c");
  }
}
ue_s1ap_ids_c::ue_s1ap_ids_c(const ue_s1ap_ids_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ue_s1ap_id_pair:
      c.init(other.c.get<ue_s1ap_id_pair_s>());
      break;
    case types::mme_ue_s1ap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_s1ap_ids_c");
  }
}
ue_s1ap_ids_c& ue_s1ap_ids_c::operator=(const ue_s1ap_ids_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ue_s1ap_id_pair:
      c.set(other.c.get<ue_s1ap_id_pair_s>());
      break;
    case types::mme_ue_s1ap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_s1ap_ids_c");
  }

  return *this;
}
void ue_s1ap_ids_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ue_s1ap_id_pair:
      j.write_fieldname("uE-S1AP-ID-pair");
      c.get<ue_s1ap_id_pair_s>().to_json(j);
      break;
    case types::mme_ue_s1ap_id:
      j.write_int("mME-UE-S1AP-ID", c.get<uint64_t>());
      break;
    default:
      log_invalid_choice_id(type_, "ue_s1ap_ids_c");
  }
  j.end_obj();
}
SRSASN_CODE ue_s1ap_ids_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::ue_s1ap_id_pair:
      HANDLE_CODE(c.get<ue_s1ap_id_pair_s>().pack(bref));
      break;
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "ue_s1ap_ids_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_s1ap_ids_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::ue_s1ap_id_pair:
      HANDLE_CODE(c.get<ue_s1ap_id_pair_s>().unpack(bref));
      break;
    case types::mme_ue_s1ap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    default:
      log_invalid_choice_id(type_, "ue_s1ap_ids_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ue_s1ap_ids_c::types_opts::to_string() const
{
  static const char* options[] = {"uE-S1AP-ID-pair", "mME-UE-S1AP-ID"};
  return convert_enum_idx(options, 2, value, "ue_s1ap_ids_c::types");
}

template struct asn1::s1ap::protocol_ie_single_container_s<ue_associated_lc_s1_conn_item_res_ack_o>;

// UEPagingID ::= CHOICE
void ue_paging_id_c::destroy_()
{
  switch (type_) {
    case types::s_tmsi:
      c.destroy<s_tmsi_s>();
      break;
    case types::imsi:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void ue_paging_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::s_tmsi:
      c.init<s_tmsi_s>();
      break;
    case types::imsi:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }
}
ue_paging_id_c::ue_paging_id_c(const ue_paging_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::s_tmsi:
      c.init(other.c.get<s_tmsi_s>());
      break;
    case types::imsi:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }
}
ue_paging_id_c& ue_paging_id_c::operator=(const ue_paging_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::s_tmsi:
      c.set(other.c.get<s_tmsi_s>());
      break;
    case types::imsi:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }

  return *this;
}
void ue_paging_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::s_tmsi:
      j.write_fieldname("s-TMSI");
      c.get<s_tmsi_s>().to_json(j);
      break;
    case types::imsi:
      j.write_str("iMSI", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }
  j.end_obj();
}
SRSASN_CODE ue_paging_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().pack(bref));
      break;
    case types::imsi:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_paging_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::s_tmsi:
      HANDLE_CODE(c.get<s_tmsi_s>().unpack(bref));
      break;
    case types::imsi:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

std::string ue_paging_id_c::types_opts::to_string() const
{
  static const char* options[] = {"s-TMSI", "iMSI"};
  return convert_enum_idx(options, 2, value, "ue_paging_id_c::types");
}

// VoiceSupportMatchIndicator ::= ENUMERATED
std::string voice_support_match_ind_opts::to_string() const
{
  static const char* options[] = {"supported", "not-supported"};
  return convert_enum_idx(options, 2, value, "voice_support_match_ind_e");
}

// WarningAreaList ::= CHOICE
void warning_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_list:
      c.destroy<ecgi_list_l>();
      break;
    case types::tracking_area_listfor_warning:
      c.destroy<tai_listfor_warning_l>();
      break;
    case types::emergency_area_id_list:
      c.destroy<emergency_area_id_list_l>();
      break;
    default:
      break;
  }
}
void warning_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_list:
      c.init<ecgi_list_l>();
      break;
    case types::tracking_area_listfor_warning:
      c.init<tai_listfor_warning_l>();
      break;
    case types::emergency_area_id_list:
      c.init<emergency_area_id_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
  }
}
warning_area_list_c::warning_area_list_c(const warning_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_list:
      c.init(other.c.get<ecgi_list_l>());
      break;
    case types::tracking_area_listfor_warning:
      c.init(other.c.get<tai_listfor_warning_l>());
      break;
    case types::emergency_area_id_list:
      c.init(other.c.get<emergency_area_id_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
  }
}
warning_area_list_c& warning_area_list_c::operator=(const warning_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switc